# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2002, 2003 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import math
import os
import random
import threading
import time
import config
import events, hub
import errors
import log
import dbitem
import requests

from helper import debug

#
# songdb class
#

class songdb(threading.Thread):
    def __init__(self, id, songdbbase, dbfile, basedir, playingstatslength, cachesize):
        threading.Thread.__init__(self)
        self.id = id
        self.songdbbase = songdbbase
        self.dbfile = dbfile
        self.basedir = basedir
        self.playingstatslength = playingstatslength

        if not os.path.isdir(self.basedir):
            raise errors.configurationerror("musicbasedir of database is not a directory.")

        if not os.access(self.basedir, os.X_OK | os.R_OK):
            raise errors.configurationerror("you are not allowed to access and read config.general.musicbasedir.")

        if self.songdbbase or self.dbfile:
            try:
                try:
                    self._initdb3(cachesize)
                except ImportError:
                    self._initdb()
            except:
                raise errors.databaseerror("cannot initialise/open song database files.")
        else:
            self.songs = {}
            self.artists = {}
            self.albums = {}
            self.genres = {}
            self.years = {}
            self.playlists = {}
            self.stats = {}

        if not self.stats:
            # insert lists into statistics db 
            if not "topplayed" in self.stats.keys():
                self.stats["topplayed"] = []
            if not "lastplayed" in self.stats.keys():
                self.stats["lastplayed"] = []
            if not "lastadded" in self.stats.keys():
                self.stats["lastadded"] = []

            # set version number for new, empty database
            self.stats["db_version"] = 2

        # set database version for older databases
        if not "db_version" in self.stats.keys():
            self.stats["db_version"] = 1

        # delete old version of lastplayed list
        if self.stats["lastplayed"] and not type(self.stats["lastplayed"][0]) == type(()):
            self.stats["lastplayed"] = []

        # upgrade databa
        if self.stats["db_version"]<2:
            self._updatefromversion1to2()
            self.stats["db_version"] = 2
            
        if self.stats["db_version"]>2:
            raise RuntimeError("database version %d not supported" % self.stats["db_version"])

        # empty cache of all artists and all songs
        # This variables contain tuples acting as caches for
        # all artists and all songs, respectively, in the database
        self.allartists = self.allsongs = None

        # as independent thread, we want our own event and request channel
        self.channel = hub.hub.newchannel()

        # we need to be informed about database changes
        self.channel.subscribe(events.updatesong, self.updatesong)
        self.channel.subscribe(events.delsong, self.delsong)
        self.channel.subscribe(events.updatealbum, self.updatealbum)
        self.channel.subscribe(events.updateartist, self.updateartist)
        self.channel.subscribe(events.registersongs, self.registersongs)
        self.channel.subscribe(events.registerplaylists, self.registerplaylists)
        self.channel.subscribe(events.quit, self.quit)

        self.done = 0

        # we are a database service provider...
        self.channel.supply(requests.getdatabaseinfo, self.getdatabaseinfo)
        self.channel.supply(requests.queryregistersong, self.queryregistersong)
        self.channel.supply(requests.getartists, self.getartists)
        self.channel.supply(requests.getalbums, self.getalbums)
        self.channel.supply(requests.getalbum, self.getalbum)
        self.channel.supply(requests.getartist, self.getartist)
        self.channel.supply(requests.getsong, self.getsong)
        self.channel.supply(requests.getsongs, self.getsongs)
        self.channel.supply(requests.getnumberofsongs, self.getnumberofsongs)
        self.channel.supply(requests.getnumberofalbums, self.getnumberofalbums)
        self.channel.supply(requests.getnumberofgenres, self.getnumberofgenres)
        self.channel.supply(requests.getnumberofdecades, self.getnumberofdecades)
        self.channel.supply(requests.getgenres, self.getgenres)
        self.channel.supply(requests.getyears, self.getyears)
        self.channel.supply(requests.getdecades, self.getdecades)
        self.channel.supply(requests.getlastplayedsongs, self.getlastplayedsongs)
        self.channel.supply(requests.gettopplayedsongs, self.gettopplayedsongs)
        self.channel.supply(requests.getlastaddedsongs, self.getlastaddedsongs)
        self.channel.supply(requests.getplaylist, self.getplaylist)
        self.channel.supply(requests.getplaylists, self.getplaylists)
        self.channel.supply(requests.getsongsinplaylist, self.getsongsinplaylist)
        self.channel.supply(requests.getsongsinplaylists, self.getsongsinplaylists)

        self.autoregisterer = songautoregisterer(self.basedir, self.id, self.channel)
        self.autoregisterer.start()

    def _initdb3(self, cachesize):
        """ initialise database using modern bsddb interface of Python 2.3 and above """
        import bsddb.dbshelve
        import bsddb.db

        # setup db environment
        self.dbenv = bsddb.db.DBEnv()

        # calculate cache parameters for bsddb for cachesize in kB
        # the factor 1000 is taken from zodb.storage.base
        gbytes, bytes = divmod(cachesize*1024, 1024 * 1024 * 1000)
        self.dbenv.set_cachesize(gbytes, bytes)

        if self.songdbbase:
            dbenvdir = os.path.dirname(self.songdbbase)
        else:
            dbenvdir = os.path.dirname(self.dbfile)
            
        self.dbenv.open(dbenvdir,
                        bsddb.db.DB_INIT_MPOOL |
                        # bsddb.db.DB_INIT_LOG |
                        # bsddb.db.DB_INIT_TXN |
                        # bsddb.db.DB_RECOVER |
                        bsddb.db.DB_CREATE |
                        bsddb.db.DB_PRIVATE)

        # setup databases (either in one total or several extra files)
        if self.dbfile:
            songdbfilename = os.path.basename(self.dbfile)
            self.songs = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="songs")
            self.artists = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="artists")
            self.albums = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="albums")
            self.playlists = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="playlists")
            self.genres = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="genres")
            self.years = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="years")
            self.stats = bsddb.dbshelve.open(songdbfilename, dbenv=self.dbenv, dbname="stats")
        else:
            songdbprefix = os.path.basename(self.songdbbase)
            self.songs = bsddb.dbshelve.open(songdbprefix + "_songs.db", dbenv=self.dbenv)
            self.artists = bsddb.dbshelve.open(songdbprefix + "_artists.db", dbenv=self.dbenv)
            self.albums = bsddb.dbshelve.open(songdbprefix + "_albums.db", dbenv=self.dbenv)
            self.playlists = bsddb.dbshelve.open(songdbprefix + "_playlists.db", dbenv=self.dbenv)
            self.genres = bsddb.dbshelve.open(songdbprefix + "_genres.db", dbenv=self.dbenv)
            self.years = bsddb.dbshelve.open(songdbprefix + "_years.db", dbenv=self.dbenv)
            self.stats = bsddb.dbshelve.open(songdbprefix + "_stats.db", dbenv=self.dbenv)

        log.info(_("database %s: type %s, basedir %s, %d songs, %d artists, %d albums, %d genres, %d playlists") %
                 (self.id, "bsddb3", self.basedir, len(self.songs),  len(self.artists),  len(self.albums),
                  len(self.genres), len(self.playlists)))
 
    def _initdb(self):
        """ initialise database using legacy bsddb interface """
        import shelve

        self.songs = shelve.open(self.songdbbase + "_songs.db")
        self.artists = shelve.open(self.songdbbase + "_artists.db")
        self.albums = shelve.open(self.songdbbase + "_albums.db")
        self.playlists = shelve.open(self.songdbbase + "_playlists.db")
        self.genres = shelve.open(self.songdbbase + "_genres.db")
        self.years = shelve.open(self.songdbbase + "_years.db")
        self.stats = shelve.open(self.songdbbase + "_stats.db")

        log.info(_("database %s: type %s, basedir %s, %d songs, %d artists, %d albums, %d genres, %d playlists") %
                 (self.id, "bsddb", self.basedir,  len(self.songs),  len(self.artists),  len(self.albums),
                  len(self.genres), len(self.playlists)))

    def _updatefromversion1to2(self):
        """ update from database version 1 to version 2 """

        log.info(_("updating song database %s from version 1 to version 2") % self.id)

        print _("Updating song database %s from version 1 to version 2:") % self.id,
        print "%d artists..." % len(self.artists),
        for artistid, artist in self.artists.items():
            newalbums = []
            for album in artist.albums:
                newalbum  = self.albums[album].name
                newalbums.append(newalbum)
            artist.albums = newalbums
            self.artists[artistid] = artist

        print "%d genres..." % len(self.genres),
        for genreid, genre in self.genres.items():
            newalbums = []
            for album in genre.albums:
                newalbum = self.albums[album].name
                newalbums.append(newalbum)
            genre.albums = newalbums
            self.genres[genreid] = genre

        print "%d years..." % len(self.years),
        for yearid, year in self.years.items():
            newalbums = []
            for album in year.albums:
                newalbum  = self.albums[album].name
                newalbums.append(newalbum)
            year.albums = newalbums
            self.years[yearid] = year

        print "%d albums..." % len(self.albums),
        for albumid, album in self.albums.items():
            if album.name in self.albums:
                newalbum = self.albums[album.name]
                newalbum.artists.append(album.artist)
                newalbum.songs.extend(album.songs)
                newalbum.genres.extend(album.genres)
                newalbum.years.extend(album.years)
            else:
                newalbum = album
                newalbum.id = album.name
                newalbum.artists = [album.artist]
                del newalbum.artist

            self.albums[newalbum.id] = newalbum
            del self.albums[albumid]
        print

    def run(self):
        """main loop"""
        while not self.done:
            self.channel.process(blocking=1)
        self.close()

    def close(self):
        if self.songdbbase:
            self.songs.close()
            self.artists.close()
            self.albums.close()
            self.genres.close()
            self.years.close()
            self.playlists.close()
            self.stats.close()

    # methods for insertion/update of respective index tables

    def _indexsong_album(self, song):
        """ insert/update album information for song """
        # marker: do we have to write new album information
        changed = 0

        # insert album in songdb, or fetch existent one from songdb
        if not self.albums.has_key(song.album):
            debug("new album: %s\n" % song.album)
            album = dbitem.album(song.album)
            changed = 1
            hub.hub.notify(events.albumaddedtosongdb(self.id, album))
        else:
            album = self.albums[song.album]

        if song.id not in album.songs:
            changed = 1
            album.songs.append(song.id)

        if song.artist not in album.artists:
            changed = 1
            album.artists.append(song.artist)

        if song.genre not in album.genres:
            changed = 1
            album.genres.append(song.genre)

        if song.year not in album.years:
            changed = 1
            album.years.append(song.year)

        if changed:
            self.albums[song.album] = album

    def _unindexsong_album(self, song):
        """ delete album information for song, but do not delete album """
        album = self.albums[song.album]

        album.songs.remove(song.id)

        # list of remaining songs
        osongs = map(self._getsong, album.songs)

        # check whether another song has the same artist
        for osong in osongs:
            if osong.artist==song.artist:
                break
        else:
            album.artists.remove(song.artist)

        # same for genres...
        for osong in osongs:
            if osong.genre==song.genre:
                break
        else:
            album.genres.remove(song.genre)

        # ... and for years
        for osong in osongs:
            if osong.year==song.year:
                break
        else:
            album.years.remove(song.year)

        self.albums[song.album] = album

    def _unindexsong_album_cleanup(self, song):
        """ delete album, if empty """
        if not self.albums[song.album].songs:
            # Note that the artist index has to be updated subsequently!
            del self.albums[song.album]

    def _indexsong_artist(self, song):
        """ insert/update artist information for song """
        changed = 0

        # insert artist in songdb, or fetch existent one from songdb
        if not self.artists.has_key(song.artist):
            debug("new artist: %s\n" % song.artist)
            artist = dbitem.artist(song.artist)
            changed = 1
            hub.hub.notify(events.artistaddedtosongdb(self.id, artist))
        else:
            artist = self.artists[song.artist]

        if song.album not in artist.albums:
            changed = 1
            artist.albums.append(song.album)

        if song.id not in artist.songs:
            changed = 1
            artist.songs.append(song.id)

        if song.genre not in artist.genres:
            changed = 1
            artist.genres.append(song.genre)

        if song.year not in artist.years:
            changed = 1
            artist.years.append(song.year)

        if changed:
            self.allartists = None
            self.artists[song.artist] = artist

    def _unindexsong_artist(self, song):
        """ delete artist information for song, but do not delete artist """
        artist = self.artists[song.artist]

        artist.songs.remove(song.id)

        # list of remaining songs
        osongs = map(self._getsong, artist.songs)

        # check whether album has already been deleted from album index
        if song.album not in self.albums:
            artist.albums.remove(song.album)

        # check whether another song has the same genre
        for osong in osongs:
            if osong.genre==song.genre:
                break
        else:
            artist.genres.remove(song.genre)

        # same for year
        for osong in osongs:
            if osong.year==song.year:
                break
        else:
            artist.years.remove(song.year)

        self.allartists = None
        self.artists[song.artist] = artist

    def _unindexsong_artist_cleanup(self, song):
        """ delete artist, if empty """
        if not self.artists[song.artist].songs:
            # Note that the genre and year indices have to be updated subsequently!
            del self.artists[song.artist]
            self.allartists = None

    def _indexsong_genre(self, song):
        """ insert/update genre information for song """

        changed = 0

        # insert genre in songdb, or fetch existent one from songdb
        if not self.genres.has_key(song.genre):
            debug("new genre: %s\n" % song.genre)
            genre = dbitem.genre(song.genre)
            changed = 1
            # XXX no event
        else:
            genre = self.genres[song.genre]

        if song.id not in genre.songs:
            changed = 1
            genre.songs.append(song.id)

        if song.artist not in genre.artists:
            changed = 1
            genre.artists.append(song.artist)

        if song.album not in genre.albums:
            changed = 1
            genre.albums.append(song.album)

        if changed:
            self.genres[song.genre] = genre

    def _unindexsong_genre(self, song):
        """ delete genre information for song """
        genre = self.genres[song.genre]

        genre.songs.remove(song.id)

        # check if either the artist is no longer present, or if it doesn't
        # contain the genre any longer. In both cases remove artist + album
        # from index
        if song.artist not in self.artists:
            genre.artists.remove(song.artist)
            genre.albums.remove(song.album)
        elif song.genre not in self.artists[song.artist].genres:
            genre.artists.remove(song.artist)
            genre.albums.remove(song.album)
        else:
            # check whether we have to merely remove the album
            if song.album not in self.artists[song.artist].albums:
                genre.albums.remove(song.album)
            else:
                for asong in self._getsongs(song.artist, song.album):
                    if song.genre==asong.genre:
                        break
                else:
                    genre.artists.remove(song.artist)

        # delete genre or write it with new values
        if not genre.artists:
            del self.genres[song.genre]
        else:
            self.genres[song.genre] = genre

    def _indexsong_year(self, song):
        """ insert/update year information for song """

        yearid = str(song.year)
        changed = 0

        # insert year in songdb, or fetch existent one from songdb

        if not self.years.has_key(yearid):
            debug("new year: %s\n" % yearid)
            year = dbitem.year(song.year)
            changed = 1
            # XXX no event
        else:
            year = self.years[yearid]

        if song.id not in year.songs:
            changed = 1
            year.songs.append(song.id)

        if song.artist not in year.artists:
            changed = 1
            year.artists.append(song.artist)

        if song.album not in year.albums:
            changed = 1
            year.albums.append(song.album)

        if changed:
            self.years[yearid] = year

    def _unindexsong_year(self, song):
        """ delete year information for song """
        yearid = str(song.year)
        year = self.years[yearid]

        year.songs.remove(song.id)

        # check if either the artist is no longer present, or if it doesn't
        # contain the year any longer. In both cases remove artist + album
        # from index
        if song.artist not in self.artists:
            year.artists.remove(song.artist)
            year.albums.remove(song.album)
        elif song.year not in self.artists[song.artist].years:
            year.artists.remove(song.artist)
            year.albums.remove(song.album)
        else:
            # check whether we have to merely remove the album
            if song.album not in self.artists[song.artist].albums:
                year.albums.remove(song.album)
            else:
                for asong in self._getsongs(song.artist, song.album):
                    if song.year==asong.year:
                        break
                else:
                    year.artists.remove(song.artist)

        if not year.artists:
            del self.years[yearid]
        else:
            self.years[yearid] = year

    def _indexsong_stats(self, song):
        """ insert/update playing (and creation) statistics for song """
        # some assumptions made:
        # - if lastplayed info is changed for a song, this song has been
        #   played and, thus, gets added to the top of the lastplayed list
        lastadded = self.stats["lastadded"]
        lastplayed = self.stats["lastplayed"]
        topplayed = self.stats["topplayed"]

        if not lastadded or self.songs[lastadded[0]].added<song.added:
            lastadded.insert(0, song.id)
            self.stats["lastadded"] = lastadded[:self.playingstatslength]

        if song.lastplayed:
            if not lastplayed or lastplayed[0][1]<song.lastplayed:
                lastplayed.insert(0, (song.id, song.lastplayed))
                self.stats["lastplayed"] = lastplayed[:self.playingstatslength]

        if song.nrplayed:
            try:
                # XXX why do we need oldsong. Is this correct?
                oldsong = self.songs[topplayed[topplayed.index(song.id)]]
                topplayed.remove(song.id)
            except ValueError:
                pass
            for i in range(len(topplayed)):
                asong = self.songs[topplayed[i]]
                if (asong.nrplayed<song.nrplayed or
                    (asong.nrplayed==song.nrplayed and asong.lastplayed<song.lastplayed)):
                    topplayed.insert(i, song.id)
                    break
            else:
                topplayed.append(song.id)
            self.stats["topplayed"] = topplayed[:self.playingstatslength]

    def _unindexsong_stats(self, song):
        """ delete  playing (and creation) statistics for song """
        # some assumptions made:
        # - if lastplayed info is changed for a song, this song has been
        #   played and, thus, gets added to the top of the lastplayed list
        lastadded = self.stats["lastadded"]
        lastplayed = self.stats["lastplayed"]
        topplayed = self.stats["topplayed"]

        if song.id in lastadded:
            lastadded.remove(song.id)
            self.stats["lastadded"] = lastadded

        newlastplayed = []
        for asongid, alastplayed in lastplayed:
            if song.id != asongid:
                newlastplayed.append((asongid, alastplayed))
        if len(lastplayed) != len(newlastplayed):
            self.stats["lastplayed"] = newlastplayed

        if song.id in topplayed:
            topplayed.remove(song.id)
            self.stats["topplayed"] = topplayed

    def _indexsong(self, song):
        self._indexsong_album(song)
        self._indexsong_artist(song)
        self._indexsong_genre(song)
        self._indexsong_year(song)
        self._indexsong_stats(song)

    def _reindexsong(self, oldsong, newsong):
        if (oldsong.album != newsong.album or
            oldsong.artist != newsong.artist or
            oldsong.genre != newsong.genre or
            oldsong.year != newsong.year):
            # The update process of the album and artist information
            # is split into three parts to prevent an intermediate
            # deletion of artist and/or album (together with its rating
            # information)
            self._unindexsong_album(oldsong)
            self._indexsong_album(newsong)
            self._unindexsong_album_cleanup(oldsong)
            self._unindexsong_artist(oldsong)
            self._indexsong_artist(newsong)
            self._unindexsong_artist_cleanup(oldsong)
            self._unindexsong_genre(oldsong)
            self._indexsong_genre(newsong)
            self._unindexsong_year(oldsong)
            self._indexsong_year(newsong)
        if (oldsong.lastplayed!=newsong.lastplayed or
            oldsong.nrplayed!=newsong.nrplayed):
            self._indexsong_stats(newsong)

    def _queryregistersong(self, path):
        """get song info from database or insert new one"""

        # we assume that the path of the song is the song id.
        # This allows to quickly verify (without reading the song itself)
        # whether we have already registered the song

        # XXX: here, we assume that path==id. Otherwise, we would
        # have to create a song instance, which is also not optimal
        # Probably, the whole song.id wasn't too good in the first place.
        if self.songs.has_key(path):
            song = self.songs[path]
        else:
            debug("new song: %s\n" % path)
            song = dbitem.song(path)
            self.songs[song.id] = song
            self.allsongs = None

        # insert into indices
        self._indexsong(song)

        return song

    def _delsong(self, song):
        """delete song from database"""
        if not self.songs.has_key(song.id):
            raise KeyError
        
        debug("delete song: %s\n" % str(song))

        self._unindexsong_album(song)
        self._unindexsong_album_cleanup(song)
        self._unindexsong_artist(song)
        self._unindexsong_artist_cleanup(song)
        self._unindexsong_genre(song)
        self._unindexsong_year(song)
        self._unindexsong_stats(song)
        del self.songs[song.id]
        if self.allsongs:
            del self.allsongs[self.allsongs.index(song)]

    def _registerplaylist(self, path):
        playlist = dbitem.playlist(path)

        # also try to register songs in playlist
        # delete song, if this fails
        for i in range(len(playlist.songs)):
            try:
                self._queryregistersong(playlist.songs[i])
            except (IOError, OSError):
                del playlist.songs[i]

        # a resulting, non-empty playlist can be written in the database
        if playlist.songs:
            self.playlists[path] = playlist

    def _updatesong(self, song):
        """updates entry of given song"""
        # update song cache if existent
        try:
            # Note that we rely on each song being uniquely specified
            # by its path and the corresponding __cmp__ method of
            # dbitem.song!
            oldsong = self.songs[song.id]
            self.allsongs[self.allsongs.index(oldsong)] = song
        except (ValueError, AttributeError):
            pass

        oldsong = self.songs[song.id]

        self.songs[song.id] = song
        self._reindexsong(oldsong, song)

    def _updatealbum(self, album):
        """updates entry of given album of artist"""
        # XXX: changes of other indices not handled correctly
        self.albums[album.id] = album

    def _updateartist(self, artist):
        """updates entry of given artist"""
        # XXX: changes of other indices not handled correctly

        # update artist cache if existent
        try:
            # Note that we rely on each artist being uniquely specified
            # by its name and the corresponding __cmp__ method of
            # dbitem.artist!
            oldartist = self.artists[artist.name]
            self.allartists[self.allartists.index(oldartist)] = artist
        except (ValueError, AttributeError):
            pass

        self.artists[artist.name] = artist

    def _getsong(self, path):
        """returns song entry with given path"""
        song = self.songs.get(path)
        return song

    def _getalbum(self, album):
        """returns given album"""
        return self.albums[album]

    def _getartist(self, artist):
        """returns given artist"""
        return self.artists.get(artist)

    def _getartists(self, genre=None, year=None):
        """return all stored artists"""
        # use cached value if existent
        if genre is None and year is None:
            if self.allartists:
                artists = self.allartists[:]
                return artists
            else:
                debug("start rebuilding artist cache\n")
                artists = map(self.artists.get, self.artists.keys())
                debug("finished rebuilding artist cache\n")
                newallartists = artists[:]
                debug("saving artist cache\n")
                self.allartists = newallartists
                return artists
        elif genre is not None and year is None:
            return map(self.artists.get, self.genres[genre].artists)
        elif genre is None and year is not None:
            return map(self.artists.get, self.years[str(year)].artists)
        else:
            artists = map(self.artists.get, self.years[str(year)].artists)
            return [artist for artist in artists if artist.genre==genre]

    def _getalbums(self, artist=None, genre=None, year=None):
        """return albums of a given artist and genre

        artist has to be a string. If it is none, all stored
        albums are returned
        """
        if artist is None:
            if genre is None and year is None:
                return map(self.albums.get, self.albums.keys())
            elif genre is not None and year is None:
                return map(self.albums.get, self.genres[genre].albums)
            elif genre is None and year is not None:
                return map(self.albums.get, self.years[str(year)].albums)
            else:
                albums = map(self.albums.get, self.years[str(year)].albums)
                albumsgenre = self.genres[genre].albums
                return [album for album in albums if album.id in albumsgenre]
        else:
            albums = map(self.albums.get, self.artists[artist].albums)
            if genre is not None:
                albumsgenre = self.genres[genre].albums
                albums = [album for album in albums if album.id in albumsgenre]
            if year is not None:
                albumsyear = self.years[str(year)].albums
                albums = [album for album in albums if album.id in albumsyear]
            return albums

    def _getsongs(self, artist=None, album=None, genre=None, year=None):
        """ returns song of given artist, album, genre and year

        All values either have to be strings (resp. an integer for
        year) or None, in which case they are ignored.
        """

        if artist is None and album is None and genre is None and year is None:
            # return all songs in songdb
            # use cached value if existent
            if self.allsongs:
                songs = self.allsongs[:]
                return songs
            else:
                songs = map(self.songs.get, self.songs.keys())
                newallsongs = songs[:]
                self.allsongs = newallsongs
                return songs

        if genre is None and year is None:
            if album is None:
                # return all songs of a given artist
                keys = self._getartist(artist).songs
                songs = map(self.songs.get, keys)
                return songs
            elif artist is None:
                # return all songs which are on albums with the given name
                albums = [ aalbum for aalbum in self._getalbums()
                           if aalbum.name==album ]
                songs = []
                for aalbum in albums:
                    keys = self._getalbum(album).songs
                    songs.extend(map(self.songs.get, keys))
                return songs
            else:
                # return all songs on a given album
                keys = self._getalbum(album).songs
                songs = map(self.songs.get, keys)
                return [song for song in songs if song.artist==artist]
        else:
            # genre or year is specified
            if genre is not None and year is None:
                # only genre specified

                if artist is None and album is None:
                    songs = map(self.songs.get, self.genres[genre].songs)
                    return songs
                else:
                    songs = self._getsongs(artist=artist, album=album)
                    return [song for song in songs if song.genre==genre]
            elif genre is None and year is not None:
                # only year specified
                if artist is None and album is None:
                    songs = map(self.songs.get, self.years[str(year)].songs)
                    return songs
                else:
                    songs = self._getsongs(artist=artist, album=album)
                    return [song for song in songs if song.year==year]
            else:
                # both year and genre specified
                if artist is None and album is None:
                    songs = map(self.songs.get, self.years[str(year)].songs)
                    return [song for song in songs if song.genre==genre]
                else:
                    songs = self._getsongs(artist=artist, album=album)
                    return [song for song in songs if song.year==year and song.genre==genre]

    def _getgenres(self):
        """return all stored genres"""
        keys = self.genres.keys()
        genres = map(self.genres.get, keys)
        return genres

    def _getyears(self):
        """return all stored years"""
        keys = self.years.keys()
        years = map(self.years.get, keys)
        return years

    def _getlastplayedsongs(self):
        """return the last played songs together with the corresponding playing time"""
        return [(self.songs[songid], playingtime) for songid, playingtime in self.stats["lastplayed"]]

    def _gettopplayedsongs(self):
        """return the top played songs"""
        keys = self.stats["topplayed"]
        return map(self.songs.get, keys)

    def _getlastaddedsongs(self):
        """return the last played songs"""
        keys = self.stats["lastadded"]
        return map(self.songs.get, keys)

    def _getplaylist(self, path):
        """returns playlist entry with given path"""
        return self.playlists.get(path)

    def _getplaylists(self):
        keys = self.playlists.keys()
        return map(self._getplaylist, keys)

    def _getsongsinplaylist(self, path):
        playlist = self._getplaylist(path)
        result = []
        for id in playlist.songs:
            try:
                result.append(self.songs[id])
            except KeyError:
                pass
        return result

    def _getsongsinplaylists(self):
        playlists = self._getplaylists()
        songs = []
        for playlist in playlists:
            songs.extend(self._getsongsinplaylist(playlist.path))
        return songs

    def _genrandomchoice(self, songs):
        """ returns random selection of songs up to the maximal length
        configured. Note that this method changes as a side-effect the
        parameter songs"""

        # choose item, avoiding duplicates. Stop after a predefined
        # total length (in seconds). Take rating of songs/albums/artists
        # into account
        length = 0
        result = []
        for i in range(len(songs)):
            # Simple heuristic algorithm to consider song ratings
            # for random selection. Certainly not optimal!
            while 1:
                nr = random.randrange(0, len(songs))
                aitem = songs[nr]
                rating = (aitem.rating or
                          self._getalbum(aitem.album).rating or
                          self._getartist(aitem.artist).rating or
                          3 )
                if aitem.lastplayed:
                    last = max(0, int((time.time()-aitem.lastplayed)/60))
                    rating -= 2*math.exp(-last/(60.0*60*24))
                if random.random()>=2.0**(-rating) or len(songs)==1:
                    break

            del songs[nr]
            length += aitem.length
            result.append(aitem)
            if length>config.general.randominsertlength:
                break
        return result

    def _selectrandom(self, request, songs):
        """ wrapper function which restrict songs to a random selection
        if specified by the caller"""
        if request.random:
            return self._genrandomchoice(songs)
        else:
            return songs

    def registerdirtree(self, dir):
        for name in os.listdir(dir):
            path = os.path.join(dir, name)
            if os.access(path, os.R_OK):
                if os.path.isdir(path):
                    try: self.registerdirtree(path)
                    except (IOError, OSError): pass
                elif name.endswith(".mp3"):
                    try: self._queryregistersong(path)
                    except (IOError, OSError): pass
                elif name.endswith(".ogg") and dbitem.oggsupport:
                    try: self._queryregistersong(path)
                    except (IOError, OSError): pass
                elif name.endswith(".m3u"):
                    try: self._registerplaylist(path)
                    except (IOError, OSError): pass

    # event handlers

    def updatesong(self, event):
        if event.songdbid==self.id:
            try:
                self._updatesong(event.song)
            except KeyError:
                pass

    def delsong(self, event):
        if event.songdbid==self.id:
            try:
                self._delsong(event.song)
            except KeyError:
                pass

    def updatealbum(self, event):
        if event.songdbid==self.id:
            try:
                self._updatealbum(event.album)
            except KeyError:
                pass

    def updateartist(self, event):
        if event.songdbid==self.id:
            try:
                self._updateartist(event.artist)
            except KeyError:
                pass

    def registersongs(self, event):
        if event.songdbid==self.id:
            for path in event.paths:
                try: self._queryregistersong(path)
                except (IOError, OSError): pass

    def registerplaylists(self, event):
        if event.songdbid==self.id:
            for path in event.paths:
                try: self._registerplaylist(path)
                except (IOError, OSError): pass

    def quit(self, event):
        self.done = 1

    # request handlers

    def getdatabaseinfo(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return ("local", self.basedir)

    def getnumberofsongs(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return len(self.songs)

    def getnumberofdecades(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        decades = []
        for year in self.years.keys():
            if year!="None" and year!="0" and int(year)/10*10 not in decades:
                decades.append(int(year)/10*10)
            elif year=="None" and year not in decades:
                decades.append(None)
        return len(decades)

    def getnumberofgenres(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        # XXX why does len(self.genres) not work???
        # return len(self.genres)
        return len(self.genres.keys())

    def getnumberofalbums(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        # see above
        return len(self.albums.keys())

    def queryregistersong(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._queryregistersong(request.path)

    def getsong(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        try:
            return self._getsong(request.path)
        except KeyError:
            return None

    def getsongs(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        if request.decade is None:
            try:
                songs = self._getsongs(request.artist, request.album, request.genre, request.year)
            except (KeyError, AttributeError):
                return []
        else:
            songs = []
            for year in range(request.decade, request.decade+10):
                try:
                    songs.extend(self._getsongs(request.artist, request.album, request.genre, year))
                except (KeyError, AttributeError):
                    pass
        return self._selectrandom(request, songs)

    def getartists(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        debug("replying to getartists request\n")
        if request.decade is None:
            try:
                return self._getartists(request.genre, request.year)
            except KeyError:
                return []
        else:
            artists = []
            for year in range(request.decade, request.decade+10):
                try:
                    newartists = self._getartists(genre=request.genre, year=year)
                    oldartistids = map(lambda a:a.id, artists)
                    for newartist in newartists:
                        if newartist.id not in oldartistids:
                            artists.append(newartist)
                except KeyError:
                    pass
            return artists

    def getartist(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        try:
            return self._getartist(request.artist)
        except KeyError:
            return None

    def getalbums(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        if request.decade is None:
            try:
                return self._getalbums(request.artist, request.genre, request.year)
            except KeyError:
                return []
        else:
            albums = []
            for year in range(request.decade, request.decade+10):
                try:
                    newalbums = self._getalbums(request.artist, genre=request.genre, year=year)
                    oldalbums = map(lambda a:a.id, albums)
                    for newalbum in newalbums:
                        if newalbum.id not in oldalbums:
                            albums.append(newalbum)
                except KeyError:
                    pass
            return albums

    def getalbum(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        try:
            return self._getalbum(request.album)
        except KeyError:
            return None

    def getgenres(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._getgenres()

    def getyears(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._getyears()

    def getdecades(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        # XXX: It would be more efficient to write a songdb method
        # returning min(years), max(years).
        years = [year.year for year in self._getyears()]
        decades = []
        if years:
            for year in years:
                if year and year/10*10 not in decades:
                    decades.append(year/10*10)
                elif year is None and year not in decades:
                    decades.append(None)

        return decades

    def getlastplayedsongs(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        # Due to the different structure of the result compared to the
        # other request, namely a list of tuples (song, playingtime), we
        # do not support random song selection here.
        return self._getlastplayedsongs()

    def gettopplayedsongs(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._selectrandom(request, self._gettopplayedsongs())

    def getlastaddedsongs(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._selectrandom(request, self._getlastaddedsongs())

    def getplaylist(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._getplaylist(request.path)

    def getplaylists(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._getplaylists()

    def getsongsinplaylist(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._selectrandom(request, self._getsongsinplaylist(request.path))

    def getsongsinplaylists(self, request):
        if self.id!=request.songdbid:
            raise hub.DenyRequest
        return self._selectrandom(request, self._getsongsinplaylists())

#
# thread for automatic registering and rescanning of songs in database
#

class songautoregisterer(threading.Thread):

    def __init__(self, basedir, songdbid, songdbchannel):
        threading.Thread.__init__(self)
        self.basedir = basedir
        self.songdbid = songdbid
        self.songdbchannel = songdbchannel
        self.done = 0

        # as independent thread, we want our own event and request channel
        self.channel = hub.hub.newchannel()
        self.channel.subscribe(events.autoregistersongs, self.autoregistersongs)
        self.channel.subscribe(events.rescansongs, self.rescansongs)
        self.channel.subscribe(events.quit, self.quit)

        # we are only servants of higher powers
        self.setDaemon(1)

    def _throttlednotify(self, event):
        # dirty trick to avoid overfilling of songdbchannel
        # due to register events
        # Such an overfilling causes, for instance, long delays
        # during the PyTone shutdown
        while self.songdbchannel.queue.qsize()>0:
            time.sleep(0.2)
        hub.hub.notify(event, -100)

    def registerdirtree(self, dir):
        self.channel.process()
        if self.done: return
        paths = []
        for name in os.listdir(dir):
            path = os.path.join(dir, name)
            if os.access(path, os.R_OK):
                if os.path.isdir(path):
                    try: self.registerdirtree(path)
                    except (IOError, OSError): pass
                elif name.endswith(".mp3"):
                    paths.append(path)
                elif name.endswith(".ogg") and dbitem.oggsupport:
                    paths.append(path)
                elif name.endswith(".m3u"):
                    self._throttlednotify(events.registerplaylists(self.songdbid, [path]))
        if paths:
            self._throttlednotify(events.registersongs(self.songdbid, paths))

    def run(self):
        while not self.done:
            self.channel.process(blocking=1)

    #
    # event handler
    #
    
    def autoregistersongs(self, event):
        if self.songdbid == event.songdbid:
            log.info(_("database %s: scanning for songs in %s") % (self.songdbid, self.basedir))
                                                                  
            start = time.time()
            self.registerdirtree(self.basedir)
            
            log.info(_("database %s: finished scanning for songs in %s") % (self.songdbid, self.basedir))

    def rescansongs(self, event):
        if self.songdbid == event.songdbid:
            log.info(_("database %s: rescanning %d songs") % (self.songdbid, len(event.songs)))

            for song in event.songs:
                song.rescan()
                while self.songdbchannel.queue.qsize()>0:
                    time.sleep(0.1)
            log.info(_("database %s: finished rescanning %d songs") % (self.songdbid, len(event.songs)))

    def quit(self, event):
        self.done = 1
