/***************************************************************************
 $RCSfile: accountview.cpp,v $
                             -------------------
    cvs         : $Id: accountview.cpp,v 1.28 2005/08/08 14:23:08 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "accountview.h"
#include "transactionwindow.h"
#include <qbanking/qbpickstartdate.h>
#include "editaccount.h"

#include <aqbanking/jobgetbalance.h>
#include <aqbanking/jobgettransactions.h>
#include <aqbanking/jobgetstandingorders.h>
#include <aqbanking/jobgetdatedtransfers.h>

#include <kbanking.h>

#include <qevent.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qlayout.h>

#include <gwenhywfar/debug.h>

#ifdef WIN32
# define strcasecmp stricmp
#endif


#define BUTTON_WIDTH 110


AccountView::AccountView(KBanking *kb,
                         QWidget* parent,
                         const char* name,
                         WFlags fl)
:AccountViewUi(parent, name, fl), _app(kb) {
  assert(kb);

  // Manually create and add layout here because the .ui-generated
  // QGroupBox doesn't have one.
  accountBox->setColumnLayout(0, Qt::Vertical );
  QBoxLayout *accountBoxLayout = new QHBoxLayout( accountBox->layout() );
  accountBoxLayout->setAlignment( Qt::AlignTop );

  _accList=new AccountListView((QWidget*)accountBox, name);
  accountBoxLayout->addWidget(_accList);

  QObject::connect((QObject*)kb->flagStaff(), SIGNAL(signalAccountsUpdated()),
                   this, SLOT(slotUpdated()));

  QObject::connect((QObject*)detailsButton, SIGNAL(clicked()),
                   this, SLOT(slotDetails()));
  QObject::connect((QObject*)transactionsButton, SIGNAL(clicked()),
                   this, SLOT(slotTransactions()));
  QObject::connect((QObject*)updateButton, SIGNAL(clicked()),
                   this, SLOT(slotUpdate()));

}



AccountView::~AccountView(){
}



bool AccountView::init(){
  GWEN_DB_NODE *db;

  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                      "gui/views/accountview/dynamic");
  if (db) {
    int i, j;
    const char *p;

    p=GWEN_DB_GetCharValue(db, "sortOrder", 0, "ascending");
    if (p) {
      if (strcasecmp(p, "ascending")==0)
        _accList->setSortOrder(Qt::Ascending);
      else
        if (strcasecmp(p, "descending")==0)
          _accList->setSortOrder(Qt::Descending);
    }
    i=GWEN_DB_GetIntValue(db, "sortColumn", 0, -1);
    if (i!=-1)
      _accList->setSortColumn(i);

    /* found settings */
    for (i=0; i<_accList->columns(); i++) {
      _accList->setColumnWidthMode(i, QListView::Manual);
      j=GWEN_DB_GetIntValue(db, "columns", i, -1);
      if (j!=-1)
        _accList->setColumnWidth(i, j);
    } /* for */
  } /* if settings */

  _accList->addAccounts(_app->getAppAccounts());

  return true;
}



bool AccountView::fini(){
  GWEN_DB_NODE *db;
  int i, j;

  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db,
                      GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      "gui/views/accountview/dynamic");

  switch(_accList->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", _accList->sortColumn());

  for (i=0; i<_accList->columns(); i++) {
    j=_accList->columnWidth(i);
    GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  return true;
}



void AccountView::slotDetails(){
  Account *acc;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(0,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), 0, 0, 0);
    return;
  }
  else {
    EditAccount w(_app, acc, false, this, "EditAccount", true);

    if (w.exec()==QDialog::Accepted) {
      DBG_ERROR(0, "Accepted account changed");
    }
    else {
      DBG_ERROR(0, "Rejected account changes");
    }

  }
}



void AccountView::slotTransactions(){
  Account *acc;
  TransactionWindow *tv;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(0,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), 0, 0, 0);
    return;
  }

  tv=new TransactionWindow(acc, 0, "TransactionView",
                           Qt::WDestructiveClose);
  tv->setCaption(tr("Transactions"));
  if (!tv->init()) {
    QMessageBox::critical(0,
                          tr("Internal Error"),
                          tr("Could not initialize transaction view."),
                          tr("Dismiss"), 0, 0, 0);
    delete tv;
    return;
  }
  tv->show();
}



void AccountView::slotUpdate(){
  Account *acc;
  int rv;
  AB_ACCOUNT *ba;
  AB_JOB *job;
  const GWEN_TIME *ti;
  AB_JOB_LIST2 *jl;
  GWEN_DB_NODE *db;
  GWEN_DB_NODE *dbAccountSettings;
  int oks=0;
  int tries=0;

  /* get my account */
  acc=_accList->getCurrentAccount();
  if (!acc) {
    DBG_DEBUG(0, "No account selected");
    QMessageBox::warning(0,
                         tr("No Selection"),
                         tr("Please select an account first."),
                         tr("Dismiss"), 0, 0, 0);
    return;
  }

  dbAccountSettings=_app->getAppData();
  assert(dbAccountSettings);
  dbAccountSettings=GWEN_DB_GetGroup(dbAccountSettings,
                                     GWEN_DB_FLAGS_DEFAULT,
                                     "settings/accounts/");
  assert(dbAccountSettings);
  dbAccountSettings=GWEN_DB_GetGroup(dbAccountSettings,
                                     GWEN_DB_FLAGS_DEFAULT,
                                     acc->getCountry().c_str());
  assert(dbAccountSettings);
  dbAccountSettings=GWEN_DB_GetGroup(dbAccountSettings,
                                     GWEN_DB_FLAGS_DEFAULT,
                                     acc->getBankCode().c_str());
  assert(dbAccountSettings);
  dbAccountSettings=GWEN_DB_GetGroup(dbAccountSettings,
                                     GWEN_DB_FLAGS_DEFAULT,
                                     acc->getAccountNumber().c_str());
  assert(dbAccountSettings);

  /* get corresponding AB_ACCOUNT */
  ba=_app->getAccount(acc->getBankingId());
  if (!ba) {
    DBG_ERROR(0, "Account not available");
    QMessageBox::critical(0,
                          tr("Account Not Available"),
                          tr("The account you requested is not available\n"
                             "with any backend."),
                          tr("Dismiss"), 0, 0, 0);
    return;
  }

  /* update transactions */
  if (acc->getOptionUpdateTransactions()) {
    tries++;
    job=AB_JobGetTransactions_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetTransactions\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      ti=acc->getLastTransactionDate();
      if (ti) {
        AB_JobGetTransactions_SetFromTime(job, ti);
      }
      else {
        int days;
        int year, month, day;
        QDate qd;

        days=AB_JobGetTransactions_GetMaxStoreDays(job);
        if (days>0) {
          GWEN_TIME *ti1;
          GWEN_TIME *ti2;

          ti1=GWEN_CurrentTime();
          ti2=GWEN_Time_fromSeconds(GWEN_Time_Seconds(ti1)-(60*60*24*days));
          GWEN_Time_free(ti1);
          ti1=ti2;

          if (GWEN_Time_GetBrokenDownDate(ti1, &day, &month, &year)) {
            DBG_ERROR(0, "Bad date");
            qd=QDate();
          }
          else
            qd=QDate(year, month+1, day);
          GWEN_Time_free(ti1);
        }

        QBPickStartDate psd(qd, QDate(), 3, this, "PickStartDate", true);
        if (psd.exec()!=QDialog::Accepted) {
          AB_Job_free(job);
          return;
        }
        qd=psd.getDate();
        if (qd.isValid()) {
          GWEN_TIME *ti1;

          ti1=GWEN_Time_new(qd.year(), qd.month()-1, qd.day(), 0, 0, 0, 0);
          AB_JobGetTransactions_SetFromTime(job, ti1);
          GWEN_Time_free(ti1);
        }
      }

      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
        DBG_NOTICE(0, "Error %d", rv);
        return;
      }
      oks++;
    }
  }

  /* update balance */
  if (acc->getOptionUpdateBalance()) {
    tries++;
    job=AB_JobGetBalance_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetBalance\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
        DBG_NOTICE(0, "Error %d", rv);
        return;
      }
      oks++;
    }
  }

  /* update standing orders */
  if (acc->getOptionUpdateStandingOrders()) {
    tries++;
    job=AB_JobGetStandingOrders_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetStandingOrders\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      GWEN_DB_SetIntValue(dbAccountSettings, GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "clearStandingOrdersOnNextImport", 1);
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
        DBG_NOTICE(0, "Error %d", rv);
        return;
      }
      oks++;
    }
  }

  /* update dated transfers */
  if (acc->getOptionUpdateDatedTransfers()) {
    tries++;
    job=AB_JobGetDatedTransfers_new(ba);
    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_ERROR(0, "Job \"GetDatedTransfers\" is not available (%d)", rv);
      AB_Job_free(job);
    }
    else {
      GWEN_DB_SetIntValue(dbAccountSettings, GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "clearDatedTransfersOnNextImport", 1);
      DBG_DEBUG(0, "Enqueuing job");
      rv=_app->enqueueJob(job);
      AB_Job_free(job);
      if (rv) {
        DBG_NOTICE(0, "Error %d", rv);
        return;
      }
      oks++;
    }
  }

  /* enqueue pending jobs for this account */
  if (GWEN_DB_GetIntValue(dbAccountSettings, "updatePending", 0, 1)) {
    tries++;
    jl=AB_Banking_GetPendingJobs(_app->getCInterface());
    if (jl) {
      AB_JOB *j;
      AB_JOB_LIST2_ITERATOR *it;

      it=AB_Job_List2_First(jl);
      assert(it);

      j=AB_Job_List2Iterator_Data(it);
      assert(j);
      while(j) {
        if (strcasecmp(AB_Job_GetCreatedBy(j),
                       _app->getAppName())==0) {
          if (AB_Job_GetAccount(j)==ba) {
            DBG_INFO(0, "Enqueueing pending job");
            rv=_app->enqueueJob(j);
            if (rv) {
              DBG_ERROR(AQBANKING_LOGDOMAIN,
                        "Error enqueueing job %d (%d)",
                        AB_Job_GetJobId(j), rv);
            }
            else
              oks++;
          }
        }
        j=AB_Job_List2Iterator_Next(it);
      } /* while */
      AB_Job_List2Iterator_free(it);
      AB_Job_List2_FreeAll(jl);
    } /* if pending jobs */
  }

  if (!oks) {
    if (!tries)
      QMessageBox::critical(0,
                            tr("No Job enqueued"),
                            tr("<qt>"
                               "No update jobs have been enqueued because "
                               "all those jobs are disabled with this "
                               "account.\n"
                               "</qt>"),
                            tr("Dismiss"), 0, 0, 0);
    else
      QMessageBox::critical(0,
                            tr("No Job enqueued"),
                            tr("<qt>"
                               "No update jobs is supported by the "
                               "bank/account/backend.\n"
                               "</qt>"),
                            tr("Dismiss"), 0, 0, 0);
    _app->statusMessage(tr("No jobs added to outbox"));
    return;
  }

  /* eventually show message */
  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_DEFAULT,
                      "gui/views/accountview/static");
  if (GWEN_DB_GetIntValue(db, "showUpdates", 0, 1)) {
    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>Update jobs have been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs.</qt>"),
                             tr("Dismiss"), 0, 0, 0);
    GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "showUpdates", 0);
  }
  _app->statusMessage(tr("Jobs added to outbox"));
}


void AccountView::slotUpdated() {
  DBG_DEBUG(0, "Account view updated");
  _accList->clear();
  _accList->addAccounts(_app->getAppAccounts());
}



