/***************************************************************************
 $RCSfile: transferview.cpp,v $
                             -------------------
    cvs         : $Id: transferview.cpp,v 1.13 2005/06/13 20:00:08 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "transferview.h"
#include "edittransaction.h"
#include "editeutransaction.h"
#include "kbanking.h"


#include <aqbanking/jobsingletransfer.h>
#include <aqbanking/jobsingledebitnote.h>
#include <aqbanking/jobeutransfer.h>

#ifdef WIN32
# define strcasecmp stricmp
#endif

#include <qevent.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qlayout.h>

#include <gwenhywfar/debug.h>



#define BUTTON_WIDTH 110


TransferView::TransferView(KBanking *app,
                           QWidget* parent,
                           const char* name,
                           WFlags fl)
:TransferViewUi(parent, name, fl), _app(app) {
  assert(app);

  // Manually create and add layout here because the .ui-generated
  // QGroupBox doesn't have one.
  transferBox->setColumnLayout(0, Qt::Vertical );
  QBoxLayout *transferBoxLayout = new QHBoxLayout( transferBox->layout() );
  transferBoxLayout->setAlignment( Qt::AlignTop );

  _xaList=new TransferListView((QWidget*)transferBox, name);
  transferBoxLayout->addWidget(_xaList);

  QObject::connect((QObject*)app->flagStaff(),
                   SIGNAL(signalAccountsUpdated()),
                   this, SLOT(slotUpdated()));

  QObject::connect((QObject*)app->flagStaff(),
                   SIGNAL(signalTransfersUpdated()),
                   this, SLOT(slotUpdated()));

  QObject::connect((QObject*)detailsButton, SIGNAL(clicked()),
                   this, SLOT(slotDetails()));
  QObject::connect((QObject*)newTransferButton, SIGNAL(clicked()),
                   this, SLOT(slotNewTransfer()));
  QObject::connect((QObject*)newDebitNoteButton, SIGNAL(clicked()),
                   this, SLOT(slotNewDebitNote()));
  QObject::connect((QObject*)newEuTransferButton, SIGNAL(clicked()),
                   this, SLOT(slotNewEuTransfer()));
  QObject::connect((QObject*)repeatButton, SIGNAL(clicked()),
                   this, SLOT(slotRepeat()));
}



TransferView::~TransferView(){
}


bool TransferView::init(){
  GWEN_DB_NODE *db;
  std::string s;
  int i, j;

  db=_app->getAppData();
  if (db) {
    db=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                        "gui/views/transferview");
    if (db) {
      int w;
      int h;
      const char *p;

      /* found settings */
      p=GWEN_DB_GetCharValue(db, "sortOrder", 0, "ascending");
      if (p) {
        if (strcasecmp(p, "ascending")==0)
          _xaList->setSortOrder(Qt::Ascending);
        else
          if (strcasecmp(p, "descending")==0)
            _xaList->setSortOrder(Qt::Descending);
      }
      i=GWEN_DB_GetIntValue(db, "sortColumn", 0, -1);
      if (i!=-1)
        _xaList->setSortColumn(i);
      w=GWEN_DB_GetIntValue(db, "width", 0, 640);
      h=GWEN_DB_GetIntValue(db, "height", 0, 480);
      resize(w, h);
      for (i=0; i<_xaList->columns(); i++) {
        _xaList->setColumnWidthMode(i, QListView::Manual);
        j=GWEN_DB_GetIntValue(db, "columns", i, -1);
        if (j!=-1)
          _xaList->setColumnWidth(i, j);
      } /* for */
    } // if transaction view setting
  } // if application settings
  slotUpdated();

  return true;
}



bool TransferView::fini(){
  GWEN_DB_NODE *db;
  int i, j;
  std::string s;

  db=_app->getAppData();
  assert(db);
  db=GWEN_DB_GetGroup(db,
                      GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      "gui/views/transferview");
  assert(db);
  db=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                      s.c_str());
  assert(db);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "width", width());
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "height", height());

  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                      "sortColumn", _xaList->sortColumn());
  switch(_xaList->sortOrder()) {
  case Qt::Ascending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "ascending");
    break;
  case Qt::Descending:
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT,
                         "sortOrder", "descending");
    break;
  default:
    break;
  }

  for (i=0; i<_xaList->columns(); i++) {
    j=_xaList->columnWidth(i);
    GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_DEFAULT,
                        "columns", j);
  } /* for */

  return true;
}



void TransferView::closeEvent(QCloseEvent *e){
  if (!fini()) {
    DBG_ERROR(0, "Could not deinit transaction view");
  }
  e->accept();
}



void TransferView::slotUpdated() {
  _xaList->clear();
  _xaList->addTransfers(_app->getTransfers());
}



void TransferView::slotDetails() {
}



AB_JOB *TransferView::_newTransferJob(Transfer::TransferType tt,
				      RefPointer<Transfer> tmpl){
  RefPointer<Transfer> t;
  bool b;
  int rv;
  AB_JOB *job=0;

  if (tmpl.isValid())
    t=new Transfer(tmpl.ref());
  else {
    t=new Transfer(tt);
    if (tt==Transfer::TransferType_DebitNote)
      t.ref().setTextKey(5);
    else
      t.ref().setTextKey(51);
  }
  t.ref().setTransactionId(_app->getNextUniqueId());

  while(1) {
    GWEN_DB_NODE *dbT;
    int bad=false;
    AB_BANKINFO_CHECKRESULT res;
    int i;
    AB_ACCOUNT *ba;
    Account *acc;

    switch(tt) {
    case Transfer::TransferType_Simple:
      b=EditTransaction::editTransfer(_app, t,
                                      tr("New Transfer"),
                                      this, true);
      break;
    case Transfer::TransferType_DebitNote:
      b=EditTransaction::editDebitNote(_app, t,
                                       tr("New Debit Note"),
                                       this, true);
      break;
    case Transfer::TransferType_EuTransfer:
      b=EditEuTransaction::editEuTransfer(_app, t,
                                          tr("New Euro Transfer"),
                                          this, true);
      break;
    default:
      b=false;
    }

    if (!b) {
      DBG_INFO(0, "User aborted");
      AB_Job_free(job);
      return 0;
    }

    // dump transaction
    DBG_NOTICE(0, "Accepted");
    dbT=GWEN_DB_Group_new("Transaction");
    if (!t.ref().toDb(dbT)) {
      DBG_ERROR(0, "Could not write DB");
    }
    else {
#ifndef WIN32
	  // In MSVC this function crashes
      GWEN_DB_Dump(dbT, stderr, 2);
#endif
    }
    GWEN_DB_Group_free(dbT);


    // get account for this new transaction
    acc=_app->findAccount(t.ref().getLocalBankCode().c_str(),
                          t.ref().getLocalAccountNumber().c_str());
    assert(acc);
    ba=_app->getAccount(acc->getBankingId());
    if (!ba) {
      DBG_ERROR(0, "Account not available");
      QMessageBox::critical(0,
                            tr("Account Not Available"),
                            tr("The account you requested is not "
                               "available\n"
                               "with any backend."),
                            tr("Dismiss"), 0, 0, 0);
      return 0;
    }

    // create job
    switch(tt) {
    case Transfer::TransferType_Simple:
      job=AB_JobSingleTransfer_new(ba);
      break;
    case Transfer::TransferType_DebitNote:
      job=AB_JobSingleDebitNote_new(ba);
      break;
    case Transfer::TransferType_EuTransfer:
      job=AB_JobEuTransfer_new(ba);
      break;
    default:
      return 0;
    }

    rv=AB_Job_CheckAvailability(job);
    if (rv) {
      DBG_NOTICE(0, "Job is not available (%d)", rv);
      AB_Job_free(job);
      QMessageBox::critical(0,
                            tr("Job Not Available"),
                            tr("The job you requested is not "
                               "available with\n"
                               "the backend which handles this account.\n"),
                            tr("Dismiss"), 0, 0, 0);
      AB_Job_free(job);
      return 0;
    }

    res=AB_Banking_CheckAccount(_app->getCInterface(),
                                t.ref().getRemoteCountry().c_str(),
                                0,
                                t.ref().getRemoteBankCode().c_str(),
                                t.ref().getRemoteAccountNumber().c_str());
    switch(res) {
    case AB_BankInfoCheckResult_NotOk:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(0,
                             tr("Warning"),
                             tr("<qt>"
                                "<p>"
                                "The given bank code and account number "
                                "combination is invalid."
                                "</p>"
                                "Do you want to review the transfer?"
                                "<qt>"
                               ),
                             tr("Yes"), tr("No"), tr("Abort"));
      if (i==0)
        bad=true;
      else if (i!=1) {
        AB_Job_free(job);
        return 0;
      }
      break;

    case AB_BankInfoCheckResult_UnknownBank:
      DBG_INFO(0, "Account check result: %d", res);
      i=QMessageBox::warning(0,
                             tr("Warning"),
                             tr("<qt>"
                                "<p>"
                                "The given bank code is invalid."
                                "</p>"
                                "Do you want to review the transfer?"
                                "<qt>"
                               ),
                             tr("Yes"), tr("No"), tr("Abort"));
      if (i==0)
        bad=true;
      else if (i!=1) {
        AB_Job_free(job);
        return 0;
      }
      break;

    case AB_BankInfoCheckResult_UnknownResult:
      DBG_INFO(0, "Nothing known about account");
      break;
    case AB_BankInfoCheckResult_Ok:
      DBG_INFO(0, "Bank code/account number combination is ok");
      break;
    default:
      break;
    }

    if (!bad) {
      AB_TRANSACTION *abt;

      abt=t.ref().toBankingTransaction();
      assert(abt);

      switch(tt) {
      case Transfer::TransferType_Simple:
        rv=AB_JobSingleTransfer_SetTransaction(job, abt);
        break;
      case Transfer::TransferType_DebitNote:
        rv=AB_JobSingleDebitNote_SetTransaction(job, abt);
        break;
      case Transfer::TransferType_EuTransfer:
        rv=AB_JobEuTransfer_SetTransaction(job, abt);
        break;
      default:
        AB_Transaction_free(abt);
        AB_Job_free(job);
        return 0;
      }

      if (rv) {
        DBG_NOTICE(0, "Could not set transfer (%d)", rv);
        if (QMessageBox::critical(0,
                                  tr("Bad Transfer Data"),
                                  tr("<qt>"
                                     "<p>"
                                     "There seems to be some errors "
                                     "in the transfer data."
                                     "</p>"
                                     "<p>"
                                     "Do you want to review the transfer?"
                                     "</p>"
                                     "<qt>"
                                    ),
                                  tr("Yes"), tr("Abort"), 0, 0)!=0){
          AB_Transaction_free(abt);
          AB_Job_free(job);
          return 0;
        }
        bad=true;
      }
      AB_Transaction_free(abt);
    } // if ok

    if (!bad) {
      break;
    }
    AB_Job_free(job);
    job=0;
  } // while

  return job;
}



void TransferView::_newTransfer(RefPointer<Transfer> tmpl) {
  AB_JOB *job;

  job=_newTransferJob(Transfer::TransferType_Simple, tmpl);
  if (job) {
    int rv;
    const AB_TRANSACTION *bat;

    DBG_NOTICE(0, "Enqueuing job");
    rv=_app->enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return;
    }
    bat=AB_JobSingleTransfer_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_Simple);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      _app->addTransfer(t);
    }
    AB_Job_free(job);
    // update transfer list
    slotUpdated();

    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>"
                                "A transfer job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             tr("Dismiss"), 0, 0, 0);
  }

}




void TransferView::slotNewTransfer() {
  _newTransfer();
}



void TransferView::_newDebitNote(RefPointer<Transfer> tmpl) {
  AB_JOB *job;

  job=_newTransferJob(Transfer::TransferType_DebitNote, tmpl);
  if (job) {
    int rv;
    const AB_TRANSACTION *bat;

    DBG_NOTICE(0, "Enqueuing job");
    rv=_app->enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return;
    }
    bat=AB_JobSingleDebitNote_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_DebitNote);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      _app->addTransfer(t);
    }
    AB_Job_free(job);
    // update transfer list
    slotUpdated();

    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>"
                                "A debut note job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             tr("Dismiss"), 0, 0, 0);
  }
}



void TransferView::slotNewDebitNote() {
  _newDebitNote();
}



void TransferView::_newEuTransfer(RefPointer<Transfer> tmpl) {
  AB_JOB *job;

  job=_newTransferJob(Transfer::TransferType_EuTransfer, tmpl);
  if (job) {
    int rv;
    const AB_TRANSACTION *bat;

    DBG_NOTICE(0, "Enqueuing job");
    rv=_app->enqueueJob(job);
    if (rv) {
      DBG_NOTICE(0, "Error %d", rv);
      AB_Job_free(job);
      return;
    }
    bat=AB_JobEuTransfer_GetTransaction(job);
    if (bat) {
      RefPointer<Transfer> t;
      GWEN_TIME *ti;

      t=new Transfer(bat, Transfer::TransferType_EuTransfer);
      ti=GWEN_CurrentTime();
      if (ti) {
        t.ref().setDate(ti);
        GWEN_Time_free(ti);
      }
      _app->addTransfer(t);
    }
    AB_Job_free(job);
    // update transfer list
    slotUpdated();

    QMessageBox::information(0,
                             tr("Job enqueued"),
                             tr("<qt>"
                                "An EU-transfer job has been enqueued.\n"
                                "Please go to the <i>outbox</i> and press\n"
                                "the <i>execute</i> button there to "
                                "actually send\n"
                                "the enqueued jobs."
                                "</qt>"),
                             tr("Dismiss"), 0, 0, 0);
  }
}



void TransferView::slotNewEuTransfer() {
  _newEuTransfer();
}



void TransferView::slotRepeat(){
  RefPointer<Transfer> t;
  Transfer::TransferType tt;

  t=_xaList->getCurrentTransfer();
  if (!t.isValid()) {
    DBG_ERROR(0, "No current transfer");
    QMessageBox::critical(0,
			  tr("No Selection"),
			  tr("<qt>"
			     "Please select a transfer"
			     "</qt"),
			  tr("Dismiss"), 0, 0, 0, 0);
    return;
  }

  tt=t.ref().getTransferType();
  if (tt==Transfer::TransferType_Unknown) {
    switch(t.ref().getTextKey()) {
    case 4:
    case 5:
      tt=Transfer::TransferType_DebitNote;
      break;
    default:
      tt=Transfer::TransferType_Simple;
    }
  }

  switch(tt) {
  case Transfer::TransferType_Simple:
    _newTransfer(t);
    break;
  case Transfer::TransferType_DebitNote:
    _newDebitNote(t);
    break;
  case Transfer::TransferType_EuTransfer:
    _newEuTransfer(t);
    break;
  default:
    DBG_ERROR(0, "Unknown transfer type %d", tt);
    QMessageBox::critical(0,
			  tr("Internal Error"),
			  tr("<qt>"
			     "Unknown Transfer Type"
			     "</qt"),
			  tr("Dismiss"), 0, 0, 0, 0);
    return;
  }
}







