/***************************************************************************
 $RCSfile: accountlist.cpp,v $
                             -------------------
    cvs         : $Id: accountlist.cpp,v 1.11 2005/06/13 20:00:08 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "accountlist.h"
#include <assert.h>
#include <qstring.h>

#include <gwenhywfar/debug.h>
#include <gwenhywfar/text.h>

#define ACCOUNTLIST_MAXVALUE ((double)(200000000.00))


AccountListViewItem::AccountListViewItem(AccountListView *parent,
                                         Account *acc)
:QListViewItem(parent)
,_account(acc){
  assert(acc);
  _populate();
}



AccountListViewItem::AccountListViewItem(const AccountListViewItem &item)
:QListViewItem(item)
,_account(0){

  if (item._account) {
    _account=item._account;
  }
}


AccountListViewItem::AccountListViewItem(AccountListView *parent,
                                         QListViewItem *after,
                                         Account *acc)
:QListViewItem(parent, after)
,_account(acc){
  assert(acc);
  _populate();
}



AccountListViewItem::~AccountListViewItem(){
}



Account *AccountListViewItem::getAccount(){
  return _account;
}


void AccountListViewItem::_populate() {
  QString tmp;
  int i;
  const AB_ACCOUNT_STATUS *ast;
  const AB_VALUE *v;

  assert(_account);

  i=0;

  // banking id
  setText(i++, QString::number(_account->getBankingId()));

  // bank code
  setText(i++, QString::fromUtf8(_account->getBankCode().c_str()));

  // bank name
  tmp=QString::fromUtf8(_account->getBankName().c_str());
  if (tmp.isEmpty())
    tmp="(unnamed)";
  setText(i++,tmp);

  // account id
  setText(i++, QString::fromUtf8(_account->getAccountNumber().c_str()));

  // account name
  tmp=QString::fromUtf8(_account->getAccountName().c_str());
  if (tmp.isEmpty())
    tmp="(unnamed)";
  setText(i++, tmp);

  // owner name
  tmp=QString::fromUtf8(_account->getOwnerName().c_str());
  if (tmp.isEmpty())
    tmp="";
  setText(i++, tmp);

  // balances
  ast=_account->getAccountStatus();
  if (ast) {
    const AB_BALANCE *bal;

    // booked balance
    bal=AB_AccountStatus_GetBookedBalance(ast);
    if (bal) {
      v=AB_Balance_GetValue(bal);
      if (v)
        setText(i++, QString::number(AB_Value_GetValue(v), 'f', 2));
      else
        setText(i++, "");
    }
    else {
      setText(i++, "");
    }

    // noted balance
    bal=AB_AccountStatus_GetNotedBalance(ast);
    if (bal) {
      v=AB_Balance_GetValue(bal);
      if (v)
        setText(i++, QString::number(AB_Value_GetValue(v), 'f', 2));
      else
        setText(i++, "");
    }
    else {
      setText(i++, "");
    }

    // bank line
    v=AB_AccountStatus_GetBankLine(ast);
    if (v)
      setText(i++, QString::number(AB_Value_GetValue(v), 'f', 2));
    else
      setText(i++, "");
  }
  else {
    setText(i++, "");
    setText(i++, "");
    setText(i++, "");
  }


}



// return string for sort
QString AccountListViewItem::key(int column, bool ascending) const{
  QString result;
  double d;
  ulong i;
  bool ok;
  char numbuf[32];

  switch(column) {
  case 0: // id
    i=text(column).toULong(&ok);
    if (ok) {
      snprintf(numbuf, sizeof(numbuf), "%012lu", i);
      result=QString(numbuf);
    }
    else
      result=text(column);
    break;

  case 6: // booked
  case 7: // noted
  case 8: // credit line
    d=text(column).toDouble(&ok);
    if (ok) {
      d=ACCOUNTLIST_MAXVALUE-d;
      snprintf(numbuf, sizeof(numbuf), "%012lf.2", d);
      result=QString(numbuf);
    }
    else
      result=text(column);
    break;

  default:
    result=text(column);
    break;
  }

  return result;
}









AccountListView::AccountListView(QWidget *parent, const char *name)
:QListView(parent, name){
  setAllColumnsShowFocus(true);
  setShowSortIndicator(true);
  addColumn(QWidget::tr("Id"),-1);
  addColumn(QWidget::tr("Institute Code"),-1);
  addColumn(QWidget::tr("Institute Name"),-1);
  addColumn(QWidget::tr("Account Number"),-1);
  addColumn(QWidget::tr("Account Name"),-1);
  addColumn(QWidget::tr("Owner"),-1);
  addColumn(QWidget::tr("Booked"),-1);
  addColumn(QWidget::tr("Noted"),-1);
  addColumn(QWidget::tr("Bank Line"),-1);
  setColumnAlignment(6, Qt::AlignRight);
  setColumnAlignment(7, Qt::AlignRight);
  setColumnAlignment(8, Qt::AlignRight);
}



AccountListView::~AccountListView(){
}



void AccountListView::addAccount(Account *acc){
  AccountListViewItem *entry;

  entry=new AccountListViewItem(this, acc);
}



void AccountListView::addAccounts(const std::list<Account*> &accs){
  std::list<Account*>::const_iterator it;

  for (it=accs.begin(); it!=accs.end(); it++) {
    AccountListViewItem *entry;

    entry=new AccountListViewItem(this, *it);
  } /* for */
}



Account *AccountListView::getCurrentAccount() {
  AccountListViewItem *entry;

  entry=dynamic_cast<AccountListViewItem*>(currentItem());
  if (!entry) {
    DBG_DEBUG(0, "No item selected in list.\n");
    return 0;
  }
  return entry->getAccount();
}



std::list<Account*> AccountListView::getSelectedAccounts(){
  std::list<Account*> accs;
  AccountListViewItem *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      entry=dynamic_cast<AccountListViewItem*>(it.current());
      if (entry)
        accs.push_back(entry->getAccount());
    }
  } // for

  return accs;
}



void AccountListView::selectAccounts(const char *bankCode,
                                     const char *accountId,
                                     bool s){
  std::list<Account*> accs;
  AccountListViewItem *entry;

  assert(bankCode);
  assert(accountId);

  DBG_DEBUG(0, "Selecting account(s): %s/%s (%d)",
             bankCode, accountId, s);

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    entry=dynamic_cast<AccountListViewItem*>(it.current());
    if (entry) {
      Account *acc;
      bool match=false;

      acc=entry->getAccount();
      assert(acc);
      if (-1!=GWEN_Text_ComparePattern(acc->getBankCode().c_str(),
                                       bankCode, 0))
        if (-1!=GWEN_Text_ComparePattern(acc->getAccountNumber().c_str(),
                                         accountId, 0)) {
          DBG_DEBUG(0, "Setting item selected (%d)", s);
          setSelected(it.current(), s);
          it.current()->repaint();
          match=true;
        }
      if (!match) {
        if (it.current()->isSelected()==s) {
          DBG_DEBUG(0, "Inverting item selection to %d", !s);
          setSelected(it.current(), !s);
          it.current()->repaint();
        }

      }
    }
  } // for
}




