/***************************************************************************
 $RCSfile: payeelist.cpp,v $
                             -------------------
    cvs         : $Id: payeelist.cpp,v 1.4 2006/01/17 22:32:57 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "payeelist.h"
#include <assert.h>
#include <qstring.h>

#include <gwenhywfar/debug.h>
#include <gwenhywfar/text.h>


#define PAYEELIST_MAXVALUE ((double)(200000000.00))


PayeeListViewItem::PayeeListViewItem(PayeeListView *parent, Payee *py)
:QListViewItem(parent)
,_payee(py){
  assert(py);
  _populate();
}



PayeeListViewItem::PayeeListViewItem(const PayeeListViewItem &item)
:QListViewItem(item)
,_payee(0){

  if (item._payee) {
    _payee=item._payee;
  }
}



PayeeListViewItem::PayeeListViewItem(PayeeListView *parent,
                                     QListViewItem *after,
                                     Payee *py)
:QListViewItem(parent, after)
,_payee(py){
  assert(py);
  _populate();
}



PayeeListViewItem::~PayeeListViewItem(){
}



Payee *PayeeListViewItem::getPayee(){
  return _payee;
}



void PayeeListViewItem::_populate() {
  QString tmp;
  int i;
  const AB_VALUE *vIn;
  const AB_VALUE *vOut;

  assert(_payee);

  i=0;

  // id
  setText(i++, QString::fromUtf8(_payee->id().c_str()));

  // name
  tmp=QString::fromUtf8(_payee->name().c_str());
  if (tmp.isEmpty())
    tmp="";
  setText(i++, tmp);

  // bank code
  setText(i++, QString::fromUtf8(_payee->bankCode().c_str()));

  // account id
  setText(i++, QString::fromUtf8(_payee->accountNumber().c_str()));

  // city
  setText(i++, QString::fromUtf8(_payee->city().c_str()));

  // amount in
  vIn=_payee->inValue();
  if (vIn)
    setText(i++, QString::number(AB_Value_GetValue(vIn), 'f', 2));
  else
    setText(i++, QString::null);

  // amount out
  vOut=_payee->outValue();
  if (vOut)
    setText(i++, QString::number(AB_Value_GetValue(vOut), 'f', 2));
  else
    setText(i++, QString::null);

  // amount out
  if (vIn && vOut) {
    AB_VALUE *v;

    v=AB_Value_dup(vIn);
    AB_Value_SubValue(v, vOut);
    setText(i++, QString::number(AB_Value_GetValue(v), 'f', 2));
    AB_Value_free(v);
  }
  else
    setText(i++, QString::null);
}



QString PayeeListViewItem::key(int column, bool ascending) const {
  QString result;
  double d;
  ulong i;
  bool ok;
  char numbuf[32];

  switch(column) {
  case 0: // id
    i=text(column).toULong(&ok);
    if (ok) {
      snprintf(numbuf, sizeof(numbuf), "%012lu", i);
      result=QString(numbuf);
    }
    else
      result=text(column);
    break;

  case 5: // amount in
  case 6: // amount out
  case 7: // balance
    d=text(column).toDouble(&ok);
    if (ok) {
      d=PAYEELIST_MAXVALUE-d;
      snprintf(numbuf, sizeof(numbuf), "%012lf.2", d);
      result=QString(numbuf);
    }
    else
      result=text(column);
    break;

  default:
    result=text(column);
    break;
  }

  return result;
}









PayeeListView::PayeeListView(QWidget *parent, const char *name)
:QListView(parent, name){
  setAllColumnsShowFocus(true);
  setShowSortIndicator(true);
  addColumn(QWidget::tr("Id"),-1);
  addColumn(QWidget::tr("Name"),-1);
  addColumn(QWidget::tr("Bank Code"),-1);
  addColumn(QWidget::tr("Account Number"),-1);
  addColumn(QWidget::tr("City"),-1);
  addColumn(QWidget::tr("Amount In"),-1);
  addColumn(QWidget::tr("Amount Out"),-1);
  addColumn(QWidget::tr("Balance"),-1);
  setColumnAlignment(5, Qt::AlignRight);
  setColumnAlignment(6, Qt::AlignRight);
  setColumnAlignment(7, Qt::AlignRight);
}



PayeeListView::~PayeeListView(){
}



void PayeeListView::addPayee(Payee *py){
  PayeeListViewItem *entry;

  entry=new PayeeListViewItem(this, py);
}



void PayeeListView::addPayees(const std::list<Payee*> &accs){
  std::list<Payee*>::const_iterator it;

  for (it=accs.begin(); it!=accs.end(); it++) {
    PayeeListViewItem *entry;

    entry=new PayeeListViewItem(this, *it);
  } /* for */
}



Payee *PayeeListView::getCurrentPayee() {
  PayeeListViewItem *entry;

  entry=dynamic_cast<PayeeListViewItem*>(currentItem());
  if (!entry) {
    DBG_DEBUG(0, "No item selected in list.\n");
    return 0;
  }
  return entry->getPayee();
}



std::list<Payee*> PayeeListView::getSelectedPayees(){
  std::list<Payee*> accs;
  PayeeListViewItem *entry;

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      entry=dynamic_cast<PayeeListViewItem*>(it.current());
      if (entry)
        accs.push_back(entry->getPayee());
    }
  } // for

  return accs;
}



void PayeeListView::selectPayee(const char *id, bool s){
  PayeeListViewItem *entry;
  bool first=true;

  assert(id);

  DBG_NOTICE(0, "Selecting payee \"%s\" (%d)",
             id, s);

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    entry=dynamic_cast<PayeeListViewItem*>(it.current());
    if (entry) {
      Payee *p;

      p=entry->getPayee();
      assert(p);
      if (-1!=GWEN_Text_ComparePattern(p->id().c_str(),
                                       id, 0)) {
        DBG_NOTICE(0, "Setting item selected (%d)", s);
        if (first) {
          ensureItemVisible(it.current());
          first=false;
        }
        setSelected(it.current(), s);
        it.current()->repaint();
      }
      else {
        if (it.current()->isSelected()==s) {
          DBG_NOTICE(0, "Inverting item selection to %d", !s);
          setSelected(it.current(), !s);
          it.current()->repaint();
        }
      }
    }
  } // for
}



void PayeeListView::selectPayeeOnly(const char *id, bool s){
  PayeeListViewItem *entry;
  bool first=true;

  assert(id);

  DBG_NOTICE(0, "Selecting payee \"%s\" (%d)",
             id, s);

  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    entry=dynamic_cast<PayeeListViewItem*>(it.current());
    if (entry) {
      Payee *p;

      p=entry->getPayee();
      assert(p);
      if (-1!=GWEN_Text_ComparePattern(p->id().c_str(),
                                       id, 0)) {
        DBG_NOTICE(0, "Setting item selected (%d)", s);
        if (first) {
          ensureItemVisible(it.current());
          first=false;
        }
        setSelected(it.current(), s);
        it.current()->repaint();
      }
    }
  } // for
}



void PayeeListView::ensureSelectionVisible(){
  // Create an iterator and give the listview as argument
  QListViewItemIterator it(this);
  // iterate through all items of the listview
  for (;it.current();++it) {
    if (it.current()->isSelected()) {
      ensureItemVisible(it.current());
      break;
    }
  } // for
}






