/***************************************************************************
  recipe.h
  -------------------
  A brew recipe class
  -------------------
  begin         Novemever 11th, 2001
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 2002, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef RECIPE_H
#define RECIPE_H

#include <qstring.h>

#include "quantity.h"
#include "grain.h"
#include "hops.h"
#include "miscingredient.h"
#include "style.h"

/**
 * A brew recipe. Contains all elements necessary for a brewing recipe,
 * including ingredients and style.
 */
class Recipe {
 public:
    // default constructor
    Recipe();
    // full constructor
    Recipe(const QString &title, const QString &brewer, bool mash,
           const Volume &size, const Style &style, const GrainList &grains,
           const HopsList &hops, const MiscIngredientList &miscs,
           const QString &recipenotes, const QString &batchnotes);
    // copy constructor
    Recipe(const Recipe &r);
    // destructor
    ~Recipe();
    // assignment operator
    Recipe operator=(const Recipe &r);

    // "get and set"
    QString& title();
    void setTitle(const QString &t);
    QString& brewer();
    void setBrewer(const QString &b);
    bool mashed();
    void setMashed(bool m);
    Volume &size();
    void setSize(const Volume &s);
    Style &style();
    void setStyle(const Style &s);
    GrainList *grains();
    void setGrains(const GrainList &gl);
    HopsList *hops();
    void setHops(const HopsList &hl);
    MiscIngredientList *miscs();
    void setMiscs(const MiscIngredientList &ml);
    QString& recipeNotes();
    void setRecipeNotes(const QString &n);
    QString& batchNotes();
    void setBatchNotes(const QString &n);

    // has the data been modified?
    bool modified();
    // set the modification
    void setModified(bool m);

    // add a grain to the recipe
    GrainIterator addGrain(const Grain &g);
    // apply a grain to the recipe
    void applyGrain(GrainIterator it, const Grain &g);
    // remove a grain from the recipe
    void removeGrain(GrainIterator it);
    // add a hop to the recipe
    HopIterator addHop(const Hops &h);
    // apply a hop to the recipe
    void applyHop(HopIterator it, const Hops &h);
    // remove a hop from the recipe
    void removeHop(HopIterator it);
    // add a misc ingredient to the recipe
    MiscIngredientIterator addMisc(const MiscIngredient &m);
    // apply a misc ingredient to the recipe
    void applyMisc(MiscIngredientIterator it, const MiscIngredient &m);
    // remove a misc ingredient from the recipe
    void removeMisc(MiscIngredientIterator it);

 private:
    QString title_;
    QString brewer_;
    bool mash_;
    Volume size_;
    Style style_;
    GrainList grains_;
    HopsList hops_;
    MiscIngredientList miscs_;
    QString recipenotes_;
    QString batchnotes_;

    bool modified_;

    friend class Calc;
    double og_;
    double ibu_;
    double srm_;

};

// Inlined Methods //////////////////////////////////////////////////////////

inline QString &Recipe::title()
    { return title_; }

inline void Recipe::setTitle(const QString &t)
    { title_ = t; modified_ = true; }

inline QString &Recipe::brewer()
    { return brewer_; }

inline void Recipe::setBrewer(const QString &b)
    { brewer_ = b;  modified_ = true; }

inline bool Recipe::mashed()
    { return mash_; }

inline void Recipe::setMashed(bool m)
    { mash_ = m;  modified_ = true; }

inline Volume &Recipe::size()
    { return size_; }

inline Style &Recipe::style()
    { return style_; }

inline void Recipe::setStyle(const Style &s)
    { style_ = s;  modified_ = true; }

inline GrainList *Recipe::grains()
    { return &grains_; }

inline void Recipe::setGrains(const GrainList &gl)
    { grains_ = gl;  modified_ = true; }

inline HopsList *Recipe::hops()
    { return &hops_; }

inline void Recipe::setHops(const HopsList &hl)
    { hops_ = hl;  modified_ = true; }

inline MiscIngredientList *Recipe::miscs()
    { return &miscs_; }

inline void Recipe::setMiscs(const MiscIngredientList &ml)
    { miscs_ = ml;  modified_ = true; }

inline QString& Recipe::recipeNotes()
    { return recipenotes_; }

inline void Recipe::setRecipeNotes(const QString &n)
    { recipenotes_ = n; modified_ = true; }

inline QString& Recipe::batchNotes()
    { return batchnotes_; }

inline void Recipe::setBatchNotes(const QString &n)
    { batchnotes_ = n; modified_ = true; }

inline bool Recipe::modified()
    { return modified_; }

inline void Recipe::setModified(bool m)
    { modified_ = m; }

#endif // RECIPE_H
