/***************************************************************************
  configure.cpp
  -------------------
  QBrew configuration dialog
  -------------------
  Copyright 2004-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qiconset.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qstyle.h>
#include <qstylefactory.h>
#include <qtoolbutton.h>

#include "fixedspinbox.h"

#include "controller.h"
#include "model.h"
#include "resource.h"

#include "configure.h"
#include "generaldialogpage.h"
#include "calcdialogpage.h"
#include "recipedialogpage.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

Configure::Configure(QWidget* parent, const char* name)
    : QTabDialog(parent, QString(name), false, WDestructiveClose),
      genpage_(0), recipepage_(0), calcpage_(0)
{
    setCaption(AppResource::ID_TITLE + " - Configure");

    setOkButton("&OK");
    setApplyButton("&Apply");
    setCancelButton("&Cancel");
    setDefaultButton("&Defaults");

    // add dialog pages
    genpage_ = new GeneralDialogPage(this);
    recipepage_ = new RecipeDialogPage(this);
    calcpage_ = new CalcDialogPage(this);
    insertTab(genpage_, "&General");
    insertTab(recipepage_, "&Recipe");
    insertTab(calcpage_, "&Calculations");

    // additional setup for apptool button
    genpage_->apptool->setIconSet(QIconSet(Controller::findImage("sm_fileopen"),
                                           Controller::findImage("fileopen")));

    // additional setup for custom FixedSpinBox widget
    recipepage_->batchlabel->setBuddy(recipepage_->batch);
    recipepage_->batch->setMinimum(0.00);
    recipepage_->batch->setMaximum(100.00);
    recipepage_->batch->setSingleStep(0.25);
    recipepage_->batch->setPrecision(2);
    recipepage_->batch->
        setSuffix(" " + Model::instance()->defaultSize().unit().symbol());

    calcpage_->efficiencylabel->setBuddy(calcpage_->efficiency);
    calcpage_->efficiency->setMinimum(0.00);
    calcpage_->efficiency->setMaximum(1.00);
    calcpage_->efficiency->setSingleStep(0.01);
    calcpage_->efficiency->setPrecision(2);

    // signals and slots connections
    connect(this, SIGNAL(defaultButtonPressed()), this, SLOT(slotDefault()));
    connect(this, SIGNAL(applyButtonPressed()), this, SLOT(slotApply()));
    connect(this, SIGNAL(cancelButtonPressed()), this, SLOT(reject()));

    connect(genpage_->apptool, SIGNAL(clicked()),
            this, SLOT(chooseAppDir()));
    connect(calcpage_->units, SIGNAL(activated(const QString&)),
            this, SLOT(convertBatchSpin(const QString&)));
}

Configure::~Configure() { ; }

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// readConfig()
// ------------
// Read in the config to get the states

void Configure::setState(const ConfigState &state)
{
    state_ = state;
    setGeneralState(state_.general);
    setRecipeState(state_.recipe);
    setCalcState(state_.calc);
}

//////////////////////////////////////////////////////////////////////////////
// setGeneralState()
// -----------------
// Set the state for the general page

void Configure::setGeneralState(const GenConfigState &state)
{
    if (genpage_) {
        genpage_->appdir->setText(state.appdir);
        genpage_->lookfeel->insertStringList(QStyleFactory::keys());
        genpage_->lookfeel->listBox()->sort();
        genpage_->largeicons->setChecked(state.largeicons);
        genpage_->showsplash->setChecked(state.showsplash);
        genpage_->autosave->setChecked(state.autosave);
        genpage_->saveinterval->setValue(state.saveinterval);
        genpage_->autobackup->setChecked(state.autobackup);
        genpage_->loadlast->setChecked(state.loadlast);
        genpage_->recentnum->setValue(state.recentnum);

        // set combo for lookfeel string
        QListBox *box = genpage_->lookfeel->listBox();
        QListBoxItem *item;
        if (state.lookfeel.isEmpty()) {
            item = box->findItem(QApplication::style().name(), Qt::ExactMatch);
        } else {
            item = box->findItem(state.lookfeel, Qt::ExactMatch);
        }
        if (item) {
            genpage_->lookfeel->setCurrentItem(box->index(item));
        } else {
            genpage_->lookfeel->insertItem(state.lookfeel);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// setCalcState()
// ----------------- 
// Set the state for the recipe page

void Configure::setRecipeState(const RecipeConfigState &state)
{
    if (recipepage_) {
        recipepage_->batch->setValue(state.batch);
        recipepage_->style->insertStringList(Model::instance()->stylesList());
        recipepage_->hopform->insertStringList(Model::instance()->formsList());
        recipepage_->mash->setChecked(state.mash);

        // set combo for style string
        QListBox *box = recipepage_->style->listBox();
        QListBoxItem *item = box->findItem(state.style, Qt::ExactMatch);
        if (item) {
            recipepage_->style->setCurrentItem(box->index(item));
        } else {
            recipepage_->style->insertItem(state.style);
        }
        // set combo for hopform string
        box = recipepage_->hopform->listBox();
        item = box->findItem(state.hopform, Qt::ExactMatch);
        if (item) {
            recipepage_->hopform->setCurrentItem(box->index(item));
        } else {
            recipepage_->hopform->insertItem(state.hopform);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// setCalcState()
// ----------------- 
// Set the state for the calculation page

void Configure::setCalcState(const CalcConfigState &state)
{
    if (calcpage_) {
        calcpage_->efficiency->setValue(state.efficiency);
        calcpage_->morey->setChecked(state.morey);
        calcpage_->tinseth->setChecked(state.tinseth);
        QStringList list; list << UNIT_METRIC << UNIT_US;
        calcpage_->units->insertStringList(list);

        // set combo for units string
        QListBox *box = calcpage_->units->listBox();
        QListBoxItem *item = box->findItem(state.units, Qt::ExactMatch);
        if (item) {
            calcpage_->units->setCurrentItem(box->index(item));
        } else {
            calcpage_->units->insertItem(state.units);
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// accept()
// --------
// OK button was pressed (QTabDialog doesn't have accept signal)

void Configure::accept()
{
    QDialog::accept();
    emit configureAccept();
}

//////////////////////////////////////////////////////////////////////////////
// slotDefault()
// -------------
// Set dialog to default values

void Configure::slotDefault()
{
    if (currentPage() == 0) return;

    if (currentPage() == (QWidget*)genpage_) {
        setGeneralState(GenConfigState());
    } else if (currentPage() == (QWidget*)recipepage_) {
        setRecipeState(RecipeConfigState());
    } else if (currentPage() == (QWidget*)calcpage_) {
        setCalcState(CalcConfigState());
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotApply()
// -----------
// Apply values

void Configure::slotApply()
{
    // TODO: should I only apply the current page?

    if (genpage_) {
        // general page
        state_.general.appdir = genpage_->appdir->text();
        state_.general.lookfeel = genpage_->lookfeel->currentText();
        state_.general.largeicons = genpage_->largeicons->isChecked();
        state_.general.showsplash = genpage_->showsplash->isChecked();
        state_.general.autosave = genpage_->autosave->isChecked();
        state_.general.saveinterval = genpage_->saveinterval->value();
        state_.general.autobackup = genpage_->autobackup->isChecked();
        state_.general.loadlast = genpage_->loadlast->isChecked();
        state_.general.recentnum = genpage_->recentnum->value();
        emit generalApply(state_.general);
    }

    if (recipepage_) {
        // recipe page
        state_.recipe.batch = recipepage_->batch->value();
        state_.recipe.style = recipepage_->style->currentText();
        state_.recipe.hopform = recipepage_->hopform->currentText();
        state_.recipe.mash = recipepage_->mash->isChecked();
        emit recipeApply(state_.recipe);
    }

    if (calcpage_) {
        // calc page
        state_.calc.efficiency = calcpage_->efficiency->value();
        state_.calc.morey = calcpage_->morey->isChecked();
        state_.calc.tinseth = calcpage_->tinseth->isChecked();
        state_.calc.units = calcpage_->units->currentText();
        emit calcApply(state_.calc);
    }
}

//////////////////////////////////////////////////////////////////////////////
// chooseAppDir()
// --------------
// Choose appdir directory via file dialog

void Configure::chooseAppDir()
{
    QDir dir(genpage_->appdir->text());
    if (!dir.isReadable()) dir = QDir::current();

    QString appdir = QFileDialog::getExistingDirectory(dir.absPath());
    if (!appdir.isEmpty()) {
        genpage_->appdir->setText(appdir);
    }
}

//////////////////////////////////////////////////////////////////////////////
// convertBatchSpin()
// ------------------
// Set the appropriate suffice for the batch spinbox

void Configure::convertBatchSpin(const QString &selection)
{
    if (selection == AppResource::UNIT_US)
        recipepage_->batch->setSuffix(" " + Volume::gallon.symbol());
    else if (selection == AppResource::UNIT_METRIC)
        recipepage_->batch->setSuffix(" " + Volume::liter.symbol());
    else
        recipepage_->batch->setSuffix("");
}
