/***************************************************************************
  hopdb.cpp
  -------------------
  A dialog page for hops
  -------------------
  Copyright 2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "controller.h"
#include "model.h"
#include "view.h"

#include "hopdb.h"

using AppResource::ID_TITLE;

// Note that this class is very similar to the GrainPage class. This is
// because they both implement very similar interfaces. They are separate
// for maintenance reasons.

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HopDB()
// --------
// Constructor

HopDB::HopDB(QWidget *parent, const char *name, Model *model)
    : HopView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    hoplistview->setColumnAlignment(HITEM_NAME, AlignLeft);
    hoplistview->setColumnAlignment(HITEM_ALPHA, AlignRight);

    alphalabel->setBuddy(alphaspin);
    alphaspin->setMinimum(0.0);
    alphaspin->setMaximum(50.0);
    alphaspin->setSingleStep(0.1);
    alphaspin->setPrecision(1);
    alphaspin->setSuffix("%");

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // hide quantity, form, time in dbmode
    hoplistview->setColumnWidthMode(HITEM_QUANTITY, QListView::Manual);
    hoplistview->setColumnWidthMode(HITEM_FORM, QListView::Manual);
    hoplistview->setColumnWidthMode(HITEM_TIME, QListView::Manual);
    hoplistview->hideColumn(HITEM_QUANTITY);
    hoplistview->hideColumn(HITEM_FORM);
    hoplistview->hideColumn(HITEM_TIME);
    quantitylabel->hide();
    quantityspin->hide();
    formlabel->hide();
    formcombo->hide();
    timelabel->hide();
    timespin->hide();

    // adjust size of leftmost labels so they line up
    namelabel->adjustSize();
    int leftmost = namelabel->width();
    alphalabel->adjustSize();
    leftmost = QMAX(leftmost, alphalabel->width());
    alphalabel->setMinimumWidth(leftmost);
    namelabel->setMinimumWidth(leftmost);

    // fixup tab orders
    QWidget::setTabOrder(namecombo, quantityspin);
    QWidget::setTabOrder(quantityspin, alphaspin);
    QWidget::setTabOrder(alphaspin, formcombo);
    QWidget::setTabOrder(alphaspin, timespin);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addHop()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyHop()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeHop()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(hoplistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
}

HopDB::~HopDB() { ; }

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields

void HopDB::refresh()
{
    // update list
    HopIterator it;
    HopItem *item;
    HopList *list = model_->hopDB();
    hoplistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new HopItem(hoplistview);
        item->setText(HITEM_NAME, (*it).name());
        item->setText(HITEM_ALPHA,
                      QString::number((*it).alpha(), 'f', 1) + '%');
        item->setIterator(it);
    }

    // update fields
    namecombo->clear();
    namecombo->insertStringList(model_->hopsList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// --------------
// Update the fields in the entry area

void HopDB::updateFields(const QString &name)
{
    Hop *thishop = model_->hop(name);
    if (thishop) { // sanity check
        alphaspin->setValue(thishop->alpha());
    }
}

//////////////////////////////////////////////////////////////////////////////
// addHop()
// ---------
// Add a hop to recipe

void HopDB::addHop()
{
    HopIterator it;
    HopItem *item;

    Hop hop(namecombo->currentText(),
             Weight(quantityspin->value(), model_->defaultHopUnit()),
             formcombo->currentText(), alphaspin->value(),
             timespin->value());

    if (model_->checkHop(hop.name())) {
        // already exists, query user
        if (QMessageBox::information(this, ID_TITLE + " - Replace?",
                                     QString("<p>Are you sure you want to "
                                             "replace the entry \"%1\"")
                                     .arg(hop.name()),
                                     QMessageBox::No | QMessageBox::Escape,
                                     QMessageBox::Yes)
            == QMessageBox::No) return;

        // remove existing item from listview
        item = (HopItem*)hoplistview->findItem(hop.name(), HITEM_NAME);
        if (item) {
            hoplistview->takeItem(item);
            delete item;
        }
    }

    // create new listview item
    item = new HopItem(hoplistview);
    item->setText(HITEM_NAME, hop.name());
    item->setText(HITEM_ALPHA, QString::number(hop.alpha(), 'f', 1) + '%');

    // add hop to database
    it = model_->addHop(hop, true);
    item->setIterator(it);

    hoplistview->ensureItemVisible(item);
    hoplistview->setCurrentItem(item);
}

//////////////////////////////////////////////////////////////////////////////
// applyHop()
// -----------
// Apply a hop to recipe

void HopDB::applyHop()
{
    if (currentitem_ != 0) {
        Hop hop(namecombo->currentText(),
                 Weight(quantityspin->value(), model_->defaultHopUnit()),
                 formcombo->currentText(), alphaspin->value(),
                 timespin->value());
        // apply hop (overwrite is simplest way)
        HopIterator it = model_->addHop(hop, true);
        currentitem_->setIterator(it);
        currentitem_->refresh();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeHop()
// ------------
// Remove a hop from recipe

void HopDB::removeHop()
{
    if (currentitem_ != 0) {
        // remove hop from recipe
        model_->removeHop(currentitem_->iterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// -----------------
// Received when user selects an item in listview

void HopDB::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<HopItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(HITEM_NAME));
        // NOTE: cannot have any suffixes or toUInt() will fail
        timespin->setValue(item->text(HITEM_TIME).toUInt());
    }
}
