/***************************************************************************
  model.cpp
  -------------------
  Model class for brewing
  -------------------
  Copyright (c) 2001-2005 David Johnson
  Please see the header file for copyright and license information.
 ***************************************************************************/

#include <qdir.h>
#include <qfile.h>
#include <qdom.h>
#include <qstringlist.h>
#include <qtextstream.h>

#include "controller.h"
#include "resource.h"

#include "model.h"

using namespace AppResource;
using namespace CalcResource;

Model *Model::instance_ = 0;

// TODO: separate database related methods from recipe related methods?

// Construction, Destruction /////////////////////////////////////////////////

// Private constructor
Model::Model()
    : QObject(0, "model"), defaultsize_(Volume(5.0, Volume::gallon)),
      defaultmash_(false), defaulthopform_(HOP_PELLET),
      defaultgrainunit_(&Weight::pound), defaulthopunit_(&Weight::ounce),
      defaultmiscunit_(&Quantity::generic), graindb_(), hopdb_(), miscdb_(),
      styledb_(), recipe_(0), datadir_("")
{
    // load data file - try user home directory first, quietly...
    if (!loadData(QDIR_HOME + "/." + ID_CALC_FILE, true)) {
        // then the default data file
        if (!loadData(Controller::instance()->dataBase() + ID_CALC_FILE)) {
            // else load in some defaults
            styledb_.append(Style());
            graindb_.append(Grain());
            hopdb_.append(Hop());
            miscdb_.append(MiscIngredient());
            UEntry u = {0, 20};
            Calc::addUEntry(u);
        }
    }
    recipe_ = new Recipe();
}

// Private destructor
Model::~Model()
{ ; }

// Return pointer to the model
Model *Model::instance()
{
    if (!instance_)
        instance_ = new Model();
    return instance_;
}

//////////////////////////////////////////////////////////////////////////////
// Data Access                                                              //
//////////////////////////////////////////////////////////////////////////////

void Model::setDefaultSize(const Volume &v)
{
    defaultsize_ = v;
    // convert recipe size
    recipe_->setSize(defaultsize_);
}

void Model::setDefaultStyle(const QString &s)
{
    StyleIterator it;
    for (it=styledb_.begin(); it!=styledb_.end(); ++it) {
        if (s == (*it).name()) defaultstyle_ = (*it);
    }
    recipe_->setStyle(defaultstyle_);
}

void Model::setDefaultStyle(const Style &s)
{
    defaultstyle_ = s;
    recipe_->setStyle(defaultstyle_);
}

void Model::setDefaultGrainUnit(Unit &u)
{
    defaultgrainunit_ = &u;
    // convert recipe grain units
    GrainIterator it;
    for (it = recipe_->grains()->begin(); it != recipe_->grains()->end(); ++it)
        (*it).weight().convert(u);    
    // convert graindb units
    for (it = graindb_.begin(); it != graindb_.end(); ++it )
        (*it).setWeight(Weight(1.0, u));    
}

void Model::setDefaultHopUnit(Unit &u)
{
    defaulthopunit_ = &u;
    // convert recipe hop units
    HopIterator it;
    for (it = recipe_->hops()->begin(); it != recipe_->hops()->end(); ++it)
        (*it).weight().convert(u);
    // convert hopdb units
    for (it = hopdb_.begin(); it != hopdb_.end(); ++it )
        (*it).setWeight(Weight(1.0, u));
}

void Model::setDefaultMiscUnit(Unit &u)
{
    defaultmiscunit_ = &u;
    // convert recipe misc units
    MiscIngredientIterator it;
    for (it = recipe_->miscs()->begin(); it != recipe_->miscs()->end(); ++it)
        (*it).quantity().convert(u);
    // convert miscdb units
    for (it = miscdb_.begin(); it != miscdb_.end(); ++it )
        (*it).setQuantity(Quantity(1.0, u));
}

//////////////////////////////////////////////////////////////////////////////
// stylesList()
// ------------
// Return a string list of available styles

QStringList Model::stylesList()
{
    QStringList list;
    StyleIterator it;
    for (it=styledb_.begin(); it!=styledb_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// style()
// -------
// Return a style given its name

Style *Model::style(const QString &name)
{
    StyleIterator it;
    for (it=styledb_.begin(); it!=styledb_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// checkStyle()
// ----------
// Check existance of style

bool Model::checkStyle(const QString &name)
{
    bool found = false;
    StyleIterator it;
    for (it=styledb_.begin(); it!=styledb_.end(); ++it) {
        if (name == (*it).name()) found = true;
    }
    return found;
}

//////////////////////////////////////////////////////////////////////////////
// addStyle()
// ----------
// Add style to database, replacing if replace parameter is set

// TODO: need to use qmap for DB
StyleIterator Model::addStyle(const Style &s, bool replace)
{
    bool found = false;
    StyleIterator it;
    for (it=styledb_.begin(); it!=styledb_.end(); ++it) {
        if (s.name() == (*it).name()) {
            found = true;
            if (replace) styledb_.remove(it);
        }
    }

    if (replace || !found) it = styledb_.append(s);
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// removeStyle()
// -------------
// Remove a style from the database

void Model::removeStyle(StyleIterator it)
{
    styledb_.remove(it);
}

//////////////////////////////////////////////////////////////////////////////
// checkGrain()
// ----------
// Check existance of ingredient

bool Model::checkGrain(const QString &name)
{
    bool found = false;
    GrainIterator it;
    for (it=graindb_.begin(); it!=graindb_.end(); ++it) {
        if (name == (*it).name()) found = true;
    }
    return found;
}

//////////////////////////////////////////////////////////////////////////////
// addGrain()
// ----------
// Add grain to database, replacing if replace parameter is set

// TODO: need to use qmap for DB. cant use it for recipe, but can use it here
GrainIterator Model::addGrain(const Grain &g, bool replace)
{
    bool found = false;
    GrainIterator it;
    for (it=graindb_.begin(); it!=graindb_.end(); ++it) {
        if (g.name() == (*it).name()) {
            found = true;
            if (replace) graindb_.remove(it);
        }
    }

    if (replace || !found) it = graindb_.append(g);
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// removeGrain()
// -------------
// Remove a grain from the database

void Model::removeGrain(GrainIterator it)
{
    graindb_.remove(it);
}

//////////////////////////////////////////////////////////////////////////////
// grainsList()
// ------------
// Return string list of available grains

QStringList Model::grainsList()
{
    QStringList list;
    GrainIterator it;
    for (it=graindb_.begin(); it!=graindb_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// grain()
// -------
// Return grain given its name

Grain* Model::grain(const QString &name)
{
    GrainIterator it;
    for (it=graindb_.begin(); it!=graindb_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// checkHop()
// ----------
// Check existance of ingredient, adding it to db if not found

bool Model::checkHop(const QString &h)
{
    bool found = false;
    HopIterator it;
    for (it=hopdb_.begin(); it!=hopdb_.end(); ++it) {
        if (h == (*it).name()) found = true;
    }
    return found;
}

//////////////////////////////////////////////////////////////////////////////
// addHop()
// ----------
// Add hop to database, replacing if replace parameter is set

// TODO: need to use qmap for DB. cant use it for recipe, but can use it here
HopIterator Model::addHop(const Hop &h, bool replace)
{
    bool found = false;
    HopIterator it;
    for (it=hopdb_.begin(); it!=hopdb_.end(); ++it) {
        if (h.name() == (*it).name()) {
            found = true;
            if (replace) hopdb_.remove(it);
        }
    }

    if (replace || !found) it = hopdb_.append(h);
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// removeHop()
// -----------
// Remove a hop from the database

void Model::removeHop(HopIterator it)
{
    hopdb_.remove(it);
}

//////////////////////////////////////////////////////////////////////////////
// hopsList()
// ----------
// Return string list of available hops

QStringList Model::hopsList()
{
    QStringList list;
    HopIterator it;
    for (it=hopdb_.begin(); it != hopdb_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// formsList()
// ----------
// Return string list of available hop forms

QStringList Model::formsList()
{
    QStringList list;
    list << HOP_PELLET << HOP_PLUG << HOP_WHOLE;
    // search through existing hopdb for other forms
    HopIterator it;
    for (it=hopdb_.begin(); it!=hopdb_.end(); ++it) {
        if ((!(*it).form().isEmpty())
            && (list.contains((*it).form()) == 0)) list += (*it).form();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// hop()
// -----
// Return hop given its name

Hop* Model::hop(const QString &name)
{
    HopIterator it;
    for (it=hopdb_.begin(); it!=hopdb_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// checkMisc()
// ----------
// Check existance of ingredient, adding it to db if not found

bool Model::checkMisc(const QString &m)
{
    bool found = false;
    MiscIngredientIterator it;
    for (it=miscdb_.begin(); it!=miscdb_.end(); ++it) {
        if (m == (*it).name()) found = true;
    }
    return found;
}

//////////////////////////////////////////////////////////////////////////////
// addMisc()
// ----------
// Add ingredient to database, replacing if replace parameter is set

// TODO: need to use qmap for DB. cant use it for recipe, but can use it here
MiscIngredientIterator Model::addMisc(const MiscIngredient &m, bool replace)
{
    bool found = false;
    MiscIngredientIterator it;
    for (it=miscdb_.begin(); it!=miscdb_.end(); ++it) {
        if (m.name() == (*it).name()) {
            found = true;
            if (replace) miscdb_.remove(it);
        }
    }

    if (replace || !found) it = miscdb_.append(m);
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// removeMisc()
// ------------
// Remove an ingredient from the database

void Model::removeMisc(MiscIngredientIterator it)
{
    miscdb_.remove(it);
}

//////////////////////////////////////////////////////////////////////////////
// miscList()
// ----------
// Return string list of available misc ingredients

QStringList Model::miscList()
{
    QStringList list;
    MiscIngredientIterator it;
    for (it=miscdb_.begin(); it!=miscdb_.end(); ++it) {
        list += (*it).name();
    }
    list.sort();
    return list;
}

//////////////////////////////////////////////////////////////////////////////
// misc()
// ------
// Return misc ingredient given its name

MiscIngredient* Model::misc(const QString &name)
{
    MiscIngredientIterator it;
    for (it=miscdb_.begin(); it!=miscdb_.end(); ++it) {
        if (name == (*it).name()) return &(*it);
    }
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
// Serialization                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// loadData()
// ----------
// Load the data

bool Model::loadData(const QString &filename, bool quiet)
{
    // TODO: need more error checking on tags and elements
    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        if (!quiet) qWarning("Error: Cannot open " + filename);
        datafile->close();
        delete (datafile);
        return false;
    }

    // open dom document
    QDomDocument doc;
    doc.setContent(datafile);
    datafile->close();
    delete (datafile);

    // check the doc type and stuff
    if (doc.doctype().name() != tagDoc) {
        // wrong file type
        if (!quiet) qWarning("Error: Wrong file type " + filename);
        return false;
    }
    QDomElement root = doc.documentElement();

    // check file version
    if (root.attribute(attrVersion) < CALC_PREVIOUS) {
        // too old of a version
        if (!quiet) qWarning("Error: Unsupported version " + filename);
        return false;
    }

    // get all styles tags
    styledb_.clear();
    QDomNodeList nodes = root.elementsByTagName(tagStyles);
    QDomNodeList subnodes;
    QDomElement element;
    QDomElement subelement;
    unsigned n, m;
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all style tags
            subnodes = element.elementsByTagName(tagStyle);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    styledb_.append(Style(subelement.text(),
                        subelement.attribute(attrOGLow).toDouble(),
                        subelement.attribute(attrOGHigh).toDouble(),
                        subelement.attribute(attrFGLow, "0.0").toDouble(),
                        subelement.attribute(attrFGHigh, "0.0").toDouble(),
                        subelement.attribute(attrIBULow).toInt(),
                        subelement.attribute(attrIBUHigh).toInt(),
                        subelement.attribute(attrSRMLow).toInt(),
                        subelement.attribute(attrSRMHigh).toInt()));
                }
            }
        }
    }

   // get all grains tags
    graindb_.clear();
    nodes = root.elementsByTagName(tagGrains);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all grain tags
            subnodes = element.elementsByTagName(tagGrain);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    graindb_.append(Grain(subelement.text(),
                        Weight(1.0, *defaultgrainunit_),
                        subelement.attribute(attrExtract).toDouble(),
                        subelement.attribute(attrColor).toDouble(),
                        subelement.attribute(attrUse)));
                }
            }
        }
    }

    // get all hops tags
    hopdb_.clear();
    nodes = root.elementsByTagName(tagHops);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all hop tags
            subnodes = element.elementsByTagName(tagHop);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    hopdb_.append(Hop(subelement.text(),
                        Weight(1.0, *defaulthopunit_), QString::null,
                        subelement.attribute(attrAlpha).toDouble(), 60));
                }
            }
        }
    }

    // get all miscingredients tags
    miscdb_.clear();
    nodes = root.elementsByTagName(tagMiscIngredients);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all miscingredient tags
            subnodes = element.elementsByTagName(tagMiscIngredient);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    miscdb_.append(MiscIngredient(subelement.text(),
                        Quantity(1.0, *defaultmiscunit_),
                        subelement.attribute(attrNotes)));
                }
            }
        }
    }

    // get all utilization tags
    nodes = root.elementsByTagName(tagUtilization);
    UEntry entry;
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all entry tags
            subnodes = element.elementsByTagName(tagEntry);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    entry.time = subelement.attribute(attrTime).toUInt();
                    entry.utilization = subelement.attribute(attrUtil).toUInt();
                    Calc::addUEntry(entry);
                }
            }
        }
    }
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// saveData()
// ------------
// Save info to data file

bool Model::saveData(const QString &filename)
{
    QDomDocument doc(tagDoc);
    doc.appendChild(doc.createProcessingInstruction("xml", "version=\"1.0\""));

    // create the root element
    QDomElement root = doc.createElement(doc.doctype().name());
    root.setAttribute(attrVersion, VERSION);
    doc.appendChild(root);

    // styles elements
    QDomElement element = doc.createElement(tagStyles);
    StyleIterator sit;
    QDomElement subelement;
    // iterate through styles list
    for (sit=styledb_.begin(); sit!=styledb_.end(); ++sit) {
        subelement = doc.createElement(tagStyle);
        subelement.appendChild(doc.createTextNode((*sit).name()));
        subelement.setAttribute(attrOGLow, (*sit).OGLow());
        subelement.setAttribute(attrOGHigh, (*sit).OGHi());
        subelement.setAttribute(attrFGLow, (*sit).FGLow());
        subelement.setAttribute(attrFGHigh, (*sit).FGHi());
        subelement.setAttribute(attrIBULow, (*sit).IBULow());
        subelement.setAttribute(attrIBUHigh, (*sit).IBUHi());
        subelement.setAttribute(attrSRMLow, (*sit).SRMLow());
        subelement.setAttribute(attrSRMHigh, (*sit).SRMHi());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // grains elements
    element = doc.createElement(tagGrains);
    GrainIterator git;
    // iterate through grains_ list
    for (git=graindb_.begin(); git!=graindb_.end(); ++git) {
        subelement = doc.createElement(tagGrain);
        subelement.appendChild(doc.createTextNode((*git).name()));
        subelement.setAttribute(attrExtract, (*git).extract());
        subelement.setAttribute(attrColor, (*git).color());
        subelement.setAttribute(attrUse, (*git).useString());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // hops elements
    element = doc.createElement(tagHops);
    HopIterator hit;
    // iterate through hops_ list
    for (hit=hopdb_.begin(); hit!=hopdb_.end(); ++hit) {
        subelement = doc.createElement(tagHop);
        subelement.appendChild(doc.createTextNode((*hit).name()));
        subelement.setAttribute(attrAlpha, (*hit).alpha());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // miscingredients elements
    element = doc.createElement(tagMiscIngredients);
    MiscIngredientIterator mit;
    // iterate through misc_ list
    for (mit=miscdb_.begin(); mit!=miscdb_.end(); ++mit) {
        subelement = doc.createElement(tagMiscIngredient);
        subelement.appendChild(doc.createTextNode((*mit).name()));
        subelement.setAttribute(attrNotes, (*mit).notes());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // utilization elements
    element = doc.createElement(tagUtilization);
    QValueList<UEntry>::Iterator uit;
    QValueList<UEntry> ulist = Calc::getUEntryList();
    // iterate through uentry list
    for (uit=ulist.begin(); uit!=ulist.end(); ++uit) {
        subelement = doc.createElement(tagEntry);
        subelement.setAttribute(attrTime, (*uit).time);
        subelement.setAttribute(attrUtil, (*uit).utilization);
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning("Error: Cannot open file " + filename);
        datafile->close();
        return false;
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 2);
    datafile->close();
    delete (datafile);

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// newRecipe()
// -----------
// Creates a new recipe

void Model::newRecipe()
{
    if (recipe_) delete recipe_;
    recipe_ = new Recipe(QString::null, QString::null, defaultmash_,
                         defaultsize_, defaultstyle_, GrainList(), HopList(),
                         MiscIngredientList(), QString::null, QString::null);
    setModified(false);
    emit (recipeChanged());
}

//////////////////////////////////////////////////////////////////////////////
// nativeFormat()
// -------------
// Is the file in native QBrew format?
// Defined as "recipe" doctype with "generator=qbrew"

bool Model::nativeFormat(const QString &filename)
{
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        qWarning("Error: Cannot open " + filename);
        datafile->close();
        delete (datafile);
        return false;
    }

    // parse document
    QDomDocument doc;
    bool status = doc.setContent(datafile);
    datafile->close();
    delete (datafile);
    if (!status) return false;

    // check the doctype
    if (doc.doctype().name() != tagRecipe) return false;

    // check generator
    QDomElement root = doc.documentElement();
    if (root.attribute(attrGenerator) != PACKAGE)  return false;

    return true;
}

//////////////////////////////////////////////////////////////////////////////
// loadRecipe()
// ------------
// load the recipe from file

bool Model::loadRecipe(const QString &filename)
{
    // TODO: need more error checking on tags and elements

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        qWarning("Error: Cannot open " + filename);
        datafile->close();
        delete (datafile);
        return false;
    }

    // open dom document
    QDomDocument doc;
    doc.setContent(datafile);
    datafile->close();
    delete (datafile);

    // check the doc type and stuff
    if (doc.doctype().name() != tagRecipe) {
        // wrong file type
        qWarning("Error: Wrong file type " + filename);
        return false;
    }
    QDomElement root = doc.documentElement();

    // check generator
    if (root.attribute(attrGenerator) != PACKAGE) {
        // right file type, wrong generator
        qWarning("Not a recipe file for " + QString(PACKAGE));
        return false;
    }

    // check file version
    if (root.attribute(attrVersion) < QBREW_PREVIOUS) {
        // too old of a version
        qWarning("Error: Unsupported version " + filename);
        return false;
    }

    // new recipe
    if (recipe_) delete recipe_;
    recipe_ = new Recipe();

    // Note: for some of these tags, only one in a document makes sense.
    // But if there is more than one, process them in order, with later
    // ones overwriting the earlier

    // get title
    QDomNodeList nodes = root.elementsByTagName(tagTitle);
    QDomElement element;
	unsigned n, m;
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            recipe_->setTitle(element.text().latin1());
        }
    }
    // get brewer
    nodes = root.elementsByTagName(tagBrewer);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            recipe_->setBrewer(element.text().latin1());
        }
    }
    // get style
    nodes = root.elementsByTagName(tagStyle);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // TODO: load/save entire style
            Style *s = style(element.text());
            recipe_->setStyle(s ? *s : defaultStyle());
        }
    }
    // get batch settings
    nodes = root.elementsByTagName(tagBatch);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            recipe_->setSize(Volume(element.attribute(attrQuantity),
                                    Volume::gallon));
            recipe_->setMashed(element.attribute(attrMash).lower() == "true");
        }
    }
    // get notes
    nodes = root.elementsByTagName(tagNotes);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            if (element.hasAttribute(attrClass)) {
                if (element.attribute(attrClass) == classRecipe) {
                    recipe_->setRecipeNotes(element.text().latin1());
                } else if (element.attribute(attrClass) == classBatch) {
                    recipe_->setBatchNotes(element.text().latin1());
                }
            }
        }
    }

    // get all grains tags
    nodes = root.elementsByTagName(tagGrains);
    QDomNodeList subnodes;
    QDomElement subelement;
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all grain tags
            subnodes = element.elementsByTagName(tagGrain);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    recipe_->addGrain(Grain(subelement.text(),
                        Weight(subelement.attribute(attrQuantity), Weight::pound),
                        subelement.attribute(attrExtract).toDouble(),
                        subelement.attribute(attrColor).toDouble(),
                        subelement.attribute(attrUse)));
                }
            }
        }
    }

    // get all hops tags
    nodes = root.elementsByTagName(tagHops);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all hop tags
            subnodes = element.elementsByTagName(tagHop);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    recipe_->addHop(Hop(subelement.text(),
                        Weight(subelement.attribute(attrQuantity), Weight::ounce),
                        subelement.attribute(attrForm),
                        subelement.attribute(attrAlpha).toDouble(),
                        subelement.attribute(attrTime).toUInt()));
                }
            }
        }
    }

    // get all miscingredients tags
    nodes = root.elementsByTagName(tagMiscIngredients);
    for (n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all miscingredient tags
            subnodes = element.elementsByTagName(tagMiscIngredient);
            for (m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    recipe_->addMisc(MiscIngredient(subelement.text(),
                        Quantity(subelement.attribute(attrQuantity),
                                 Quantity::generic),
                        subelement.attribute(attrNotes)));
                }
            }
        }
    }
    // calculate the numbers
    Calc::recalc(recipe_);
    // just loaded recipes are not modified
    setModified(false);
    emit (recipeChanged());
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// importRecipe()
// -------------
// Import a recipe into the model

bool Model::importRecipe(const QString &filename)
{
    qWarning("STUBBY: Model::importRecipe()");
    return false;
}

//////////////////////////////////////////////////////////////////////////////
// saveRecipe()
// ------------
// Save the recipe out to file

bool Model::saveRecipe(const QString &filename)
{
    QDomDocument doc(tagRecipe);
    doc.appendChild(doc.createProcessingInstruction("xml", "version=\"1.0\""));

    // create the root element
    QDomElement root = doc.createElement(doc.doctype().name());
    root.setAttribute(attrGenerator, PACKAGE);
    root.setAttribute(attrVersion, VERSION);
    doc.appendChild(root);

    // title
    QDomElement element = doc.createElement(tagTitle);
    element.appendChild(doc.createTextNode(recipe_->title()));
    root.appendChild(element);
    // brewer
    element = doc.createElement(tagBrewer);
    element.appendChild(doc.createTextNode(recipe_->brewer()));
    root.appendChild(element);
    // style
    // TODO: load/save entire style
    element = doc.createElement(tagStyle);
    element.appendChild(doc.createTextNode(recipe_->style().name()));
    root.appendChild(element);
    // batch settings
    element = doc.createElement(tagBatch);
    element.setAttribute(attrQuantity, recipe_->size().toQString());
    element.setAttribute(attrMash, recipe_->mashed() ? "true" : "false");
    root.appendChild(element);
    // notes
    if (!recipe_->recipeNotes().isEmpty()) {
        element = doc.createElement(tagNotes);
        element.setAttribute(attrClass, classRecipe);
        element.appendChild(doc.createTextNode(recipe_->recipeNotes()));
        root.appendChild(element);
    }
    if (!recipe_->batchNotes().isEmpty()) {
        element = doc.createElement(tagNotes);
        element.setAttribute(attrClass, classBatch);
        element.appendChild(doc.createTextNode(recipe_->batchNotes()));
        root.appendChild(element);
    }

    // grains elements
    element = doc.createElement(tagGrains);
    GrainIterator git;
    QDomElement subelement;
    // iterate through grains list
    for (git=recipe_->grains()->begin(); git!=recipe_->grains()->end(); ++git) {
        subelement = doc.createElement(tagGrain);
        subelement.appendChild(doc.createTextNode((*git).name()));
        subelement.setAttribute(attrQuantity, (*git).weight().toQString());
        subelement.setAttribute(attrExtract, (*git).extract());
        subelement.setAttribute(attrColor, (*git).color());
        subelement.setAttribute(attrUse, (*git).useString());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // hops elements
    element = doc.createElement(tagHops);
    HopIterator hit;
    // iterate through hops list
    for (hit=recipe_->hops()->begin(); hit!=recipe_->hops()->end(); ++hit) {
        subelement = doc.createElement(tagHop);
        subelement.appendChild(doc.createTextNode((*hit).name()));
        subelement.setAttribute(attrQuantity, (*hit).weight().toQString());
        subelement.setAttribute(attrForm, (*hit).form());
        subelement.setAttribute(attrAlpha, (*hit).alpha());
        subelement.setAttribute(attrTime, (*hit).time());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // miscingredients elements
    element = doc.createElement(tagMiscIngredients);
    MiscIngredientIterator mit;
    // iterate through misc list
    for (mit=recipe_->miscs()->begin(); mit!=recipe_->miscs()->end(); ++mit) {
        subelement = doc.createElement(tagMiscIngredient);
        subelement.appendChild(doc.createTextNode((*mit).name()));
        subelement.setAttribute(attrQuantity, (*mit).quantity().toQString());
        subelement.setAttribute(attrNotes, (*mit).notes());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning("Error: Cannot open file " + filename);
        datafile->close();
        return false;
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 2);
    datafile->close();
    delete (datafile);

    // document is saved, so set flags accordingly
    recipe_->setModified(false);
    return true;
}
///////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                             //
///////////////////////////////////////////////////////////////////////////////

void Model::setModified(bool mod)
{
    recipe_->setModified(mod);
    if (mod) emit recipeModified();
}

