/***************************************************************************
  model.h
  -------------------
  Model class for brewing
  -------------------
  Copyright (c) 2001-2005 David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation
     and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef MODEL_H
#define MODEL_H

#include <qmap.h>
#include <qobject.h>
#include <qstring.h>

#include "calc.h"
#include "grain.h"
#include "hops.h"
#include "miscingredient.h"
#include "recipe.h"
#include "style.h"

class QDomElement;

/**
 * A homebrewing domain model class. This is a singleton class, since there is
 * only one brewing domain model. Currently the model can only access one
 * recipe at a time. Multiple recipes is a future trajectory.
 */
class Model : public QObject {
    Q_OBJECT
 public:
    // Return pointer to the model.
    static Model *instance();

    // Create a new recipe
    void newRecipe();
    // Is the file a native format?
    bool nativeFormat(const QString &filename);
    // Load a recipe into the model
    bool loadRecipe(const QString &filename);
    // Import a recipe into the model
    bool importRecipe(const QString &filename);
    // Save the current recipe
    bool saveRecipe(const QString &filename);

    // get/set the default size volume
    void setDefaultSize(const Volume &v);
    const Volume &defaultSize() const;
    // get/set the default style
    void setDefaultStyle(const QString &s);
    void setDefaultStyle(const Style &s);
    const Style &defaultStyle() const;
    // get/set default mash
    void setDefaultMash(bool mash);
    bool defaultMash() const;
    // get/set default hop form
    void setDefaultHopform(const QString &form);
    const QString &defaultHopform() const;
    // get/set the default grain units
    void setDefaultGrainUnit(Unit &u);
    const Unit &defaultGrainUnit() const;
    // get/set the default hop units
    void setDefaultHopUnit(Unit &u);
    const Unit &defaultHopUnit() const;
    // get/set the default misc units
    void setDefaultMiscUnit(Unit &u);
    const Unit &defaultMiscUnit() const;

    // return a list of styles
    QStringList stylesList();
    StyleList *styleDB();
    // return given style from name
    Style *style(const QString &name);
    // add style to database, replacing if asked
    StyleIterator addStyle(const Style &s, bool replace);
    // remove style from database
    void removeStyle(StyleIterator it);
    // check existance of style
    bool checkStyle(const QString &s);
    // return a list of grains
    QStringList grainsList();
    GrainList *grainDB();
    // return given grain from name
    Grain* grain(const QString &name);
    // add grain to database, replacing if asked
    GrainIterator addGrain(const Grain &g, bool replace);
    // remove grain from database
    void removeGrain(GrainIterator it);
    // check existance of grain
    bool checkGrain(const QString &g);
    // return a list of hops
    QStringList hopsList();
    HopList *hopDB();
    // return given hop from name
    Hop* hop(const QString &name);
    // add hop to database, replacing if asked
    HopIterator addHop(const Hop &g, bool replace);
    // remove hop from database
    void removeHop(HopIterator it);
    // check existance of hop
    bool checkHop(const QString &h);
    // return a list of hop forms (TODO: this should be in Hop class)
    QStringList formsList();
    // return a list of misc ingredients
    QStringList miscList();
    MiscIngredientList *miscDB();
    // return given misc ingredient from name
    MiscIngredient* misc(const QString &name);
    // add ingredient to database, replacing if asked
    MiscIngredientIterator addMisc(const MiscIngredient &g, bool replace);
    // remove ingredient from database
    void removeMisc(MiscIngredientIterator it);
    // check existance of ingredients
    bool checkMisc(const QString &m);

    // return pointer to current recipe
    Recipe *recipe();

    // is the current recipe modified?
    bool modified();
    // set modified status
    void setModified(bool mod);

    // recalculate recipe
    void recalc();

    // set the data directory
    void setDataDir(const QString &datadir);

    // load the data
    bool loadData(const QString &filename, bool quiet=false);
    // save the data
    bool saveData(const QString &filename);

    // return the recipe as plain text
    QString recipeText();
    // export the recipe as plain text
    bool exportText(const QString &filename);
    // return the recipe as an html string
    QString recipeHTML();
    // export the recipe as html
    bool exportHTML(const QString &filename);
    // export the recipe in BeerXML format
    bool exportBeerXML(const QString &filename);

 signals:
    // a recipe has been loaded or created
    void recipeChanged();
    // send that the document has been modified
    void recipeModified();

 private:
    // Private constructor.
    Model();
    // Private destructor
    ~Model();

    // load native file format
    bool loadNativeRecipe(const QDomElement &root);

    // figure out the base directory for the data
    QString dataBase();

 private:
    static Model *instance_;

    Volume defaultsize_;
    Style defaultstyle_;
    bool defaultmash_;
    QString defaulthopform_;
    Unit *defaultgrainunit_;
    Unit *defaulthopunit_;
    Unit *defaultmiscunit_;

    GrainList graindb_;
    HopList hopdb_;
    MiscIngredientList miscdb_;
    StyleList styledb_;

    Recipe *recipe_;

    QString datadir_;
};

// Inlined Methods ///////////////////////////////////////////////////////////

inline void Model::setDefaultMash(bool mash) { defaultmash_ = mash; }

inline void Model::setDefaultHopform(const QString &f) { defaulthopform_ = f; }

inline const Volume &Model::defaultSize() const { return defaultsize_; }

inline const Style &Model::defaultStyle() const { return defaultstyle_; }

inline bool Model::defaultMash() const { return defaultmash_; }

inline const QString &Model::defaultHopform() const { return defaulthopform_; }

inline const Unit &Model::defaultGrainUnit() const { return *defaultgrainunit_; }

inline const Unit &Model::defaultHopUnit() const { return *defaulthopunit_; }

inline const Unit &Model::defaultMiscUnit() const { return *defaultmiscunit_; }

inline StyleList *Model::styleDB() { return &styledb_; }

inline GrainList *Model::grainDB() { return &graindb_; }

inline HopList *Model::hopDB() { return &hopdb_; }

inline MiscIngredientList *Model::miscDB() { return &miscdb_; }

inline Recipe *Model::recipe() { return recipe_; }

inline bool Model::modified() { return recipe_->modified(); }

inline void Model::recalc() { Calc::recalc(recipe_); }

inline void Model::setDataDir(const QString &datadir) { datadir_ = datadir; }

#endif // MODEL_H
