/***************************************************************************
  style.cpp
  -------------------
  AHA-like style class
  -------------------
  Copyright 1999-2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qregexp.h>
#include <qstringlist.h>
#include "resource.h"

#include "style.h"

using AppResource::CHAR_LATIN_DEGREE;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Style()
// -------
// Constructor

Style::Style(const QString name, const double &oglow, const double &oghi,
             const double &fglow, const double &fghi,
             const int &ibulow, const int &ibuhi,
             const int &srmlow, const int &srmhi)
    : name_(name), oglow_(oglow), oghi_(oghi), fglow_(fglow), fghi_(fghi),
      ibulow_(ibulow), ibuhi_(ibuhi), srmlow_(srmlow), srmhi_(srmhi)
{
    // older qbrewdata files might not have FG values
    if (fglow_ == 0.0) fglow_ = ((oglow_ - 1.0) * 0.25) + 1.0;
    if (fghi_ == 0.0) fghi_ = ((oghi_ - 1.0) * 0.25) + 1.0;
}

//////////////////////////////////////////////////////////////////////////////
// Style(const Style &)
// --------------------
// Copy onstructor

Style::Style(const Style &s)
    : name_(s.name_), oglow_(s.oglow_), oghi_(s.oghi_), fglow_(s.fglow_),
      fghi_(s.fghi_), ibulow_(s.ibulow_), ibuhi_(s.ibuhi_),
      srmlow_(s.srmlow_), srmhi_(s.srmhi_)
{ ; }

//////////////////////////////////////////////////////////////////////////////
// ~Style()
// --------
// Destructor

Style::~Style() {}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// operator=
// ---------
// Assignment operator

Style Style::operator=(const Style &s)
{
    if (&s != this) {
        name_ = s.name_;
        oglow_ = s.oglow_; oghi_ = s.oghi_;
        fglow_ = s.fglow_; fghi_ = s.fghi_;
        ibulow_ = s.ibulow_; ibuhi_ = s.ibuhi_;
        srmlow_ = s.srmlow_; srmhi_ = s.srmhi_;
    }
    return *this;
}

//////////////////////////////////////////////////////////////////////////////
// operator==
// ----------
// Equivalence operator

bool Style::operator==(const Style &s) const
{
    return ( (name_ == s.name_) &&
             (oglow_ == s.oglow_) &&
             (oghi_ == s.oghi_) &&
             (fglow_ == s.fglow_) &&
             (fghi_ == s.fghi_) &&
             (ibulow_ == s.ibulow_) &&
             (ibuhi_ == s.ibuhi_) &&
             (srmlow_ == s.srmlow_) &&
             (srmhi_ == s.srmhi_) );
}

//////////////////////////////////////////////////////////////////////////////
// StyleItem class                                                          //
//////////////////////////////////////////////////////////////////////////////

StyleItem::StyleItem(QListView * parent) : QListViewItem(parent), sit_(0) { ; }

StyleItem::~StyleItem() { ; }

void StyleItem::refresh()
{
    setText(SITEM_NAME, (*sit_).name());
    setText(SITEM_OGLOW, QString::number((*sit_).OGLow(), 'f', 3));
    setText(SITEM_OGHI, QString::number((*sit_).OGHi(), 'f', 3));
    setText(SITEM_FGLOW, QString::number((*sit_).FGLow(), 'f', 3));
    setText(SITEM_FGHI, QString::number((*sit_).FGHi(), 'f', 3));
    setText(SITEM_IBULOW, QString::number((*sit_).IBULow()));
    setText(SITEM_IBUHI, QString::number((*sit_).IBUHi()));
    setText(SITEM_SRMLOW,
            QString::number((*sit_).SRMLow()) + CHAR_LATIN_DEGREE);
    setText(SITEM_SRMHI,
            QString::number((*sit_).SRMHi()) + CHAR_LATIN_DEGREE);
}

// reimplemented for numerical sorting
int StyleItem::compare(QListViewItem* item, int column, bool asc) const
{
    switch (column) {
      case SITEM_OGLOW:
      case SITEM_OGHI:
      case SITEM_FGLOW:
      case SITEM_FGHI:
      case SITEM_SRMLOW:
      case SITEM_SRMHI: { // remove suffixes and decimals in these fields
          QRegExp regex("[0-9.]+");
          regex.search(text(column));
          int first = (int) regex.cap(0).toDouble() * 1000;
          regex.search(item->text(column));
          int second = (int) regex.cap(0).toDouble() * 1000;
          return (first - second);
      }

      default:
          return QListViewItem::compare(item, column, asc);
    }
}
