/***************************************************************************
  styledb.cpp
  -------------------
  A dialog page for styles
  -------------------
  Copyright 2005, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcombobox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qstringlist.h>

#include "fixedspinbox.h"

#include "resource.h"
#include "controller.h"
#include "model.h"
#include "view.h"

#include "styledb.h"

using namespace AppResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// StyleDB()
// ---------
// Constructor

StyleDB::StyleDB(QWidget *parent, const char *name,  Model *model)
    : StyleView(parent, name), model_(model), currentitem_(0)
{
    // additional widget setup
    stylelistview->setColumnAlignment(SITEM_NAME, AlignLeft);
    stylelistview->setColumnAlignment(SITEM_OGLOW, AlignRight);
    stylelistview->setColumnAlignment(SITEM_OGHI, AlignRight);
    stylelistview->setColumnAlignment(SITEM_FGLOW, AlignRight);
    stylelistview->setColumnAlignment(SITEM_FGHI, AlignRight);
    stylelistview->setColumnAlignment(SITEM_IBULOW, AlignRight);
    stylelistview->setColumnAlignment(SITEM_IBUHI, AlignRight);
    stylelistview->setColumnAlignment(SITEM_SRMLOW, AlignRight);
    stylelistview->setColumnAlignment(SITEM_SRMHI, AlignRight);

    oglowspin->setMinimum(1.000);
    oglowspin->setMaximum(1.150);
    oglowspin->setSingleStep(0.001);
    oglowspin->setPrecision(3);

    oghispin->setMinimum(1.000);
    oghispin->setMaximum(1.150);
    oghispin->setSingleStep(0.001);
    oghispin->setPrecision(3);

    fglowspin->setMinimum(0.990);
    fglowspin->setMaximum(1.050);
    fglowspin->setSingleStep(0.001);
    fglowspin->setPrecision(3);

    fghispin->setMinimum(0.990);
    fghispin->setMaximum(1.050);
    fghispin->setSingleStep(0.001);
    fghispin->setPrecision(3);

    srmlowspin->setSuffix(AppResource::CHAR_LATIN_DEGREE);
    srmhispin->setSuffix(AppResource::CHAR_LATIN_DEGREE);

    addbutton->setPixmap(Controller::findImage("sm_add"));
    applybutton->setPixmap(Controller::findImage("sm_apply"));
    removebutton->setPixmap(Controller::findImage("sm_remove"));

    // fixup tab orders
    QWidget::setTabOrder(namecombo, oglowspin);
    QWidget::setTabOrder(oglowspin, oghispin);
    QWidget::setTabOrder(oghispin, fglowspin);
    QWidget::setTabOrder(fglowspin, fghispin);
    QWidget::setTabOrder(fghispin, ibulowspin);
    QWidget::setTabOrder(ibulowspin, ibuhispin);
    QWidget::setTabOrder(ibuhispin, srmlowspin);
    QWidget::setTabOrder(srmlowspin, srmhispin);
    QWidget::setTabOrder(srmhispin, addbutton);

    // setup connections
    connect(addbutton, SIGNAL(clicked()), SLOT(addStyle()));
    connect(applybutton, SIGNAL(clicked()), SLOT(applyStyle()));
    connect(removebutton, SIGNAL(clicked()), SLOT(removeStyle()));
    connect(namecombo, SIGNAL(activated(const QString &)),
            SLOT(updateFields(const QString &)));
    connect(stylelistview, SIGNAL(selectionChanged(QListViewItem *)),
            SLOT(itemSelected(QListViewItem *)));
    // connections to keep lows <= highs
    connect(oglowspin, SIGNAL(valueChanged(double)), this, SLOT(ogChanged()));
    connect(oghispin, SIGNAL(valueChanged(double)), this, SLOT(ogChanged()));
    connect(fglowspin, SIGNAL(valueChanged(double)), this, SLOT(fgChanged()));
    connect(fghispin, SIGNAL(valueChanged(double)), this, SLOT(fgChanged()));
    connect(ibulowspin, SIGNAL(valueChanged(int)), this, SLOT(ibuChanged()));
    connect(ibuhispin, SIGNAL(valueChanged(int)), this, SLOT(ibuChanged()));
    connect(srmlowspin, SIGNAL(valueChanged(int)), this, SLOT(srmChanged()));
    connect(srmhispin, SIGNAL(valueChanged(int)), this, SLOT(srmChanged()));
}

StyleDB::~StyleDB() {}

//////////////////////////////////////////////////////////////////////////////
// Implementation                                                           //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// refresh()
// ---------
// initialize list and fields

void StyleDB::refresh()
{
    // update list
    StyleIterator it;
    StyleItem *item;
    StyleList *list = model_->styleDB();
    stylelistview->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new StyleItem(stylelistview);
        item->setText(SITEM_NAME, (*it).name());
        item->setText(SITEM_OGLOW, QString::number((*it).OGLow(),'f',3));
        item->setText(SITEM_OGHI, QString::number((*it).OGHi(),'f',3));
        item->setText(SITEM_FGLOW, QString::number((*it).FGLow(),'f',3));
        item->setText(SITEM_FGHI, QString::number((*it).FGHi(),'f',3));
        item->setText(SITEM_IBULOW, QString::number((*it).IBULow()));
        item->setText(SITEM_IBUHI, QString::number((*it).IBUHi()));
        item->setText(SITEM_SRMLOW, QString::number((*it).SRMLow()));
        item->setText(SITEM_SRMHI, QString::number((*it).SRMHi()));
        item->setIterator(it);
    }

    // update fields
    namecombo->clear();
    namecombo->insertStringList(model_->stylesList());
    updateFields(namecombo->currentText());
}

//////////////////////////////////////////////////////////////////////////////
// updateFields()
// --------------
// Update the fields in the entry area

void StyleDB::updateFields(const QString &name)
{
    Style *thisstyle = model_->style(name);
    if (thisstyle) { // sanity check
        oglowspin->setValue(thisstyle->OGLow());
        oghispin->setValue(thisstyle->OGHi());
        fglowspin->setValue(thisstyle->FGLow());
        fghispin->setValue(thisstyle->FGHi());
        ibulowspin->setValue(thisstyle->IBULow());
        ibuhispin->setValue(thisstyle->IBUHi());
        srmlowspin->setValue(thisstyle->SRMLow());
        srmhispin->setValue(thisstyle->SRMHi());
    }
}

//////////////////////////////////////////////////////////////////////////////
// addStyle()
// ----------
// Add a style to the recipe

void StyleDB::addStyle()
{
    StyleIterator it;
    StyleItem *item;

    Style style(namecombo->currentText(),
                oglowspin->value(), oghispin->value(),
                fglowspin->value(), fghispin->value(),
                ibulowspin->value(), ibuhispin->value(),
                srmlowspin->value(), srmhispin->value());

    if (model_->checkStyle(style.name())) {
        // already exists, query user
        if (QMessageBox::information(this, ID_TITLE + " - Replace?",
                                     QString("<p>Are you sure you want to "
                                             "replace the entry \"%1\"")
                                     .arg(style.name()),
                                     QMessageBox::No | QMessageBox::Escape,
                                     QMessageBox::Yes)
            == QMessageBox::No) return;

        // remove existing item from listview
        item = (StyleItem*)stylelistview->findItem(style.name(), SITEM_NAME);
        if (item) {
            stylelistview->takeItem(item);
            delete item;
        }
    }

    // create new listview item
    item = new StyleItem(stylelistview);
    item->setText(GITEM_NAME, style.name());
        item->setText(SITEM_OGLOW,QString::number(style.OGLow(),'f',3));
        item->setText(SITEM_OGHI,QString::number(style.OGHi(),'f',3));
        item->setText(SITEM_FGLOW,QString::number(style.FGLow(),'f',3));
        item->setText(SITEM_FGHI,QString::number(style.FGHi(),'f',3));
        item->setText(SITEM_IBULOW, QString::number(style.IBULow()));
        item->setText(SITEM_IBUHI, QString::number(style.IBUHi()));
        item->setText(SITEM_SRMLOW, QString::number(style.SRMLow()));
        item->setText(SITEM_SRMHI, QString::number(style.SRMHi()));

    // add style to database
    it = model_->addStyle(style, true);
    item->setIterator(it);

    stylelistview->ensureItemVisible(item);
    stylelistview->setCurrentItem(item);
}

//////////////////////////////////////////////////////////////////////////////
// applyStyle()
// ------------
// Apply a style to the database

void StyleDB::applyStyle()
{
    if (currentitem_ != 0) {
        Style style(namecombo->currentText(),
                    oglowspin->value(), oghispin->value(),
                    fglowspin->value(), fghispin->value(),
                    ibulowspin->value(), ibuhispin->value(),
                    srmlowspin->value(), srmhispin->value());
        // apply style (overwrite is simplest way)
        StyleIterator it = model_->addStyle(style, true);
        currentitem_->setIterator(it);
        currentitem_->refresh();
    }
}

//////////////////////////////////////////////////////////////////////////////
// removeStyle()
// -------------
// Remove a style from the database

void StyleDB::removeStyle()
{
    if (currentitem_ != 0) {
        model_->removeStyle(currentitem_->iterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// itemSelected()
// --------------
// A style is selected in the listview

void StyleDB::itemSelected(QListViewItem *item)
{
    if (item) {
        currentitem_ = dynamic_cast<StyleItem*>(item);
        if (currentitem_ == 0) {
            qWarning("qbrew: bad dynamic cast, line" + __LINE__);
            return;
        }
        namecombo->setEditText(item->text(SITEM_NAME));
        oglowspin->setValue(item->text(SITEM_OGLOW));
        oghispin->setValue(item->text(SITEM_OGHI));
        fglowspin->setValue(item->text(SITEM_FGLOW));
        fghispin->setValue(item->text(SITEM_FGHI));
        ibulowspin->setValue(item->text(SITEM_IBULOW).toUInt());
        ibuhispin->setValue(item->text(SITEM_IBUHI).toUInt());
        // NOTE: cannot have any suffixes or toUInt() will fail
        srmlowspin->setValue(item->text(SITEM_SRMLOW).toUInt());
        srmhispin->setValue(item->text(SITEM_SRMHI).toUInt());
    }
}

//////////////////////////////////////////////////////////////////////////////
// xxxChanged()
// ------------
// keep low values less than high values

void StyleDB::ogChanged()
{
    oglowspin->setMaximum(oghispin->value());
    oghispin->setMinimum(oglowspin->value());
}

void StyleDB::fgChanged()
{
    fglowspin->setMaximum(fghispin->value());
    fghispin->setMinimum(fglowspin->value());
}

void StyleDB::ibuChanged()
{
    ibulowspin->setMaxValue(ibuhispin->value());
    ibuhispin->setMinValue(ibulowspin->value());
}

void StyleDB::srmChanged()
{
    srmlowspin->setMaxValue(srmhispin->value());
    srmhispin->setMinValue(srmlowspin->value());
}
