/***************************************************************************
                          ractedit.cpp  -  description                              
                             -------------------                                         
    begin                : Wed Apr 28 1999                                           
    copyright            : (C) 1999 by Andreas Mustun                         
    email                : mustun@ribbonsoft.com                                     
 ***************************************************************************/


#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qpixmap.h>

#include "rappwin.h"
#include "ractedit.h"
#include "rattribdialog.h"
#include "rbehaviordef.h"
#include "rfontdialog.h"
#include "rgraphic.h"
#include "rinputdialog.h"
#include "rinputpanel.h"
#include "rconfig.h"
#include "rlibrarybrowser.h"
#include "rlog.h"
#include "rmath.h"
#include "rparameterdialog.h"
#include "rsnap.h"
#include "rstatuspanel.h"

#include "xpm/helpeditmove.xpm"
#include "xpm/helpeditrotate.xpm"
#include "xpm/helpeditscale.xpm"
#include "xpm/helpeditmirror.xpm"
#include "xpm/helpeditmoverotate.xpm"
#include "xpm/helpeditrotaterotate.xpm"

#include "xpm/roundtrimon.xpm"
#include "xpm/roundtrimoff.xpm"
#include "xpm/beveltrimon.xpm"
#include "xpm/beveltrimoff.xpm"


/*! \class RactEdit
    \brief Action handling for edit fuinctions
    \author Andreas Mustun
*/


/*! Constructor
*/
RActEdit::RActEdit()
{

}



/*! Destructor
*/
RActEdit::~RActEdit()
{
}



/*! Serves an event to the corresponding function
    \param _action The action (e.g. ACT_EDIT_MOVE)
    \param _rEvent The event (e.g. REVENT_BEGIN)
*/
void
RActEdit::serveEvent(int _action,
                     int _rEvent )
{
  if(_action>=ACT_EDIT_FIRST &&
     _action<=ACT_EDIT_LAST     ) {

    /*
    RLOG("\nserveEvent: ");
    RLOG(_action);
    RLOG(" ");
    RLOG(_rEvent);
    */

    switch(_action) {

      case ACT_EDIT_MOVE:
        move(_rEvent);
        break;

      case ACT_EDIT_ROTATE:
        rotate(_rEvent);
        break;

      case ACT_EDIT_SCALE:
        scale(_rEvent);
        break;

      case ACT_EDIT_MIRROR:
        mirror(_rEvent);
        break;

      case ACT_EDIT_MOVEROTATE:
        moveRotate(_rEvent);
        break;

      case ACT_EDIT_ROTATEROTATE:
        rotateRotate(_rEvent);
        break;

      case ACT_EDIT_TRIM:
        trim(_rEvent);
        break;

      case ACT_EDIT_TRIM2:
        trim2(_rEvent);
        break;

      case ACT_EDIT_TRIM3:
        trim3(_rEvent);
        break;

      case ACT_EDIT_CUT:
        cut(_rEvent);
        break;

      case ACT_EDIT_CUT2:
        cut2(_rEvent);
        break;

      case ACT_EDIT_STRETCH:
        stretch(_rEvent);
        break;

      case ACT_EDIT_ROUND:
        round(_rEvent);
        break;

      case ACT_EDIT_BEVEL:
        bevel(_rEvent);
        break;

      case ACT_EDIT_TEXTELEMENTS:
        textElements(_rEvent);
        break;

      case ACT_EDIT_TEXT:
        text(_rEvent);
        break;

      case ACT_EDIT_DELETE:
        del(_rEvent);
        break;

      case ACT_EDIT_LAYER:
        layer(_rEvent);
        break;

      case ACT_EDIT_ZERO:
        zero(_rEvent);
        break;

      case ACT_EDIT_ATTRIB:
        attrib(_rEvent);
        break;

      case ACT_EDIT_CLIP:
        clip(_rEvent);
        break;

      case ACT_EDIT_CLIPCOPY:
        clipCopy(_rEvent);
        break;

      case ACT_EDIT_CLIPCUT:
        clipCut(_rEvent);
        break;

      case ACT_EDIT_CLIPPASTE:
        clipPaste(_rEvent);
        break;

      case ACT_EDIT_INSERTPART:
        insertPart(_rEvent);
        break;

      default:
        break;
    }
  }
}



/*! Lets the user move elements
    <p>States:
    <ul>
      <li>1: Set reference pos
      <li>2: Set target pos
    </ul>
*/
void
RActEdit::move(int _rEvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first point
  static double cy1=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Reference point"),
                                     tr("R: Back"),
                                     tr("Move objects"));

      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Angle:"), false);
      inputPanel()->setValue(0, 0.0);
      inputPanel()->show();
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting reference point:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting target point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        double pAngle = inputPanel()->getValue(0);

        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->previewMoved(px, py, pAngle)) {
            currentDoc()->drawPreview(px, py, pAngle);
          }
        }

        currentDoc()->setPreviewUpdate(true);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set reference point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          currentDoc()->fillPreviewWithTagged(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Target point"));
          currentDoc()->setCurrentState(2);

          /*inputPanel()->reset();
          inputPanel()->hide();
          inputPanel()->addLineEdit(tr("Angle:"), false);
          inputPanel()->setValue(0, 0.0);
          inputPanel()->show();*/
        }
      }

      // Set target point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          currentDoc()->setPreviewUpdate(false);

          double pAngle = inputPanel()->getValue(0);

          // Show move Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Move / Copy"),
                                 *(new QPixmap(helpeditmove_xpm)));

          dlg->addEntry(tr("Number of copies (n):"), copies);

          if(dlg->exec()) {
            copies=dlg->getIntInput(0);
            currentDoc()->editMove(copies, cx1, cy1, px-cx1, py-cy1, pAngle, false, true);
          }
          delete dlg;
        }
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Reference point"));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user rotate elements
    <p>States:
    <ul>
      <li>1: Set center
    </ul>

*/
void
RActEdit::rotate(int _rEvent)
{
  static double cx=DEF_AREAMAX;  // Coordinates of center point
  static double cy=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies
  static double angle=30.0;      // Angle of rotation

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Rotation center"),
                                     tr("R: Back"),
                                     tr("Rotate objects"));
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting rotation center:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:

      // Set rotation center:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx, &cy, true)) {
          currentDoc()->moveRelZero(cx, cy);
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);
          currentDoc()->fillPreviewWithTagged(0.0, 0.0);
          statusPanel()->setLeftButtonStatus(tr("L: Rotation center"));


          // Show rotate Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Rotate"),
                                 *(new QPixmap(helpeditrotate_xpm)));

          dlg->addEntry(tr("Angle:"), angle);
          dlg->addEntry(tr("Number of copies (n):"), copies);

          if(dlg->exec()) {
            angle=dlg->getFloatInput(0);
            copies=dlg->getIntInput(1);
            currentDoc()->editRotate(copies, angle, cx, cy, true);
          }

          delete dlg;
        }
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      break;

    default:
      break;
  }

}



/*! Lets the user scale elements
    <p>States:
    <ul>
      <li>1: Set scaling center
    </ul>
*/
void
RActEdit::scale(int _rEvent)
{
  static double cx=DEF_AREAMAX;  // Coordinates of scaling center
  static double cy=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies
  static double factor=1.5;      // Factor for scaling

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Scaling center"),
                                     tr("R: Back"),
                                     tr("Scale objects"));
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting rotation center:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:

      // Set scaling center:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx, &cy, true)) {
          currentDoc()->moveRelZero(cx, cy);
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);

          currentDoc()->fillPreviewWithTagged(0.0, 0.0);

          // Show scaling Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Scale"),
                                 *(new QPixmap(helpeditscale_xpm)));
          dlg->addEntry(tr("Factor:"), factor);
          dlg->addEntry(tr("Number of copies (n):"), copies);
          if(dlg->exec()) {
            factor=dlg->getFloatInput(0);
            copies=dlg->getIntInput(1);
            currentDoc()->editScale(copies, factor, cx, cy, true);
          }
          delete dlg;
        }
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      break;

    default:
      break;
  }

}



/*! Lets the user mirror elements
    <p>States:
    <ul>
      <li>1: Set 1st point of axis
      <li>2: Set 2nd point on axis
    </ul>
*/
void
RActEdit::mirror(int _rEvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first (last) point
  static double cy1=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: First point of mirror axis"),
                                     tr("R: Back"),
                                     tr("Mirror objects"));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd point :
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE, cx1, cy1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point of axis:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Second point of mirror axis"));

          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd point of axis:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);

          currentDoc()->fillPreviewWithTagged(0.0, 0.0);

          // Show mirroring Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Mirror"),
                                 *(new QPixmap(helpeditmirror_xpm)));
          dlg->addEntry(tr("Copies (0/1):"), copies);
          if(dlg->exec()) {
            copies=dlg->getIntInput(0);
            currentDoc()->editMirror(copies, cx1, cy1, px, py, true);
          }
          delete dlg;

          cx1=px;
          cy1=py;
        }
        statusPanel()->setLeftButtonStatus(tr("L: First point of mirror axis"));
        currentDoc()->setCurrentState(1);
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();

      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: First point of mirror axis"));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user move and rotate elements at once
    <p>States:
    <ul>
      <li>1: Set reference point for moving
      <li>2: Set relative reference point for rotating
      <li>3: Set target point
    </ul>
*/
void
RActEdit::moveRotate(int _rEvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of 1st reference point (movement)
  static double cy1=DEF_AREAMAX;  //
  static double cx2=DEF_AREAMAX;  // Coordinates of 2nd reference point (rotation)
  static double cy2=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies
  static double angle=0.0;       // Angle of rotation

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Reference point for movement"),
                                     tr("R: Back"),
                                     tr("Move and rotate object"));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st reference point (movement) is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move before 2nd reference point (rotation) is set:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapPoint();
      }

      // Move for setting target point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        double px, py;

        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->previewMoved(px, py)) {
            currentDoc()->drawPreview(px, py);
          }
        }

        currentDoc()->setPreviewUpdate(true);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st reference point (movement):
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          currentDoc()->fillPreviewWithTagged(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Referencepoint for rotation"));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd reference point (rotation):
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&cx2, &cy2, true)) {
          currentDoc()->moveRelZero(cx2,cy2);
          statusPanel()->setLeftButtonStatus(tr("L: Target point"));
          currentDoc()->setCurrentState(3);
        }
      }

      // Set target point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);

          // Show move rotate Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Move and rotate"),
                                 *(new QPixmap(helpeditmoverotate_xpm)));
          dlg->addEntry(tr("Angle:"), angle);
          dlg->addEntry(tr("Number of copies (n):"), copies);
          if(dlg->exec()) {
            angle=dlg->getFloatInput(0);
            copies=dlg->getIntInput(1);
            currentDoc()->editMoveRotate(copies,
                                         angle,
                                         px-cx1, py-cy1,
                                         cx1, cy1, cx2, cy2,
                                         true);
          }
          delete dlg;
        }
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st reference point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st reference point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Reference point for movement"));
      }

      // Go back to setting of 2nd reference point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Referencepoint for rotation"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user rotate elements around two centers
    <p>States:
    <ul>
      <li>1: Set reference point for rotation
      <li>2: Set reference point for relative rotation
    </ul>
*/
void
RActEdit::rotateRotate(int _rEvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of 1st reference point (big rotation)
  static double cy1=DEF_AREAMAX;  //
  static int copies=1;          // Number of copies
  static double angle1;          // Angle of big rotation
  static double angle2;          // Angle of small rotation

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Reference point for the absolute rotation"),
                                     tr("R: Back"),
                                     tr("Rotate objects around two centers"));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st reference point (rotation 1 (big)) is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move before 2nd reference point (rotation 2 (small)) is set:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapPoint();
      }

      break;

    case REVENT_LBUTTONUP:

      // Set 1st reference point (rotation 1 (big)):
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          statusPanel()->setLeftButtonStatus(tr("L: Reference point for the reative rotation"));

          currentDoc()->setCurrentState(2);
        }
      }

      // Set 1st reference point (rotation 2 (small)):
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);

          snapper()->stop();

          currentDoc()->fillPreviewWithTagged(px, py);
          currentDoc()->setPreviewUpdate(false);

          // Show move rotate Dialog:
          //
          RParameterDialog* dlg =
            new RParameterDialog(qApp->mainWidget(),
                                 tr("Rotate and Counter-Rotate"),
                                 *(new QPixmap(helpeditrotaterotate_xpm)));
          dlg->addEntry(tr("Angle a"), angle1);
          dlg->addEntry(tr("Angle b"), angle2);
          dlg->addEntry(tr("Number of copies (n):"), copies);
          if(dlg->exec()) {
            angle1=dlg->getFloatInput(0);
            angle2=dlg->getFloatInput(1);
            copies=dlg->getIntInput(2);
            currentDoc()->editRotateRotate(copies,
                                           angle1, angle2,
                                           cx1, cy1, px, py,
                                           true);
          }
          delete dlg;
        }
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st reference point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st reference point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Reference point for the absolute rotation"));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user trim elements
    <p>States:
    <ul>
      <li>1: Set trim element
      <li>2: Set limit element
    </ul>
*/
void
RActEdit::trim(int _rEvent)
{
  static RElement* trimEl=0;             // Trim element
  static double mx1=0.0;
  static double my1=0.0;                  // 1st mouse pos

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Trim element"),
                                     tr("R: Back"),
                                     tr("Trim objects"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }
      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_ALL, trimEl);
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set trim element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          trimEl = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx1 = currentDoc()->getRealMousePosX();
          my1 = currentDoc()->getRealMousePosY();

          currentDoc()->setHighlightFilter(T_ALL, trimEl);
          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Limit element"));

        }
      }

      else if(currentDoc()->getCurrentState()==2) {

        // Set limit element / do trimming:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          // The limit element:
          //
          RElement* limitEl = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          double mx2 = currentDoc()->getRealMousePosX();
          double my2 = currentDoc()->getRealMousePosY();

          // Variables for the intersection(s):
          //   ix2/iy2 is only used for full circles (trim two points)
          //
          bool i1=false;
          double ix1=DEF_AREAMAX, iy1=DEF_AREAMAX;
          bool i2=false;
          double ix2=DEF_AREAMAX, iy2=DEF_AREAMAX;
          bool itan=false;

          // Angle from center to mouse curs.
          //
          double mang=0.0;

          // Get intersection point:
          //
          trimEl->getIntersection(limitEl,
                                  &i1, &ix1, &iy1,
                                  &i2, &ix2, &iy2,
                                  &itan,
                                  false);


          if(i1) {

            // We have two intersections:
            //
            if(i2 && !itan) {
              // Distance from 2nd click to intersection 1:
              //
              double dis1 = mtGetDistance(mx2, my2,
                                         ix1, iy1);
              // Distance from 2nd click to intersection 2:
              //
              double dis2 = mtGetDistance(mx2, my2,
                                         ix2, iy2);

              // Which intersection is next to the 2nd klick?
              //
              if(dis1>dis2) {
                double dum=ix1;
                ix1 = ix2;
                ix2 = dum;
                dum = iy1;
                iy1=iy2;
                iy2=dum;
              }
            }

            // Get angle from circle's center to 1st mouse click:
            //
            if(trimEl->isInGroup(T_CIRCLE)) {
              mang = mtGetAngle(trimEl->getCx(),
                                trimEl->getCy(),
                                mx1, my1);
            }

            bool endPoint = trimEl->getTrimPoint(mx1, my1,
                                                 ix1, iy1);

            currentDoc()->editTrim(trimEl,
                                   endPoint,
                                   ix1, iy1,
                                   ix2, iy2,
                                   mang,
                                   true);
          }

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: Trim element"));
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: Trim element"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user trim two elements at once
    <p>States:
    <ul>
      <li>1: Set 1st trim element
      <li>2: Set 2nd trim element
    </ul>
*/
void
RActEdit::trim2(int _rEvent)
{
  static RElement* trimEl1=0;            // 1st Trim element
  static double mx1=0.0, my1=0.0;         // 1st mouse pos

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: 1st Trim element"),
                                     tr("R: Back"),
                                     tr("Trim two objects"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }
      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_ALL, trimEl1);
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set 1st trim element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          trimEl1 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx1 = currentDoc()->getRealMousePosX();
          my1 = currentDoc()->getRealMousePosY();

          currentDoc()->setHighlightFilter(T_ALL, trimEl1);
          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: 2nd Trim element"));

        }
      }

      else if(currentDoc()->getCurrentState()==2) {

        // Set 2nd trim element / do trimming:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          // The 2nd trim element:
          //
          RElement* trimEl2 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          double mx2 = currentDoc()->getRealMousePosX();
          double my2 = currentDoc()->getRealMousePosY();

          // Variables for the intersection(s):
          //   ix2/iy2 is only used for full circles (trim two points)
          //
          bool i1=false;
          double ix1=0.0, iy1=0.0;
          bool i2=false;
          double ix2=0.0, iy2=0.0;
          bool itan=false;

          // Angle from center to mouse curs.
          //
          double mang1=0.0;
          double mang2=0.0;

          // Get intersection point:
          //
          trimEl1->getIntersection(trimEl2,
                                   &i1, &ix1, &iy1,
                                   &i2, &ix2, &iy2,
                                   &itan,
                                   false);

          if(i1) {

            // We have two intersections:
            //
            if(i2 && !itan) {
              // Distance from 2nd click to intersection 1:
              //
              double dis1 = mtGetDistance(mx2, my2,
                                         ix1, iy1);
              // Distance from 2nd click to intersection 2:
              //
              double dis2 = mtGetDistance(mx2, my2,
                                         ix2, iy2);

              // Which intersection is next to the 2nd klick?
              //
              if(dis1>dis2) {
                double dum=ix1;
                ix1 = ix2;
                ix2 = dum;
                dum = iy1;
                iy1=iy2;
                iy2=dum;
              }

              // Get angle from 1st circle's center to 1st mouse click:
              //
              if(trimEl1->isInGroup(T_CIRCLE)) {
                mang1 = mtGetAngle(trimEl1->getCx(),
                                   trimEl1->getCy(),
                                   mx1, my1);
              }

              // Get angle from 2nd circle's center to 2nd mouse click:
              //
              if(trimEl2->isInGroup(T_CIRCLE)) {
                mang2 = mtGetAngle(trimEl2->getCx(),
                                   trimEl2->getCy(),
                                   mx2, my2);
              }
            }

            // Trim 1st element:
            //
            bool endPoint = trimEl1->getTrimPoint(mx1, my1,
                                                  ix1, iy1);

            currentDoc()->editTrim(trimEl1,
                                   endPoint,
                                   ix1, iy1,
                                   ix2, iy2,
                                   mang1,
                                   true);
            // Trim 2nd element:
            //
            endPoint = trimEl2->getTrimPoint(mx2, my2,
                                             ix1, iy1);

            currentDoc()->editTrim(trimEl2,
                                   endPoint,
                                   ix1, iy1,
                                   ix2, iy2,
                                   mang2,
                                   true);
          }

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: 1st Trim element"));
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: 1st Trim element"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user trim elements by a given amount
    <p>States:
    <ul>
      <li>1: Set element
    </ul>
*/
void
RActEdit::trim3(int _myREvent)
{
  static double amount=1.0;


  switch(_myREvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Trim element"),
                                     tr("R: Back"),
                                     tr("Trim lines by a given amount"));
      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Amount:"), false);
      inputPanel()->show();
      inputPanel()->setValue(0, amount);
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);

        // Preview nothing:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          // No preview
        }
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Draw trimmed amount:
        //
        RElement* el;
        if(snapper() &&
           (el=snapper()->getSnappedElement())) {

          // Get amount from input panel:
          //
          amount = inputPanel()->getValue(0);

          snapper()->stop();

          bool endP=false;
          double mx, my;
          currentDoc()->getRealMousePos( &mx, &my );

          // Trim endpoint?
          if( mtGetDistance( mx, my, el->getX1(), el->getY1() ) >
              mtGetDistance( mx, my, el->getX2(), el->getY2() )    ) {
            endP=true;
          }
          currentDoc()->editTrim3( el, amount, endP, true );
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}



/*! Lets the user cut elements
    <p>States:
    <ul>
      <li>1: Set element
      <li>2: Set cutting point
    </ul>
*/
void
RActEdit::cut(int _rEvent)
{
  static RElement* cutEl=0;            // Cut element

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Element"),
                                     tr("R: Back"),
                                     tr("Divide"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set cut element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          cutEl = snapper()->getSnappedElement();

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Dividing point"));
        }
      }


      else if(currentDoc()->getCurrentState()==2) {

        // Set cut point:
        //
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();

          // Is cutting point on element:
          //
          if(cutEl->isPointOnElement(px, py)) {

            // Cut element:
            //
            currentDoc()->editCut(cutEl,
                                  px, py,
                                  true);
          }

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: Element"));
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: Element"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user cut elements between two points
    <p>States:
    <ul>
      <li>1: Set element
      <li>2: Set cutting point 1
      <li>3: Set cutting point 2
    </ul>
*/
void
RActEdit::cut2(int _rEvent)
{
  static RElement* cutEl=0;            // Cut element
  static double px1=0.0, py1=0.0;
  double px2, py2;

  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Element"),
                                     tr("R: Back"),
                                     tr("Divide"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapPoint();
      }
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set cut element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          cutEl = snapper()->getSnappedElement();

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: First dividing point"));
        }
      }
      else if(currentDoc()->getCurrentState()==2) {
        px1=0.0;
        py1=0.0;

        // Set first cut point:
        //
        if(snapper()->snapPoint(&px1, &py1, true)) {
          snapper()->stop();
          // Is cutting point on element:
          //
          if(cutEl->isPointOnElement(px1, py1)) {
            currentDoc()->setCurrentState(3);
            statusPanel()->setLeftButtonStatus(tr("L: Second dividing point"));
          }
        }
      }
      else if(currentDoc()->getCurrentState()==3) {
        // Set second cut point:
        //
        if(snapper()->snapPoint(&px2, &py2, true)) {
          snapper()->stop();

          // Is cutting point on element:
          //
          if(cutEl->isPointOnElement(px2, py2)) {

            // Cut element:
            //
            currentDoc()->editCut2(cutEl,
                                  px1, py1,
                                  px2, py2,
                                  true);
          }

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: Element"));
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: Element"));
      }
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(tr("L: First dividing point"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user stretch elements
    <p>States:
    <ul>
      <li>1: Set 1st corner of area
      <li>2: Set 2nd corner of area
      <li>3: Set reference point

      <li>4: Set target point
    </ul>
*/
void
RActEdit::stretch(int _rEvent)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first corner
  static double cy1=DEF_AREAMAX;  //
  static double cx2=DEF_AREAMAX;  // Coordinates of second corner
  static double cy2=DEF_AREAMAX;  //
  static double rx=DEF_AREAMAX;  // Coordinates of reference point
  static double ry=DEF_AREAMAX;  //


  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: First corner of the range"),
                                     tr("R: Back"),
                                     tr("Stretch"));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st corner is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd corner:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_RECT, cx1, cy1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting reference point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->snapPoint();
      }

      // Move for setting target point:
      //
      else if(currentDoc()->getCurrentState()==4) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st corner:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->moveRelZero(cx1, cy1);
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Second corner of the range"));
        }
      }

      // Set 2nd corner:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&cx2, &cy2, true)) {
          currentDoc()->moveRelZero(cx2, cy2);
          snapper()->stop();
          currentDoc()->setCurrentState(3);
          statusPanel()->setLeftButtonStatus(tr("L: Reference point"));
        }
        snapper()->snapPoint();
      }

      // Set reference point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        if(snapper()->snapPoint(&rx, &ry, true)) {
          currentDoc()->moveRelZero(rx, ry);
          snapper()->stop();
          currentDoc()->setCurrentState(4);
          statusPanel()->setLeftButtonStatus(tr("L: Target point"));
        }
        snapper()->snapPoint();
      }

      // Set target point:
      //
      else if(currentDoc()->getCurrentState()==4) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();

          // Stretch:
          //
          currentDoc()->tagRange(cx1, cy1, cx2, cy2,
                                 true, true, true);

          currentDoc()->editStretch(px-rx, py-ry, true);

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: First corner of the range"));
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st corner is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st corner:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: First corner of the range"));
      }

      // Go back to setting of 2nd corner:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Second corner of the range"));
      }

      // Go back to setting of reference point:
      //
      else if(currentDoc()->getCurrentState()==4) {
        snapper()->stop();
        currentDoc()->setCurrentState(3);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: Reference point"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user round edges
    <p>States:
    <ul>
      <li>1: Set 1st rounding element
      <li>2: Set 2nd rounding element
      <li>3: Set rounding
    </ul>
*/
void
RActEdit::round(int _rEvent)
{
  static RElement* roundEl1=0;            // 1st Round element
  static RElement* roundEl2=0;            // 2nd Round element
  static double mx1=0.0, my1=0.0;         // 1st clicked mouse pos (for trimming)
  static double mx2=0.0, my2=0.0;         // 2nd clicked mouse pos (for trimming)
  static double mang1=0.0;                // Angle from center of el1 to mouse curs. (circles)
  static double mang2=0.0;                // Angle from center of el2 to mouse curs. (circles)
  static bool   endPoint1=false;          // Trim Endpoint of 1st element

  static bool   endPoint2=false;          // Trim Endpoint of 2nd element
  static bool   ires[8];                  // Are there intersection / which?
  static double ipx[8];                   // Intersection points of parallels (possible centers for rounding)
  static double ipy[8];                   // "
  static double radius=10.0;              // Round Radius
  static bool   doTrim=true;              // Trim?

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: 1st round element"),
                                     tr("R: Back"),
                                     tr("Round"));

      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Radius:"), false);
      inputPanel()->setValue(0, radius);
      inputPanel()->newGroup(tr("Trimming"));
      inputPanel()->addRadioButton(QPixmap(roundtrimon_xpm), tr("Trimming on"), 0, 2);
      inputPanel()->addRadioButton(QPixmap(roundtrimoff_xpm), tr("Trimming off"), 1, 2);
      inputPanel()->endGroup();
      if(doTrim) inputPanel()->setCheckedRadioButton(0);
      else       inputPanel()->setCheckedRadioButton(1);

      inputPanel()->show();

      RLOG("\nBegin rounding");
      break;

    case REVENT_MOUSEMOVE:

      // Set 1st round element:
      //
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      }

      // Set 2nd round element:
      //
      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_ALL, roundEl1);
      }

      // Set rounding:
      //
      else if(currentDoc()->getCurrentState()==3) {
        // Find parallel intersection next to mouse cursor:
        //
        int i;                          // Counter
        int cRound=-1;                  // Found index to round
        double minDist = DEF_AREAMAX;    // Smalles distance found so far
        double dist;                     // Current distance
        double mx, my;                   // Real mouse pos (no snapping)

        currentDoc()->getRealMousePos(&mx, &my);

        round(REVENT_RECALCULATE);

        for(i=0; i<8; ++i) {
          dist = mtGetDistance(mx, my, ipx[i], ipy[i]);
          if(dist<minDist) {
            minDist = dist;
            cRound = i;
          }
        }

        if(currentDoc()->setPreviewElement(T_CIRCLE, ipx[cRound], ipy[cRound], radius)) {
          currentDoc()->drawPreviewElement();
        }
      }

      break;

    case REVENT_RECALCULATE:
      // Get radius from input panel:
      //
      radius = inputPanel()->getValue(0);

      roundEl1->getParallelIntersections(roundEl2,
                                         radius,
                                         &ires[0], &ipx[0], &ipy[0]);
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set 1st round element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          roundEl1 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx1 = currentDoc()->getRealMousePosX();
          my1 = currentDoc()->getRealMousePosY();

          currentDoc()->setHighlightFilter(T_ALL, roundEl1);
          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: 2nd round element"));

        }
      }

      else if(currentDoc()->getCurrentState()==2) {

        // Set 2nd round element / prepare trimming:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          // The 2nd round element:
          //
          roundEl2 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx2 = currentDoc()->getRealMousePosX();
          my2 = currentDoc()->getRealMousePosY();

          // Variables for the intersection(s):
          //
          bool i1=false;
          double ix1=0.0, iy1=0.0;
          bool i2=false;
          double ix2=0.0, iy2=0.0;
          bool itan=false;

          // Get intersection point:
          //
          roundEl1->getIntersection(roundEl2,
                                    &i1, &ix1, &iy1,
                                    &i2, &ix2, &iy2,
                                    &itan,
                                    false);

          if(!i1) {
            roundEl1->getGliding(roundEl2,
                                 &i1, &ix1, &iy1);
            i2=false;
            ix2=iy2=DEF_AREAMAX;
            itan=false;
          }

          round(REVENT_RECALCULATE);

          if(i1) {

            // We have two intersections:
            //
            if(i2 && !itan) {
              // Distance from 2nd click to intersection 1:
              //
              double dis1 = mtGetDistance(mx2, my2,
                                         ix1, iy1);
              // Distance from 2nd click to intersection 2:
              //
              double dis2 = mtGetDistance(mx2, my2,
                                         ix2, iy2);

              // Which intersection is next to the 2nd klick?
              //
              if(dis1>dis2) {
                double dum=ix1;
                ix1 = ix2;
                ix2 = dum;
                dum = iy1;
                iy1=iy2;
                iy2=dum;
              }

              // Get angle from 1st circle's center to 1st mouse click:
              //
              if(roundEl1->isInGroup(T_CIRCLE)) {
                mang1 = mtGetAngle(roundEl1->getCx(),
                                   roundEl1->getCy(),
                                   mx1, my1);
              }

              // Get angle from 2nd circle's center to 2nd mouse click:
              //
              if(roundEl2->isInGroup(T_CIRCLE)) {
                mang2 = mtGetAngle(roundEl2->getCx(),
                                   roundEl2->getCy(),
                                   mx2, my2);
              }
            }

            // Prepare trimming 1st element:
            //
            endPoint1 = roundEl1->getTrimPoint(mx1, my1,
                                               ix1, iy1);

            // Prepare trimming 2nd element:
            //
            endPoint2 = roundEl2->getTrimPoint(mx2, my2,
                                               ix1, iy1);

            currentDoc()->setCurrentState(3);
            statusPanel()->setLeftButtonStatus(tr("L: Set rounding"));
          }
        }
      }

      else if(currentDoc()->getCurrentState()==3) {

        // Find parallel intersection next to mouse cursor:
        //
        int i;                          // Counter
        int cRound=-1;                  // Found index to round
        double minDist = DEF_AREAMAX;    // Smalles distance found so far
        double dist;                     // Current distance
        double mx, my;                   // Real mouse pos (no snapping)

        currentDoc()->getRealMousePos(&mx, &my);

        for(i=0; i<8; ++i) {
          dist = mtGetDistance(mx, my, ipx[i], ipy[i]);
          if(dist<minDist) {
            minDist = dist;
            cRound = i;
          }
        }

        currentDoc()->delPreviewElement();
        currentDoc()->resetPreviewElement();

        if(cRound>=0 && cRound<=7) {
          doTrim = (inputPanel()->isRadioButtonChecked(0)==1 ? true : false);
          currentDoc()->editRound(roundEl1,
                                  roundEl2,
                                  radius,
                                  doTrim,
                                  endPoint1,
                                  endPoint2,
                                  ipx[cRound], ipy[cRound],
                                  mang1, mang2,
                                  true);
        }

        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: 1st round element"));
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->delPreviewElement();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: 1st round element"));
      }

      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(tr("L: 2nd round element"));
      }
      break;

    default:
      break;
  }

}



/*! Lets the user bevelling elements
    <p>States:
    <ul>
      <li>1: Set 1st bevel element
      <li>2: Set 2nd bevel element
    </ul>
*/
void
RActEdit::bevel(int _rEvent)
{
  static RElement* bevelEl1=0;           // 1st Bevel element
  static RElement* bevelEl2=0;           // 2nd Bevel element
  static double mx1=0.0, my1=0.0;         // 1st clicked mouse pos (for trimming)
  static double mx2=0.0, my2=0.0;         // 2nd clicked mouse pos (for trimming)
  static bool  endPoint1=false;          // Trim Endpoint of 1st element
  static bool  endPoint2=false;          // Trim Endpoint of 2nd element
  static double dist1=10.0;               // Bevel Distance 1
  static double dist2=10.0;               // Bevel Distance 2
  static bool  doTrim=true;                // Trim?

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: 1st element"),
                                     tr("R: Back"),
                                     tr("Bevel"));
      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Dist. 1"), false);
      inputPanel()->addLineEdit(tr("Dist. 2"), false);
      inputPanel()->setValue(0, dist1);
      inputPanel()->setValue(1, dist2);

      inputPanel()->newGroup(tr("Trimming"));
      inputPanel()->addRadioButton(QPixmap(beveltrimon_xpm), tr("Trimming on"), 0, 2);
      inputPanel()->addRadioButton(QPixmap(beveltrimoff_xpm), tr("Trimming off"), 1, 2);
      inputPanel()->endGroup();
      if(doTrim) inputPanel()->setCheckedRadioButton(0);
      else       inputPanel()->setCheckedRadioButton(1);

      inputPanel()->show();

      RLOG("\nBegin beveling");
      break;

    case REVENT_MOUSEMOVE:

      // Set 1st bevel element:
      //
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE_ARC);
      }

      // Set 2nd bevel element:
      //
      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE_ARC, bevelEl1);
      }

      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set 1st bevel element:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          snapper()->stop();

          bevelEl1 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx1 = currentDoc()->getRealMousePosX();
          my1 = currentDoc()->getRealMousePosY();

          currentDoc()->setHighlightFilter(T_ALL, bevelEl1);
          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: 2nd element"));

        }
      }

      else if(currentDoc()->getCurrentState()==2) {

        // Set 2nd bevel element / prepare trimming:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

          // The 2nd bevel element:
          //
          bevelEl2 = snapper()->getSnappedElement();

          // Get Mouse pos:
          //
          mx2 = currentDoc()->getRealMousePosX();
          my2 = currentDoc()->getRealMousePosY();

          // Variables for the intersection(s):
          //   ix2/iy2 is only used for full circles (trim two points)
          //
          bool i1=false;
          double ix1=0.0, iy1=0.0;
          bool i2=false;
          double ix2=0.0, iy2=0.0;
          bool itan=false;

          // Get intersection point:
          //
          bevelEl1->getIntersection(bevelEl2,
                                    &i1, &ix1, &iy1,
                                    &i2, &ix2, &iy2,
                                    &itan,
                                    false);

          if(!i1) {
            bevelEl1->getGliding(bevelEl2,
                                 &i1, &ix1, &iy1);
            i2=false;
            ix2=iy2=DEF_AREAMAX;
            itan=false;
          }

          if(i1) {

            // We have two intersections:
            //
            if(i2 && !itan) {
              // Distance from 2nd click to intersection 1:
              //
              double dis1 = mtGetDistance(mx2, my2,
                                         ix1, iy1);
              // Distance from 2nd click to intersection 2:
              //
              double dis2 = mtGetDistance(mx2, my2,
                                         ix2, iy2);

              // Which intersection is next to the 2nd klick?
              //
              if(dis1>dis2) {
                double dum=ix1;
                ix1 = ix2;
                ix2 = dum;
                dum = iy1;
                iy1=iy2;
                iy2=dum;
              }

            }

            // Prepare trimming 1st element:
            //
            endPoint1 = bevelEl1->getTrimPoint(mx1, my1,
                                               ix1, iy1);

            // Prepare trimming 2nd element:
            //
            endPoint2 = bevelEl2->getTrimPoint(mx2, my2,
                                               ix1, iy1);


            // Get distances from input panel:
            //
            dist1 = inputPanel()->getValue(0);
            dist2 = inputPanel()->getValue(1);

            // Get trim attribut from input panel:
            //
            doTrim = (inputPanel()->isRadioButtonChecked(0)==1 ? true : false);

            currentDoc()->editBevel(bevelEl1,
                                    bevelEl2,
                                    dist1,
                                    dist2,
                                    doTrim,
                                    endPoint1,
                                    endPoint2,
                                    ix1, iy1,
                                    true);

            currentDoc()->setCurrentState(1);
            statusPanel()->setLeftButtonStatus(tr("L: 1st element"));
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->delPreviewElement();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }

      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(tr("L: 1st element"));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user convert text/hatches to elements
*/
void
RActEdit::textElements(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->editTextElements(true);
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:

      break;
  }
}



/*! Lets the user edit texts
    <p>States:
    <ul>
      <li>1: Set text
    </ul>
*/
void
RActEdit::text(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setHighlightFilter(T_TEXT);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("Text to edit"),
                                     tr("R: Back"),
                                     tr("Edit text"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_TEXT);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set the text element:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()) {
          if(snapper()->snapElement(true, T_TEXT)) {
            currentDoc()->editText(snapper()->getSnappedElement(), true);
          	currentDoc()->repaint();
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}



/*! Lets the user delete elements
    <p>States:
    <ul>
      <li>1: Set point
    </ul>
*/
void
RActEdit::del(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->editDelete(true);
      currentDoc()->resetCurrentState();
      currentDoc()->resetCurrentAction();

      if( RCONFIG->getSettingInt("CAD:AutoRedraw")==1 ) currentDoc()->repaint();
      break;

    default:
      break;
  }

}



/*! Lets the user change the layer for elements
*/
void
RActEdit::layer(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      if(true) {

        QString layerName;

        snapper()->stop();

        currentDoc()->setPreviewUpdate(false);

        // Show layer Dialog:
        //
        RInputDialog* dlg =
          new RInputDialog(tr("Change Layer"),
                           DEF_DEFAULTLAYER,
                           'l',
                           true,
                           currentDoc(),
                           qApp->mainWidget());
        if(dlg->exec()) {
          layerName = dlg->getInput();
        }

        currentDoc()->editLayer(layerName, true);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();

        delete dlg;
      }
      break;


    default:
      break;
  }

}



/*! Lets the user set the zero point
    <p>States:
    <ul>
      <li>1: Set zero point
    </ul>
*/
void
RActEdit::zero(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Position of the new zero point"),
                                     tr("R: Back"),
                                     tr("Move zero"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:
      if( currentDoc()->getCurrentState()==1 ) {
        double px, py;
        if( snapper()->snapPoint(&px, &py, true) ) {
          currentDoc()->moveRelZero( -px, -py );
          currentDoc()->fillPreviewWithAll( 0.0, 0.0, true );
          currentDoc()->editMove( 0, 0.0, 0.0, -px, -py, 0.0, true, true, true );
          currentDoc()->setOffsetX( currentDoc()->getOffsetX() + (currentDoc()->realToScreenX(px)-currentDoc()->realToScreenX(0.0)) );
          currentDoc()->setOffsetY( currentDoc()->getOffsetY() + (currentDoc()->realToScreenY(py)-currentDoc()->realToScreenY(0.0)) );
          currentDoc()->setPageOriginX( currentDoc()->getPageOriginX() - px * currentDoc()->getPaperSpace() );
          currentDoc()->setPageOriginY( currentDoc()->getPageOriginY() - py * currentDoc()->getPaperSpace() );
          currentDoc()->repaint();
        }
        snapper()->snapPoint();
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        statusPanel()->clear();
        currentDoc()->setCursor(arrowCursor);
      }
      break;

    default:
      break;
  }
}



/*! Lets the user change attributes for elements
*/
void
RActEdit::attrib(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      if(true) {
        snapper()->stop();
        currentDoc()->setPreviewUpdate(false);

        // Show attrib Dialog:
        //
        RAttribDialog* dlg =
          new RAttribDialog(DEF_APPNAME,
                            tr("Choose the new attributes for the selected objects."),
                            qApp->mainWidget());
        if(dlg->exec()) {
          int newColor = dlg->getColor();
          int newWidth = dlg->getWidth();
          int newStyle = dlg->getStyle();

          currentDoc()->editAttrib(newColor, newWidth, newStyle, true);
        }

        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();

        delete dlg;
      }
      break;


    default:
      break;
  }

}



/*! Lets the user copy elements to clipboard
    <p>States:
    <ul>
      <li>1: Set reference point
    </ul>
*/
void
RActEdit::clipCopy(int _rEvent, bool _cut)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Reference point"),
                                     tr("R: Back"),
                                     tr("Copy selected elements"));
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting reference point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;

        snapper()->snapPoint(&px, &py);

        currentDoc()->setPreviewUpdate(true);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set reference point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          currentDoc()->editClipCopy(-px, -py);
          if(_cut) currentDoc()->editDelete(true);
          statusPanel()->setLeftButtonStatus(tr("L: Reference point"));
          snapper()->stop();
          currentDoc()->resetPreview();
          currentDoc()->setBehavior(BVR_NO);
          currentDoc()->resetCurrentState();
          currentDoc()->resetCurrentAction();
          currentDoc()->setCursor(arrowCursor);
          statusPanel()->clear();
          RAppWin::getRAppWin()->changeMenu(TOOLBAR_MAIN);
        }
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        RAppWin::getRAppWin()->changeMenu(TOOLBAR_MAIN);
      }
      break;

    default:
      break;
  }
}



/*! Lets the user paste elements from clipboard
    <p>States:
    <ul>
      <li>1: Set target point
    </ul>
*/
void
RActEdit::clipPaste(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Target point"),
                                     tr("R: Back"),
                                     tr("Paste objects"));

      currentDoc()->fillPreviewWithClipboard();

      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Angle:"), false);
      inputPanel()->setValue(0, 0.0);
      inputPanel()->show();
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting target point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;
        double pAngle = inputPanel()->getValue(0);

        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->previewMoved(px, py, pAngle)) {
            currentDoc()->drawPreview(px, py, pAngle);
          }
        }

        currentDoc()->setPreviewUpdate(true);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set target point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);

          double pAngle = inputPanel()->getValue(0);

          currentDoc()->movePreviewToCurrentLayer();
          currentDoc()->editMove(1, 0.0, 0.0, px, py, pAngle, true, true);

          currentDoc()->moveRelZero(px, py);
        }
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}




/*! Lets the user insert elements from library browser
    <p>States:
    <ul>
      <li>1: Set target point
    </ul>
*/
void
RActEdit::insertPart(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(tr("L: Target point"),
                                     tr("R: Back"),
                                     tr("Paste objects"));

      if( true ) {
        //RLibraryBrowser* lb = RAppWin::getRAppWin()->getLibraryBrowser();
        RLibraryBrowser* lb = RAppWin::getRAppWin()->getLibraryBrowser();

        currentDoc()->fillPreviewWithGraphic( lb->getSelectedGraphic(),
                                              lb->getFactor(),
                                              //lb->getAngle(),
                                              0.0,    // Rotation later in the preview...
                                              lb->getLength(),
                                              lb->getHeight() );
      }

      /*inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Angle:"), false);
      inputPanel()->setValue(0, 0.0);
      inputPanel()->show();*/
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting target point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;
        double pAngle = RAppWin::getRAppWin()->getLibraryBrowser()->getAngle();

        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->previewMoved(px, py, pAngle)) {
            currentDoc()->drawPreview(px, py, pAngle);
          }
        }

        currentDoc()->setPreviewUpdate(true);
      }
      break;

    case REVENT_LBUTTONUP:

      // Set target point:
      //
      if(currentDoc()->getCurrentState()==1) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();

          currentDoc()->setPreviewUpdate(false);

          double pAngle = RAppWin::getRAppWin()->getLibraryBrowser()->getAngle();

          currentDoc()->editMove(1, 0.0, 0.0, px, py, pAngle, true, true);

          currentDoc()->moveRelZero(px, py);
        }
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->resetPreview();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        //inputPanel()->reset();
        //inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}


/*! Lets the user clip elements (cut out a gap in the middle of the element)
    <p>States:

    <ul>
      <li>1: Set element
    </ul>
*/
void
RActEdit::clip(int _myREvent)
{
  static double gap=0.1;

  switch(_myREvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Clip element"),
                                     tr("R: Back"),
                                     tr("Clip elements"));
      inputPanel()->reset();
      inputPanel()->addLineEdit(tr("Gap:"), false);
      inputPanel()->show();
      inputPanel()->setValue(0, gap);
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);

        // Preview nothing:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          // Get distance from input panel:
          //
          /*gap = inputPanel()->getValue(0);

          prevEl.reset();
          snapper()->getSnappedElement()->getParallel(&prevEl,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY(),
                                                      paraDistance);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
          */
        }
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Draw gap:
        //
        RElement* el;
        if(snapper() &&
           (el=snapper()->getSnappedElement())) {

          // Get gap from input panel:
          //
          gap = inputPanel()->getValue(0);

          snapper()->stop();
          if( gap<el->getLength() ) {
            currentDoc()->editClip( el, gap, true );
          }
          else {
            statusPanel()->setStatus( tr("Gap too wide for object") );
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}


// EOF






















