/***************************************************************************
                          racttag.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


#include <math.h>
#include <stdlib.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>

#include "rappwin.h"
#include "racttag.h"
#include "rbehaviordef.h"
#include "rgraphic.h"
#include "rinputdialog.h"
#include "rlog.h"
#include "rsnap.h"
#include "rstatuspanel.h"

/*! \class RActTag
    \brief Event handling for tag functions

    \author Andrew Mustun
*/

/*! Constructor:
*/
RActTag::RActTag()
{
  
}



/*! Destructor:
*/
RActTag::~RActTag()
{
}



/*! Serves an event to the corresponding function
    \param _action The Action (e.g. ACT_LINES_NORMAL)
    \param _rEvent The event (e.g. REVENT_BEGIN, REVENT_END, REVENT_LBUTTONDOWN, ...)
*/
void
RActTag::serveEvent(int _action,
                    int _rEvent )
{
  if(_action>=ACT_TAG_FIRST &&
     _action<=ACT_TAG_LAST     ) {

    switch(_action) {

      case ACT_TAG_ELEMENT:
        tagElement(_rEvent);
        break;
        
      case ACT_TAG_CONTOUR:
        tagContour(_rEvent);
        break;
        
      case ACT_UNTAG_RANGE:
        tagRange(_rEvent, false);
        break;
        
      case ACT_TAG_RANGE:
        tagRange(_rEvent, true);
        break;

      case ACT_UNTAG_INTERS:
        tagInters(_rEvent, false);
        break;
        
      case ACT_TAG_INTERS:
        tagInters(_rEvent, true);
        break;
        
      case ACT_TAG_LAYER:
        tagLayer(_rEvent);
        break;
        
      default:
        break;
    }
  }
}



/*! Handles events for tagging single elements
*/
void
RActTag::tagElement(int _rEvent)
{
 
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: (Un-)tag element"),
                                     tr("R: Back"),
                                     tr("(Un-)tag single elements"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        if(snapper()) snapper()->getSnappedElement();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Tag element:
        //
        if(snapper() && 
           snapper()->getSnappedElement()) {

          if(snapper()->getSnappedElement()->getFlag(E_TAGGED)) {
            snapper()->getSnappedElement()->untag();
          }
          else {
            snapper()->getSnappedElement()->tag();
          }
          
          currentDoc()->drawElement(snapper()->getSnappedElement());
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}



/*! Handles events for tagging contours
*/
void
RActTag::tagContour(int _rEvent)
{
 
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: (Un-)tag contour"),
                                     tr("R: Back"),
                                     tr("(Un-)tag Contours"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        if(snapper()) snapper()->getSnappedElement();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Tag contour:
        //
        if(snapper() && 
           snapper()->getSnappedElement()) {

          if(snapper()->getSnappedElement()->getFlag(E_TAGGED)) {
            currentDoc()->tagContour(snapper()->getSnappedElement(), 
                                     false, true);
          }
          else {
            currentDoc()->tagContour(snapper()->getSnappedElement(), 
                                     true, true);
          }
          
          currentDoc()->drawElement(snapper()->getSnappedElement());
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}




/*! Handles events for (un-)tagging ranges
    \param _tag true=tag / false=untag
*/
void
RActTag::tagRange(int _rEvent, bool _tag)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first (last) point
  static double cy1=DEF_AREAMAX;  //
 
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      if(_tag) statusPanel()->setActionStatus(tr("L: First edge of range"),
                                              tr("R: Back"),
                                              tr("Tag range"));
      else     statusPanel()->setActionStatus(tr("L: First edge of range"),
                                              tr("R: Back"),
                                              tr("Untag range"));
      break;

    case REVENT_MOUSEMOVE:
      
      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      
      // Move for setting next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_RECT, cx1, cy1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Second edge of range"));
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          
          currentDoc()->tagRange(cx1, cy1, px, py, _tag, true);
          
          cx1=DEF_AREAMAX;
          cy1=DEF_AREAMAX;
          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: First edge of range"));
        }
        snapper()->snapPoint();
      }
      
      break;

    case REVENT_RBUTTONUP:
     
      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      
      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: First edge of range"));
      }
      break;

    default:
      break;
  }
}




/*! Handles events for (un-)tagging elements intersected by a line
    \param _tag true=tag / false=untag
*/
void
RActTag::tagInters(int _rEvent, bool _tag)
{
  static double cx1=DEF_AREAMAX;  // Coordinates of first (last) point
  static double cy1=DEF_AREAMAX;  //
 
  switch(_rEvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      if(_tag) statusPanel()->setActionStatus(tr("L: First point of intersection line"),
                                              tr("R: Back"),
                                              tr("Tag elements intersected by a line"));
      else     statusPanel()->setActionStatus(tr("L: First point of intersection line"),
                                              tr("R: Back"),
                                              tr("Untag elements intersected by a line"));
      break;

    case REVENT_MOUSEMOVE:
      
      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      
      // Move for setting next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE, cx1, cy1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx1, &cy1, true)) {
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(tr("L: Second point of intersection line"));
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        double px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          
          currentDoc()->tagInters(cx1, cy1, px, py, _tag, true);
          
          cx1=DEF_AREAMAX;
          cy1=DEF_AREAMAX;
          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(tr("L: First point of intersection line"));
        }
        snapper()->snapPoint();
      }
      
      break;

    case REVENT_RBUTTONUP:
     
      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      
      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(tr("L: First point of intersection line"));
      }
      break;

    default:
      break;
  }
}



/*! Handles events for tagging layers
*/
void
RActTag::tagLayer(int _rEvent)
{
  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(tr("L: Reference element"),
                                     tr("R: Back"),
                                     tr("(Un-)tag all elements of a layer"));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        if(snapper()) snapper()->getSnappedElement();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Tag layer:
        //
        if(snapper() && 
           snapper()->getSnappedElement()) {

          if(snapper()->getSnappedElement()->getFlag(E_TAGGED)) {
            currentDoc()->tagLayer(snapper()->getSnappedElement()->getLayer(), 
                                   false, true);
          }
          else {
            currentDoc()->tagLayer(snapper()->getSnappedElement()->getLayer(), 
                                   true, true);
          }
          
          currentDoc()->drawElement(snapper()->getSnappedElement());
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}


// EOF

