/***************************************************************************
                          rappwin.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

#include "rappwin.h"

#include <qwindowdefs.h>
#include <qregexp.h>

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef DEF_SOLARIS
#include <unistd.h>
#include <sys/wait.h>
#endif

#ifdef DEF_LINUX
#include <unistd.h>
#include <sys/dir.h>
#include <sys/wait.h>
#endif

#ifdef DEF_SGI
#include <unistd.h>
#include <sys/dir.h>
#include <sys/wait.h>
#endif

#ifdef DEF_FREEBSD
#include <dirent.h>
#endif

#ifdef DEF_WINDOWS
#include <dir.h>
#include <process.h>
#endif

#ifdef DEF_CAM_EXPERT
#include "rsimulationpanel.h"
#include "rconfigmach.h"
#include "rchoosemach.h"
#include "rvalidate.h"
#endif

#include <qapplication.h>
#include <qbitmap.h>
#include <qfiledialog.h>
#include <qimage.h>
#include <qkeycode.h>
#include <qlayout.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpaintdevicemetrics.h>
#include <qpixmap.h>
#include <qpixmapcache.h>
#include <qpopupmenu.h>
#include <qprinter.h>
#include <qstatusbar.h>
#include <qstring.h>
#include <qstrlist.h>
#include <qtoolbutton.h>
#include <qtooltip.h>

#include "raboutdialog.h"
#include "rcadwidget.h"
#include "rcombobox.h"
#include "rconfig.h"
#include "rcursors.h"
#include "rdimdlg.h"
#include "reventdef.h"
#include "rfile.h"
#include "rfilecxf.h"
#include "rgraphic.h"
#include "rhelpwindow.h"
#include "rintro.h"
#include "rinputdialog.h"
#include "rinputpanel.h"
#include "rlayoutdef.h"
#include "rlibrarybrowser.h"
#include "rlog.h"
#include "rmath.h"
#include "rpainter.h"
#include "rparameterdialog.h"
#include "rpreferencesdialog.h"
#include "rprgdef.h"

#include "rsnap.h"
#include "rsplitter.h"
#include "rstatuspanel.h"
#include "rstring.h"
#include "rtoolbar.h"
#include "rtoolbutton.h"
#include "rwidget.h"
#include "rxpmapp.h"

#include "xpm/simgrid.xpm"
#include "xpm/layerlist.xpm"


/*! \class RAppWin
    \brief The Application Window Class (Parent: QMainWindow)

    \author Andrew Mustun
*/

RAppWin* RAppWin::theOneAndOnlyRAppWin=0;



/*!
   Gets the one and only RAppWin-Object
   (creates a new one on first call only)
   \return Pointer to RAppWin
*/
RAppWin*
RAppWin::getRAppWin()
{
  if(theOneAndOnlyRAppWin==0) {
    RLOG("\nInit: create app win\n");
    theOneAndOnlyRAppWin = new RAppWin();
    RLOG("\nInit: app win created\n");
  }
  return theOneAndOnlyRAppWin;
}



/*! The Constructor of the Application Window
*/
RAppWin::RAppWin()
    : QMainWindow()
{
  RLOG("\n\nRAppWin constructor");
}



/*! Destructs the object.
*/
RAppWin::~RAppWin()
{
  delete printer;
  delete printMetrics;
  delete clipboard;

  for(int c=0; c<DEF_MAXDOCUMENTS; ++c) {
    if(graphicDoc[c]!=0) {
      delete graphicDoc[c];
    }
  }
  delete[] graphicDoc;
}



/** Initializes the AppWin
*/
void
RAppWin::init()
{
  int c;

  RLOG("RAppWin::init() begin");
  

  // Reset values:
  //
  intro=0;
  printer=0;
  fileMenu=0;
  camMenu=0;
  zoomMenu=0;
  windowMenu=0;
  optionsMenu=0;
  helpMenu=0;
  cadWidget1=0;
  cadWidget2=0;
  
#ifdef DEF_CAM_EXPERT
  simPanel=0;
#endif  
  
  inputPanel=0;
  inputPanel2=0;

  bSubmenuPoints=0;
  bSubmenuLines=0;
  bSubmenuArcs=0;
  bSubmenuCircles=0;
  bSubmenuTexts=0;
  bSubmenuDims=0;
  bSubmenuMeasure=0;
  bSubmenuHatch=0;
  bSubmenuTag=0;
  bSubmenuEdit=0;
  bSubmenuCam=0;
  
  bPointsSingle=0;

  bLinesNormal=0;
  bLinesRect=0;
  bLinesPara=0;
  bLinesBisector=0;
  bLinesTan1=0;
  bLinesTan2=0;
  bLinesOrtho=0;
  bLinesOrtho2=0;

  bDimsPara=0;
  bDimsHor=0;
  bDimsVer=0;
  bDimsAngle=0;
  bDimsDiam=0;
  bDimsRad=0;

  bMeasurePtPt=0;
  bMeasureLnPt=0;
  bMeasureParallelLines=0;
  bMeasureAngleLnLn=0;
  bMeasureEditElement=0;
	bMeasureArea=0;
  
  bTagAll=0;
  bTagNothing=0;
  bTagElement=0;
  bTagContour=0;
  bUntagRange=0;
  bTagRange=0;
  bUntagInters=0;
  bTagInters=0;
  bTagInvert=0;
  bTagLayer=0;
  bTagDoubles=0;
  bTagEnough=0;
  
  bEditMove=0;
  bEditRotate=0;
  bEditScale=0;
  bEditMirror=0;
  bEditMoveRotate=0;
  bEditRotateRotate=0;
  bEditTrim=0;
  bEditTrim2=0;
  bEditTrim3=0;
  bEditCut=0;
  bEditCut2=0;
  bEditStretch=0;
  bEditRound=0;
  bEditBevel=0;
  bEditTextElements=0;
  bEditText=0;
  bEditDelete=0;
  bEditLayer=0;
  bEditZero=0;
  bEditAttrib=0;
  bEditClip=0;

  bSnapFree=0;
  bSnapGrid=0;
  bSnapEndpoints=0;
  bSnapCenters=0;
  bSnapMiddles=0;
  bSnapAutoInter=0;
  bSnapManInter=0;
  bSnapDist=0;
  bSnapOnElement=0;
  bSnapKeyXY=0;
  bSnapKeyAR=0;
  
  bCamAuto=0;
  bCamManual=0;
  
  bFileNew=0;
  bFileOpen=0;
  bFileClose=0;
  bFileCloseOutput=0;
  bFileSave=0;

  bFilePrintPreview=0;
  bFilePrint=0;
  
  bZoomRedraw=0;
  bZoomAuto=0;
  bZoomToPage=0;
  bZoomIncrease=0;
  bZoomDecrease=0;
  bZoomWindow=0;
  bZoomPan=0;
  bZoomTagged=0;
  bZoomOne=0;

  bClipCopy=0;
  bClipCut=0;
  bClipPaste=0;
  bUndo=0;
  bRedo=0;

  bShowGrid=0;
  bShowLayerList=0;
  bShowLibraryBrowser=0;

  cbAttribColor=0;
  cbAttribWidth=0;
  cbAttribStyle=0;
  
  toolBarFile=0;
  toolBarZoom=0;
  toolBarUndo=0;
  toolBarShow=0;
  toolBarAttrib=0;

  currentMenu=-1;
  for(c=0; c<DEF_MENUBUF; ++c) lastMenu[c]=-1;
  ilastMenu=0;
  curDocInd=-1;

  keyDown = false;

  docNumber=1;
 
  RLOG("\nInit: central widget");
  centralWidget = new RWidget( this, "centralWidget", 0, true );
  //setCentralWidget( centralWidget );  
  centralWidget->show();

  // Set Caption:
  //
  setCaption(DEF_APPNAME);

  // Set Icon
#ifdef DEF_SGI
  setIcon(QPixmap(qcadsgi_xpm));
#else
  setIcon(QPixmap(qcad_xpm));
#endif
    
  // Make printer object:
  //
  RLOG("\nInit: printer");
  printer = new QPrinter;
  printer->setFullPage( true );
  ////printer->setPrinterName("");
  //printer->setMinMax(1, 1);
  ////printer->setPageSize(QPrinter::A4);
  ////printer->setOrientation(QPrinter::Portrait);
  //printMetrics = new QPaintDeviceMetrics(printer);
  printMetrics=0;
  printerInitialized=false;

  // Check for $PRINTER:
#ifdef DEF_LINUX
/*
  char *envPrinter;
  envPrinter = getenv( "PRINTER" );
  if ( !envPrinter || !*envPrinter ) {
    setenv( "PRINTER", RCONFIG->getSetting("Print:PrinterName").latin1(), 1 );
  }
  if( !RCONFIG->getSetting("Print:PrinterName").isEmpty() ) {
    printer->setPrinterName( RCONFIG->getSetting("Print:PrinterName") );
  }
  if( !RCONFIG->getSetting("Print:PrintProgram").isEmpty() ) {
    printer->setPrintProgram( RCONFIG->getSetting("Print:PrintProgram") );
  }
*/
#endif

  paperWidth=4800;
  paperHeight=6680;
  paperWidthMM=203;
  paperHeightMM=282;
  
  // Create Menus:
  //
  createMenus();

  recentFiles.setAutoDelete( true );

  // Create ToolBars:
  //
  RLOG("\nInit: toolbars");
  createToolBars();
  
  changeMenu(TOOLBAR_MAIN);
  //changeMenu(TOOLBAR_FILE);
  //changeMenu(TOOLBAR_ZOOM);

  // Splitter between cad widgets:
  //
  RLOG("\nInit: splitter");
  splitter = new RSplitter(centralWidget, "splitter");
  //splitter->hide();
  splitter->show();

  // Cad widgets:
  //
  RLOG("\nInit: cad widgets");
  cadWidget1 = new RCadWidget(true, this, splitter);
  //cadWidget1->hide();
  cadWidget1->show();
  cadWidget2 = new RCadWidget(false, this, splitter);
  //cadWidget2->hide();
  cadWidget2->show();

  // Mouse Cursors:
  //
  curGlass = new QCursor(QBitmap(32, 32, cur_glass_bmp,  true), 
                         QBitmap(32, 32, cur_glass_mask, true), 
                         6, 6);
  curCross = new QCursor(QBitmap(32, 32, cur_cross_bmp,  true), 
                         QBitmap(32, 32, cur_cross_mask, true), 
                         16, 16);

  // Panels:
  //
  RLOG("\nInit: panels");

#ifdef DEF_CAM_EXPERT
  // Simulation Panel:
  //
  simPanel = new RSimulationPanel(centralWidget, "simPanel");
#endif

  // Input Panel:
  //
  inputPanel = new RInputPanel(centralWidget, "inputPanel");
  inputPanel->hide();

  // Input Panel 2:
  //
  inputPanel2 = new RInputPanel(centralWidget, "inputPanel2");
  inputPanel2->hide();

  // Dimension Dialog:
  //
  dimensionDlg = new RDimDlg(centralWidget, "dimDlg");
  dimensionDlg->hide();

  // Status Panel:
  //
  statusPanel = new RStatusPanel(centralWidget, "statusPanel");
  statusPanel->show();

  // Ini Clipboard:
  //
  RLOG("\nInit: clipboard");
  clipboard = new RGraphic();

  // Ini graphics:
  //
  RLOG("\nInit: graphics");
  graphicDoc = new RGraphic*[DEF_MAXDOCUMENTS];
  for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
    graphicDoc[c]=0;
  }

  graphicOut = new RGraphic(cadWidget2);
  cadWidget2->setGraphic(graphicOut);

  RLOG("\n\nGraphics created successfully!");

  // Open a new doc at beginning / status bar:
  //
  RLOG("\nInit: new file");
  if(fileNew()>=0) {
    // Ready:
    //
    //statusBar()->message(tr("Ready"), 2000);
  }
  else {
    // Can't create new document
    //
    //statusBar()->message(tr("Can't create new document"), 2000);
  }

  //--statusBar()->hide();
  
  RLOG("\n\nStatus bar hidden");

  // Connect menubar entries with function showDocument
  //   (allows switching to specific loaded file by menu):
  //
  connect(windowMenu, SIGNAL(activated(int)), 
          this, SLOT(showDocument(int)));

  RLOG("\nMenus connected!");
  
  if( RCONFIG->getSettingInt("CAD:LayerListVisible")==1 ) bShowLayerList->toggle();
  if( RCONFIG->getSettingInt("CAD:LibraryBrowserVisible")==1 ) bShowLibraryBrowser->toggle();
  
  RLOG("\nAppWin created successfully!");
  
  // Resize / Move by values set in INI-File:
  //
  int winWidth = RCONFIG->getSettingInt("Application:ApplicationWindowWidth");
  int winHeight = RCONFIG->getSettingInt("Application:ApplicationWindowHeight");
  if(winWidth >0 && winHeight>0    ) {
    resize( winWidth,winHeight );
  }

  int winX = RCONFIG->getSettingInt("Application:ApplicationWindowX");
  int winY = RCONFIG->getSettingInt("Application:ApplicationWindowY");
  if( winX>-100 && winY>-100 ) {
    move( winX,winY );
  }
  
  setConstructionMode();
  
  resizeChildren();

  RLOG("\nRAppWin init ended!");
}





/** Reinit application from configuration
*/
void
RAppWin::reinit()
{
  if(graphicDoc) {
    for(int i=0; i<DEF_MAXDOCUMENTS; ++i) {
      if(graphicDoc[i]) {
        graphicDoc[i]->init();
      }
    }
  }
}



/*! Activates a document for (bring it to the top). All future actions
    go to this doc.
    \param *_graphic pointer to the document which you wish to bring to top
*/
void
RAppWin::activateDocument(RGraphic* _graphic)
{
  if(_graphic && (_graphic==currentOut() || _graphic==currentDoc()) ) {

    // Deactivate output graphic:
    //
    if(graphicOut && _graphic!=graphicOut && graphicOut->isActive()) {
      graphicOut->setActive(false);
      graphicOut->drawActiveBorder(false);
    }

    // Deactivate all other documents:
    //
    if(graphicDoc) {
      for(int i=0; i<DEF_MAXDOCUMENTS; ++i) {
        if(graphicDoc[i]) {
          if(_graphic!=graphicDoc[i] && graphicDoc[i]->isActive()) {
            graphicDoc[i]->setActive(false);
            graphicDoc[i]->drawActiveBorder(false);
          }
        }
      }
    }
    
    _graphic->setActive(true);
    
#ifdef DEF_CAM_EXPERT
    if(simPanel) simPanel->setGraphic(_graphic);
#endif

    if(inputPanel) inputPanel->setGraphic(_graphic);
    if(inputPanel2) inputPanel2->setGraphic(_graphic);
    _graphic->drawActiveBorder(true);
  }
}



/*! Shows the Main Window and all children (overwrites existing function)
*/
void
RAppWin::show()
{
  QMainWindow::show();

  setMinimumSize(DEF_MINAPPWIDTH, DEF_MINAPPHEIGHT);
	return;

  if(splitter) {
    splitter->show();
    splitter->moveToRight();
  }
  if(cadWidget1) cadWidget1->show();
  if(cadWidget2) cadWidget2->show();
  
#ifdef DEF_CAM_EXPERT
  
  if(simPanel) {
    if(RCONFIG->getSettingInt("CAM:SimulationPanelVisible")==1) {
      simPanel->show();
    }
    else {
      simPanel->hide();
    }
  }
#endif

  setMinimumSize(DEF_MINAPPWIDTH, DEF_MINAPPHEIGHT);
  
  resizeChildren();
}



/*! Shows the Splash window (start up)
    \sa closeIntro()
*/
void
RAppWin::showIntro() 
{
  intro=new RIntro(DEF_VERSION, this, "intro");
  intro->show();
  enableAll(false);
}


/*! Closes the Splash window
    \sa showIntro()
*/
void
RAppWin::closeIntro() 
{
  delete intro;
  enableAll(true);
}


//
//
// Recent files handling
//
//

/*! Clears list with recent files.
*/
void
RAppWin::clearRecentFiles()
{
  recentFiles.clear();
}



/*! Adds a new file to recent file list and
   removes the last entry if we've more than
   'QAPS_RECENTNUMBER' entries. Double entries
   are avoided. The last added entry is always
   in index 0.
 */
void
RAppWin::addRecentFile( QString recentFile, bool updateMenu )
{
  if( recentFile.isEmpty() ) return;

  QString* it;

  // Remove identical previous entries:
  //
  for( it=recentFiles.first(); it!=0; it=recentFiles.next() ) {
    if( *it==recentFile ) {
      recentFiles.remove( it );
      //it=recentFiles.first();
      recentFiles.first();
    }
  }

  // Insert new recent file:
  //
  recentFiles.insert( 0, new QString(recentFile) );
  if( recentFiles.count()>DEF_RECENTFILES ) {
    recentFiles.removeLast();
  }

  if( updateMenu ) updateRecentFileMenu();
}



/*! Gets recent file with the given index.
*/
QString
RAppWin::getRecentFile(int _index)
{
  if( _index>=0 && _index<(int)recentFiles.count() ) {
    return *recentFiles.at(_index);
  }
  else {
    return 0;
  }
}


/*!
 * Updates the recent files in menu file.
 */
void
RAppWin::updateRecentFileMenu()
{
  // Add recent files to list in Menu "File":
  //
  int c;
  QString str;
  QString* it;
  for( it=recentFiles.first(), c=0; it!=0; it=recentFiles.next(), ++c ) {
    if( !it->isEmpty() ) {

      // Change item:
      //
      if( DEF_RECENTPOS+c < (int)fileMenu->count() ) {
        fileMenu->changeItem( 32000+c, *it );
      }

      // Add item:
      //
      else {
        fileMenu->insertItem( QIconSet( QPixmap(filerecall_xpm) ), *it, 32000+c );
      }

      str.setNum(c);
      RCONFIG->addSetting( "RecentFiles:File" + str, *it );
    }
  }
}



/** Updates the Comboboxes color, width and style
*/
void
RAppWin::updateAttribToolbar(){
  cbAttribColor->setCurrentItem(currentDoc()->colorToNumber(
  															currentDoc()->layerCurrent()->getColor()));
  cbAttribWidth->setCurrentItem(currentDoc()->layerCurrent()->getWidth());
  cbAttribStyle->setCurrentItem(currentDoc()->styleToNumber(
  															currentDoc()->layerCurrent()->getStyle()));
}



/** Get pointer to library browser which is part of the cad widget 1
*/
RLibraryBrowser*
RAppWin::getLibraryBrowser()
{
  return cadWidget1->getLibraryBrowser();
}



/** Gets pixmap for a given edit-function
*/
QPixmap
RAppWin::getPixmapForEditFunction( int _fkt )
{
  switch( _fkt ) {
    case ACT_EDIT_MOVE        : return QPixmap(editmove_xpm);
    case ACT_EDIT_ROTATE      : return QPixmap(editrotate_xpm);
    case ACT_EDIT_SCALE       : return QPixmap(editscale_xpm);
    case ACT_EDIT_MIRROR      : return QPixmap(editmirror_xpm);
    case ACT_EDIT_MOVEROTATE  : return QPixmap(editmoverot_xpm);
    case ACT_EDIT_ROTATEROTATE: return QPixmap(editrotrot_xpm);
    case ACT_EDIT_TRIM        : return QPixmap(edittrim_xpm);
    case ACT_EDIT_TRIM2       : return QPixmap(edittrim2_xpm);
    case ACT_EDIT_TRIM3       : return QPixmap(edittrim3_xpm);
    case ACT_EDIT_CUT         : return QPixmap(editcut_xpm);
    case ACT_EDIT_CUT2        : return QPixmap(editcut2_xpm);
    case ACT_EDIT_STRETCH     : return QPixmap(editstretch_xpm);
    case ACT_EDIT_ROUND       : return QPixmap(editround_xpm);
    case ACT_EDIT_BEVEL       : return QPixmap(editbevel_xpm);
    case ACT_EDIT_TEXTELEMENTS: return QPixmap(edittextelement_xpm);
    case ACT_EDIT_TEXT        : return QPixmap(edittext_xpm);
    case ACT_EDIT_DELETE      : return QPixmap(editdelete_xpm);
    case ACT_EDIT_LAYER       : return QPixmap(editlayer_xpm);
    case ACT_EDIT_ZERO        : return QPixmap(editzero_xpm);
    case ACT_EDIT_ATTRIB      : return QPixmap(editattrib_xpm);
    case ACT_EDIT_CLIP        : return QPixmap(editclip_xpm);

    case ACT_HATCH            : return QPixmap(cadmenuhatch_xpm);
  }

  return QPixmap( tagenough_xpm );

}



/** Gets the page size of chosen print settings.
*/
QPrinter::PageSize
RAppWin::getPageSize()
{
  if( printer ) {
    return printer->pageSize();
  }
  else {
    return QPrinter::A4;
  }
}



//
//   Menu functions:
//

/*! File -> New:
    Creates a new file (document) as a user action
    \return index of new document
*/
int
RAppWin::fileNew()
{
  return fileNew(false);
}



/*! Creates a new file (document) (user action).
    curDocInd is set to new index.
    \param _load true: creates a new document for loading a file into it later
               / false: make a new file with a default name
    \return index of new document
*/
int
RAppWin::fileNew(bool _load)
{
  int newDocIndex;

  newDocIndex = getNewDocumentIndex();
  if(isDocumentValid(newDocIndex)) {

    RLOG("\nfileNew: Document Index is valid");

    hideDocuments(newDocIndex);
    curDocInd=newDocIndex;

    RLOG("\nSwitch to construction mode...");

    setConstructionMode();

    // New file with default name and menu entry:
    //
    if(!_load) {

      // Set a default name for the new document:
      //
      QString docName;

      // Document #:
      //
      docName = QString("%1 %2")
                       .arg(tr("Document"))
		       .arg(docNumber);
      ++docNumber;
      currentDoc()->init();
      currentDoc()->setFileName(docName);

      int newMenuId;
      newMenuId = windowMenu->insertItem(QIconSet(QPixmap(document_xpm)), docName, -2);
      currentDoc()->setMenuId(newMenuId);
      currentDoc()->setActive(true);

      RLOG("\nfileNew: New Window-Menu Entry: ");
      RLOG(docName);
      RLOG(" Id: ");
      RLOG(newMenuId);


      updateCaption();
    }

    RLOG("\nShow current doc...");

    showCurrentDocument();
    //if(currentDoc()) currentDoc()->zoomToPage();
    if(RCONFIG->getSettingInt("CAD:ShowPageBorder")!=0) zoomToPage();
    else                                                zoomAuto();

    RLOG("Show Main Menu...");

    changeMenu(TOOLBAR_MAIN);

    RLOG("\nThe new created doc has index: ");
    RLOG(curDocInd);

    return curDocInd;
  }
  else {
    return -1;
  }
}



/*! Closes the file (document) which is currently active
    \param _ask Ask the user if he/she's sure.
    \return index of the new active document or
            -1 if this was the last document closed or
            -2 if the closing action was aborted by the
               user
*/
int
RAppWin::fileClose(bool _ask)
{
  int c;

  if(isDocumentValid(curDocInd)) {

    if( _ask && graphicDoc[curDocInd]->getFlag(G_CHANGED) ) {
      // Not saved: Ask user if he/her's sure:
      //
      QString shortName = graphicDoc[curDocInd]->getFileName();
      if(shortName.length()>64) shortName = shortName.left(30) + "..." + shortName.right(31);
      QMessageBox mb( DEF_APPNAME,
                      shortName + "\n" + tr("has been modified but not saved! - Save now?"),   // 428
                      QMessageBox::Information,
                      QMessageBox::Yes,
                      QMessageBox::No,
                      QMessageBox::Cancel | QMessageBox::Escape );
      mb.setButtonText( QMessageBox::Yes, tr("Yes") );
      mb.setButtonText( QMessageBox::No, tr("No") );
      mb.setButtonText( QMessageBox::Cancel, tr("Cancel") );

      RLOG("\nfileClose");

      int ret = mb.exec();
      if( ret==QMessageBox::Cancel ) {
        RLOG("\nCancel");
        return -2;
      }
      else if( ret==QMessageBox::Yes ) {
        if(!fileSave()) {
          RLOG("\nNot saved");
          return -2;
        }
      }
    }

    RLOG("\nClose doc with menu ID: ");
    RLOG(currentDoc()->getMenuId());

    // Remove menu entry:
    //
    windowMenu->removeItem(currentDoc()->getMenuId());

    RLOG("\nRemove Window-Menu Entry: ");
    RLOG(" Id: ");
    RLOG(currentDoc()->getMenuId());

    // Set current document index to
    //   the next existing one:
    //
    int oldDocIndex=curDocInd;
    curDocInd=-1;
    for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c] && c!=oldDocIndex) {
        curDocInd=c;
        RLOG("\nNew Doc Index: ");
        RLOG(curDocInd);
      }
    }

    if(curDocInd==-1) {
      if(cadWidget1) {
        cadWidget1->setGraphic(0);
      }
      
#ifdef DEF_CAM_EXPERT
      if(simPanel)    simPanel->setGraphic(0);
#endif
      
      if(inputPanel)  inputPanel->setGraphic(0);
      if(inputPanel2) inputPanel2->setGraphic(0);
    }

    showCurrentDocument();
    RLOG("\nCurrent Doc Shown");

    // Close current document:
    //
    delete graphicDoc[oldDocIndex];
    graphicDoc[oldDocIndex]=0;

    RLOG("\nDeleted.");

    changeMenu(TOOLBAR_MAIN);

    return curDocInd;
  }

  return -1;
}



/*! File -> Close output.
    Closes the output file (document)
    \return 1 on success, -1 if there was no output
*/
int
RAppWin::fileCloseOutput()
{
  if(isOutputValid()) {

    RLOG("\nClose output");

    if(cadWidget2) {
      RLOG("\ncadWidget2: graphic set");
    }

    // Close current output:
    //
    graphicOut->deleteElements();
    graphicOut->repaint();

    RLOG("\nOutput Deleted.");

    return 1;
  }

  return -1;
}


/*! File -> Open
    Allows the user to open a file with file selection box
    \return true: succesful / false: user has cancled
*/
bool
RAppWin::fileOpen(bool add)
{
  QString fn;                           // File name
  QFileDialog fileDlg(this,0,true);     // File open dialog
  QStringList filters;                  // Filter (dxf, plt, ...)
  QString openFilter = RCONFIG->getSetting("Application:OpenFilter");
  QString newFilter;
  RGraphic prev;                        // The preview widget
  prev.setFlag(G_PREVIEW);

  if( !openFilter.isEmpty() ) {
    filters.append( openFilter );
  }

  if( (newFilter="Drawing Exchange (*.dxf)") != openFilter )    filters.append(newFilter);
  if( (newFilter="Microstation (*.dgn)") != openFilter )        filters.append(newFilter);
  if( (newFilter="HPGL (*.plt *.hpgl *.inc)") != openFilter )   filters.append(newFilter);
#ifdef DEF_CAM_EXPERT
  if( (newFilter="GERBER (*.g*)") != openFilter )               filters.append(newFilter);
  if( (newFilter="NC Program (*.cnc *.nc *.d)") != openFilter ) filters.append(newFilter);
#endif
  if( (newFilter="Fonts (*.cxf)") != openFilter )               filters.append(newFilter);
  if( (newFilter="All Files (*.*)") != openFilter )             filters.append(newFilter);

  fileDlg.setFilters( filters );
  fileDlg.setMode( QFileDialog::ExistingFile );
  fileDlg.setCaption( tr("File open") );
  if( !RCONFIG->getSetting("Application:OpenPath").isEmpty() ) {
    fileDlg.setDir( RCONFIG->getSetting("Application:OpenPath") );
  }
  /*if( !RCONFIG->getSetting("Application:OpenFilter").isEmpty() ) {
    fileDlg.setFilter( RCONFIG->getSetting("Application:OpenFilter") );
  }*/
  fileDlg.setContentsPreviewEnabled( true );
  fileDlg.setContentsPreview( &prev, &prev );
  if( !RCONFIG->getSetting("Application:OpenViewMode").isEmpty() ) {
    fileDlg.setViewMode( (QFileDialog::ViewMode)(RCONFIG->getSettingInt("Application:OpenViewMode")) );
  }
  if( !RCONFIG->getSetting("Application:OpenPreviewMode").isEmpty() ) {
    fileDlg.setPreviewMode( (QFileDialog::PreviewMode)(RCONFIG->getSettingInt("Application:OpenPreviewMode")) );
  }

  if( !fileDlg.exec() ) return false;

  fn = fileDlg.selectedFile();

  if(!fn.isEmpty()) {
    RCONFIG->addSetting( "Application:OpenPath", QFileInfo( fn ).dirPath( true ) );
    RCONFIG->addSetting( "Application:OpenFilter", fileDlg.selectedFilter() );
    RCONFIG->addSetting( "Application:OpenViewMode", (int)(fileDlg.viewMode()) );
    RCONFIG->addSetting( "Application:OpenPreviewMode", (int)(fileDlg.previewMode()) );

    if(fileOpen(fn, add)) {
      addRecentFile( fn );
      changeMenu(TOOLBAR_MAIN);
      return true;
    }
    else {
      return false;
    }
  }
  else {
    // Can't create new document
    //
    //statusBar()->message(tr("Can't create new document"), 2000);
    return false;
  }
}



// Load file 'fileName':
//
bool
RAppWin::fileOpen(const QString& _fileName, bool add)
{
  if(add || fileNew(true)>=0) {
    if(isDocumentValid(curDocInd) &&
       currentDoc()->loadFile(_fileName, add)) {
       
      if(RCONFIG->getSettingInt("CAD:ShowPageBorder")!=0) currentDoc()->zoomToPage();
      else                                                currentDoc()->zoomAuto();
      currentDoc()->repaint();
      currentDoc()->setPrevVisible(false);
      currentDoc()->setActive( true );

      if(!add) {
        // Insert menu Item for the new Document:
        //
        int newMenuId = windowMenu->insertItem(QPixmap(document_xpm), _fileName, -2);
        currentDoc()->setMenuId(newMenuId);

        updateCaption();
      }
      else {
        // File changed:
        currentDoc()->graphicHasChanged();
      }

      changeMenu(TOOLBAR_MAIN);

      return true;
    }
    else {
      if(!add) fileClose(false);
    }
  }

  return false;
}



// File -> recent file
//
void
RAppWin::fileOpenRecent(int _id)
{
  QString str = fileMenu->text(_id);

  // Is it a recent file?
  //
  if(str.contains(".") && str.contains("/")) {
    // Load file:
    //
    fileOpen( str );
    addRecentFile( str );
  }
}



// File -> Save As:
//
bool
RAppWin::fileSaveAs()
{

#if defined(DEF_CAM_EXPERT)
  RValidate v;                       // Validate reg.txt (CAM Expert)
  if(!v.validate(false)) exit(1);
#endif

  QString fn;        // File name
  bool ret=false;    // return value

  QFileDialog fileDlg(this,0,true);     // File open dialog
  QStringList filters;                  // Filter (dxf, plt, ...)

  filters.append("Drawing Exchange (*.dxf)");
  filters.append("EPS (*.eps)");
  filters.append("Font (*.cxf)");
  fileDlg.setFilters(filters);
  fileDlg.setMode(QFileDialog::AnyFile);
  fileDlg.setCaption(tr("File save"));
  fileDlg.setDir( RCONFIG->getSetting("Application:SavePath") );
  int res = fileDlg.exec();

  RLOG( "\nres: " );
  RLOG( res );

  fn = fileDlg.selectedFile();

  // Valid name:
  //
  if(res!=0 && !fn.isEmpty()) {

    RLOG("\nfileSave");

    bool usersChoice;

    RCONFIG->addSetting( "Application:SavePath", QFileInfo( fn ).dirPath( true ) );

    // No Extension set / append default extension:
    //
    if(fn.find('.')==-1) {
      if(fileDlg.selectedFilter()=="Font (*.cxf)") fn+=".cxf";
      else if(fileDlg.selectedFilter()=="EPS (*.eps)") fn+=".eps";
      else                                         fn+=".dxf";
    }

    QFileInfo fi(fn);

    if(fi.exists()) {
      usersChoice=fileOverwriteWarning(fn,
                                       tr("The file"),
                                       RAppWin::getRAppWin());
    }
    else {
      usersChoice=true;
    }

    if(!fi.exists() || usersChoice) {

      RLOGTIME();

      if(fileSave(fn)) {

        RLOGTIME();

        // Add to recent file list:
        //   test if it is already in list -> set to top
        //
        addRecentFile(fn);
        ret=true;
      }
      else {
        QMessageBox::warning(this, tr("Warning"),
                             QString(tr("Saving of the file failed! This may have one\nof the following reasons:\n- You don't have write permission to the place you've chosen.\n- The disk is full.")) +
                             tr("\nPlease try to save the file to your home directory or delete\nsome files you don't need anymore to make free space\navailable on your harddisk."));
        ret = false;
      }
    }
  }
  else {
    // Saving aborted
    //
    //statusBar()->message(tr("Saving aborted"), 2000);
  }

  return ret;
}



// File -> Save (called directly from menu):
//
bool
RAppWin::fileSave()
{
  if(isDocumentValid(curDocInd) &&
     !currentDoc()->getFileName().contains('.')) {
    return fileSaveAs();
  }
  else {
    return fileSave( currentDoc()->getFileName() );
  }
}



// File -> Save:
//
bool
RAppWin::fileSave(const QString& _fileName)
{
#if defined(DEF_CAM_EXPERT)
  RValidate v;                       // Validate reg.txt (CAM Expert)
  if(!v.validate(false)) exit(1);
#endif

  if(isDocumentValid(curDocInd) &&
     currentDoc()->saveFile(_fileName)) {

    //fileClose(false);
    //fileOpen(_fileName);

    updateCaption();

    // Saved document
    //
    //QString s;
    //s.sprintf("%s %s", tr("Saved document"), _fileName);
    //statusBar()->message(s, 2000);

    return true;
  }

  return false;
}


// File -> Print:
//
void
RAppWin::filePrint( bool _setup )
{
  RLOG("\nRAppWin::filePrint");

#if defined(DEF_CAM_EXPERT)
  RValidate v;                       // Validate reg.txt (CAM Expert)
  if(!v.validate(false)) exit(1);
#endif

  // printer dialog
  //
  RLOG("\nOpening printer dialog");

  if(isDocumentValid(curDocInd) && ( !_setup || printer->setup(this) ) ) {

    RLOG("\nPrinter name: ");
    RLOG(printer->printerName());

    RLOG("\nPrint program: ");
    RLOG(printer->printProgram());
    
#ifdef DEF_LINUX
    char *envPrinter;
    envPrinter = getenv( "PRINTER" );
    if ( !envPrinter || !*envPrinter ) {
      setenv( "PRINTER", printer->printerName(), 1 );
    }
#endif
    
    printerInitialized=true;

    if(printMetrics) delete printMetrics;
    printMetrics = new QPaintDeviceMetrics(printer);
    updatePrintMetrics();

    // Printing...
    //
    RLOG("\nStarting print...");
    RLOG("\nPrinter Name: ");
    RLOG(printer->printerName());
    statusPanel->setStatus(tr("Printing..."));

    int resEnh=1;      // Resolution enhancement

    if(RUNNING_LINUX) {
      resEnh = RCONFIG->getSettingInt("Print:PrintEnhancement");
    }

    QRect r(0, 0, paperWidth*resEnh, paperHeight*resEnh);
    double oriZoom = currentDoc()->getZoom();
    int oriOffsetX = currentDoc()->getOffsetX();
    int oriOffsetY = currentDoc()->getOffsetY();

    // Paint on printer
    //
    RPainter p(printer);
    //p.begin(printer);

    p.setWindow(0, 0, paperWidth*resEnh, paperHeight*resEnh);

    // Set printer factor (only for zero and grid)
    //
    if(paperWidthMM>0.1 && paperHeightMM>0.1) {
      currentDoc()->setPrinterFactor(paperWidth*resEnh/paperWidthMM,
                                     paperHeight*resEnh/paperHeightMM);
    }

    double f=RCONFIG->getSettingDouble("Print:PrintAdjustmentFactor");
    if(f<0.000001 && f>-0.000001) f=1.0;
    currentDoc()->setZoom(currentDoc()->screenToRealX(0),
                          currentDoc()->screenToRealY(0),
                          currentDoc()->screenToRealX((int)(currentDoc()->width()/f)),
                          currentDoc()->screenToRealY((int)(currentDoc()->height()/f)),
                          0, &r);

    currentDoc()->draw(&p, false, true);

    //RLOG( "\np.end():" );
    //p.end();                      // send job to printer
    //RLOG( "\ndone" );

    currentDoc()->setZoom(oriZoom);
    currentDoc()->setOffsetX(oriOffsetX);
    currentDoc()->setOffsetY(oriOffsetY);

    // Printing completed:
    //
    RLOG("\nPrinting completed");
    RLOG("\nPrinter Name: ");
    RLOG(printer->printerName());
    statusPanel->setStatus(tr("Printing completed"));
  }
  else {
    // Printing aborted:
    //
    RLOG("\nPrinting aborted");
    statusPanel->setStatus(tr("Printing aborted"));
  }
}



/*! Updates the print metrics from print object
    to local variables. The print object gets
    reset sometimes...
*/
void
RAppWin::updatePrintMetrics()
{
  if(printMetrics) {
    paperWidth = printMetrics->width();
    paperHeight = printMetrics->height();
    paperWidthMM = printMetrics->widthMM();
    paperHeightMM = printMetrics->heightMM();

    RLOG("\npaperWidth: ");
    RLOG(paperWidth);
    RLOG("\npaperHeight: ");
    RLOG(paperHeight);

    RLOG("\npaperWidthMM: ");
    RLOG(paperWidthMM);
    RLOG("\npaperHeightMM: ");
    RLOG(paperHeightMM);

  }
}



// File -> Printer Setup:
//
void
RAppWin::filePrintSetup()
{
  RLOG("\nRAppWin::filePrintSetup");
  if(cadWidget1 && cadWidget1->getGraphic()) {
    if(printer->setup(this)) {
      RLOG("\nPrinter Name: ");
      RLOG(printer->printerName());
      printerInitialized=true;
      if(printMetrics) delete printMetrics;
      printMetrics = new QPaintDeviceMetrics(printer);
      updatePrintMetrics();

      cadWidget1->resizeChildren();
      //cadWidget1->getGraphic()->zoomAuto();
    }
  }
}



// File -> Printpreview:
//
void
RAppWin::filePrintPreview()
{
  RLOG("\nRAppWin::filePrintPreview");

  //setenv( "PRINTER", "lp", 1 );

  // Switch preview on (=true) or off (=false):
  //
  bool switchPreviewOn=false;

  if(cadWidget1 && cadWidget1->getGraphic()) {
    if(!cadWidget1->getPrintPreview()) {

      // Ini printer first:
      //
      if(!printerInitialized) {
        if(printer->setup(this)) {
          printerInitialized=true;
          if(printMetrics) delete printMetrics;
          printMetrics = new QPaintDeviceMetrics(printer);
          updatePrintMetrics();
          switchPreviewOn=true;
        }
      }

      // Printer already set up:
      //
      else {
        switchPreviewOn=true;
      }
    }

    if(switchPreviewOn) {
      bFilePrintPreview->setOn(true);
      cadWidget1->setPrintPreview(true);
      cadWidget1->getGraphic()->setFrameStyle(QFrame::Box|QFrame::Plain);
    }

    else {
      bFilePrintPreview->setOn(false);
      cadWidget1->setPrintPreview(false);
      cadWidget1->getGraphic()->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
    }
    cadWidget1->resizeChildren();
    cadWidget1->getGraphic()->zoomOne(paperWidthMM, paperHeightMM);
  }
}


// File -> Quit:
//
void
RAppWin::fileQuit()
{
  int ret;

  do {
    ret = fileClose( true );
  } while(ret!=-2 && ret!=-1);

  if(ret==-1) qApp->quit();

  /*
  QMessageBox mb( DEF_APPNAME,
                  tr("Do you really want to exit?"),
                  QMessageBox::Information,
                  QMessageBox::Yes,
                  QMessageBox::No,
                  QMessageBox::Cancel | QMessageBox::Escape );
  mb.setButtonText( QMessageBox::Yes, tr("Yes") );
  mb.setButtonText( QMessageBox::No, tr("No") );

  if( mb.exec()==QMessageBox::Yes ) {
    qApp->quit();
  }
  */

}


/*! File -> Exit.
    Called if the user closes the app using the X at top right
    of the window. We ask him/her if he/she's sure if there are
    unsaved files.
 */
void
RAppWin::closeEvent(QCloseEvent* ce)
{
  RLOG("closeEvent");

  int ret;

  do {
    ret = fileClose( true );
  } while(ret!=-2 && ret!=-1);

  if(ret==-1) ce->accept();
  else        ce->ignore();


  /*if (!e->edited()) {
    ce->accept();
    return;
  }*/

  // Ask user if he/her's sure:
  //
  /*QMessageBox mb( DEF_APPNAME,
                  tr("Do you really want to exit?"),
                  QMessageBox::Information,
                  QMessageBox::Yes,
                  QMessageBox::No,
                  QMessageBox::Cancel | QMessageBox::Escape );
  mb.setButtonText( QMessageBox::Yes, tr("Yes") );
  mb.setButtonText( QMessageBox::No, tr("No") );

  if( mb.exec()==QMessageBox::Yes ) {
    ce->accept();
  }
  else {
    ce->ignore();
  }*/
}




// Zoom -> Redraw:
//
void
RAppWin::zoomRedraw()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->repaint();
    if(isOutputValid()) {
      //synchronizeZoom();
      currentOut()->repaint();
    }
  }
}


// Zoom -> Autozoom:
//
// SLOT Version
//
void
RAppWin::zoomAuto()
{
  zoomAuto(true);
}


// Zoom -> Autozoom:
//
// _udate: Update the screen? (Std: true)
//
void
RAppWin::zoomAuto(bool _update)
{

  switch(mode) {

    // In Construction Mode:
    //
    case 'n':
      if(isDocumentValid(curDocInd)) {
        currentDoc()->zoomAuto();
        if(_update) currentDoc()->repaint();
      }
      break;

    // In Simulation Mode:
    //
    case 's':
      if(isOutputValid()) {
        currentOut()->zoomAuto();
        if(_update) currentOut()->repaint();
      }
      break;

    // In Comparison Mode:
    //
    case 'm':
      if(isDocumentValid(curDocInd) && isOutputValid()) {
        currentOut()->zoomAuto();
        currentDoc()->zoomAuto();

        // Adjust zoom to left frame:
        //
        if(currentDoc()->getZoom()<currentOut()->getZoom() ||
           currentOut()->count()==0) {
          synchronizeZoom(false);
        }

        // Adjust zoom to right frame:
        //
        else {
          synchronizeZoom(true);
        }
        if(_update) currentOut()->repaint();
        if(_update) currentDoc()->repaint();
      }
      break;

    default:
      break;

  }
}



// Zoom -> To page:
//
void
RAppWin::zoomToPage()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->zoomToPage();
    synchronizeZoom();
    currentDoc()->repaint();
    currentOut()->repaint();
  }
}



// Zoom -> Increase:
//
void
RAppWin::zoomIncrease()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->zoomIncrease();
    synchronizeZoom();
    currentDoc()->repaint();
    currentOut()->repaint();
  }
}



// Zoom -> Decrease:
//
void
RAppWin::zoomDecrease()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->zoomDecrease();
    synchronizeZoom();
    currentDoc()->repaint();
    currentOut()->repaint();
  }
}



// Zoom -> Window:
//
void
RAppWin::zoomWindow()
{
  if(isDocumentValid(curDocInd)) {

    if(currentDoc()->getCurrentAction()<ACT_ZOOM_FIRST ||
       currentDoc()->getCurrentAction()>ACT_ZOOM_LAST     ) {

      currentDoc()->saveCurrentActionSettings();
      currentDoc()->setCurrentAction(ACT_ZOOM_WINDOW);
      currentDoc()->distributeEvent(REVENT_BEGIN);
    }
  }

  if(isOutputValid()) {

    if(currentOut()->getCurrentAction()<ACT_ZOOM_FIRST ||
       currentOut()->getCurrentAction()>ACT_ZOOM_LAST     ) {

      currentOut()->saveCurrentActionSettings();
      currentOut()->setCurrentAction(ACT_ZOOM_WINDOW);
      currentOut()->distributeEvent(REVENT_BEGIN);
    }
  }
}



// Zoom -> Pan:
//
void
RAppWin::zoomPan()
{
  if(isDocumentValid(curDocInd)) {
    if(currentDoc()->getCurrentAction()<ACT_ZOOM_FIRST ||
       currentDoc()->getCurrentAction()>ACT_ZOOM_LAST     ) {

      currentDoc()->saveCurrentActionSettings();
      currentDoc()->setCurrentAction(ACT_ZOOM_PAN);
      currentDoc()->distributeEvent(REVENT_BEGIN);
    }
  }

  if(isOutputValid()) {
    if(currentOut()->getCurrentAction()<ACT_ZOOM_FIRST ||
       currentOut()->getCurrentAction()>ACT_ZOOM_LAST     ) {

      currentOut()->saveCurrentActionSettings();
      currentOut()->setCurrentAction(ACT_ZOOM_PAN);
      currentOut()->distributeEvent(REVENT_BEGIN);
    }
  }
}



// Zoom -> Tagged:
//
void
RAppWin::zoomTagged()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->zoomTagged();
    synchronizeZoom();
    currentOut()->repaint();
    currentDoc()->repaint();
  }
}



// Zoom -> 1:x:
//
void
RAppWin::zoomOne()
{
  if(isDocumentValid(curDocInd)) {

    currentDoc()->zoomOne(paperWidthMM, paperHeightMM);
    synchronizeZoom();
    currentOut()->repaint();
    currentDoc()->repaint();
  }
}


/** Show/hide layer list
*/
void
RAppWin::viewLayerList()
{
  showLayerList( !bShowLayerList->isOn() );
  bShowLayerList->toggle();
}


/** Show/hide library browser
*/
void
RAppWin::viewLibraryBrowser()
{
  showLibraryBrowser( !bShowLibraryBrowser->isOn() );
  bShowLibraryBrowser->toggle();
}


/** Show/hide grid
*/
void
RAppWin::viewGrid()
{
  showGrid( !bShowGrid->isOn() );
  bShowGrid->toggle();
}


// Show -> Grid:
//
void
RAppWin::showGrid(bool _on)
{
  if(isDocumentValid(curDocInd)) {
    if(currentDoc()->isActive()) currentDoc()->setGrid(_on);
  }
  if(currentOut()->isActive()) currentOut()->setGrid(_on);
}


// Show -> Layerlist:
//
void
RAppWin::showLayerList(bool _on)
{
  if(isDocumentValid(curDocInd)) {
    if(currentDoc()->isActive()) {
      cadWidget1->showLayerWidget(_on);
      cadWidget1->resizeChildren();
      RCONFIG->addSetting( "CAD:LayerListVisible", (_on ? "1" : "0") );
    }
  }

  if(currentOut()->isActive()) {
    cadWidget2->showLayerWidget(_on);
    cadWidget2->resizeChildren();
  }
}


// Show -> Object Browser:
//
void
RAppWin::showLibraryBrowser(bool _on)
{
  //if(_on) libraryBrowser->show();
  //else    libraryBrowser->hide();

  if(isDocumentValid(curDocInd)) {
    if(currentDoc()->isActive()) {
      cadWidget1->showLibraryBrowser(_on);
      cadWidget1->resizeChildren();
      RCONFIG->addSetting( "CAD:LibraryBrowserVisible", (_on ? "1" : "0") );
    }
  }

}



// Draw -> Points -> Single:
//
void
RAppWin::pointsSingle()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_POINTS_SINGLE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Normal:
//
void
RAppWin::linesNormal()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_NORMAL);

    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Horizontal:
//
void
RAppWin::linesHor()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_HOR);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Vertical:
//
void
RAppWin::linesVer()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_VER);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> with angle:
//
void
RAppWin::linesAngle()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_ANGLE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Lines -> Rectangles
//
void
RAppWin::linesRect()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_RECT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Parallels:
//
void
RAppWin::linesPara()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_PARA);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Bisectors:
//
void
RAppWin::linesBisector()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_BISECTOR);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Tangents from points to circles:
//
void
RAppWin::linesTan1()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_TAN1);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Tangents from circles to circles:
//
void
RAppWin::linesTan2()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_TAN2);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Lines -> Ortho:
//
void
RAppWin::linesOrtho()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_ORTHO);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Lines -> Ortho with angle:
//
void
RAppWin::linesOrtho2()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_LINES_ORTHO2);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Arcs -> Center, Radius, Angle, Angle:
//
void
RAppWin::arcsCRAA()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_ARCS_CRAA);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Arcs -> 3 points:
//
void
RAppWin::arcsPPP()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_ARCS_PPP);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Arcs -> Parallels:
//
void
RAppWin::arcsPara()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_ARCS_PARA);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Circles -> Center, Radius:
//
void
RAppWin::circlesCR()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_CIRCLES_CR);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Circles -> Center, Linepoint:
//
void
RAppWin::circlesCL()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_CIRCLES_CL);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Circles -> Three points:
//
void
RAppWin::circles3P()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_CIRCLES_3P);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Texts
//
void
RAppWin::textsNormal()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TEXTS_NORMAL);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Dimensions parallel
//
void
RAppWin::dimsPara()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_PARA);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Dimensions horiz
//
void
RAppWin::dimsHor()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_HOR);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Dimensions vertical
//
void
RAppWin::dimsVer()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_VER);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Dimensions angle
//
void
RAppWin::dimsAngle()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_ANGLE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Draw -> Dimensions diameter
//
void
RAppWin::dimsDiam()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_DIAM);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// Draw -> Dimensions radius
//
void
RAppWin::dimsRad()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_RAD);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Draw -> Dimensions -> Arrow:
//
void
RAppWin::dimsArrow()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_DIMS_ARROW);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



// Measure -> Distance Point-Point:
//
void
RAppWin::measurePtPt()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_MEASURE_PTPT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// Measure -> Distance Line-Point:
//
void
RAppWin::measureLnPt()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_MEASURE_LNPT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// Measure -> Distance parallel Line-Line:
//
void
RAppWin::measureParallelLines()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_MEASURE_PARALN);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// Measure -> Angle Line-Line:
//
void
RAppWin::measureAngleLnLn()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_MEASURE_ANGLELNLN);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Measure -> Edit an Element:
//
void
RAppWin::measureEditElement()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_MEASURE_EDITELEMENT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// Measure -> calculate area                                                    
//                                                                              
void                                                                            
RAppWin::measureArea()                                                          
{                                                                               
  if(isDocumentValid(curDocInd)) {                                              
    currentDoc()->setCurrentAction(ACT_MEASURE_AREA);                           
    currentDoc()->distributeEvent(REVENT_BEGIN);                                
  }                                                                             
}


// Draw -> Hatching
//
void
RAppWin::hatching()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_HATCH);
    showTag2Menu();
  }
}



// Tag -> Nothing:
//
//   Untag all:
//
void
RAppWin::tagNothing()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->tagNothing();
  }
}



// Tag -> All:
//
//   Tag all:
//
void
RAppWin::tagAll()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->tagAll();
  }
}



// Tag -> Element:
//
//   Tag single elements:
//
void
RAppWin::tagElement()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TAG_ELEMENT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Contour:
//
//   Tag contours:
//
void
RAppWin::tagContour()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TAG_CONTOUR);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Untag range:
//
//   Untag range:
//
void
RAppWin::untagRange()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_UNTAG_RANGE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Tag range:
//
//   Tag range:
//
void
RAppWin::tagRange()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TAG_RANGE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Untag inters:
//
//   Untag intersected elements:
//
void
RAppWin::untagInters()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_UNTAG_INTERS);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Tag inters:
//
//   Tag intersected elements:
//
void
RAppWin::tagInters()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TAG_INTERS);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Invert:
//
//   Invert selection:
//
void
RAppWin::tagInvert()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->tagInvert();
  }
}



// Tag -> Layer:
//
//   Tag layer:
//
void
RAppWin::tagLayer()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_TAG_LAYER);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}




// Tag -> Doubles:
//
//   Tag double elements
//
void
RAppWin::tagDoubles()
{
  if(isDocumentValid(curDocInd)) {

    RInputDialog idlg(tr("Type the tolerance in mm:"),
                      16,
                      "0.001",
                      qApp->mainWidget(),
                      "idlg");

    if(idlg.exec()) {
      double tolerance = atof(idlg.getInput());
      currentDoc()->tagDoubles(tolerance);
    }
  }
}




// Tag -> Enough:
//
//   Enough: go on with edit function
//
void
RAppWin::tagEnough()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->activateNextAction();
  }
}



// User Action: Edit: move
//
void
RAppWin::editMove()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_MOVE);
    showTag2Menu();
  }
}


// User Action: Edit: rotate
//
void
RAppWin::editRotate()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_ROTATE);
    showTag2Menu();
  }
}


// User Action: Edit: scale
//
void
RAppWin::editScale()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_SCALE);
    showTag2Menu();
  }
}


// User Action: Edit: mirror
//
void
RAppWin::editMirror()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_MIRROR);
    showTag2Menu();
  }
}


// User Action: Edit: move/rotate
//
void
RAppWin::editMoveRotate()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_MOVEROTATE);
    showTag2Menu();
  }
}


// User Action: Edit: rotate/rotate
//
void
RAppWin::editRotateRotate()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_ROTATEROTATE);
    showTag2Menu();
  }
}


// User Action: Edit: trim:
//
void
RAppWin::editTrim()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_TRIM);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: trim 2
//
void
RAppWin::editTrim2()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_TRIM2);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: trim 3
//
void
RAppWin::editTrim3()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_TRIM3);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: cut
//
void
RAppWin::editCut()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_CUT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: cut2
//
void
RAppWin::editCut2()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_CUT2);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: stretch
//
void
RAppWin::editStretch()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_STRETCH);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// User Action: Edit: round
//
void
RAppWin::editRound()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_ROUND);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}

// User Action: Edit: bevel
//
void
RAppWin::editBevel()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_BEVEL);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: textelements
//
void
RAppWin::editTextElements()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_TEXTELEMENTS);
    showTag2Menu();
  }
}


// User Action: Edit: Text
//
void
RAppWin::editText()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_TEXT);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: delete
//
void
RAppWin::editDelete()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_DELETE);
    showTag2Menu();
  }
}


// User Action: Edit: layer
//
void
RAppWin::editLayer()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_LAYER);
    showTag2Menu();
  }
}


// User Action: Edit: zero
//
void
RAppWin::editZero()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_ZERO);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


// User Action: Edit: attributes
//
void
RAppWin::editAttrib()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_ATTRIB);
    showTag2Menu();
  }
}



// User Action: Edit: clip
//
void
RAppWin::editClip()
{
#ifdef DEF_CAM_EXPERT
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_CLIP);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
#endif
}



/*!
 * Edit -> Copy:
 */
void
RAppWin::editClipCopy()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_CLIPCOPY);
    showTag2Menu();
  }
}


/*!
 * Edit -> Cut:
 */
void
RAppWin::editClipCut()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setNextAction(ACT_EDIT_CLIPCUT);
    showTag2Menu();
  }
}


/*!
 * Edit -> Paste:
 */
void
RAppWin::editClipPaste()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_CLIPPASTE);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}


/*!
 * Edit -> Insert part from object library:
 */
void
RAppWin::editInsertPart()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_EDIT_INSERTPART);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
}



/*!
 * Edit -> Convert drawing to font:
 */
void
RAppWin::editConvertToFont()
{
  if(isDocumentValid(curDocInd)) {
    RFileCxf cxf( "", currentDoc() );
    cxf.convertToFont();
  }
}






// Undo last change:
//
void
RAppWin::undo()
{
  currentDoc()->undo();
}



// Redo last change:
//
void
RAppWin::redo()
{
  currentDoc()->redo();
}


// Choose Color:
//
void
RAppWin::chooseColor(int _col)
{
  if(cbAttribColor) {
    for(int c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        graphicDoc[c]->setColor(_col);
      }
    }
  }
}


// Choose Width:
//
void
RAppWin::chooseWidth(int _width)
{
  if(cbAttribWidth) {
    for(int c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        graphicDoc[c]->setWidth(_width);
      }
    }
  }
}


// Choose Style:
//
void
RAppWin::chooseStyle(int _style)
{
  if(cbAttribStyle) {
    for(int c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        graphicDoc[c]->setStyle(RGraphic::numberToStyle(_style));
      }
    }
  }
}


// Snap to nothing:
//
void
RAppWin::snapFree()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setFreeMode();
    untoggleSnapMenu(SNAP_FREE);
  }
}



// Snap to grid:
//
void
RAppWin::snapGrid()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setGridMode();
    untoggleSnapMenu(SNAP_GRID);
  }
}



// Snap to endpoints:
//
void
RAppWin::snapEndpoints()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setEndpointsMode();
    untoggleSnapMenu(SNAP_ENDPOINTS);
  }
}



// Snap to centers:
//
void
RAppWin::snapCenters()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setCentersMode();
    untoggleSnapMenu(SNAP_CENTERS);
  }
}



// Snap to middles:
//
void
RAppWin::snapMiddles()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setMiddlesMode();
    untoggleSnapMenu(SNAP_MIDDLES);
  }
}



// Snap to auto intersections:
//
void
RAppWin::snapAutoInter()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setAutoInterMode();
    untoggleSnapMenu(SNAP_AUTOINTER);
  }
}



// Snap to manual intersections:
//
void
RAppWin::snapManInter()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setManInterMode();
    untoggleSnapMenu(SNAP_MANINTER);
  }
}



// Snap to points with distance on element:
//
void
RAppWin::snapDist()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    inputPanel2->addLineEdit(tr("Distance:"));
    inputPanel2->setValue(0, currentDoc()->snapper()->getSnapDistance());
    currentDoc()->snapper()->setDistMode(currentDoc()->snapper()->getSnapDistance());
    resizeChildren();
    inputPanel2->show();
  }
}



// Snap to points on elements:
//
void
RAppWin::snapOnElement()
{
  if(isDocumentValid(curDocInd)) {
    inputPanel2->reset();
    inputPanel2->hide();
    currentDoc()->snapper()->setOnElementMode();
    untoggleSnapMenu(SNAP_ONELEMENT);
  }
}


// Snap. Keyboard X/Y:
//
void
RAppWin::snapKeyXY()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->snapper()->setKeyXYMode();
    untoggleSnapMenu(SNAP_KEYXY);

    inputPanel2->reset();
    inputPanel2->hide();
    inputPanel2->addLineEdit("X pos.: ");
    inputPanel2->addLineEdit("Y pos.: ");
    inputPanel2->newGroup(tr("Mode:"));
    inputPanel2->addRadioButton(QPixmap(modeabs_xpm), tr("Absolute positioning"), 0, 2);
    inputPanel2->addRadioButton(QPixmap(moderel_xpm), tr("Relative positioning"), 1, 2);
    inputPanel2->endGroup();
    inputPanel2->setValue(0, currentDoc()->snapper()->getLastUsedXPos());
    inputPanel2->setValue(1, currentDoc()->snapper()->getLastUsedYPos());
    inputPanel2->setCheckedRadioButton(currentDoc()->snapper()->getLastUsedPosMode());

    resizeChildren();
    inputPanel2->show();
  }
}


// Snap. Keyboard Alpha/Radius:
//
void
RAppWin::snapKeyAR()
{
  if(isDocumentValid(curDocInd)) {
    currentDoc()->snapper()->setKeyARMode();
    untoggleSnapMenu(SNAP_KEYAR);

    inputPanel2->reset();
    inputPanel2->hide();
    inputPanel2->addLineEdit(tr("Angle:"));
    inputPanel2->addLineEdit(tr("Radius:"));
    inputPanel2->newGroup(tr("Mode:"));
    inputPanel2->addRadioButton(QPixmap(modeabs_xpm), tr("Absolute positioning"), 0, 2);
    inputPanel2->addRadioButton(QPixmap(moderel_xpm), tr("Relative positioning"), 1, 2);
    inputPanel2->endGroup();

    inputPanel2->setValue(0, currentDoc()->snapper()->getLastUsedAngle());
    inputPanel2->setValue(1, currentDoc()->snapper()->getLastUsedRadius());
    inputPanel2->setCheckedRadioButton( currentDoc()->snapper()->getLastUsedPosMode() );
    resizeChildren();
    inputPanel2->show();

  }
}


// Untoggle all buttons from snap menu except _exception:
//
// _exception: e.g.: SNAP_FREE
//
void
RAppWin::untoggleSnapMenu(int _exception)
{
  if(bSnapFree) {
    if(_exception!=SNAP_FREE)      bSnapFree->setOn(false);
    else                           bSnapFree->setOn(true);
  }

  if(bSnapGrid) {
    if(_exception!=SNAP_GRID)      bSnapGrid->setOn(false);
    else                           bSnapGrid->setOn(true);
  }

  if(bSnapEndpoints) {
    if(_exception!=SNAP_ENDPOINTS) bSnapEndpoints->setOn(false);
    else                           bSnapEndpoints->setOn(true);
  }

  if(bSnapCenters) {
    if(_exception!=SNAP_CENTERS)   bSnapCenters->setOn(false);
    else                           bSnapCenters->setOn(true);
  }

  if(bSnapMiddles) {
    if(_exception!=SNAP_MIDDLES)   bSnapMiddles->setOn(false);
    else                           bSnapMiddles->setOn(true);
  }

  if(bSnapAutoInter) {
    if(_exception!=SNAP_AUTOINTER) bSnapAutoInter->setOn(false);
    else                           bSnapAutoInter->setOn(true);
  }

  if(bSnapManInter) {
    if(_exception!=SNAP_MANINTER)  bSnapManInter->setOn(false);
    else                           bSnapManInter->setOn(true);
  }

  if(bSnapDist) {
    if(_exception!=SNAP_DIST)      bSnapDist->setOn(false);
    else                           bSnapDist->setOn(true);
  }

  if(bSnapOnElement) {
    if(_exception!=SNAP_ONELEMENT) bSnapOnElement->setOn(false);
    else                           bSnapOnElement->setOn(true);
  }

  if(bSnapKeyXY) {
    if(_exception!=SNAP_KEYXY) bSnapKeyXY->setOn(false);
    else                       bSnapKeyXY->setOn(true);
  }

  if(bSnapKeyAR) {
    if(_exception!=SNAP_KEYAR) bSnapKeyAR->setOn(false);
    else                       bSnapKeyAR->setOn(true);
  }
}



// CAM -> Auto:
//
void
RAppWin::camAuto()
{
#ifdef DEF_CAM_EXPERT
  if(isDocumentValid(curDocInd) && currentDoc()->count()!=0) {
    currentDoc()->setCurrentAction(ACT_CAM_AUTO);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
  else {
    QMessageBox::warning(this, DEF_APPNAME, tr("You must load a drawing to create an NC program from it."));
  }
#endif
}



// CAM -> Manual:
//
void
RAppWin::camManual()
{
#ifdef DEF_CAM_EXPERT
  if(isDocumentValid(curDocInd)) {
    currentDoc()->setCurrentAction(ACT_CAM_MANUAL);
    currentDoc()->distributeEvent(REVENT_BEGIN);
  }
#endif
}



/** Toggles simulation panel
 */
void
RAppWin::toggleSimulationPanel()
{
#ifdef DEF_CAM_EXPERT
  if( simPanel ) {
    if( simPanel->isVisible() ) {
      simPanel->hide();
      RCONFIG->addSetting("CAM:SimulationPanelVisible", "0");
    }
    else {
      simPanel->show();
      RCONFIG->addSetting("CAM:SimulationPanelVisible", "1");
    }
    resizeChildren();
  }
#endif
}


// Config Machinegenerator:
//
void
RAppWin::configMachineGenerator()
{
#ifdef DEF_CAM_EXPERT
  RChooseMach chooseMach(this, "choosemach");
  if(chooseMach.exec()) {
    RConfigMach configMach( chooseMach.getMachine(), this, "configmach" );
    configMach.exec();
  }
#endif
}



/*! Options -> Preferences
*/
void
RAppWin::optionsPreferences()
{
  RPreferencesDialog prefDlg( this, false, "prefDlg", true );
  prefDlg.exec();
  if(isDocumentValid(curDocInd)) {
    currentDoc()->repaint();
  }

  if(isOutputValid()) {
    currentOut()->repaint();
  }
}


/*! Options -> Drawing
*/
void
RAppWin::optionsDrawing()
{
  if(isDocumentValid(curDocInd)) {
    RPreferencesDialog prefDlg( this, true, "drwDlg", true );
    prefDlg.exec();
    currentDoc()->repaint();
  }
}


/*! Debug -> Contour Check
*/
void
RAppWin::debugContourCheck()
{
  currentDoc()->debugContourCheck();
}


/** Library -> Open Object Browser
*/
/*void
RAppWin::libraryOpenBrowser()
{
  / *if( !libraryBrowser->isVisible() ) {
    //libraryBrowser = new RLibraryBrowser( 0, "libraryBrowser" );
    libraryBrowser->setGeometry( x()+width()-250, y()+80, 153, 400 );
  }* /

  //libraryBrowser->show();
  if(!bShowLibraryBrowser->isDown()) bShowLibraryBrowser->toggle();
}
*/


// Help About:
//
void
RAppWin::helpAbout()
{
  RAboutDialog aboutDlg(this);
  aboutDlg.exec();
}



// Help Manual:
//
void
RAppWin::helpManual()
{
  helpManual( "index.html" );
}



// Help Manual:
//
void
RAppWin::helpManual( const QString& _what )
{
  QString manualStr;

  manualStr = fileSearchSubdir( "doc/" + RCONFIG->getSetting("Language:Language") );

  // Use English as default:
  if( !(QDir( manualStr ).exists()) ) {
    manualStr = fileSearchSubdir( "doc/en" );
  }

  // File given:
  if( _what.contains(".htm") ) {
    manualStr += "/" + _what;
  }

  // OK - Let's search:
  else {
    /** TODO: Add a search routine for finding chapters independant of file names... **/
    //QDir manDir(manualStr);
    //QStringList fileList = manDir.entryList("*.html *.htm");
  }

  RHelpWindow *hw = new RHelpWindow( manualStr, 0 );
  hw->setCaption(QString(DEF_APPNAME) + " " +
                 tr("Help System") + " - " + _what);
  hw->show();


/*  QString manualStr;
  manualStr = RCONFIG->getPrgDir().path();
  manualStr += "/doc/index.html";


#ifndef DEF_WINDOWS
  pid_t   pid;
  //int     status;

  pid = fork();

  RLOG("\nfork: ");
  RLOG((int)pid);

  switch(pid) {

    // Error
    case -1:
      RLOG("\nError (-1): ");
      break;

    case 0:
      if(true) {
        if(!QFileInfo(manualStr).exists()) {
          manualStr = "	/usr/share/doc/qcad/html/index.html";
        }

        RLOG("\nEXEC:");

        RLOG("\n");
        RLOG(RCONFIG->getSetting("Paths:BrowserPath"));
        RLOG("\n");
        RLOG(manualStr);

        execl("/bin/sh",
              "/bin/sh",
              "-c", (RCONFIG->getSetting("Paths:BrowserPath") + " " + manualStr).latin1(), (char*)0);
        exit(1);
      }
      break;

    default:
      //wait(&status);
      //if(status!=0) { RLOG("\nError (wait): "); }
      break;
  }

  //exit(0);

#endif

#ifdef DEF_WINDOWS
 manualStr = manualStr.replace( QRegExp( "/" ), "\\" );
 QFileInfo fi( RCONFIG->getSetting("Paths:BrowserPath") );
 QString exeFile = fi.fileName();
 //spawnl( P_NOWAITO, "C:\\Progra~1\\Intern~1\\Iexplore.exe", "IExplore.exe", manualStr.latin1(), NULL );
 spawnl( P_NOWAITO, RCONFIG->getSetting("Paths:BrowserPath"), exeFile, manualStr.latin1(), NULL );
#endif

*/
}



//
// Event handling:
//

// Resize event:
//
void
RAppWin::resizeEvent(QResizeEvent*)
{
  setMinimumSize(DEF_MINAPPWIDTH, DEF_MINAPPHEIGHT);

  RCONFIG->addSetting("Application:ApplicationWindowWidth", width());
  RCONFIG->addSetting("Application:ApplicationWindowHeight", height());

  resizeChildren();
}



/*! Resizes all children in the application window
*/
void
RAppWin::resizeChildren()
{
  // Some coordinates:
  //
  int toolBarLeftWidth   = DEF_BX2+DEF_BSIZE2+2;
  int toolBarTopHeight   = (RUNNING_WINDOWS ? 27 : 25); 
  int menuBarHeight      = menuBar()->height();
  int panelHeight;                                     // Height of simulation panel for CAM Expert
  int statusBarHeight    = DEF_STATUSBARHEIGHT;
  
  centralWidget->setGeometry( 0,menuBarHeight+toolBarTopHeight,
                              size().width(),size().height()-toolBarTopHeight-menuBarHeight );

#ifdef DEF_CAM_EXPERT
  if( !simPanel || !simPanel->isVisible() ) {
    panelHeight=0;
  }
  else {
    panelHeight = DEF_PANELHEIGHT;
  }
#endif

#ifndef DEF_CAM_EXPERT
  panelHeight=0;
#endif

  int splitterWidth      = centralWidget->size().width()-toolBarLeftWidth;
  int splitterHeight     = centralWidget->size().height()-statusBarHeight-
                           panelHeight;

  RLOG("\nmenuBarHeight: ");
  RLOG(menuBarHeight);
  RLOG("\ntoolBarTopHeight: ");
  RLOG(toolBarTopHeight);
  
  RLOG("\ncentralWidget height: ");
  RLOG( centralWidget->size().height() );
  RLOG("\nthis->height: ");
  RLOG( this->size().height() );
  RLOG("\nsplitter->height: ");
  RLOG( splitterHeight );
  RLOG("\npanelHeight: ");
  RLOG( panelHeight );
  


  // Resize Toolbar Left:
  //
  if(fToolBar) {
    fToolBar->move(0,0);
    fToolBar->resize(toolBarLeftWidth,
                     splitterHeight+panelHeight);
  }

  RLOG("\nResize Splitter");

  // Resize Splitter:
  //
  if(splitter) {
    splitter->move(toolBarLeftWidth,0);
    splitter->resize(splitterWidth,
                     splitterHeight);

         if(mode=='s') splitter->moveToLeft();
    else if(mode=='m') splitter->moveToCenter();
    else               splitter->moveToRight();
  }

  RLOG("\nSplitter resized");

  // Resize Simulation Panel:
  //
#ifdef DEF_CAM_EXPERT
  if(simPanel) {
    simPanel->move(toolBarLeftWidth,
                   splitterHeight);
    simPanel->resize(splitterWidth,
                     panelHeight);
  }
#endif

  // Resize Status Panel:
  //
  if(statusPanel) {
    statusPanel->move(0,
                      splitterHeight+panelHeight);
    statusPanel->resize(centralWidget->size().width(),
                        statusBarHeight);
  }


  // Resize Dimension Dialog:
  //
  if(dimensionDlg) {
    dimensionDlg->move(toolBarLeftWidth+2, 0);

    /*if(inputPanel && inputPanel->isVisible()) {
      dimensionDlg->move(dimensionDlg->x(),
                        inputPanel->y() + inputPanel->height());
    }
    if(inputPanel2 && inputPanel2->isVisible()) {
      dimensionDlg->move(dimensionDlg->x(),
                        inputPanel2->y() + inputPanel2->height());
    }*/
  }

  // Resize Input Panel:
  //
  if(inputPanel) {
    inputPanel->move(toolBarLeftWidth+2, 0);

    if(dimensionDlg && dimensionDlg->isVisible()) {
      inputPanel->move(inputPanel->x(),
                       dimensionDlg->y() + dimensionDlg->height());
    }
  }

  // Resize Input Panel 2:
  //
  if(inputPanel2) {
    inputPanel2->move(toolBarLeftWidth+2, 0);

    if(dimensionDlg && dimensionDlg->isVisible()) {
      inputPanel2->move(inputPanel2->x(),
                        dimensionDlg->y() + dimensionDlg->height());
    }

    if(inputPanel && inputPanel->isVisible()) {
      inputPanel2->move(inputPanel2->x(),
                        inputPanel->y() + inputPanel->height());
    }

  }

  if(cadWidget1) cadWidget1->resizeChildren();
  if(cadWidget2) cadWidget2->resizeChildren();

  RLOG("\nChildren were resized");

}



/*! Synchronize Zoom of the graphics
    \param _back true = sync from output to input. Default=false
*/
void
RAppWin::synchronizeZoom(bool _back)
{
  if(isOutputValid() &&
     isDocumentValid(curDocInd)) {
    if(!_back) {
      currentOut()->setZoomFromGraphic(currentDoc());
    }
    else {
      currentDoc()->setZoomFromGraphic(currentOut());
    }
  }
}



/*! Sets the current program mode (actually nothing more
    than the splitter position)
    \param: _mode 'n': construction, 'm': comparison, 's': Simulation
*/
void
RAppWin::setMode(char _mode)
{
  mode=_mode;

  RLOG("\nsetMode: New Mode is: ");
  RLOG(mode);

  resizeChildren();
}



// Mouse Move event:
//
void
RAppWin::moveEvent(QMoveEvent*)
{
  RCONFIG->addSetting("Application:ApplicationWindowX", x());
  RCONFIG->addSetting("Application:ApplicationWindowY", y());
}



// Mouse Release event:
//
void
RAppWin::mouseReleaseEvent(QMouseEvent* _ev)
{
  if( _ev->button()==RightButton ||
      (_ev->button()==LeftButton && (_ev->state()&ControlButton))) {
    //backMenu();
    if(currentDoc()) {
      currentDoc()->distributeEvent(REVENT_RBUTTONUP);
    }
  }
}



void
RAppWin::keyPressEvent( QKeyEvent * _ev )
{
  RLOG( "\nkeypressevent" );

  //static QTime now;
  //if( now.msecsTo( QTime::currentTime() ) < 250 ) return;
  //now = QTime::currentTime();

  if( keyDown ) return;

  keyDown=true;

  // Special function keys:
  //
  switch( _ev->key() ) {
    case Key_Escape:
      if(currentDoc()) {
        currentDoc()->distributeEvent(REVENT_RBUTTONUP);
      }
      break;

    case Key_Minus: zoomDecrease();    break;
    case Key_Plus: zoomIncrease();     break;

#ifdef DEF_RDEBUG
    case Key_F7: if(currentDoc()) currentDoc()->debugElements(); break;
#endif

    default:                           break;
  }

  // Normal keys:
  //
  switch( _ev->ascii() ) {

    case 'u': undo();                  break;
    case 'U': redo();                  break;

    case 'a': zoomAuto( true );        break;
    case 'r': zoomRedraw();            break;
    case 'w': zoomWindow();            break;
    case 'P': zoomPan();               break;

    case 'T': editTrim();              break;

    case 's': tagAll();                break;
    case 'S': tagNothing();            break;
    case 'o': tagEnough();             break;

    case 'f': snapFree();              break;
    case 'g': snapGrid();              break;
    case 'e': snapEndpoints();         break;
    case 'c': snapCenters();           break;
    case 'm': snapMiddles();           break;
    case 'x': snapAutoInter();         break;
    case 'd': snapDist();              break;
    case 'l': snapOnElement();         break;
    case 'k': snapKeyXY();             break;

    case 'v': editConvertToFont();     break;

    default:                           break;
  }

  // Ctrl-Keys:
  //
  if( _ev->state()==ControlButton ) {
    switch( _ev->key() ) {
      case Key_Z: undo();              break;
      default:                         break;
    }
  }

  // Ctrl-Shift-Keys:
  //
  else if( _ev->state()==(ControlButton|ShiftButton) ) {
    switch( _ev->key() ) {
      case Key_Z: redo();              break;
      default:                         break;
    }
  }

  _ev->accept();

  QMainWindow::keyPressEvent( _ev );
}



void
RAppWin::keyReleaseEvent( QKeyEvent * _ev )
{
  RLOG( "\nkeyreleaseevent" );

  QMainWindow::keyReleaseEvent( _ev );

  keyDown=false;

  _ev->accept();
}



//
//   Document handling:
//


// Clear all documents:
//
void
RAppWin::clearAllDocuments()
{
  int c;

  if(graphicDoc) {
    for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        windowMenu->removeItem(graphicDoc[c]->getMenuId());

        RLOG("\nRemove Menu Item at (clearAllDocuments): ");
        RLOG(graphicDoc[c]->getMenuId());

        delete graphicDoc[c];
      }
      graphicDoc[c]=0;
    }
  }
  curDocInd=-1;
}



// Get the index of the first free document
//   we've found in list 'graphicDoc':
//
// return: - index of free graphic, which is
//           initialized by zero (0)
//         - -1: no free doc found
//
int
RAppWin::getFreeDocumentIndex()
{
  int c;

  if(graphicDoc) {
    for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(!graphicDoc[c]) {
        return c;
      }
    }
  }

  return -1;
}



// Create a new document in list 'graphicDoc'
//   sets curDocInd to new index
//
// return: - index of new graphic, which is
//           initialized by a new graphic
//         - -1: no free doc found
//
int
RAppWin::getNewDocumentIndex()
{
  int newDocIndex;

  newDocIndex=getFreeDocumentIndex();

  if(newDocIndex>=0               &&
     newDocIndex<DEF_MAXDOCUMENTS &&
     cadWidget1                   &&
     graphicDoc                      ) {

    graphicDoc[newDocIndex] = new RGraphic(cadWidget1);
    curDocInd=newDocIndex;

    RLOG("\nNew Doc Index: ");
    RLOG(newDocIndex);

    return newDocIndex;
  }

  return -1;
}



/*! Gets pointer to current document
    \return Pointer to the current (open) document
            (RGraphic) or 0 if no doc is open
*/
RGraphic*
RAppWin::currentDoc()
{
  if(isDocumentValid(curDocInd)) {
    return graphicDoc[curDocInd];
  }
  else {
    return 0;
  }
}



/*! Gets pointer to current output document
    \return Pointer to the current (open) output document
            (RGraphic) or 0 if no output doc exists
*/
RGraphic*
RAppWin::currentOut()
{
  return graphicOut;
}



// Test, if document _docIndex is valid:
//
bool
RAppWin::isDocumentValid(int _docIndex)
{
  if(_docIndex>=0               &&
     _docIndex<DEF_MAXDOCUMENTS &&
     graphicDoc                 &&
     graphicDoc[_docIndex]         ) {

    return true;
  }
  else {
    return false;
  }
}



// Test, if output is valid:
//
bool
RAppWin::isOutputValid()
{
  if(graphicOut) {
    return true;
  }
  else {
    return false;
  }
}



// Show document with Id _id:
//   (SLOT)
//
void
RAppWin::showDocument(int _id)
{
  RLOG("\nShow doc with menu ID: ");
  RLOG(_id);

  int c;
  bool found=false;     // document found
                        // (doc entry selected from menu)

  if(graphicDoc) {

    // Find document with Id _id:
    //
    for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        if(graphicDoc[c]->getMenuId()==_id) {

          // Set index of document to show:
          //
          curDocInd=c;
          found=true;

          RLOG("\nDoc found at index: ");
          RLOG(c);

          break;
        }
      }
    }

    if(found) {

      // Hide other documents:
      //
      hideDocuments(curDocInd);

      // Show document:
      //
      showCurrentDocument();

      showMainMenu();
    }
  }
}



// Show current document (with curDocInd):
//
void
RAppWin::showCurrentDocument()
{
  if(isDocumentValid(curDocInd) &&
     cadWidget1) {

    cadWidget1->setGraphic(currentDoc());

#ifdef DEF_CAM_EXPERT
    if(simPanel)    simPanel->setGraphic(currentDoc());
#endif

    if(inputPanel)  inputPanel->setGraphic(currentDoc());
    if(inputPanel2) inputPanel2->setGraphic(currentDoc());
    cadWidget1->resizeChildren();
    currentDoc()->show();

    if(cbAttribColor) chooseColor(cbAttribColor->currentItem());
    if(cbAttribWidth) chooseWidth(cbAttribWidth->currentItem());
    if(cbAttribStyle) chooseStyle(cbAttribStyle->currentItem());

    RLOG("\nDo connections for the current doc...");

    // Connect signal zoomChanged() from cad graphic
    //   with slot synchronizeZoom() from this:
    //
    disconnect(currentDoc());

    connect(currentDoc(),  SIGNAL(zoomChanged()),
            this,          SLOT(synchronizeZoom()));

    connect(currentDoc(),  SIGNAL(graphicChanged()),
            this,          SLOT(updateCaption()));
			
    RLOG("\ndone");

    synchronizeZoom();
  }
  updateCaption();
}



// Hide all documents except _exception:
//   _exception can be -1: hide all
//
void
RAppWin::hideDocuments(int _exception)
{
  int c;

  // Hide documents except _exception:
  //
  if(graphicDoc) {
    for(c=0; c<DEF_MAXDOCUMENTS; ++c) {
      if(graphicDoc[c]) {
        if(c!=_exception) {
          disconnect(graphicDoc[c],0, this,0);
          graphicDoc[c]->hide();
        }
      }
    }
  }
}




/*!
 * Show the Tag Menu
 */
void
RAppWin::showTagMenu()
{
  changeMenu(TOOLBAR_TAG);
  //bTagElement->setDown(true);
  tagElement();
}



/*!
 * Show the Tag Menu with enough button
 */
void
RAppWin::showTag2Menu()
{
  changeMenu(TOOLBAR_TAG2);
  //bTagElement->setDown(true);
  tagElement();
}



// Update caption:
//   - Program name - Path and file name
//   or
//   - Document xx (for new docs)
//
void 
RAppWin::updateCaption()
{
  QString s;
  
  if(isDocumentValid(curDocInd)) { 
    s = QString(DEF_APPNAME) + " - " + currentDoc()->getFileName() + " ";
    if( currentDoc()->getFlag(G_CHANGED) ) {
      s += tr("(modified)");
    }
    else {
      s += tr("(unmodified)");
    }
  }
  else {
    s = DEF_APPNAME;
  }
  
  setCaption(s);
}







//
//   ToolBar related functions:
//


// Create toolbars for:
//
void
RAppWin::createToolBars()
{
  fToolBar = new RWidget( centralWidget, "ftoolbar", 0, true );
  fToolBar->setFrameStyle( QFrame::Panel|QFrame::Raised );
  fToolBar->show();

  /*
  bBack = new RToolButton( fToolBar, 0, true );
  bBack->setText( tr("back") );
  QToolTip::add( bBack, tr("back") );
  connect( bBack, SIGNAL(clicked()), this, SLOT(backMenu()) );
  bBack->setGeometry( 2, 2, width()-2, DEF_BSIZE2/2 );
  */

  // Tool Bar Main:
  //
    bSubmenuPoints = new RToolButton(fToolBar, 0, true);
    bSubmenuPoints->setPixmap(QPixmap(cadmenupoint_xpm));
    QToolTip::add(bSubmenuPoints, tr("Points"));
    connect(bSubmenuPoints, SIGNAL(clicked()), this, SLOT(showPointMenu()));
    bSubmenuPoints->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuLines = new RToolButton(fToolBar, 0, true);
    bSubmenuLines->setPixmap(QPixmap(cadmenuline_xpm));
    QToolTip::add(bSubmenuLines, tr("Lines"));
    connect(bSubmenuLines, SIGNAL(clicked()), this, SLOT(showLineMenu()));
    bSubmenuLines->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuArcs = new RToolButton(fToolBar, 0, true);
    bSubmenuArcs->setPixmap(QPixmap(cadmenuarc_xpm));
    QToolTip::add(bSubmenuArcs, tr("Arcs"));
    connect(bSubmenuArcs, SIGNAL(clicked()), this, SLOT(showArcMenu()));
    bSubmenuArcs->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuCircles = new RToolButton(fToolBar, 0, true);
    bSubmenuCircles->setPixmap(QPixmap(cadmenucircle_xpm));
    QToolTip::add(bSubmenuCircles, tr("Circles"));
    connect(bSubmenuCircles, SIGNAL(clicked()), this, SLOT(showCircleMenu()));
    bSubmenuCircles->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuTexts = new RToolButton(fToolBar, 0, true);
    bSubmenuTexts->setPixmap(QPixmap(cadmenutext_xpm));
    //QToolTip::add(bSubmenuTexts, tr("Texts"));
    QToolTip::add(bSubmenuTexts, tr("Create Texts"));
    //connect(bSubmenuTexts, SIGNAL(clicked()), this, SLOT(showTextMenu()));
    connect(bSubmenuTexts, SIGNAL(clicked()), this, SLOT(textsNormal()));
    bSubmenuTexts->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuDims = new RToolButton(fToolBar, 0, true);
    bSubmenuDims->setPixmap(QPixmap(cadmenudim_xpm));
    QToolTip::add(bSubmenuDims, tr("Dimensions"));
    connect(bSubmenuDims, SIGNAL(clicked()), this, SLOT(showDimMenu()));
    bSubmenuDims->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuHatch = new RToolButton(fToolBar, 0, true);
    bSubmenuHatch->setPixmap(QPixmap(cadmenuhatch_xpm));
    QToolTip::add(bSubmenuHatch, tr("Create hatchings"));
    connect(bSubmenuHatch, SIGNAL(clicked()), this, SLOT(hatching()));
    bSubmenuHatch->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuMeasure = new RToolButton(fToolBar, 0, true);
    bSubmenuMeasure->setPixmap(QPixmap(cadmenumeasure_xpm));
    QToolTip::add(bSubmenuMeasure, tr("Measure"));
    connect(bSubmenuMeasure, SIGNAL(clicked()), this, SLOT(showMeasureMenu()));
    bSubmenuMeasure->setGeometry(DEF_BX2, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuTag = new RToolButton(fToolBar, 0, true);
    bSubmenuTag->setPixmap(QPixmap(cadmenutag_xpm));
    QToolTip::add(bSubmenuTag, tr("Tag"));
    connect(bSubmenuTag, SIGNAL(clicked()), this, SLOT(showTagMenu()));
    bSubmenuTag->setGeometry(DEF_BX1, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bSubmenuEdit = new RToolButton(fToolBar, 0, true);
    bSubmenuEdit->setPixmap(QPixmap(cadmenuedit_xpm));
    QToolTip::add(bSubmenuEdit, tr("Edit"));
    connect(bSubmenuEdit, SIGNAL(clicked()), this, SLOT(showEditMenu()));
    bSubmenuEdit->setGeometry(DEF_BX2, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

#ifdef DEF_CAM_EXPERT
    bSubmenuCam = new RToolButton(fToolBar, 0, true);
    bSubmenuCam->setPixmap(QPixmap(cadmenucam_xpm));
    QToolTip::add(bSubmenuCam, tr("CAM"));
    //connect(bSubmenuCam, SIGNAL(clicked()), this, SLOT(showCamMenu()));
    connect(bSubmenuCam, SIGNAL(clicked()), this, SLOT(camAuto()));   // No submenu
    bSubmenuCam->setGeometry(DEF_BX1, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);
#endif

  // Tool Bar CAM:
  //
#ifdef DEF_CAM_EXPERT
    bCamAuto = new RToolButton(fToolBar, 0, true);
    bCamAuto->setPixmap(QPixmap(camauto_xpm));
    QToolTip::add(bCamAuto, tr("Automatic Creation"));
    connect(bCamAuto, SIGNAL(clicked()), this, SLOT(camAuto()));
    bCamAuto->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);
#endif

    /*
    bCamManual = new RToolButton(fToolBar, 0, true);
    bCamManual->setPixmap(QPixmap(cammanual_xpm));
    QToolTip::add(bCamManual, tr("Manual Creation"));
    connect(bCamManual, SIGNAL(clicked()), this, SLOT(camManual()));
    bCamManual->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);
    */

  // Tool Bar Points:
  //
    bPointsSingle = new RToolButton(fToolBar, 0, true);
    bPointsSingle->setPixmap(QPixmap(cadpointsingle_xpm));
    QToolTip::add(bPointsSingle, tr("Create points"));
    connect(bPointsSingle, SIGNAL(clicked()), this, SLOT(pointsSingle()));
    bPointsSingle->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);


  // Tool Bar Lines:
  //
    bLinesNormal = new RToolButton(fToolBar, 0, true);
    bLinesNormal->setPixmap(QPixmap(cadlinesnormal_xpm));
    QToolTip::add(bLinesNormal, tr("Create lines"));
    connect(bLinesNormal, SIGNAL(clicked()), this, SLOT(linesNormal()));
    bLinesNormal->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bLinesAngle = new RToolButton(fToolBar, 0, true);
    bLinesAngle->setPixmap(QPixmap(cadlinesangle_xpm));
    QToolTip::add(bLinesAngle, tr("Create lines with a given angle"));
    connect(bLinesAngle, SIGNAL(clicked()), this, SLOT(linesAngle()));
    bLinesAngle->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bLinesHor = new RToolButton(fToolBar, 0, true);
    bLinesHor->setPixmap(QPixmap(cadlineshor_xpm));
    QToolTip::add(bLinesHor, tr("Create horizontal lines"));
    connect(bLinesHor, SIGNAL(clicked()), this, SLOT(linesHor()));
    bLinesHor->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bLinesVer = new RToolButton(fToolBar, 0, true);
    bLinesVer->setPixmap(QPixmap(cadlinesver_xpm));
    QToolTip::add(bLinesVer, tr("Create vertical lines"));
    connect(bLinesVer, SIGNAL(clicked()), this, SLOT(linesVer()));
    bLinesVer->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bLinesRect = new RToolButton(fToolBar, 0, true);
    bLinesRect->setPixmap(QPixmap(cadlinesrect_xpm));
    QToolTip::add(bLinesRect, tr("Create rectangles"));
    connect(bLinesRect, SIGNAL(clicked()), this, SLOT(linesRect()));
    bLinesRect->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bLinesPara = new RToolButton(fToolBar, 0, true);
    bLinesPara->setPixmap(QPixmap(cadlinespara_xpm));
    QToolTip::add(bLinesPara, tr("Create parallels"));
    connect(bLinesPara, SIGNAL(clicked()), this, SLOT(linesPara()));
    bLinesPara->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bLinesBisector = new RToolButton(fToolBar, 0, true);
    bLinesBisector->setPixmap(QPixmap(cadlinesbisector_xpm));
    QToolTip::add(bLinesBisector, tr("Create bisectors"));
    connect(bLinesBisector, SIGNAL(clicked()), this, SLOT(linesBisector()));
    bLinesBisector->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bLinesTan1 = new RToolButton(fToolBar, 0, true);
    bLinesTan1->setPixmap(QPixmap(cadlinestan1_xpm));
    QToolTip::add(bLinesTan1, tr("Create tangents (Point-Circle)"));
    connect(bLinesTan1, SIGNAL(clicked()), this, SLOT(linesTan1()));
    bLinesTan1->setGeometry(DEF_BX2, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bLinesTan2 = new RToolButton(fToolBar, 0, true);
    bLinesTan2->setPixmap(QPixmap(cadlinestan2_xpm));
    QToolTip::add(bLinesTan2, tr("Create tangents (Circle-Circle)"));
    connect(bLinesTan2, SIGNAL(clicked()), this, SLOT(linesTan2()));
    bLinesTan2->setGeometry(DEF_BX1, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bLinesOrtho = new RToolButton(fToolBar, 0, true);
    bLinesOrtho->setPixmap(QPixmap(cadlinesortho_xpm));
    QToolTip::add(bLinesOrtho, tr("Create orthogonal lines"));
    connect(bLinesOrtho, SIGNAL(clicked()), this, SLOT(linesOrtho()));
    bLinesOrtho->setGeometry(DEF_BX2, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bLinesOrtho2 = new RToolButton(fToolBar, 0, true);
    bLinesOrtho2->setPixmap(QPixmap(cadlinesortho2_xpm));
    QToolTip::add(bLinesOrtho2, tr("Create lines with relative angles"));
    connect(bLinesOrtho2, SIGNAL(clicked()), this, SLOT(linesOrtho2()));
    bLinesOrtho2->setGeometry(DEF_BX1, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);

  // Tool Bar Arcs:
  //
    bArcsCRAA = new RToolButton(fToolBar, 0, true);
    bArcsCRAA->setPixmap(QPixmap(cadarcscraa_xpm));
    QToolTip::add(bArcsCRAA, tr("Create arcs with center, radius and angles"));
    connect(bArcsCRAA, SIGNAL(clicked()), this, SLOT(arcsCRAA()));
    bArcsCRAA->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bArcsPPP = new RToolButton(fToolBar, 0, true);
    bArcsPPP->setPixmap(QPixmap(cadarcsppp_xpm));
    QToolTip::add(bArcsPPP, tr("Create arcs with three points"));
    connect(bArcsPPP, SIGNAL(clicked()), this, SLOT(arcsPPP()));
    bArcsPPP->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bArcsPara = new RToolButton(fToolBar, 0, true);
    bArcsPara->setPixmap(QPixmap(cadarcspara_xpm));
    QToolTip::add(bArcsPara, tr("Create arcs parallel to others"));
    connect(bArcsPara, SIGNAL(clicked()), this, SLOT(arcsPara()));
    bArcsPara->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

  // Tool bar Circles:
  //
    bCirclesCR = new RToolButton(fToolBar, 0, true);
    bCirclesCR->setPixmap(QPixmap(cadcirclescr_xpm));
    QToolTip::add(bCirclesCR, tr("Create Circles by Center and Radius"));
    connect(bCirclesCR, SIGNAL(clicked()), this, SLOT(circlesCR()));
    bCirclesCR->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bCirclesCL = new RToolButton(fToolBar, 0, true);
    bCirclesCL->setPixmap(QPixmap(cadcirclescl_xpm));
    QToolTip::add(bCirclesCL, tr("Create circles by center and point"));
    connect(bCirclesCL, SIGNAL(clicked()), this, SLOT(circlesCL()));
    bCirclesCL->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bCircles3P = new RToolButton(fToolBar, 0, true);
    bCircles3P->setPixmap(QPixmap(cadcircles3p_xpm));
    QToolTip::add(bCircles3P, tr("Create circles by three points"));
    connect(bCircles3P, SIGNAL(clicked()), this, SLOT(circles3P()));
    bCircles3P->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

  // Tool bar Dimensions:
  //
    bDimsPara = new RToolButton(fToolBar, 0, true);
    bDimsPara->setPixmap(QPixmap(caddimspara_xpm));
    QToolTip::add(bDimsPara, tr("Create dimension lines parallel to elements"));
    connect(bDimsPara, SIGNAL(clicked()), this, SLOT(dimsPara()));
    bDimsPara->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bDimsHor = new RToolButton(fToolBar, 0, true);
    bDimsHor->setPixmap(QPixmap(caddimshor_xpm));
    QToolTip::add(bDimsHor, tr("Create horizontal dimension lines"));
    connect(bDimsHor, SIGNAL(clicked()), this, SLOT(dimsHor()));
    bDimsHor->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bDimsVer = new RToolButton(fToolBar, 0, true);
    bDimsVer->setPixmap(QPixmap(caddimsver_xpm));
    QToolTip::add(bDimsVer, tr("Create vertical dimension lines"));
    connect(bDimsVer, SIGNAL(clicked()), this, SLOT(dimsVer()));
    bDimsVer->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bDimsAngle = new RToolButton(fToolBar, 0, true);
    bDimsAngle->setPixmap(QPixmap(caddimsangle_xpm));
    QToolTip::add(bDimsAngle, tr("Angle dimensioning"));
    connect(bDimsAngle, SIGNAL(clicked()), this, SLOT(dimsAngle()));
    bDimsAngle->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bDimsDiam = new RToolButton(fToolBar, 0, true);
    bDimsDiam->setPixmap(QPixmap(caddimsdiam_xpm));
    QToolTip::add(bDimsDiam, tr("Diameter dimensioning"));
    connect(bDimsDiam, SIGNAL(clicked()), this, SLOT(dimsDiam()));
    bDimsDiam->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bDimsRad = new RToolButton(fToolBar, 0, true);
    bDimsRad->setPixmap(QPixmap(caddimsrad_xpm));
    QToolTip::add(bDimsRad, tr("Radius dimensioning"));
    connect(bDimsRad, SIGNAL(clicked()), this, SLOT(dimsRad()));
    bDimsRad->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bDimsArrow = new RToolButton(fToolBar, 0, true);
    bDimsArrow->setPixmap(QPixmap(caddimsarrow_xpm));
    QToolTip::add(bDimsArrow, tr("Create arrows"));
    connect(bDimsArrow, SIGNAL(clicked()), this, SLOT(dimsArrow()));
    bDimsArrow->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

  // Tool bar Measure:
  //
    bMeasurePtPt = new RToolButton(fToolBar, 0, true);
    bMeasurePtPt->setPixmap(QPixmap(measureptpt_xpm));
    QToolTip::add(bMeasurePtPt, tr("Distance between two points"));
    connect(bMeasurePtPt, SIGNAL(clicked()), this, SLOT(measurePtPt()));
    bMeasurePtPt->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bMeasureLnPt = new RToolButton(fToolBar, 0, true);
    bMeasureLnPt->setPixmap(QPixmap(measurelnpt_xpm));
    QToolTip::add(bMeasureLnPt, tr("Distance between a line and a point"));
    connect(bMeasureLnPt, SIGNAL(clicked()), this, SLOT(measureLnPt()));
    bMeasureLnPt->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bMeasureParallelLines = new RToolButton(fToolBar, 0, true);
    bMeasureParallelLines->setPixmap(QPixmap(measureparallellines_xpm));
    QToolTip::add(bMeasureParallelLines, tr("Distance between two parallel lines"));
    connect(bMeasureParallelLines, SIGNAL(clicked()), this, SLOT(measureParallelLines()));
    bMeasureParallelLines->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bMeasureAngleLnLn = new RToolButton(fToolBar, 0, true);
    bMeasureAngleLnLn->setPixmap(QPixmap(measureanglelnln_xpm));
    QToolTip::add(bMeasureAngleLnLn, tr("Angle between two lines"));
    connect(bMeasureAngleLnLn, SIGNAL(clicked()), this, SLOT(measureAngleLnLn()));
    bMeasureAngleLnLn->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bMeasureEditElement = new RToolButton(fToolBar, 0, true);
    bMeasureEditElement->setPixmap(QPixmap(measureeditelement_xpm));
    QToolTip::add(bMeasureEditElement, tr("Edit geometry of an element"));
    connect(bMeasureEditElement, SIGNAL(clicked()), this, SLOT(measureEditElement()));
    bMeasureEditElement->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

		bMeasureArea = new RToolButton(fToolBar, 0, true);                          
    bMeasureArea->setPixmap(QPixmap(measurearea_xpm));                          
    QToolTip::add(bMeasureArea, tr("calculate area/circumference"));                                     
    connect(bMeasureArea, SIGNAL(clicked()), this, SLOT(measureArea()));        
    bMeasureArea->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

  // Tool bar Edit:
  //
    bEditMove = new RToolButton(fToolBar, 0, true);
    bEditMove->setPixmap(QPixmap(editmove_xpm));
    QToolTip::add(bEditMove, tr("Move objects"));
    connect(bEditMove, SIGNAL(clicked()), this, SLOT(editMove()));
    bEditMove->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bEditRotate = new RToolButton(fToolBar, 0, true);
    bEditRotate->setPixmap(QPixmap(editrotate_xpm));
    QToolTip::add(bEditRotate, tr("Rotate objects"));
    connect(bEditRotate, SIGNAL(clicked()), this, SLOT(editRotate()));
    bEditRotate->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bEditScale = new RToolButton(fToolBar, 0, true);
    bEditScale->setPixmap(QPixmap(editscale_xpm));
    QToolTip::add(bEditScale, tr("Scale objects"));
    connect(bEditScale, SIGNAL(clicked()), this, SLOT(editScale()));
    bEditScale->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bEditMirror = new RToolButton(fToolBar, 0, true);
    bEditMirror->setPixmap(QPixmap(editmirror_xpm));
    QToolTip::add(bEditMirror, tr("Mirror objects"));
    connect(bEditMirror, SIGNAL(clicked()), this, SLOT(editMirror()));
    bEditMirror->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bEditMoveRotate = new RToolButton(fToolBar, 0, true);
    bEditMoveRotate->setPixmap(QPixmap(editmoverot_xpm));
    QToolTip::add(bEditMoveRotate, tr("Move and rotate object"));
    connect(bEditMoveRotate, SIGNAL(clicked()), this, SLOT(editMoveRotate()));
    bEditMoveRotate->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bEditRotateRotate = new RToolButton(fToolBar, 0, true);
    bEditRotateRotate->setPixmap(QPixmap(editrotrot_xpm));
    QToolTip::add(bEditRotateRotate, tr("Rotate objects around two centers"));
    connect(bEditRotateRotate, SIGNAL(clicked()), this, SLOT(editRotateRotate()));
    bEditRotateRotate->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bEditTrim = new RToolButton(fToolBar, 0, true);
    bEditTrim->setPixmap(QPixmap(edittrim_xpm));
    //bEditTrim->setAccel( SHIFT+'T' );
    QToolTip::add(bEditTrim, tr("Trim objects"));
    connect(bEditTrim, SIGNAL(clicked()), this, SLOT(editTrim()));
    bEditTrim->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bEditTrim2 = new RToolButton(fToolBar, 0, true);
    bEditTrim2->setPixmap(QPixmap(edittrim2_xpm));
    QToolTip::add(bEditTrim2, tr("Trim two objects"));
    connect(bEditTrim2, SIGNAL(clicked()), this, SLOT(editTrim2()));
    bEditTrim2->setGeometry(DEF_BX2, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bEditTrim3 = new RToolButton(fToolBar, 0, true);
    bEditTrim3->setPixmap(QPixmap(edittrim3_xpm));
    QToolTip::add(bEditTrim3, tr("Trim lines by a given amount"));
    connect(bEditTrim3, SIGNAL(clicked()), this, SLOT(editTrim3()));
    bEditTrim3->setGeometry(DEF_BX1, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bEditCut = new RToolButton(fToolBar, 0, true);
    bEditCut->setPixmap(QPixmap(editcut_xpm));
    QToolTip::add(bEditCut, tr("Divide"));
    connect(bEditCut, SIGNAL(clicked()), this, SLOT(editCut()));
    bEditCut->setGeometry(DEF_BX2, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bEditCut2 = new RToolButton(fToolBar, 0, true);
    bEditCut2->setPixmap(QPixmap(editcut2_xpm));
    QToolTip::add(bEditCut2, tr("Break between two points"));
    connect(bEditCut2, SIGNAL(clicked()), this, SLOT(editCut2()));
    bEditCut2->setGeometry(DEF_BX2, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);

    bEditStretch = new RToolButton(fToolBar, 0, true);
    bEditStretch->setPixmap(QPixmap(editstretch_xpm));
    QToolTip::add(bEditStretch, tr("Stretch"));
    connect(bEditStretch, SIGNAL(clicked()), this, SLOT(editStretch()));
    bEditStretch->setGeometry(DEF_BX1, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);

    bEditRound = new RToolButton(fToolBar, 0, true);
    bEditRound->setPixmap(QPixmap(editround_xpm));
    QToolTip::add(bEditRound, tr("Round"));
    connect(bEditRound, SIGNAL(clicked()), this, SLOT(editRound()));
    bEditRound->setGeometry(DEF_BX2, DEF_BY7, DEF_BSIZE2, DEF_BSIZE2);

    bEditBevel = new RToolButton(fToolBar, 0, true);
    bEditBevel->setPixmap(QPixmap(editbevel_xpm));
    QToolTip::add(bEditBevel, tr("Bevel"));
    connect(bEditBevel, SIGNAL(clicked()), this, SLOT(editBevel()));
    bEditBevel->setGeometry(DEF_BX1, DEF_BY7, DEF_BSIZE2, DEF_BSIZE2);

    bEditTextElements = new RToolButton(fToolBar, 0, true);
    bEditTextElements->setPixmap(QPixmap(edittextelement_xpm));
    QToolTip::add(bEditTextElements, tr("Convert Text into\nsingle elements"));
    connect(bEditTextElements, SIGNAL(clicked()), this, SLOT(editTextElements()));
    bEditTextElements->setGeometry(DEF_BX2, DEF_BY8, DEF_BSIZE2, DEF_BSIZE2);

    bEditText = new RToolButton(fToolBar, 0, true);
    bEditText->setPixmap(QPixmap(edittext_xpm));
    QToolTip::add(bEditText, tr("Edit text"));
    connect(bEditText, SIGNAL(clicked()), this, SLOT(editText()));
    bEditText->setGeometry(DEF_BX1, DEF_BY8, DEF_BSIZE2, DEF_BSIZE2);

    bEditDelete = new RToolButton(fToolBar, 0, true);
    bEditDelete->setPixmap(QPixmap(editdelete_xpm));
    QToolTip::add(bEditDelete, tr("Delete objects"));
    connect(bEditDelete, SIGNAL(clicked()), this, SLOT(editDelete()));
    bEditDelete->setGeometry(DEF_BX2, DEF_BY9, DEF_BSIZE2, DEF_BSIZE2);

    bEditLayer = new RToolButton(fToolBar, 0, true);
    bEditLayer->setPixmap(QPixmap(editlayer_xpm));
    QToolTip::add(bEditLayer, tr("Move objects to a layer"));
    connect(bEditLayer, SIGNAL(clicked()), this, SLOT(editLayer()));
    bEditLayer->setGeometry(DEF_BX1, DEF_BY9, DEF_BSIZE2, DEF_BSIZE2);

    bEditZero = new RToolButton(fToolBar, 0, true);
    bEditZero->setPixmap(QPixmap(editzero_xpm));
    QToolTip::add(bEditZero, tr("Move zero"));
    connect(bEditZero, SIGNAL(clicked()), this, SLOT(editZero()));
    bEditZero->setGeometry(DEF_BX2, DEF_BY10, DEF_BSIZE2, DEF_BSIZE2);

    bEditAttrib = new RToolButton(fToolBar, 0, true);
    bEditAttrib->setPixmap(QPixmap(editattrib_xpm));
    QToolTip::add(bEditAttrib, tr("Change attributes"));
    connect(bEditAttrib, SIGNAL(clicked()), this, SLOT(editAttrib()));
    bEditAttrib->setGeometry(DEF_BX1, DEF_BY10, DEF_BSIZE2, DEF_BSIZE2);

#ifdef DEF_CAM_EXPERT
    bEditClip = new RToolButton(fToolBar, 0, true);
    bEditClip->setPixmap(QPixmap(editclip_xpm));
    QToolTip::add(bEditClip, tr("Clip with given gap"));
    connect(bEditClip, SIGNAL(clicked()), this, SLOT(editClip()));
    bEditClip->setGeometry(DEF_BX2, DEF_BY11, DEF_BSIZE2, DEF_BSIZE2);
#endif


  // Tool bar Snap:
  //
    bSnapFree = new RToolButton(fToolBar, 0, true);
    bSnapFree->setPixmap(QPixmap(snapfree_xpm));
    //bSnapFree->setStyle(MotifStyle);
    bSnapFree->setToggleButton(true);
    bSnapFree->setOn(false);
    //bSnapFree->setAccel(Key_F);
    QToolTip::add(bSnapFree, tr("Snap to nothing [F]"));
    connect(bSnapFree, SIGNAL(clicked()), this, SLOT(snapFree()));
    bSnapFree->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bSnapGrid = new RToolButton(fToolBar, 0, true);
    bSnapGrid->setPixmap(QPixmap(snapgrid_xpm));
    //bSnapGrid->setStyle(MotifStyle);
    bSnapGrid->setToggleButton(true);
    bSnapGrid->setOn(false);
    //bSnapGrid->setAccel(Key_G);
    QToolTip::add(bSnapGrid, tr("Snap to grid points [G]"));
    connect(bSnapGrid, SIGNAL(clicked()), this, SLOT(snapGrid()));
    bSnapGrid->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bSnapEndpoints = new RToolButton(fToolBar, 0, true);
    bSnapEndpoints->setPixmap(QPixmap(snapendpoints_xpm));
    //bSnapEndpoints->setStyle(MotifStyle);
    bSnapEndpoints->setToggleButton(true);
    bSnapEndpoints->setOn(false);
    //bSnapEndpoints->setAccel(Key_E);
    QToolTip::add(bSnapEndpoints, tr("Snap to endpoints [E]"));
    connect(bSnapEndpoints, SIGNAL(clicked()), this, SLOT(snapEndpoints()));
    bSnapEndpoints->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bSnapCenters = new RToolButton(fToolBar, 0, true);
    bSnapCenters->setPixmap(QPixmap(snapcenters_xpm));
    //bSnapCenters->setStyle(MotifStyle);
    bSnapCenters->setToggleButton(true);
    bSnapCenters->setOn(false);
    //bSnapCenters->setAccel(Key_C);
    QToolTip::add(bSnapCenters, tr("Snap to centers [C]"));
    connect(bSnapCenters, SIGNAL(clicked()), this, SLOT(snapCenters()));
    bSnapCenters->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bSnapMiddles = new RToolButton(fToolBar, 0, true);
    bSnapMiddles->setPixmap(QPixmap(snapmiddles_xpm));
    //bSnapMiddles->setStyle(MotifStyle);
    bSnapMiddles->setToggleButton(true);
    bSnapMiddles->setOn(false);
    //bSnapMiddles->setAccel(Key_M);
    QToolTip::add(bSnapMiddles, tr("Snap to middles [M]"));
    connect(bSnapMiddles, SIGNAL(clicked()), this, SLOT(snapMiddles()));
    bSnapMiddles->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bSnapAutoInter = new RToolButton(fToolBar, 0, true);
    bSnapAutoInter->setPixmap(QPixmap(snapautointer_xpm));
    //bSnapAutoInter->setStyle(MotifStyle);
    bSnapAutoInter->setToggleButton(true);
    bSnapAutoInter->setOn(false);
    //bSnapAutoInter->setAccel(Key_X);
    QToolTip::add(bSnapAutoInter, tr("Snap automatically to intersections [X]"));
    connect(bSnapAutoInter, SIGNAL(clicked()), this, SLOT(snapAutoInter()));
    bSnapAutoInter->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bSnapManInter = new RToolButton(fToolBar, 0, true);
    bSnapManInter->setPixmap(QPixmap(snapmaninter_xpm));
    //bSnapManInter->setStyle(MotifStyle);
    bSnapManInter->setToggleButton(true);
    bSnapManInter->setOn(false);
    QToolTip::add(bSnapManInter, tr("Snap manually to intersections"));
    connect(bSnapManInter, SIGNAL(clicked()), this, SLOT(snapManInter()));
    bSnapManInter->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bSnapDist = new RToolButton(fToolBar, 0, true);
    bSnapDist->setPixmap(QPixmap(snapdist_xpm));
    //bSnapDist->setStyle(MotifStyle);
    bSnapDist->setToggleButton(true);
    bSnapDist->setOn(false);
    //bSnapDist->setAccel(Key_D);
    QToolTip::add(bSnapDist, tr("Snap to points with a defined\ndistance from endpoint"));
    connect(bSnapDist, SIGNAL(clicked()), this, SLOT(snapDist()));
    bSnapDist->setGeometry(DEF_BX2, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bSnapOnElement = new RToolButton(fToolBar, 0, true);
    bSnapOnElement->setPixmap(QPixmap(snaponelement_xpm));
    //bSnapOnElement->setStyle(MotifStyle);
    bSnapOnElement->setToggleButton(true);
    bSnapOnElement->setOn(false);
    //bSnapOnElement->setAccel(Key_L);
    QToolTip::add(bSnapOnElement, tr("Snap to points on elements [L]"));
    connect(bSnapOnElement, SIGNAL(clicked()), this, SLOT(snapOnElement()));
    bSnapOnElement->setGeometry(DEF_BX1, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bSnapKeyXY = new RToolButton(fToolBar, 0, true);
    bSnapKeyXY->setPixmap(QPixmap(snapkeyxy_xpm));
    bSnapKeyXY->setToggleButton(true);
    bSnapKeyXY->setOn(false);
    //bSnapKeyXY->setAccel(Key_K);
    QToolTip::add(bSnapKeyXY, tr("Input coordinate (X/Y) [K]"));
    connect(bSnapKeyXY, SIGNAL(clicked()), this, SLOT(snapKeyXY()));
    bSnapKeyXY->setGeometry(DEF_BX2, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bSnapKeyAR = new RToolButton(fToolBar, 0, true);
    bSnapKeyAR->setPixmap(QPixmap(snapkeyar_xpm));
    bSnapKeyAR->setToggleButton(true);
    bSnapKeyAR->setOn(false);
    QToolTip::add(bSnapKeyAR, tr("Input coordinate (Angle/Radius)"));
    connect(bSnapKeyAR, SIGNAL(clicked()), this, SLOT(snapKeyAR()));
    bSnapKeyAR->setGeometry(DEF_BX1, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);

  // Tool bar Tag:
  //
    bTagNothing = new RToolButton(fToolBar, 0, true);
    bTagNothing->setPixmap(QPixmap(tagnothing_xpm));
    QToolTip::add(bTagNothing, tr("Untag all elements"));
    connect(bTagNothing, SIGNAL(clicked()), this, SLOT(tagNothing()));
    bTagNothing->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bTagAll = new RToolButton(fToolBar, 0, true);
    bTagAll->setPixmap(QPixmap(tagall_xpm));
    QToolTip::add(bTagAll, tr("Tag all elements"));
    connect(bTagAll, SIGNAL(clicked()), this, SLOT(tagAll()));
    bTagAll->setGeometry(DEF_BX2, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);

    bTagElement = new RToolButton(fToolBar, 0, true);
    bTagElement->setPixmap(QPixmap(tagelement_xpm));
    QToolTip::add(bTagElement, tr("(Un-)tag single elements"));
    connect(bTagElement, SIGNAL(clicked()), this, SLOT(tagElement()));
    bTagElement->setGeometry(DEF_BX1, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bTagContour = new RToolButton(fToolBar, 0, true);
    bTagContour->setPixmap(QPixmap(tagcontour_xpm));
    QToolTip::add(bTagContour, tr("(Un-)tag Contours"));
    connect(bTagContour, SIGNAL(clicked()), this, SLOT(tagContour()));
    bTagContour->setGeometry(DEF_BX2, DEF_BY2, DEF_BSIZE2, DEF_BSIZE2);

    bUntagRange = new RToolButton(fToolBar, 0, true);
    bUntagRange->setPixmap(QPixmap(untagrange_xpm));
    QToolTip::add(bUntagRange, tr("Untag range"));
    connect(bUntagRange, SIGNAL(clicked()), this, SLOT(untagRange()));
    bUntagRange->setGeometry(DEF_BX1, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bTagRange = new RToolButton(fToolBar, 0, true);
    bTagRange->setPixmap(QPixmap(tagrange_xpm));
    QToolTip::add(bTagRange, tr("Tag range"));
    connect(bTagRange, SIGNAL(clicked()), this, SLOT(tagRange()));
    bTagRange->setGeometry(DEF_BX2, DEF_BY3, DEF_BSIZE2, DEF_BSIZE2);

    bUntagInters = new RToolButton(fToolBar, 0, true);
    bUntagInters->setPixmap(QPixmap(untaginters_xpm));
    QToolTip::add(bUntagInters, tr("Untag elements intersected by a line"));
    connect(bUntagInters, SIGNAL(clicked()), this, SLOT(untagInters()));
    bUntagInters->setGeometry(DEF_BX1, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bTagInters = new RToolButton(fToolBar, 0, true);
    bTagInters->setPixmap(QPixmap(taginters_xpm));
    QToolTip::add(bTagInters, tr("Tag elements intersected by a line"));
    connect(bTagInters, SIGNAL(clicked()), this, SLOT(tagInters()));
    bTagInters->setGeometry(DEF_BX2, DEF_BY4, DEF_BSIZE2, DEF_BSIZE2);

    bTagInvert = new RToolButton(fToolBar, 0, true);
    bTagInvert->setPixmap(QPixmap(taginvert_xpm));
    QToolTip::add(bTagInvert, tr("Invert selection"));
    connect(bTagInvert, SIGNAL(clicked()), this, SLOT(tagInvert()));
    bTagInvert->setGeometry(DEF_BX1, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bTagLayer = new RToolButton(fToolBar, 0, true);
    bTagLayer->setPixmap(QPixmap(taglayer_xpm));
    QToolTip::add(bTagLayer, tr("(Un-)tag all elements of a layer"));
    connect(bTagLayer, SIGNAL(clicked()), this, SLOT(tagLayer()));
    bTagLayer->setGeometry(DEF_BX2, DEF_BY5, DEF_BSIZE2, DEF_BSIZE2);

    bTagDoubles = new RToolButton(fToolBar, 0, true);
    bTagDoubles->setPixmap(QPixmap(tagdoubles_xpm));
    QToolTip::add(bTagDoubles, tr("Tag double elements"));
    connect(bTagDoubles, SIGNAL(clicked()), this, SLOT(tagDoubles()));
    bTagDoubles->setGeometry(DEF_BX1, DEF_BY6, DEF_BSIZE2, DEF_BSIZE2);

    bTagEnough = new RToolButton(fToolBar, 0, true);
    bTagEnough->setPixmap(QPixmap(tagenough_xpm));
    QToolTip::add(bTagEnough, tr("Enough (go on with the edit function)"));
    connect(bTagEnough, SIGNAL(clicked()), this, SLOT(tagEnough()));
    bTagEnough->setGeometry(DEF_BX1, DEF_BY7, 2*DEF_BSIZE2, DEF_BSIZE2);


  // Tool Bar File (top):
  //
  toolBarFile = new QToolBar("Tool Bar File", this, Top, false, "toolbarfile");
  //toolBarFile->setGeometry(0, 0, 0, 0);
    bFileNew = new QToolButton(QPixmap(filenew_xpm),
                               tr("File new"), tr("Create a new Drawing"),
                               this, SLOT(fileNew()), toolBarFile, "filenew");
    bFileNew->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bFileOpen = new QToolButton(QPixmap(fileopen_xpm),
                                tr("File open"), tr("Open a file"),
                                this, SLOT(fileOpen()), toolBarFile, "fileopen");
    bFileOpen->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bFileClose = new QToolButton(QPixmap(fileclose_xpm),
                                 tr("Close input file"),
                                 tr("Close current file (left side)"),
                                 this, SLOT(fileClose()), toolBarFile, "fileclose");
    bFileClose->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
/*#ifdef DEF_CAM_EXPERT
    bFileCloseOutput = new QToolButton(QPixmap(filecloseoutput_xpm), tr("Close output file"), tr("Close current output (right)"),
                               this, SLOT(fileCloseOutput()), toolBarFile, "filecloseoutput");
    bFileCloseOutput->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
#endif*/
    bFileSave = new QToolButton(QPixmap(filesave_xpm),
                                tr("File save"), tr("Save drawing as DXF"),
                                this, SLOT(fileSave()), toolBarFile, "submenuarc");
    bFileSave->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    toolBarFile->addSeparator();

    bFilePrintPreview = new QToolButton(QPixmap(fileprintprev_xpm),
                                        tr("Print Preview"),
                                        tr("Preview Drawing for Printing"),
                                        this, SLOT(filePrintPreview()), toolBarFile, "printpreview");
    bFilePrintPreview->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bFilePrintPreview->setToggleButton(true);
    bFilePrintPreview->setOn(false);
    bFilePrint = new QToolButton(QPixmap(fileprint_xpm),
                                 tr("Print"), tr("Print Drawing"),
                                 this, SLOT(filePrint()), toolBarFile, "fileprint");
    bFilePrint->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

  // Tool Bar Zoom (top):
  //
  toolBarZoom = new QToolBar("Tool Bar Zoom", this, Top, false, "toolbarzoom");
  toolBarZoom->setGeometry(0, 0, 0, 0);
    bZoomRedraw = new QToolButton(QPixmap(zoomredraw_xpm),
                                  tr("Redraw"), tr("Redraw the graphic"),
                                  this, SLOT(zoomRedraw()), toolBarZoom, "zoomredraw");
    bZoomRedraw->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomAuto = new QToolButton(QPixmap(zoomauto_xpm),
                                tr("Autozoom"), tr("Show the whole graphic"),
                                this, SLOT(zoomAuto()), toolBarZoom, "zoomauto");
    bZoomAuto->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomToPage = new QToolButton(QPixmap(zoomtopage_xpm),
                                  tr("Zoom to page"), tr("Zoom to page"),
                                  this, SLOT(zoomToPage()), toolBarZoom, "zoomtopage");
    bZoomToPage->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomIncrease = new QToolButton(QPixmap(zoomincrease_xpm),
                                    tr("Increase Zoom"), tr("Increase the zoom by factor 1.5"),
                                    this, SLOT(zoomIncrease()), toolBarZoom, "zoomincrease");
    bZoomIncrease->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomDecrease = new QToolButton(QPixmap(zoomdecrease_xpm),
                                    tr("Decrease Zoom"),
                                    tr("Decrease the zoom by factor 1.5"),
                                    this, SLOT(zoomDecrease()), toolBarZoom, "zoomdecrease");
    bZoomDecrease->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomWindow = new QToolButton(QPixmap(zoomwindow_xpm),
                                  tr("Zoom Window"), tr("Draw Window to zoom"),
                                  this, SLOT(zoomWindow()), toolBarZoom, "zoomwindow");
    bZoomWindow->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomPan = new QToolButton(QPixmap(zoompan_xpm),
                               tr("Panzoom"), tr("Move the current window"),
                               this, SLOT(zoomPan()), toolBarZoom, "zoompan");
    bZoomPan->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    //bZoomTagged = new QToolButton(QPixmap(zoomtagged_xpm), tr("Zoom Tagged"), tr("Zoom to all tagged objects"),
    //                           this, SLOT(zoomTagged()), toolBarZoom, "zoomtagged");
    //bZoomTagged->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);
    bZoomOne = new QToolButton(QPixmap(zoomone_xpm),
                               tr("Zoom 1:x"),
                               tr("Zoom to scale (Only available in print preview)"),
                               this, SLOT(zoomOne()), toolBarZoom, "zoomone");
    bZoomOne->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);


  // Tool Bar Undo/Redo (top):
  //
  toolBarUndo = new QToolBar("Tool Bar Undo", this, Top, false, "toolbarundo");
  toolBarUndo->setGeometry(0, 0, 0, 0);

    bClipCut = new QToolButton(QPixmap(cut_xpm),
                               tr("Cut"), tr("Cut selected objects"),
                               this, SLOT(editClipCut()), toolBarUndo);
    bClipCut->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    bClipCopy = new QToolButton(QPixmap(copy_xpm),
                                tr("Copy"), tr("Copy selected elements"),
                                this, SLOT(editClipCopy()), toolBarUndo);
    bClipCopy->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    bClipPaste = new QToolButton(QPixmap(paste_xpm),
                                 tr("Paste"), tr("Paste objects"),
                                 this, SLOT(editClipPaste()), toolBarUndo);
    bClipPaste->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    toolBarUndo->addSeparator();

    bUndo = new QToolButton(QPixmap(undo_xpm),
                            tr("Undo"), tr("Undo last action"),
                            this, SLOT(undo()), toolBarUndo);
    bUndo->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    bRedo = new QToolButton(QPixmap(redo_xpm),
                            tr("Redo"), tr("Redo last undone action"),
                            this, SLOT(redo()), toolBarUndo);
    bRedo->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

  // Tool Bar Show (top):
  //
  toolBarShow = new QToolBar("Tool Bar Show", this, Top, false, "toolbarshow");
  toolBarShow->setGeometry(0, 0, 0, 0);

    bShowGrid = new QToolButton(toolBarShow, "grid");
    bShowGrid->setPixmap(QPixmap(simgrid_xpm));
    bShowGrid->setToggleButton(true);
    bShowGrid->setOn(true);
    QToolTip::add(bShowGrid, tr("Show/ hide grid"));
    connect(bShowGrid, SIGNAL(toggled(bool)), this, SLOT(showGrid(bool)));
    bShowGrid->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    bShowLayerList = new QToolButton(toolBarShow, "layerlist");
    bShowLayerList->setPixmap(QPixmap(layerlist_xpm));
    bShowLayerList->setToggleButton(true);
    bShowLayerList->setOn(false);
    QToolTip::add(bShowLayerList, tr("Show/ hide Layerlist"));
    connect(bShowLayerList, SIGNAL(toggled(bool)), this, SLOT(showLayerList(bool)));
    bShowLayerList->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

    bShowLibraryBrowser = new QToolButton(toolBarShow, "librarybrowser");
    bShowLibraryBrowser->setPixmap(QPixmap(librarybrowser_xpm));
    bShowLibraryBrowser->setToggleButton(true);
    bShowLibraryBrowser->setOn(false);
    QToolTip::add(bShowLibraryBrowser, tr("Show/ hide library browser"));
    connect(bShowLibraryBrowser, SIGNAL(toggled(bool)), this, SLOT(showLibraryBrowser(bool)));
    bShowLibraryBrowser->setFixedSize(DEF_BSIZE1, DEF_BSIZE1-1);

  toolBarAttrib = new QToolBar("Tool Bar Attrib", this, Top, false, "toolbarattrib");
  toolBarAttrib->setGeometry(0, 0, 0, 0);

    cbAttribColor = new RComboBox(false, toolBarAttrib, "color");
    cbAttribColor->fillInColors();
    QToolTip::add(cbAttribColor, tr("Color"));
    connect(cbAttribColor, SIGNAL(activated(int)), this, SLOT(chooseColor(int)));
    cbAttribColor->setFixedSize(DEF_BSIZE1*2, DEF_BSIZE1);

    cbAttribWidth = new RComboBox(false, toolBarAttrib, "width");
    cbAttribWidth->fillInWidths();
    QToolTip::add(cbAttribWidth, tr("Width"));
    connect(cbAttribWidth, SIGNAL(activated(int)), this, SLOT(chooseWidth(int)));
    cbAttribWidth->setFixedSize(DEF_BSIZE1*2, DEF_BSIZE1);

    cbAttribStyle = new RComboBox(false, toolBarAttrib, "style");
    cbAttribStyle->fillInStyles();
    QToolTip::add(cbAttribStyle, tr("Style"));
    connect(cbAttribStyle, SIGNAL(activated(int)), this, SLOT(chooseStyle(int)));
    cbAttribStyle->setFixedSize(DEF_BSIZE1*2, DEF_BSIZE1);
}


// Create toolbars for CAM Expert:
//
void
RAppWin::createToolBarsCamExpert()
{
    /*
    b = new QToolButton(fToolBar);
    b->setPixmap(QPixmap());
    _xpmQToolTip::add(b, "");
    connect(b, SIGNAL(clicked()), this, SLOT(()));
    b->setGeometry(DEF_BX1, DEF_BY1, DEF_BSIZE2, DEF_BSIZE2);
    */
    

}


// Hide left toolbars:
//
void
RAppWin::hideToolBarsLeft()
{
  toolBarMainShow(false);
  toolBarPointShow(false);
  toolBarLineShow(false);
  toolBarTag2Show(false);
  toolBarEditShow(false);
  toolBarArcShow(false);
  toolBarCircleShow(false);
  toolBarTextShow(false);
  toolBarDimShow(false);
  toolBarMeasureShow(false);
  toolBarSnapShow(false);
  toolBarCamShow(false);
}



// Hide top toolbars:
//
void
RAppWin::hideToolBarsTop()
{
  //toolBarFile->hide();
}



/*! Changes the menu to new index
    \param _num index of new menu
    \param _back Change menu back (dont save the last menu)
*/
void
RAppWin::changeMenu(int _num, bool _back)
{
  if(_num!=currentMenu) {
   
    // Hide left menu:
    //
    if(_num>=TOOLBAR_LEFT_FIRST && 
       _num<=TOOLBAR_LEFT_LAST     ) {
     
      hideToolBarsLeft();
    }
  
    switch(_num) {
      case TOOLBAR_MAIN:
        toolBarMainShow(true);
        ilastMenu=0;
        break;
        
      case TOOLBAR_POINT:
        toolBarPointShow(true);
        break;
  
      case TOOLBAR_LINE:
        toolBarLineShow(true);
        break;

      case TOOLBAR_ARC:
        toolBarArcShow(true);
        break;

      case TOOLBAR_CIRCLE:
        toolBarCircleShow(true);
        break;

      case TOOLBAR_DIM:
        toolBarDimShow(true);
        break;

      case TOOLBAR_MEASURE:
        toolBarMeasureShow(true);
        break;

      case TOOLBAR_TAG:
        toolBarTagShow(true);
        break;
        
      case TOOLBAR_TAG2:
        toolBarTag2Show(true);
        break;

        
      case TOOLBAR_EDIT:
        toolBarEditShow(true);
        break;

      case TOOLBAR_SNAP:
        toolBarSnapShow(true);
        if(isDocumentValid(curDocInd)) {
          // Don't remember keyboard positioning / snap with distance:
          if(currentDoc()->snapper()->getSnapMode()==SNAP_KEYXY ||
             currentDoc()->snapper()->getSnapMode()==SNAP_KEYAR ||
             currentDoc()->snapper()->getSnapMode()==SNAP_DIST) {
            currentDoc()->snapper()->setFreeMode();
          }
          untoggleSnapMenu(currentDoc()->snapper()->getSnapMode());
        }
        break;
  
      case TOOLBAR_CAM:
        toolBarCamShow(true);
        break;
  
      case TOOLBAR_FILE:
        hideToolBarsTop();
        toolBarFile->show();
        break;
  
      case TOOLBAR_ZOOM:
        toolBarZoom->show();
        break;
      
      case TOOLBAR_UNDO:
        toolBarUndo->show();
        break;

      case TOOLBAR_SHOW:
        toolBarShow->show();
        break;
  
      default:
        break;
    }
    
    // Set new current menu (only for left toolbar):
    //
    if(_num>=TOOLBAR_LEFT_FIRST && 
       _num<=TOOLBAR_LEFT_LAST     ) {
       
      if(!_back) {
        if(ilastMenu>=0 && ilastMenu<DEF_MENUBUF) {
          lastMenu[ilastMenu]=currentMenu;
        }
        if(ilastMenu<DEF_MENUBUF) ++ilastMenu;
      }
      
      currentMenu=_num;
    }
  }
}



/*! Changes the menu to the last one (back)
*/
void
RAppWin::backMenu()
{
  if(currentMenu!=TOOLBAR_MAIN) {
    if(inputPanel) {
      inputPanel->reset();
      inputPanel->hide();
    }
    if(inputPanel2) {
      inputPanel2->reset();
      inputPanel2->hide();
    }
    if(ilastMenu>0) --ilastMenu;
    changeMenu(lastMenu[ilastMenu], true);
  }
}


// Show/hide Main menu:
//
void
RAppWin::toolBarMainShow(bool _show)
{
  if(_show) {
    if(bSubmenuPoints ) bSubmenuPoints->show();
    if(bSubmenuLines  ) bSubmenuLines->show();
    if(bSubmenuArcs   ) bSubmenuArcs->show();
    if(bSubmenuCircles) bSubmenuCircles->show();
    if(bSubmenuTexts  ) bSubmenuTexts->show();
    if(bSubmenuDims   ) bSubmenuDims->show();
    if(bSubmenuMeasure) bSubmenuMeasure->show();
    if(bSubmenuHatch  ) bSubmenuHatch->show();
    if(bSubmenuTag    ) bSubmenuTag->show();
    if(bSubmenuEdit   ) bSubmenuEdit->show();
    if(bSubmenuCam    ) bSubmenuCam->show();
  }
  else {
    if(bSubmenuPoints ) bSubmenuPoints->hide();
    if(bSubmenuLines  ) bSubmenuLines->hide();
    if(bSubmenuArcs   ) bSubmenuArcs->hide();
    if(bSubmenuCircles) bSubmenuCircles->hide();
    if(bSubmenuTexts  ) bSubmenuTexts->hide();
    if(bSubmenuDims   ) bSubmenuDims->hide();
    if(bSubmenuMeasure) bSubmenuMeasure->hide();
    if(bSubmenuHatch  ) bSubmenuHatch->hide();
    if(bSubmenuTag    ) bSubmenuTag->hide();
    if(bSubmenuEdit   ) bSubmenuEdit->hide();
    if(bSubmenuCam    ) bSubmenuCam->hide();
  }

}



// Show/hide Point Menu:
//
void
RAppWin::toolBarPointShow(bool _show)
{
  if(_show) {
    if(bPointsSingle) bPointsSingle->show();
  }
  else {
    if(bPointsSingle) bPointsSingle->hide();
  }
}




// Show/hide Line Menu:
//
void
RAppWin::toolBarLineShow(bool _show)
{
  if(_show) {
    if(bLinesNormal) bLinesNormal->show();
    if(bLinesHor) bLinesHor->show();
    if(bLinesVer) bLinesVer->show();
    if(bLinesAngle) bLinesAngle->show();
    if(bLinesRect) bLinesRect->show();
    if(bLinesPara) bLinesPara->show();
    if(bLinesBisector) bLinesBisector->show();
    if(bLinesTan1) bLinesTan1->show();
    if(bLinesTan2) bLinesTan2->show();
    if(bLinesOrtho) bLinesOrtho->show();
    if(bLinesOrtho2) bLinesOrtho2->show();
  }
  else {
    if(bLinesNormal) bLinesNormal->hide();
    if(bLinesHor) bLinesHor->hide();
    if(bLinesVer) bLinesVer->hide();
    if(bLinesAngle) bLinesAngle->hide();
    if(bLinesRect) bLinesRect->hide();
    if(bLinesPara) bLinesPara->hide();
    if(bLinesBisector) bLinesBisector->hide();
    if(bLinesTan1) bLinesTan1->hide();
    if(bLinesTan2) bLinesTan2->hide();
    if(bLinesOrtho) bLinesOrtho->hide();
    if(bLinesOrtho2) bLinesOrtho2->hide();
  }
}



// Show/hide Arc Menu:
//
void
RAppWin::toolBarArcShow(bool _show)
{
  if(_show) {
    if(bArcsCRAA) bArcsCRAA->show();
    if(bArcsPPP)  bArcsPPP->show();
    if(bArcsPara) bArcsPara->show();
    
  }
  else {
    if(bArcsCRAA) bArcsCRAA->hide();
    if(bArcsPPP)  bArcsPPP->hide();
    if(bArcsPara) bArcsPara->hide();
  }
}



// Show/hide Circle Menu:
//
void
RAppWin::toolBarCircleShow(bool _show)
{
  if(_show) {
    if(bCirclesCR) bCirclesCR->show();
    if(bCirclesCL) bCirclesCL->show();
    if(bCircles3P) bCircles3P->show();
  }
  else {
    if(bCirclesCR) bCirclesCR->hide();
    if(bCirclesCL) bCirclesCL->hide();
    if(bCircles3P) bCircles3P->hide();
  }
}



// Show/hide Text Menu:
//
void
RAppWin::toolBarTextShow(bool /*_show*/)
{
}


// Show/hide Dimension Menu:
//
void
RAppWin::toolBarDimShow(bool _show)
{
  if(_show) {
    if(bDimsPara) bDimsPara->show();
    if(bDimsHor)  bDimsHor->show();
    if(bDimsVer)  bDimsVer->show();
    if(bDimsAngle) bDimsAngle->show();
    if(bDimsDiam)  bDimsDiam->show();
    if(bDimsRad)  bDimsRad->show();
    if(bDimsArrow)  bDimsArrow->show();
  }
  else {
    if(bDimsPara) bDimsPara->hide();
    if(bDimsHor)  bDimsHor->hide();
    if(bDimsVer)  bDimsVer->hide();
    if(bDimsAngle) bDimsAngle->hide();
    if(bDimsDiam)  bDimsDiam->hide();
    if(bDimsRad)  bDimsRad->hide();
    if(bDimsArrow)  bDimsArrow->hide();
  }
}


// Show/hide Measure Menu:
//
void
RAppWin::toolBarMeasureShow(bool _show)
{
  if(_show) {
    if(bMeasurePtPt) bMeasurePtPt->show();
    if(bMeasureLnPt) bMeasureLnPt->show();
    if(bMeasureParallelLines) bMeasureParallelLines->show();
    if(bMeasureAngleLnLn) bMeasureAngleLnLn->show();
    if(bMeasureEditElement) bMeasureEditElement->show();
		if(bMeasureArea) bMeasureArea->show();
  }
  else {
    if(bMeasurePtPt) bMeasurePtPt->hide();
    if(bMeasureLnPt) bMeasureLnPt->hide();
    if(bMeasureParallelLines) bMeasureParallelLines->hide();
    if(bMeasureAngleLnLn) bMeasureAngleLnLn->hide();
    if(bMeasureEditElement) bMeasureEditElement->hide();
		if(bMeasureArea) bMeasureArea->hide();
  }
}


// Show/hide Tag Menu:
//
void
RAppWin::toolBarTagShow(bool _show)
{
  if(_show) {
    if(bTagNothing)  bTagNothing->show();
    if(bTagAll)      bTagAll->show();
    if(bTagElement)  bTagElement->show();
    if(bTagContour)  bTagContour->show();
    if(bUntagRange)  bUntagRange->show();
    if(bTagRange)    bTagRange->show();
    if(bUntagInters) bUntagInters->show();
    if(bTagInters)   bTagInters->show();
    if(bTagInvert)   bTagInvert->show();
    if(bTagLayer)    bTagLayer->show();
    if(bTagDoubles)  bTagDoubles->show();
  }
  else {
    if(bTagNothing)  bTagNothing->hide();
    if(bTagAll)      bTagAll->hide();
    if(bTagElement)  bTagElement->hide();
    if(bTagContour)  bTagContour->hide();
    if(bUntagRange)  bUntagRange->hide();
    if(bTagRange)    bTagRange->hide();
    if(bUntagInters) bUntagInters->hide();
    if(bTagInters)   bTagInters->hide();
    if(bTagInvert)   bTagInvert->hide();
    if(bTagLayer)    bTagLayer->hide();
    if(bTagDoubles)  bTagDoubles->hide();
  }
}



// Show/hide extended Tag Menu:
//
void
RAppWin::toolBarTag2Show(bool _show)
{
  toolBarTagShow(_show);
  
  if(_show) {
    if(bTagEnough)  {
      bTagEnough->setPixmap( getPixmapForEditFunction( currentDoc()->getNextAction() ) );
      bTagEnough->show();
    }
  }
  else {
    if(bTagEnough)  bTagEnough->hide();
  }
}



// Show/hide Edit Menu:
//
void
RAppWin::toolBarEditShow(bool _show)
{
  if(_show) {
    if(bEditMove)         bEditMove->show();
    if(bEditRotate)       bEditRotate->show();
    if(bEditScale)        bEditScale->show();
    if(bEditMirror)       bEditMirror->show();
    if(bEditMoveRotate)   bEditMoveRotate->show();
    if(bEditRotateRotate) bEditRotateRotate->show();
    if(bEditTrim)         bEditTrim->show();
    if(bEditTrim2)        bEditTrim2->show();
    if(bEditTrim3)        bEditTrim3->show();
    if(bEditCut)          bEditCut->show();
    if(bEditCut2)         bEditCut2->show();
    if(bEditStretch)      bEditStretch->show();
    if(bEditRound)        bEditRound->show();
    if(bEditBevel)        bEditBevel->show();
    if(bEditTextElements) bEditTextElements->show();
    if(bEditText)         bEditText->show();
    if(bEditDelete)       bEditDelete->show();
    if(bEditLayer)        bEditLayer->show();
    if(bEditZero)         bEditZero->show();
    if(bEditAttrib)       bEditAttrib->show();
    if(bEditClip)         bEditClip->show();
  }
  else {
    if(bEditMove)         bEditMove->hide();
    if(bEditRotate)       bEditRotate->hide();
    if(bEditScale)        bEditScale->hide();
    if(bEditMirror)       bEditMirror->hide();
    if(bEditMoveRotate)   bEditMoveRotate->hide();
    if(bEditRotateRotate) bEditRotateRotate->hide();
    if(bEditTrim)         bEditTrim->hide();
    if(bEditTrim2)        bEditTrim2->hide();
    if(bEditTrim3)        bEditTrim3->hide();
    if(bEditCut)          bEditCut->hide();
    if(bEditCut2)         bEditCut2->hide();
    if(bEditStretch)      bEditStretch->hide();
    if(bEditRound)        bEditRound->hide();
    if(bEditBevel)        bEditBevel->hide();
    if(bEditTextElements) bEditTextElements->hide();
    if(bEditText)         bEditText->hide();
    if(bEditDelete)       bEditDelete->hide();
    if(bEditLayer)        bEditLayer->hide();
    if(bEditZero)         bEditZero->hide();
    if(bEditAttrib)       bEditAttrib->hide();
    if(bEditClip)         bEditClip->hide();
  }
}


// Show/hide Snap Menu:
//
void
RAppWin::toolBarSnapShow(bool _show)
{
  if(_show) {
    if(bSnapFree)      bSnapFree->show();
    if(bSnapGrid)      bSnapGrid->show();
    if(bSnapEndpoints) bSnapEndpoints->show();
    if(bSnapCenters)   bSnapCenters->show();
    if(bSnapMiddles)   bSnapMiddles->show();
    if(bSnapAutoInter) bSnapAutoInter->show();
    if(bSnapManInter)  bSnapManInter->show();
    if(bSnapDist)      bSnapDist->show();
    if(bSnapOnElement) bSnapOnElement->show();
    if(bSnapKeyXY)     bSnapKeyXY->show();
    if(bSnapKeyAR)     bSnapKeyAR->show();
  }
  else {
    if(bSnapFree)      bSnapFree->hide();
    if(bSnapGrid)      bSnapGrid->hide();
    if(bSnapEndpoints) bSnapEndpoints->hide();
    if(bSnapCenters)   bSnapCenters->hide();
    if(bSnapMiddles)   bSnapMiddles->hide();
    if(bSnapAutoInter) bSnapAutoInter->hide();
    if(bSnapManInter)  bSnapManInter->hide();
    if(bSnapDist)      bSnapDist->hide();
    if(bSnapOnElement) bSnapOnElement->hide();
    if(bSnapKeyXY)     bSnapKeyXY->hide();
    if(bSnapKeyAR)     bSnapKeyAR->hide();
  }
}



// Show/hide CAM Menu:
//
void
RAppWin::toolBarCamShow(bool _show)
{
  if(_show) {
    if(bCamAuto   ) bCamAuto->show();
    if(bCamManual ) bCamManual->show();
  }
  else {
    if(bCamAuto   ) bCamAuto->hide();
    if(bCamManual ) bCamManual->hide();
  }
}








//
//   Menu related functions:
//

// Create Menus:
//
void
RAppWin::createMenus()
{
  menuBar()->setFont( RCONFIG->getFont(0) );

  // Menu "File":
  //
  fileMenu = new QPopupMenu();
  fileMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&File"), fileMenu);

    fileMenu->insertItem(QPixmap(filenew_xpm),   tr("New"), this, SLOT(fileNew()),      CTRL+Key_N);
    fileMenu->insertItem(QPixmap(fileopen_xpm),  tr("Open"), this, SLOT(fileOpen()),     CTRL+Key_O);
    fileMenu->insertItem(                        tr("Add"), this, SLOT(fileAdd()));
    fileMenu->insertItem(QPixmap(filesave_xpm),  tr("Save"), this, SLOT(fileSave()),     CTRL+Key_S);
    fileMenu->insertItem(                        tr("Save As"), this, SLOT(fileSaveAs()));
    fileMenu->insertSeparator();
    fileMenu->insertItem(QPixmap(fileprintsetup_xpm), tr("Setup printer..."), this, SLOT(filePrintSetup()));
    fileMenu->insertItem(QPixmap(fileprintprev_xpm), tr("Print Preview"), this, SLOT(filePrintPreview()));
    fileMenu->insertItem(QPixmap(fileprint_xpm), tr("Print..."), this, SLOT(filePrint()),    CTRL+Key_P);
    fileMenu->insertSeparator();
    fileMenu->insertItem(QPixmap(fileclose_xpm), tr("Close input file"), this, SLOT(fileClose()),    CTRL+Key_W);
#ifdef DEF_CAM_EXPERT
    fileMenu->insertItem(QPixmap(filecloseoutput_xpm), tr("Close output file"), this, SLOT(fileCloseOutput()));
#endif
    fileMenu->insertItem(                        tr("Quit"), this, SLOT(fileQuit()),     CTRL+Key_Q);
    fileMenu->insertSeparator();

    QString settingName;
    for(int c=DEF_RECENTFILES-1; c>=0; --c) {
      settingName.sprintf( "RecentFiles:File%d", c );
      addRecentFile( RCONFIG->getSetting(settingName), false );
    }

    updateRecentFileMenu();

    // Connect menubar entries with function fileOpenRecent
    //   (allows to reload a recent file in menu "File"):
    //
    connect( fileMenu, SIGNAL(activated(int)),
             this, SLOT(fileOpenRecent(int)) );


  // Menu "Edit":
  //
  editMenu = new QPopupMenu();
  editMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&Edit"), editMenu);

    editMenu->insertItem( QPixmap(undo_xpm),   tr("Undo"), this, SLOT(undo()) );
    editMenu->insertItem( QPixmap(redo_xpm),   tr("Redo"), this, SLOT(redo()) );

    editMenu->insertSeparator();

    editMenu->insertItem(QPixmap(copy_xpm),  tr("Copy"), this, SLOT(editClipCopy()),  CTRL+Key_C);
    editMenu->insertItem(QPixmap(cut_xpm),   tr("Cut"), this, SLOT(editClipCut()),   CTRL+Key_X);
    editMenu->insertItem(QPixmap(paste_xpm), tr("Paste"), this, SLOT(editClipPaste()), CTRL+Key_V);

    
  // Menu "Zoom":
  //
  zoomMenu = new QPopupMenu();
  zoomMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&Zoom"), zoomMenu);

    zoomMenu->insertItem(QPixmap(zoomredraw_xpm),   tr("Redraw"), this, SLOT(zoomRedraw()) );
    zoomMenu->insertItem(QPixmap(zoomauto_xpm),     tr("Autozoom"), this, SLOT(zoomAuto()) );
    zoomMenu->insertItem(QPixmap(zoomtopage_xpm),   tr("Zoom to page"), this, SLOT(zoomToPage()) );
    zoomMenu->insertItem(QPixmap(zoomincrease_xpm), tr("Increase"), this, SLOT(zoomIncrease()) );

    zoomMenu->insertItem(QPixmap(zoomdecrease_xpm), tr("Decrease"), this, SLOT(zoomDecrease()) );
    zoomMenu->insertItem(QPixmap(zoomwindow_xpm),   tr("Window"), this, SLOT(zoomWindow()) );
    zoomMenu->insertItem(QPixmap(zoompan_xpm),      tr("Pan zoom"), this, SLOT(zoomPan()) );
    zoomMenu->insertItem(QPixmap(zoomtagged_xpm),   tr("Tagged"), this, SLOT(zoomTagged()) );
    zoomMenu->insertItem(QPixmap(zoomone_xpm),      tr("Zoom 1:x"), this, SLOT(zoomOne()) );

  // Menu "View":
  //
  viewMenu = new QPopupMenu();
  viewMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&View"), viewMenu);

    viewMenu->insertItem( QPixmap(layerlist_xpm),      tr("Show/ hide Layerlist"), this, SLOT(viewLayerList()) );
    viewMenu->insertItem( QPixmap(librarybrowser_xpm), tr("Show/ hide library browser"), this, SLOT(viewLibraryBrowser()) );
    viewMenu->insertItem( QPixmap(simgrid_xpm),        tr("Show/ hide grid"), this, SLOT(viewGrid()) );

  // Menu "Snap":
  //
  /*snapMenu = new QPopupMenu();
  //snapMenu->setFont(QFont(RCONFIG->getSetting( "Application:FontName" ), RIni::getRIni()->getFontSize0()));
  snapMenu->setCheckable(true);
  menuBar()->insertItem(tr("Snap"), snapMenu);

    snapMenu->insertItem(QPixmap(snapfree_xpm),      tr("Snap to nothing [F]"), this, SLOT(snapFree()),      Key_F);
    snapMenu->insertItem(QPixmap(snapgrid_xpm),      tr("Snap to grid points [G]"), this, SLOT(snapGrid()),      Key_G);
    snapMenu->insertItem(QPixmap(snapendpoints_xpm), tr("Snap to endpoints [E]"), this, SLOT(snapEndpoints()), Key_E);
    snapMenu->insertItem(QPixmap(snapcenters_xpm),   tr("Snap to centers [C]"), this, SLOT(snapCenters()),   Key_C);
    snapMenu->insertItem(QPixmap(snapmiddles_xpm),   tr("Snap to middles [M]"), this, SLOT(snapMiddles()),   Key_M);
    snapMenu->insertItem(QPixmap(snapautointer_xpm), tr("Snap automatically to intersections [X]"), this, SLOT(snapAutoInter()), Key_X);
    snapMenu->insertItem(QPixmap(snapmaninter_xpm),  tr("Snap manually to intersections"), this, SLOT(snapManInter()));
    snapMenu->insertItem(QPixmap(snapdist_xpm),      tr("Snap to points with a defined\ndistance from endpoint"), this, SLOT(snapDist()),      Key_D);
    snapMenu->insertItem(QPixmap(snaponelement_xpm), tr("Snap to points on elements [L]"), this, SLOT(snapOnElement()), Key_L);*/


#ifdef DEF_CAM_EXPERT
  // Menu "CAM":
  //
  camMenu = new QPopupMenu();
  camMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&CAM"), camMenu);

    camMenu->insertItem(QPixmap(camauto_xpm),   tr("Automatic Creation"),   this, SLOT(camAuto())    );
    //camMenu->insertItem(QPixmap(cammanual_xpm), tr("Manual"), this, SLOT(camManual())  );
    camMenu->insertSeparator();
    camMenu->insertItem(QPixmap(modeconstruction_xpm), tr("Construction"), this, SLOT(setConstructionMode()), CTRL+Key_1);
    camMenu->insertItem(QPixmap(modecomparison_xpm),   tr("Comparison"), this, SLOT(setComparisonMode()), CTRL+Key_2);
    camMenu->insertItem(QPixmap(modesimulation_xpm),   tr("Simulation"), this, SLOT(setSimulationMode()), CTRL+Key_3);
    camMenu->insertSeparator();
    camMenu->insertItem(tr("Show/ hide simulation panel"), this, SLOT(toggleSimulationPanel()));
    camMenu->insertSeparator();
    camMenu->insertItem(QPixmap(confmachgen_xpm), tr("Configure"),   this, SLOT(configMachineGenerator()));
#endif

  // Menu "Options":
  //
  optionsMenu = new QPopupMenu();
  optionsMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&Options"), optionsMenu);

    optionsMenu->insertItem(tr("&General Preferences"), this, SLOT(optionsPreferences()));
    optionsMenu->insertItem(tr("Current drawing settings"), this, SLOT(optionsDrawing()));

  // Menu "Windows":
  //  
  windowMenu = new QPopupMenu();
  windowMenu->setFont( RCONFIG->getFont(0) );
    menuBar()->insertItem(tr("&Documents"), windowMenu);

#ifdef DEF_RDEBUGMENU
  // Menu "Debug":
  //
  debugMenu = new QPopupMenu();
  debugMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem("Debug", debugMenu);
    debugMenu->insertItem("Contourcheck", this, SLOT(debugContourCheck()));
    debugMenu->insertItem("Convert To Font", this, SLOT(editConvertToFont()));
#endif

  // Menu "Help":
  //
  helpMenu = new QPopupMenu();
  helpMenu->setFont( RCONFIG->getFont(0) );
  menuBar()->insertItem(tr("&Help"), helpMenu);

    helpMenu->insertItem(QPixmap(helpabout_xpm), tr("&About..."), this, SLOT(helpAbout()));
    helpMenu->insertItem(QPixmap(helpmanual_xpm), tr("Manual"), this, SLOT(helpManual()), Key_F1);



}



/*! Enables all buttons in all menus
    \param _enable true: enable / false: disable
*/
void 
RAppWin::enableAll(bool _enable)
{
  enableMenus(_enable);
  enableToolBars(_enable);
}



/*! Enables all menus
    \param _enable true: enable / false: disable

*/
void 
RAppWin::enableMenus(bool _enable)
{
  if(fileMenu)   fileMenu->setEnabled(_enable);
  if(zoomMenu)   zoomMenu->setEnabled(_enable);
  if(camMenu)    camMenu->setEnabled(_enable);
  if(windowMenu) windowMenu->setEnabled(_enable);
  if(helpMenu)   helpMenu->setEnabled(_enable);
}



/*! Enables tool bars (deprecated)
    \param _enable true: enable / false: disable
*/
void 
RAppWin::enableToolBars(bool /*_enable*/ )
{
  /*
  if(bSubmenuPoints) bSubmenuPoints->setEnabled(_enable);
  if(bSubmenuLines) bSubmenuLines->setEnabled(_enable);
  if(bSubmenuArcs) bSubmenuArcs->setEnabled(_enable);
  if(bSubmenuCircles) bSubmenuCircles->setEnabled(_enable);
  if(bSubmenuTexts) bSubmenuTexts->setEnabled(_enable);
  if(bSubmenuTag) bSubmenuTag->setEnabled(_enable);
  if(bSubmenuEdit) bSubmenuEdit->setEnabled(_enable);
  if(bSubmenuCam) bSubmenuCam->setEnabled(_enable);
  
  if(bPointsSingle) bPointsSingle->setEnabled(_enable);

  if(bLinesNormal) bLinesNormal->setEnabled(_enable);
  if(bLinesRect) bLinesRect->setEnabled(_enable);
  if(bLinesPara) bLinesPara->setEnabled(_enable);
  if(bLinesBisector) bLinesBisector->setEnabled(_enable);
  if(bLinesTan1) bLinesTan1->setEnabled(_enable);
  if(bLinesTan2) bLinesTan2->setEnabled(_enable);

  if(bArcsCRAA) bArcsCRAA->setEnabled(_enable);
  if(bArcsPPP) bArcsPPP->setEnabled(_enable);
  if(bArcsPara) bArcsPara->setEnabled(_enable);

  if(bCirclesCR) bCirclesCR->setEnabled(_enable);
  
  if(bEditMove)         bEditMove->setEnabled(_enable);
  if(bEditRotate)       bEditRotate->setEnabled(_enable);
  if(bEditScale)        bEditScale->setEnabled(_enable);
  if(bEditMirror)       bEditMirror->setEnabled(_enable);
  if(bEditMoveRotate)   bEditMoveRotate->setEnabled(_enable);
  if(bEditRotateRotate) bEditRotateRotate->setEnabled(_enable);
  if(bEditTrim)         bEditTrim->setEnabled(_enable);
  if(bEditTrim2)        bEditTrim2->setEnabled(_enable);
  if(bEditTrim3)        bEditTrim3->setEnabled(_enable);
  if(bEditCut)          bEditCut->setEnabled(_enable);
  if(bEditCut2)         bEditCut2->setEnabled(_enable);
  if(bEditStretch)      bEditStretch->setEnabled(_enable);
  if(bEditRound)        bEditRound->setEnabled(_enable);
  if(bEditBevel)        bEditBevel->setEnabled(_enable);
  if(bEditTextElements) bEditTextElements->setEnabled(_enable);
  if(bEditText)         bEditText->setEnabled(_enable);
  if(bEditDelete)       bEditDelete->setEnabled(_enable);
  if(bEditLayer)        bEditLayer->setEnabled(_enable);
  if(bEditZero)         bEditZero->setEnabled(_enable);
  
  if(bTagNothing) bTagNothing->setEnabled(_enable);
  if(bTagAll) bTagAll->setEnabled(_enable);
  if(bTagElement) bTagElement->setEnabled(_enable);
  if(bTagContour) bTagContour->setEnabled(_enable);
  if(bUntagRange) bUntagRange->setEnabled(_enable);
  if(bTagRange) bTagRange->setEnabled(_enable);
  if(bUntagInters) bUntagInters->setEnabled(_enable);
  if(bTagInters) bTagInters->setEnabled(_enable);
  if(bTagInvert) bTagInvert->setEnabled(_enable);
  if(bTagLayer) bTagLayer->setEnabled(_enable);
  if(bTagDoubles) bTagDoubles->setEnabled(_enable);

  if(bSnapFree) bSnapFree->setEnabled(_enable);
  if(bSnapGrid) bSnapGrid->setEnabled(_enable);
  if(bSnapEndpoints) bSnapEndpoints->setEnabled(_enable);
  if(bSnapCenters) bSnapCenters->setEnabled(_enable);
  if(bSnapMiddles) bSnapMiddles->setEnabled(_enable);
  if(bSnapAutoInter) bSnapAutoInter->setEnabled(_enable);
  if(bSnapManInter) bSnapManInter->setEnabled(_enable);
  if(bSnapDist) bSnapDist->setEnabled(_enable);
  if(bSnapOnElement) bSnapOnElement->setEnabled(_enable);
  
  
  if(bCamAuto) bCamAuto->setEnabled(_enable);
  if(bCamManual) bCamManual->setEnabled(_enable);
  
  if(bFileNew) bFileNew->setEnabled(_enable);
  if(bFileOpen) bFileOpen->setEnabled(_enable);
  if(bFileClose) bFileClose->setEnabled(_enable);
  if(bFileSave) bFileSave->setEnabled(_enable);
  //toolBarFile->setEnabled(_enable);
  
  if(bFilePrintPreview) bFilePrintPreview->setEnabled(_enable);
  if(bFilePrint) bFilePrint->setEnabled(_enable);
  
  if(bZoomRedraw) bZoomRedraw->setEnabled(_enable);
  if(bZoomAuto) bZoomAuto->setEnabled(_enable);
  if(bZoomIncrease) bZoomIncrease->setEnabled(_enable);
  if(bZoomDecrease) bZoomDecrease->setEnabled(_enable);
  if(bZoomWindow) bZoomWindow->setEnabled(_enable);
  if(bZoomPan) bZoomPan->setEnabled(_enable);
  if(bZoomTagged) bZoomTagged->setEnabled(_enable);
  
  if(bUndo) bUndo->setEnabled(_enable);
  if(bRedo) bRedo->setEnabled(_enable);
  */
}



// EOF




















































































































