/***************************************************************************
                          rfiledxf.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rfiledxf.cpp 1998/08/28 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include "rfiledxf.h"

#include <stdlib.h>
#include <string.h>

#include <qstring.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qregexp.h>

#include "rappwin.h"
#include "rfilebase.h"
#include "rfonts.h"
#include "rgraphic.h"
#include "rconfig.h"
#include "rlog.h"
#include "rmath.h"
#include "rprgdef.h"
#include "rstring.h"
#include "rstatuspanel.h"

// Constructor:
//
RFileDxf::RFileDxf(const QString& _name, RGraphic* _graphic)
  :RFileBase(_name)
{
  graphic=_graphic;
}



// Destructor:
//
RFileDxf::~RFileDxf()
{
  
}



// load a dxf file (read in buffer / count / read from buffer):
//
bool
RFileDxf::load(bool add)
{
  if(readFileInBuffer()) {
    separateBuf();
    return readFromBuffer(add);
  }
  else {
    return false;
  }
}



// read a dxf file from buffer:
//
bool
RFileDxf::readFromBuffer(bool add)
{
  RLOG( "\nDXF: Read from buffer" );

  bool      ret;                    // returned value
  QString   dxfLine;                // A line in the dxf file
  QString   dxfCode;                // A Code in the dxf file as string
  int       code=-1;                // Dxf-code as number
  double    vx1=0.0, vy1=0.0,       // Start point
            vx2=0.0, vy2=0.0,       // End point
            vcx=0.0, vcy=0.0,       // Centre
            vcr=0.0,                // Radius
            va1=0.0, va2=0.0,       // Start / End Angle
            vab=0.0,                // Bulge
            vpx=0.0, vpy=0.0;       // First Polyline point
  double    ax=0.0, ay=0.0;         // Current coordinate
  bool      plClose=false;          // Polyline closed-flag
  QString   lastLayer;              // Last used layer name (test adding only
                                    //   if the new layer!=lastLayer)
  int       currentLayerNum=0;      // Current layer number
  RLayer*   currentLayer=0;         // Pointer to current layer
  QList<RGraphic> blockList;        // List of blocks
  blockList.setAutoDelete( true );
  bool      oldColorNumbers=false;  // use old color numbers (qcad<1.5.3)

  if(!add) graphic->clearLayers();

  graphic->addLayer(DEF_DEFAULTLAYER);

  RLOG( "\nDefault layer added" );

  // Loaded graphics without unit information: load as unit less:
  graphic->setUnit( None );

  RLOG( "\nUnit set" );

  resetBufP();

  if(!graphic->getFlag(G_PREVIEW)) statusPanel()->iniProgress(fSize, QObject::tr("Loading File..."));

  if(fBuf) {

    RLOG( "\nBuffer OK" );
    RLOG( "\nBuffer: " );
    RLOG( fBuf );
  
    do {
      dxfLine=getBufLine();

      RLOG( "\ndxfLine: " );
      RLOG( dxfLine );
      
      if(!graphic->getFlag(G_PREVIEW)) statusPanel()->setProgress(fBufP, 1000);

      // $-Setting in the header of DXF found
      //
      if( dxfLine &&
          dxfLine[0]=='$' ) {

        // Units:
        //
        if( dxfLine=="$INSUNITS" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==70 ) {
              if( dxfLine=getBufLine() ) {
                switch( dxfLine.toInt() ) {
                  case  0: graphic->setUnit( None );       break;
                  case  1: graphic->setUnit( Inch );       break;
                  case  2: graphic->setUnit( Foot );       break;
                  case  3: graphic->setUnit( Mile );       break;
                  case  4: graphic->setUnit( Millimeter ); break;
                  case  5: graphic->setUnit( Centimeter ); break;
                  case  6: graphic->setUnit( Meter );      break;
                  case  7: graphic->setUnit( Kilometer );  break;
                  case  8: graphic->setUnit( Microinch );  break;
                  case  9: graphic->setUnit( Mil );        break;
                  case 10: graphic->setUnit( Yard );       break;
                  case 11: graphic->setUnit( Angstrom );   break;
                  case 12: graphic->setUnit( Nanometer );  break;
                  case 13: graphic->setUnit( Micron );     break;
                  case 14: graphic->setUnit( Decimeter );  break;
                  case 15: graphic->setUnit( Decameter );  break;
                  case 16: graphic->setUnit( Hectometer ); break;
                  case 17: graphic->setUnit( Gigameter );  break;
                  case 18: graphic->setUnit( Astro );      break;
                  case 19: graphic->setUnit( Lightyear );  break;
                  case 20: graphic->setUnit( Parsec );     break;
                }

                graphic->setDimensionUnit( graphic->getUnit() );
                //graphic->setGridUnit( graphic->getUnit() );
              }
            }
          }
        }

        // Dimenison Units:
        //
        else if( dxfLine=="$DIMALT" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==70 ) {
              if( dxfLine=getBufLine() ) {
                switch( dxfLine.toInt() ) {
                  case  0: graphic->setDimensionUnit( None );       break;
                  case  1: graphic->setDimensionUnit( Inch );       break;
                  case  2: graphic->setDimensionUnit( Foot );       break;
                  case  3: graphic->setDimensionUnit( Mile );       break;
                  case  4: graphic->setDimensionUnit( Millimeter ); break;
                  case  5: graphic->setDimensionUnit( Centimeter ); break;
                  case  6: graphic->setDimensionUnit( Meter );      break;
                  case  7: graphic->setDimensionUnit( Kilometer );  break;
                  case  8: graphic->setDimensionUnit( Microinch );  break;
                  case  9: graphic->setDimensionUnit( Mil );        break;
                  case 10: graphic->setDimensionUnit( Yard );       break;
                  case 11: graphic->setDimensionUnit( Angstrom );   break;
                  case 12: graphic->setDimensionUnit( Nanometer );  break;
                  case 13: graphic->setDimensionUnit( Micron );     break;
                  case 14: graphic->setDimensionUnit( Decimeter );  break;
                  case 15: graphic->setDimensionUnit( Decameter );  break;
                  case 16: graphic->setDimensionUnit( Hectometer ); break;
                  case 17: graphic->setDimensionUnit( Gigameter );  break;
                  case 18: graphic->setDimensionUnit( Astro );      break;
                  case 19: graphic->setDimensionUnit( Lightyear );  break;
                  case 20: graphic->setDimensionUnit( Parsec );     break;
                }

              }
            }
          }
        }

        // Dimension Format:
        //
        else if( dxfLine=="$DIMLUNIT" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==70 ) {
              if( dxfLine=getBufLine() ) {
                switch( dxfLine.toInt() ) {
                  case 1: graphic->setDimensionFormat( Scientific ); break;
                  case 2:
                  case 3: graphic->setDimensionFormat( Decimal ); break;
                  case 4:
                  case 5: graphic->setDimensionFormat( Fractional ); break;
                  default: break;
                }
              }
            }
          }
        }

        // Dimension Arrow Size:
        //
        else if( dxfLine=="$DIMASZ" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setDimensionArrowSize( dxfLine.toDouble() );
              }
            }
          }
        }

        // Dimension Scale:
        //
        /*
        else if( dxfLine=="$DIMSCALE" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setDimensionScale( dxfLine.toDouble() );
              }
            }
          }
        }
        */

        // Dimension Text Height:
        //
        else if( dxfLine=="$DIMTXT" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setDimensionTextHeight( dxfLine.toDouble() );
              }
            }
          }
        }

        // Dimension exactness:
        //
        else if( dxfLine=="$DIMRND" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
	        if( dxfLine.toDouble()>0.000001 ) {
                  graphic->setDimensionExactness( dxfLine.toDouble() );
		}
              }
            }
          }
        }

        // Dimension over length:
        //
        else if( dxfLine=="$DIMEXE" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setDimensionOverLength( dxfLine.toDouble() );
              }
            }
          }
        }

        // Dimension under length:
        //
        else if( dxfLine=="$DIMEXO" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setDimensionUnderLength( dxfLine.toDouble() );
              }
            }
          }
        }


        // Angle dimension format:
        //
        else if( dxfLine=="$DIMAUNIT" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==70 ) {
              if( dxfLine=getBufLine() ) {
                switch( dxfLine.toInt() ) {
                  case 0: graphic->setAngleDimensionFormat( DecimalDegrees ); break;
                  case 1: graphic->setAngleDimensionFormat( DegreesMinutesSeconds ); break;
                  case 2: graphic->setAngleDimensionFormat( Gradians ); break;
                  case 3: graphic->setAngleDimensionFormat( Radians ); break;
                  case 4: graphic->setAngleDimensionFormat( Surveyor ); break;
                  default: break;
                }
              }
            }
          }
        }

        // Angle dimension exactness:
        //
        else if( dxfLine=="$DIMADEC" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==70 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setAngleDimensionExactness( pow(0.1, dxfLine.toInt()) );
              }
            }
          }
        }

        // Grid x/y:
        //
        else if( dxfLine=="$GRIDUNIT" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==10 ) {
              if( dxfLine=getBufLine() ) {
                double gx=dxfLine.toDouble();
                if (gx<0.0001) gx=0.0001;
                graphic->setMinGridX(gx);
                graphic->setGridFormat( Fractional );

                for( double q=0.00000001; q<=100000.0; q*=10.0 ) {
                  if( mtCompFloat(gx, q, q/1000.0) ) {
                    graphic->setGridFormat( Decimal );
                    break;
                  }
                }
              }
            }
          }
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==20 ) {
              if( dxfLine=getBufLine() ) {
                double gy=dxfLine.toDouble();
                if (gy<0.0001) gy=0.0001;
                graphic->setMinGridY(gy);
              }
            }
          }
        }

        // Page limits min x/y:
        //
        else if( dxfLine=="$PLIMMIN" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==10 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setPageOriginX( dxfLine.toDouble() );
              }
            }
          }
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==20 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setPageOriginY( dxfLine.toDouble() );
              }
            }
          }
        }

        // Page limits min x/y:
        //
        else if( dxfLine=="$PLIMMAX" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==10 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setPageSizeX( dxfLine.toDouble() - graphic->getPageOriginX() );
              }
            }
          }
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==20 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setPageSizeY( dxfLine.toDouble() - graphic->getPageOriginY() );
              }
            }
          }
        }

        // Paper space scale:
        //
        else if( dxfLine=="$PSVPSCALE" ) {
          if(dxfCode=getBufLine()) {
            if( dxfCode.toInt()==40 ) {
              if( dxfLine=getBufLine() ) {
                graphic->setPaperSpace( dxfLine.toDouble() );
              }
            }
          }
        }

      }

      // Entity
      //
      else if(dxfLine &&
              dxfLine[0]>='A' && dxfLine[0]<='Z') {
         	
        if(dxfLine=="EOF") {
        	// End of file reached
        	//
        }
      
        // ------
        // Layer:
        // ------
        else if(dxfLine=="LAYER") {
          currentLayer=0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  2:  // Layer name
                    currentLayerNum = graphic->addLayer(dxfLine);
                    currentLayer = graphic->getLayer( graphic->getLayerIndex(currentLayerNum) );
                    lastLayer=dxfLine;
                    break;
                  case 70:  // Visibility
                    /*
                    if(dxfLine.toInt()&5) {
                      if(currentLayerNum>=0 && currentLayerNum<DEF_MAXLAYERS) {
                        graphic->layer[currentLayerNum].DelFlag(Y_VISIBLE);
                      }
                    }
                    */
                    break;
                  case  6:  // style
                    if(currentLayer) currentLayer->setStyle( graphic->nameToStyle(dxfLine) );
                    break;
                  case 39:  // Thickness
                    if(currentLayer) currentLayer->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
				  	if (dxfLine.toInt()==0) {
						oldColorNumbers=true;
					}
                    if(currentLayer) {
						currentLayer->setColor( graphic->numberToColor(dxfLine.toInt(), !oldColorNumbers));
					}
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          graphic->setStyle("CONTINOUS");
          graphic->setWidth(0);
          graphic->setColor(0, false);
        }

        // ------
        // Block:
        // ------
        else if(dxfLine=="BLOCK") {

          RLOG( "\nWe've found a block..." );

          QString blockName;
          double relX=0.0, relY=0.0;

          // Read block information until first entity:
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  2:  // Block name
                    blockName = dxfLine;
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // Block-reference point X1
                    relX = dxfLine.toDouble();
                    break;
                  case 20:  // Block-reference point Y1
                    relY = dxfLine.toDouble();
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // proccess block:
          if(dxfCode) {
            bool done=false;
            bool first=true;

            // Read the whole contents of this block:
            //
            QString blockContents="";

            do {
              if(first) {
                first=false;
                code=0;
              }
              else {
                dxfCode=getBufLine();
                blockContents+=dxfCode+"\n";
                if(dxfCode) code=dxfCode.toInt();
              }

              if(dxfCode) {
                dxfLine=getBufLine();
                if(code!=0 || dxfLine!="ENDBLK") blockContents+=dxfLine+"\n";
              }

              if(dxfCode && code==0 && dxfLine=="ENDBLK") {
                done=true;
              }

            }while(dxfCode && !done);

            RGraphic* newBlock = new RGraphic( 0 );
            RFileDxf blockBuffer( blockName, newBlock );

            bool beg=true;
            char* buf = new char[ blockContents.length() ];
            for( int i=0; i<(int)blockContents.length(); ++i ) {
              if( blockContents.at(i).latin1()!='\0' || !beg ) {
                buf[i] = blockContents.at(i).latin1();
                beg=false;
              }
            }

            //RLOG( "\nThe buffer is: " );
            //RLOG( blockContents );

            blockBuffer.setBuf( buf );
            blockBuffer.setFSize( blockContents.length() );

            //blockBuffer.setBuf( blockContents.latin1() );
            blockBuffer.separateBuf();

            RLOG( "\nRead buffer..." );
            bool rb = blockBuffer.readFromBuffer();
            RLOG( "\nBuffer read: " );
            RLOG( rb );

            delete buf;

            RLOG( "\nNumber of elements in this block: " );
            RLOG( newBlock->count() );

            if(relX!=0.0 || relY!=0.0) {
              //newBlock->tagAll();
              newBlock->editMove( 0, 0.0, 0.0, -relX, -relY, true, false );
            }
            newBlock->setFileName( blockName );
            blockList.append( newBlock );
          }
        }

        // -------------
        // Insert Block:
        // -------------
        else if(dxfLine=="INSERT") {
          QString blockName;
          double factorX=1.0;
          //double factorY=1.0;
          double posX=0.0, posY=0.0;
          double angle=0.0;
          //int rowCount=1, colCount=1;
          //double rowSpacing=0.0, colSpacing=0.0;

          do {
            dxfCode=getBufLine();

            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();

              RLOG( "\ninsertLine: " );
              RLOG( dxfLine );

              if(dxfLine) {
                switch(code) {
                  case  2:  // Block name
                    blockName = dxfLine;
                    RLOG( "\nblockName found: " );
                    RLOG( blockName );
                    break;
                  case 10:  // Block-reference point X1
                    posX = dxfLine.toDouble();
                    break;
                  case 20:  // Block-reference point Y1
                    posY = dxfLine.toDouble();
                    break;
                  case 41:  // Factor X
                    factorX = dxfLine.toDouble();
                    break;
                  case 42:  // Factor Y
                    //factorY = dxfLine.toDouble();
                    break;
                  case 50:  // Angle
                    angle = dxfLine.toDouble();
                    break;
                  case 70:  // Col count
                    //colCount = dxfLine.toInt();
                    break;
                  case 71:  // Row count
                    //rowCount = dxfLine.toInt();
                    break;
                  case 44:  // Col spacing
                    //colSpacing = dxfLine.toDouble();
                    break;
                  case 45:  // Row spacing
                    //rowSpacing = dxfLine.toDouble();
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Search block:

          RLOG( "\nSearch block: " );
          RLOG( blockName );

          RElement* newElement;
          for( RGraphic* bl=blockList.first(); bl!=0; bl=blockList.next() ) {

            RLOG( "\nList contains Block: " );
            RLOG( bl->getFileName() );

            if(bl->getFileName()==blockName) {

              RLOG( "\nBlock found: " );
              RLOG( blockName );

              // Insert block with given parameters:
              for( RElement* el=bl->elementFirst(); el!=0; el=bl->elementNext() ) {
                newElement = new RElement( graphic );

                RLOG( "\nNew Layer: " );
                RLOG( bl->getLayerName( el->getLayer() ) );

                newElement->getMeasuresFrom( el );
                newElement->setLayer( graphic->addLayer( bl->getLayerName( el->getLayer() ) ) );
                newElement->rotate( angle, 0.0, 0.0, false );
                newElement->scale( factorX, 0.0, 0.0, false );
                newElement->move( posX, posY );
                //newElement->setFlag(E_VISIBLE|E_ACTIVE);
                graphic->addElement( newElement, add );
                //graphic->appendElement( newElement );
              }

              break;
            }
          }
        }

        // ------
        // Point:
        // ------
        else if(dxfLine=="POINT") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // X1
                    dxfLine.replace( QRegExp(","), "." );
                    vx1 = dxfLine.toDouble();
                    break;
                  case 20:  // Y1
                    dxfLine.replace( QRegExp(","), "." );
                    vy1 = dxfLine.toDouble();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          graphic->addPoint(vx1, vy1, currentLayerNum, add);
        }
        
        // -----
        // Line:
        // -----
        else if(dxfLine=="LINE") {
          do {
            dxfCode=getBufLine();

            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              
              dxfLine=getBufLine();

              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // X1
                    dxfLine.replace( QRegExp(","), "." );
                    vx1 = dxfLine.toDouble();
                    break;
                  case 20:  // Y1
                    dxfLine.replace( QRegExp(","), "." );
                    vy1 = dxfLine.toDouble();
                    break;
                  case 11:  // X2
                    dxfLine.replace( QRegExp(","), "." );
                    vx2 = dxfLine.toDouble();
                    break;
                  case 21:  // Y2
                    dxfLine.replace( QRegExp(","), "." );
                    vy2 = dxfLine.toDouble();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          
          if(!mtCompFloat(vx1, vx2) || !mtCompFloat(vy1, vy2)) {
            graphic->addLine(vx1, vy1, vx2, vy2, currentLayerNum, add);
          }
        }
        
        // ----
        // Arc:
        // ----
        else if(dxfLine=="ARC") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // Centre X
                    dxfLine.replace( QRegExp(","), "." );
                    vcx = dxfLine.toDouble();
                    break;
                  case 20:  // Centre Y
                    dxfLine.replace( QRegExp(","), "." );
                    vcy = dxfLine.toDouble();
                    break;
                  case 40:  // Radius
                    dxfLine.replace( QRegExp(","), "." );
                    vcr = dxfLine.toDouble();
                    break;
                  case 50:  // Start Angle
                    dxfLine.replace( QRegExp(","), "." );
                    va1 = mtCorrAngle(dxfLine.toDouble());
                    break;
                  case 51:  // End Angle
                    dxfLine.replace( QRegExp(","), "." );
                    va2 = mtCorrAngle(dxfLine.toDouble());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          if(vcr>0.0 && !mtCompFloat(va1, va2)) {
            graphic->addArc(vcx, vcy, vcr, va1, va2, false, currentLayerNum, add);
          }
        }
        
        // -------
        // Circle:
        // -------
        else if(dxfLine=="CIRCLE") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // Centre X
                    dxfLine.replace( QRegExp(","), "." );
                    vcx = dxfLine.toDouble();
                    break;
                  case 20:  // Centre Y
                    dxfLine.replace( QRegExp(","), "." );
                    vcy = dxfLine.toDouble();
                    break;
                  case 40:  // Radius
                    dxfLine.replace( QRegExp(","), "." );
                    vcr = dxfLine.toDouble();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          if(vcr>0.0) {
            graphic->addCircle(vcx, vcy, vcr, 0.0, 360.0, false, currentLayerNum, add);
          }
        }

        // --------
        // Ellipse:
        // --------
        else if(dxfLine=="ELLIPSE") {
          double vapx=0.0, vapy=0.0; // Apex (endpoint of center axis)
          double vratio=1.0;         // Ratio of minor axis to major axis
          bool  flipY=false;        // Flip y-axis (if extrusion direction is -1)
          do {
            dxfCode=getBufLine();
            
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // Centre X
                    dxfLine.replace( QRegExp(","), "." );
                    vcx = dxfLine.toDouble();
                    break;
                  case 20:  // Centre Y
                    dxfLine.replace( QRegExp(","), "." );
                    vcy = dxfLine.toDouble();
                    break;
                  case 11:  // Endpoint of major axis X
                    dxfLine.replace( QRegExp(","), "." );
                    vapx = dxfLine.toDouble();
                    break;
                  case 21:  // Endpoint of major axis Y
                    dxfLine.replace( QRegExp(","), "." );
                    vapy = dxfLine.toDouble();
                    break;
                  case 40:  // Ratio of minor axis to major axis (eg.: 0.75)
                    dxfLine.replace( QRegExp(","), "." );
                    vratio = dxfLine.toDouble();
                    break;
                  case 41:  // Start angle in rad (0.0 for a full ellipse)
                    dxfLine.replace( QRegExp(","), "." );
                    va1 = dxfLine.toDouble();
                    break;
                  case 42:  // End angle in rad (2PI=6.283 for a full ellipse)
                    dxfLine.replace( QRegExp(","), "." );
                    va2 = dxfLine.toDouble();
                    break;
                  case 230: // extrusion direction
                    flipY = ((int)dxfLine.toDouble()==-1);
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Create the ellipse as lines:
          //
          double ac;            // angle counter
          double refAng;        // ref angle (major axis)
          double tx, ty;        // tmp pos on ellipse
          double otx, oty;      // old tmp pos on ellipse
          double aStep = RCONFIG->getSettingDouble("DXF:DxfEllipseSegmentAngle");  // angle step

          if(aStep<0.01) aStep=0.01;

          vapx+=vcx;
          vapy+=vcy;
          vcr=mtGetDistance(vcx, vcy, vapx, vapy);
          refAng=mtGetAngle(vcx, vcy, vapx, vapy);

          otx=vcx + cos(va1) * vcr;
          if(!flipY) oty=vcy + sin(va1) * vcr * vratio;
          else       oty=vcy - sin(va1) * vcr * vratio;
          mtRotatePoint(vcx, vcy, &otx, &oty, refAng);
          
          for(ac=va1*ARAD+aStep; ac<=va2*ARAD; ac+=aStep) {
            tx=vcx + cos(ac/ARAD) * vcr;
            if(!flipY) ty=vcy + sin(ac/ARAD) * vcr * vratio;
            else       ty=vcy - sin(ac/ARAD) * vcr * vratio;

            if(!mtCompFloat(refAng, 0.0)) {
              mtRotatePoint(vcx, vcy, &tx, &ty, refAng);
            }
            
            if(!mtCompFloat(otx, tx) || !mtCompFloat(oty, ty)) {
              graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
            }

            otx=tx;
            oty=ty;
          }

          if(!mtCompFloat(ac, va2*ARAD)) {
            tx=vcx + cos(va2) * vcr;
            if(!flipY) ty=vcy + sin(va2) * vcr * vratio;
            else       ty=vcy - sin(va2) * vcr * vratio;
            mtRotatePoint(vcx, vcy, &tx, &ty, refAng);

            if(!mtCompFloat(otx, tx) || !mtCompFloat(oty, ty)) {
              graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
            }
          }
        }

        // -------
        // Spline:
        // -------
        else if(dxfLine=="SPLINE") {
          //bool  flipY=false;      // Flip y-axis (if extrusion direction is -1)
          int   ctrlPts=1;        // number of control points
          bool  closed=false;

          // Get number of control points and flags:
          //
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case 70:
                    closed=(bool)(1&dxfLine.toInt());
                    break;
                  case 73:  // number of control points
                    ctrlPts=dxfLine.toInt();
                    code=0;  // break while found what we need...
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          if(closed) ++ctrlPts;

          // allocate memory for ctrl-points:
          //
          double* px;      // tmp used coordinates of ctrl pts
          double* py;      // 
          int    ac=0;    // array counter

          px=new double[ctrlPts];
          py=new double[ctrlPts];
          
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // X-value
                    dxfLine.replace( QRegExp(","), "." );
                    vx1 = dxfLine.toDouble();
                    break;
                  case 20:  // Y-value
                    dxfLine.replace( QRegExp(","), "." );
                    vy1 = dxfLine.toDouble();

                    px[ac]=vx1;
                    py[ac]=vy1;

                    // closed spline: start and end the same:
                    //
                    if(ac==0 && closed) {
                      px[ctrlPts-1]=vx1;
                      py[ctrlPts-1]=vy1;
                    }

                    // dont read too much points:
                    //
                    if(ac<ctrlPts-1) ++ac;

                    break;
                  case 40:  // Knot value (? kind of an angle)
                    break;
                  case 73:  // number of control points
                    ctrlPts=dxfLine.toInt();
                    break;
                  case 230: // extrusion direction
                    //flipY = ((int)dxfLine.toDouble()==-1);
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Create the spline as lines:
          //
          int   i;           // counter for control points
          double u;           // counter for double value (loop)
          double nc1, nc2,    // what ever...
                nc3, nc4;
          double tx, ty;      // tmp pos on spline
          double otx, oty;    // old tmp pos on spline
          int   ao1, ao2,    // array offsets
                ao3, ao4;
          int   segm;        // segments

          segm=RCONFIG->getSettingInt("DXF:DxfSplineSegments");
          if(segm<=0) segm=1;

          otx=px[0];
          oty=py[0];

          for(i=0; i<=ctrlPts-1; ++i) {
            for(u=0.0; u<=1.0; u+=1.0/segm) {
              nc1=-(u*u*u/6.0)+u*u/2.0-u/2.0+1.0/6.0;
              nc2=u*u*u/2.0-u*u+2.0/3.0;
              nc3=(-u*u*u+u*u+u)/2.0+1.0/6.0;
              nc4=u*u*u/6.0;

              ao1=-1;     // normal case
              ao2= 0;
              ao3= 1;
              ao4= 2;

              if     (i==0        ) { ao1=0;        }   // 1st part of curve
              else if(i==ctrlPts-2) { ao4=1;        }   // 2nd last part of curve
              else if(i==ctrlPts-1) { ao3=0; ao4=0; }   // last part of curve

              tx = nc1*px[i+ao1] + nc2*px[i+ao2] +
                   nc3*px[i+ao3] + nc4*px[i+ao4]   ;
              ty = nc1*py[i+ao1] + nc2*py[i+ao2] +
                   nc3*py[i+ao3] + nc4*py[i+ao4]   ;

              if(!mtCompFloat(otx, tx, 0.0015) || !mtCompFloat(oty, ty, 0.0015)) {
                graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
                otx=tx;
                oty=ty;
              }
            }
          }

          if(!mtCompFloat(otx, px[ctrlPts-1], 0.0001) || !mtCompFloat(oty, py[ctrlPts-1], 0.0001)) {
            graphic->addLine(otx, oty, px[ctrlPts-1], py[ctrlPts-1], currentLayerNum, add);
          }

          delete[] px;
          delete[] py;
        }

        // ---------------------
        // LWPOLYLINE
        // ---------------------
        else if(dxfLine=="LWPOLYLINE") {
          plClose=false;
          double plX=DEF_AREAMAX,              // current position
                plY=DEF_AREAMAX;              //
          double oplX=0.0, oplY=0.0;           // last position
          double fstX=0.0, fstY=0.0;           // first position
          bool  firstPoint=true;              // first point in list?
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case  9:
                    break;
                  case 10:
                    oplX=plX;
                    dxfLine.replace( QRegExp(","), "." );
                    plX=dxfLine.toDouble();

                    if(firstPoint) {
                      fstX=plX;
                    }
                    break;
                  case 20:
                    oplY=plY;
                    dxfLine.replace( QRegExp(","), "." );
                    plY=dxfLine.toDouble();

                    if(firstPoint) {
                      fstY=plY;
                      firstPoint=false;
                    }
                    else {
                      if(!mtCompFloat(oplX, plX) || !mtCompFloat(oplY, plY)) {
                        graphic->addLine(oplX, oplY, plX, plY, currentLayerNum, add);
                      }
                    }
                    break;
                  case 70:  // Flag "Polyline closed"
                    plClose = (bool)(1&dxfLine.toInt());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Close Polyline:
          //
          if(plClose) {
            if(!mtCompFloat(plX, fstX) || !mtCompFloat(plY, fstY)) {
              graphic->addLine(plX, plY, fstX, fstY, currentLayerNum, add);
            }
          }
        }

        // ------
        // Hatch:
        // ------
        /*
        if(dxfLine=="HATCH") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // X1
                    vx1 = dxfLine.toDouble();
                    break;
                  case 20:  // Y1
                    vy1 = dxfLine.toDouble();
                    //graphic->Vec[vc].CreatePoint(vy1, vx1, currentLayerNum);
                    //if(vc<vElements-1) ++vc;
                    break;
                  case 11:  // X2
                    vx2 = dxfLine.toDouble();
                    break;
                  case 21:  // Y2
                    vy2 = dxfLine.toDouble();
                    //graphic->Vec[vc].CreatePoint(vy2, vx2, currentLayerNum);
                    //if(vc<vElements-1) ++vc;
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          / *
          if(!mt.CompFloat(vx1, vx2) || !mt.CompFloat(vy1, vy2)) {
            graphic->Vec[vc].CreateLine(vx1, vy1, vx2, vy2, currentLayerNum);
            if(vc<vElements-1) ++vc;
          }
          if(++updProgress==1000) {
            np->getStateWin()->UpdateProgressBar((int)(pcFact*vc)+25);
            updProgress=0;
          }
          * /
        }
        */

        // ---------------------
        // First Polyline-Point:
        // ---------------------
        else if(dxfLine=="POLYLINE") {
          plClose=false;
          // Get Flag "Polyline closed":
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case  9:
                    break;
                  case 70:  // Flag "Polyline closed"
                    plClose = (bool)(1&dxfLine.toInt());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          gotoBufLine("VERTEX");
          vab=0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case  9:
                    break;
                  case 10:   // Next Coordinate X
                    dxfLine.replace( QRegExp(","), "." );
                    vpx = ax = dxfLine.toDouble();
                    break;
                  case 20:   // Next Coordinate Y
                    dxfLine.replace( QRegExp(","), "." );
                    vpy = ay = dxfLine.toDouble();
                    break;
                  case 42:   // Next Bulge
                    dxfLine.replace( QRegExp(","), "." );
                    vab = dxfLine.toDouble();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
        }

        // --------------------
        // Next Polyline-Point:
        // --------------------
        else if(dxfLine=="VERTEX") {
          //graphic->Vec[vc].Reset();
          double nvab=0.0;     // Next Bulge
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case  9:
                    break;
                  case 10:    // Next Coordinate X
                    dxfLine.replace( QRegExp(","), "." );
                    vx2 = dxfLine.toDouble();
                    break;
                  case 20:    // Next Coordinate Y
                    dxfLine.replace( QRegExp(","), "." );
                    vy2 = dxfLine.toDouble();
                    break;
                  case 42:    // Next Bulge
                    dxfLine.replace( QRegExp(","), "." );
                    nvab = dxfLine.toDouble();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          //if(!mtCompFloat(ax, vx2) || !mtCompFloat(ay, vy2) || !mtCompFloat(vab, 0.0)) {
          // removed: 20000201: Doesn't work for polylines with very high resolution
            graphic->addPolylineElement(ax, ay,
                                        vx2, vy2,
                                        vab,
                                        currentLayerNum,
                                        add);
          //}
          ax = vx2;
          ay = vy2;
          vab=nvab;
        }

        // ----------------
        // Close Polylinie:
        // ----------------
        else if(dxfLine=="SEQEND" && plClose) {
          if(!mtCompFloat(ax, vpx) || !mtCompFloat(ay, vpy) || !mtCompFloat(vab, 0.0)) {
            graphic->addPolylineElement(ax, ay,
                                        vpx, vpy,
                                        vab,
                                        currentLayerNum,
                                        add);
          }
        }

        // -----
        // Text:
        // -----
        else if(dxfLine=="TEXT" || dxfLine=="MTEXT") {

          bool  mtext = (dxfLine=="MTEXT");
          QString vtext;          // the text
          char  vtextStyle[256];  // text style (normal_ro, cursive_ri, normal_st, ...)
          double vheight=10.0,     // text height
                vtextAng=0.0,     // text angle
                vradius=0.0,      // text radius
                vletterspace=2.0, // Text letter space
                vwordspace=6.0;   // Text wordspace
          QString vfont;          // font "normal", "cursive", ...
          int   valign=0;         // alignment (0=left, 1=center, 2=right)
          int   vattachement=7;   // 1=top left, 2, 3, 4, 5, 6, 7, 8, 9=bottom right
          uint  vfl=0;            // special flags
          bool  codeSeven=false;  // Have we found a code seven?

          vtextStyle[0] = '\0';
          vfont="normal";

          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              if(code!=1 && code!=3 && code!=7) dxfLine=getBufLine();
              if(dxfLine || code==1 || code==3 || code==7) {

                switch(code) {

                  case  1:  // Text itself
                    vtext=getBufLine();
                    strDecodeDxfString(vtext);
                    break;

                  case  3:  // Text parts (always 250 chars)
                    vtext=getBufLine();
                    break;

                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;

                  case  7:  // Text style (normal_ro#50.0, cursive_ri#20.0, normal_st)
                    qstrncpy(vtextStyle, getBufLine(), 249);

                    // get font typ:
                    //
                    if(true) {
                      char dummy[256];
                      sscanf(vtextStyle, "%[^_#\n]", dummy);
                      vfont=dummy;
                    }

                    // get text style:
                    //
                    if(strstr(vtextStyle, "_ro"))      vfl=vfl|E_ROUNDOUT;
                    else if(strstr(vtextStyle, "_ri")) vfl=vfl|E_ROUNDIN;
                    else                               vfl=vfl|E_STRAIGHT;

                    if(strstr(vtextStyle, "_fix"))     vfl=vfl|E_FIXEDWIDTH;

                    // get radius, letterspace, wordspace:
                    //
                    if(true) {
                      char *ptr;  // pointer to value
                      ptr = strchr(vtextStyle, '#');
                      if(ptr) {
                        // Parse radius
                        if(vfl&E_ROUNDOUT || vfl&E_ROUNDIN) {
                          ++ptr;
                          if(ptr[0]) {
                            sscanf(ptr, "%lf", &vradius);
                          }
                          ptr = strchr(ptr, '#');
                        }
                        if(ptr) {
                          // Parse letter space:
                          ++ptr;
                          if(ptr[0]) {
                            sscanf(ptr, "%lf", &vletterspace);
                          }
                          // Parse word space:
                          ptr = strchr(ptr, '#');
                          if(ptr) {
                            ++ptr;
                            if(ptr[0]) {
                              sscanf(ptr, "%lf", &vwordspace);
                            }
                          }
                        }
                      }
                    }
                    codeSeven=true;
                    break;

                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;

                  case 10:  // X1
                    dxfLine.replace( QRegExp(","), "." );
                    vx1 = dxfLine.toDouble();
                    break;
                  case 20:  // Y1
                    dxfLine.replace( QRegExp(","), "." );
                    vy1 = dxfLine.toDouble();
                    break;
                  case 40:  // height
                    dxfLine.replace( QRegExp(","), "." );
                    vheight = dxfLine.toDouble();
                    if(!codeSeven) {
                      vletterspace = vheight*0.2;
                      vwordspace = vheight*0.6;
                    }
                    break;
                  case 50:  // angle
                    dxfLine.replace( QRegExp(","), "." );
                    vtextAng = dxfLine.toDouble();
                    break;
                  case 71:  // attachement point
                    if(mtext) {
                      vattachement = dxfLine.toInt();
                      switch(vattachement) {
                        case 1: vfl = vfl|E_LEFT  |E_TOP   ; break;
                        case 2: vfl = vfl|E_CENTER|E_TOP   ; break;
                        case 3: vfl = vfl|E_RIGHT |E_TOP   ; break;
                        case 4: vfl = vfl|E_LEFT  |E_MIDDLE; break;
                        case 5: vfl = vfl|E_CENTER|E_MIDDLE; break;
                        case 6: vfl = vfl|E_RIGHT |E_MIDDLE; break;
                        case 7: vfl = vfl|E_LEFT  |E_BOTTOM; break;
                        case 8: vfl = vfl|E_CENTER|E_BOTTOM; break;
                        case 9: vfl = vfl|E_RIGHT |E_BOTTOM; break;
                        default: break;
                      }
                    }
                    break;
                  case 72:  // alignment
                    if(!mtext) {
                      valign = dxfLine.toInt();
                      if(valign==1)      vfl=vfl|E_CENTER;
                      else if(valign==2) vfl=vfl|E_RIGHT;
                      else               vfl=vfl|E_LEFT;
                    }
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          graphic->addText(vx1, vy1,
                           RFonts::getRFonts()->getFontNumber(vfont),
                           vtext,
                           vfl,
                           vheight,
                           vtextAng,
                           vradius,
                           vletterspace,
                           vwordspace,
                           vheight*1.4,
                           currentLayerNum,
                           add);
          //graphic->addSingleElements(graphic->elementCurrent());
        }

        // ----------
        // Dimension:
        // ----------
        else if(dxfLine=="DIMENSION") {
          int typ=1;
          double v10=0.0, v20=0.0,
                v13=0.0, v23=0.0,
                v14=0.0, v24=0.0,
                v15=0.0, v25=0.0,
                v16=0.0, v26=0.0,
                v40=0.0, v50=0.0;
          QString dimText;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  1:  // Text (if any)
                    dimText=dxfLine;

                    // Mend unproper savings of older versions:
                    if(dimText==" ") dimText="";

                    //else dimText.replace(QRegExp("%%c"), "");
                    else strDecodeDxfString(dimText);
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // line position x
                    dxfLine.replace( QRegExp(","), "." );
                    v10 = dxfLine.toDouble();
                    break;
                  case 20:  // line position y
                    dxfLine.replace( QRegExp(","), "." );
                    v20 = dxfLine.toDouble();
                    break;
                  case 13:  // X1
                    dxfLine.replace( QRegExp(","), "." );
                    v13 = dxfLine.toDouble();
                    break;
                  case 23:  // Y1
                    dxfLine.replace( QRegExp(","), "." );
                    v23 = dxfLine.toDouble();
                    break;
                  case 14:  // X2
                    dxfLine.replace( QRegExp(","), "." );
                    v14 = dxfLine.toDouble();
                    break;
                  case 24:  // Y2
                    dxfLine.replace( QRegExp(","), "." );
                    v24 = dxfLine.toDouble();
                    break;
                  case 15:  // X2
                    dxfLine.replace( QRegExp(","), "." );
                    v15 = dxfLine.toDouble();
                    break;
                  case 25:  // Y2
                    dxfLine.replace( QRegExp(","), "." );
                    v25 = dxfLine.toDouble();
                    break;
                  case 16:  // X2
                    dxfLine.replace( QRegExp(","), "." );
                    v16 = dxfLine.toDouble();
                    break;
                  case 26:  // Y2
                    dxfLine.replace( QRegExp(","), "." );
                    v26 = dxfLine.toDouble();
                    break;
                  case 40:
                    dxfLine.replace( QRegExp(","), "." );
                    v40 = dxfLine.toDouble();
                    break;
                  case 50:
                    dxfLine.replace( QRegExp(","), "." );
                    v50 = dxfLine.toDouble();
                    break;
                  case 70:  // Typ
                    typ = dxfLine.toInt();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;

                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          double dist;
          RElement tmpEl;

          // Remove Bit values:
          if(typ>=128) typ-=128;   // Location of Text
          if(typ>= 64) typ-= 64;   // Ordinate

          switch(typ) {
            // Horiz. / vert.:
            case 0:
              if(mtCompFloat(v50, 0.0)) {
                graphic->addDimension(v13, v23, v14, v24,
                                      0.0, 0.0,
                                      v20,
                                      E_STRAIGHT|E_HORIZONTAL,
                                      currentLayerNum,
                                      add);
              }
              else {
                graphic->addDimension(v13, v23, v14, v24,
                                      0.0, 0.0,
                                      v10,
                                      E_STRAIGHT|E_VERTICAL,
                                      currentLayerNum,
                                      add);
              }
              break;

            // Aligned:
            case 1:
            default:
              tmpEl.createLine(v13, v23, v14, v24);
              dist = tmpEl.getDistanceToPoint(v10, v20, true);
              if(tmpEl.getRelAngleToPoint(v10, v20, true)<0.0) dist*=-1;
              graphic->addDimension(v13, v23, v14, v24,
                                    0.0, 0.0,
                                    dist,
                                    E_STRAIGHT,
                                    currentLayerNum,
                                    add);
              break;

            // Angle:
            case 2:
              {
                RElement tmpEl1, tmpEl2;
                tmpEl1.createLine(v13, v23, v14, v24);
                tmpEl2.createLine(v10, v20, v15, v25);
                
                if (!tmpEl1.isParallelWith (&tmpEl2)) {
                
                  bool inters=false;
                  tmpEl1.getIntersection(&tmpEl2,
                                         &inters, &vcx, &vcy, 0,0,0,0, false);
                  if (inters) {                       
                    vcr = mtGetDistance(vcx, vcy, v16, v26);
  
                    if(mtGetDistance(vcx,vcy, v13,v23)<vcr) {
                      va1 = tmpEl1.getDirection1();
                    }
                    else {
                      va1 = tmpEl1.getDirection2();
                    }
  
                    if(mtGetDistance(vcx,vcy, v10,v20)<vcr) {
                      va2 = tmpEl2.getDirection1();
                    }
                    else {
                      va2 = tmpEl2.getDirection2();
                    }
                    graphic->addDimension(vcx, vcy, va1, va2,
                                          mtGetDistance(vcx, vcy, v13, v23),
                                          mtGetDistance(vcx, vcy, v10, v20),
                                          vcr,
                                          E_ROUNDOUT,
                                          currentLayerNum,
                                          add);
                  }
                }
              }
              break;

            // Radius:
            case 4:
              graphic->addDimension(v10, v20, v15, v25,
                                    0.0, 0.0,
                                    v40,
                                    E_STRAIGHT|E_RADIUS,
                                    currentLayerNum,
                                    add);
              break;

            // Arrow:
            case 7:
              graphic->addDimension(v13, v23, v14, v24,
                                    0.0, 0.0, 0.0,
                                    E_STRAIGHT|E_ARROW,
                                    currentLayerNum,
                                    add);
              break;
          }
          graphic->elementCurrent()->setText(dimText);
        }

        // ---------
        // Hatching:
        // ---------
        else if(dxfLine=="HATCH") {
          QString patternName="45";
          double patternScale=1.0;
          //int numPaths=1;
          //int numEdges=1;
          int nextObjectTyp=T_LINE;
          double v10=0.0, v20=0.0,
                v11=0.0, v21=0.0,
                v40=0.0, v50=0.0,
                v51=0.0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  2:
                    patternName = dxfLine;
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine);
                      lastLayer=dxfLine;
                    }
                    break;
                  case 10:  // Start point/center of boundary line/arc
                    dxfLine.replace( QRegExp(","), "." );
                    v10=dxfLine.toDouble();
                    break;
                  case 20:  // Start point/center of boundary line/arc
                    dxfLine.replace( QRegExp(","), "." );
                    v20=dxfLine.toDouble();
                    break;
                  case 11:  // End point of boundary line
                    dxfLine.replace( QRegExp(","), "." );
                    v11=dxfLine.toDouble();
                    break;
                  case 21:  // End point of boundary line
                    dxfLine.replace( QRegExp(","), "." );
                    v21=dxfLine.toDouble();
                    if(nextObjectTyp==T_LINE) {
                      int elnu=graphic->addLine(v10, v20, v11, v21, currentLayerNum, add);
                      graphic->elementAt(elnu)->setFlag(E_TAGGED);
                    }
                    break;
                  case 40:  // Radius of boundary entity
                    dxfLine.replace( QRegExp(","), "." );
                    v40=dxfLine.toDouble();
                    break;
                  case 50:  // Start angle
                    dxfLine.replace( QRegExp(","), "." );
                    v50=dxfLine.toDouble();
                    break;
                  case 51:  // End angle
                    dxfLine.replace( QRegExp(","), "." );
                    v51=dxfLine.toDouble();
                    break;
                  case 73:  // Counterclockwise?
                    if(nextObjectTyp==T_ARC) {
                      int elnu;
                      if( mtCompFloat( v50, 0.0 ) && mtCompFloat( v51, 0.0 ) ) {
                        elnu=graphic->addCircle(v10, v20, v40, 0.0, 360.0, (bool)dxfLine.toInt(), currentLayerNum, add);
                      }
                      else {
                        elnu=graphic->addArc(v10, v20, v40, v50, v51, (bool)dxfLine.toInt(), currentLayerNum, add);
                      }
                      graphic->elementAt(elnu)->setFlag(E_TAGGED);
                      //newEl = new RElement( graphic );
                      //newEl->createArc(v10, v20, v40, v50, v51, (bool)dxfLine.toInt());
                      //boundaryList.append(newEl);
                    }
                    break;
                  case 41:  // Scale
                    dxfLine.replace( QRegExp(","), "." );
                    patternScale=dxfLine.toDouble();
                    break;
                  case 52:  // Angle

                    break;
                  case 70:  // Solid (=1) or pattern (=0)

                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt(), !oldColorNumbers);
                    break;
                  case 91:  // Number of boundary paths (loops)
                    //numPaths=dxfLine.toInt();
                    break;
                  case 92:  // Typ of boundary

                    break;
                  case 93:  // Number of edges in this boundary
                    //numEdges=dxfLine.toInt();
                    break;
                  case 72:  // Edge typ
                    switch(dxfLine.toInt()) {
                      case 1: nextObjectTyp=T_LINE; break;
                      case 2: nextObjectTyp=T_ARC;  break;
                      default: break;
                    }
                    break;

                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          graphic->addHatching(patternScale,
                               patternName,
                               currentLayerNum,
                               add);

          graphic->editDelete(false);

          /*RElement* el;
          for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
            graphic->
          }*/

          /*for(el=boundaryList.first(); el!=0; el=boundaryList.next()) {

          }*/

        }

      }
    } while(dxfLine && dxfLine!="EOF");
    
    graphic->terminateAction();

    graphic->debugElements();

    ret=true;
  }
  else {
    ret=false;
  }

  if(!graphic->getFlag(G_PREVIEW)) statusPanel()->delProgress();

  return ret;
}



// save a dxf file:
//
bool
RFileDxf::save()
{
  RLOG( "\nRFileDxf::save()" );

  bool      ret;       // returned value
  int       i;         // counter
  RLayer* lay;         // Pointer to a layer in the graphic
  RElement *el;        // pointer which walks through elements

  statusPanel()->iniProgress(graphic->count(), QObject::tr("Saving File..."));

  RLOG( "\nstatusPanel()->iniProgress done" );

  fPointer = fopen(name(), "wt");
  if(fPointer!=NULL) {

    RLOG( "\nfPointer!=NULL" );

    // Write DXF-header:
    
    // HEADER:
    //
    fprintf(fPointer,
            "  0\nSECTION\n  2\nHEADER\n");
    fflush( fPointer );

    // Settings:
    //

    // LIMITS:
    //
    double minX, minY, maxX, maxY; // Borders of graphic
    graphic->getBorders(minX, maxY, maxX, minY);
    fprintf(fPointer,
            "  9\n$LIMMIN\n 10\n%f\n 20\n%f\n",
            minX, minY);
    fprintf(fPointer,
            "  9\n$LIMMAX\n 10\n%f\n 20\n%f\n",
            maxX, maxY);
    fprintf(fPointer,
            "  9\n$EXTMIN\n 10\n%f\n 20\n%f\n",
            minX, minY);
    fprintf(fPointer,
            "  9\n$EXTMAX\n 10\n%f\n 20\n%f\n",
            maxX, maxY);

    // Angle base:
    fprintf(fPointer,
            "  9\n$ANGBASE\n 50\n%f\n",
            0.0);

    // Angle direction:
    fprintf(fPointer,
            "  9\n$ANGDIR\n 70\n%d\n",
            0);

    // Units:
    fprintf(fPointer,
            "  9\n$INSUNITS\n 70\n%d\n",
            (int)graphic->getUnit() );

    // Dimension Units:
    fprintf(fPointer,
            "  9\n$DIMALT\n 70\n%d\n",
            (int)graphic->getDimensionUnit() );

    // Dimension Format:
    int f;
    switch( graphic->getDimensionFormat() ) {
      case Scientific: f=1; break;
      case Decimal:    f=2; break;
      case Fractional: f=5; break;
      default:         f=2; break;
    }
    fprintf(fPointer,
            "  9\n$DIMLUNIT\n 70\n%d\n",
            f );

    // Dimension Arrow Size:
    fprintf(fPointer,
            "  9\n$DIMASZ\n 40\n%f\n",
            graphic->getDimensionArrowSize() );


    // Dimension Scale:
    /*fprintf(fPointer,
            "  9\n$DIMSCALE\n 40\n%f\n",
            graphic->getDimensionScale() );*/

    // Dimension Text Height:
    fprintf(fPointer,
            "  9\n$DIMTXT\n 40\n%f\n",
            graphic->getDimensionTextHeight() );

    // Dimension exactness:
    fprintf(fPointer,
            "  9\n$DIMRND\n 40\n%f\n",
            graphic->getDimensionExactness() );

    // Dimension over length:
    fprintf(fPointer,
            "  9\n$DIMEXE\n 40\n%f\n",
            graphic->getDimensionOverLength() );

    // Dimension under length:
    fprintf(fPointer,
            "  9\n$DIMEXO\n 40\n%f\n",
            graphic->getDimensionUnderLength() );

    // Angle dimension format:
    fprintf(fPointer,
            "  9\n$DIMAUNIT\n 70\n%d\n",
            (int)graphic->getAngleDimensionFormat() );

    // Angle dimension exactness:
    fprintf(fPointer,
            "  9\n$DIMADEC\n 70\n%d\n",
            mtRound(- log10( graphic->getAngleDimensionExactness() )) );

    // Grid x/y:
    fprintf(fPointer,
            "  9\n$GRIDUNIT\n 10\n%f\n",
            graphic->getMinGridX() );
    fprintf(fPointer,
            " 20\n%f\n",
            graphic->getMinGridY() );

    // Page min x/y:
    fprintf(fPointer,
            "  9\n$PLIMMIN\n 10\n%f\n",
            graphic->getPageOriginX() );
    fprintf(fPointer,
            " 20\n%f\n",
            graphic->getPageOriginY() );

    // Page max x/y:
    fprintf(fPointer,
            "  9\n$PLIMMAX\n 10\n%f\n",
            graphic->getPageOriginX()+graphic->getPageSizeX() );
    fprintf(fPointer,
            " 20\n%f\n",
            graphic->getPageOriginY()+graphic->getPageSizeY() );

    // Paper space scale:
    fprintf(fPointer,
            "  9\n$PSVPSCALE\n 40\n%f\n",
            graphic->getPaperSpace() );


    fprintf(fPointer, "  0\nENDSEC\n  0\nSECTION\n  2\nTABLES\n  0\n");
    fprintf(fPointer, "TABLE\n  2\nLTYPE\n 70\n     1\n  0\nLTYPE\n  2\n");
    fprintf(fPointer, "CONTINUOUS\n 70\n    64\n  3\nAusgez\n 72\n    65\n 73\n");
    fprintf(fPointer, "  0\n 40\n0.000000\n  0\nENDTAB\n  0\nTABLE\n");
    fflush( fPointer );

    RLOG( "\nHeader done" );
    
    // LAYER:
    //
    fprintf(fPointer, "  2\nLAYER\n 70\n     1\n");
    for(i=0; i<graphic->countLayers(); ++i) {
      lay = graphic->getLayer(i);
      if(lay->getFlag(Y_USED)) {
        fprintf(fPointer, "  0\nLAYER\n  2\n");
        fprintf(fPointer, "%s\n", (const char*)lay->getName().local8Bit());
        fprintf(fPointer, " 70\n    64\n");

				// Not accepted by Autocad R14:
        //fprintf(fPointer, " 39\n%d\n",
        //                  lay->getWidth());

		int colNum = graphic->colorToNumber(lay->getColor(), true);
        fprintf(fPointer, " 62\n%d\n",
                          (colNum==0 ? 7 : colNum));
        fprintf(fPointer, "  6\n%s\n",
                          graphic->styleToName(lay->getStyle()).latin1());
      }
    }

    RLOG( "\nLayers done" );
    
    // HEADER END:
    //
    fprintf(fPointer, "  0\nENDTAB\n  0\nENDSEC\n  0\nSECTION\n  2\n");
    fprintf(fPointer, "BLOCKS\n  0\nENDSEC\n  0\nSECTION\n  2\nENTITIES\n");
    fflush( fPointer );

    RLOG( "\nHeader end done" );

    // ELEMENTS:
    //
    for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
      //if(el->getFlag(E_VISIBLE)) {   // Save elements on invisible layers too
      if(el->getFlag(E_ACTIVE) && !el->getFlag(E_UNDO)) {

        statusPanel()->setProgress(graphic->elementAt(), 200);

        switch(el->getElementTyp()) {
        
          // Point:
          //
          case T_POINT:
            fprintf(fPointer, "  0\nPOINT\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getX1());
            fprintf(fPointer, " 20\n%f\n",
                              el->getY1());

            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;
            
          // Line:
          //
          case T_LINE:
            fprintf(fPointer, "  0\nLINE\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getX1());
            fprintf(fPointer, " 20\n%f\n",
                              el->getY1());
            fprintf(fPointer, " 11\n%f\n",
                              el->getX2());
            fprintf(fPointer, " 21\n%f\n",
                              el->getY2());
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;
            
          // Arc:
          //
          case T_ARC:
            fprintf(fPointer, "  0\nARC\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getCx());
            fprintf(fPointer, " 20\n%f\n",
                              el->getCy());
            fprintf(fPointer, " 40\n%f\n",
                              el->getCr());
            if(!el->getFlag(E_REVERSED)) {
              fprintf(fPointer, " 50\n%f\n",
                                el->getA1());
              fprintf(fPointer, " 51\n%f\n",
                                el->getA2());
            } 
            else {
              fprintf(fPointer, " 50\n%f\n",
                                el->getA2());
              fprintf(fPointer, " 51\n%f\n",
                                el->getA1());
            }
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;
            
          // Circle:
          //
          case T_CIRCLE:
            fprintf(fPointer, "  0\nCIRCLE\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getCx());
            fprintf(fPointer, " 20\n%f\n",
                              el->getCy());
            fprintf(fPointer, " 40\n%f\n",
                              el->getCr());
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;
            
          // Text:
          //
          case T_TEXT:
            if(true) {
              int   align=0;          // Align: (0=left, 1=center, 2=right)
              char  style[255];       // Style: (normal_ro#, cursive_ri#)
              int   tlc;              // Textline-Counter
              QString textLine;       // Style: (normal_ro#, cursive_ri#)
              double radius;          // radius counter
              double xp, yp;          // position
              
              if(el->getFlag(E_CENTER)) align=1;
              if(el->getFlag(E_RIGHT )) align=2;
              
              style[0]='\0';
            
              // Style: (normal, cursive, ...)
              //
              strcat(style, RFonts::getRFonts()->getFontName((int)el->getFont()));
  
              if(el->getFlag(E_ROUNDOUT)) strcat(style, "_ro");
              if(el->getFlag(E_ROUNDIN )) strcat(style, "_ri");
              if(el->getFlag(E_FIXEDWIDTH)) strcat(style, "_fix");
  
              xp = el->getX1();
              yp = el->getY1();
              
              radius = el->getCr();
              for(tlc=0; tlc<el->getNumberOfTextLines(); ++tlc) {
                 textLine = el->getTextLine(tlc);
                 if(!textLine.isEmpty()) {
            
                  fprintf(fPointer, "  0\nTEXT\n");
                  fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
                  fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
                  fprintf(fPointer, " 10\n%f\n",
                                    xp);
                  fprintf(fPointer, " 20\n%f\n",
                                    yp);
                  fprintf(fPointer, " 40\n%f\n",
                                    el->getY2());
                  fprintf(fPointer, " 50\n%f\n",
                                    el->getA1());
                  fprintf(fPointer, " 72\n%d\n",
                                    align);
                  fprintf(fPointer, "  7\n%s",
                                    style);
                  
                  if(el->getFlag(E_ROUNDIN|E_ROUNDOUT)) {
                    fprintf(fPointer, "#%.6f", radius);
                  }
                  fprintf(fPointer, "#%.6f#%.6f",
                                    el->getX2(), el->getCx()); // Letterdist. / Worddist
                  fprintf(fPointer, "\n");
  
                  strEncodeDxfString(textLine);
                  fprintf(fPointer, "  1\n%s\n",
                                    (const char*)textLine.local8Bit());
                  fprintf(fPointer, " 39\n%d\n",
                                    el->getWidth());
                  fprintf(fPointer, " 62\n%d\n",
                                    graphic->colorToNumber(el->getColor(), true));
                }
  
                // Go to pos of next line:
                //
                if(el->getFlag(E_ROUNDIN)) {
                  radius-=el->getCy();
                }
                else {
                  if(el->getFlag(E_ROUNDOUT)) {
                    radius+=el->getCy();
                  }
                  else {
                    xp+=sin(el->getA1()/ARAD) * el->getCy();
                    yp-=cos(el->getA1()/ARAD) * el->getCy();
                  }
                }
              }
            }
            fflush( fPointer );
            break;

          // Dimension:
          //
          case T_DIMENSION:
            fprintf(fPointer, "  0\nDIMENSION\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            {QString dText=el->getText();
            if(!dText.isEmpty()) {
              strEncodeDxfString(dText);
              fprintf(fPointer, "  1\n%s\n", (const char*)dText.local8Bit());}
            }

            if(el->getFlag(E_STRAIGHT)) {
              if(el->getFlag(E_HORIZONTAL)) {
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getX1());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getDimDist());
              }
              else if(el->getFlag(E_VERTICAL)) {
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getDimDist());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getY1());
              }
              else if(el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 10\n%f\n",    // dim center x
                                  el->getCx());
                fprintf(fPointer, " 20\n%f\n",    // dim center y
                                  el->getCy());
                fprintf(fPointer, " 15\n%f\n",    // dim center x
                                  el->getX1());
                fprintf(fPointer, " 25\n%f\n",    // dim center y
                                  el->getY1());
                fprintf(fPointer, " 40\n%f\n",    // dim radius
                                  el->getCr());
              }
              else {
                double ang = mtGetAngle(el->getX1(), el->getY1(), el->getX2(), el->getY2());
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getX1() + cos((ang+90.0)/ARAD) * el->getDimDist());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getY1() + sin((ang+90.0)/ARAD) * el->getDimDist());
              }

              if(!el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 13\n%f\n",
                                  el->getX1());
                fprintf(fPointer, " 23\n%f\n",
                                  el->getY1());
                fprintf(fPointer, " 14\n%f\n",
                                  el->getX2());
                fprintf(fPointer, " 24\n%f\n",
                                  el->getY2());
              }

              if(el->getFlag(E_HORIZONTAL)) {
                fprintf(fPointer, " 50\n%f\n", 0.0);
                fprintf(fPointer, " 70\n%d\n", 0);
              }
              else if(el->getFlag(E_VERTICAL)) {
                fprintf(fPointer, " 50\n%f\n", 90.0);
                fprintf(fPointer, " 70\n%d\n", 0);
              }
              else if(el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 70\n%d\n", 4);
              }
              else if(el->getFlag(E_ARROW)) {
                fprintf(fPointer, " 70\n%d\n", 7);   // Not DXF compatible!
              }
              else {
                fprintf(fPointer, " 70\n%d\n", 1);
              }
            }
            else if(el->getFlag(E_ROUNDOUT)) {
              // 1st extension line
              fprintf(fPointer, " 13\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getX1());
              fprintf(fPointer, " 23\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getX1());
              fprintf(fPointer, " 14\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 24\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getDimDist());

              // 2nd extension line
              fprintf(fPointer, " 10\n%f\n",
                                el->getCx() + cos(el->getA2()/ARAD) * el->getX2());
              fprintf(fPointer, " 20\n%f\n",
                                el->getCy() + sin(el->getA2()/ARAD) * el->getX2());
              fprintf(fPointer, " 15\n%f\n",
                                el->getCx() + cos(el->getA2()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 25\n%f\n",
                                el->getCy() + sin(el->getA2()/ARAD) * el->getDimDist());

              // Def line:
              fprintf(fPointer, " 16\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 26\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getDimDist());

              fprintf(fPointer, " 70\n%d\n",
                                2);
            }
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;

          // Hatching:
          //
          case T_HATCH:
            fprintf(fPointer, "  0\nHATCH\n");
            fprintf(fPointer, "  8\n%s\n", (const char*)graphic->getLayerName(el->getLayer()).local8Bit());
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, "  1\n \n");
            fprintf(fPointer, "  2\n%s\n",
                              el->getText().latin1());
            fprintf(fPointer, " 70\n%d\n",
                              0);
            fprintf(fPointer, " 91\n%d\n",
                              1);

            // Save boundary:
            if(true) {
              for(i=0; i<el->getSubElementNum(); ++i) {
                if(!el->getSubElement(i)->getFlag(E_VISIBLE)) {
                  switch(el->getSubElement(i)->getElementTyp()) {
                    case T_LINE:
                      fprintf(fPointer, " 72\n%d\n",
                                        1);
                      fprintf(fPointer, " 10\n%f\n",
                                        el->getSubElement(i)->getX1());
                      fprintf(fPointer, " 20\n%f\n",
                                        el->getSubElement(i)->getY1());
                      fprintf(fPointer, " 11\n%f\n",
                                        el->getSubElement(i)->getX2());
                      fprintf(fPointer, " 21\n%f\n",
                                        el->getSubElement(i)->getY2());
                      break;
  
                    case T_ARC:
                    case T_CIRCLE:
                      fprintf(fPointer, " 72\n%d\n",
                                        2);
                      fprintf(fPointer, " 10\n%f\n",
                                        el->getSubElement(i)->getCx());
                      fprintf(fPointer, " 20\n%f\n",
                                        el->getSubElement(i)->getCy());
                      fprintf(fPointer, " 40\n%f\n",
                                        el->getSubElement(i)->getCr());
                      fprintf(fPointer, " 50\n%f\n",
                                        el->getSubElement(i)->getA1());
                      fprintf(fPointer, " 51\n%f\n",
                                        el->getSubElement(i)->getA2());
                      fprintf(fPointer, " 73\n%d\n",
                                       el->getSubElement(i)->getFlag(E_REVERSED));
                      break;
                    default:
                      break;
                  }
                }
              }
            }

            fprintf(fPointer, " 75\n%d\n",
                              0);
            fprintf(fPointer, " 76\n%d\n",
                              1);
            fprintf(fPointer, " 52\n%f\n",
                              0.0);
            fprintf(fPointer, " 91\n%d\n",
                              1);
            fprintf(fPointer, " 41\n%f\n",
                              el->getCr());
            fprintf(fPointer, " 77\n%d\n",
                              0);
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor(), true));
            fflush( fPointer );
            break;
            
          default:
            break;
        }
      }
    }

    RLOG( "\nDXF saved" );

    // DXF-End:
    //
    fprintf(fPointer, "  0\nENDSEC\n  0\nEOF\n");
  
    fclose(fPointer);

    RLOG( "\nDXF closed" );

    statusPanel()->delProgress();

    RLOG( "\nstatusPanel()->delProgress() done" );

    ret=true;
  }
  else {
    ret=false;
  }

  RLOG( "\nReturn" );

  return ret;
  
}


// EOF



















































