/*
	Description: qgit revision list view

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qlineedit.h>
#include <qtextbrowser.h>
#include <qlistview.h>
#include <qpainter.h>
#include <qstringlist.h>
#include <qlistbox.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qheader.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qaction.h>
#include <qdragobject.h>
#include <qtoolbutton.h>
#include "common.h"
#include "git.h"
#include "domain.h"
#include "treeview.h"
#include "listview.h"
#include "filelist.h"
#include "revdesc.h"
#include "patchview.h"
#include "mainimpl.h"
#include "revsview.h"

RevsView::RevsView(MainImpl* mi, Git* g) : Domain(mi, g) {

	listViewLog = new ListView(this, git, m()->listViewLog, false);
	textBrowserDesc = new RevDesc(this, m()->textBrowserDesc);
	textBrowserDesc_2 = new RevDesc(this, m()->textBrowserDesc_2);
	listBoxFiles = new ListBoxFiles(this, git, m()->listBoxFiles);
	listBoxFiles_2 = new ListBoxFiles(this, git, m()->listBoxFiles_2);
	treeView = new TreeView(this, git, m()->treeView);

	connect(git, SIGNAL(newRevsAdded(const QValueVector<QString>&)),
		listViewLog, SLOT(on_newRevsAdded(const QValueVector<QString>&)));

	connect(git, SIGNAL(loadCompleted(const QString&)),
		this, SLOT(on_loadCompleted(const QString&)));

	connect(listViewLog, SIGNAL(lanesContextMenuRequested(const QStringList&,
		const QStringList&)), this, SLOT(on_lanesContextMenuRequested
		(const QStringList&, const QStringList&)));

	connect(listViewLog, SIGNAL(droppedRevisions(const QString&,
		const QStringList&, const QString&)), this, SLOT(on_droppedRevisions
		(const QString&, const QStringList&, const QString&)));

	connect(listViewLog, SIGNAL(contextMenu(const QString&, int)),
		d, SLOT(on_contextMenu(const QString&, int)));

	connect(treeView, SIGNAL(contextMenu(const QString&, int)),
		d, SLOT(on_contextMenu(const QString&, int)));

	connect(listBoxFiles, SIGNAL(contextMenu(const QString&, int)),
		d, SLOT(on_contextMenu(const QString&, int)));

	connect(listBoxFiles_2, SIGNAL(contextMenu(const QString&, int)),
		d, SLOT(on_contextMenu(const QString&, int)));
}

void RevsView::clear(bool keepState) {

	if (!keepState)
		st.clear();
	listViewLog->clear();
	textBrowserDesc->clear();
	textBrowserDesc_2->clear();
	listBoxFiles->clear();
	listBoxFiles_2->clear();
	treeView->clear();
	updateLineEditSHA(true);
	if (patchView)
		patchView->clear();
}

void RevsView::setPatchView(bool b) {

	if (b && patchView)
		return;

	if (b) {
		patchView = new PatchView(this, git, m()->textEditDiff,
			m()->lineEditDiff, m()->radioButtonSha);

		connect(listBoxFiles, SIGNAL(fileSelected(const QString&)),
			patchView, SLOT(on_fileSelected(const QString&)));

		connect(m(), SIGNAL(highlightPatch(const QString&, bool)),
			   patchView, SLOT(on_highlightPatch(const QString&, bool)));

		patchView->update(true);
		patchView->on_fileSelected(st.fileName());
	} else
		delete patchView;
}

void RevsView::on_loadCompleted(const QString& stats) {

	if (st.sha().isEmpty()) { // point to first one in list
		if (m()->listViewLog->firstChild()) {
			SCRef firstRev(m()->listViewLog->firstChild()->text(QGit::COMMIT_COL));
			st.setSha(firstRev);
			st.setSelectItem(true);
		}
	}
	UPDATE_DOMAIN(d);

	// queue message after UPDATE_DOMAIN()
	QApplication::postEvent(d, new MessageEvent(stats));
}

bool RevsView::doUpdate() {

	try {
		EM_REGISTER(exSetRepositoryCalled);
		EM_REGISTER(exExiting);

		bool dataCleared = m()->lineEditSHA->text().isEmpty();
		bool found = listViewLog->update();

		if (!found) { // views are updated only if sha is found

			const QString tmp("Sorry, revision " + st.sha() +
					" has not been found in main view");
			m()->statusBar()->message(tmp);

		} else {
			bool shaChanged = (st.sha(true) != st.sha(false));
			bool diffToChanged = (st.diffToSha(true) != st.diffToSha(false));
			bool allChanged = (st.allMergeFiles(true) != st.allMergeFiles(false));

			if (shaChanged || dataCleared) {
				updateLineEditSHA();
				SCRef d(git->getDesc(st.sha(), m()->shortLogRE, m()->longLogRE));
				textBrowserDesc->setText(d);
				textBrowserDesc_2->setText(d);
				m()->statusBar()->message(git->getRevInfo(st.sha(), false));
			}
			const RevFile* files = NULL;

			if (shaChanged || diffToChanged || allChanged || dataCleared)
				// could call processEvents()
				files = git->getFiles(st.sha(), st.diffToSha(), st.allMergeFiles());

			listBoxFiles->update(files);
			listBoxFiles_2->update(files);

			if (m()->treeView->isVisible())
				treeView->update(); // blocking call

			if (st.selectItem()) {
				bool isDir = treeView->isDir(st.fileName());
				m()->updateContextActions(st.sha(), st.fileName(), isDir, found);
			}
			// at the end update diffs that is the slowest and must be
			// run after update of file list for 'diff to sha' to work
			if (patchView)
				patchView->update(dataCleared);
		}
		if (!found && dataCleared && m()->listViewLog->currentItem()) {
			// we are in an inconsistent state: list view current item is
			// not selected and secondary panes are empty.
			// This could happen as example after removing a tree filter.
			// At least populate secondary panes
			SCRef firstRev(m()->listViewLog->currentItem()->text(QGit::COMMIT_COL));
			st.setSha(firstRev);
			st.setSelectItem(false);
			customEvent(new UpdateDomainEvent()); // will be queued immediately
		}
		EM_REMOVE(exExiting);
		EM_REMOVE(exSetRepositoryCalled);

		return found;

	} catch(int i) {

		EM_REMOVE(exExiting);
		EM_REMOVE(exSetRepositoryCalled);

		if (EM_MATCH(i, exSetRepositoryCalled, "update views")) {
			EM_CHECK_PENDING;
			return false;
		}
		if (EM_MATCH(i, exExiting, "update main view")) {
			EM_CHECK_PENDING;
			return false;
		}
		const QString info("Exception \'" + EM_DESC(i) + "\' "
				"not handled in main view update...re-throw");
		dbp("%1", info);
		throw i;
	}
}

void RevsView::updateLineEditSHA(bool clear) {

	QLineEdit* l = m()->lineEditSHA;

	if (clear)
		l->setText(""); // clears history

	else if (l->text() != st.sha()) {

		if (l->text().isEmpty())
			l->setText(st.sha()); // first rev clears history
		else {
			// setText() clears undo/redo history so
			// we use clear() + insert() instead
			l->clear();
			l->insert(st.sha());
		}
	}
	m()->ActBack->setEnabled(l->isUndoAvailable());
	m()->ActForward->setEnabled(l->isRedoAvailable());
}

void RevsView::on_lanesContextMenuRequested(SCList parents, SCList childs) {

	QPopupMenu contextMenu;
	uint i = 0;
	QStringList::const_iterator it(childs.constBegin());
	for ( ; it != childs.constEnd(); ++it, i++)
		contextMenu.insertItem("Child: " + git->getShortLog(*it), i);

	for (it = parents.constBegin() ; it != parents.constEnd(); ++it, i++) {

		QString log(git->getShortLog(*it));
		if (log.isEmpty())
			log = *it;

		contextMenu.insertItem("Parent: " + log, i);
	}
	int id = contextMenu.exec(QCursor::pos()); // modal exec
	if (id == -1)
		return;

	int cc = (int)childs.count();
	SCRef target((id < cc) ? childs[id] : parents[id - cc]);
	st.setSha(target);
	UPDATE_DOMAIN(d);
}

void RevsView::on_droppedRevisions(const QString& sha, const QStringList& remoteRevs,
							const QString& remoteRepo) {

	if (isDropping()) // avoid reentrancy
		return;

	QDir dr;
	if (!dr.exists(remoteRepo) || m()->curDir == remoteRepo)
		return;

	dr.setPath(m()->curDir + QGit::PATCHES_DIR);
	if (dr.exists()) {
		const QString tmp("Please remove stale import directory " + dr.absPath());
		m()->statusBar()->message(tmp);
		return;
	}
	// ok, let's go.
	dr.setFilter(QDir::Files);
	setDropping(true);
	bool ok = true;
	QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
	m()->raise();
	EM_PROCESS_EVENTS;

	for (uint i = 1; i <= remoteRevs.count(); i++) {

		SCRef sha(remoteRevs[remoteRevs.count() - i].section('@', 0, 0));
		const QStringList shaList(sha); // we create patches one by one

		const QString tmp(QString("Importing revision %1 of %2")
				.arg(i).arg(remoteRevs.count()));
		m()->statusBar()->message(tmp);

		if (!git->formatPatch(shaList, dr.absPath(), remoteRepo)) {
			ok = false;
			break;
		}
		dr.refresh();
		if (dr.count() != 1) {
			ok = false;
			break;
		}
		if (!git->applyPatchFile(dr.absFilePath(dr[0]), true)) {
			ok = false;
			break;
		}
		dr.remove(dr[0]);
	}
	if (!ok)
		m()->statusBar()->message("Failed to import revision " + sha);
	else
		m()->statusBar()->clear();

	dr.rmdir(dr.absPath());
	QApplication::restoreOverrideCursor();
	setDropping(false);
	m()->refreshRepo();
}
