/****************************************************************************
** $Id: qt/dirview.cpp   3.3.4   edited Oct 28 2003 $
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of an example program for Qt.  This example
** program may be used, distributed and modified without limitation.
**
** Heavily modified by Marco Costalba
*****************************************************************************/
#include <qapplication.h>
#include <qcursor.h>
#include <qpainter.h>
#include <qaction.h>
#include <qlabel.h>
#include "git.h"
#include "domain.h"
#include "mainimpl.h"
#include "treeview.h"

static QPixmap* folderClosed = NULL;
static QPixmap* folderOpen = NULL;
static QPixmap* fileDefault = NULL;

// ******************************* FileItem ****************************

void FileItem::setPixmap(QPixmap* p) {

	pix = p;
	setup();
	widthChanged(0);
	invalidateHeight();
	repaint();
}

const QPixmap* FileItem::pixmap(int i) const {

	if (i > 0)
		return 0;
	return pix;
}

QString FileItem::text(int column) const {

	if (column == 0)
		return name;
	return "";
}

QString FileItem::fullName() {

	QString s; // root directory has no name
	if (p) {
		s = p->fullName();
		if (!s.isEmpty())
			s.append("/");
		s.append(name);
	}
	return s;
}

void FileItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int wdt, int ali) {

	p->save();
	if (isModified) {
		QFont f(p->font());
		f.setBold(true);
		p->setFont(f);
	}
	QListViewItem::paintCell(p, cg, col, wdt, ali);
	p->restore();
}


// ******************************* DirItem ****************************

using namespace QGit;

DirItem::DirItem(DirItem* par, SCRef ts, SCRef nm, TreeView* t)
	: FileItem(par, nm), treeSha(ts), tv(t), isWorkingDir(par->isWorkingDir)
	{ setPixmap(folderClosed); }

DirItem::DirItem(QListView* par, SCRef ts, SCRef nm, TreeView* t)
	: FileItem(par, nm), treeSha(ts), tv(t), isWorkingDir(ts == ZERO_SHA) {}

void DirItem::setup() {

	setExpandable(true);
	QListViewItem::setup();
}

void TreeView::getTree(SCRef treeSha, SList names,
	SList shas, SList types, bool wd, SCRef treePath) {

	git->getTree(treeSha, names, shas, types, wd, treePath); // calls processEvents()
}

void DirItem::setOpen(bool o) {

	if (o)
		setPixmap(folderOpen);
	else
		setPixmap(folderClosed);

	bool alreadyWaiting = false;
	if (QApplication::overrideCursor())
		alreadyWaiting = (QApplication::overrideCursor()->shape() == Qt::WaitCursor);

	if (!alreadyWaiting)
		QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

	if (o && !childCount()) {

		QStringList names, types, shas;
		tv->getTree(treeSha, names, shas, types, isWorkingDir, fullName());

		if (!names.empty()) {
			QStringList::const_iterator it(names.constBegin());
			QStringList::const_iterator itSha(shas.constBegin());
			QStringList::const_iterator itTypes(types.constBegin());
			while (it != names.constEnd()) {

				if (*itTypes == "tree") {
					DirItem* item = new DirItem(this, *itSha, *it, tv);
					item->setModified(tv->isModified(item->fullName(), true));
				} else {
					FileItem* item = new FileItem(this, *it);
					item->setPixmap(tv->mimePix(*it));
					item->setModified(tv->isModified(item->fullName()));
				}
				++it;
				++itSha;
				++itTypes;
			}
		}
	}
	QListViewItem::setOpen(o);
	if (!alreadyWaiting)
		QApplication::restoreOverrideCursor();
}

// ******************************* TreeView ****************************

TreeView::TreeView(Domain* dm, Git* g, QListView* lv) :
		QObject(dm), d(dm), git(g), listView(lv) {

	EM_INIT(exTreeCleared, "Resetting tree view");

	st = &(d->st);
	oldRoot = NULL;
	ignoreCurrentChanged = false;

	initMimePix();

	connect(listView, SIGNAL(currentChanged(QListViewItem*)),
		   this, SLOT(slotCurrentChanged(QListViewItem*)));
	connect(listView, SIGNAL(contextMenuRequested(QListViewItem*,const QPoint&,int)),
		   this, SLOT(contextMenuRequested(QListViewItem*,const QPoint&,int)));
}

void TreeView::initMimePix() {

	MainImpl* m = const_cast<MainImpl*>(d->m());

	setMimePix("#FOLDER_CLOSED", m->lblFolder->pixmap());
	setMimePix("#FOLDER_OPEN", m->lblFolderOpen->pixmap());
	setMimePix("#DEFAULT", m->lblFile->pixmap());
	setMimePix("c", m->lblC->pixmap());
	setMimePix("cpp", m->lblCpp->pixmap());
	setMimePix("h", m->lblH->pixmap());
	setMimePix("txt", m->lblTxt->pixmap());
	setMimePix("sh", m->lblSh->pixmap());
	setMimePix("perl", m->lblPl->pixmap());
	setMimePix("pl", m->lblPl->pixmap());
	setMimePix("py", m->lblPy->pixmap());

	m->lblFolder->hide();
	m->lblFolderOpen->hide();
	m->lblFile->hide();
	m->lblC->hide();
	m->lblCpp->hide();
	m->lblH->hide();
	m->lblTxt->hide();
	m->lblSh->hide();
	m->lblPl->hide();
	m->lblPy->hide();
}

void TreeView::slotCurrentChanged(QListViewItem* item) {

	if (item) {
		SCRef fn = ((FileItem*)item)->fullName();
		if (!ignoreCurrentChanged && fn != st->fileName()) {
			st->setFileName(fn);
			st->setSelectItem(true);
			UPDATE_DOMAIN(d);
		}
	}
}

void TreeView::contextMenuRequested(QListViewItem* item, const QPoint&,int) {

	if (!item)
		return;

	emit contextMenu(fullName(item), POPUP_TREE_EV);
}

void TreeView::clear() {

	EM_RAISE(exTreeCleared);

	oldRoot = NULL;
	rootName = "";
	listView->clear();
}

bool TreeView::isModified(SCRef path, bool isDir) {

	if (isDir)
		return (modifiedDirs.findIndex(path) != -1);

	return (modifiedFiles.findIndex(path) != -1);
}

bool TreeView::isDir(SCRef fileName) {

	// if currentItem is NULL or is different from fileName
	// return false, because treeview is not updated while
	// not visible, so could be out of sync.
	if (listView->currentItem() == NULL)
		return false;

	FileItem* item = (FileItem*)listView->currentItem();
	if (item->fullName() != fileName)
		return false;

	DirItem* d = dynamic_cast<DirItem*>(listView->currentItem());
	return (d != NULL);
}

const QString TreeView::fullName(QListViewItem* item) {

	if (item == NULL)
		return QString();

	FileItem* f = static_cast<FileItem*>(item);
	return f->fullName();
}

void TreeView::getTreeSelectedItems(QStringList& selectedItems) {

	selectedItems.clear();
	QListViewItemIterator it(listView);
	while (it.current()) {
		QListViewItem* item = it.current();
		if (item->isSelected())
			selectedItems.append(((FileItem*)item)->fullName());
		++it;
	}
}

void TreeView::setMimePix(SCRef ext, QPixmap* pix) {

	// common cases first
	if (ext == "#FOLDER_CLOSED")
		folderClosed = pix;
	if (ext == "#FOLDER_OPEN")
		folderOpen = pix;
	if (ext == "#DEFAULT")
		fileDefault = pix;

	//added extensions
	if (mimePixMap.find(ext) != 0)
		return;
	mimePixMap.insert(ext, pix);
}

QPixmap* TreeView::mimePix(SCRef fileName) {

	SCRef ext = fileName.section('.', -1, -1);
	if (ext.isEmpty())
		return fileDefault;

	QPixmap* pix = mimePixMap.find(ext);
	if (pix)
		return pix;

	return fileDefault;
}

void TreeView::setTreeName(SCRef treeName) {

	rootName = treeName;
}

void TreeView::setTree(SCRef treeSha) {

	oldRoot = listView->firstChild();
	if (!oldRoot) {
		git->getWorkDirFiles(MODIFIED, modifiedFiles, modifiedDirs);
		QStringList f, d;
		git->getWorkDirFiles(DELETED, f, d);
		modifiedFiles += f;
		modifiedDirs += d;
		git->getWorkDirFiles(UNKNOWN, f, d);
		modifiedFiles += f;
		modifiedDirs += d;
	}
	if (!treeSha.isEmpty()) {
		// insert a new dir at the beginning of the list
		DirItem* root = new DirItem(listView, treeSha, rootName, this);
		root->setOpen(true); // be interesting
	}
}

void TreeView::update() {

	if (st->sha().isEmpty())
		return;

	// qt emits currentChanged() signal when populating the
	// list view, so we should ignore while here.
	ignoreCurrentChanged = true;

	try {
		EM_REGISTER(exTreeCleared);

		QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

		bool newTree = true;
		DirItem* root = (DirItem*)(listView->firstChild());
		if (root)
			newTree = (root->treeSha != st->sha());

		if (newTree && root && st->sha() != ZERO_SHA &&
			root->treeSha != ZERO_SHA)
			// root->treeSha could reference a different sha from current
			// one in case the tree is the same, i.e. has the same files.
			// so we must use the state sha to call isSameFiles() and
			// benefit from the early skip logic
			newTree = !git->isSameFiles(st->sha(false), st->sha());

		if (newTree)
			setTree(st->sha());

		if (st->fileName().isEmpty()) {
			if (newTree && oldRoot) {
				while (oldRoot->itemBelow())
					delete oldRoot->itemBelow();

				delete oldRoot;
				oldRoot = NULL;
			}
			ignoreCurrentChanged = false;
			QApplication::restoreOverrideCursor();
			EM_REMOVE(exTreeCleared);
			return;
		}
		FileItem* cur = (FileItem*)listView->currentItem();
		if (cur && !newTree) {
			if (cur->fullName() == st->fileName()) {
				ignoreCurrentChanged = false;
				QApplication::restoreOverrideCursor();
				EM_REMOVE(exTreeCleared);
				return;
			}
		}
		listView->setUpdatesEnabled(false);
		QStringList lst(QStringList::split("/", st->fileName()));
		QListViewItem* item(listView->firstChild());
		item = item->itemBelow(); // first item is repository name
		loopList(it2, lst) {
			while (item && item != oldRoot) {
				if (item->text(0) == *it2) {
					// could be a different subdirectory with the
					// same name that appears before in tree view
					// to be sure we need to check the names
					if (st->fileName().startsWith(((FileItem*)item)->fullName())) {
						item->setOpen(true);
						break;
					}
				}
				item = item->itemBelow();
			}
		}
		if (item == oldRoot)
			item = NULL;

		if (oldRoot != NULL) {
			while (oldRoot->itemBelow())
				delete oldRoot->itemBelow();

			delete oldRoot;
			oldRoot = NULL;
		}
		if (item) {
			listView->clearSelection();
			listView->setSelected(item, true);
			listView->setCurrentItem(item); // calls slotCurrentChanged()
			listView->ensureItemVisible(item);
		}
		listView->setUpdatesEnabled(true);
		listView->triggerUpdate();
		ignoreCurrentChanged = false;
		QApplication::restoreOverrideCursor();
		EM_REMOVE(exTreeCleared);

	} catch (int i) {
		EM_REMOVE(exTreeCleared);

		if (EM_MATCH(i, exTreeCleared, "updating tree")) {

			QApplication::restoreOverrideCursor();
			listView->setUpdatesEnabled(true);
			ignoreCurrentChanged = false;
			EM_CHECK_PENDING;
			return;
		}
		const QString info("Exception \'" + EM_DESC(i) + "\' "
				"not handled in tree view...re-throw");
		dbp("%1", info);
		throw i;
	}
}
