#!/usr/bin/env python
# File created on 13 Jul 2012
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso"]
__license__ = "GPL"
__version__ = "1.7.0"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"
__status__ = "Release"

from shutil import rmtree
from glob import glob
from os.path import exists, join
from cogent.util.unit_test import TestCase, main
from cogent.util.misc import remove_files, create_dir
from qiime.parallel.alpha_diversity import ParallelAlphaDiversity
from qiime.util import (get_qiime_temp_dir, 
                        get_tmp_filename,
                        count_seqs_in_filepaths)
from qiime.test import initiate_timeout, disable_timeout

class ParallelAlphaDiversityTests(TestCase):
    
    def setUp(self):
        """ """
        self.files_to_remove = []
        self.dirs_to_remove = []
        
        tmp_dir = get_qiime_temp_dir()
        self.test_out = get_tmp_filename(tmp_dir=tmp_dir,
                                         prefix='qiime_parallel_tests_',
                                         suffix='',
                                         result_constructor=str)
        self.dirs_to_remove.append(self.test_out)
        create_dir(self.test_out)
        
        self.rt_fps = []
        for rt in [rt1, rt2, rt3, rt4]:
            rt_fp = get_tmp_filename(tmp_dir=self.test_out,
                                     prefix='qiime_rt',
                                     suffix='.biom')
            rt_f = open(rt_fp,'w')
            rt_f.write(rt)
            rt_f.close()
            self.rt_fps.append(rt_fp)
            self.files_to_remove.append(rt_fp)
        
        self.tree_fp = get_tmp_filename(tmp_dir=self.test_out,
                                     prefix='qiime',
                                     suffix='.tre')
        tree_f = open(self.tree_fp,'w')
        tree_f.write(tree)
        tree_f.close()
        self.files_to_remove.append(self.tree_fp)
        
        initiate_timeout(60)
    
    def tearDown(self):
        """ """
        disable_timeout()
        remove_files(self.files_to_remove)
        # remove directories last, so we don't get errors
        # trying to remove files which may be in the directories
        for d in self.dirs_to_remove:
            if exists(d):
                rmtree(d)

    def test_parallel_alpha_diversity(self):
        """parallel alpha diversity functions as expected with basic input
        """
        params = {'metrics':'observed_species,chao1,PD_whole_tree',
                  'tree_path':self.tree_fp,
                  'jobs_to_start':2
                  }
        app = ParallelAlphaDiversity()
        r = app(self.rt_fps,
                self.test_out,
                params,
                job_prefix='ATEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        # confirm that the total number of output sequences equals the total
        # number of input sequences
        output_fps = glob(join(self.test_out,'*txt'))
        self.assertEqual(len(output_fps),len(self.rt_fps))
        
    def test_parallel_alpha_diversity_wo_tree(self):
        """parallel alpha diversity functions as expected without tree
        """
        params = {'metrics':'observed_species,chao1',
                  'tree_path':None,
                  'jobs_to_start':2
                  }
        app = ParallelAlphaDiversity()
        r = app(self.rt_fps,
                self.test_out,
                params,
                job_prefix='ATEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        # confirm that the total number of output sequences equals the total
        # number of input sequences
        output_fps = glob(join(self.test_out,'*txt'))
        self.assertEqual(len(output_fps),len(self.rt_fps))


rt1 = """{"rows": [{"id": "0", "metadata": {}}, {"id": "1", "metadata": {}}, {"id": "2", "metadata": {}}, {"id": "3", "metadata": {}}, {"id": "4", "metadata": {}}, {"id": "5", "metadata": {}}, {"id": "6", "metadata": {}}, {"id": "8", "metadata": {}}, {"id": "9", "metadata": {}}, {"id": "10", "metadata": {}}, {"id": "11", "metadata": {}}, {"id": "12", "metadata": {}}, {"id": "13", "metadata": {}}, {"id": "14", "metadata": {}}, {"id": "16", "metadata": {}}, {"id": "17", "metadata": {}}, {"id": "23", "metadata": {}}, {"id": "24", "metadata": {}}, {"id": "25", "metadata": {}}, {"id": "26", "metadata": {}}, {"id": "27", "metadata": {}}, {"id": "29", "metadata": {}}, {"id": "30", "metadata": {}}, {"id": "31", "metadata": {}}, {"id": "32", "metadata": {}}, {"id": "33", "metadata": {}}, {"id": "34", "metadata": {}}, {"id": "35", "metadata": {}}, {"id": "37", "metadata": {}}, {"id": "38", "metadata": {}}, {"id": "39", "metadata": {}}, {"id": "41", "metadata": {}}, {"id": "42", "metadata": {}}, {"id": "43", "metadata": {}}, {"id": "44", "metadata": {}}, {"id": "45", "metadata": {}}, {"id": "46", "metadata": {}}, {"id": "48", "metadata": {}}, {"id": "49", "metadata": {}}, {"id": "51", "metadata": {}}, {"id": "52", "metadata": {}}, {"id": "53", "metadata": {}}, {"id": "54", "metadata": {}}, {"id": "56", "metadata": {}}, {"id": "57", "metadata": {}}, {"id": "58", "metadata": {}}, {"id": "59", "metadata": {}}, {"id": "60", "metadata": {}}, {"id": "62", "metadata": {}}, {"id": "63", "metadata": {}}, {"id": "64", "metadata": {}}, {"id": "65", "metadata": {}}, {"id": "66", "metadata": {}}, {"id": "70", "metadata": {}}, {"id": "71", "metadata": {}}, {"id": "72", "metadata": {}}, {"id": "73", "metadata": {}}, {"id": "74", "metadata": {}}, {"id": "75", "metadata": {}}, {"id": "76", "metadata": {}}, {"id": "77", "metadata": {}}, {"id": "78", "metadata": {}}, {"id": "81", "metadata": {}}, {"id": "82", "metadata": {}}, {"id": "83", "metadata": {}}, {"id": "84", "metadata": {}}, {"id": "86", "metadata": {}}, {"id": "87", "metadata": {}}, {"id": "89", "metadata": {}}, {"id": "90", "metadata": {}}, {"id": "93", "metadata": {}}, {"id": "94", "metadata": {}}, {"id": "95", "metadata": {}}, {"id": "96", "metadata": {}}, {"id": "98", "metadata": {}}, {"id": "99", "metadata": {}}, {"id": "101", "metadata": {}}, {"id": "104", "metadata": {}}, {"id": "107", "metadata": {}}, {"id": "108", "metadata": {}}, {"id": "109", "metadata": {}}, {"id": "110", "metadata": {}}, {"id": "111", "metadata": {}}, {"id": "112", "metadata": {}}, {"id": "113", "metadata": {}}, {"id": "116", "metadata": {}}, {"id": "118", "metadata": {}}, {"id": "119", "metadata": {}}, {"id": "120", "metadata": {}}, {"id": "121", "metadata": {}}, {"id": "123", "metadata": {}}, {"id": "124", "metadata": {}}, {"id": "125", "metadata": {}}, {"id": "126", "metadata": {}}, {"id": "128", "metadata": {}}, {"id": "129", "metadata": {}}, {"id": "130", "metadata": {}}, {"id": "132", "metadata": {}}, {"id": "133", "metadata": {}}, {"id": "134", "metadata": {}}, {"id": "135", "metadata": {}}, {"id": "136", "metadata": {}}, {"id": "137", "metadata": {}}, {"id": "140", "metadata": {}}, {"id": "141", "metadata": {}}, {"id": "142", "metadata": {}}, {"id": "143", "metadata": {}}, {"id": "144", "metadata": {}}, {"id": "145", "metadata": {}}, {"id": "146", "metadata": {}}, {"id": "148", "metadata": {}}, {"id": "149", "metadata": {}}, {"id": "150", "metadata": {}}, {"id": "151", "metadata": {}}, {"id": "152", "metadata": {}}, {"id": "153", "metadata": {}}, {"id": "156", "metadata": {}}, {"id": "160", "metadata": {}}, {"id": "161", "metadata": {}}, {"id": "163", "metadata": {}}, {"id": "164", "metadata": {}}, {"id": "165", "metadata": {}}, {"id": "166", "metadata": {}}, {"id": "168", "metadata": {}}, {"id": "169", "metadata": {}}, {"id": "170", "metadata": {}}, {"id": "171", "metadata": {}}, {"id": "174", "metadata": {}}, {"id": "175", "metadata": {}}, {"id": "176", "metadata": {}}, {"id": "177", "metadata": {}}, {"id": "181", "metadata": {}}, {"id": "182", "metadata": {}}, {"id": "183", "metadata": {}}, {"id": "185", "metadata": {}}, {"id": "186", "metadata": {}}, {"id": "188", "metadata": {}}, {"id": "191", "metadata": {}}, {"id": "192", "metadata": {}}, {"id": "193", "metadata": {}}, {"id": "194", "metadata": {}}, {"id": "195", "metadata": {}}, {"id": "196", "metadata": {}}, {"id": "197", "metadata": {}}, {"id": "198", "metadata": {}}, {"id": "199", "metadata": {}}, {"id": "201", "metadata": {}}, {"id": "202", "metadata": {}}, {"id": "203", "metadata": {}}, {"id": "204", "metadata": {}}, {"id": "207", "metadata": {}}, {"id": "208", "metadata": {}}, {"id": "210", "metadata": {}}, {"id": "211", "metadata": {}}, {"id": "212", "metadata": {}}, {"id": "213", "metadata": {}}, {"id": "214", "metadata": {}}, {"id": "216", "metadata": {}}, {"id": "217", "metadata": {}}, {"id": "218", "metadata": {}}, {"id": "219", "metadata": {}}, {"id": "221", "metadata": {}}, {"id": "222", "metadata": {}}, {"id": "223", "metadata": {}}, {"id": "224", "metadata": {}}, {"id": "225", "metadata": {}}, {"id": "226", "metadata": {}}, {"id": "227", "metadata": {}}, {"id": "228", "metadata": {}}, {"id": "229", "metadata": {}}, {"id": "231", "metadata": {}}, {"id": "232", "metadata": {}}, {"id": "233", "metadata": {}}, {"id": "234", "metadata": {}}, {"id": "235", "metadata": {}}, {"id": "236", "metadata": {}}, {"id": "237", "metadata": {}}, {"id": "238", "metadata": {}}, {"id": "239", "metadata": {}}, {"id": "240", "metadata": {}}, {"id": "241", "metadata": {}}, {"id": "242", "metadata": {}}, {"id": "244", "metadata": {}}, {"id": "246", "metadata": {}}, {"id": "247", "metadata": {}}, {"id": "248", "metadata": {}}, {"id": "250", "metadata": {}}, {"id": "251", "metadata": {}}, {"id": "252", "metadata": {}}, {"id": "253", "metadata": {}}, {"id": "254", "metadata": {}}, {"id": "255", "metadata": {}}, {"id": "256", "metadata": {}}, {"id": "257", "metadata": {}}, {"id": "261", "metadata": {}}, {"id": "262", "metadata": {}}, {"id": "265", "metadata": {}}, {"id": "266", "metadata": {}}, {"id": "267", "metadata": {}}, {"id": "268", "metadata": {}}, {"id": "269", "metadata": {}}, {"id": "270", "metadata": {}}, {"id": "271", "metadata": {}}, {"id": "272", "metadata": {}}, {"id": "273", "metadata": {}}, {"id": "274", "metadata": {}}, {"id": "275", "metadata": {}}, {"id": "276", "metadata": {}}, {"id": "277", "metadata": {}}, {"id": "278", "metadata": {}}, {"id": "279", "metadata": {}}, {"id": "280", "metadata": {}}, {"id": "281", "metadata": {}}, {"id": "282", "metadata": {}}, {"id": "283", "metadata": {}}, {"id": "284", "metadata": {}}, {"id": "285", "metadata": {}}, {"id": "286", "metadata": {}}, {"id": "287", "metadata": {}}, {"id": "288", "metadata": {}}, {"id": "289", "metadata": {}}, {"id": "291", "metadata": {}}, {"id": "292", "metadata": {}}, {"id": "293", "metadata": {}}, {"id": "295", "metadata": {}}, {"id": "296", "metadata": {}}, {"id": "297", "metadata": {}}, {"id": "298", "metadata": {}}, {"id": "300", "metadata": {}}, {"id": "301", "metadata": {}}, {"id": "303", "metadata": {}}, {"id": "304", "metadata": {}}, {"id": "306", "metadata": {}}, {"id": "307", "metadata": {}}, {"id": "310", "metadata": {}}, {"id": "311", "metadata": {}}, {"id": "312", "metadata": {}}, {"id": "313", "metadata": {}}, {"id": "314", "metadata": {}}, {"id": "315", "metadata": {}}, {"id": "316", "metadata": {}}, {"id": "317", "metadata": {}}, {"id": "318", "metadata": {}}, {"id": "320", "metadata": {}}, {"id": "322", "metadata": {}}, {"id": "323", "metadata": {}}, {"id": "324", "metadata": {}}, {"id": "325", "metadata": {}}, {"id": "326", "metadata": {}}, {"id": "327", "metadata": {}}, {"id": "328", "metadata": {}}, {"id": "329", "metadata": {}}, {"id": "331", "metadata": {}}, {"id": "334", "metadata": {}}, {"id": "335", "metadata": {}}, {"id": "336", "metadata": {}}, {"id": "338", "metadata": {}}, {"id": "339", "metadata": {}}, {"id": "340", "metadata": {}}, {"id": "341", "metadata": {}}, {"id": "342", "metadata": {}}, {"id": "343", "metadata": {}}, {"id": "345", "metadata": {}}, {"id": "347", "metadata": {}}, {"id": "348", "metadata": {}}, {"id": "349", "metadata": {}}, {"id": "350", "metadata": {}}, {"id": "352", "metadata": {}}, {"id": "354", "metadata": {}}, {"id": "355", "metadata": {}}, {"id": "357", "metadata": {}}, {"id": "358", "metadata": {}}, {"id": "359", "metadata": {}}, {"id": "360", "metadata": {}}, {"id": "362", "metadata": {}}, {"id": "363", "metadata": {}}, {"id": "365", "metadata": {}}, {"id": "366", "metadata": {}}, {"id": "367", "metadata": {}}, {"id": "368", "metadata": {}}, {"id": "369", "metadata": {}}, {"id": "370", "metadata": {}}, {"id": "371", "metadata": {}}, {"id": "372", "metadata": {}}, {"id": "373", "metadata": {}}, {"id": "375", "metadata": {}}, {"id": "376", "metadata": {}}, {"id": "378", "metadata": {}}, {"id": "379", "metadata": {}}, {"id": "380", "metadata": {}}, {"id": "381", "metadata": {}}, {"id": "383", "metadata": {}}, {"id": "385", "metadata": {}}, {"id": "386", "metadata": {}}, {"id": "388", "metadata": {}}, {"id": "391", "metadata": {}}, {"id": "392", "metadata": {}}, {"id": "393", "metadata": {}}, {"id": "394", "metadata": {}}, {"id": "396", "metadata": {}}, {"id": "397", "metadata": {}}, {"id": "399", "metadata": {}}, {"id": "400", "metadata": {}}, {"id": "401", "metadata": {}}, {"id": "403", "metadata": {}}, {"id": "404", "metadata": {}}, {"id": "406", "metadata": {}}, {"id": "408", "metadata": {}}, {"id": "409", "metadata": {}}, {"id": "410", "metadata": {}}, {"id": "411", "metadata": {}}, {"id": "412", "metadata": {}}, {"id": "413", "metadata": {}}, {"id": "414", "metadata": {}}, {"id": "415", "metadata": {}}, {"id": "416", "metadata": {}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 1, 1.0], [2, 0, 1.0], [3, 2, 1.0], [4, 3, 1.0], [5, 0, 1.0], [6, 4, 1.0], [7, 0, 1.0], [7, 1, 1.0], [7, 2, 1.0], [8, 5, 1.0], [9, 3, 1.0], [10, 3, 1.0], [11, 6, 1.0], [12, 4, 2.0], [12, 6, 1.0], [13, 1, 1.0], [14, 1, 1.0], [15, 8, 1.0], [16, 7, 1.0], [17, 1, 2.0], [18, 3, 1.0], [19, 1, 1.0], [20, 7, 1.0], [21, 8, 1.0], [22, 1, 1.0], [23, 8, 1.0], [24, 3, 1.0], [25, 8, 1.0], [26, 2, 1.0], [27, 1, 1.0], [28, 3, 2.0], [29, 7, 1.0], [30, 7, 1.0], [31, 3, 1.0], [31, 7, 1.0], [32, 3, 1.0], [32, 4, 1.0], [33, 7, 1.0], [34, 5, 1.0], [35, 4, 12.0], [35, 5, 6.0], [36, 0, 6.0], [36, 1, 2.0], [36, 7, 2.0], [36, 8, 5.0], [37, 7, 1.0], [38, 5, 1.0], [39, 5, 1.0], [40, 3, 1.0], [41, 3, 2.0], [41, 6, 2.0], [42, 0, 25.0], [42, 1, 4.0], [42, 8, 3.0], [43, 0, 5.0], [43, 1, 4.0], [43, 3, 2.0], [43, 4, 1.0], [43, 5, 1.0], [43, 6, 3.0], [43, 7, 9.0], [43, 8, 2.0], [44, 5, 1.0], [45, 0, 1.0], [46, 0, 1.0], [46, 1, 1.0], [46, 2, 7.0], [46, 3, 1.0], [46, 4, 1.0], [46, 5, 16.0], [46, 6, 1.0], [47, 3, 1.0], [48, 7, 1.0], [49, 2, 1.0], [50, 6, 1.0], [51, 7, 1.0], [52, 1, 1.0], [53, 7, 1.0], [54, 6, 1.0], [55, 0, 1.0], [55, 1, 1.0], [55, 8, 2.0], [56, 8, 1.0], [57, 0, 1.0], [57, 1, 2.0], [57, 5, 1.0], [57, 7, 1.0], [57, 8, 1.0], [58, 7, 4.0], [59, 4, 1.0], [60, 0, 1.0], [60, 1, 2.0], [61, 1, 1.0], [62, 2, 1.0], [63, 0, 13.0], [63, 1, 1.0], [63, 8, 1.0], [64, 7, 1.0], [65, 2, 1.0], [66, 3, 1.0], [67, 3, 1.0], [68, 1, 1.0], [69, 0, 1.0], [69, 1, 2.0], [69, 2, 3.0], [69, 6, 2.0], [69, 7, 2.0], [70, 2, 1.0], [70, 3, 5.0], [70, 6, 2.0], [71, 2, 1.0], [72, 4, 1.0], [73, 0, 1.0], [74, 7, 1.0], [75, 1, 1.0], [75, 3, 1.0], [75, 8, 3.0], [76, 7, 1.0], [77, 3, 1.0], [77, 5, 2.0], [78, 6, 1.0], [79, 8, 2.0], [80, 8, 1.0], [81, 4, 1.0], [82, 7, 1.0], [83, 4, 1.0], [84, 5, 1.0], [85, 2, 1.0], [85, 3, 1.0], [86, 3, 1.0], [87, 2, 1.0], [88, 1, 1.0], [88, 8, 2.0], [89, 0, 1.0], [90, 2, 1.0], [90, 4, 1.0], [90, 6, 3.0], [91, 0, 1.0], [91, 6, 1.0], [92, 4, 1.0], [93, 1, 1.0], [93, 2, 1.0], [94, 4, 1.0], [95, 2, 1.0], [95, 4, 2.0], [95, 6, 2.0], [96, 5, 1.0], [97, 5, 1.0], [98, 4, 1.0], [99, 1, 1.0], [100, 4, 1.0], [101, 1, 1.0], [102, 8, 1.0], [103, 7, 1.0], [104, 3, 1.0], [105, 1, 1.0], [106, 7, 1.0], [107, 2, 1.0], [108, 1, 1.0], [109, 1, 1.0], [109, 3, 5.0], [110, 1, 2.0], [110, 4, 3.0], [110, 6, 8.0], [111, 0, 1.0], [112, 1, 1.0], [113, 1, 4.0], [113, 8, 1.0], [114, 2, 1.0], [115, 5, 1.0], [116, 3, 1.0], [117, 7, 1.0], [118, 6, 2.0], [119, 0, 1.0], [119, 1, 1.0], [120, 0, 1.0], [120, 1, 1.0], [121, 2, 7.0], [121, 3, 1.0], [121, 4, 9.0], [121, 6, 1.0], [122, 2, 3.0], [122, 4, 5.0], [122, 5, 2.0], [123, 5, 1.0], [124, 2, 6.0], [124, 4, 2.0], [124, 6, 2.0], [125, 5, 1.0], [126, 7, 1.0], [127, 5, 1.0], [128, 0, 1.0], [128, 8, 1.0], [129, 8, 1.0], [130, 5, 7.0], [131, 5, 6.0], [131, 7, 1.0], [132, 0, 1.0], [132, 5, 1.0], [133, 3, 1.0], [134, 7, 1.0], [135, 1, 1.0], [136, 1, 1.0], [136, 3, 1.0], [137, 4, 1.0], [137, 5, 1.0], [138, 2, 1.0], [139, 2, 1.0], [140, 2, 1.0], [141, 2, 2.0], [142, 2, 1.0], [143, 2, 1.0], [144, 2, 1.0], [144, 6, 2.0], [145, 7, 1.0], [146, 1, 2.0], [147, 3, 3.0], [147, 4, 1.0], [147, 6, 1.0], [147, 7, 1.0], [148, 2, 9.0], [148, 4, 18.0], [148, 6, 1.0], [149, 7, 1.0], [150, 0, 1.0], [151, 0, 1.0], [152, 3, 1.0], [153, 8, 10.0], [154, 3, 2.0], [155, 8, 1.0], [156, 4, 1.0], [156, 7, 4.0], [157, 2, 1.0], [158, 1, 4.0], [158, 5, 2.0], [159, 1, 1.0], [159, 7, 2.0], [160, 1, 1.0], [160, 2, 2.0], [161, 3, 4.0], [162, 3, 1.0], [163, 2, 1.0], [163, 7, 2.0], [164, 0, 1.0], [165, 3, 1.0], [165, 4, 2.0], [165, 6, 1.0], [166, 2, 1.0], [167, 2, 1.0], [167, 4, 1.0], [168, 6, 4.0], [169, 4, 2.0], [170, 7, 1.0], [171, 3, 2.0], [171, 5, 12.0], [171, 7, 13.0], [172, 1, 1.0], [173, 0, 1.0], [174, 1, 1.0], [174, 8, 1.0], [175, 3, 1.0], [175, 4, 1.0], [176, 7, 1.0], [177, 1, 1.0], [178, 4, 1.0], [179, 3, 1.0], [180, 4, 1.0], [181, 8, 1.0], [182, 4, 1.0], [183, 0, 1.0], [183, 1, 2.0], [183, 7, 4.0], [184, 1, 1.0], [185, 7, 1.0], [186, 5, 1.0], [187, 2, 1.0], [187, 4, 1.0], [188, 5, 1.0], [189, 5, 1.0], [190, 3, 1.0], [191, 1, 1.0], [192, 3, 1.0], [193, 3, 1.0], [193, 7, 1.0], [194, 3, 1.0], [195, 4, 1.0], [196, 0, 1.0], [197, 2, 1.0], [198, 7, 1.0], [199, 5, 1.0], [200, 0, 1.0], [201, 1, 1.0], [202, 4, 1.0], [202, 6, 1.0], [203, 4, 1.0], [203, 6, 1.0], [204, 0, 1.0], [204, 2, 2.0], [204, 3, 2.0], [204, 4, 1.0], [204, 6, 4.0], [205, 0, 1.0], [206, 2, 1.0], [207, 7, 1.0], [208, 2, 1.0], [209, 1, 1.0], [209, 7, 4.0], [210, 3, 1.0], [211, 1, 1.0], [212, 6, 1.0], [213, 5, 1.0], [214, 8, 4.0], [215, 6, 1.0], [216, 8, 1.0], [217, 5, 1.0], [218, 3, 1.0], [219, 2, 1.0], [219, 4, 1.0], [220, 6, 1.0], [221, 7, 1.0], [222, 0, 1.0], [223, 1, 1.0], [224, 3, 1.0], [225, 1, 1.0], [226, 1, 1.0], [227, 1, 1.0], [227, 7, 1.0], [228, 3, 1.0], [229, 2, 1.0], [230, 2, 1.0], [231, 5, 3.0], [231, 7, 1.0], [232, 2, 2.0], [232, 7, 5.0], [233, 1, 1.0], [234, 3, 2.0], [235, 1, 1.0], [236, 0, 1.0], [237, 1, 1.0], [238, 2, 3.0], [238, 3, 11.0], [238, 4, 8.0], [238, 5, 1.0], [238, 6, 8.0], [239, 0, 1.0], [239, 8, 1.0], [240, 8, 1.0], [241, 8, 1.0], [242, 2, 2.0], [242, 3, 1.0], [242, 4, 1.0], [243, 3, 1.0], [244, 7, 1.0], [245, 2, 1.0], [246, 0, 1.0], [247, 0, 1.0], [248, 4, 1.0], [249, 7, 1.0], [250, 7, 1.0], [251, 4, 1.0], [252, 6, 1.0], [253, 7, 1.0], [254, 2, 2.0], [255, 7, 1.0], [256, 1, 1.0], [257, 5, 1.0], [258, 1, 1.0], [259, 0, 1.0], [260, 0, 2.0], [260, 5, 1.0], [261, 0, 2.0], [261, 1, 1.0], [261, 2, 1.0], [262, 0, 1.0], [263, 0, 2.0], [263, 4, 1.0], [263, 5, 1.0], [263, 8, 1.0], [264, 7, 1.0], [265, 5, 3.0], [266, 1, 1.0], [266, 3, 2.0], [266, 6, 2.0], [267, 3, 1.0], [268, 8, 1.0], [269, 3, 2.0], [269, 5, 4.0], [269, 6, 1.0], [270, 7, 1.0], [270, 8, 1.0], [271, 1, 1.0], [272, 8, 1.0], [273, 1, 1.0], [274, 6, 1.0], [274, 8, 2.0], [275, 1, 1.0], [276, 8, 1.0], [277, 6, 1.0], [278, 6, 1.0], [279, 6, 1.0], [280, 7, 1.0], [281, 1, 5.0], [281, 2, 1.0], [281, 3, 3.0], [281, 6, 1.0], [281, 7, 3.0], [281, 8, 1.0], [282, 1, 1.0], [283, 3, 2.0], [284, 1, 1.0], [285, 7, 1.0], [286, 8, 1.0], [287, 0, 2.0], [287, 1, 2.0], [287, 7, 2.0], [287, 8, 7.0], [288, 0, 3.0], [288, 1, 1.0], [288, 3, 1.0], [288, 4, 2.0], [288, 6, 13.0], [288, 8, 13.0], [289, 2, 1.0], [290, 2, 1.0], [291, 6, 1.0], [291, 7, 1.0], [292, 4, 1.0], [293, 2, 1.0], [294, 2, 1.0], [295, 8, 1.0], [296, 2, 2.0], [296, 5, 1.0], [296, 6, 10.0], [296, 8, 1.0], [297, 0, 1.0], [297, 2, 1.0], [298, 2, 1.0], [299, 2, 1.0], [299, 6, 1.0], [300, 1, 1.0], [300, 2, 1.0], [300, 5, 6.0], [300, 6, 1.0], [301, 8, 2.0], [302, 0, 2.0], [302, 2, 3.0], [302, 3, 1.0], [302, 5, 4.0], [302, 6, 1.0], [302, 8, 5.0], [303, 8, 1.0], [304, 4, 1.0], [305, 6, 1.0], [306, 2, 1.0], [306, 3, 2.0], [306, 6, 1.0], [306, 8, 1.0], [307, 8, 1.0], [308, 0, 1.0], [309, 3, 6.0], [309, 8, 2.0], [310, 6, 1.0], [311, 7, 1.0], [312, 0, 1.0], [313, 0, 1.0], [313, 1, 7.0], [313, 8, 5.0], [314, 6, 1.0], [315, 2, 1.0], [315, 3, 2.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2826", "matrix_type": "sparse", "shape": [316, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T16:24:01.170353", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

rt2 = """{"rows": [{"id": "1", "metadata": {}}, {"id": "3", "metadata": {}}, {"id": "5", "metadata": {}}, {"id": "8", "metadata": {}}, {"id": "10", "metadata": {}}, {"id": "11", "metadata": {}}, {"id": "12", "metadata": {}}, {"id": "13", "metadata": {}}, {"id": "14", "metadata": {}}, {"id": "15", "metadata": {}}, {"id": "16", "metadata": {}}, {"id": "17", "metadata": {}}, {"id": "18", "metadata": {}}, {"id": "20", "metadata": {}}, {"id": "21", "metadata": {}}, {"id": "23", "metadata": {}}, {"id": "24", "metadata": {}}, {"id": "25", "metadata": {}}, {"id": "26", "metadata": {}}, {"id": "27", "metadata": {}}, {"id": "28", "metadata": {}}, {"id": "29", "metadata": {}}, {"id": "30", "metadata": {}}, {"id": "31", "metadata": {}}, {"id": "32", "metadata": {}}, {"id": "33", "metadata": {}}, {"id": "34", "metadata": {}}, {"id": "35", "metadata": {}}, {"id": "36", "metadata": {}}, {"id": "37", "metadata": {}}, {"id": "39", "metadata": {}}, {"id": "40", "metadata": {}}, {"id": "41", "metadata": {}}, {"id": "42", "metadata": {}}, {"id": "43", "metadata": {}}, {"id": "44", "metadata": {}}, {"id": "45", "metadata": {}}, {"id": "46", "metadata": {}}, {"id": "48", "metadata": {}}, {"id": "51", "metadata": {}}, {"id": "52", "metadata": {}}, {"id": "53", "metadata": {}}, {"id": "54", "metadata": {}}, {"id": "55", "metadata": {}}, {"id": "56", "metadata": {}}, {"id": "57", "metadata": {}}, {"id": "59", "metadata": {}}, {"id": "60", "metadata": {}}, {"id": "62", "metadata": {}}, {"id": "63", "metadata": {}}, {"id": "64", "metadata": {}}, {"id": "66", "metadata": {}}, {"id": "67", "metadata": {}}, {"id": "68", "metadata": {}}, {"id": "69", "metadata": {}}, {"id": "70", "metadata": {}}, {"id": "72", "metadata": {}}, {"id": "73", "metadata": {}}, {"id": "74", "metadata": {}}, {"id": "75", "metadata": {}}, {"id": "76", "metadata": {}}, {"id": "77", "metadata": {}}, {"id": "82", "metadata": {}}, {"id": "83", "metadata": {}}, {"id": "84", "metadata": {}}, {"id": "85", "metadata": {}}, {"id": "86", "metadata": {}}, {"id": "87", "metadata": {}}, {"id": "88", "metadata": {}}, {"id": "89", "metadata": {}}, {"id": "90", "metadata": {}}, {"id": "91", "metadata": {}}, {"id": "92", "metadata": {}}, {"id": "93", "metadata": {}}, {"id": "94", "metadata": {}}, {"id": "95", "metadata": {}}, {"id": "96", "metadata": {}}, {"id": "98", "metadata": {}}, {"id": "99", "metadata": {}}, {"id": "103", "metadata": {}}, {"id": "104", "metadata": {}}, {"id": "106", "metadata": {}}, {"id": "107", "metadata": {}}, {"id": "108", "metadata": {}}, {"id": "109", "metadata": {}}, {"id": "110", "metadata": {}}, {"id": "111", "metadata": {}}, {"id": "113", "metadata": {}}, {"id": "114", "metadata": {}}, {"id": "117", "metadata": {}}, {"id": "118", "metadata": {}}, {"id": "119", "metadata": {}}, {"id": "120", "metadata": {}}, {"id": "121", "metadata": {}}, {"id": "122", "metadata": {}}, {"id": "123", "metadata": {}}, {"id": "124", "metadata": {}}, {"id": "126", "metadata": {}}, {"id": "129", "metadata": {}}, {"id": "130", "metadata": {}}, {"id": "131", "metadata": {}}, {"id": "133", "metadata": {}}, {"id": "134", "metadata": {}}, {"id": "135", "metadata": {}}, {"id": "136", "metadata": {}}, {"id": "137", "metadata": {}}, {"id": "138", "metadata": {}}, {"id": "139", "metadata": {}}, {"id": "140", "metadata": {}}, {"id": "141", "metadata": {}}, {"id": "142", "metadata": {}}, {"id": "144", "metadata": {}}, {"id": "145", "metadata": {}}, {"id": "146", "metadata": {}}, {"id": "147", "metadata": {}}, {"id": "148", "metadata": {}}, {"id": "150", "metadata": {}}, {"id": "151", "metadata": {}}, {"id": "154", "metadata": {}}, {"id": "155", "metadata": {}}, {"id": "156", "metadata": {}}, {"id": "157", "metadata": {}}, {"id": "160", "metadata": {}}, {"id": "161", "metadata": {}}, {"id": "163", "metadata": {}}, {"id": "164", "metadata": {}}, {"id": "165", "metadata": {}}, {"id": "166", "metadata": {}}, {"id": "167", "metadata": {}}, {"id": "168", "metadata": {}}, {"id": "169", "metadata": {}}, {"id": "171", "metadata": {}}, {"id": "173", "metadata": {}}, {"id": "175", "metadata": {}}, {"id": "176", "metadata": {}}, {"id": "177", "metadata": {}}, {"id": "178", "metadata": {}}, {"id": "180", "metadata": {}}, {"id": "181", "metadata": {}}, {"id": "182", "metadata": {}}, {"id": "184", "metadata": {}}, {"id": "186", "metadata": {}}, {"id": "187", "metadata": {}}, {"id": "188", "metadata": {}}, {"id": "190", "metadata": {}}, {"id": "192", "metadata": {}}, {"id": "193", "metadata": {}}, {"id": "194", "metadata": {}}, {"id": "195", "metadata": {}}, {"id": "197", "metadata": {}}, {"id": "198", "metadata": {}}, {"id": "200", "metadata": {}}, {"id": "201", "metadata": {}}, {"id": "202", "metadata": {}}, {"id": "203", "metadata": {}}, {"id": "204", "metadata": {}}, {"id": "207", "metadata": {}}, {"id": "208", "metadata": {}}, {"id": "209", "metadata": {}}, {"id": "210", "metadata": {}}, {"id": "211", "metadata": {}}, {"id": "212", "metadata": {}}, {"id": "213", "metadata": {}}, {"id": "214", "metadata": {}}, {"id": "216", "metadata": {}}, {"id": "217", "metadata": {}}, {"id": "218", "metadata": {}}, {"id": "219", "metadata": {}}, {"id": "220", "metadata": {}}, {"id": "221", "metadata": {}}, {"id": "223", "metadata": {}}, {"id": "224", "metadata": {}}, {"id": "225", "metadata": {}}, {"id": "227", "metadata": {}}, {"id": "228", "metadata": {}}, {"id": "229", "metadata": {}}, {"id": "231", "metadata": {}}, {"id": "232", "metadata": {}}, {"id": "233", "metadata": {}}, {"id": "235", "metadata": {}}, {"id": "237", "metadata": {}}, {"id": "238", "metadata": {}}, {"id": "239", "metadata": {}}, {"id": "240", "metadata": {}}, {"id": "242", "metadata": {}}, {"id": "243", "metadata": {}}, {"id": "244", "metadata": {}}, {"id": "245", "metadata": {}}, {"id": "246", "metadata": {}}, {"id": "247", "metadata": {}}, {"id": "248", "metadata": {}}, {"id": "249", "metadata": {}}, {"id": "250", "metadata": {}}, {"id": "251", "metadata": {}}, {"id": "253", "metadata": {}}, {"id": "254", "metadata": {}}, {"id": "255", "metadata": {}}, {"id": "256", "metadata": {}}, {"id": "257", "metadata": {}}, {"id": "258", "metadata": {}}, {"id": "259", "metadata": {}}, {"id": "262", "metadata": {}}, {"id": "265", "metadata": {}}, {"id": "267", "metadata": {}}, {"id": "268", "metadata": {}}, {"id": "271", "metadata": {}}, {"id": "272", "metadata": {}}, {"id": "273", "metadata": {}}, {"id": "274", "metadata": {}}, {"id": "275", "metadata": {}}, {"id": "276", "metadata": {}}, {"id": "278", "metadata": {}}, {"id": "282", "metadata": {}}, {"id": "283", "metadata": {}}, {"id": "284", "metadata": {}}, {"id": "288", "metadata": {}}, {"id": "289", "metadata": {}}, {"id": "290", "metadata": {}}, {"id": "291", "metadata": {}}, {"id": "292", "metadata": {}}, {"id": "293", "metadata": {}}, {"id": "294", "metadata": {}}, {"id": "295", "metadata": {}}, {"id": "296", "metadata": {}}, {"id": "298", "metadata": {}}, {"id": "299", "metadata": {}}, {"id": "300", "metadata": {}}, {"id": "301", "metadata": {}}, {"id": "302", "metadata": {}}, {"id": "303", "metadata": {}}, {"id": "304", "metadata": {}}, {"id": "305", "metadata": {}}, {"id": "306", "metadata": {}}, {"id": "307", "metadata": {}}, {"id": "310", "metadata": {}}, {"id": "312", "metadata": {}}, {"id": "313", "metadata": {}}, {"id": "314", "metadata": {}}, {"id": "315", "metadata": {}}, {"id": "318", "metadata": {}}, {"id": "319", "metadata": {}}, {"id": "320", "metadata": {}}, {"id": "322", "metadata": {}}, {"id": "323", "metadata": {}}, {"id": "324", "metadata": {}}, {"id": "325", "metadata": {}}, {"id": "327", "metadata": {}}, {"id": "328", "metadata": {}}, {"id": "331", "metadata": {}}, {"id": "332", "metadata": {}}, {"id": "333", "metadata": {}}, {"id": "335", "metadata": {}}, {"id": "337", "metadata": {}}, {"id": "340", "metadata": {}}, {"id": "341", "metadata": {}}, {"id": "342", "metadata": {}}, {"id": "343", "metadata": {}}, {"id": "344", "metadata": {}}, {"id": "345", "metadata": {}}, {"id": "346", "metadata": {}}, {"id": "347", "metadata": {}}, {"id": "349", "metadata": {}}, {"id": "350", "metadata": {}}, {"id": "351", "metadata": {}}, {"id": "352", "metadata": {}}, {"id": "354", "metadata": {}}, {"id": "355", "metadata": {}}, {"id": "356", "metadata": {}}, {"id": "357", "metadata": {}}, {"id": "358", "metadata": {}}, {"id": "359", "metadata": {}}, {"id": "360", "metadata": {}}, {"id": "361", "metadata": {}}, {"id": "362", "metadata": {}}, {"id": "363", "metadata": {}}, {"id": "364", "metadata": {}}, {"id": "366", "metadata": {}}, {"id": "367", "metadata": {}}, {"id": "369", "metadata": {}}, {"id": "370", "metadata": {}}, {"id": "371", "metadata": {}}, {"id": "372", "metadata": {}}, {"id": "373", "metadata": {}}, {"id": "374", "metadata": {}}, {"id": "375", "metadata": {}}, {"id": "376", "metadata": {}}, {"id": "378", "metadata": {}}, {"id": "379", "metadata": {}}, {"id": "380", "metadata": {}}, {"id": "381", "metadata": {}}, {"id": "382", "metadata": {}}, {"id": "383", "metadata": {}}, {"id": "384", "metadata": {}}, {"id": "385", "metadata": {}}, {"id": "386", "metadata": {}}, {"id": "387", "metadata": {}}, {"id": "388", "metadata": {}}, {"id": "390", "metadata": {}}, {"id": "391", "metadata": {}}, {"id": "392", "metadata": {}}, {"id": "393", "metadata": {}}, {"id": "395", "metadata": {}}, {"id": "396", "metadata": {}}, {"id": "397", "metadata": {}}, {"id": "398", "metadata": {}}, {"id": "399", "metadata": {}}, {"id": "400", "metadata": {}}, {"id": "401", "metadata": {}}, {"id": "402", "metadata": {}}, {"id": "403", "metadata": {}}, {"id": "405", "metadata": {}}, {"id": "406", "metadata": {}}, {"id": "407", "metadata": {}}, {"id": "408", "metadata": {}}, {"id": "409", "metadata": {}}, {"id": "410", "metadata": {}}, {"id": "411", "metadata": {}}, {"id": "413", "metadata": {}}, {"id": "414", "metadata": {}}, {"id": "415", "metadata": {}}, {"id": "416", "metadata": {}}, {"id": "418", "metadata": {}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 1, 1.0], [1, 2, 1.0], [2, 0, 1.0], [3, 0, 1.0], [4, 3, 1.0], [5, 1, 1.0], [5, 3, 1.0], [6, 6, 1.0], [7, 6, 1.0], [8, 1, 1.0], [9, 7, 1.0], [10, 1, 1.0], [11, 8, 1.0], [12, 8, 1.0], [13, 8, 1.0], [14, 3, 1.0], [14, 4, 1.0], [15, 7, 1.0], [16, 1, 2.0], [17, 3, 1.0], [18, 1, 1.0], [19, 7, 2.0], [20, 0, 1.0], [21, 8, 1.0], [22, 1, 1.0], [23, 8, 1.0], [24, 3, 1.0], [25, 8, 1.0], [26, 1, 1.0], [27, 1, 1.0], [28, 7, 1.0], [29, 3, 3.0], [30, 7, 1.0], [31, 6, 1.0], [32, 3, 1.0], [32, 7, 2.0], [33, 3, 1.0], [33, 4, 1.0], [34, 7, 1.0], [35, 5, 1.0], [36, 4, 12.0], [36, 5, 10.0], [37, 0, 5.0], [37, 1, 1.0], [37, 7, 2.0], [37, 8, 4.0], [38, 7, 1.0], [39, 5, 1.0], [40, 3, 1.0], [41, 1, 1.0], [41, 3, 2.0], [41, 6, 1.0], [42, 0, 22.0], [42, 1, 9.0], [42, 3, 1.0], [42, 8, 4.0], [43, 5, 1.0], [44, 0, 5.0], [44, 1, 3.0], [44, 3, 1.0], [44, 5, 1.0], [44, 6, 3.0], [44, 7, 8.0], [45, 5, 1.0], [46, 1, 1.0], [46, 2, 5.0], [46, 3, 1.0], [46, 4, 2.0], [46, 5, 15.0], [46, 6, 1.0], [47, 3, 1.0], [48, 7, 1.0], [49, 2, 1.0], [50, 6, 1.0], [51, 1, 1.0], [52, 3, 1.0], [53, 3, 1.0], [54, 6, 1.0], [55, 7, 1.0], [56, 0, 1.0], [56, 1, 1.0], [56, 8, 2.0], [57, 8, 1.0], [58, 0, 1.0], [58, 1, 3.0], [58, 5, 1.0], [58, 7, 2.0], [58, 8, 1.0], [59, 7, 5.0], [60, 4, 1.0], [61, 0, 2.0], [61, 1, 1.0], [62, 0, 12.0], [62, 1, 2.0], [62, 3, 1.0], [62, 8, 1.0], [63, 7, 1.0], [64, 2, 1.0], [65, 3, 1.0], [66, 3, 1.0], [67, 3, 1.0], [68, 5, 1.0], [69, 1, 1.0], [70, 0, 1.0], [70, 1, 1.0], [70, 2, 2.0], [70, 5, 1.0], [70, 6, 3.0], [70, 7, 2.0], [71, 2, 1.0], [72, 2, 1.0], [73, 2, 1.0], [73, 3, 4.0], [73, 4, 1.0], [73, 6, 2.0], [74, 2, 1.0], [75, 4, 1.0], [76, 0, 1.0], [77, 7, 1.0], [78, 1, 1.0], [78, 8, 2.0], [79, 0, 1.0], [80, 3, 1.0], [80, 5, 3.0], [81, 3, 1.0], [82, 6, 1.0], [83, 8, 3.0], [84, 8, 1.0], [85, 4, 1.0], [86, 7, 1.0], [87, 5, 1.0], [88, 4, 1.0], [89, 4, 1.0], [90, 3, 1.0], [91, 2, 1.0], [92, 1, 1.0], [92, 8, 2.0], [93, 0, 1.0], [93, 6, 1.0], [94, 6, 1.0], [95, 2, 1.0], [95, 4, 1.0], [95, 6, 2.0], [96, 0, 1.0], [97, 2, 1.0], [98, 2, 2.0], [98, 4, 2.0], [98, 6, 2.0], [99, 2, 1.0], [99, 5, 1.0], [99, 6, 1.0], [100, 5, 1.0], [101, 4, 1.0], [102, 1, 1.0], [103, 2, 1.0], [103, 4, 1.0], [104, 1, 1.0], [104, 4, 1.0], [105, 8, 1.0], [106, 8, 1.0], [107, 4, 1.0], [108, 7, 1.0], [109, 3, 1.0], [110, 1, 1.0], [111, 2, 1.0], [112, 1, 1.0], [113, 1, 1.0], [113, 3, 3.0], [114, 7, 1.0], [115, 1, 1.0], [115, 3, 1.0], [115, 4, 4.0], [115, 6, 5.0], [116, 1, 1.0], [117, 0, 2.0], [117, 1, 3.0], [117, 8, 1.0], [118, 0, 1.0], [119, 8, 1.0], [120, 1, 1.0], [120, 3, 1.0], [121, 8, 1.0], [122, 7, 1.0], [123, 6, 1.0], [124, 0, 1.0], [124, 1, 1.0], [125, 0, 1.0], [125, 1, 1.0], [126, 2, 10.0], [126, 4, 9.0], [126, 6, 1.0], [127, 2, 1.0], [127, 4, 4.0], [127, 5, 2.0], [128, 1, 1.0], [129, 5, 1.0], [130, 2, 5.0], [130, 4, 2.0], [130, 6, 3.0], [131, 7, 1.0], [132, 0, 1.0], [133, 0, 1.0], [133, 8, 1.0], [134, 8, 1.0], [135, 4, 1.0], [135, 5, 9.0], [136, 8, 1.0], [137, 3, 1.0], [138, 5, 6.0], [139, 0, 1.0], [139, 5, 2.0], [140, 3, 1.0], [141, 1, 1.0], [142, 6, 1.0], [143, 3, 1.0], [143, 7, 1.0], [144, 1, 1.0], [145, 2, 1.0], [146, 2, 1.0], [147, 2, 1.0], [148, 2, 2.0], [149, 2, 1.0], [150, 2, 1.0], [150, 6, 2.0], [151, 7, 1.0], [152, 1, 1.0], [153, 3, 3.0], [153, 4, 2.0], [154, 2, 9.0], [154, 4, 19.0], [155, 7, 1.0], [156, 0, 1.0], [157, 0, 1.0], [158, 2, 2.0], [159, 3, 1.0], [160, 8, 6.0], [161, 3, 2.0], [162, 8, 1.0], [163, 7, 2.0], [164, 2, 1.0], [165, 1, 5.0], [165, 5, 1.0], [166, 1, 1.0], [167, 1, 1.0], [167, 2, 2.0], [168, 6, 1.0], [169, 3, 3.0], [170, 1, 1.0], [170, 2, 1.0], [171, 0, 1.0], [172, 4, 2.0], [172, 6, 1.0], [173, 3, 1.0], [173, 4, 1.0], [174, 3, 1.0], [174, 4, 1.0], [174, 6, 1.0], [175, 4, 2.0], [176, 7, 1.0], [177, 3, 2.0], [177, 4, 1.0], [177, 5, 12.0], [177, 7, 14.0], [178, 1, 1.0], [179, 8, 1.0], [180, 7, 1.0], [181, 1, 1.0], [182, 4, 1.0], [183, 3, 1.0], [184, 8, 1.0], [185, 1, 1.0], [186, 4, 1.0], [187, 7, 1.0], [188, 0, 1.0], [188, 1, 1.0], [188, 7, 5.0], [189, 1, 1.0], [190, 7, 1.0], [191, 7, 1.0], [192, 5, 1.0], [193, 4, 1.0], [194, 5, 1.0], [195, 3, 1.0], [196, 1, 1.0], [197, 3, 2.0], [198, 3, 1.0], [199, 1, 1.0], [200, 2, 1.0], [201, 4, 1.0], [202, 0, 1.0], [203, 7, 1.0], [204, 6, 1.0], [205, 2, 1.0], [205, 4, 1.0], [205, 6, 1.0], [206, 6, 1.0], [207, 0, 1.0], [207, 2, 3.0], [207, 3, 2.0], [207, 6, 3.0], [208, 0, 1.0], [209, 2, 1.0], [210, 7, 1.0], [211, 1, 2.0], [211, 7, 5.0], [212, 5, 1.0], [213, 8, 3.0], [214, 6, 1.0], [215, 2, 1.0], [216, 6, 1.0], [217, 3, 1.0], [218, 7, 1.0], [219, 0, 1.0], [220, 1, 1.0], [221, 7, 1.0], [222, 3, 1.0], [223, 1, 1.0], [224, 7, 1.0], [225, 6, 1.0], [226, 3, 1.0], [227, 2, 1.0], [228, 7, 1.0], [229, 2, 1.0], [230, 5, 4.0], [230, 7, 1.0], [231, 7, 1.0], [232, 2, 2.0], [232, 4, 1.0], [232, 7, 5.0], [233, 1, 1.0], [234, 3, 2.0], [235, 0, 1.0], [236, 1, 1.0], [237, 2, 4.0], [237, 3, 7.0], [237, 4, 10.0], [237, 5, 2.0], [237, 6, 8.0], [238, 0, 1.0], [238, 1, 1.0], [239, 2, 2.0], [240, 8, 1.0], [241, 3, 1.0], [242, 7, 1.0], [243, 2, 1.0], [244, 0, 1.0], [245, 0, 1.0], [246, 7, 1.0], [247, 7, 1.0], [248, 6, 1.0], [249, 2, 1.0], [249, 4, 1.0], [250, 2, 1.0], [251, 2, 2.0], [252, 1, 1.0], [253, 1, 1.0], [254, 0, 1.0], [255, 0, 1.0], [255, 3, 1.0], [255, 5, 1.0], [255, 6, 1.0], [256, 0, 2.0], [256, 1, 1.0], [256, 2, 2.0], [257, 1, 1.0], [258, 0, 1.0], [259, 8, 1.0], [260, 0, 3.0], [260, 4, 1.0], [260, 5, 2.0], [260, 8, 1.0], [261, 5, 3.0], [262, 1, 1.0], [262, 3, 5.0], [262, 6, 1.0], [263, 3, 1.0], [264, 3, 1.0], [265, 8, 1.0], [266, 3, 2.0], [266, 4, 1.0], [266, 5, 2.0], [267, 8, 1.0], [268, 7, 1.0], [268, 8, 1.0], [269, 1, 1.0], [270, 5, 2.0], [270, 8, 1.0], [271, 1, 1.0], [272, 1, 1.0], [273, 8, 1.0], [274, 1, 1.0], [275, 8, 1.0], [276, 6, 1.0], [277, 6, 1.0], [278, 7, 1.0], [279, 1, 3.0], [279, 2, 2.0], [279, 3, 3.0], [279, 4, 1.0], [279, 6, 2.0], [279, 7, 2.0], [279, 8, 1.0], [280, 1, 1.0], [281, 3, 2.0], [282, 1, 1.0], [283, 1, 1.0], [284, 7, 1.0], [285, 8, 1.0], [286, 0, 1.0], [286, 1, 3.0], [286, 7, 4.0], [286, 8, 6.0], [287, 0, 3.0], [287, 1, 3.0], [287, 3, 1.0], [287, 4, 1.0], [287, 6, 15.0], [287, 8, 16.0], [288, 2, 1.0], [289, 2, 1.0], [290, 3, 1.0], [291, 6, 1.0], [291, 7, 1.0], [292, 2, 1.0], [293, 4, 1.0], [294, 2, 1.0], [295, 2, 1.0], [296, 2, 1.0], [297, 2, 1.0], [298, 8, 1.0], [299, 2, 2.0], [299, 5, 2.0], [299, 6, 13.0], [299, 8, 2.0], [300, 2, 1.0], [301, 3, 1.0], [302, 2, 1.0], [302, 6, 1.0], [303, 0, 2.0], [303, 2, 1.0], [303, 5, 5.0], [303, 6, 1.0], [303, 7, 1.0], [303, 8, 1.0], [304, 7, 1.0], [305, 8, 1.0], [306, 0, 3.0], [306, 2, 1.0], [306, 3, 1.0], [306, 5, 2.0], [306, 6, 2.0], [306, 8, 2.0], [307, 8, 1.0], [308, 6, 1.0], [309, 4, 1.0], [310, 3, 1.0], [311, 0, 1.0], [311, 2, 1.0], [311, 3, 1.0], [311, 6, 2.0], [311, 8, 1.0], [312, 8, 1.0], [313, 8, 1.0], [314, 0, 1.0], [315, 3, 6.0], [315, 8, 2.0], [316, 0, 1.0], [317, 0, 1.0], [318, 0, 2.0], [318, 1, 3.0], [318, 8, 7.0], [319, 6, 1.0], [320, 3, 1.0], [321, 0, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2826", "matrix_type": "sparse", "shape": [322, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T16:24:01.254089", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

rt3 = """{"rows": [{"id": "0", "metadata": {}}, {"id": "2", "metadata": {}}, {"id": "4", "metadata": {}}, {"id": "5", "metadata": {}}, {"id": "7", "metadata": {}}, {"id": "8", "metadata": {}}, {"id": "9", "metadata": {}}, {"id": "10", "metadata": {}}, {"id": "11", "metadata": {}}, {"id": "12", "metadata": {}}, {"id": "13", "metadata": {}}, {"id": "14", "metadata": {}}, {"id": "18", "metadata": {}}, {"id": "19", "metadata": {}}, {"id": "20", "metadata": {}}, {"id": "21", "metadata": {}}, {"id": "24", "metadata": {}}, {"id": "25", "metadata": {}}, {"id": "27", "metadata": {}}, {"id": "28", "metadata": {}}, {"id": "29", "metadata": {}}, {"id": "30", "metadata": {}}, {"id": "31", "metadata": {}}, {"id": "32", "metadata": {}}, {"id": "33", "metadata": {}}, {"id": "34", "metadata": {}}, {"id": "35", "metadata": {}}, {"id": "36", "metadata": {}}, {"id": "37", "metadata": {}}, {"id": "38", "metadata": {}}, {"id": "39", "metadata": {}}, {"id": "40", "metadata": {}}, {"id": "41", "metadata": {}}, {"id": "42", "metadata": {}}, {"id": "43", "metadata": {}}, {"id": "44", "metadata": {}}, {"id": "45", "metadata": {}}, {"id": "46", "metadata": {}}, {"id": "48", "metadata": {}}, {"id": "50", "metadata": {}}, {"id": "52", "metadata": {}}, {"id": "53", "metadata": {}}, {"id": "54", "metadata": {}}, {"id": "55", "metadata": {}}, {"id": "56", "metadata": {}}, {"id": "57", "metadata": {}}, {"id": "58", "metadata": {}}, {"id": "59", "metadata": {}}, {"id": "62", "metadata": {}}, {"id": "65", "metadata": {}}, {"id": "66", "metadata": {}}, {"id": "67", "metadata": {}}, {"id": "68", "metadata": {}}, {"id": "69", "metadata": {}}, {"id": "70", "metadata": {}}, {"id": "72", "metadata": {}}, {"id": "73", "metadata": {}}, {"id": "74", "metadata": {}}, {"id": "75", "metadata": {}}, {"id": "77", "metadata": {}}, {"id": "78", "metadata": {}}, {"id": "80", "metadata": {}}, {"id": "81", "metadata": {}}, {"id": "82", "metadata": {}}, {"id": "83", "metadata": {}}, {"id": "84", "metadata": {}}, {"id": "85", "metadata": {}}, {"id": "86", "metadata": {}}, {"id": "87", "metadata": {}}, {"id": "88", "metadata": {}}, {"id": "90", "metadata": {}}, {"id": "91", "metadata": {}}, {"id": "93", "metadata": {}}, {"id": "94", "metadata": {}}, {"id": "95", "metadata": {}}, {"id": "96", "metadata": {}}, {"id": "98", "metadata": {}}, {"id": "99", "metadata": {}}, {"id": "103", "metadata": {}}, {"id": "104", "metadata": {}}, {"id": "106", "metadata": {}}, {"id": "107", "metadata": {}}, {"id": "108", "metadata": {}}, {"id": "111", "metadata": {}}, {"id": "113", "metadata": {}}, {"id": "114", "metadata": {}}, {"id": "115", "metadata": {}}, {"id": "116", "metadata": {}}, {"id": "117", "metadata": {}}, {"id": "118", "metadata": {}}, {"id": "120", "metadata": {}}, {"id": "121", "metadata": {}}, {"id": "122", "metadata": {}}, {"id": "123", "metadata": {}}, {"id": "124", "metadata": {}}, {"id": "125", "metadata": {}}, {"id": "126", "metadata": {}}, {"id": "127", "metadata": {}}, {"id": "128", "metadata": {}}, {"id": "129", "metadata": {}}, {"id": "130", "metadata": {}}, {"id": "131", "metadata": {}}, {"id": "132", "metadata": {}}, {"id": "134", "metadata": {}}, {"id": "136", "metadata": {}}, {"id": "137", "metadata": {}}, {"id": "138", "metadata": {}}, {"id": "139", "metadata": {}}, {"id": "140", "metadata": {}}, {"id": "143", "metadata": {}}, {"id": "144", "metadata": {}}, {"id": "145", "metadata": {}}, {"id": "146", "metadata": {}}, {"id": "147", "metadata": {}}, {"id": "148", "metadata": {}}, {"id": "150", "metadata": {}}, {"id": "151", "metadata": {}}, {"id": "154", "metadata": {}}, {"id": "155", "metadata": {}}, {"id": "156", "metadata": {}}, {"id": "157", "metadata": {}}, {"id": "158", "metadata": {}}, {"id": "159", "metadata": {}}, {"id": "160", "metadata": {}}, {"id": "161", "metadata": {}}, {"id": "162", "metadata": {}}, {"id": "163", "metadata": {}}, {"id": "164", "metadata": {}}, {"id": "165", "metadata": {}}, {"id": "166", "metadata": {}}, {"id": "168", "metadata": {}}, {"id": "169", "metadata": {}}, {"id": "170", "metadata": {}}, {"id": "174", "metadata": {}}, {"id": "175", "metadata": {}}, {"id": "176", "metadata": {}}, {"id": "177", "metadata": {}}, {"id": "178", "metadata": {}}, {"id": "179", "metadata": {}}, {"id": "180", "metadata": {}}, {"id": "181", "metadata": {}}, {"id": "182", "metadata": {}}, {"id": "183", "metadata": {}}, {"id": "185", "metadata": {}}, {"id": "186", "metadata": {}}, {"id": "187", "metadata": {}}, {"id": "188", "metadata": {}}, {"id": "190", "metadata": {}}, {"id": "191", "metadata": {}}, {"id": "194", "metadata": {}}, {"id": "195", "metadata": {}}, {"id": "198", "metadata": {}}, {"id": "199", "metadata": {}}, {"id": "200", "metadata": {}}, {"id": "201", "metadata": {}}, {"id": "202", "metadata": {}}, {"id": "203", "metadata": {}}, {"id": "204", "metadata": {}}, {"id": "205", "metadata": {}}, {"id": "207", "metadata": {}}, {"id": "208", "metadata": {}}, {"id": "209", "metadata": {}}, {"id": "211", "metadata": {}}, {"id": "212", "metadata": {}}, {"id": "213", "metadata": {}}, {"id": "214", "metadata": {}}, {"id": "215", "metadata": {}}, {"id": "217", "metadata": {}}, {"id": "218", "metadata": {}}, {"id": "219", "metadata": {}}, {"id": "221", "metadata": {}}, {"id": "222", "metadata": {}}, {"id": "223", "metadata": {}}, {"id": "224", "metadata": {}}, {"id": "225", "metadata": {}}, {"id": "226", "metadata": {}}, {"id": "227", "metadata": {}}, {"id": "228", "metadata": {}}, {"id": "229", "metadata": {}}, {"id": "231", "metadata": {}}, {"id": "232", "metadata": {}}, {"id": "233", "metadata": {}}, {"id": "235", "metadata": {}}, {"id": "237", "metadata": {}}, {"id": "238", "metadata": {}}, {"id": "239", "metadata": {}}, {"id": "240", "metadata": {}}, {"id": "241", "metadata": {}}, {"id": "242", "metadata": {}}, {"id": "243", "metadata": {}}, {"id": "244", "metadata": {}}, {"id": "246", "metadata": {}}, {"id": "247", "metadata": {}}, {"id": "248", "metadata": {}}, {"id": "249", "metadata": {}}, {"id": "250", "metadata": {}}, {"id": "251", "metadata": {}}, {"id": "252", "metadata": {}}, {"id": "253", "metadata": {}}, {"id": "254", "metadata": {}}, {"id": "255", "metadata": {}}, {"id": "256", "metadata": {}}, {"id": "257", "metadata": {}}, {"id": "259", "metadata": {}}, {"id": "260", "metadata": {}}, {"id": "261", "metadata": {}}, {"id": "263", "metadata": {}}, {"id": "264", "metadata": {}}, {"id": "265", "metadata": {}}, {"id": "266", "metadata": {}}, {"id": "267", "metadata": {}}, {"id": "268", "metadata": {}}, {"id": "269", "metadata": {}}, {"id": "270", "metadata": {}}, {"id": "271", "metadata": {}}, {"id": "272", "metadata": {}}, {"id": "273", "metadata": {}}, {"id": "274", "metadata": {}}, {"id": "275", "metadata": {}}, {"id": "276", "metadata": {}}, {"id": "277", "metadata": {}}, {"id": "278", "metadata": {}}, {"id": "279", "metadata": {}}, {"id": "280", "metadata": {}}, {"id": "281", "metadata": {}}, {"id": "283", "metadata": {}}, {"id": "284", "metadata": {}}, {"id": "288", "metadata": {}}, {"id": "289", "metadata": {}}, {"id": "293", "metadata": {}}, {"id": "295", "metadata": {}}, {"id": "296", "metadata": {}}, {"id": "297", "metadata": {}}, {"id": "298", "metadata": {}}, {"id": "299", "metadata": {}}, {"id": "302", "metadata": {}}, {"id": "303", "metadata": {}}, {"id": "304", "metadata": {}}, {"id": "306", "metadata": {}}, {"id": "307", "metadata": {}}, {"id": "308", "metadata": {}}, {"id": "309", "metadata": {}}, {"id": "310", "metadata": {}}, {"id": "311", "metadata": {}}, {"id": "312", "metadata": {}}, {"id": "313", "metadata": {}}, {"id": "314", "metadata": {}}, {"id": "315", "metadata": {}}, {"id": "318", "metadata": {}}, {"id": "319", "metadata": {}}, {"id": "320", "metadata": {}}, {"id": "321", "metadata": {}}, {"id": "322", "metadata": {}}, {"id": "323", "metadata": {}}, {"id": "324", "metadata": {}}, {"id": "326", "metadata": {}}, {"id": "327", "metadata": {}}, {"id": "328", "metadata": {}}, {"id": "329", "metadata": {}}, {"id": "330", "metadata": {}}, {"id": "331", "metadata": {}}, {"id": "332", "metadata": {}}, {"id": "335", "metadata": {}}, {"id": "337", "metadata": {}}, {"id": "338", "metadata": {}}, {"id": "340", "metadata": {}}, {"id": "341", "metadata": {}}, {"id": "342", "metadata": {}}, {"id": "343", "metadata": {}}, {"id": "344", "metadata": {}}, {"id": "345", "metadata": {}}, {"id": "347", "metadata": {}}, {"id": "349", "metadata": {}}, {"id": "350", "metadata": {}}, {"id": "352", "metadata": {}}, {"id": "353", "metadata": {}}, {"id": "354", "metadata": {}}, {"id": "355", "metadata": {}}, {"id": "356", "metadata": {}}, {"id": "359", "metadata": {}}, {"id": "360", "metadata": {}}, {"id": "361", "metadata": {}}, {"id": "362", "metadata": {}}, {"id": "363", "metadata": {}}, {"id": "365", "metadata": {}}, {"id": "366", "metadata": {}}, {"id": "367", "metadata": {}}, {"id": "368", "metadata": {}}, {"id": "369", "metadata": {}}, {"id": "370", "metadata": {}}, {"id": "373", "metadata": {}}, {"id": "374", "metadata": {}}, {"id": "377", "metadata": {}}, {"id": "378", "metadata": {}}, {"id": "379", "metadata": {}}, {"id": "381", "metadata": {}}, {"id": "382", "metadata": {}}, {"id": "383", "metadata": {}}, {"id": "384", "metadata": {}}, {"id": "387", "metadata": {}}, {"id": "388", "metadata": {}}, {"id": "390", "metadata": {}}, {"id": "391", "metadata": {}}, {"id": "392", "metadata": {}}, {"id": "393", "metadata": {}}, {"id": "394", "metadata": {}}, {"id": "395", "metadata": {}}, {"id": "396", "metadata": {}}, {"id": "397", "metadata": {}}, {"id": "398", "metadata": {}}, {"id": "399", "metadata": {}}, {"id": "400", "metadata": {}}, {"id": "401", "metadata": {}}, {"id": "402", "metadata": {}}, {"id": "405", "metadata": {}}, {"id": "406", "metadata": {}}, {"id": "407", "metadata": {}}, {"id": "408", "metadata": {}}, {"id": "409", "metadata": {}}, {"id": "410", "metadata": {}}, {"id": "411", "metadata": {}}, {"id": "414", "metadata": {}}, {"id": "415", "metadata": {}}, {"id": "416", "metadata": {}}, {"id": "417", "metadata": {}}, {"id": "418", "metadata": {}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 0, 1.0], [2, 3, 1.0], [3, 0, 1.0], [4, 3, 1.0], [5, 0, 1.0], [5, 1, 1.0], [5, 2, 1.0], [5, 4, 1.0], [6, 5, 1.0], [7, 3, 1.0], [8, 1, 1.0], [9, 6, 1.0], [10, 4, 1.0], [10, 6, 1.0], [11, 1, 1.0], [12, 8, 1.0], [13, 2, 1.0], [14, 8, 1.0], [15, 3, 1.0], [15, 4, 1.0], [16, 1, 1.0], [17, 3, 1.0], [18, 7, 2.0], [19, 0, 1.0], [20, 8, 1.0], [21, 1, 1.0], [22, 8, 2.0], [23, 1, 1.0], [23, 3, 1.0], [24, 8, 1.0], [25, 2, 1.0], [26, 1, 1.0], [27, 7, 1.0], [28, 3, 3.0], [29, 7, 1.0], [30, 7, 1.0], [31, 6, 1.0], [32, 3, 1.0], [32, 7, 1.0], [33, 3, 1.0], [33, 4, 1.0], [34, 7, 1.0], [35, 5, 1.0], [36, 4, 9.0], [36, 5, 8.0], [37, 0, 6.0], [37, 1, 1.0], [37, 7, 2.0], [37, 8, 5.0], [38, 7, 1.0], [39, 4, 1.0], [40, 3, 1.0], [41, 3, 1.0], [41, 6, 1.0], [42, 0, 26.0], [42, 1, 7.0], [42, 3, 1.0], [42, 8, 3.0], [43, 5, 1.0], [44, 0, 3.0], [44, 1, 3.0], [44, 3, 2.0], [44, 5, 1.0], [44, 6, 2.0], [44, 7, 5.0], [44, 8, 1.0], [45, 5, 1.0], [46, 0, 1.0], [47, 0, 1.0], [47, 2, 7.0], [47, 3, 2.0], [47, 4, 1.0], [47, 5, 19.0], [47, 6, 1.0], [48, 7, 1.0], [49, 7, 1.0], [50, 1, 1.0], [51, 3, 1.0], [52, 3, 1.0], [53, 6, 1.0], [54, 7, 1.0], [55, 0, 1.0], [55, 1, 1.0], [55, 8, 1.0], [56, 0, 1.0], [56, 8, 1.0], [57, 1, 1.0], [57, 5, 2.0], [57, 7, 1.0], [57, 8, 1.0], [58, 7, 4.0], [59, 0, 1.0], [59, 1, 2.0], [60, 1, 1.0], [61, 2, 1.0], [62, 2, 1.0], [63, 0, 10.0], [63, 1, 1.0], [63, 8, 1.0], [64, 7, 1.0], [65, 2, 1.0], [66, 3, 1.0], [67, 3, 1.0], [68, 3, 1.0], [69, 5, 1.0], [70, 1, 2.0], [70, 2, 3.0], [70, 5, 1.0], [70, 6, 3.0], [70, 7, 2.0], [71, 2, 1.0], [72, 3, 3.0], [72, 6, 3.0], [73, 2, 1.0], [74, 4, 2.0], [75, 0, 1.0], [76, 7, 1.0], [77, 1, 1.0], [77, 3, 1.0], [77, 8, 2.0], [78, 0, 1.0], [79, 5, 3.0], [80, 3, 1.0], [81, 6, 1.0], [82, 8, 5.0], [83, 7, 1.0], [84, 5, 1.0], [85, 4, 1.0], [86, 0, 1.0], [87, 2, 1.0], [87, 3, 1.0], [88, 3, 1.0], [88, 4, 1.0], [88, 7, 1.0], [89, 3, 1.0], [90, 1, 1.0], [90, 8, 2.0], [91, 0, 1.0], [92, 6, 1.0], [93, 6, 3.0], [94, 0, 1.0], [94, 6, 1.0], [95, 4, 1.0], [96, 1, 1.0], [96, 2, 1.0], [97, 7, 1.0], [98, 4, 1.0], [99, 2, 2.0], [99, 4, 1.0], [100, 2, 1.0], [100, 5, 1.0], [101, 5, 1.0], [102, 5, 1.0], [103, 1, 1.0], [104, 1, 1.0], [104, 4, 1.0], [105, 8, 1.0], [106, 8, 1.0], [107, 4, 1.0], [108, 7, 1.0], [109, 7, 1.0], [110, 2, 1.0], [111, 1, 1.0], [112, 1, 1.0], [112, 3, 5.0], [113, 7, 1.0], [114, 1, 1.0], [114, 3, 2.0], [114, 4, 4.0], [114, 6, 4.0], [115, 1, 1.0], [116, 0, 2.0], [116, 1, 4.0], [116, 8, 1.0], [117, 0, 1.0], [118, 8, 1.0], [119, 1, 1.0], [119, 3, 1.0], [120, 8, 1.0], [121, 1, 1.0], [122, 7, 1.0], [123, 7, 1.0], [124, 6, 2.0], [125, 6, 1.0], [126, 0, 1.0], [126, 1, 1.0], [127, 0, 1.0], [127, 1, 1.0], [128, 2, 11.0], [128, 3, 1.0], [128, 4, 11.0], [128, 6, 1.0], [129, 2, 4.0], [129, 4, 5.0], [129, 5, 2.0], [130, 5, 1.0], [131, 2, 5.0], [131, 4, 1.0], [131, 6, 3.0], [132, 5, 1.0], [133, 5, 1.0], [134, 0, 1.0], [135, 8, 1.0], [136, 4, 1.0], [136, 5, 6.0], [137, 8, 1.0], [138, 7, 1.0], [139, 3, 1.0], [140, 5, 9.0], [140, 7, 1.0], [141, 0, 1.0], [141, 5, 2.0], [142, 3, 1.0], [143, 7, 2.0], [144, 1, 1.0], [145, 6, 1.0], [146, 0, 1.0], [147, 1, 1.0], [148, 4, 1.0], [149, 2, 1.0], [150, 2, 1.0], [151, 2, 1.0], [151, 6, 2.0], [152, 7, 1.0], [153, 7, 1.0], [154, 1, 3.0], [155, 3, 4.0], [155, 4, 2.0], [155, 6, 1.0], [155, 7, 1.0], [156, 2, 7.0], [156, 4, 18.0], [157, 7, 1.0], [158, 1, 1.0], [159, 0, 1.0], [160, 0, 1.0], [161, 2, 1.0], [162, 8, 9.0], [163, 3, 1.0], [164, 8, 1.0], [165, 4, 1.0], [165, 7, 3.0], [166, 0, 1.0], [167, 1, 2.0], [167, 5, 2.0], [168, 1, 1.0], [168, 7, 1.0], [169, 1, 1.0], [169, 2, 2.0], [170, 3, 2.0], [171, 3, 1.0], [172, 2, 1.0], [172, 7, 2.0], [173, 0, 1.0], [174, 4, 1.0], [174, 6, 2.0], [175, 2, 1.0], [175, 4, 1.0], [176, 2, 1.0], [176, 3, 1.0], [176, 4, 1.0], [177, 4, 1.0], [177, 6, 3.0], [178, 4, 1.0], [179, 7, 2.0], [180, 3, 5.0], [180, 4, 1.0], [180, 5, 9.0], [180, 7, 12.0], [181, 1, 1.0], [182, 8, 1.0], [183, 7, 1.0], [184, 1, 1.0], [185, 4, 1.0], [186, 3, 1.0], [187, 4, 1.0], [188, 8, 1.0], [189, 1, 1.0], [190, 4, 1.0], [191, 1, 1.0], [191, 7, 3.0], [192, 7, 1.0], [193, 7, 1.0], [194, 8, 1.0], [195, 5, 1.0], [196, 4, 1.0], [197, 5, 1.0], [198, 5, 1.0], [199, 3, 1.0], [200, 1, 1.0], [201, 3, 1.0], [202, 7, 1.0], [203, 2, 1.0], [204, 1, 1.0], [205, 3, 1.0], [206, 0, 1.0], [207, 2, 1.0], [208, 8, 1.0], [209, 2, 1.0], [210, 7, 1.0], [211, 5, 1.0], [211, 6, 1.0], [212, 0, 1.0], [213, 1, 1.0], [213, 4, 1.0], [214, 4, 1.0], [215, 4, 1.0], [216, 2, 3.0], [216, 3, 1.0], [216, 4, 1.0], [216, 6, 4.0], [217, 0, 1.0], [218, 2, 1.0], [219, 7, 1.0], [220, 2, 1.0], [221, 1, 3.0], [221, 7, 4.0], [222, 3, 1.0], [223, 1, 1.0], [224, 6, 1.0], [225, 8, 4.0], [226, 6, 1.0], [227, 2, 2.0], [227, 4, 1.0], [228, 6, 1.0], [229, 1, 1.0], [230, 3, 1.0], [231, 1, 1.0], [232, 1, 1.0], [233, 1, 1.0], [234, 6, 1.0], [235, 7, 1.0], [236, 2, 2.0], [237, 5, 3.0], [237, 7, 1.0], [238, 7, 6.0], [239, 1, 1.0], [240, 7, 1.0], [241, 6, 1.0], [242, 3, 2.0], [243, 1, 1.0], [244, 0, 1.0], [245, 1, 1.0], [246, 2, 3.0], [246, 3, 10.0], [246, 4, 8.0], [246, 5, 1.0], [246, 6, 8.0], [247, 0, 1.0], [247, 1, 1.0], [247, 8, 1.0], [248, 2, 1.0], [248, 3, 1.0], [248, 4, 2.0], [249, 8, 1.0], [250, 3, 1.0], [251, 2, 1.0], [252, 7, 1.0], [253, 2, 1.0], [254, 0, 1.0], [255, 2, 1.0], [255, 4, 1.0], [256, 7, 1.0], [257, 7, 1.0], [258, 4, 1.0], [259, 7, 1.0], [260, 6, 1.0], [261, 2, 1.0], [261, 4, 1.0], [262, 2, 2.0], [263, 1, 1.0], [264, 1, 1.0], [265, 1, 1.0], [266, 0, 1.0], [267, 0, 2.0], [267, 3, 1.0], [268, 0, 2.0], [268, 1, 1.0], [268, 2, 2.0], [269, 1, 1.0], [270, 0, 1.0], [271, 0, 2.0], [271, 4, 1.0], [271, 5, 2.0], [272, 5, 3.0], [273, 1, 1.0], [273, 3, 5.0], [273, 6, 2.0], [274, 3, 1.0], [275, 5, 1.0], [276, 8, 1.0], [277, 3, 1.0], [277, 5, 4.0], [278, 8, 1.0], [279, 5, 1.0], [279, 8, 1.0], [280, 1, 1.0], [281, 1, 1.0], [282, 6, 1.0], [282, 8, 3.0], [283, 1, 1.0], [284, 8, 1.0], [285, 6, 1.0], [286, 6, 1.0], [287, 6, 1.0], [288, 7, 1.0], [289, 1, 3.0], [289, 2, 2.0], [289, 3, 1.0], [289, 4, 1.0], [289, 6, 1.0], [289, 7, 5.0], [289, 8, 1.0], [290, 1, 1.0], [291, 1, 1.0], [292, 4, 1.0], [293, 0, 1.0], [293, 1, 3.0], [293, 7, 3.0], [293, 8, 6.0], [294, 0, 3.0], [294, 1, 2.0], [294, 3, 1.0], [294, 4, 1.0], [294, 6, 11.0], [294, 8, 14.0], [295, 2, 1.0], [296, 3, 1.0], [297, 6, 1.0], [298, 2, 1.0], [299, 2, 1.0], [300, 2, 1.0], [301, 2, 1.0], [302, 8, 1.0], [303, 2, 2.0], [303, 6, 13.0], [303, 8, 2.0], [304, 0, 1.0], [304, 2, 1.0], [305, 2, 1.0], [306, 3, 1.0], [307, 6, 2.0], [308, 0, 3.0], [308, 2, 1.0], [308, 5, 2.0], [308, 7, 1.0], [309, 7, 1.0], [310, 8, 1.0], [311, 0, 1.0], [311, 2, 2.0], [311, 3, 1.0], [311, 5, 3.0], [311, 6, 1.0], [311, 8, 2.0], [312, 8, 1.0], [313, 6, 1.0], [314, 3, 1.0], [315, 0, 1.0], [315, 1, 1.0], [315, 2, 1.0], [315, 3, 2.0], [315, 6, 2.0], [316, 8, 1.0], [317, 8, 2.0], [318, 0, 1.0], [319, 3, 6.0], [319, 8, 2.0], [320, 6, 1.0], [321, 0, 2.0], [321, 1, 7.0], [321, 8, 5.0], [322, 6, 1.0], [323, 2, 1.0], [323, 3, 1.0], [324, 6, 1.0], [325, 0, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2826", "matrix_type": "sparse", "shape": [326, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T16:24:01.347554", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

rt4 = """{"rows": [{"id": "0", "metadata": {}}, {"id": "2", "metadata": {}}, {"id": "4", "metadata": {}}, {"id": "5", "metadata": {}}, {"id": "6", "metadata": {}}, {"id": "7", "metadata": {}}, {"id": "8", "metadata": {}}, {"id": "9", "metadata": {}}, {"id": "10", "metadata": {}}, {"id": "11", "metadata": {}}, {"id": "13", "metadata": {}}, {"id": "14", "metadata": {}}, {"id": "15", "metadata": {}}, {"id": "16", "metadata": {}}, {"id": "17", "metadata": {}}, {"id": "19", "metadata": {}}, {"id": "20", "metadata": {}}, {"id": "21", "metadata": {}}, {"id": "22", "metadata": {}}, {"id": "24", "metadata": {}}, {"id": "25", "metadata": {}}, {"id": "26", "metadata": {}}, {"id": "27", "metadata": {}}, {"id": "29", "metadata": {}}, {"id": "30", "metadata": {}}, {"id": "31", "metadata": {}}, {"id": "32", "metadata": {}}, {"id": "33", "metadata": {}}, {"id": "34", "metadata": {}}, {"id": "35", "metadata": {}}, {"id": "37", "metadata": {}}, {"id": "38", "metadata": {}}, {"id": "39", "metadata": {}}, {"id": "40", "metadata": {}}, {"id": "41", "metadata": {}}, {"id": "42", "metadata": {}}, {"id": "43", "metadata": {}}, {"id": "45", "metadata": {}}, {"id": "46", "metadata": {}}, {"id": "47", "metadata": {}}, {"id": "48", "metadata": {}}, {"id": "50", "metadata": {}}, {"id": "51", "metadata": {}}, {"id": "53", "metadata": {}}, {"id": "54", "metadata": {}}, {"id": "56", "metadata": {}}, {"id": "59", "metadata": {}}, {"id": "60", "metadata": {}}, {"id": "61", "metadata": {}}, {"id": "62", "metadata": {}}, {"id": "63", "metadata": {}}, {"id": "64", "metadata": {}}, {"id": "66", "metadata": {}}, {"id": "67", "metadata": {}}, {"id": "68", "metadata": {}}, {"id": "69", "metadata": {}}, {"id": "70", "metadata": {}}, {"id": "71", "metadata": {}}, {"id": "72", "metadata": {}}, {"id": "73", "metadata": {}}, {"id": "74", "metadata": {}}, {"id": "75", "metadata": {}}, {"id": "76", "metadata": {}}, {"id": "77", "metadata": {}}, {"id": "82", "metadata": {}}, {"id": "83", "metadata": {}}, {"id": "84", "metadata": {}}, {"id": "87", "metadata": {}}, {"id": "88", "metadata": {}}, {"id": "89", "metadata": {}}, {"id": "90", "metadata": {}}, {"id": "91", "metadata": {}}, {"id": "92", "metadata": {}}, {"id": "93", "metadata": {}}, {"id": "94", "metadata": {}}, {"id": "95", "metadata": {}}, {"id": "96", "metadata": {}}, {"id": "97", "metadata": {}}, {"id": "98", "metadata": {}}, {"id": "99", "metadata": {}}, {"id": "101", "metadata": {}}, {"id": "104", "metadata": {}}, {"id": "105", "metadata": {}}, {"id": "107", "metadata": {}}, {"id": "108", "metadata": {}}, {"id": "111", "metadata": {}}, {"id": "112", "metadata": {}}, {"id": "113", "metadata": {}}, {"id": "114", "metadata": {}}, {"id": "115", "metadata": {}}, {"id": "116", "metadata": {}}, {"id": "117", "metadata": {}}, {"id": "118", "metadata": {}}, {"id": "119", "metadata": {}}, {"id": "120", "metadata": {}}, {"id": "121", "metadata": {}}, {"id": "123", "metadata": {}}, {"id": "124", "metadata": {}}, {"id": "126", "metadata": {}}, {"id": "128", "metadata": {}}, {"id": "129", "metadata": {}}, {"id": "131", "metadata": {}}, {"id": "134", "metadata": {}}, {"id": "135", "metadata": {}}, {"id": "136", "metadata": {}}, {"id": "137", "metadata": {}}, {"id": "138", "metadata": {}}, {"id": "139", "metadata": {}}, {"id": "141", "metadata": {}}, {"id": "142", "metadata": {}}, {"id": "143", "metadata": {}}, {"id": "144", "metadata": {}}, {"id": "145", "metadata": {}}, {"id": "146", "metadata": {}}, {"id": "148", "metadata": {}}, {"id": "149", "metadata": {}}, {"id": "150", "metadata": {}}, {"id": "151", "metadata": {}}, {"id": "153", "metadata": {}}, {"id": "154", "metadata": {}}, {"id": "156", "metadata": {}}, {"id": "157", "metadata": {}}, {"id": "159", "metadata": {}}, {"id": "160", "metadata": {}}, {"id": "161", "metadata": {}}, {"id": "162", "metadata": {}}, {"id": "163", "metadata": {}}, {"id": "164", "metadata": {}}, {"id": "165", "metadata": {}}, {"id": "166", "metadata": {}}, {"id": "167", "metadata": {}}, {"id": "168", "metadata": {}}, {"id": "169", "metadata": {}}, {"id": "173", "metadata": {}}, {"id": "174", "metadata": {}}, {"id": "175", "metadata": {}}, {"id": "176", "metadata": {}}, {"id": "177", "metadata": {}}, {"id": "179", "metadata": {}}, {"id": "180", "metadata": {}}, {"id": "181", "metadata": {}}, {"id": "182", "metadata": {}}, {"id": "183", "metadata": {}}, {"id": "184", "metadata": {}}, {"id": "185", "metadata": {}}, {"id": "186", "metadata": {}}, {"id": "188", "metadata": {}}, {"id": "189", "metadata": {}}, {"id": "191", "metadata": {}}, {"id": "192", "metadata": {}}, {"id": "194", "metadata": {}}, {"id": "195", "metadata": {}}, {"id": "196", "metadata": {}}, {"id": "198", "metadata": {}}, {"id": "199", "metadata": {}}, {"id": "200", "metadata": {}}, {"id": "201", "metadata": {}}, {"id": "202", "metadata": {}}, {"id": "203", "metadata": {}}, {"id": "204", "metadata": {}}, {"id": "205", "metadata": {}}, {"id": "206", "metadata": {}}, {"id": "210", "metadata": {}}, {"id": "211", "metadata": {}}, {"id": "212", "metadata": {}}, {"id": "213", "metadata": {}}, {"id": "214", "metadata": {}}, {"id": "216", "metadata": {}}, {"id": "217", "metadata": {}}, {"id": "219", "metadata": {}}, {"id": "220", "metadata": {}}, {"id": "221", "metadata": {}}, {"id": "222", "metadata": {}}, {"id": "223", "metadata": {}}, {"id": "224", "metadata": {}}, {"id": "225", "metadata": {}}, {"id": "226", "metadata": {}}, {"id": "227", "metadata": {}}, {"id": "228", "metadata": {}}, {"id": "229", "metadata": {}}, {"id": "230", "metadata": {}}, {"id": "231", "metadata": {}}, {"id": "232", "metadata": {}}, {"id": "233", "metadata": {}}, {"id": "235", "metadata": {}}, {"id": "236", "metadata": {}}, {"id": "237", "metadata": {}}, {"id": "238", "metadata": {}}, {"id": "239", "metadata": {}}, {"id": "240", "metadata": {}}, {"id": "241", "metadata": {}}, {"id": "242", "metadata": {}}, {"id": "244", "metadata": {}}, {"id": "245", "metadata": {}}, {"id": "246", "metadata": {}}, {"id": "247", "metadata": {}}, {"id": "248", "metadata": {}}, {"id": "249", "metadata": {}}, {"id": "250", "metadata": {}}, {"id": "251", "metadata": {}}, {"id": "252", "metadata": {}}, {"id": "253", "metadata": {}}, {"id": "254", "metadata": {}}, {"id": "255", "metadata": {}}, {"id": "256", "metadata": {}}, {"id": "257", "metadata": {}}, {"id": "258", "metadata": {}}, {"id": "259", "metadata": {}}, {"id": "260", "metadata": {}}, {"id": "262", "metadata": {}}, {"id": "263", "metadata": {}}, {"id": "264", "metadata": {}}, {"id": "265", "metadata": {}}, {"id": "268", "metadata": {}}, {"id": "269", "metadata": {}}, {"id": "270", "metadata": {}}, {"id": "271", "metadata": {}}, {"id": "272", "metadata": {}}, {"id": "273", "metadata": {}}, {"id": "274", "metadata": {}}, {"id": "276", "metadata": {}}, {"id": "277", "metadata": {}}, {"id": "278", "metadata": {}}, {"id": "280", "metadata": {}}, {"id": "282", "metadata": {}}, {"id": "283", "metadata": {}}, {"id": "284", "metadata": {}}, {"id": "285", "metadata": {}}, {"id": "288", "metadata": {}}, {"id": "289", "metadata": {}}, {"id": "291", "metadata": {}}, {"id": "292", "metadata": {}}, {"id": "293", "metadata": {}}, {"id": "296", "metadata": {}}, {"id": "297", "metadata": {}}, {"id": "298", "metadata": {}}, {"id": "299", "metadata": {}}, {"id": "300", "metadata": {}}, {"id": "301", "metadata": {}}, {"id": "302", "metadata": {}}, {"id": "303", "metadata": {}}, {"id": "304", "metadata": {}}, {"id": "306", "metadata": {}}, {"id": "308", "metadata": {}}, {"id": "309", "metadata": {}}, {"id": "310", "metadata": {}}, {"id": "311", "metadata": {}}, {"id": "314", "metadata": {}}, {"id": "315", "metadata": {}}, {"id": "316", "metadata": {}}, {"id": "317", "metadata": {}}, {"id": "318", "metadata": {}}, {"id": "321", "metadata": {}}, {"id": "323", "metadata": {}}, {"id": "325", "metadata": {}}, {"id": "326", "metadata": {}}, {"id": "329", "metadata": {}}, {"id": "330", "metadata": {}}, {"id": "331", "metadata": {}}, {"id": "332", "metadata": {}}, {"id": "333", "metadata": {}}, {"id": "334", "metadata": {}}, {"id": "335", "metadata": {}}, {"id": "336", "metadata": {}}, {"id": "339", "metadata": {}}, {"id": "340", "metadata": {}}, {"id": "341", "metadata": {}}, {"id": "342", "metadata": {}}, {"id": "343", "metadata": {}}, {"id": "345", "metadata": {}}, {"id": "347", "metadata": {}}, {"id": "349", "metadata": {}}, {"id": "350", "metadata": {}}, {"id": "352", "metadata": {}}, {"id": "353", "metadata": {}}, {"id": "354", "metadata": {}}, {"id": "355", "metadata": {}}, {"id": "356", "metadata": {}}, {"id": "357", "metadata": {}}, {"id": "358", "metadata": {}}, {"id": "359", "metadata": {}}, {"id": "360", "metadata": {}}, {"id": "362", "metadata": {}}, {"id": "365", "metadata": {}}, {"id": "368", "metadata": {}}, {"id": "369", "metadata": {}}, {"id": "370", "metadata": {}}, {"id": "372", "metadata": {}}, {"id": "373", "metadata": {}}, {"id": "374", "metadata": {}}, {"id": "375", "metadata": {}}, {"id": "376", "metadata": {}}, {"id": "378", "metadata": {}}, {"id": "379", "metadata": {}}, {"id": "380", "metadata": {}}, {"id": "382", "metadata": {}}, {"id": "383", "metadata": {}}, {"id": "384", "metadata": {}}, {"id": "385", "metadata": {}}, {"id": "387", "metadata": {}}, {"id": "388", "metadata": {}}, {"id": "389", "metadata": {}}, {"id": "390", "metadata": {}}, {"id": "391", "metadata": {}}, {"id": "392", "metadata": {}}, {"id": "393", "metadata": {}}, {"id": "394", "metadata": {}}, {"id": "395", "metadata": {}}, {"id": "396", "metadata": {}}, {"id": "397", "metadata": {}}, {"id": "398", "metadata": {}}, {"id": "399", "metadata": {}}, {"id": "400", "metadata": {}}, {"id": "403", "metadata": {}}, {"id": "405", "metadata": {}}, {"id": "406", "metadata": {}}, {"id": "407", "metadata": {}}, {"id": "408", "metadata": {}}, {"id": "409", "metadata": {}}, {"id": "410", "metadata": {}}, {"id": "411", "metadata": {}}, {"id": "412", "metadata": {}}, {"id": "413", "metadata": {}}, {"id": "414", "metadata": {}}, {"id": "415", "metadata": {}}, {"id": "416", "metadata": {}}, {"id": "417", "metadata": {}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 0, 1.0], [2, 3, 1.0], [3, 0, 1.0], [3, 1, 1.0], [4, 4, 1.0], [5, 3, 1.0], [6, 0, 1.0], [6, 1, 1.0], [6, 2, 1.0], [6, 4, 1.0], [7, 5, 1.0], [8, 3, 1.0], [9, 1, 1.0], [9, 3, 1.0], [10, 4, 1.0], [11, 1, 1.0], [11, 2, 1.0], [12, 7, 1.0], [13, 1, 1.0], [14, 8, 1.0], [15, 2, 1.0], [16, 8, 1.0], [17, 3, 1.0], [17, 4, 1.0], [18, 7, 1.0], [19, 1, 2.0], [20, 3, 1.0], [21, 1, 1.0], [22, 7, 1.0], [23, 8, 1.0], [24, 1, 1.0], [25, 8, 1.0], [26, 3, 1.0], [27, 8, 1.0], [28, 1, 1.0], [29, 1, 1.0], [30, 3, 1.0], [31, 7, 1.0], [32, 7, 2.0], [33, 6, 1.0], [34, 3, 1.0], [34, 7, 2.0], [35, 3, 1.0], [36, 7, 1.0], [37, 4, 10.0], [37, 5, 10.0], [38, 0, 4.0], [38, 1, 2.0], [38, 7, 1.0], [38, 8, 3.0], [39, 3, 1.0], [40, 7, 1.0], [41, 4, 1.0], [42, 5, 1.0], [43, 1, 1.0], [43, 3, 2.0], [43, 6, 2.0], [44, 0, 23.0], [44, 1, 4.0], [44, 8, 1.0], [45, 0, 4.0], [45, 1, 3.0], [45, 2, 1.0], [45, 4, 1.0], [45, 5, 1.0], [45, 6, 3.0], [45, 7, 7.0], [45, 8, 1.0], [46, 0, 1.0], [46, 1, 1.0], [46, 2, 7.0], [46, 3, 2.0], [46, 4, 2.0], [46, 5, 13.0], [46, 6, 1.0], [47, 3, 1.0], [48, 2, 1.0], [49, 7, 1.0], [50, 2, 1.0], [51, 6, 1.0], [52, 1, 1.0], [53, 3, 1.0], [54, 3, 1.0], [55, 6, 1.0], [56, 7, 1.0], [57, 6, 1.0], [58, 0, 2.0], [58, 1, 2.0], [58, 8, 1.0], [59, 8, 2.0], [60, 1, 4.0], [60, 5, 2.0], [60, 7, 2.0], [60, 8, 1.0], [61, 7, 4.0], [62, 4, 1.0], [63, 0, 1.0], [63, 1, 1.0], [64, 0, 13.0], [64, 1, 1.0], [64, 3, 1.0], [65, 7, 1.0], [66, 2, 1.0], [67, 3, 1.0], [68, 5, 1.0], [69, 1, 1.0], [70, 1, 1.0], [70, 2, 3.0], [70, 5, 1.0], [70, 6, 4.0], [71, 2, 1.0], [72, 2, 1.0], [73, 2, 1.0], [73, 3, 4.0], [73, 4, 1.0], [73, 6, 1.0], [74, 2, 1.0], [75, 4, 2.0], [76, 0, 1.0], [77, 8, 1.0], [78, 7, 1.0], [79, 1, 1.0], [79, 3, 1.0], [79, 8, 2.0], [80, 7, 1.0], [81, 5, 2.0], [82, 6, 1.0], [83, 6, 1.0], [84, 8, 3.0], [85, 7, 1.0], [86, 4, 1.0], [87, 5, 1.0], [88, 4, 1.0], [89, 0, 1.0], [90, 3, 1.0], [91, 3, 1.0], [91, 4, 1.0], [91, 7, 1.0], [92, 3, 1.0], [93, 2, 1.0], [94, 8, 1.0], [95, 0, 1.0], [95, 6, 1.0], [96, 2, 1.0], [96, 6, 2.0], [97, 0, 1.0], [98, 2, 1.0], [99, 4, 1.0], [100, 2, 2.0], [100, 4, 1.0], [100, 6, 2.0], [101, 5, 1.0], [102, 1, 1.0], [103, 2, 1.0], [104, 1, 1.0], [105, 8, 1.0], [106, 8, 1.0], [107, 4, 1.0], [108, 3, 1.0], [109, 1, 1.0], [110, 7, 1.0], [111, 2, 1.0], [112, 1, 1.0], [113, 1, 1.0], [113, 3, 6.0], [114, 1, 2.0], [114, 3, 1.0], [114, 4, 4.0], [114, 6, 7.0], [115, 0, 1.0], [116, 1, 1.0], [117, 0, 2.0], [117, 1, 4.0], [117, 8, 1.0], [118, 5, 1.0], [119, 0, 1.0], [120, 1, 1.0], [120, 3, 1.0], [121, 8, 1.0], [122, 7, 1.0], [123, 7, 2.0], [124, 6, 1.0], [125, 6, 1.0], [126, 1, 1.0], [127, 0, 1.0], [127, 1, 1.0], [128, 2, 8.0], [128, 3, 1.0], [128, 4, 10.0], [128, 6, 1.0], [129, 2, 3.0], [129, 4, 5.0], [129, 5, 1.0], [130, 1, 1.0], [131, 5, 1.0], [132, 2, 5.0], [132, 4, 1.0], [132, 6, 3.0], [133, 0, 1.0], [134, 5, 1.0], [135, 0, 1.0], [135, 8, 1.0], [136, 8, 1.0], [137, 4, 1.0], [137, 5, 6.0], [138, 7, 1.0], [139, 3, 1.0], [140, 5, 5.0], [140, 7, 1.0], [141, 0, 2.0], [141, 5, 3.0], [142, 3, 1.0], [143, 3, 1.0], [144, 7, 2.0], [145, 1, 1.0], [146, 0, 1.0], [146, 3, 1.0], [146, 7, 1.0], [147, 5, 1.0], [148, 4, 1.0], [148, 5, 1.0], [149, 2, 1.0], [150, 2, 1.0], [151, 2, 2.0], [152, 2, 1.0], [153, 6, 2.0], [154, 7, 1.0], [155, 7, 1.0], [156, 1, 1.0], [156, 8, 1.0], [157, 3, 4.0], [157, 4, 1.0], [157, 6, 1.0], [158, 2, 9.0], [158, 4, 17.0], [159, 7, 1.0], [160, 1, 1.0], [161, 2, 1.0], [162, 3, 1.0], [163, 8, 9.0], [164, 3, 1.0], [165, 8, 1.0], [166, 7, 4.0], [167, 2, 1.0], [168, 1, 2.0], [168, 5, 2.0], [169, 2, 1.0], [170, 6, 1.0], [171, 3, 2.0], [172, 3, 1.0], [173, 2, 1.0], [173, 7, 2.0], [174, 0, 1.0], [175, 3, 1.0], [175, 4, 1.0], [175, 6, 2.0], [176, 2, 1.0], [176, 4, 1.0], [177, 2, 1.0], [177, 3, 1.0], [177, 4, 1.0], [178, 3, 1.0], [178, 4, 1.0], [178, 6, 4.0], [179, 4, 2.0], [180, 7, 1.0], [181, 7, 2.0], [182, 3, 3.0], [182, 4, 1.0], [182, 5, 12.0], [182, 7, 13.0], [183, 1, 1.0], [184, 1, 1.0], [184, 8, 1.0], [185, 4, 1.0], [186, 7, 1.0], [187, 1, 1.0], [188, 4, 1.0], [189, 3, 1.0], [190, 4, 1.0], [191, 8, 1.0], [192, 4, 1.0], [193, 7, 1.0], [194, 0, 1.0], [194, 7, 3.0], [195, 1, 1.0], [196, 7, 1.0], [197, 8, 1.0], [198, 5, 1.0], [199, 4, 1.0], [200, 5, 1.0], [201, 5, 1.0], [202, 3, 1.0], [203, 1, 1.0], [204, 3, 2.0], [205, 3, 1.0], [206, 1, 1.0], [207, 2, 1.0], [208, 1, 1.0], [209, 4, 1.0], [210, 0, 1.0], [211, 2, 1.0], [212, 0, 1.0], [213, 5, 1.0], [213, 6, 1.0], [214, 0, 1.0], [215, 1, 1.0], [216, 2, 1.0], [216, 6, 1.0], [217, 4, 1.0], [218, 0, 1.0], [218, 2, 2.0], [218, 3, 2.0], [218, 6, 3.0], [219, 0, 1.0], [220, 7, 1.0], [221, 2, 1.0], [222, 1, 3.0], [222, 7, 3.0], [223, 1, 1.0], [224, 5, 1.0], [225, 8, 3.0], [226, 8, 1.0], [227, 8, 1.0], [228, 2, 1.0], [228, 4, 1.0], [229, 6, 1.0], [230, 7, 1.0], [231, 0, 2.0], [232, 1, 1.0], [233, 1, 1.0], [234, 1, 1.0], [235, 1, 1.0], [235, 7, 1.0], [236, 6, 1.0], [237, 3, 1.0], [238, 2, 1.0], [239, 7, 1.0], [240, 2, 1.0], [241, 5, 3.0], [241, 7, 1.0], [242, 2, 1.0], [242, 7, 3.0], [243, 7, 1.0], [244, 6, 1.0], [245, 3, 1.0], [246, 1, 1.0], [247, 2, 3.0], [247, 3, 8.0], [247, 4, 6.0], [247, 5, 1.0], [247, 6, 6.0], [248, 8, 1.0], [249, 8, 1.0], [250, 8, 1.0], [251, 2, 1.0], [251, 4, 1.0], [252, 2, 1.0], [253, 2, 1.0], [254, 0, 1.0], [255, 2, 1.0], [255, 4, 1.0], [256, 4, 1.0], [257, 7, 1.0], [258, 6, 1.0], [259, 4, 1.0], [260, 2, 1.0], [261, 7, 1.0], [262, 2, 1.0], [263, 7, 1.0], [264, 5, 2.0], [265, 1, 1.0], [266, 0, 1.0], [267, 0, 2.0], [267, 5, 1.0], [268, 1, 1.0], [268, 2, 1.0], [269, 0, 1.0], [270, 0, 1.0], [270, 4, 1.0], [270, 5, 1.0], [270, 8, 2.0], [271, 5, 3.0], [272, 1, 2.0], [272, 3, 4.0], [272, 6, 1.0], [273, 3, 1.0], [274, 5, 1.0], [275, 8, 1.0], [276, 3, 2.0], [276, 4, 1.0], [276, 5, 3.0], [276, 6, 1.0], [277, 8, 1.0], [278, 7, 1.0], [279, 1, 1.0], [280, 5, 1.0], [281, 1, 1.0], [282, 6, 1.0], [282, 8, 3.0], [283, 8, 1.0], [284, 6, 1.0], [285, 7, 1.0], [286, 1, 3.0], [286, 2, 1.0], [286, 3, 4.0], [286, 4, 1.0], [286, 7, 4.0], [287, 3, 2.0], [288, 1, 1.0], [289, 1, 1.0], [290, 7, 1.0], [291, 8, 1.0], [292, 0, 3.0], [292, 1, 2.0], [292, 7, 3.0], [292, 8, 3.0], [293, 0, 3.0], [293, 1, 4.0], [293, 3, 1.0], [293, 4, 2.0], [293, 6, 11.0], [293, 8, 19.0], [294, 2, 1.0], [295, 3, 1.0], [296, 6, 1.0], [296, 7, 1.0], [297, 2, 1.0], [298, 2, 1.0], [298, 4, 1.0], [299, 1, 1.0], [299, 2, 1.0], [300, 2, 1.0], [301, 2, 1.0], [302, 2, 1.0], [303, 8, 1.0], [304, 2, 2.0], [304, 5, 2.0], [304, 6, 14.0], [304, 8, 2.0], [305, 0, 1.0], [305, 2, 1.0], [306, 2, 1.0], [307, 3, 1.0], [308, 2, 1.0], [308, 6, 2.0], [309, 0, 2.0], [309, 1, 1.0], [309, 5, 7.0], [309, 6, 1.0], [309, 7, 1.0], [310, 7, 1.0], [311, 8, 2.0], [312, 0, 4.0], [312, 2, 2.0], [312, 5, 2.0], [312, 6, 2.0], [312, 8, 3.0], [313, 4, 1.0], [314, 3, 1.0], [315, 0, 1.0], [315, 1, 1.0], [315, 2, 1.0], [315, 3, 1.0], [315, 6, 2.0], [315, 8, 1.0], [316, 8, 1.0], [317, 8, 3.0], [318, 0, 1.0], [319, 3, 6.0], [319, 8, 2.0], [320, 0, 1.0], [320, 6, 1.0], [321, 7, 1.0], [322, 0, 1.0], [323, 0, 2.0], [323, 1, 8.0], [323, 8, 5.0], [324, 6, 1.0], [325, 3, 2.0], [326, 6, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2826", "matrix_type": "sparse", "shape": [327, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T16:24:01.431794", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

tree = """((((117:0.01623,(196:0.02706,306:0.01045)0.430:0.00015)0.914:0.00014,(314:0.02521,104:0.00015)0.465:0.01855)0.885:0.02154,407:0.02223)0.848:0.01046,(259:0.01763,185:0.02203)0.762:0.00511,((353:0.04271,(335:0.01062,(394:0.04443,169:0.00014)0.910:0.01046)0.671:0.00521)0.239:0.00014,((245:0.05025,((229:0.01061,95:0.00523)0.825:0.02163,(45:0.00543,(380:0.02839,(366:0.02249,((100:0.00528,411:0.00534)0.801:0.00586,((80:0.02709,62:0.00014)0.787:0.00524,4:0.01624)0.743:0.00448)0.385:0.00528)0.821:0.01601)0.407:0.00986)0.749:0.0047)1.000:0.00014)0.884:0.01045,(((404:0.02473,(321:0.05743,345:0.04116)0.770:0.01042)0.871:0.0175,(159:0.00512,(((((130:0.00524,223:0.02109)0.768:0.00506,53:0.00505)0.700:0.00015,((417:0.00455,(((309:0.04229,(332:0.00014,(152:0.03495,49:0.01132)0.757:0.00501)0.726:0.00689)0.484:0.01637,93:0.00014)0.782:0.00501,(128:0.01468,(254:0.00886,367:0.00882)0.813:0.00016)0.838:0.01501)0.773:0.00525)0.893:0.01078,(350:0.00518,(203:0.01589,7:0.03797)0.303:0.00015)0.778:0.00522)0.803:0.00528)0.893:0.01013,(388:0.03663,(110:0.02502,144:0.00823)0.883:0.02317)0.392:0.01286)0.340:0.00015,((251:0.03298,129:0.00595)0.743:0.02131,((187:0.01683,(192:0.04094,333:0.02254)0.819:0.01106)0.075:0.00016,(291:0.03997,374:0.02052)0.773:0.01301)0.706:0.00748)0.960:0.03217)0.827:0.00518)0.783:0.00016)0.968:0.01009,((((((331:0.0213,60:0.01357)0.425:0.01773,((109:0.02075,(317:0.07537,264:0.01319)0.577:0.02234)0.840:0.02594,(312:0.13538,173:0.0804)0.863:0.04059)0.906:0.04474)0.732:0.00594,(250:0.02106,253:0.01877)0.861:0.01467)1.000:0.08412,((37:0.02019,(284:0.01874,301:0.05561)0.553:0.01707)0.821:0.01394,(((391:0.01339,94:0.03006)0.863:0.01092,((141:0.02689,(90:0.01568,166:0.01544)0.845:0.01038)0.773:0.00831,(216:0.02563,288:0.04234)0.553:0.02164)0.778:0.00999)0.751:0.00694,((174:0.04134,(19:0.01184,((11:0.01042,(279:0.0051,(97:0.03227,172:0.00015)0.826:0.00518)0.773:0.00507)0.865:0.01734,(202:0.02027,(198:0.0444,(16:0.02344,297:0.01386)0.707:0.01593)0.900:0.01926)0.904:0.01639)0.717:0.00775)0.879:0.01642)0.783:0.01157,((((124:0.06649,((405:0.00158,89:0.0157)0.911:0.02924,(81:0.05513,((32:0.01302,(111:0.01924,418:0.01395)0.079:0.00502)0.876:0.01356,(67:0.01066,140:0.00015)0.894:0.01716)0.882:0.02725)0.951:0.03825)0.231:0.01554)0.841:0.01263,(((((41:0.0103,171:0.0168)0.841:0.01081,(278:0.01609,305:0.01125)0.772:0.00474)0.784:0.00577,((84:0.0344,(186:0.04377,142:0.03554)0.843:0.01608)0.912:0.02736,(372:0.02034,(287:0.03183,409:0.01693)0.856:0.01073)0.745:0.00582)0.571:0.00506)0.753:0.00517,((268:0.03304,(213:0.01051,382:0.01052)0.811:0.00998)0.952:0.02716,(9:0.08093,(52:0.03741,(359:0.02766,58:0.07021)0.869:0.02854)0.679:0.01388)0.836:0.01208)0.521:0.00015)0.899:0.01075,(((364:0.0264,66:0.01994)0.913:0.0207,(222:0.00015,10:0.02739)0.845:0.01067)0.860:0.01237,(61:0.03289,(((330:0.04019,351:0.02244)0.698:0.0276,(323:0.12995,(289:0.00535,(413:0.02178,139:0.01068)0.762:0.00528)0.942:0.00015)0.819:0.0102)0.938:0.00015,303:0.01596)0.858:0.01157)0.209:0.00476)0.954:0.02615)0.496:0.00014)0.832:0.00517,(((214:0.05163,230:0.07569)0.927:0.03951,((280:0.03044,(194:0.01081,(65:0.01809,(316:0.04535,(((387:0.00014,(30:0.02953,145:0.00015)0.831:0.00568)0.806:0.00587,(204:0.01043,346:0.00438)0.793:0.00431)0.960:0.03307,((343:0.01764,393:0.01827)0.727:0.00451,(127:0.02335,83:0.0179)0.759:0.00672)0.588:0.00574)0.643:0.00015)0.950:0.0228)0.748:0.00571)0.961:0.04287)0.925:0.03217,(241:0.01939,43:0.08019)0.857:0.02177)0.849:0.01507)0.829:0.01924,((((132:0.03373,(221:0.02342,352:0.04396)0.946:0.03287)0.888:0.01823,((373:0.10091,177:0.13147)0.751:0.00015,(((293:0.0161,86:0.00014)0.968:0.03626,102:0.04302)0.693:0.00897,(((123:0.01071,(209:0.00014,(376:0.0,156:0.0):0.00014)0.842:0.01038)0.945:0.0265,(210:0.00016,146:0.01595)0.549:0.01437)0.654:0.00218,((12:0.0484,8:0.02104)0.969:0.0424,(272:0.02828,(22:0.01045,29:0.02189)0.760:0.00525)0.749:0.00487)0.701:0.01121)0.821:0.01228)0.883:0.01772)0.724:0.00651)0.902:0.02042,(((((408:0.00014,(38:0.0526,78:0.00074)0.996:0.08306)0.998:0.10223,((76:0.08846,182:0.04382)0.958:0.0539,(410:0.0971,15:0.10716)0.678:0.01328)0.691:0.0071)0.897:0.03797,(((0:0.00016,292:0.02665)1.000:0.17598,(((236:0.08389,165:0.0064)0.425:0.03841,(328:0.07297,91:0.05669)0.981:0.09592)0.924:0.06437,(355:0.00016,55:0.0732)0.675:0.01105)0.763:0.07187)0.636:0.01026,((319:0.09501,(((200:0.02705,98:0.02199)0.983:0.0685,(24:0.06833,398:0.03572)0.511:0.02027)0.798:0.02205,((322:0.01463,39:0.05175)0.967:0.0826,((274:0.04802,18:0.02754)0.792:0.0172,((176:0.03312,(35:0.03377,17:0.0252)0.746:0.01061)0.939:0.03209,300:0.04129)0.818:0.01326)0.988:0.09877)0.924:0.06455)0.891:0.03516)0.560:0.03086,362:0.07093)0.915:0.04272)0.692:0.01378)0.979:0.07885,(((304:0.26243,(267:0.00968,118:0.08099)0.885:0.05434)0.524:0.05109,(371:0.07182,(73:0.0922,31:0.21665)0.855:0.03897)0.740:0.02336)0.929:0.05844,((220:0.0938,(21:0.01109,320:0.00014)0.968:0.07128)0.872:0.03879,(207:0.00418,327:0.03108)0.983:0.07498)0.770:0.01237)0.473:0.02492)0.802:0.01473,((((136:0.00661,153:0.00473)0.997:0.09292,((71:0.02198,(271:0.00016,(((168:0.03029,369:0.01489)0.765:0.01385,122:0.01842)0.806:0.01117,361:0.02448)0.835:0.01151)0.936:0.02566)0.911:0.028,(199:0.00914,239:0.08898)0.899:0.02838)0.827:0.02548)0.927:0.02861,(((((266:0.00016,(((383:0.02613,(226:0.02876,201:0.14721)0.798:0.02147)0.884:0.02148,((((360:0.00015,(337:0.02264,(414:0.0107,72:0.01132)0.862:0.01097)0.621:0.01705)0.998:0.19154,(131:0.00205,(334:0.01891,(232:0.00015,143:0.02891)0.846:0.01454)0.999:0.09463)0.978:0.1571)0.878:0.07516,(324:0.0258,(219:0.00015,224:0.03191)0.981:0.04417)0.807:0.0125)0.867:0.03328,(((164:0.05516,(235:0.05657,27:0.0195)0.806:0.01096)0.678:0.00115,((315:0.00529,113:0.05507)0.889:0.02149,(348:0.09547,((311:0.00508,163:0.00014)0.965:0.00015,(56:0.00015,(205:0.04314,101:0.0156)0.748:0.00014)0.902:0.01018)0.166:0.01624)0.845:0.02166)1.000:0.07732)0.801:0.01079,(5:0.0081,((((400:0.0051,((63:0.016,(396:0.04485,64:0.00015)0.879:0.01096)0.971:0.02743,181:0.03152)0.722:0.00016)0.907:0.01479,(((215:0.02059,54:0.00014)0.202:0.00016,(2:0.01021,103:0.02082)0.886:0.01027)0.994:0.03906,(3:0.04134,(((339:0.00014,(170:0.00507,59:0.00519)0.815:0.00508)0.335:0.00016,397:0.01549)0.690:0.00014,(269:0.01064,310:0.0052)0.866:0.01042)0.861:0.01459)0.870:0.01673)0.925:0.00014)0.880:0.01752,(225:0.02073,((249:0.05068,(246:0.02824,138:0.00448)0.961:0.02374)0.177:0.00015,75:0.05899)0.734:0.01674)0.832:0.01477)0.654:0.00602,(74:0.00488,(28:0.03622,((273:0.03289,390:0.00016)0.990:0.03262,363:0.01567)0.796:0.00459)0.869:0.00642)0.981:0.03561)0.824:0.01317)0.887:0.01496)0.779:0.00739)0.883:0.02152)0.702:0.01128,(((283:0.00014,399:0.0678)0.963:0.03671,((121:0.0375,96:0.00525)0.874:0.0219,154:0.07411)0.824:0.02578)0.927:0.03762,(260:0.10849,(((((120:0.02607,(155:0.06721,137:0.00495)0.974:0.02947)0.535:0.0036,242:0.04238)0.861:0.02023,(349:0.04521,(((((82:0.00014,341:0.06325)0.993:0.03755,379:0.00016)0.786:0.00014,234:0.06992)0.997:0.06111,69:0.00014)0.994:0.04534,211:0.00332)0.381:0.00588)0.481:0.00823)0.938:0.03703,157:0.07989)0.717:0.00692,57:0.00882)0.961:0.07989)0.916:0.05076)0.798:0.02675)0.998:0.096)0.929:0.03959,((285:0.00015,(108:0.00014,(149:0.07407,((344:0.01413,151:0.01634)0.736:0.00657,(134:0.02781,(212:0.02652,(378:0.00015,(238:0.04745,247:0.04301)0.703:0.00014)0.966:0.02148)0.015:0.00015)0.466:0.01024)0.756:0.00588)0.883:0.01526)0.969:0.03759)0.988:0.03167,370:0.00014)0.867:0.02372)0.998:0.08133,158:0.00015)0.922:0.03621,((358:0.01281,(20:0.02969,99:0.02372)0.985:0.06249)0.998:0.11815,(190:0.06903,(357:0.06998,356:0.00015)0.829:0.02015)0.858:0.03439)0.919:0.05706)0.807:0.02571,193:0.01711)0.882:0.02352)0.729:0.01709,(248:0.0579,((244:0.03253,(206:0.05079,(((262:0.01325,255:0.03775)0.835:0.01673,88:0.02516)0.336:0.00015,40:0.02313)0.090:0.01515)0.865:0.01872)0.925:0.03028,(263:0.0345,(36:0.0456,183:0.04272)0.933:0.04591)0.868:0.02359)0.930:0.04794)0.957:0.06027)0.972:0.06398)0.617:0.00519)0.724:0.00431,((277:0.01408,(227:0.04382,233:0.00014)1.000:0.09125)0.682:0.0064,((((((((14:0.00015,188:0.00525)0.779:0.00517,(257:0.01105,375:0.04818)0.740:0.00015)0.775:0.01082,(119:0.00016,(313:0.01642,347:0.00015)0.843:0.01076)0.674:0.0108)0.000:0.00016,(298:0.0345,((325:0.05824,(184:0.00016,(178:0.06962,(340:0.06024,(((175:0.08004,(270:0.02469,282:0.06081)0.944:0.04574)0.357:0.01243,(258:0.06066,((365:0.00016,((26:0.00534,70:0.03876)0.939:0.01609,(195:0.03389,(44:0.01056,(180:0.0222,386:0.02757)0.969:0.00014)0.868:0.02803)0.745:0.00533)0.343:0.01063)0.802:0.00638,125:0.0208)0.354:0.01817)0.372:0.00767)0.820:0.0178,(218:0.03567,147:0.00245)0.981:0.05918)0.768:0.0159)0.847:0.01215)0.768:0.00569)0.978:0.02296)0.000:0.00015,(237:0.03489,296:0.01051)0.925:0.01719)0.896:0.01736)0.515:0.0105)0.741:0.01842,((126:0.01118,179:0.02795)0.939:0.02231,(160:0.01105,(329:0.01155,(336:0.00015,34:0.01644)0.840:0.02223)0.949:0.02512)0.759:0.00599)0.000:0.00334)0.952:0.0329,384:0.01969)0.777:0.00942,133:0.04451)0.888:0.02052,(208:0.0403,(23:0.01161,((115:0.00837,46:0.08317)0.999:0.11513,(276:0.02789,162:0.02676)0.797:0.03037)0.884:0.03986)0.910:0.0309)0.796:0.0182)0.951:0.03291)0.958:0.03269)0.545:0.00537)0.583:0.00014)0.727:0.00654,(((((295:0.02551,(368:0.02676,47:0.03946)0.602:0.00136)0.826:0.00588,((((228:0.00015,((290:0.01639,354:0.01649)0.568:0.01074,197:0.0162)0.879:0.00016)0.924:0.00529,256:0.00519)0.880:0.00015,(231:0.01531,403:0.02714)0.818:0.00041)0.733:0.00505,(191:0.0051,(33:0.02234,377:0.03943)0.879:0.01609)0.882:0.01086)0.858:0.01046)0.759:0.00506,(77:0.02734,(114:0.02206,50:0.03351)0.705:0.00412)0.855:0.01095)0.861:0.01075,((48:0.01954,(389:0.00015,13:0.01579)0.776:0.00512)0.800:0.00644,(((107:0.00513,116:0.01096)0.872:0.01062,(150:0.01086,(401:0.00014,189:0.05119)0.390:0.01048)0.911:0.01611)0.887:0.01052,(307:0.00016,((395:0.01672,402:0.01628)0.732:0.00525,(((((243:0.01546,(265:0.03394,412:0.01126)0.542:0.01041)0.781:0.00609,(326:0.01055,167:0.0162)0.760:0.0052)0.814:0.0052,(294:0.05971,(308:0.01435,381:0.0377)0.751:0.00763)0.895:0.0161)0.987:0.00014,(112:0.03413,25:0.01612)0.905:0.00014)0.876:0.00015,(252:0.00015,342:0.01556)0.980:0.02642)0.502:0.00526)0.893:0.01027)0.907:0.01031)0.851:0.00014)0.868:0.01018)0.889:0.01626,(((106:0.00831,(302:0.03114,85:0.01229)0.901:0.02187)0.886:0.01264,((((275:0.01585,(217:0.00016,338:0.03721)0.892:0.03778)0.770:0.00488,(392:0.00516,(1:0.00527,(240:0.01873,(51:0.02977,281:0.0106)0.492:0.01443)0.777:0.00831)0.772:0.00526)0.778:0.00519)0.833:0.00503,(406:0.01023,416:0.00515)0.865:0.00015)0.799:0.01576,68:0.01626)0.749:0.00507)0.757:0.00518,(87:0.02794,161:0.00014)0.233:0.00511)0.904:0.0159)0.896:0.02029)0.683:0.02168)0.833:0.01463)0.781:0.00532)0.723:0.00556)0.820:0.01376,((92:0.03867,((318:0.01724,(415:0.02669,148:0.00014)0.736:0.00422)0.837:0.01019,(261:0.00516,385:0.00502)0.967:0.02157)0.000:0.00015)0.895:0.01312,(6:0.04864,(((135:0.00901,(286:0.06254,105:0.01165)0.753:0.01766)0.580:0.0171,79:0.01655)0.632:0.00016,42:0.00015)0.829:0.00928)0.759:0.00636)0.758:0.01044)0.646:0.00165,299:0.00508)0.922:0.00016)0.860:0.01021)0.765:0.00517)0.776:0.00522);"""


if __name__ == "__main__":
    main()