#!/usr/bin/env python
# File created on 07 Jul 2012
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso", "Jens Reeder"]
__license__ = "GPL"
__version__ = "1.7.0"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"
__status__ = "Release"

from shutil import rmtree
from glob import glob
from os.path import exists, join
from cogent.util.unit_test import TestCase, main
from cogent.util.misc import remove_files, create_dir
from qiime.parallel.pick_otus import (ParallelPickOtusUclustRef,
                                      ParallelPickOtusBlast,
                                      ParallelPickOtusTrie,
                                      greedy_partition)
from qiime.util import (get_qiime_temp_dir,
                        get_tmp_filename)
from qiime.test import initiate_timeout, disable_timeout
from qiime.parse import parse_otu_map

class ParallelPickOtusTests(TestCase):
    
    def setUp(self):
        """ """
        self.files_to_remove = []
        self.dirs_to_remove = []
        
        tmp_dir = get_qiime_temp_dir()
        self.test_out = get_tmp_filename(tmp_dir=tmp_dir,
                                         prefix='qiime_parallel_tests_',
                                         suffix='',
                                         result_constructor=str)
        self.dirs_to_remove.append(self.test_out)
        create_dir(self.test_out)
        
        self.refseqs1_fp = get_tmp_filename(tmp_dir=self.test_out,
                                            prefix='qiime_refseqs',
                                            suffix='.fasta')
        refseqs1_f = open(self.refseqs1_fp,'w')
        refseqs1_f.write(refseqs1)
        refseqs1_f.close()
        self.files_to_remove.append(self.refseqs1_fp)
        
        self.inseqs1_fp = get_tmp_filename(tmp_dir=self.test_out,
                                            prefix='qiime_inseqs',
                                            suffix='.fasta')
        inseqs1_f = open(self.inseqs1_fp,'w')
        inseqs1_f.write(inseqs1)
        inseqs1_f.close()
        self.files_to_remove.append(self.inseqs1_fp)
        
        initiate_timeout(60)

    
    def tearDown(self):
        """ """
        disable_timeout()
        remove_files(self.files_to_remove)
        # remove directories last, so we don't get errors
        # trying to remove files which may be in the directories
        for d in self.dirs_to_remove:
            if exists(d):
                rmtree(d)

class ParallelPickOtusUclustRefTests(ParallelPickOtusTests):

    def test_parallel_pick_otus_uclust_ref(self):
        """ parallel_pick_otus_uclust_ref functions as expected """
        
        params = {'refseqs_fp':self.refseqs1_fp,
          'similarity':0.97,
          'max_accepts':1,
          'max_rejects':8,
          'stepwords':8,
          'word_length':8,
          'stable_sort':True,
          'enable_rev_strand_match':True,
          'optimal_uclust':False,
          'exact_uclust':False,
          'stable_sort':True,
          'save_uc_files':True
        }
        
        app = ParallelPickOtusUclustRef()
        r = app(self.inseqs1_fp,
                self.test_out,
                params,
                job_prefix='PTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        otu_map_fp = glob(join(self.test_out,'*otus.txt'))[0]
        otu_map = parse_otu_map(open(otu_map_fp,'U'))
        # some basic sanity checks: at least one OTU per reference sequence
        self.assertTrue(len(otu_map[0])>5)
        self.assertEqual(set(otu_map[2]),set(['r1','r2','r3','r4','r5']))

class ParallelPickOtusBlastTests(ParallelPickOtusTests):

    def test_parallel_pick_otus_blast(self):
        """ parallel_pick_otus_blast functions as expected """
        
        params = {'refseqs_fp':self.refseqs1_fp,
          'similarity':0.97,
          'blast_db':None,
          'max_e_value':1e-10,
          'min_aligned_percent':0.50
        }
        
        app = ParallelPickOtusBlast()
        r = app(self.inseqs1_fp,
                self.test_out,
                params,
                job_prefix='BTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        otu_map_fp = glob(join(self.test_out,'*otus.txt'))[0]
        otu_map = parse_otu_map(open(otu_map_fp,'U'))
        # some basic sanity checks: at least one OTU per reference sequence
        self.assertTrue(len(otu_map[0])>5)
        self.assertEqual(set(otu_map[2]),set(['r1','r2','r3','r4','r5']))
        
class ParallelPickOtusTrieTests(ParallelPickOtusTests):

    def setUp(self):
        """
        """

        super(ParallelPickOtusTrieTests, self).setUp()
        seqs = [\
         ('s1 some description','ACGTAATGGT'),\
         ('s2','ACGTATTTTAATTTGGCATGGT'),\
         ('s3','ACGTAAT'),\
         ('s4','ACGTA'),\
         ('s5','ATTTAATGGT'),\
         ('s6','ATTTAAT'),\
         ('s7','AAATAAAAA')
        ]
        self.small_seq_path = get_tmp_filename(
            prefix='TrieOtuPickerTest_', suffix='.fasta')
        self.files_to_remove = [self.small_seq_path]
        f = open(self.small_seq_path, 'w')
        f.write('\n'.join(['>%s\n%s' % s for s in seqs]))
        f.close()
        
    def test_parallel_pick_otus_trie(self):
        """ parallel_pick_otus_trie functions as expected """
        
        params = {
            'jobs_to_start': 2,
            'prefix_length': 1
            }
        
        app = ParallelPickOtusTrie()
        r = app(self.small_seq_path,
                self.test_out,
                params,
                job_prefix='POTU_TEST_',
                poll_directly=True,
                suppress_submit_jobs=False)

        otu_map_fp = glob(join(self.test_out,'*otus.txt'))[0]
        otu_map = parse_otu_map(open(otu_map_fp,'U'))

        expected =( {(1, 2): 1,
                     (0, 0): 1,
                     (2, 4): 1,
                     (1, 3): 1,
                     (3, 6): 1,
                     (1, 1): 1,
                     (3, 5): 1} ,
                    ['s2', 's1', 's3', 's4', 's7', 's5', 's6'],
                    ['0', '1', '2', '3'])

        self.assertEqual(len(otu_map[0]), 7)
        self.assertEqual(otu_map, expected)

    def test_parallel_pick_otus_trie_raises(self):
        """ parallel_pick_otus_trie raises error"""
        
        params = {
            'jobs_to_start': 2,
            'prefix_length': 6
            }
        
        app = ParallelPickOtusTrie()
        self.assertRaises( ValueError, app ,
                           self.small_seq_path,
                           self.test_out,
                           params,
                           job_prefix='POTU_TEST_',
                           poll_directly=True,
                           suppress_submit_jobs=False)

class ParallelPickOtusFunctionTests(TestCase):
    
    def test_greedy_partition(self):
        """greedy_partition works as expected"""

        #(non) partition into one bucket
        obs_part, obs_levels = greedy_partition({'1':2,
                                                 '2':1,
                                                 '3':3 }, 1)
        self.assertEquals(obs_levels, [6])
        self.assertEquals(obs_part, [['3','1','2']])

        # two buckets
        obs_part, obs_levels = greedy_partition({'1':2,
                                                 '2':1,
                                                 '3':3}, 2)

        self.assertEquals(obs_levels, [3,3])
        self.assertEquals(obs_part, [['3'],['1','2']])

        # larger input
        obs_part, obs_levels = greedy_partition({'1':1, '2':2, '3':3,
                                                 '4':4,'5':5, '6':6}, 2)
        self.assertEquals(obs_levels, [11,10])
        self.assertEquals(obs_part, [['6','3','2'],['5','4','1']])


refseqs1 = """>r1
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
>r2
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>r3
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>r4
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
>r5
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
"""

inseqs1 = """>PC.634_1 FLP3FBN01ELBSX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
>PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
>PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_6 FLP3FBN01A55WZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_7 FLP3FBN01D7O1S orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>PC.634_8 FLP3FBN01CV5ZU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_9 FLP3FBN01DQ783 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_10 FLP3FBN01DDPFF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGGTACGCGTTACTC
>PC.634_11 FLP3FBN01CPD70 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_12 FLP3FBN01BBAE6 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_13 FLP3FBN01AWYZD orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.634_14 FLP3FBN01AM0P3 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
>PC.635_15 FLP3FBN01BO4IE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGTGATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGG
>PC.635_16 FLP3FBN01BPX14 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGCAAGTTGCATACGCG
>PC.356_17 FLP3FBN01DB5I7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.634_18 FLP3FBN01AK9OO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGTCAGGTTG
>PC.634_19 FLP3FBN01ANGF2 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_20 FLP3FBN01AF994 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTTCACCCGTCGCGG
>PC.634_21 FLP3FBN01AHXU8 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.593_22 FLP3FBN01APRWO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_23 FLP3FBN01C645E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCAGTCG
>PC.634_24 FLP3FBN01A2K0Q orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCAT
>PC.634_25 FLP3FBN01DJY30 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTGCCGCTGCACCATGCGGTGCTGTGGCCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACG
>PC.355_26 FLP3FBN01EB63O orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.634_27 FLP3FBN01DUWHJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCGGAATCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_28 FLP3FBN01ESJ5R orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_29 FLP3FBN01BJH6S orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTCCGAGCGATATAATCTTTGATACCAAAGGGATGTCCCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGTAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.634_30 FLP3FBN01BF988 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.634_31 FLP3FBN01DRASA orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_32 FLP3FBN01EMWGJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCG
>PC.634_33 FLP3FBN01BMERW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.634_34 FLP3FBN01A724Q orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_35 FLP3FBN01AKXJC orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAGCCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTATCCACGCGTT
>PC.634_36 FLP3FBN01EC04T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_37 FLP3FBN01AX7NV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_38 FLP3FBN01CS9UE orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGACAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCAGGTT
>PC.634_39 FLP3FBN01A38B0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_40 FLP3FBN01CB1XF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_41 FLP3FBN01ARU00 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCAGACGCGGGGCCATCTCATACCACCGGAGTTTTTCACACCGTGTCATGCGACACTGTGCGCTTATGCGGCATTAGCAGTCATTTCTAACTGTTATTCCCCTGTATGAGGCAGGTTCCCCACGCGTTACT
>PC.634_42 FLP3FBN01A24G3 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.593_43 FLP3FBN01AYGTR orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTATTTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_44 FLP3FBN01DPZMM orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTCTGGATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAACTATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATATGCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTTACTCAGCCGTCCG
>PC.634_45 FLP3FBN01AJ7M7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCCCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_46 FLP3FBN01DV3TK orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCG
>PC.634_47 FLP3FBN01AL42P orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGCAAGTTGCATACGCGTTACTC
>PC.634_48 FLP3FBN01BYPDD orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_49 FLP3FBN01CTMCP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACCGGGGCATGCGCCCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_50 FLP3FBN01BAJ58 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTTCGAGCGATAAAATCTTTGATACCAGCAGGATGTCCTCCCGGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.634_51 FLP3FBN01A4K2C orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_52 FLP3FBN01DRBCR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTTCTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAT
>PC.634_53 FLP3FBN01BWH0G orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGGTGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.634_54 FLP3FBN01D0JVY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCACGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.634_55 FLP3FBN01EDR9T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGGACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_56 FLP3FBN01CDTMI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_57 FLP3FBN01AFERM orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGG
>PC.634_58 FLP3FBN01BFQFB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.635_59 FLP3FBN01D3JOZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGCTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.634_60 FLP3FBN01DVIVE orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCCCTCAGGGAGTGCACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGACAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_61 FLP3FBN01EQVOU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGAGCCATTACCTCACCAACTAACTAATACGCCGCGGGATCATCTAAGAGCGACAGCAAAGCCGTCTTTCATTCATGAACCATGCGGTTCGCGAATCTATGCGGTATTAGCATCCGTTTCCGAATGTTATCCCCCACTCCTAGGCAGATTCCCC
>PC.634_62 FLP3FBN01DH4LG orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_63 FLP3FBN01EEU42 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_64 FLP3FBN01AE4G6 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_65 FLP3FBN01A559N orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCTTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.355_66 FLP3FBN01BYAA8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_67 FLP3FBN01CSQSY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCTAGG
>PC.634_68 FLP3FBN01DTXOJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.481_69 FLP3FBN01EJD3A orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTCCATGCGAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGGCAGGTTGCCCACGCGTT
>PC.634_70 FLP3FBN01CIW8K orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTCTCTCAGTTCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGCGATCCCGTCATTTCATGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAAGG
>PC.481_71 FLP3FBN01BYVGX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGGCTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCAGGTTGCCCACGCG
>PC.635_72 FLP3FBN01BKL5B orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCGATTAGACGCCGTTTCCTAGCGATTATTCCCCTCGATGAGGGCAAGTTGCTCACG
>PC.634_73 FLP3FBN01DX9GS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.593_74 FLP3FBN01BVHTK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAACAACCTAATGGAACGCATCCCCATCCTTTACCGGAATCCTTTAATAATGAAACCATGCGGAATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAG
>PC.607_75 FLP3FBN01A9PL2 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAAGAAGATGCCTCCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_76 FLP3FBN01C3PFP orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACTCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAACTGTTATTCCCCACTCCAAGGCAGGTTACT
>PC.634_77 FLP3FBN01A17GS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCGACCCAGATATATTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCG
>PC.634_78 FLP3FBN01DM0XL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACCCGTCATCGCCTTGGTGAGCCGTTACCTCACCAACTAACTAATAGGCCGCGAGCCCATCCCCAAGCGCATTGCTGCTTTAATCTTTCGATACTATGCGGTATTAATCCCAGTTTCCCAGGGCTATCCCGCTCTCGGGGGCAGGTTACTCACGTGTTACTCACCCGTGCGCC
>PC.634_79 FLP3FBN01DFQEI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCATTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCGGGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCGCCGGTCCG
>PC.607_80 FLP3FBN01DN5MU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGGGTCGCGAGCCCATCCTATACCGATAAATCTTTTACCCCTGTATCATGCGATACTGTGGTCTTATACCGTATTAGCACAAATTTCTCTGTGTTATCCCGTTGTATAGGGCAGGTTGCTCACGTGTTACTCACCCGTTCG
>PC.607_81 FLP3FBN01B1L15 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCGCTCTATCAGCGATAGCCTCTCTCGAGTCCGCCACCTTTCCTTCTGCCATCATGCGATGACTGAACCTTATGCGGTATTAGCACTGATTTCTCATTGTTATTCCCCT
>PC.634_82 FLP3FBN01AW01I orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_83 FLP3FBN01BI38I orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_84 FLP3FBN01DSC1C orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGG
>PC.355_85 FLP3FBN01AIELL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_86 FLP3FBN01D1T0J orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_87 FLP3FBN01EOHM5 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTCACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCT
>PC.634_88 FLP3FBN01BMLLI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCGGGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.634_89 FLP3FBN01D74QJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCGG
>PC.634_90 FLP3FBN01EAAV0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_91 FLP3FBN01BC3GH orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_92 FLP3FBN01D9A6M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_93 FLP3FBN01A8V4Q orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTCCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.634_94 FLP3FBN01DCU7E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_95 FLP3FBN01AM688 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_96 FLP3FBN01DMRPA orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCATCCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.607_97 FLP3FBN01BAZV6 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGAGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCGCCGGCGGAGCAAAG
>PC.634_98 FLP3FBN01DEBKQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGCAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.634_99 FLP3FBN01BAWOL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTTCGCCACTCGGGCAC
>PC.636_100 FLP3FBN01D7AVV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAGTGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_101 FLP3FBN01B5T3T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGGTGGGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGTCG
>PC.635_102 FLP3FBN01B02WO orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_103 FLP3FBN01B2DRT orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCACTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTATTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.634_104 FLP3FBN01A8KH9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_105 FLP3FBN01AU6ML orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_106 FLP3FBN01AJ4R9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.481_107 FLP3FBN01B32LN orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_108 FLP3FBN01COYKA orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTGGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.593_109 FLP3FBN01C4BE9 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.634_110 FLP3FBN01BU1CU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAACTAATCAGACGCAGGCCCATCCCTTACCGATAAATCTTTCACATCCCTGATATGCATCAGACTTGCATTATGAGGTTTTAATCCCGGTTTCCCGAGGCTATCCCTCTGTAAGGGGCAGGTTGCCTACGCGTTACTCACCCGTCCGCCG
>PC.634_111 FLP3FBN01AYTJU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_112 FLP3FBN01DXJGO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_113 FLP3FBN01BOOHE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACGTATCGTTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATACGGCGCGGGTCCATCTATAAGTGACAGCCGAAACCGTCTTTCAACATTGAACCATGCGGTTCAATATATTATCCGGTATTAGCCCCGGTTTCCCGGAGTTATCCCAGTCTTATAGGTAGGTTACCCACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_114 FLP3FBN01BJL13 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTTCCAGTTCCAGTGTGACTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACCAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCACATAAGATAGATATGCAGCGTATCCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAACTTCTTGG
>PC.634_115 FLP3FBN01CTRWB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCCTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGACCATGCAGTCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACT
>PC.636_116 FLP3FBN01BJCG1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.481_117 FLP3FBN01CORWL orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_118 FLP3FBN01C68BF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_119 FLP3FBN01B449W orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGAAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCACTAACTCATACAT
>PC.607_120 FLP3FBN01EZ4SI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTTCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCCGGTGTGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.634_121 FLP3FBN01DD5NY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCC
>PC.355_122 FLP3FBN01BHKSP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.634_123 FLP3FBN01DHYS5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_124 FLP3FBN01D837F orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_125 FLP3FBN01DC9HR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_126 FLP3FBN01DLDHU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTTACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_127 FLP3FBN01D6Q98 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_128 FLP3FBN01CNKON orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTTACAGGTAGGTTGCCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_129 FLP3FBN01BCWX5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCCTCCCTTTCGGAAGGTTCGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.635_130 FLP3FBN01EA91G orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_131 FLP3FBN01B06QS orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCTTCCACCACAAAAAGCTTTCAACCCAGAGAGATGCCTCTCCGAATTATATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTGAAAGGTAGGTTGCATACGCGTTACGCACCCGTCCGCCGGTCG
>PC.634_132 FLP3FBN01D0SFK orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_133 FLP3FBN01DVTEG orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGGTAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCAGGTTGCTCACGCGTTACTCACCCGTCCGCC
>PC.634_134 FLP3FBN01CECHO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGAGCTGCTACCTCACCTACTACCTGATGGGCCGCGACCCCATCCCAGACCGCAAGAGCTTTCATCCCTCCGCCATGCGGTGGAAGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTCTGGGGCAGGTTGGTCACGTGTT
>PC.634_135 FLP3FBN01ES2BT orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTAGTCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTCGCG
>PC.636_136 FLP3FBN01BJO3F orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.635_137 FLP3FBN01AMOUT orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.356_138 FLP3FBN01CNT1Q orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCGACCCAGATATATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCATCCAAGGCAGGTT
>PC.634_139 FLP3FBN01DR739 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTCTCTCAGTTCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGCGATCCCGTTCATTTCATTGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAAGG
>PC.634_140 FLP3FBN01BBDL9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCTTTTACTCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCTCAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACT
>PC.634_141 FLP3FBN01CLP3T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
ATGGACCGTGTCTCAGTTCCATTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATTACCTACGTGTTACTCACCCGTTCG
>PC.355_142 FLP3FBN01C1O95 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_143 FLP3FBN01DHLQN orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.634_144 FLP3FBN01BA44D orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.634_145 FLP3FBN01EHG6O orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTG
>PC.356_146 FLP3FBN01EE1C3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACCGGGCCATGCGGCCCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.634_147 FLP3FBN01AER1M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGCAGGTTCG
>PC.636_148 FLP3FBN01A1RXA orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.634_149 FLP3FBN01B5Z5E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.634_150 FLP3FBN01EB7G2 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTTCCAGTTCCAGTGTGACTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACCAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCACATAAGATAGATATGCAGCGTATTCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAACTT
>PC.607_151 FLP3FBN01AYGA0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCCTGTCCGGCCGAAGCCTTTCCTGCCTCCGGGATGCCCCGGTGGCATGTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.481_152 FLP3FBN01AL8JX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGGGCATGCGCCCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_153 FLP3FBN01B7E8J orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.634_154 FLP3FBN01D6BHF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.634_155 FLP3FBN01BLTKB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.636_156 FLP3FBN01EK2JX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.634_157 FLP3FBN01EF15V orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGCAAGTTGCATACGCGTTACTC
>PC.634_158 FLP3FBN01BB8KH orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACTAA
>PC.481_159 FLP3FBN01AMFGP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGAACATGTGTCCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_160 FLP3FBN01A6LEJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_161 FLP3FBN01BUCNS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_162 FLP3FBN01ET0T4 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_163 FLP3FBN01B0AX7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCATGTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_164 FLP3FBN01BM4QE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAAGTTGCATACGTGTTACTC
>PC.593_165 FLP3FBN01EPSX1 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACT
>PC.634_166 FLP3FBN01DZ97Y orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTAGTCTTGGTGGGCCGTTACCCCCCAACAAACTAATGGAACGCATCCCCATCTCATACCGAATTTCTTTAATATAAAACCCATGCGGGAAATATATGCTATCGGATATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTATGAGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCGG
>PC.634_167 FLP3FBN01BFDT5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_168 FLP3FBN01BKTRL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGGCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTTGGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGG
>PC.634_169 FLP3FBN01C9NT9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.634_170 FLP3FBN01BE65F orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_171 FLP3FBN01AY2QI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_172 FLP3FBN01BYYFZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCACGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_173 FLP3FBN01AFUF1 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.634_174 FLP3FBN01B7I8E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.356_175 FLP3FBN01EIDVP orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.634_176 FLP3FBN01DOR0M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTTGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>PC.634_177 FLP3FBN01DO8MU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_178 FLP3FBN01ATR31 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_179 FLP3FBN01CO4O9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCGTGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.356_180 FLP3FBN01CZUXO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCACGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_181 FLP3FBN01A9EX0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_182 FLP3FBN01BYSJT orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_183 FLP3FBN01AV5YV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_184 FLP3FBN01DU3OR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTACGGGCAGGTTGCCA
>PC.636_185 FLP3FBN01EWW35 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_186 FLP3FBN01A3LZ0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCG
>PC.634_187 FLP3FBN01D55K5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAGTCTTTGACAGCCAACCCCATGCGGGATAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGCAGATTG
>PC.636_188 FLP3FBN01CHNXI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGGTAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_189 FLP3FBN01EETLW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCCCTCAGGGAGTGTACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGCTATGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.634_190 FLP3FBN01B4E5A orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCGATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.356_191 FLP3FBN01ET99E orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACCACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGCGGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_192 FLP3FBN01BS7JZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>PC.634_193 FLP3FBN01DW7MZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_194 FLP3FBN01BKWR9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAACTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATTGCTACTTCATGCGAAGCTGCAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGCAGGTT
>PC.636_195 FLP3FBN01DLUYP orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_196 FLP3FBN01DBVV5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.607_197 FLP3FBN01B5DKJ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTATACCACCGGAGTTTTTCACACCGGAGCATGCGCTCCTGTGCGCTTATGCGGTATTAACAGTCGTTTCCAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.634_198 FLP3FBN01ASQES orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_199 FLP3FBN01DPNWR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_200 FLP3FBN01DY5C2 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTTGCACCATGCGGTGCTGTGCGCTTATGCGGTATTAGCAGTCCATTTCTAACTGTTACTCCCCTGTATAAGGCAGGTTACCACGCGTTACTCACCCG
>PC.634_201 FLP3FBN01BLIY9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTTACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_202 FLP3FBN01DQGLV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGGGCCACTGCCCCGCCTACTACCTGATGGGCCGCGACCCCATCCTGCACCGCCGGAGCTTTCATCCGCTCCCCATGCGGGGTGCGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTGCAGGGCAGGTTGGTCACGTGTTACT
>PC.354_203 FLP3FBN01CUQNU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTT
>PC.634_204 FLP3FBN01B0QGX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGGTGGCCGATCGCCCTCCTCAGGCCGGCTACCCATCGCAGGCTAGGTGGGCCGTTGCCCCGCCTACTACCTAATGGGCCGCGACCCCATCCCGCACCGTCGGAGCTTTCCTCCGTGGCGCATGCGCGCCTCGGAGAGTATCCGGTATTAGCCGCCGTTTCCGGCGGTTGTCCCGGGGTGCGGGGCAGGTTGGTCACGTGTTACTCACCCGTTCGCC
>PC.634_205 FLP3FBN01A3PGP orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_206 FLP3FBN01ALY2B orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.634_207 FLP3FBN01BT8QN orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTCTGGATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAACTATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATATGCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTTACTCAGCCGTCCG
>PC.634_208 FLP3FBN01DATE7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGCGGGGGGTTAACCTCTCAGTCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.635_209 FLP3FBN01A5XYK orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCACTGTACCATGCAGTACCGTGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_210 FLP3FBN01BAT5U orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_211 FLP3FBN01ECTYY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCATCCTCCCCCGATGAATCTTTCCTCCTCCGGAGATGCCTCCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_212 FLP3FBN01A2T1M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.634_213 FLP3FBN01CBDAE orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_214 FLP3FBN01D8Y20 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCGCCAACCAGCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.634_215 FLP3FBN01A6MXL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGATGTCTTGGTGGGCCGTTACCCCGCCAACAAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTTCTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_216 FLP3FBN01ED1E9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATTGGAACCAGATGCCTGATTCCATCGCATGGGGCATTAGACGCCGTTTCCAGCCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.356_217 FLP3FBN01AYIUY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAGACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCAGGTTATCCACG
>PC.634_218 FLP3FBN01A0UCW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>PC.636_219 FLP3FBN01CDJ70 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCTCTACCCCGCCAACTGGCTAATCAGACGCGGGCCCCTCCCATACCACTATTGTTTTTCACACAGAACCATGCGGTCCCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_220 FLP3FBN01EFY7W orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCT
>PC.634_221 FLP3FBN01AGI0O orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.356_222 FLP3FBN01CUN6D orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_223 FLP3FBN01EFPI1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_224 FLP3FBN01A8V8L orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAGGCCCATCCTTTGCCGATAAATCTTTGACCAAACAGCCATGTGACTATTTAGTATTATGAGGTTTTAATCATCGTTTCCAATGGCTATCCCTCTGCAAAGGGCAGGTTGCCTACGCGTTACGT
>PC.354_225 FLP3FBN01BZOEE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_226 FLP3FBN01EGX4W orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_227 FLP3FBN01D2AYU orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.634_228 FLP3FBN01BHVDZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGTAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.481_229 FLP3FBN01ED5UR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_230 FLP3FBN01CU37V orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATCCCCAAGGGATGTCCCTCAAGGATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.354_231 FLP3FBN01B86XL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.355_232 FLP3FBN01DJ1VW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_233 FLP3FBN01AKD0L orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACTCTCTCAAGCCGGCTACTGATCGTTGTCTTGGTAGGCCGTTACCCTGCCAACTAACTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGGCACAGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTGCCCACGCG
>PC.356_234 FLP3FBN01D6H8C orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.635_235 FLP3FBN01DQFJG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGGGCCTCTACCCCGCCAACTATCTAATCAGCCGCGAGTCCATCTCCCATCGGATTTCTCCTTTGACTATCCAAGGATGCCCTCAAATAGTATTATGCGGTATTAGCGTCCGTTTCCAGACGTTATCCCCCTTTGAAAGGTAGGTTACTCACGTGTTACTCACCCGTCCG
>PC.635_236 FLP3FBN01AGC0G orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_237 FLP3FBN01D0FVD orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_238 FLP3FBN01BG1KW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.356_239 FLP3FBN01BW4TT orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGACCGTATCTCAGTTCCAATGTGGCCGATCAGCCTCTCAGCTCGGCTATGCATCGTTGCCTTGGTAGGCCATTGCCCCACCAACTAGCTAATACACCGCAAGCTCATCCTAAGGTGAAGCAAACGCTTCTTTTAACATATCTAGTTATCTAGTTATGTATCATCCGGTATTAGCGTTCGTTTCCAAACGTTATCCCAGTCCCTAGGGTAGATTACCC
>PC.635_240 FLP3FBN01BOFGD orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCCCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCTCACGCGTTACTCACCCGTCCGCC
>PC.355_241 FLP3FBN01CWLZU orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.593_242 FLP3FBN01AMXMK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTTCACACTGTACCATGCAGTACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.354_243 FLP3FBN01BHNXZ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.356_244 FLP3FBN01DRT2O orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.607_245 FLP3FBN01ATU7A orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.607_246 FLP3FBN01DQMNE orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCAGGGGCATGCGCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_247 FLP3FBN01BGO68 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_248 FLP3FBN01EC56I orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCTCTTACCTCACCAACTAGCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTTCCACAAGGTCATGCGACCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACT
>PC.636_249 FLP3FBN01D9EK7 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTCACTCACGTGTTACT
>PC.356_250 FLP3FBN01BXX60 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACT
>PC.636_251 FLP3FBN01A208L orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_252 FLP3FBN01EBSVN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTCCGCCGCT
>PC.636_253 FLP3FBN01BDPN5 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.635_254 FLP3FBN01DF6NE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTTAGACCGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_255 FLP3FBN01BSM84 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.354_256 FLP3FBN01BJIH7 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.354_257 FLP3FBN01D8FO0 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_258 FLP3FBN01AGYWU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTT
>PC.481_259 FLP3FBN01D6UF4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAACCAACAGGGATGCCCCCGTCGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCG
>PC.635_260 FLP3FBN01AEB9M orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.355_261 FLP3FBN01CIT14 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGCCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_262 FLP3FBN01DZG2M orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_263 FLP3FBN01CN7JX orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGACTGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCACGGAGATGCCTCCGCGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCACTAACTCAAGG
>PC.636_264 FLP3FBN01B74TP orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTCCGCCACTAGGGCG
>PC.636_265 FLP3FBN01BXZ1T orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATACGGCGCGGGTCCATCTATAAGTGATAGCAAAACCATCTTTCACTTTAGAACCATGCGGTTCTAAATGTTATCCGGTATTAGCTCCGGTTTCCCGAAGTTATCCCAGTCTTATAGGTAGGTTACCCACGTGTTACTCACCCGTCCGCCGCTAAG
>PC.355_266 FLP3FBN01C6LRW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_267 FLP3FBN01AOR4O orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_268 FLP3FBN01CSZZD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATGCGCCGCATGCCCATCCTCCGCCGGAATCGCTTCCTTTTAACACCCGTGGGATGCCCCACAGATGTACCACGCGGTATTAGTCCGGATTTCTCCGGGTTATCCCCCTGCGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_269 FLP3FBN01DNS9D orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.356_270 FLP3FBN01EGW4M orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTCGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.636_271 FLP3FBN01C9TUA orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTATGGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCCAACGCGGACCCATCCTATGCCGCCTCAGCTTTTCACACCGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCCGTTTCCGGATGTTATCCCCCTGCATAGGGCAGGTTGTCCACGCGTTACTCACCCGTCCGCCG
>PC.635_272 FLP3FBN01DZ0UC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_273 FLP3FBN01D9E65 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTGCTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_274 FLP3FBN01ECWT9 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.607_275 FLP3FBN01AHH1K orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTATACCATGCGGTACTGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.636_276 FLP3FBN01DQLWU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.636_277 FLP3FBN01ALCMH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACTCACCCG
>PC.635_278 FLP3FBN01C4N17 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.607_279 FLP3FBN01B8F4K orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCG
>PC.481_280 FLP3FBN01CATW7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_281 FLP3FBN01B97H8 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGATTGCTCCTTTGACCGCCCCGGGATGTCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_282 FLP3FBN01BA934 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.636_283 FLP3FBN01A3YNV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGACCCATCCCGCACCGCATGCGCTTTCCGCGCGGCCCCATGCGGGGCCGTGCGCATATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGCGGGGTAGGTTGTCCACGCGTTACTCACCCG
>PC.481_284 FLP3FBN01BJOS4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_285 FLP3FBN01EP59G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.354_286 FLP3FBN01CTU4A orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_287 FLP3FBN01DD28G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_288 FLP3FBN01BW0YB orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.355_289 FLP3FBN01AZT44 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTTCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_290 FLP3FBN01AGMG5 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_291 FLP3FBN01ARUVC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_292 FLP3FBN01D86I8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.481_293 FLP3FBN01COOOS orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.355_294 FLP3FBN01D8J4C orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTTACT
>PC.636_295 FLP3FBN01BR61D orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTTTAATC
>PC.356_296 FLP3FBN01BF0UO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTCCTCCTTTCGCCCGGTCCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTCGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.354_297 FLP3FBN01DB7BE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCTCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCGTCTCGGCTTTTCACCCCGAACCATGCGGTTCTGTGTGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_298 FLP3FBN01AZSCT orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.636_299 FLP3FBN01D3EWI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_300 FLP3FBN01A55LW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCCGCACCGGATCGCTCCTTTGACCGCTCCCCCATGCAGAGGAACGGTGTCATGCCGTATTAGTCCGGATTTCTCCGGGTTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.355_301 FLP3FBN01DZ6P8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTCTCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.355_302 FLP3FBN01AEB2N orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_303 FLP3FBN01B9U1E orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.354_304 FLP3FBN01A77ZB orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_305 FLP3FBN01BCQ7B orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.635_306 FLP3FBN01D5TAZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.635_307 FLP3FBN01BX26A orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCTGTGCGCCAGTCGCCGG
>PC.481_308 FLP3FBN01DTX5C orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.354_309 FLP3FBN01BCTWA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.636_310 FLP3FBN01BA22V orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.636_311 FLP3FBN01A4UET orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_312 FLP3FBN01EII4M orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTCCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.607_313 FLP3FBN01DPG83 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.593_314 FLP3FBN01AYGIQ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_315 FLP3FBN01AGF7L orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCCTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_316 FLP3FBN01A4NV1 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGGAGTCCCCATGCAGGGTCCCCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAAACAGTCCGGGG
>PC.635_317 FLP3FBN01AZ963 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCTAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.481_318 FLP3FBN01DMBQY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGAATCATGCGATTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.636_319 FLP3FBN01A6SLO orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.635_320 FLP3FBN01CQHPF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.635_321 FLP3FBN01CSQG7 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.481_322 FLP3FBN01AMENY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCACTAAGGCCCGCGCCTTCCGGGT
>PC.481_323 FLP3FBN01D1PPR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGATACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCG
>PC.607_324 FLP3FBN01CPKVQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_325 FLP3FBN01BZK6L orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGTAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTCAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.356_326 FLP3FBN01DF6XB orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_327 FLP3FBN01CF2XK orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.355_328 FLP3FBN01CXD3D orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAGAT
>PC.355_329 FLP3FBN01CQGY3 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTCA
>PC.481_330 FLP3FBN01B52I9 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACTCACCCG
>PC.607_331 FLP3FBN01AS2O8 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCG
>PC.636_332 FLP3FBN01D7MLX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.607_333 FLP3FBN01EUKQQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAATCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_334 FLP3FBN01AYCPW orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTGTCTCGATGATGCCATCGAAACACATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACG
>PC.354_335 FLP3FBN01BDSSU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.636_336 FLP3FBN01ELLAQ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCACCAGAGATGCCTCCGGTGATTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCG
>PC.356_337 FLP3FBN01CEUYF orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCG
>PC.636_338 FLP3FBN01CI8Z2 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.481_339 FLP3FBN01ENO6X orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGGTTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.636_340 FLP3FBN01B33P1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.636_341 FLP3FBN01ALS1C orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCATCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCC
>PC.356_342 FLP3FBN01AY1NE orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGTCAGTACCATGCGGTACCAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_343 FLP3FBN01D9HWD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGGTT
>PC.593_344 FLP3FBN01B0EIF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCAGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAGCCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTATCCACG
>PC.354_345 FLP3FBN01BPI1O orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_346 FLP3FBN01BBDRA orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.354_347 FLP3FBN01C3FYG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.356_348 FLP3FBN01DCLBL orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGACCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCTTGGGATGTCCCTAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.607_349 FLP3FBN01D7ZOI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.636_350 FLP3FBN01C0173 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.635_351 FLP3FBN01DPEUG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCTCCGGAGATGCCTCCGGAGATTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTT
>PC.636_352 FLP3FBN01B7KTL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTC
>PC.607_353 FLP3FBN01CXEBD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_354 FLP3FBN01BVDVK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.636_355 FLP3FBN01CMYHR orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.356_356 FLP3FBN01D6N8J orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_357 FLP3FBN01COUSC orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.481_358 FLP3FBN01AZPBJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.354_359 FLP3FBN01A8PO2 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_360 FLP3FBN01CVZXE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.607_361 FLP3FBN01D1IC0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCACCAACCAGCTAATCAGACGCGGGCCCATCTTGCACCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACT
>PC.355_362 FLP3FBN01AF0HW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAGG
>PC.593_363 FLP3FBN01DS5AU orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.356_364 FLP3FBN01DVSSP orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.635_365 FLP3FBN01AYYJN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTGGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGACCCATCGTGTACCGTACTAGATAAGATCTAGGCTTTCCGCCCTGTGCCATGCGGCACTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTACACGGCAGGTTG
>PC.607_366 FLP3FBN01BQL2S orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCATGTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_367 FLP3FBN01C6OIE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCATACGG
>PC.635_368 FLP3FBN01D8PY9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGGCGTCCAGAGAGATGCCTCCCAGACGCATCATGCGGTATTAGCGGCTGTTTCCAACCGTTATTCCCCACTCCAAGG
>PC.635_369 FLP3FBN01BRKZB orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.593_370 FLP3FBN01EHLMD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.354_371 FLP3FBN01BOOJJ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.481_372 FLP3FBN01CO1SB orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAGGTTGCATACGTGTTACTC
>PC.355_373 FLP3FBN01D9QTP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCG
>PC.636_374 FLP3FBN01CRC8T orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.355_375 FLP3FBN01DMQWE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.355_376 FLP3FBN01EU78N orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_377 FLP3FBN01AN2IX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.481_378 FLP3FBN01AMBQN orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGGCACCGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.354_379 FLP3FBN01B8NID orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_380 FLP3FBN01A2TMU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.607_381 FLP3FBN01AELDH orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_382 FLP3FBN01ED2F6 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGACTGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCAAGGAGATGCCTCCATGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCCCCCGTGCGCCAGTCGCCG
>PC.355_383 FLP3FBN01EABDR orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.635_384 FLP3FBN01DIYZ3 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_385 FLP3FBN01APLDI orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCG
>PC.355_386 FLP3FBN01D0DDK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_387 FLP3FBN01CDWQ4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGCCCATCCTTGTCCGGATAAATCCTTTGATCGAATTCTCATGCGAGAACCCGATGTCACGCGGTATTAGACCGGATTTCTCCGGCTTATCCCCCTGACAAGGGTAGGTTGCATACGTGTTACTC
>PC.354_388 FLP3FBN01AGMY0 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.635_389 FLP3FBN01DQSWF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTCTCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_390 FLP3FBN01CXUVC orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGTCTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTAAAGGCCAGGTTACTTATGTATTACTCACCCGTTCGCCACTCGGGC
>PC.635_391 FLP3FBN01EFNNG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCTACCTCTCAGTACGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGAGCCCATCTTTAAGCGATAAATCTTTGATACACAAACCATGCGATTCATGTATATTATGCGGTATTAGCGGTCGTTTCCGACCGTTATCCCACTCTTAAAGGCAGGTTGCTC
>PC.607_392 FLP3FBN01EAC1O orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTCCGAGCGATAAATCTTTGATACCAAAGGGATGTCCCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.636_393 FLP3FBN01DO7JM orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACG
>PC.593_394 FLP3FBN01B5KK8 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_395 FLP3FBN01BDDUQ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.607_396 FLP3FBN01ENY6F orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCATCCCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGTGGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.636_397 FLP3FBN01B8DUH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCGCCG
>PC.481_398 FLP3FBN01AX3YR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGCACCATGCAGTACTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.635_399 FLP3FBN01BI664 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.355_400 FLP3FBN01A3VZS orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGGCTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.593_401 FLP3FBN01C65QI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGTGCTTTTACCACTCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCAGATTTCTCCTTTTCACACGAATACATGCGTCTTCGTGCGCTTATGCGGTTTTACCACCAGTTTCCCGATGCTATCCCCCTGTATGAGGCAGGTTG
>PC.607_402 FLP3FBN01AMRRA orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATCCATACCATGCGATACAGATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAGAGGTAGGTTCCTCACG
>PC.636_403 FLP3FBN01CSPWG orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.355_404 FLP3FBN01CMVBH orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.635_405 FLP3FBN01C8VGW orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.607_406 FLP3FBN01CMRIY orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.356_407 FLP3FBN01DHESU orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCACCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.636_408 FLP3FBN01AMVJR orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>PC.593_409 FLP3FBN01D9BZ0 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACTAGCTAATCAGACGCGGATCCATCGTATACCACCAGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCG
>PC.636_410 FLP3FBN01CZIJ2 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.356_411 FLP3FBN01A8P22 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGTCAGAGAGATGCCTCCCAACTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGCAG
>PC.354_412 FLP3FBN01DVDLD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_413 FLP3FBN01DYU35 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.355_414 FLP3FBN01AJSD6 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTATGCCGTTACCATACCAACTAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTTCGCCACTAAGA
>PC.593_415 FLP3FBN01B5GN0 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCCATAGCGATAAATCTTTGAAGAAAAAGACATGCGTCTCTTTCCTGCTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTCTATGGGGCAGGTTACCCACGCGTT
>PC.636_416 FLP3FBN01CIJW1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_417 FLP3FBN01D44U7 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.636_418 FLP3FBN01DTYVK orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGCCCATCCTTGTCCGGATAAATCCTTTGATCGAATTCTCATGCGAGAACCCGATGTCACGCGGTATTAGACCGGATTTCTCCGGCTTATCCCCCTGACAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.481_419 FLP3FBN01DKJQE orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGAC
>PC.607_420 FLP3FBN01D6990 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCCCATACCGCGAAAGCTTTCCATGCAGGGGCATGCGCCCCCGCATGTGCATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTACCCACGCGTT
>PC.593_421 FLP3FBN01A5MND orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGCTCCATGCGGAGCTGTGCGCTCATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTATCCACGCGTTACT
>PC.636_422 FLP3FBN01AFDPD orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.607_423 FLP3FBN01C9021 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.636_424 FLP3FBN01D6MDJ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.636_425 FLP3FBN01BD6JX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCGACCTCTCAGTCCGGCTACCCATCGCGGGCACGGTGGGCCGTTACCCCGCCGTCTACCTAATGGGCCGCGACCCCATCCCGCGCCGTCTCGGCTTTCCCGGCAGGGCCATGGGGCCCTCCCGGAGTATCCGGTATTAGCCGCCGTTTCCAGCGGTTGTCCCGGAGCGCGGGGCAGGTTGGTCACGTGTTACTCACCCGTTCGCCACTCTATGTCCGCCCGAGGGCGG
>PC.481_426 FLP3FBN01CKTYU orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACT
>PC.593_427 FLP3FBN01EIOYS orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.636_428 FLP3FBN01EMEL5 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTTCCAGTTCCAGTGTGACTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACCAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCACATAAGATAGATATGCAGCGTATTCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAACTT
>PC.354_429 FLP3FBN01ACUB7 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_430 FLP3FBN01BGEG7 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCA
>PC.636_431 FLP3FBN01DLFJ6 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_432 FLP3FBN01DDMH6 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCG
>PC.481_433 FLP3FBN01C7M7W orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_434 FLP3FBN01EFF35 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_435 FLP3FBN01D8F9Q orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCTTTACCCCGCCAACTAGCTAATCAGACGCGGATCCATCGTATACCACCAGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACG
>PC.356_436 FLP3FBN01APXMT orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTTGACCATGCAGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.356_437 FLP3FBN01DQIRW orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCCGTTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGGTTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.354_438 FLP3FBN01D1PYD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_439 FLP3FBN01A6R98 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGCTACTCACCCGTGCGCCGGTCG
>PC.635_440 FLP3FBN01DEJQB orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTTGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_441 FLP3FBN01CUA96 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.354_442 FLP3FBN01DFTRG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCTATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.354_443 FLP3FBN01CZE2X orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACAAGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_444 FLP3FBN01B1OGH orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATTCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.635_445 FLP3FBN01A00UU orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_446 FLP3FBN01DL2J4 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_447 FLP3FBN01D4O1Y orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAAAG
>PC.636_448 FLP3FBN01EOUWL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.636_449 FLP3FBN01EEO6R orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.356_450 FLP3FBN01BV60P orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.593_451 FLP3FBN01EGU6E orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_452 FLP3FBN01AF9TE orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGCTGTCTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGTCCATCCCGTACCACCGTGGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCG
>PC.607_453 FLP3FBN01BI23K orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCCACTAAT
>PC.356_454 FLP3FBN01BZKF3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCGGGTTCTCCACGCGTTACT
>PC.593_455 FLP3FBN01AX4MJ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTC
>PC.481_456 FLP3FBN01EENPI orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATCAGACGCGGACCCATCGTGTACCACTATCGTTTTCCGCCCTGTGCCATGCGGCACTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTACACGGCAGGTTGTCCACGCGTT
>PC.356_457 FLP3FBN01A2E42 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACT
>PC.607_458 FLP3FBN01A0QD0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.354_459 FLP3FBN01B3PY0 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGCTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.355_460 FLP3FBN01B533J orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_461 FLP3FBN01EZQR8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGGTGGAGGACCTTCCTCTCCAGAACCCCTAGACATCGTCGCCTTGGTCGGGCCGTTACCCCGCCAACTAGCTAATGTCACCGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAGAAGATGCCTCCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACGGGCAGATTGCATACGCGTTACTCACCCG
>PC.635_462 FLP3FBN01A2G7F orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATTCAAACCATGCGATTTAAATGTATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACG
>PC.354_463 FLP3FBN01CAN8M orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCGCCACT
>PC.593_464 FLP3FBN01EDLIV orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_465 FLP3FBN01AN9E6 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.354_466 FLP3FBN01AIA4D orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.636_467 FLP3FBN01DSMJS orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCGCCCGTGCG
>PC.636_468 FLP3FBN01DHCC5 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATTCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_469 FLP3FBN01CLZRI orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTAGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_470 FLP3FBN01CEU1Z orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCAGG
>PC.354_471 FLP3FBN01DRJEK orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGATCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.636_472 FLP3FBN01A7RFW orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.354_473 FLP3FBN01DYJU9 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.635_474 FLP3FBN01BLL5V orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCCAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCCTCCTCCGGAGATGCCTCCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCG
>PC.635_475 FLP3FBN01DUUQ8 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCTCAGTCACAAAAATCTT
>PC.635_476 FLP3FBN01EABFG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.481_477 FLP3FBN01CZPD2 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.636_478 FLP3FBN01BU66X orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACTCACCCG
>PC.607_479 FLP3FBN01EDUZ7 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.636_480 FLP3FBN01A5RCL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCATCCTCTCAGACCGGCTACTGATCGCAGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACATGAGCCCATCCTTCAGCGATAAATCTTTGATTTCAGGAAGATGCCTTCCCAAAATGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAGGGCAGGTTGCTC
>PC.635_481 FLP3FBN01AULX0 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGATGCATACGCGTTACG
>PC.481_482 FLP3FBN01BULUD orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACTGCGCCATGCGGCGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCGGTATACGGCAGGTTCTCCACGCGTTACT
>PC.635_483 FLP3FBN01EZAZQ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.481_484 FLP3FBN01EQG8B orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.356_485 FLP3FBN01D7715 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.356_486 FLP3FBN01EYHU5 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTCGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCCGCCG
>PC.481_487 FLP3FBN01BTVUO orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.356_488 FLP3FBN01B8NKS orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_489 FLP3FBN01BLVU7 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGACAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCAGGTTATCCACGCGTTACTCACCCG
>PC.481_490 FLP3FBN01AVPQ2 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCATCCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.354_491 FLP3FBN01EAFRV orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAGGTTGCATACGTGTTACTC
>PC.354_492 FLP3FBN01DQ4XV orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_493 FLP3FBN01EVPJE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_494 FLP3FBN01D1105 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.636_495 FLP3FBN01CZHBB orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.593_496 FLP3FBN01DWQWF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_497 FLP3FBN01CHR1X orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACT
>PC.607_498 FLP3FBN01EHASU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.636_499 FLP3FBN01ARODF orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGATACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACTCACCCCG
>PC.481_500 FLP3FBN01DM0L1 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.635_501 FLP3FBN01AMXVM orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGGCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGTTGGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCC
>PC.607_502 FLP3FBN01EN0R5 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACTAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCTGTAAAGGATGCTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCT
>PC.354_503 FLP3FBN01ANMU2 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCGCC
>PC.354_504 FLP3FBN01DH6AB orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_505 FLP3FBN01AMFQA orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCCAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCTTCCTCCGGTACGATGCCTCCGGAGGATATACGCGGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAGGTTGCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_506 FLP3FBN01ANJSA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_507 FLP3FBN01BAA4G orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_508 FLP3FBN01DZYQE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_509 FLP3FBN01EAKES orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGACCATCCTGTACCACCGGAGTTTTCCCTGCGCTGCCATGCGGCAGCGCAGGATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCAGGTTACTCACG
>PC.354_510 FLP3FBN01B495L orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_511 FLP3FBN01BS5XB orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.356_512 FLP3FBN01DRO6W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.636_513 FLP3FBN01B08NI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_514 FLP3FBN01BYZHF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.635_515 FLP3FBN01AVTON orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCTTGCA
>PC.636_516 FLP3FBN01BR8CU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCATACCTATGCGGTCTTAGCTACCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGGTTACTTA
>PC.354_517 FLP3FBN01C2EP9 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_518 FLP3FBN01EMTB4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.354_519 FLP3FBN01EYX93 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTATCAGGGTCATGCGGCCCCAATACTTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGCGTTACTCACCAG
>PC.481_520 FLP3FBN01AE4FR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCACTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_521 FLP3FBN01DX6WZ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_522 FLP3FBN01CHYG5 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGGCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCG
>PC.354_523 FLP3FBN01D75ZT orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCTTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAGACAG
>PC.355_524 FLP3FBN01B67MI orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCACTCCGAGGCAGGTTCCTCACGCGTTACTCACCCGTCCG
>PC.354_525 FLP3FBN01C1PP4 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCAGGGGCATGCGCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_526 FLP3FBN01CCNA1 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGGGCATGCACTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_527 FLP3FBN01EEO1S orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.481_528 FLP3FBN01B39O5 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTG
>PC.635_529 FLP3FBN01DQGNZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCTCTCTCAAGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGACCATCCTGTACCACCGGAGTTTTCCCTGCGCTGCCATGCGGCAGCGCAGGATTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTCCCCACGCGTT
>PC.607_530 FLP3FBN01DD2LQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCATTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCGGCGATAAATCTTTGATGTCCAGTGCCATGCGACACAGACATATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCAGG
>PC.607_531 FLP3FBN01CKK5R orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCC
>PC.636_532 FLP3FBN01C1EUH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGAAG
>PC.636_533 FLP3FBN01AXLO6 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_534 FLP3FBN01BAHPU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_535 FLP3FBN01B85QO orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTCCCTTTGGCCGCAGGGCCATGCAGCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.636_536 FLP3FBN01DOCE5 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACG
>PC.593_537 FLP3FBN01B8169 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_538 FLP3FBN01D5N1L orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_539 FLP3FBN01BKT0E orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_540 FLP3FBN01D0BE4 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCG
>PC.356_541 FLP3FBN01BSMR0 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCGTATACCACCTCAGTTTTTCACACTGCGCCATGCGCCGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTT
>PC.607_542 FLP3FBN01BYGK2 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTATGTATCGTCGCCTAGGTGAGCCTTTACCTCACCTACTAGCTAATACAACGCAGGTCCATCTGGTAGTGATGCAGTTGCATCTTTCAAACTTCAATCATGCAATCAAAGATATTATGCGGTATTAGCTATCGTTTCCAATAGTTATCCCCCGCTACCAGGCAGG
>PC.356_543 FLP3FBN01D868W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_544 FLP3FBN01EAMCO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCA
>PC.355_545 FLP3FBN01B0H89 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.355_546 FLP3FBN01CPQSL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTTGACTTGGTAGGCCATTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATTAGCGGATTTCTCCTTTCCCACTAGAACCATGTGGTTCCTATGGCATATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTAATAGGCAGGTTGCTTACG
>PC.355_547 FLP3FBN01DZX6B orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAACTATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTGGAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCTCTGATAGGTAGGTTGCATACGCGTTACTTCACCCGTCGCG
>PC.593_548 FLP3FBN01ARBPM orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCACCGGAGATGCCTCCGACGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_549 FLP3FBN01B1VVL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAAAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.636_550 FLP3FBN01BAWPK orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTGCCGCTGCACCATGCGGTGCTGTGGCCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCG
>PC.635_551 FLP3FBN01EUT65 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCCACCAACTAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTACCACTGCACCATGCGGCGCTGTGGTCTTATGCGGTATTAGCAGCCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.593_552 FLP3FBN01EPDN4 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_553 FLP3FBN01BSTU3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTGCCCCGCCAACTACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCTCACCATGCGGCAAAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGCGTT
>PC.356_554 FLP3FBN01DP84W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_555 FLP3FBN01A9328 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.356_556 FLP3FBN01A6SQ3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGGCTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCAGTGCGGGGCAGGTTGCCCACG
>PC.635_557 FLP3FBN01C6I0F orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.481_558 FLP3FBN01D1YSR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_559 FLP3FBN01DAS79 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGCGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>PC.356_560 FLP3FBN01CR7MF orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTGTCTCGATGATGCCATCGAAACACATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACG
>PC.635_561 FLP3FBN01D7F5H orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGATACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACTCACCCG
>PC.635_562 FLP3FBN01CSG40 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_563 FLP3FBN01BBORZ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGAGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGT
>PC.356_564 FLP3FBN01BY17V orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.607_565 FLP3FBN01AQTYD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.481_566 FLP3FBN01BGT75 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.635_567 FLP3FBN01AJFDA orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCGGCCGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGCTGCATCGGGATGCCCCATCGCAGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTATCCCCCTCTGAAAGGCAGGTTGCTCACG
>PC.636_568 FLP3FBN01D5SZS orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.354_569 FLP3FBN01EB4D8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGGGTCCATCTCTATCCGTCAAGGCTTTTCACACCGTGCCATGCGGCACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGATAGAGGCAGGTTACCCACGCG
>PC.607_570 FLP3FBN01A7HFD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_571 FLP3FBN01BZTX6 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGACAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.481_572 FLP3FBN01AQFZV orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAATCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.635_573 FLP3FBN01ESUQV orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_574 FLP3FBN01C1PAK orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGGTTGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATATCAGTGCCATGCGACTCCGATATTTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCG
>PC.635_575 FLP3FBN01DVCKT orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCTCTACCCCGCCAACTGGCTAATCGGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACTGTATCATGCGTTACTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGGTATCCCCCGGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_576 FLP3FBN01DR7VU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.635_577 FLP3FBN01CZ007 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGGCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTTGGTTGCACACGTGTTACTCACCCGTACGCCGGTCGCCACCAGTG
>PC.635_578 FLP3FBN01C903U orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.481_579 FLP3FBN01C68GY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.356_580 FLP3FBN01A0XKK orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAACAG
>PC.355_581 FLP3FBN01D0C70 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_582 FLP3FBN01DXGDW orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCGTACCTATGCGGTCTTAGCTACCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGGTTACTT
>PC.607_583 FLP3FBN01D37W3 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.636_584 FLP3FBN01BPYDP orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTT
>PC.636_585 FLP3FBN01COSQ8 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACAAGCTAATCAGACGCGAGCTCATCCATATCCGATAAATCTTTGATCAAACGAAGATGCCTTCAAATGATATTATGCGGTATTATCAGTCGTTTCCAACTGTTATCCCCCAGATATGGGCAGATTG
>PC.593_586 FLP3FBN01DSGMM orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.636_587 FLP3FBN01CDLEW orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_588 FLP3FBN01AHK73 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_589 FLP3FBN01DX2PW orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.607_590 FLP3FBN01BE4YH orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTCGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.481_591 FLP3FBN01A38MX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACCAGATCATGCGATCCTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACT
>PC.636_592 FLP3FBN01ALQ01 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.636_593 FLP3FBN01CE5R8 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGCGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAACTATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTGGAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCTCTGATAGGTAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.356_594 FLP3FBN01CDDGT orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.354_595 FLP3FBN01EMNUG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.607_596 FLP3FBN01BT0AE orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAGGCCCATCCTTTGCCGATAAATCTTTGACCAAACAGCCATGTGACTATTTAGTATTATGAGGTTTTAATCATCGTTTCCAATGGCTATCCCTCTGCAAAGGGCAGGTTGCCTACGCGTTACTCACCCG
>PC.481_597 FLP3FBN01D7CTS orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.481_598 FLP3FBN01CXKKK orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.593_599 FLP3FBN01CXUT9 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCTTCAGAGATGCCTCCAAAAGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_600 FLP3FBN01AR4YL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.354_601 FLP3FBN01A7NE9 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTTGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATCGGACGCGAGCCCATCTCCCAGCGGATTGCTCCTTTGATATATCTACCATGTGATAAATATATGTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGGAAGGCAGGTTGCTCACGTGTTACT
>PC.607_602 FLP3FBN01ELYL8 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGCTGCCCCGCCAACTGCCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_603 FLP3FBN01BKQF8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTCA
>PC.635_604 FLP3FBN01D4OI7 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGCTGCCCCGCCAACTGGCTAATCGGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTTACCCATGCGGGTATGTGCGCTTATGCGGTATTAGCAGCTGTTTCCAGCTGGTATCCCCCGGTACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.636_605 FLP3FBN01AUBJF orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATTGGACGCGAGCCCATCTCTCAGCGGATTTCTCCTTTGGTCCAGACAGGATGTCCCATCCGGACGTTATGCGGTATTAGCGATCGTTTCCAATCGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGCGTTACTCACCCGTCCG
>PC.635_606 FLP3FBN01AZSFJ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCG
>PC.593_607 FLP3FBN01C847I orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGCTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTT
>PC.355_608 FLP3FBN01DU20K orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGCAGGTTG
>PC.356_609 FLP3FBN01B5XY2 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGGCTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACG
>PC.607_610 FLP3FBN01C354R orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTT
>PC.354_611 FLP3FBN01C667Q orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTCTATCCGTCTCGGCTTTTCACACCGTGCCATGCGGCACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACT
>PC.481_612 FLP3FBN01DJCPD orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGTTAACCTCTCAGTCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCCATAACCACCGGAGTTTTCAATCAAAAGGGATGCCCCTCTTGATGTTATGGGATATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTATGGGTAGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.593_613 FLP3FBN01C1ONI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTTACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.635_614 FLP3FBN01ARRYC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.356_615 FLP3FBN01AKJ6W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_616 FLP3FBN01DD2QI orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCGTCACCCCGCCAACTAACTAATCGGACGCAGGCCCATCCCTCGCCGTTCCCTTTGGCCCCTCGAGCATGCGCTCCAAAGGCTTCATGGGGTCTTAATCCCGGTTTCCCGAGGCTGTCCCCCAGCGAGGGGCAGGTTGCCTACGCGTTACTCACCCGTCCGCCGCTTTCCGCCATCG
>PC.635_617 FLP3FBN01AOLSN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.636_618 FLP3FBN01AMKWI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGC
>PC.636_619 FLP3FBN01AFTMX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_620 FLP3FBN01DQAD5 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTGGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.607_621 FLP3FBN01CHB2R orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGCCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.635_622 FLP3FBN01EV5MZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGGCCGTCTACCCTCTCAGGCCGGCTACGTATCATCGCTTTGGTGGGCCGTTACCCGCCAACTGGCTAATACGCCGCAGGTCCATCCATGTCCACACCCGAAGGTCCTTTAATATAGTCAAGATGCCTTGTCTATACCTATCCGGTTTTAGCAGACGTTTCCATCTGTTATCCCGGGTACATGGGCAGGTTCCCTACGTGTTACTCACCCGTTCG
>PC.635_623 FLP3FBN01CQDN5 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCG
>PC.356_624 FLP3FBN01EBIVZ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGG
>PC.354_625 FLP3FBN01D06GF orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTATATGAGGCAGGTTACCCACGCGTT
>PC.593_626 FLP3FBN01DNQQE orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.481_627 FLP3FBN01CJ7ZY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATCCATACCATGCGATACAGATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAGAGGCAGGTTCCT
>PC.481_628 FLP3FBN01BW1QP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGAAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTTTTACTCACCCGTTCG
>PC.607_629 FLP3FBN01D9D5Q orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.607_630 FLP3FBN01D0IW2 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.481_631 FLP3FBN01B6DNW orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGGCCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.607_632 FLP3FBN01B54IG orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAAATCTTTGGCAGTCAGGGAGATGTCTCCCAACTGCTTCATGCGGTATTAGCAGATGTTTCCATCTGTTATTCCCCACTCCAAGGCAGGTT
>PC.635_633 FLP3FBN01DECHE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGAC
>PC.593_634 FLP3FBN01BDNJS orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAACAACCTAATGGAACGCATCCCCATCCTTTACCGGAATCCTTTAATAATGAAACCATGCGGAATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAG
>PC.635_635 FLP3FBN01DVZQC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTATGCCGTTACCATACCAACTAGCTAATCGGACGCAGGGCCATCCCTTACCACCTTAGTTTTTCACTCTGCCTCATGCGAGGCTGGGCGCTTATGCGGTATTAGCAGTTGTTTCCATCTGTTATCCCCCTGTAAGGGCCAGGTTCCCTACGCGTTACTCACCCGTCCG
>PC.636_636 FLP3FBN01EI5JL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGCCGGCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAACCAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCCGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.607_637 FLP3FBN01EPSYJ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGTCCGATCAGCCTCTCAGCTCGGATATGCATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATACAACGCAAGCCCATCTCAAAGTGAAGCAAATGCTCCTTTTAACGTATCTAGTTATCTAATTACGTATCATCTGGTATTAGCGTTCATTTCTAAACGTTATCCCAGTCTCTGAGGTAGGTTACCCACG
>PC.356_638 FLP3FBN01EOGU6 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.635_639 FLP3FBN01CL1LE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCTGG
>PC.356_640 FLP3FBN01DQSSH orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCG
>PC.355_641 FLP3FBN01CHYQT orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCCTATACCACCTCAGTTTTTCACACCGAACCATGCGGTTTTAGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAACAG
>PC.355_642 FLP3FBN01DI11Q orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCTACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.593_643 FLP3FBN01A3YSL orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAATTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_644 FLP3FBN01CUX8S orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGTCCATCCCATACCGTCTCGGCTTTCCCCTCCCGGCCATGCGGCCAAGAGGAATCATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_645 FLP3FBN01A4SET orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGTCCAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCCTCCTCCGGAGATGCCTCCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_646 FLP3FBN01BVAH3 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTAGGCCACTGCCCCACCTACTACCTGATGGGCCGCGACCCCATCTCACGCCGCGAAAGCTTTCATCCGCTCCTCATGCGAGGTGCGGATAATATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGCGTGAGGCAGGTTGGTC
>PC.354_647 FLP3FBN01AHKPL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.635_648 FLP3FBN01ET7S1 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGGCCCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>PC.356_649 FLP3FBN01DBO5X orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCACGTGCAAGGCAGGTTACCCACGCGTT
>PC.593_650 FLP3FBN01EPNQQ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGG
>PC.356_651 FLP3FBN01BXFJH orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.355_652 FLP3FBN01CP0TU orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_653 FLP3FBN01CTP94 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCTACCTCTCAGTACGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGATCCATCTCATACCGCCTACAGTCTTTTCACACCGGAAACATGCATCTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCCTAACTTGTTATCCCCTGTATGAGGCAGGTTAT
>PC.355_654 FLP3FBN01AMBXT orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTT
>PC.607_655 FLP3FBN01DRS0G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGATACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACTCACCCG
>PC.356_656 FLP3FBN01AXM4C orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.607_657 FLP3FBN01D82DP orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.635_658 FLP3FBN01ED2R2 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCG
>PC.607_659 FLP3FBN01B82L6 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.607_660 FLP3FBN01BW42C orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGTCCATCTGTTAGCGAATTGCTCCTTTCCCACTATTATCATGTGATAATTATGGCATATGCGGTATTAGCAATGATTTCTCATTGTTATTCCCCTCTAACAGGCAGGTTACTTACG
>PC.354_661 FLP3FBN01A86UW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTT
>PC.355_662 FLP3FBN01AJ5LD orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.593_663 FLP3FBN01DLI9X orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.635_664 FLP3FBN01BIKSR orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGATTTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCGAGAGCAGGTT
>PC.356_665 FLP3FBN01DUZ3A orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.354_666 FLP3FBN01BCBOG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCATACGG
>PC.481_667 FLP3FBN01AEVVE orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGAGCTCATCTTTGGACAATAAATCTTTGGCAGCAGGACCATGCGATCCCGTTGCATTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCATCCAAAGGCAGATTGCT
>PC.593_668 FLP3FBN01D5ZFA orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGG
>PC.356_669 FLP3FBN01BPHY3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCTCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCGTCTCGGCTTTTCACCCCGAACCATGCGGTTCTGTGTGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_670 FLP3FBN01B4FWO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTTTTACCTCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACATTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACT
>PC.354_671 FLP3FBN01AXQ2Z orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATATGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACGCG
>PC.354_672 FLP3FBN01ECMN2 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_673 FLP3FBN01EDX65 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.607_674 FLP3FBN01AP0N3 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>PC.354_675 FLP3FBN01EADWQ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.481_676 FLP3FBN01CEH6T orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.481_677 FLP3FBN01D28RJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGTCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACTCACCCG
>PC.354_678 FLP3FBN01AWE2B orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.593_679 FLP3FBN01B9JHP orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
>PC.356_680 FLP3FBN01DE1KU orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCAAGCCATGCAGCTCAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGCGTT
>PC.593_681 FLP3FBN01A9EJ1 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.481_682 FLP3FBN01A8MZH orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.481_683 FLP3FBN01ALP0V orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACAGAACCATGCGGTTCCATGCGCTTATGCGGTATTAGCAGTCATTTCTGGCTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.356_684 FLP3FBN01DMENX orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGGCTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.356_685 FLP3FBN01B4MLO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTCACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGTCG
>PC.636_686 FLP3FBN01AQ9FC orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_687 FLP3FBN01DVF2L orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTAACTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTCCTGCTGTACCATGCGGCACTGTAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACGCGTTACTCACCCG
>PC.636_688 FLP3FBN01CWQIO orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCTTTACCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTGTACCATGCGGTACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.635_689 FLP3FBN01BC6PA orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCCCTCAGGGAGTGCACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGCTATGGGCAGGTTGCATACGTGTTACTC
>PC.593_690 FLP3FBN01CHIXK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.356_691 FLP3FBN01BSQTU orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.635_692 FLP3FBN01D15YG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAGCAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.354_693 FLP3FBN01BTIK9 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_694 FLP3FBN01AHV2W orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAATCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_695 FLP3FBN01EGJ5Q orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.355_696 FLP3FBN01A2YVL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACAGCAGGTTGGATACGTGTT
>PC.354_697 FLP3FBN01A5KXW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACT
>PC.593_698 FLP3FBN01D0KPW orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_699 FLP3FBN01EF8RN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATTGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATATCAGTGCCATGCGACTCCGATATTTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCG
>PC.481_700 FLP3FBN01A0OWM orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCCATAACCACCGGAGTTTTCAATCAAAAGGGATGCCCCTCTTGATGTTATGGGATATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.356_701 FLP3FBN01DUQQ0 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCAACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCAAGCCATGCAGCTCAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGCG
>PC.355_702 FLP3FBN01CIWNC orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATCTGGCCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGCTCATCCAAAGGCAATAAATCTTTGGCAGCTTGGCCATGCGACCACGCTGCGTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCACCTAAGGGCAGATTGCTCACGCGTT
>PC.635_703 FLP3FBN01D07UJ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.354_704 FLP3FBN01B27CZ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_705 FLP3FBN01DQI3A orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGCTGCCCCGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_706 FLP3FBN01AT5GO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.356_707 FLP3FBN01CP6X7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.356_708 FLP3FBN01DJ7Y0 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_709 FLP3FBN01AXR34 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGCGG
>PC.636_710 FLP3FBN01C9YGU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.354_711 FLP3FBN01AVACO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_712 FLP3FBN01DTNE9 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.607_713 FLP3FBN01DLTYU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.635_714 FLP3FBN01DIYX9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.481_715 FLP3FBN01CMQ1H orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGGAAAGCCATGCGGCTCCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.481_716 FLP3FBN01ECGSC orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.481_717 FLP3FBN01AP7UV orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.356_718 FLP3FBN01COR6A orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_719 FLP3FBN01D7CGI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.636_720 FLP3FBN01BHLXV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.593_721 FLP3FBN01CXMVM orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTT
>PC.481_722 FLP3FBN01A7TP4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGACACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_723 FLP3FBN01CWOPF orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_724 FLP3FBN01C1B5E orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.607_725 FLP3FBN01EGXD4 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACAAGCTAATCAGACGCGAGCTCATCCATATCCGATAAATCTTTGATGGAAAGAAGATGCCTTCAAACCATGTTATGCGGTATTATCAGTCGTTTCCAACTGTTATCCCCCAGATATGGGCAGATTG
>PC.593_726 FLP3FBN01APPTP orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_727 FLP3FBN01C4BE2 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.607_728 FLP3FBN01ASJ6H orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_729 FLP3FBN01DI8XZ orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCGGAAG
>PC.354_730 FLP3FBN01DXZGT orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.593_731 FLP3FBN01CVPVO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGTCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACT
>PC.635_732 FLP3FBN01C6P40 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCGGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGGCCAACTACCTAACTGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_733 FLP3FBN01EFG5X orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTCTTACCCCTCCAACTAGCTAATCAGACGCGGACTCATCCTGCACCGAATTTCTTTTCCTACTCCCTCATGCGAGGCTGTAGGCTTATGCGGTATTAGCAGCCATTTCTAGCTGTTGTCCCCCTGTGCAGGGCAGATTATCC
>PC.607_734 FLP3FBN01B7UNT orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGTCTTGGTGGGCCGCTACCCCGCCAACTAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.355_735 FLP3FBN01CSORT orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGTAGTTTTCTAGGGAAAAGCCATGCGGCTTCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.607_736 FLP3FBN01DY4CI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.607_737 FLP3FBN01BE4FD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.356_738 FLP3FBN01B4V0D orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_739 FLP3FBN01BZ1F6 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCATCGCAGGCTCGGTGGGCCGTTACCCCGCCGACTACCTAATGGGCCGCGACCCCATCTCACACCGCGAGAGCTTTCATCCGGCTAACATGTGTTGGCCGGATAGTATCCGGTATTAGCTACCGTTTCCAGTAGTTATCCCGGTGTGTGAGGCAGGTTGGTCACGTGTTACTCACCCGTTCG
>PC.635_740 FLP3FBN01AZK2X orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.355_741 FLP3FBN01BSEFO orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATTGTCGCTTTGGTAGGCCGTTACCCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGCAACATGCGTTGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACT
>PC.354_742 FLP3FBN01BCXZU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.635_743 FLP3FBN01ALCR0 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACTCAGCGCCATGCGGCTCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCG
>PC.356_744 FLP3FBN01AO474 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_745 FLP3FBN01EJKEX orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGCATCGTCGTCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTCGATGAGGGCAAGTTGCTCACGCG
>PC.354_746 FLP3FBN01CPIOY orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTGCACCATGCGGTGCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.636_747 FLP3FBN01BUOD1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_748 FLP3FBN01AWUMH orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_749 FLP3FBN01BI3RV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.355_750 FLP3FBN01DQWOP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCG
>PC.635_751 FLP3FBN01BSFE9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGATACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACTCACCCG
>PC.636_752 FLP3FBN01DA3NU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.607_753 FLP3FBN01EU5A3 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGGCAGGTTGCCCACGCGTTACT
>PC.354_754 FLP3FBN01EON7J orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAGGTTGCATACGTGTTACTC
>PC.593_755 FLP3FBN01CIA5C orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.593_756 FLP3FBN01BE5VX orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.355_757 FLP3FBN01DKS5K orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_758 FLP3FBN01AIWWA orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.593_759 FLP3FBN01ES3TI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_760 FLP3FBN01DVTZC orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAAATCTTTGACATTCAAACCATGCGATTCAAATGTATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGCG
>PC.635_761 FLP3FBN01CAD0I orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_762 FLP3FBN01C82RZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGC
>PC.354_763 FLP3FBN01DQ1LW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_764 FLP3FBN01C3YCV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGACAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGCAGTA
>PC.481_765 FLP3FBN01ETH11 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.636_766 FLP3FBN01A449C orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCCACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCGCCG
>PC.356_767 FLP3FBN01CK4EX orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGCTGCCCCGCCAACTGCCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTCCGCCCCATGCCATGCGGCATTTGGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.593_768 FLP3FBN01ASO4F orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.635_769 FLP3FBN01ET6LV orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCATCTTACACCGATGAAATCTTTTCCGTCAGAGTCATGCGACTCCAGCGGGTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.593_770 FLP3FBN01DR37T orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCGGATCCATTCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAGTCCCGTGCGCTTATGCGGTATTAGCACGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTATCCACGCGTT
>PC.355_771 FLP3FBN01BS9WF orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGTACCATGCAGTACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_772 FLP3FBN01DH65L orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGTAGTG
>PC.607_773 FLP3FBN01DQKWR orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGACCGTGTTTCAGTTCCAGTGTGGCCGGTCATCCTCTTAGACCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGGGACGCGGACTCATCCCCTCGCGGAAGCTTTCATGAAGAGGCCTCCTTTCCTCCACCATACAATTGGTGGAGCTTATCCGGTATTAGCTGCCGTTTCCAGCAGTTATCCCGAACAAGAGGGCAGATTATCCACGCGTT
>PC.354_774 FLP3FBN01DVCUY orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACACTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_775 FLP3FBN01A6VU8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.481_776 FLP3FBN01B9O3L orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGAGCTCATCTTTGGACGATAAATCTTTGGCAGCAAAATCATGCGGTCTCGCTGCATCATGCGGTATTAGCAGTCGTTTCCGGCTGTTATCCCCCATCCAAAGGCAGATTG
>PC.635_777 FLP3FBN01AJ7P4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_778 FLP3FBN01CTE7A orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTTCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.636_779 FLP3FBN01EC99X orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_780 FLP3FBN01DQR2G orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTTGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCAA
>PC.635_781 FLP3FBN01CFBG4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.635_782 FLP3FBN01B5NT1 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_783 FLP3FBN01EZ9L3 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGATTGCTCCTTTGACCGCCCCGGGATGTCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_784 FLP3FBN01CRN6U orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.635_785 FLP3FBN01APRO8 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATAGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.635_786 FLP3FBN01BUMCM orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCCCTGTACCATGCGGTACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCTAAGT
>PC.636_787 FLP3FBN01EEB9U orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_788 FLP3FBN01A0HNZ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAACTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTCCTGCTGTGCCATGCGGCACCGCAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTGTACAGGGCAGGTTGCCCACACGTTACTCACCCG
>PC.635_789 FLP3FBN01AXZN7 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTATCTCACCAACTAGCTAATCAGATGCGGGCCCATCTTTTACCGAATATTTCCTTTCCTTCTCGGAAGATGCCTTCCAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGGCAGGTTGCCCACACGTTACTCACCCG
>PC.355_790 FLP3FBN01CN9ZN orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_791 FLP3FBN01CM7ZF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_792 FLP3FBN01A8C9C orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.636_793 FLP3FBN01DYE6O orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGGCCCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>PC.635_794 FLP3FBN01B03Q3 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTCACCCTTCCAACTAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATCTTCAGAGGCCATCTTTCCTTAAAGAATCATGCGATCCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTCCCCACAAAAGGGCA
>PC.356_795 FLP3FBN01D79PA orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCCGTTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGGTTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTTTCCACGCGTTACTCACCCGTCCGCCACTAACTCA
>PC.636_796 FLP3FBN01AR65G orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGAATCGCTTCCTTTGATCTCCCGGAGATGCCTCCGGAAGATGTCACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGTGACGGGCAGGTTGCATACGTGTTACTC
>PC.635_797 FLP3FBN01B8KLV orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTGGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTTTATCATGCGATCTCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.635_798 FLP3FBN01DGRID orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCTAATGTGGCCGGCCGCCCTCTCAGGCCGGCTGCTGATCGCGGTCTTGGTGGGCCGTTACCCCGCCAACTAACTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACGCAGAGCCATGCGGCTCCGCGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.481_799 FLP3FBN01D7KO6 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.635_800 FLP3FBN01DI2NJ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.354_801 FLP3FBN01COFU8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCGAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.354_802 FLP3FBN01BTCQ8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGTACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_803 FLP3FBN01BN228 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGTACCATGCAGTACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_804 FLP3FBN01BD9IY orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTATTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.635_805 FLP3FBN01AQPXC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGAGCCCATCCTATACCGGATTTCTCCTTTTACCCCTCAGCCATGCGACTGCGTGGTCTTATACCGTATTAACTACAGTTTCCCGTAGGTATCCCGTTGTATAGGGCAGGTTGCTCACGTGTTACTCACCCG
>PC.481_806 FLP3FBN01EIL6N orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAATCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_807 FLP3FBN01DAXF4 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.636_808 FLP3FBN01DF870 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.481_809 FLP3FBN01B65YI orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCGTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACCTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTTACAATAATATTATGGGTGGTTAATCCACGTTTCCATGGGCTATCCCCCTCGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.607_810 FLP3FBN01DY7BY orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_811 FLP3FBN01AUK9B orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.354_812 FLP3FBN01DWFUS orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.635_813 FLP3FBN01B4PVG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.356_814 FLP3FBN01CEUV1 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACAGGCTAATCAGACGCGGGTCCATCCATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_815 FLP3FBN01BYVKN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_816 FLP3FBN01CHCL3 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.593_817 FLP3FBN01AWPFQ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGTCCATCCCATACCGTCTCGGCTTTCCCCTCCCGGCCATGCGGCCAAGAGGAATCATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.354_818 FLP3FBN01CAKFA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGCGATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>PC.635_819 FLP3FBN01CKTVO orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCATCCTCTCAGACCAGCTACCCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATGGGACGCGGACTCATCGAAAAGCGGAAGCATATGCAGAGGCCCCCTTTCCTGCATAGACATATATGCAGCGTATGCGGTATTAGCAGCCGTTTCCAACTGTTATTCCCCACTCCCCGGTAGATTAT
>PC.355_820 FLP3FBN01CSJ87 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_821 FLP3FBN01A6FDX orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.636_822 FLP3FBN01EECRG orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_823 FLP3FBN01B4KSX orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGGCCGGATCGTGCGATCCTTCCGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.354_824 FLP3FBN01DJ1EH orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGATTTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCAGAGGCAGGTTG
>PC.636_825 FLP3FBN01DT1QK orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGACTGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCACGGAGATGCCTCCGCGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.636_826 FLP3FBN01EM8RT orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_827 FLP3FBN01DRFRX orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGTCCCATGCGGGCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_828 FLP3FBN01CG2MW orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.356_829 FLP3FBN01CQO5F orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTCCTCCTTTCGCCCGGTCCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.354_830 FLP3FBN01D8YZ5 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAGACAG
>PC.481_831 FLP3FBN01D32U0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.354_832 FLP3FBN01EBLMN orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGCTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.354_833 FLP3FBN01A6BZY orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGTTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGCAACATGCGTTGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.354_834 FLP3FBN01CT7NN orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_835 FLP3FBN01CKH8P orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.635_836 FLP3FBN01CYE9Z orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.354_837 FLP3FBN01EF1R5 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_838 FLP3FBN01C4K5P orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTTACT
>PC.354_839 FLP3FBN01C2MYD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGATACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
>PC.356_840 FLP3FBN01EOUN6 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.593_841 FLP3FBN01EKS2G orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.607_842 FLP3FBN01C77VC orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.355_843 FLP3FBN01A18XL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.355_844 FLP3FBN01EVR1U orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_845 FLP3FBN01D9H69 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACTAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.356_846 FLP3FBN01DPA1A orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_847 FLP3FBN01D0Z6N orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.355_848 FLP3FBN01BZ72V orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.636_849 FLP3FBN01D3QET orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCTGTACCGGGATCGCTCCCTTTCCGCATCTGCCCATGCGCGCTGATGCGGTTATGCGGTATTAGCATCTGTTTCCAGATGGTATCCCCCTGTACAGGGCAGGTTG
>PC.356_850 FLP3FBN01AZ92N orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_851 FLP3FBN01DUG1R orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGTCCACGCGTTACTCACCCGTCCGCCGCTAAGCC
>PC.481_852 FLP3FBN01CUD3O orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACCACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGCGGTCTTATGCGGTATTAGCAGTCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCTCACGCGTTACTCACCCGTCCGCC
>PC.481_853 FLP3FBN01CTZRH orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_854 FLP3FBN01D7ZG7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAATCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCCTCCATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_855 FLP3FBN01BA9G8 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_856 FLP3FBN01B0SNX orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCAGACGCGGGGCCATCTCATACCACCGGAGTTTTTCCTGCTGTACCATGCGGTACTGTAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_857 FLP3FBN01AYO4X orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCCGG
>PC.635_858 FLP3FBN01DSQSF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.636_859 FLP3FBN01D6CH7 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.481_860 FLP3FBN01DSRTP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGACACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTT
>PC.481_861 FLP3FBN01ENV4O orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCCCACCAACTACCTAATCAGACGCGGGTCCATCTTACACCACCTCAGTTTTTACCACTGTACCATGCAGTACTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.607_862 FLP3FBN01DZOKU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.356_863 FLP3FBN01C5M5W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGCGATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>PC.356_864 FLP3FBN01CLMSS orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCAGTCTCTTGGGCAGGTTACCCACGTGTTACT
>PC.593_865 FLP3FBN01DY2AX orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_866 FLP3FBN01CFM8N orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_867 FLP3FBN01DYIFI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGAAG
>PC.635_868 FLP3FBN01A3EIE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.354_869 FLP3FBN01D02CB orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.355_870 FLP3FBN01EY83L orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.355_871 FLP3FBN01CS0ML orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.354_872 FLP3FBN01EDF0K orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTTATTTCTAAGTGTTACTCCCCCGGTATACGGCTAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.481_873 FLP3FBN01AO1DG orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCCCCTGTACAGGGCAGGTTCCCCACGCGTT
>PC.354_874 FLP3FBN01CLZGK orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.636_875 FLP3FBN01A2AJL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCGTATACCGTCTCGGCTTTTCACACTCCAGTATGCACTGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTT
>PC.355_876 FLP3FBN01AW4PZ orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACC
>PC.481_877 FLP3FBN01DYX26 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATTGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCGGACGCGAGCCCATCTCTCAGCGGATTGCTCCTTTGATATATCTACCATGCGATACATATATTTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAGAGGCAGGTTGCTC
>PC.355_878 FLP3FBN01DUDZY orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_879 FLP3FBN01A44JU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.481_880 FLP3FBN01C0GMZ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAACTAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGCCTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGGCAGGTTGCCCACACGTTACTCACCCG
>PC.354_881 FLP3FBN01ARHYF orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACACGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_882 FLP3FBN01DNVVZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGATCACGCGTTACG
>PC.356_883 FLP3FBN01BAT9T orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.607_884 FLP3FBN01BAL72 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.354_885 FLP3FBN01CIG7X orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.607_886 FLP3FBN01DVJS5 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.635_887 FLP3FBN01ARTG1 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGCGG
>PC.354_888 FLP3FBN01BV86P orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_889 FLP3FBN01EFIS6 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAACTAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGGCCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.607_890 FLP3FBN01BJO96 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAAATCTTTGGCAGTCAGGGAGATGTCTCCCAACTGCTTCATGCGGTATTAGCAGATGTTTCCATCTGTTATTCCCCACTCCAAGGCAGGTTACTCACG
>PC.593_891 FLP3FBN01EMI0C orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_892 FLP3FBN01AOU7Q orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCACCTCGGTTTTTCACACTGTCCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.607_893 FLP3FBN01DWLJX orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCCACTAAT
>PC.635_894 FLP3FBN01CKW1J orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_895 FLP3FBN01D7325 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_896 FLP3FBN01BR4U6 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.355_897 FLP3FBN01A2KBL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.481_898 FLP3FBN01AT6OD orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGATACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCTCGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.355_899 FLP3FBN01BD8E7 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.635_900 FLP3FBN01EKI61 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACCGATCGATGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCGGACGCATCCCCATCCTTTACCGATAAATCTTTGCTCCGACACAGATGTCCGGGACGGAGAACATGGGGGATTAATCCGCCTTTCGACGGGCTGAACCCCACGTAAAGGGAAAGGGTTGGATACGCGTTACTCACCCGTGCGCCGGTCGCC
>PC.481_901 FLP3FBN01EDR54 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.593_902 FLP3FBN01CODQH orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.354_903 FLP3FBN01BVQJR orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.635_904 FLP3FBN01EV1JR orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.636_905 FLP3FBN01BQAKZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCAGGTT
>PC.356_906 FLP3FBN01BDWJA orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.593_907 FLP3FBN01CNXDO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_908 FLP3FBN01ASXZX orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_909 FLP3FBN01B9E1U orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCCTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTACCGCTGTGCCATGCGGCACCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.354_910 FLP3FBN01AKJ8H orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACAAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACT
>PC.593_911 FLP3FBN01BD3KF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCC
>PC.354_912 FLP3FBN01DBIH6 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.636_913 FLP3FBN01C61FL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_914 FLP3FBN01ENY5A orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_915 FLP3FBN01ESGKH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_916 FLP3FBN01DMOT4 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_917 FLP3FBN01BEC0A orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGGCAATCAGAGAGATGCCTCCCAATTGCATCATGCGGTATTAGCGGCTGTTTCCAACCGTTATTCCCCACTCCAAGGCAG
>PC.356_918 FLP3FBN01C1F87 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAACTAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGCCTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGGCAGGTTGCCCACACGTTACTTACCCGTCCG
>PC.356_919 FLP3FBN01AR785 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCCGCACCGGATCGCTCCTTTGACCGCTCCCCCATGCAGAGGAACGGTGTCATGCCGTATTAGTCCGGATTTCTCCGGGTTATCCGTCTGTGCGGGGCAGGTCGCATACGCGTTACT
>PC.635_920 FLP3FBN01DT66J orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGATTTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCAGAGGCAGGTTG
>PC.481_921 FLP3FBN01ER6TY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_922 FLP3FBN01BWI9J orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACATGTCCGTGATGCCACGGTCATGTGTCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGTCG
>PC.481_923 FLP3FBN01BR0YS orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.635_924 FLP3FBN01BTOX2 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTTGCTTTGGTGGGCCGTTGCCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGACACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.593_925 FLP3FBN01BR1S4 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGCCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_926 FLP3FBN01BI8Q8 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_927 FLP3FBN01C73G0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.607_928 FLP3FBN01EIDZ1 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACTAGCTAATCAGACGCGGGCCCATCCCACACCGCCGGAGCTTTCCGCACCGCCCCATGCGGGGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCTGTGTGGGGTAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.636_929 FLP3FBN01AE1JZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGGTGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_930 FLP3FBN01EEXTN orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCACCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.356_931 FLP3FBN01BVGJF orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCATATGCCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCG
>PC.635_932 FLP3FBN01BWF4P orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTGCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTTTATCATGCGATCTCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.593_933 FLP3FBN01BU6SJ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACGCATCGTCGCCGTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCG
>PC.355_934 FLP3FBN01DY8W5 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGCGATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>PC.635_935 FLP3FBN01CXZHC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.356_936 FLP3FBN01EDY3K orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.356_937 FLP3FBN01AF3UI orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.481_938 FLP3FBN01CNBHT orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_939 FLP3FBN01BW3YV orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.635_940 FLP3FBN01DZVEP orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACTGTTCCATGCGAAACCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_941 FLP3FBN01EK704 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.355_942 FLP3FBN01DQ5NS orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_943 FLP3FBN01CX6KS orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTGCCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_944 FLP3FBN01BDSR0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.593_945 FLP3FBN01D5PH5 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.607_946 FLP3FBN01AMPHD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.593_947 FLP3FBN01D88A5 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACTAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_948 FLP3FBN01DTZ2Y orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.354_949 FLP3FBN01EAT6O orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCCTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_950 FLP3FBN01AYJGP orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.636_951 FLP3FBN01BDJRH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.635_952 FLP3FBN01B2CCU orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCTGTATCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGAGCCCATCCTATACCGGATTTCTCCTTTTACCCCTCAGCCATGCGACTGCGTGGTCTTATACCGTATTAACTACAGTTTCCCGTAGGTATCCCGTTGTATAGGGCAGGTTGCTCACGTGTTACT
>PC.636_953 FLP3FBN01BSTZH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACGTATCGTTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATACGGCGCGGGTCCATCTATAAGTGACAGCCGAAACCGTCTTTCAACATTGAACCATGCGGTTCAATATATTATCCGGTATTAGCCCCGGTTTCCCGGAGTTATCCCAGTCTTATAGGTAGGTTACCCACGTGTTACTCACCCGTCCGCCGCTAACA
>PC.636_954 FLP3FBN01EG5XE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACG
>PC.593_955 FLP3FBN01DHSDB orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.636_956 FLP3FBN01BG5PZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCTCACCAACTATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTGGAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTAGTTCCCCTCGTACAGGCCAGGTTGGCCCACCGCGTTACTCACCCGTCCGCCACTCA
>PC.607_957 FLP3FBN01ERW7M orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCGGCCGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGCTGCATCGGGATGCCCCATCGCAGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTATCCCCCTCTGAAAGGCAGGTTGCTC
>PC.356_958 FLP3FBN01A3D61 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.635_959 FLP3FBN01AOYAO orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.636_960 FLP3FBN01A5KJQ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_961 FLP3FBN01BOMSP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAATTTTCAAGTAAAAGACATGCGTCTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCGCCACT
>PC.356_962 FLP3FBN01D9NXA orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTT
>PC.481_963 FLP3FBN01D0A3H orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.481_964 FLP3FBN01DKHEO orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCTTTACCCCGCCAACTGCCTAATCAGACGCGGGCCCATCCCATACCGCAAAAGCTTTCCATGCCAGGGCATGCGCCCCCGCATGTGCATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_965 FLP3FBN01EJF8M orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.356_966 FLP3FBN01DBC7G orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGCCCCCTCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_967 FLP3FBN01CKKP0 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.481_968 FLP3FBN01A38C0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_969 FLP3FBN01DR8GQ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTGCCCCGCCAACTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGGCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTTGGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCCGGCG
>PC.636_970 FLP3FBN01C170V orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.636_971 FLP3FBN01DOBTJ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.355_972 FLP3FBN01DI8DL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGGCCCAACTGCATTATGCGGTATTAGCACATCTTTCGATGTGTTATTCCCCACTCCAAGGCAGG
>PC.636_973 FLP3FBN01BW7BY orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTCACACCGCCAACTACCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.607_974 FLP3FBN01BWI6J orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.355_975 FLP3FBN01CN8VK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAG
>PC.593_976 FLP3FBN01B12IL orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTT
>PC.593_977 FLP3FBN01DTCGT orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.635_978 FLP3FBN01C7P3U orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_979 FLP3FBN01BUFD0 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGCTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.607_980 FLP3FBN01BI3Q6 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACAGATACCATGCGATACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACTCACCCG
>PC.356_981 FLP3FBN01CKM98 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.481_982 FLP3FBN01C4VM7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCGAGCCATGCGGCTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCAGGTTGCCCACGCGTTACT
>PC.481_983 FLP3FBN01CVDF5 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.356_984 FLP3FBN01CM73M orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCG
>PC.354_985 FLP3FBN01CLMTS orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTCACCCTTCCAACTAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATCTTCAGAGGCCATCTTTCCTTAAAGAATCATGCGATCCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTCCCCACAAAAGGGCAGGTT
>PC.607_986 FLP3FBN01ANKI1 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTTTTACCTCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCG
>PC.636_987 FLP3FBN01CEL0P orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGG
>PC.354_988 FLP3FBN01DQMD6 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCCGG
>PC.636_989 FLP3FBN01AIT2F orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGACTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_990 FLP3FBN01ENHOZ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_991 FLP3FBN01A5RUT orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.354_992 FLP3FBN01AY6CG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTTCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.607_993 FLP3FBN01EQJ0R orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCTTCTACCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTATACCATGCGGTACTGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTACC
>PC.635_994 FLP3FBN01DNNDF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.481_995 FLP3FBN01BBWRU orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGTATCATGCGATACTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCC
>PC.607_996 FLP3FBN01CJZCS orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.354_997 FLP3FBN01EJZ09 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACT
>PC.356_998 FLP3FBN01BGWL1 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_999 FLP3FBN01AYFQO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_1000 FLP3FBN01C9PO9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCGATTAGACGCCGGTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.593_1001 FLP3FBN01DR99Q orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTTCG
>PC.635_1002 FLP3FBN01BHOAC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCAT
>PC.607_1003 FLP3FBN01CGYKR orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTACGTATCATCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATACGCCGCAGGTCCATCCATGTCCACACCCGAAGGTCCTTTAATATAGTCAAGATGCCTTGTCTATACCTATCCGGTTTTAGCAGACGTTTCCATCTGTTATCTCGGGTACATGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.607_1004 FLP3FBN01BJE5P orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATCTGGCCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCGACGCTTCCATGCGAAAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCAGGTTGCCCACGTGTTACTCAGCCGTTCGCCACTATCC
>PC.607_1005 FLP3FBN01DI95G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_1006 FLP3FBN01DHLY4 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_1007 FLP3FBN01BHLW4 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTT
>PC.481_1008 FLP3FBN01B2YIQ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.354_1009 FLP3FBN01A1YKO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCGGGTCGCTCTCCCAGCGGCAGCTGCTCTGCTCTGCCACCTTTCTTTCTTCCATCATGCGATGGTTGAACCTCATGCGGTATTAGCTGTGATTTCTCACGGTTATT
>PC.356_1010 FLP3FBN01A44HD orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTCTCCACGCGTT
>PC.354_1011 FLP3FBN01ADKEK orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
ATGGACCGTGTCTCAGTTCCATTGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCCAGAGTGATAGCCAAAGCCATCTTTCAAACAAAAGCCATGTGGCTTTTGTTGTTATGCGGTATTAGCATCTGTTTCCAAATGTTATCCCCCGCTCCGGGGCAGG
>PC.355_1012 FLP3FBN01A4HWP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.593_1013 FLP3FBN01A7BP1 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_1014 FLP3FBN01AS674 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_1015 FLP3FBN01A1FYD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.607_1016 FLP3FBN01BJH0F orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.635_1017 FLP3FBN01EEXQK orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGAATCGCTTCCTTTGATCTCCCGGAGATGCCTCCGGAAGATGTCACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCG
>PC.354_1018 FLP3FBN01ADKU8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.481_1019 FLP3FBN01D4CIQ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAGCTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_1020 FLP3FBN01BN9BB orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGTCAGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGACAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTT
>PC.593_1021 FLP3FBN01AEJLU orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.635_1022 FLP3FBN01CHVM4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAACCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.481_1023 FLP3FBN01EBPW0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCTATAAATATTTGATAGCAATATCATGCGGTATCGCTATGTTATGCGGTGTTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTACAGGGCAGGTTACCCACG
>PC.636_1024 FLP3FBN01BGO9L orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_1025 FLP3FBN01EDVUS orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAACTATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTGGAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCCTCCTGATAGGTAGGTTGCATACGCGTTACTCACCCGTCGCGCCGG
>PC.481_1026 FLP3FBN01BOAPJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGGCTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACG
>PC.356_1027 FLP3FBN01B9SW2 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGATGCAGGTCCAGTCCTAC
>PC.354_1028 FLP3FBN01AST80 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.635_1029 FLP3FBN01DOAO5 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGCTGGTAGGTTGCATACGCGTTACTCACCCG
>PC.636_1030 FLP3FBN01CSLHO orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCGCCG
>PC.356_1031 FLP3FBN01CSRAJ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGACTTGGTAGGCCTCTACCCCACCAACTATCTAATCAGACGCGGGCCCATCCCGCTACCGCCGCAGCTTTCCGCCGGGGGCCATGCGGCCCCCGCCGTGTATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCAGTGCCGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.356_1032 FLP3FBN01DVOX8 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_1033 FLP3FBN01ETLIN orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGAAAACCTCTCAGTCCCCCTATGTATCGTAGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTCTAACCATCGGAATTTTCAACCATCTAAGATGCCTTAAATGATATTATGGGGTATTAGACCAAATTTCTTCGGATTATCCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGG
>PC.356_1034 FLP3FBN01EIT5M orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.635_1035 FLP3FBN01DAUO4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGGCTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCT
>PC.593_1036 FLP3FBN01DH3EM orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.607_1037 FLP3FBN01A6YVR orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGACGCGAGCCCATCTCATACCGGATTTCTCCTTTTACCCCTCTACCATGCGGTAGTGTGGTCTTATACCGTATTAACCGAAGTTTCCCTCGGGTATCCCGTTGTATGAGGCAGGTTGCTCACGCGTTACTCACCCGTTCG
>PC.354_1038 FLP3FBN01ATZTQ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATGCGCCATAAGTCCATCCTTCAGCTATCCCGTGGGATATTTAATATGTCGATCATGCGATCTTCATACCTATGCGGTCTTAGCTATCATTTCTAATAGTTATCCCCCTCTTAAGGCCAGGTTACTTA
>PC.635_1039 FLP3FBN01EV4Z5 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAACGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.635_1040 FLP3FBN01ELE68 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTGCTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACGAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACCAGGGCATGCGCCCCTGTGCGCTTATGCGGTATTACCAGCCGTTTCCAGCTGCTATCCCCCTGTATGGGCCAGGTTATCCACGCGTTACTCACCCG
>PC.354_1041 FLP3FBN01DNL4E orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_1042 FLP3FBN01COXAR orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_1043 FLP3FBN01B6YON orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGCCCCCTCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.635_1044 FLP3FBN01C8AH6 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGCCAGGGAGATGCCTCCCAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGCAG
>PC.354_1045 FLP3FBN01B2ALM orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.354_1046 FLP3FBN01EXM1D orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAGGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.355_1047 FLP3FBN01C6ISY orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGCTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_1048 FLP3FBN01C44O7 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATCTGGCCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCAACGCTTCCATGCGAAAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCAGGTTGCCCACGTGTTACTCAGCCGTTCGCCACTATCC
>PC.355_1049 FLP3FBN01C5Q76 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.481_1050 FLP3FBN01DOZ1S orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1051 FLP3FBN01AZ93K orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGACGAAGTCATGCGACTCCCTCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCG
>PC.635_1052 FLP3FBN01CK0M3 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGAGCCCATCCTGTACCTATAAATATTTGATGGAAATATCATGCGGTATCTCCATGTTATGCGGTTTTAGCAGCCGTTTCCAACTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTT
>PC.481_1053 FLP3FBN01ENPIT orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTAGGCTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1054 FLP3FBN01B3DCW orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCAACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCCCACCAACTACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCCTACCATGCGATAAGATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGCGTTACT
>PC.354_1055 FLP3FBN01BMNXJ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.607_1056 FLP3FBN01DAQ2Q orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.356_1057 FLP3FBN01B4Q1P orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.607_1058 FLP3FBN01A7RZC orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGGCAGGTTGCCCACGCGTTACT
>PC.481_1059 FLP3FBN01CFA5S orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCCACTGATCGTCGCCTTGGTGGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGGAAAGCCATGCGGCTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTGGG
>PC.607_1060 FLP3FBN01DPJ0D orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.481_1061 FLP3FBN01DOYEX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACTCTCTCAAGCCGGCTACTGATCGTTGTCTTGGTAGGCCGTTACCCTGCCAACTAACTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_1062 FLP3FBN01A13CQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.593_1063 FLP3FBN01CMHTY orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCC
>PC.481_1064 FLP3FBN01DD801 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATTACCTACGTGTTACTCACCCGTTCG
>PC.607_1065 FLP3FBN01B1MJU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGATAAAAGAGGCTCATGCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCAGGTTATCCACGCGTTACTCACCCG
>PC.593_1066 FLP3FBN01D8L02 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTT
>PC.356_1067 FLP3FBN01DWXHY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.607_1068 FLP3FBN01DO0PN orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.607_1069 FLP3FBN01A2RID orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGACCGTGTCCCAGTCCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGGGTCGCGAGCCCATCCTATACCGATAAATCTTTTACCCCTGTATCATGCGATACTGTGGTCTTATACCGTATTAGCACAAATTTCTCTGTGTTATCCCGTTGTATAGGGCAGGTTGCTCACGTGTTACTCACCCGTTCG
>PC.481_1070 FLP3FBN01AJ4LN orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACAAGACCATGCAGTCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCAGGTTATCCACGCGTTACT
>PC.481_1071 FLP3FBN01DJSHD orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGGCTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCAGGTTGCCCACGCG
>PC.356_1072 FLP3FBN01ES2CK orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_1073 FLP3FBN01D5PDA orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGGCCGGATCATGCGATCCTTCCGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1074 FLP3FBN01CPOTU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_1075 FLP3FBN01DDGAY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.354_1076 FLP3FBN01DDAHB orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.356_1077 FLP3FBN01A8N1I orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_1078 FLP3FBN01ELV1R orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.356_1079 FLP3FBN01DZNIH orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCGTATACCGTTTACGCTTTTCACACTGCTCCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.593_1080 FLP3FBN01DWOHH orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCC
>PC.356_1081 FLP3FBN01EKZQ1 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCAGG
>PC.354_1082 FLP3FBN01EVVYX orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_1083 FLP3FBN01BX9OT orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATAGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_1084 FLP3FBN01C11Y6 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAACTAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGGCCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_1085 FLP3FBN01BORQ9 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.607_1086 FLP3FBN01CORTK orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCG
>PC.607_1087 FLP3FBN01DDXJ9 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_1088 FLP3FBN01D3MIZ orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.481_1089 FLP3FBN01COS84 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACT
>PC.607_1090 FLP3FBN01CJOJP orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGCCAGAGCCATGCGGCTCAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGCAG
>PC.354_1091 FLP3FBN01AH4QL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGGCCATGCGGCCCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.354_1092 FLP3FBN01ARXOA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGCGGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.356_1093 FLP3FBN01BOZTS orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGACAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.356_1094 FLP3FBN01CP1E3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.356_1095 FLP3FBN01AYTXJ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_1096 FLP3FBN01ESMLF orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.355_1097 FLP3FBN01CAOAX orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTCCACACTGCTTCATGCAAAGCTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCGGGGCAGGTTGCCCACGCG
>PC.354_1098 FLP3FBN01D5KU1 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCTCACACCACCGGAGTTTTTCACACCAGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCAGGTTATCCACGCGTTACTCACCCG
>PC.356_1099 FLP3FBN01DR6KE orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.481_1100 FLP3FBN01BWBS4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.481_1101 FLP3FBN01B9R0O orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGCAGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.354_1102 FLP3FBN01EGUWG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1103 FLP3FBN01CMT8N orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1104 FLP3FBN01DAJ0V orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1105 FLP3FBN01EDYEP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.354_1106 FLP3FBN01DAF3E orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_1107 FLP3FBN01BVUDC orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGACCGTGTTTCAGTTCCAGTGTGGCCGGTCATCCTCTTAGACCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGGGACGCGGACTCATTCCCTCGCGGAAGCTTTCATGAAGAGGCCTCCTTTCCTCCACCATACAATTGGTGGAGCTTATCCGGTATTAGCTGCCGTTTCCAGTAGTTATCCCGGTGTGTGAGGCAGGTTGGTCACGTGTT
>PC.481_1108 FLP3FBN01D7PW9 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.593_1109 FLP3FBN01C62R4 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCGCCCGTGCG
>PC.481_1110 FLP3FBN01CV26L orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAATCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.355_1111 FLP3FBN01DGJIL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_1112 FLP3FBN01ENBFP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTACCTCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCGCACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACT
>PC.481_1113 FLP3FBN01D5Y47 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCTCAGTTTTTCACACTGTCCCATGCGAGACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCAGGTTATCCACG
>PC.356_1114 FLP3FBN01D6TCI orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGATTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.607_1115 FLP3FBN01DBSZG orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.355_1116 FLP3FBN01A6KUH orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_1117 FLP3FBN01CFKFW orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGCACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.355_1118 FLP3FBN01CWV71 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.607_1119 FLP3FBN01BUKY6 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAGATCTTTGATGTCCAGTACCATGTGATACAGACATATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGTAGG
>PC.354_1120 FLP3FBN01APUP8 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACT
>PC.356_1121 FLP3FBN01DVAA4 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTTTTAGCACCTATCTCTAAGTGTTATCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.354_1122 FLP3FBN01AYSZA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAACTAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGCCTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGGCAGGTTGCCCACACGTTACT
>PC.481_1123 FLP3FBN01E11F6 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.355_1124 FLP3FBN01D2E5F orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_1125 FLP3FBN01AY6JF orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCGTTCG
>PC.607_1126 FLP3FBN01AOSD0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.481_1127 FLP3FBN01DHOF4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTACCACTGTACCATGCAGTACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCGTTACT
>PC.356_1128 FLP3FBN01DP53S orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGGTGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_1129 FLP3FBN01DPT2H orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.481_1130 FLP3FBN01A9OKC orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAGACCATGCGGTCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_1131 FLP3FBN01C45DS orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.356_1132 FLP3FBN01BTP39 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACAATCCCATGCGGGATTGGGCGCTTATGCGGTTTTAGCACCTGTTTCCAGGTGTTATCCCCCGGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAGGTT
>PC.607_1133 FLP3FBN01D8UA3 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.607_1134 FLP3FBN01BTBC0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAATCAGTACCATGCGATACAATTGCATTATGCGGTATTAGCACAGCTTTCGTTGTGTTATTCCCCACTCCAAGGCAGGTT
>PC.481_1135 FLP3FBN01BZF87 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTCCCCACGCGTTACT
>PC.593_1136 FLP3FBN01ANQ3B orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTTACCCTTCCAACTAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATTCTCAGAGGCCATCTTTCCTTAGAAAATCATGCGATTCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTCCCCACAAAAGGGCAGGTT
>PC.607_1137 FLP3FBN01ATVZT orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.355_1138 FLP3FBN01BY1GK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.481_1139 FLP3FBN01B8WRO orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.354_1140 FLP3FBN01A68PA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1141 FLP3FBN01B6F7P orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_1142 FLP3FBN01CURUA orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACAAGCTAATCAGACGCGAGCCCATCCTGCACCGTATTACTTTCCACACCGCCCCATGCAGGGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.356_1143 FLP3FBN01DEVFX orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGTCTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCGCTAAG
>PC.607_1144 FLP3FBN01DJVP5 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTCCGAGCGATAAAATCTTTGATACCAAAGGGATGTCCCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGTAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.356_1145 FLP3FBN01AVA3U orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCTGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1146 FLP3FBN01A1JXR orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATGTCACGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAAGAAGATGCCTTCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACGGGCAGGTTGCATACGCGTTACTCACCCGTGCG
>PC.356_1147 FLP3FBN01CU9QX orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGTCAGTACCATGCGGTACCAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.355_1148 FLP3FBN01EFEUM orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.356_1149 FLP3FBN01DR3RW orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACAAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCG
>PC.607_1150 FLP3FBN01BTNQ7 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.355_1151 FLP3FBN01DG439 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_1152 FLP3FBN01BJV71 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGCTCCATGCGGAGCTGTGCGCTCATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTATCCACGCGTTACT
>PC.354_1153 FLP3FBN01BBGPO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_1154 FLP3FBN01CPEDQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCAACGCTTCCATGCGAAAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCAGGTTGCCCACGTGTTACTCAGCCGTTCGCCACT
>PC.355_1155 FLP3FBN01BJETT orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.354_1156 FLP3FBN01D81I4 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.355_1157 FLP3FBN01CXUH7 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAGGTTGCATACGTGTTACTC
>PC.607_1158 FLP3FBN01D66OZ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.354_1159 FLP3FBN01EP18M orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.354_1160 FLP3FBN01AYADU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTATCAGGGTCATGCGGCCCCAATACTTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.354_1161 FLP3FBN01DZ6HR orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.356_1162 FLP3FBN01DQS40 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCGGACGCGGACCCATCGCATGCCGGGATCGCTCCCTTTCCGCACTGCGCCATGCGGCGCCGTGCGCATATGCGGTATTAGCGGTTGTTTCCAGCCGGTATCCCCCTGCATGCGGCAGGTTG
>PC.355_1163 FLP3FBN01ETR1Z orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGTACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.355_1164 FLP3FBN01BYEMY orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_1165 FLP3FBN01DWFB9 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.481_1166 FLP3FBN01B350B orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTACACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.356_1167 FLP3FBN01D8VJ6 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTATACCACCTCAGTTTTTCACACCGAACCATGCGGTTTTAGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.481_1168 FLP3FBN01AWH2Z orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAGCTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.356_1169 FLP3FBN01A0B3E orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCCCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_1170 FLP3FBN01B6CNT orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_1171 FLP3FBN01AJVO0 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCG
>PC.354_1172 FLP3FBN01A5TI5 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.354_1173 FLP3FBN01CSGAL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.354_1174 FLP3FBN01A26LS orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1175 FLP3FBN01D1OTA orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.356_1176 FLP3FBN01EOJO5 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTGCCCCACCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACTCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>PC.607_1177 FLP3FBN01EEA7Q orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTTGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCAGACGCGGGCCCATCCCATACCGCGAAAGCTTTCCATGCAGGGGCATGCGCCCCCGCATGTGCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGCGTTACTC
>PC.607_1178 FLP3FBN01C7IA4 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGATAAAAGAGGCTCATGCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCAGGTTATCCACGCGTTACTCACCCG
>PC.593_1179 FLP3FBN01DIPTV orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGTTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.356_1180 FLP3FBN01A95YV orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAGAT
>PC.355_1181 FLP3FBN01C17R1 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGTAGTTTTTACCTCAGAACCATGCGGTTCTGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTACTCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_1182 FLP3FBN01ENDTE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.356_1183 FLP3FBN01CGNQ7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGACCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.356_1184 FLP3FBN01CPBZJ orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.355_1185 FLP3FBN01C94TW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.355_1186 FLP3FBN01AJ8QN orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.607_1187 FLP3FBN01ERY9Y orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACATTCAAACCATGCGATTCAAATGTGTCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGCG
>PC.607_1188 FLP3FBN01D1576 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.607_1189 FLP3FBN01D9ANJ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGACAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTCGTGAAAGGCAGGTTATCCACGCGTTACTCACCCGTCCCG
>PC.607_1190 FLP3FBN01DE016 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTGGGCCATGCAGCCCCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.607_1191 FLP3FBN01BEAYH orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.481_1192 FLP3FBN01B4KRR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
ATGGACCGTGTCTCAGTTCCATTGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCCAGAGTGATAGCCAAAGCCATCTTTCAAACAAAAGCCATGTGGCTTTTGTTGTTATGCGGTATTAGCATCTGTTTCCAAATGTTATCCCCCGCTCCGGGGCAGG
>PC.593_1193 FLP3FBN01EDWJK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAACAACCTAATGGAACGCATCCCCATCCTTTATCGGAATCCTTTAATAATGAAACCATGCGGAATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.481_1194 FLP3FBN01B2A3Z orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTTACCAACCAGCTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTTCCATGCGGCACAGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACT
>PC.355_1195 FLP3FBN01EEKR5 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_1196 FLP3FBN01CQ5AB orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.356_1197 FLP3FBN01DXG0S orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCTGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGCCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.354_1198 FLP3FBN01B1B5Q orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.593_1199 FLP3FBN01CQKDB orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGGTACTCACCCGTGCGCCGG
>PC.355_1200 FLP3FBN01CB8GB orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.356_1201 FLP3FBN01C03VW orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCATCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1202 FLP3FBN01ETEMV orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1203 FLP3FBN01DYMXO orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTCCCCACGCGTTACTCACCCGTCCG
>PC.593_1204 FLP3FBN01D0XGC orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.355_1205 FLP3FBN01D069Y orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACCTGTTTATCCCCCTGTGTTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.356_1206 FLP3FBN01C8P52 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.607_1207 FLP3FBN01C1Q79 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCGGAG
>PC.481_1208 FLP3FBN01BJ2ZS orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAACCAACAGGGATGCCCCCGTCGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCG
>PC.481_1209 FLP3FBN01AK0GJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1210 FLP3FBN01D8L81 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGAGCTCATCTTTGGACGATAAATCTTTGGCAGCAAAATCATGCGGTCCTGCTGCATTATGCGGTATTAGCAGTCGTTTCCGGCTGTTATCCCCCATCCAAAGGCAGATTG
>PC.354_1211 FLP3FBN01EHQW9 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.593_1212 FLP3FBN01APYG5 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1213 FLP3FBN01CXOXZ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_1214 FLP3FBN01C6H6I orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGCGTCGGAGTCATGCGGCTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACGCG
>PC.481_1215 FLP3FBN01BB5R9 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGGGGTCCCCATGCAGGGTCCCCTGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCGGTACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAAACAGTCCGGGG
>PC.355_1216 FLP3FBN01BF51P orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.356_1217 FLP3FBN01DN1Z3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.481_1218 FLP3FBN01CJJE7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1219 FLP3FBN01AK9WE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACT
>PC.356_1220 FLP3FBN01D2BCC orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGCACAGATACCATGCGATACCCATGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.354_1221 FLP3FBN01D2DVT orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_1222 FLP3FBN01CC1O1 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATGCGCCGCATGCCCATCCTCCGCCGGAATCGCTTCCTTTTAACACCCGTGGGATGCCCCACAGATGTACCACGCGGTATTAGTCCGGATTTCTCCGGGTTATCCCCCTGCGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.354_1223 FLP3FBN01BBP7L orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_1224 FLP3FBN01DPA2Y orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.481_1225 FLP3FBN01D9LCY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_1226 FLP3FBN01C0XC7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTACCGCTGTGCCATGCGGCACCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCCGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_1227 FLP3FBN01EKL55 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACT
>PC.355_1228 FLP3FBN01CG3KS orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCACCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGCGATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGGCAGG
>PC.355_1229 FLP3FBN01D5RTH orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.356_1230 FLP3FBN01D97XE orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1231 FLP3FBN01ASUWG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.355_1232 FLP3FBN01BTIOW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.355_1233 FLP3FBN01EM86A orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTATATGAGGCAGGTTACCCACGCGTT
>PC.356_1234 FLP3FBN01CHFB7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.356_1235 FLP3FBN01CNAWV orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.354_1236 FLP3FBN01AKUQT orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1237 FLP3FBN01AP4SS orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACT
>PC.481_1238 FLP3FBN01B770T orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGATACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCG
>PC.355_1239 FLP3FBN01BV9MF orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCG
>PC.355_1240 FLP3FBN01EEO8Q orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.354_1241 FLP3FBN01BC5FW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGCTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACT
>PC.355_1242 FLP3FBN01BOTE5 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCAGTATACGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.593_1243 FLP3FBN01BUMJK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.355_1244 FLP3FBN01EGB8E orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCGCT
>PC.481_1245 FLP3FBN01AUN6V orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.354_1246 FLP3FBN01CBJWL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTACCTCACCAACTAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.356_1247 FLP3FBN01DH3NR orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACT
>PC.355_1248 FLP3FBN01EM0GU orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_1249 FLP3FBN01D69SQ orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.356_1250 FLP3FBN01BFKOF orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCGTCTCGGCTTTTCACTCGGTATCATGCGATACCGGGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1251 FLP3FBN01D0CQF orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_1252 FLP3FBN01CWRA1 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.356_1253 FLP3FBN01A85QC orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACAAGCTAATCAGACGCGGAGCCATCTTACACCACCTCAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTCCTCCACGCGTTACTCACCCGTCCG
>PC.355_1254 FLP3FBN01CBFWS orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.481_1255 FLP3FBN01CSXJW orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCGTTCG
>PC.481_1256 FLP3FBN01CABHI orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.356_1257 FLP3FBN01BIK18 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTTACTC
>PC.356_1258 FLP3FBN01D26CI orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_1259 FLP3FBN01B5LFD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.481_1260 FLP3FBN01ADQU7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_1261 FLP3FBN01D5SXD orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCG
>PC.356_1262 FLP3FBN01BKOLX orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCGGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.356_1263 FLP3FBN01CEGMP orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.355_1264 FLP3FBN01E2P4M orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACT
>PC.355_1265 FLP3FBN01C64QW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGGCTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_1266 FLP3FBN01DGS0V orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.356_1267 FLP3FBN01C2VAB orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.355_1268 FLP3FBN01DYU4B orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1269 FLP3FBN01AIIWJ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_1270 FLP3FBN01BJV1A orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.356_1271 FLP3FBN01ANBIL orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTCGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_1272 FLP3FBN01AJ8QF orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCCCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACTCACCCG
>PC.481_1273 FLP3FBN01C4QQE orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCAGTCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCAACGCGTTACTCACCCGTCCGCCG
>PC.593_1274 FLP3FBN01BS0JY orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.356_1275 FLP3FBN01DFM2G orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCGTATACCGTTTACGCTTTTCACACTGCTCCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCCG
>PC.356_1276 FLP3FBN01AIAA5 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTCTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCA
>PC.356_1277 FLP3FBN01CAKPN orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.355_1278 FLP3FBN01A6R51 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>PC.481_1279 FLP3FBN01EH9DF orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAACTAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACAGATACCATGCGATACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCAGGTTCCTCACGCGTTACT
>PC.593_1280 FLP3FBN01C1ALJ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTGTAAGTGGATTGCTCCTTTCCTCACTCTAACATGTGTCAGTGTGAGCATATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTTACAGG
>PC.481_1281 FLP3FBN01B0TAP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATGCCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTGTTAGCACCTATTTCTAAGTGTTATCCCCCTGCATACGGCAGGTTACCCACGCGTTACT
>PC.356_1282 FLP3FBN01C2UWD orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGTGTTACTCACCCG
>PC.355_1283 FLP3FBN01CKQDF orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGATACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.355_1284 FLP3FBN01A1GYR orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCG
>PC.356_1285 FLP3FBN01CD97W orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCAGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_1286 FLP3FBN01DIARK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCGTATACCGTCTCGGCTTTTCACACTCCAGTATGCACTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.356_1287 FLP3FBN01B438A orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCACCGGAGTTTTCAAAGGATTACCATGCGGTATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCGTGCGCCAGTCG
>PC.356_1288 FLP3FBN01D0219 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1289 FLP3FBN01EA4TG orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATTCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.356_1290 FLP3FBN01BGW2P orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.593_1291 FLP3FBN01CLS7R orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAACAAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGAGCCATGCAGCTCCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACG
>PC.593_1292 FLP3FBN01D8ZMU orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACT
>PC.593_1293 FLP3FBN01COFBD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCAGGTCCATCTTACACCGATAAAATCTTTTCCGTCCGGGCCATGCGGCCCTAGCGGGTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.355_1294 FLP3FBN01A88RD orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGGTCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGTGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.355_1295 FLP3FBN01AIDK4 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCGAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.355_1296 FLP3FBN01DCKQ6 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACT
>PC.593_1297 FLP3FBN01BZNMV orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.355_1298 FLP3FBN01BLVR2 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.355_1299 FLP3FBN01D9KO8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCCG
>PC.593_1300 FLP3FBN01EWA03 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_1301 FLP3FBN01CUHT2 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTATGGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCCAACGCGGACCCATCCTATGCCGCCTCAGCTTTTCACACCGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCCGTTTCCGGATGTTATCCCCCTGCATAGGGCAGGTTGTCCACGCGTTACTCACCCGTCCG
>PC.593_1302 FLP3FBN01DZILX orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_1303 FLP3FBN01EMNLG orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGCCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.593_1304 FLP3FBN01CJ0R3 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCC
>PC.593_1305 FLP3FBN01EPZM5 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_1306 FLP3FBN01DKURH orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACTCACCCGTTCG
>PC.593_1307 FLP3FBN01A52PB orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.593_1308 FLP3FBN01DKNLI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCCTCACGCGTT
>PC.593_1309 FLP3FBN01BIX4J orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1310 FLP3FBN01CCNMN orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.593_1311 FLP3FBN01EO993 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.593_1312 FLP3FBN01A1THR orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1313 FLP3FBN01A1UPS orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGACTTGGTGGGCCGTTACCTCACCAACTATCTAATCGGGTGCGAGCCCATCTCTAAGCGGATTTCTCCTTTGCCGGTTGTACAATGCTATACTGCCGGTTTATGTGGTATTACCACCCGTTTCCAGGTGCTATCCCTCTCTTAAAGGTAGGTTGCTCACATGTTACTCACCC
>PC.593_1314 FLP3FBN01D1GVB orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1315 FLP3FBN01DCLZP orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCAAGCCCATCTGTAAGTGGATTGCTCCTTTCCTCACTCTAACATGTGTCAGTGTGAGCATATGCGGTATTAGCACCCTATTTCTAACGTGTTATCCCCCAGTATACGGCAGGTTCTCCACG
>PC.593_1316 FLP3FBN01CYQG0 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.593_1317 FLP3FBN01BLZYI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.593_1318 FLP3FBN01CYH88 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.593_1319 FLP3FBN01C5F5F orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_1320 FLP3FBN01BHIQC orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.593_1321 FLP3FBN01B03J2 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTACTCCCCCTTCTTTTGGGCAGGGTTCGCCCCACGTTGTTACT
>PC.593_1322 FLP3FBN01CKVKR orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_1323 FLP3FBN01DBF9Z orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGTCCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAACAAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCAGTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTACTCACCCGTGCGCC
>PC.593_1324 FLP3FBN01APHXO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACT
>PC.593_1325 FLP3FBN01C0GRI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.593_1326 FLP3FBN01CEZZE orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_1327 FLP3FBN01AGO5G orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCGTTCGCC
>PC.593_1328 FLP3FBN01A9WKI orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_1329 FLP3FBN01AMKUE orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1330 FLP3FBN01CD50B orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGTCTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_1331 FLP3FBN01CXEQE orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTT
>PC.593_1332 FLP3FBN01C6MIF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.593_1333 FLP3FBN01CKQYM orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_1334 FLP3FBN01A68AG orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCACCGGAGATGCCTCCAACGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
"""


if __name__ == "__main__":
    main()
