/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA
**********************************************************************************************/

#include "CToolViewDist.h"
#include "CCentralResources.h"
#include "CGarminDBRoute.h"
#include "CGarminRoute.h"
#include "icons.h"

#include <QtGui>

CToolViewDist::CToolViewDist(QWidget * parent, CGarminDBRoute* db)
: QWidget(parent)
, originator(false)
{
    setupUi(this);
    setObjectName("Distance");

    connect(db,SIGNAL(sigDistListChanged()),this,SLOT(slotUpdateDistList()));
    connect(db,SIGNAL(sigRouteChanged()),this,SLOT(slotUpdateDistList()));

    connect(listDistances,SIGNAL(itemClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemClicked(QTreeWidgetItem*)));
    connect(listDistances,SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int) ),this,SLOT(slotItemDoubleClicked(QTreeWidgetItem*)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Delete"),this,SLOT(slotDelete()));

    connect(listDistances,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

}


CToolViewDist::~CToolViewDist()
{

}


void CToolViewDist::keyPressEvent(QKeyEvent * e)
{
    if(e->key() == Qt::Key_Delete) {
        QStringList keys;
        QTreeWidgetItem * item;
        const QList<QTreeWidgetItem*>& items = listDistances->selectedItems();
        foreach(item,items) {
            keys << item->text(0);
            delete item;
        }
        originator = true;
        gpResources->routedb().removeDistances(keys);
        originator = false;
    }
    else {
        QWidget::keyPressEvent(e);
    }
}


void CToolViewDist::slotUpdateDistList()
{
    if(originator) return;
    CGarminRoute * actRoute = gpResources->routedb().getCurrentRoute();
    listDistances->clear();

    const QMap<QString,CGarminRoute*>& routes = gpResources->routedb().getDistanceDict();
    QMap<QString,CGarminRoute*>::const_iterator route = routes.begin();
    while(route != routes.end()) {
        QTreeWidgetItem * entry = new QTreeWidgetItem(listDistances);
        entry->setText(0,(*route)->getName());

        QString str = gpResources->distance2str((*route)->getDistance());
        entry->setText(1,str);
        if(*route == actRoute) entry->setSelected(true);
        ++route;
    }
}


void CToolViewDist::slotItemClicked(QTreeWidgetItem * item)
{
    originator = true;
    gpResources->routedb().highlightDistance(item->text(0));
    originator = false;
}


void CToolViewDist::slotItemDoubleClicked(QTreeWidgetItem * item)
{
    QString key = item->text(0);
    CGarminRoute* route = gpResources->routedb().getDistanceDict()[key];

    QRectF area;

    QList<XY>::const_iterator pt1 = route->routeDegree.begin();
    QList<XY>::const_iterator pt2 = route->routeDegree.begin();
    area.setTopLeft(QPointF(pt1->v,pt1->u));
    area.setBottomRight(QPointF(pt1->v,pt1->u));
    ++pt1;
    while(pt1 != route->routeDegree.end()) {
        QRectF r;
        r.setTopLeft(QPointF(pt1->u,pt1->v));
        r.setBottomRight(QPointF(pt2->u,pt2->v));

        area |= r;
        ++pt1; ++pt2;
    }

    qDebug() << area;
    gpResources->canvas().zoom(area);
}


void CToolViewDist::slotContextMenu(const QPoint& pos)
{
    if(!listDistances->selectedItems().isEmpty()) {
        QPoint p = listDistances->mapToGlobal(pos);
        p.setY(p.y() + listDistances->header()->height());
        contextMenu->exec(p);
    }
}


void CToolViewDist::slotDelete()
{
    QStringList keys;
    QTreeWidgetItem * item;
    const QList<QTreeWidgetItem*>& items = listDistances->selectedItems();
    foreach(item,items) {
        keys << item->text(0);
        delete item;
    }
    originator = true;
    gpResources->routedb().removeDistances(keys);
    originator = false;
}
