/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/
#include "CToolViewTrackInfo.h"
#include "CCentralResources.h"
#include "CGarminDBTrack.h"
#include "GeoMath.h"
#include "icons.h"

#include <IDevice.h>

#include <QtGui>

using namespace Garmin;
using namespace std;

CToolViewTrackInfo::CToolViewTrackInfo(QWidget * parent, CGarminDBTrack * db)
: QWidget(parent)
, originator(false)
{
    setupUi(this);

    connect(butApply,SIGNAL(pressed()),this,SLOT(slotApply()));
    connect(checkRemoveDelTrkPt,SIGNAL(clicked(bool)),this,SLOT(slotCheckRemove(bool)));
    connect(checkResetDelTrkPt,SIGNAL(clicked(bool)),this,SLOT(slotCheckReset(bool)));
    connect(listPoints,SIGNAL(itemSelectionChanged()),this,SLOT(slotPointSelectionChanged()));
    connect(listPoints,SIGNAL(itemClicked(MyQTreeWidgetItem*,int)),this,SLOT(slotPointSelection(MyQTreeWidgetItem*)));

    connect(db,SIGNAL(sigTrackListChanged()),this,SLOT(slotTrackSelectionChanged()));

    listPoints->header()->setResizeMode(0,QHeaderView::Interactive);
    for(int i=0; i< eMaxColumn - 1; ++i) {
        listPoints->resizeColumnToContents(i);
    }

    QPixmap icon(16,8);
    for(int i=0; i < 17; ++i) {
        icon.fill(CGarminTrack::colors[i]);
        comboColor->addItem(icon,"",QVariant(i));
    }

    connect(listPoints,SIGNAL(customContextMenuRequested(const QPoint&)),this,SLOT(slotContextMenu(const QPoint&)));

    contextMenu = new QMenu(this);
    contextMenu->addAction(QPixmap(iconTrash16x16),tr("Purge"),this,SLOT(slotPurge()));

    setEnabled(false);
}


CToolViewTrackInfo::~CToolViewTrackInfo()
{

}


void CToolViewTrackInfo::slotCheckReset(bool checked)
{
    if(checked) {
        checkRemoveDelTrkPt->setChecked(false);
    }
}


void CToolViewTrackInfo::slotCheckRemove(bool checked)
{
    if(checked) {
        checkResetDelTrkPt->setChecked(false);
    }
}


void CToolViewTrackInfo::slotTrackSelectionChanged()
{
    if(originator) return;

    // reset tree widget
    listPoints->clear();
    listPoints->sortItems(eNum,Qt::AscendingOrder);

    track = gpResources->trackdb().highlightedTrack();
    if(track.isNull()) {
        lineIdent->clear();
        setEnabled(false);
        return;
    }
    setEnabled(true);

    QTime time;
    time = time.addSecs(track->getTotalTime());
    lblTotalTime->setText(time.toString("HH:mm:ss"));

    lblTotalDist->setText(gpResources->distance2str(track->getTotalDistance()));

    if(track->getAvgSpeed() > 0) {
        lblAvgSpeed->setText(gpResources->speed2str(track->getAvgSpeed()));
    }
    else {
        lblAvgSpeed->setText(" - ");
    }

    comboColor->setCurrentIndex(((Garmin::Track_t*)track)->color > 16 ? 4 : ((Garmin::Track_t*)track)->color);
    lineIdent->setText(track->getName());

    quint32 ptcnt = 0;
    MyQTreeWidgetItem * focus = 0;
    QVector<CGarminTrack::TrkPt_t>::const_iterator trackpoint = track->begin();
    while(trackpoint != track->end()) {

        // add trackpoint to tree widget
        QString str;
        MyQTreeWidgetItem * item = new MyQTreeWidgetItem(listPoints);
        item->setData(0, Qt::UserRole, trackpoint->idx);

        // gray shade deleted items
        if(trackpoint->flags & CGarminTrack::eDeleted) {
            item->setFlags(item->flags() & ~Qt::ItemIsEnabled);
        }

        // temp. store item of user focus
        if(trackpoint->flags & CGarminTrack::eFocus) {
            focus = item;
        }

        // point number
        item->setText(eNum,QString::number(++ptcnt));

        // timestamp
        if(trackpoint->time != 0x00000000 && trackpoint->time != 0xFFFFFFFF) {
            QDateTime time = QDateTime::fromTime_t(trackpoint->time + gpResources->getUTCOffset());
            time.setTimeSpec(Qt::LocalTime);
            str = time.toString();
        }
        else {
            str = "-";
        }

        item->setText(eTime,str);

        // altitude
        if(trackpoint->alt != 1e25f  || trackpoint->dpth != 1e25f) {
            if(trackpoint->alt != 1e25f) {
                str = gpResources->altitude2str(trackpoint->alt);
            }
            else if(trackpoint->dpth != 1e25f) {
                str = gpResources->altitude2str(-trackpoint->dpth);
            }
        }
        else {
            str = "-";
        }

        item->setText(eAltitude,str);

        str = gpResources->distance2str(trackpoint->delta);
        item->setText(eDelta,str);

        str.sprintf("%1.0f\260",trackpoint->azimuth);
        item->setText(eAzimuth,str);

        str = gpResources->distance2str(trackpoint->distance);
        item->setText(eDistance,str);

        // speed
        if(trackpoint->speed > 0) {
            str = gpResources->speed2str(trackpoint->speed);
        }
        else {
            str = "-";
        }
        item->setText(eSpeed,str);

        //ascend
        str = gpResources->altitude2str(trackpoint->ascend);
        item->setText(eAscend,str);

        //descend
        str = gpResources->altitude2str(trackpoint->descend);
        item->setText(eDescend,str);

        // position
        GPS_Math_Deg_To_Str(trackpoint->lon,trackpoint->lat,str);
        item->setText(ePosition,str);

        item->setTextAlignment(eNum,Qt::AlignLeft);
        item->setTextAlignment(eAltitude,Qt::AlignRight);
        item->setTextAlignment(eDelta,Qt::AlignRight);
        item->setTextAlignment(eAzimuth,Qt::AlignRight);
        item->setTextAlignment(eDistance,Qt::AlignRight);
        item->setTextAlignment(eSpeed,Qt::AlignRight);
        item->setTextAlignment(eAscend,Qt::AlignRight);
        item->setTextAlignment(eDescend,Qt::AlignRight);

        ++trackpoint;
    }

    // adjust column sizes to fit
    listPoints->header()->setResizeMode(0,QHeaderView::Interactive);
    for(int i=0; i< eMaxColumn - 1; ++i) {
        listPoints->resizeColumnToContents(i);
    }

    // scroll to item of user focus
    if(focus) {
        listPoints->setCurrentItem(focus);
        listPoints->scrollToItem(focus);
    }

    lblNoPoints->setText(QString::number(ptcnt));
}


void CToolViewTrackInfo::slotPointSelection(MyQTreeWidgetItem * item)
{
    if(track.isNull()) return;
    originator = true;
    track->setPointOfFocus(item->data(0,Qt::UserRole).toInt());
    originator = false;
}


void CToolViewTrackInfo::slotPointSelectionChanged()
{
    if(track.isNull()) return;

    // reset previous selections
    QVector<CGarminTrack::TrkPt_t>::iterator trkpt = track->begin();
    while(trkpt != track->end()) {
        trkpt->flags &= ~CGarminTrack::eSelected;
        ++trkpt;
    }

    // set eSelected flag for selected points
    QList<QTreeWidgetItem*> items = listPoints->selectedItems();
    QList<QTreeWidgetItem*>::const_iterator item = items.begin();
    while(item != items.end()) {
        quint32 idxTrkPt = (*item)->data(0,Qt::UserRole).toUInt();
        (*track)[idxTrkPt].flags |= CGarminTrack::eSelected;
        ++item;
    }
    originator = true;
    track->update();
    originator = false;
}


void CToolViewTrackInfo::slotApply()
{
    if(track.isNull()) return;

    originator = true;

    if(checkResetDelTrkPt->isChecked()) {
        QVector<CGarminTrack::TrkPt_t>::iterator pt = track->begin();
        while(pt != track->end()) {
            pt->flags &= ~CGarminTrack::eDeleted;
            ++pt;
        }
        checkResetDelTrkPt->setChecked(false);
        originator = false;
    }

    if(checkRemoveDelTrkPt->isChecked()) {
        QMessageBox::warning(0,tr("Remove track points ...")
            ,tr("You are about to remove purged track points permanently. If you press 'yes', all information will be lost.")
            ,QMessageBox::Yes|QMessageBox::No);
        QVector<CGarminTrack::TrkPt_t>::iterator pt = track->begin();
        while(pt != track->end()) {
            if(pt->flags & CGarminTrack::eDeleted) {
                pt = track->erase(pt);
            }
            else {
                ++pt;
            }
        }
        checkRemoveDelTrkPt->setChecked(false);
        originator = false;
    }

    track->setName(lineIdent->text());
    track->setColor(comboColor->currentIndex());
    track->update(true);
    originator = false;
}


void CToolViewTrackInfo::keyPressEvent(QKeyEvent * e)
{
    if(track.isNull()) return;

    if(e->key() == Qt::Key_Delete) {
        slotPurge();
    }
    else {
        QWidget::keyPressEvent(e);
    }
}


void CToolViewTrackInfo::slotPurge()
{
    QList<QTreeWidgetItem*> items = listPoints->selectedItems();
    QList<QTreeWidgetItem*>::const_iterator item = items.begin();
    while(item != items.end()) {
        quint32 idxTrkPt = (*item)->data(0,Qt::UserRole).toUInt();
        if((*track)[idxTrkPt].flags & CGarminTrack::eDeleted) {
            (*track)[idxTrkPt].flags &= ~CGarminTrack::eDeleted;
        }
        else {
            (*track)[idxTrkPt].flags |= CGarminTrack::eDeleted;
        }

        ++item;
    }
    track->update();
}


void CToolViewTrackInfo::slotContextMenu(const QPoint& pos)
{
    if(!listPoints->selectedItems().isEmpty()) {
        QPoint p = listPoints->mapToGlobal(pos);
        p.setY(p.y() + listPoints->header()->height());
        contextMenu->exec(p);
    }
}


bool MyQTreeWidgetItem::operator< ( const QTreeWidgetItem & other ) const
{
    const QString speed("/h");
    const QRegExp distance("(ft|ml|m|km)");
    double d1 = 0, d2 = 0;

    int sortCol = treeWidget()->sortColumn();
    QString str1 = text(sortCol);
    QString str2 = other.text(sortCol);

    if (str1.contains(speed) && str2.contains(speed)) {
        d1 = gpResources->str2speed(str1);
        d2 = gpResources->str2speed(str2);
    }
    else if (str1.contains(distance) && str2.contains(distance)) {
        d1 = gpResources->str2distance(str1);
        d2 = gpResources->str2distance(str2);
    }
    else {
        /* let's assume it's a double without any unit ... */
        d1 = str1.toDouble();
        d2 = str2.toDouble();
    }

    return d1 < d2;
}
