// prefs.C
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997-1999

#include "prefs.h"
#include "qps.h"

#include <qpushbutton.h>
#include <qaccel.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qvalidator.h>

#include <qapplication.h>
#include <qcombobox.h>

// class that validates input for the "swap warning limit" field
class Swapvalid : public QValidator
{
public:
    Swapvalid(QWidget *parent) : QValidator(parent) {};
    virtual State validate(QString &s, int &)
#if QT_VERSION >= 200
	const
#endif
	;
};

struct Boxvar
{
    const char *text;
    bool *variable;
    CrossBox *cb;
};

static Boxvar statbar_boxes[] = {
    {"Graphic Load Display", &Qps::show_load_graph, 0},
    {"Graphic CPU Display", &Qps::show_cpu_bar, 0},
    {"Graphic Memory Display", &Qps::show_mem_bar, 0},
    {"Graphic Swap Display", &Qps::show_swap_bar, 0},
    {"Load Graph in Icon", &Qps::load_in_icon, 0},
    {"Vertical CPU Bar", &Qps::vertical_cpu_bar, 0},
    {0, 0, 0}
};

#ifdef LINUX
static Boxvar sockinfo_boxes[] = {
    {"Host Name Lookup", &Qps::hostname_lookup, 0},
    {"Service Name Lookup", &Qps::service_lookup, 0},
    {0, 0, 0}
};
#endif

static Boxvar tree_boxes[] = {
    {"Disclosure Triangles", &Qps::tree_gadgets, 0},
    {"Branch Lines", &Qps::tree_lines, 0},
    {0, 0, 0}
};

static Boxvar misc_boxes[] = {
    {"Auto Save Settings on Exit", &Qps::auto_save_options, 0},
    {"Set X11 Selection to PIDs", &Qps::pids_to_selection, 0},
#ifdef SOLARIS
    {"Normalize NICE", &Qps::normalize_nice, 0},
#endif
    {0, 0, 0}
};

// dual use function: both validate and apply changes
QValidator::State Swapvalid::validate(QString &s, int &)
#if QT_VERSION >= 200
    const
#endif
{
    // only accept /^[0-9]*[%kKmM]?$/
    int len = s.length();
    int i = 0;
    while(i < len && s[i] >= '0' && s[i] <= '9') i++;
    if(i < len && QString("kKmM%").contains(s[i]) == 0 || i < len - 1)
	return Invalid;
    if(s[i] == 'k')
	s[i] = 'K';
    if(s[i] == 'm')
	s[i] = 'M';
    int val = atoi((const char *)s);
    bool percent;
    if(s[i] == '%') {
	percent = TRUE;
    } else {
	percent = FALSE;
	if(s[i] == 'M')
	    val <<= 10;
    }
    Qps::swaplimit = val;
    Qps::swaplim_percent = percent;
    return Acceptable;
}

struct Cbgroup
{
    const char *caption;
    Boxvar *boxvar;
};

static Cbgroup groups[] = {
    {"Status Bar", statbar_boxes},
#ifdef LINUX
    {"Socket Info Window", sockinfo_boxes},
#endif
    {"Tree View", tree_boxes},
    {"Miscellaneous", misc_boxes},
    {0, 0}
};


Preferences::Preferences()
           : QDialog(0)
{
    setCaption("Preferences");
    QVBoxLayout *tl = new QVBoxLayout(this, 10);

    const int border_x = 10;
    int min_x = 0;
    int top_margin = (int)(fontMetrics().lineSpacing() * 1.5);

    for(Cbgroup *g = groups; g->caption; g++) {
	QButtonGroup *grp = new QButtonGroup(g->caption, this);
	int box_y = top_margin;
	for(Boxvar *b = g->boxvar; b->text; b++) {
	    b->cb = new CrossBox(b->text, grp);
	    b->cb->adjustSize();
	    b->cb->move(border_x, box_y);
	    min_x = QMAX(min_x,  b->cb->sizeHint().width());
	    box_y += b->cb->sizeHint().height();
	}
	connect(grp, SIGNAL(clicked(int)), SLOT(update_reality(int)));
	grp->setMinimumWidth(min_x + 2 * border_x);
	grp->setFixedHeight(box_y + 5);
	tl->addWidget(grp);
    }

    update_boxes();

    QGroupBox *wgrp = new QButtonGroup("Swap Warning", this);
    QLabel *lbl = new QLabel("Warn when free swap below", wgrp);
    lbl->adjustSize();
    lbl->move(10, top_margin);
    QLineEdit *swaped = new QLineEdit(wgrp);
    QString s;
    if(Qps::swaplim_percent) {
	s.sprintf("%d%%", Qps::swaplimit);
    } else {
	if(Qps::swaplimit >= 1024 && (Qps::swaplimit & 0x7ff) == 0)
	    s.sprintf("%dM", Qps::swaplimit >> 10);
	else
	    s.sprintf("%dK", Qps::swaplimit);
    }
    swaped->setText(s);
    swaped->setMaxLength(7);
    swaped->resize(50, swaped->sizeHint().height());
    swaped->move(lbl->x() + lbl->width() + border_x,
		 lbl->y() + (lbl->height() - swaped->height()) / 2);
    swaped->setValidator(new Swapvalid(this));
    connect(swaped, 
#if QT_VERSION < 200
	    SIGNAL(textChanged(const char*)),
#else
	    SIGNAL(textChanged(const QString &)),
#endif
	    SIGNAL(prefs_change()));
    wgrp->setMinimumWidth(swaped->x() + swaped->width() + border_x);
    wgrp->setFixedHeight(swaped->height() + 30);
    tl->addWidget(wgrp);

    QPushButton *closebut = new QPushButton("Close", this);
    closebut->setDefault(TRUE);
    closebut->adjustSize();
    closebut->setFixedSize(closebut->size());
    connect(closebut, SIGNAL(clicked()), SLOT(closed()));
    tl->addSpacing(10);
    tl->addWidget(closebut);
    
    QAccel *acc = new QAccel(this);
    acc->connectItem(acc->insertItem(ALT + Key_W),
		     this, SLOT(closed()));

    tl->freeze();
}

// slot: update check boxes to reflect current status
void Preferences::update_boxes()
{
    for(Cbgroup *g = groups; g->caption; g++)
	for(Boxvar *b = g->boxvar; b->text; b++)
	    b->cb->setChecked(*b->variable);
}

// slot: update flags and repaint to reflect state of check boxes
void Preferences::update_reality(int)
{
    for(Cbgroup *g = groups; g->caption; g++)
	for(Boxvar *b = g->boxvar; b->text; b++)
	    *b->variable = b->cb->isChecked();
    emit prefs_change();
}

void Preferences::closed()
{
    hide();
}
