/* Copyright (C) 2003 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#include "config.h"

#include <vector>

#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qfontinfo.h>

#include "qtadsio.h"
#include "qtadssettings.h"
#include "qtadsmainwindow.h"


/* Returns a string that contains user-visible information about a
 * font.  For example: "Times New Roman [12pt, b]".
 *
 * This is an internal non-member function; it has not been generated
 * by Qt Designer.
 */
static QString
makeFontInfo( const QFont& font )
{
	QFontInfo info(font);
	QString text(info.family() + " [");

	if (info.pointSize() != -1) {
		text.append(QString::number(info.pointSize()));
		text.append("pt");
	} else {
		// The font size is specified in pixels instead of points.
		text.append(QString::number(info.pixelSize()));
		text.append("pix");
	}

	if (info.italic() or info.bold()) {
		text.append(", ");
		info.bold() ? text.append("b") : text.append("i");
	}
	text.append("]");
	return text;
}


void
QTadsConfDialog::init()
{
	const std::vector<QString>& themes = QTadsIO::settings().themes();
	for (std::vector<QString>::const_iterator i(themes.begin()); i != themes.end(); ++i) {
		this->themeBox->insertItem(*i);
		if (*i == QTadsIO::settings().nameOfCurrentTheme()) {
			this->themeBox->setCurrentText(*i);
		}
	}
	this->switchToTheme(this->themeBox->currentText());
}


void
QTadsConfDialog::selectMainFont()
{
	this->fSelectedMainFont = QFontDialog::getFont(0, this->previewText->currentFont(),
						       this, "main font selection dialog");

	this->fFontInfo->setText(makeFontInfo(this->fSelectedMainFont));
	this->updatePreview();
}


void
QTadsConfDialog::selectMainTextColor()
{
	QColor color(QColorDialog::getColor(this->previewText->color(), this,
					    "main text color selection dialog"));

	if (not color.isValid()) {
		return;
	}

	this->fSelectedMainTextColor = color;
	this->updatePreview();
}


void
QTadsConfDialog::selectMainBgColor()
{
	QColor color(QColorDialog::getColor(this->previewText->paper().color(), this,
					    "main background color selection dialog"));

	if (not color.isValid()) {
		return;
	}

	this->fSelectedMainBgColor = color;
	this->updatePreview();
}


void
QTadsConfDialog::selectStatusFont()
{
	this->fSelectedStatusFont
		= QFontDialog::getFont(0, QTadsIO::settings().currentTheme().statusFont(),
				       this, "status font selection dialog");

	this->fStatusFontInfo->setText(makeFontInfo(this->fSelectedStatusFont));
}


void
QTadsConfDialog::selectStatusTextColor()
{
	QColor color(QColorDialog::getColor(QTadsIO::settings().currentTheme().statusTextColor(),
					    this, "status text color selection dialog"));

	if (not color.isValid()) {
		return;
	}

	this->fSelectedStatusTextColor = color;
}


void
QTadsConfDialog::selectStatusBgColor()
{
	QColor color(QColorDialog::getColor(QTadsIO::settings().currentTheme().statusBgColor(),
					    this,
					    "status background color selection dialog"));

	if (not color.isValid()) {
		return;
	}

	this->fSelectedStatusBgColor = color;
}


void
QTadsConfDialog::leftAlignPreview( bool on )
{
	if (not on) return;

	this->fSelectedAlignment = Qt::AlignLeft;
	this->updatePreview();
}


void
QTadsConfDialog::centerAlignPreview( bool on )
{
	if (not on) return;

	this->fSelectedAlignment = Qt::AlignCenter;
	this->updatePreview();
}


void
QTadsConfDialog::justifyAlignPreview( bool on )
{
	if (not on) return;

	this->fSelectedAlignment = Qt::AlignJustify;
	this->updatePreview();
}


void
QTadsConfDialog::rightAlignPreview( bool on )
{
	if (not on) return;

	this->fSelectedAlignment = Qt::AlignRight;
	this->updatePreview();
}


void
QTadsConfDialog::accept()
{
	// Save the currently selected theme.
	this->saveTheme();

	// Save the settings that aren't part of the theme.
	switch (this->bufferSizeComboBox->currentItem()) {
	  case 0:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize1);
		break;
	  case 1:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize2);
		break;
	  case 2:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize3);
		break;
	  case 3:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize4);
		break;
	  case 4:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize5);
		break;
	  case 5:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize6);
		break;
	  case 6:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize7);
		break;
	  case 7:
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize8);
		break;
	  default:
		// Paranoia.
		QTadsIO::settings().scrollBufferSize(QTadsSettings::BufferSize1);
	}
	QTadsIO::settings().immediateQuit(this->immediateQuitButton->isChecked());
	QTadsIO::settings().immediateRestart(this->immediateRestartButton->isChecked());

	// Activate the currently selected theme and apply the settings.
	QTadsIO::settings().currentTheme(this->themeBox->currentText());
	QTadsIO::applySettings();

	// Update the main window's "Display->Theme" menu, since the
	// user might have switched the current theme.
	QTadsIO::mainWindow().updateThemeList();

	// Let the inherited class take care of deleting widgets and
	// stuff.
	QDialog::accept();
}


void
QTadsConfDialog::updatePreview()
{
	this->previewText->setReadOnly(false);
	for (int i = 0; i < this->previewText->paragraphs(); ++i) {
		this->previewText->setCursorPosition(i, 0);
		this->previewText->setAlignment(this->fSelectedAlignment);
	}
	this->previewText->setReadOnly(true);
	this->previewText->setUpdatesEnabled(false);
	this->previewText->setFont(this->fSelectedMainFont);
	this->previewText->setPaper(this->fSelectedMainBgColor);
	this->previewText->setPaletteForegroundColor(this->fSelectedMainTextColor);
	this->previewText->setUpdatesEnabled(true);
	this->previewText->repaint(false);
}


/* Delete the currently selected theme.
 */
void
QTadsConfDialog::deleteTheme()
{
	// Remove it from the list.  If false is returned, it means
	// that it could not be removed; the only way this can happen
	// is when we try to remove the "Default" theme.
	if (not QTadsIO::settings().removeTheme(this->themeBox->currentText())) {
		Q_ASSERT(this->themeBox->currentText() == "Default");

		QMessageBox::information(this, tr("Cannot delete theme"),
					 tr("It's not possible to delete this theme."),
					 QMessageBox::Ok);
		return;
	}

	// Also remove it from the dialog's themeBox.
	this->themeBox->removeItem(this->themeBox->currentItem());

	// Since we removed the current item, we must activate the one
	// that is now selected (whatever that is).
	this->switchToTheme(this->themeBox->currentText());

	// Update the main window's "Display->Theme" menu.
	QTadsIO::mainWindow().updateThemeList();
}


/* Create a new theme by copying the currently selected one.
 */
void
QTadsConfDialog::newTheme()
{
	QString themeName;
	bool ok = true;

	// Repeatedly ask the user for a name until that name is valid.
	while (themeName.isEmpty() and ok) {
		themeName = QInputDialog::getText(tr("Theme name"),
						  tr("What name would you like to give the new theme?"),
						  QLineEdit::Normal, QString::null, &ok, this,
						  "theme name input dialog");
		// No empty string allowed.
		if (themeName.isEmpty() and ok) {
			QMessageBox::information(this, tr("Invalid theme name"),
						 tr("Obviously, an empty string is not a valid name for a theme."),
						    QMessageBox::Ok);
		}

		// Check if this name is already used by another theme.
		for (int i = 0; i < this->themeBox->count() and not themeName.isEmpty(); ++i) {
			if (themeName == this->themeBox->text(i)) {
				// Name already in use.  Make the
				// top-level loop execute again.
				themeName = QString::null;
				QMessageBox::information(this, tr("Invalid theme name"),
							 tr("This name is already used by another theme. Please choose a different one."),
							 QMessageBox::Ok);
			}
		}
	}

	if (not ok) {
		// The user cancelled.  Do nothing.
		return;
	}

	// When we get here, the name should be valid.

	// First, create an entry in the theme box and make it the
	// current one.
	this->themeBox->insertItem(themeName);
	this->themeBox->setCurrentText(themeName);

	// Now simply save it as a new theme.
	this->saveTheme();

	// Let the main window know.
	QTadsIO::mainWindow().updateThemeList();
}


/* Save the currently selected theme.
 */
void
QTadsConfDialog::saveTheme()
{
	// Construct the theme.
	QTadsTheme theme(this->fSelectedMainBgColor, this->fSelectedMainTextColor,
			 this->fSelectedStatusBgColor, this->fSelectedStatusTextColor,
			 this->fSelectedMainFont, this->fSelectedStatusFont,
			 this->fSelectedAlignment, this->leftMarginSpinBox->value(),
			 this->rightMarginSpinBox->value(),
			 this->doubleSpaceCheckBox->isEnabled() ?
			     this->doubleSpaceCheckBox->isChecked()
			   : false,
			 this->inputBoldCheckBox->isChecked(),
			 this->inputItalicCheckBox->isChecked(),
			 this->inputUnderlinedCheckBox->isChecked(),
			 this->curlyQuotesCheckBox->isChecked(),
			 this->curlyApostrophesCheckBox->isChecked(),
			 this->dashConversionCheckBox->isChecked()
			);

	// Add it to the list.  If it already exists, it will be
	// updated with the one we just created.
	QTadsIO::settings().addTheme(theme, this->themeBox->currentText());
}


void
QTadsConfDialog::switchToTheme( const QString& theme )
{
	const QTadsSettings& sett = QTadsIO::settings();
	const QTadsTheme& thm = QTadsIO::settings().theme(theme);

	// Initialize the current settings.
	this->fSelectedMainFont = thm.gameFont();
	this->fSelectedMainTextColor = thm.gameTextColor();
	this->fSelectedMainBgColor = thm.gameBgColor();
	this->fSelectedStatusFont = thm.statusFont();
	this->fSelectedStatusTextColor = thm.statusTextColor();
	this->fSelectedStatusBgColor = thm.statusBgColor();
	this->fSelectedAlignment = thm.alignment();
	this->fSelectedLeftMargin = thm.leftMargin();
	this->fSelectedRightMargin = thm.rightMargin();
	this->fSelectedDoubleSpace = thm.doubleSpace();
	this->fSelectedBufferSize = sett.scrollBufferSize();
	this->fSelectedImmediateQuit = sett.immediateQuit();
	this->fSelectedImmediateRestart = sett.immediateRestart();

	// Initialize the dialog's widgets with the current settings.
	this->fFontInfo->setText(makeFontInfo(this->fSelectedMainFont));
	this->fStatusFontInfo->setText(makeFontInfo(this->fSelectedStatusFont));
	this->doubleSpaceCheckBox->setChecked(this->fSelectedDoubleSpace);
	this->inputBoldCheckBox->setChecked(thm.boldInput());
	this->inputItalicCheckBox->setChecked(thm.italicInput());
	this->inputUnderlinedCheckBox->setChecked(thm.underlinedInput());
	this->curlyQuotesCheckBox->setChecked(thm.curlyQuotes());
	this->curlyApostrophesCheckBox->setChecked(thm.curlyApostrophes());
	this->dashConversionCheckBox->setChecked(thm.dashConversion());
	this->leftMarginSpinBox->setValue(this->fSelectedLeftMargin);
	this->rightMarginSpinBox->setValue(this->fSelectedRightMargin);
	switch (this->fSelectedAlignment) {
	  case Qt::AlignLeft:
		this->fAlignLeftButton->setChecked(true);
		break;
	  case Qt::AlignRight:
		this->fAlignRightButton->setChecked(true);
		break;
	  case Qt::AlignCenter:
		this->fAlignCenterButton->setChecked(true);
		break;
	  case Qt::AlignJustify:
		this->fAlignJustifyButton->setChecked(true);
		break;
	  default:
		// Paranoia.
		this->fAlignLeftButton->setChecked(true);
	}
	switch (this->fSelectedBufferSize) {
	  case QTadsSettings::BufferSize1:
		this->bufferSizeComboBox->setCurrentItem(0);
		break;
	  case QTadsSettings::BufferSize2:
		this->bufferSizeComboBox->setCurrentItem(1);
		break;
	  case QTadsSettings::BufferSize3:
		this->bufferSizeComboBox->setCurrentItem(2);
		break;
	  case QTadsSettings::BufferSize4:
		this->bufferSizeComboBox->setCurrentItem(3);
		break;
	  case QTadsSettings::BufferSize5:
		this->bufferSizeComboBox->setCurrentItem(4);
		break;
	  case QTadsSettings::BufferSize6:
		this->bufferSizeComboBox->setCurrentItem(5);
		break;
	  case QTadsSettings::BufferSize7:
		this->bufferSizeComboBox->setCurrentItem(6);
		break;
	  case QTadsSettings::BufferSize8:
		this->bufferSizeComboBox->setCurrentItem(7);
		break;
	  default:
		// Paranoia.
		this->bufferSizeComboBox->setCurrentItem(0);
	}
	if (this->fSelectedImmediateQuit) {
		this->immediateQuitButton->setChecked(true);
	} else {
		this->gameQuitButton->setChecked(true);
	}
	if (this->fSelectedImmediateRestart) {
		this->immediateRestartButton->setChecked(true);
	} else {
		this->gameRestartButton->setChecked(true);
	}

	// Update the preview.
	this->updatePreview();
}
