/****************************************************************************
**
** Copyright (C) 2006-2007 Urs Wolfer <uwolfer @ fwo.ch>
**
** This file is part of QtEmu.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU Library General Public License
** along with this library; see the file COPYING.LIB.  If not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
** Boston, MA 02110-1301, USA.
**
****************************************************************************/

#include <QCoreApplication>
#include <QSettings>
#include <QTime>
#include <QDir>

#include "config.h"

#include "machineprocess.h"

MachineProcess::MachineProcess(QObject *parent)
    : QProcess(parent)
{
    pathString = QString();
    cdRomPathString = QString();
    snapshotEnabled = false;
    networkEnabled = false;
}

void MachineProcess::start()
{
    QStringList arguments;
    arguments << pathString;

    if (snapshotEnabled)
        arguments << "-snapshot";

    if (networkEnabled)
        arguments << "-net" << "nic" << "-net" << "user";
    else
        arguments << "-net" << "none";

    if (cdRomPathString != QString())
    {
        arguments << "-cdrom" << cdRomPathString;
        if (bootFromCdEnabled)
            arguments << "-boot" << "d";
    }

    if (floppyDiskPathString != QString())
    {
        arguments << "-fda" << floppyDiskPathString;
        if (bootFromFloppyEnabled)
            arguments << "-boot" << "a";
    }

    arguments << "-m" << QString::number(memoryInt);

    arguments << "-smp" << QString::number(cpuInt);

    if (mouseEnabled)
        arguments << "-usb" << "-usbdevice" << "tablet";

    if (timeEnabled)
        arguments << "-localtime";

#ifdef DEVELOPER
    QString debugString = QString();
    for (int i = 0; i < arguments.size(); ++i)
        debugString = debugString + arguments.at(i).toLocal8Bit().constData() + ' ';
    qDebug(debugString.toLocal8Bit().constData());
#endif

    connect(this, SIGNAL(finished(int)), SLOT(afterExitExecute()));

    QSettings settings("QtEmu", "QtEmu");

    QString command = settings.value("beforeStart").toString();
    if (!command.isEmpty())
    {
        QStringList commandList;
        commandList = command.split("\n");
        for (int i = 0; i < commandList.size(); ++i)
        {
            QProcess::start(commandList.at(i).toLocal8Bit().constData());
            while (waitForFinished())
            {
                QTime sleepTime = QTime::currentTime().addMSecs(5);
                while (QTime::currentTime() < sleepTime)
                    QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
            }
        }
    }

#ifndef Q_OS_WIN32
    QProcess::start(settings.value("command", "qemu").toString(), arguments);
#elif defined(Q_OS_WIN32)
    arguments << "-L" << ".";
    QString qemuCommand = settings.value("command", QCoreApplication::applicationDirPath() + "/qemu/qemu.exe").toString();
    QDir *path = new QDir(qemuCommand);
    path->cdUp();
    setWorkingDirectory(path->path());
    QProcess::start(qemuCommand, arguments);
#endif
}

void MachineProcess::afterExitExecute()
{
    QSettings settings("QtEmu", "QtEmu");

    QString command = settings.value("afterExit").toString();
    if (!command.isEmpty())
    {
        QStringList commandList;
        commandList = command.split("\n");
        for (int i = 0; i < commandList.size(); ++i)
        {
            QProcess::start(commandList.at(i).toLocal8Bit().constData());
            while (waitForFinished())
            {
                QTime sleepTime = QTime::currentTime().addMSecs(5);
                while (QTime::currentTime() < sleepTime)
                    QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
            }
        }
    }
}

void MachineProcess::path(const QString &newPath)
{
    pathString = newPath;
}

void MachineProcess::cdRomPath(const QString &newPath)
{
    cdRomPathString = newPath;
}

void MachineProcess::cdBoot(int value)
{
    if (value == Qt::Checked)
        bootFromCdEnabled = true;
    else
        bootFromCdEnabled = false;
}

void MachineProcess::floppyDiskPath(const QString &newPath)
{
    floppyDiskPathString = newPath;
}

void MachineProcess::floppyBoot(int value)
{
    if (value == Qt::Checked)
        bootFromFloppyEnabled = true;
    else
        bootFromFloppyEnabled = false;
}

void MachineProcess::snapshot(int value)
{
    if (value == Qt::Checked)
        snapshotEnabled = true;
    else
        snapshotEnabled = false;
#ifdef DEVELOPER
    snapshotEnabled = true;
#endif
}

void MachineProcess::network(int value)
{
    if (value == Qt::Checked)
        networkEnabled = true;
    else
        networkEnabled = false;
}

void MachineProcess::time(int value)
{
    if (value == Qt::Checked)
        timeEnabled = true;
    else
        timeEnabled = false;
}

void MachineProcess::mouse(int value)
{
    if (value == Qt::Checked)
        mouseEnabled = true;
    else
        mouseEnabled = false;
}

void MachineProcess::memory(int value)
{
    memoryInt = value;
}

void MachineProcess::cpu(int value)
{
    cpuInt = value;
}
