/****************************************************************************
**
** Copyright (C) 2012 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of the QtLocation module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
***************************************************************************/

#include "qdeclarativegeocircle.h"

#include <QtCore/qnumeric.h>
#include <QtLocation/QGeoCircle>

#include <QtCore/QDebug>

QT_BEGIN_NAMESPACE

/*!
    \qmlbasictype geocircle
    \inqmlmodule QtLocation 5.0
    \ingroup qml-QtLocation5-positioning
    \since Qt Location 5.0

    \brief The geocircle type represents a circular geographic area.

    The \c geocircle type is a \l {QtLocation5::geoshape}{geoshape} that represents a circular
    geographic area.  It is defined in terms of a \l {QtLocation5::coordinate}{coordinate} which
    specifies the \c center of the circle and a qreal which specifies the \c radius of the circle
    in meters.

    The circle is considered invalid if the center coordinate is invalid or if the radius is less
    than zero.

    \section2 Example Usage

    Use properties of type \l variant to store a \c {geocircle}.  To create a \c geocircle value,
    use the \l {QtLocation5::QtLocation}{QtLocation.circle()} function:

    \qml
    import QtLocation 5.0

    Item {
        property variant region: QtLocation.circle(QtLocation.coordinate(-27.5, 153.1), 1000)
    }
    \endqml

    When integrating with C++, note that any QGeoCircle value passed into QML from C++ is
    automatically converted into a \c geocircle value, and vise-versa.
*/

GeoCircleValueType::GeoCircleValueType(QObject *parent)
:   GeoShapeValueType(qMetaTypeId<QGeoCircle>(), parent)
{
}

GeoCircleValueType::~GeoCircleValueType()
{
}

/*
    This property holds the coordinate of the center of the geocircle.
*/
QGeoCoordinate GeoCircleValueType::center()
{
    return QGeoCircle(v).center();
}

void GeoCircleValueType::setCenter(const QGeoCoordinate &coordinate)
{
    QGeoCircle c = v;

    if (c.center() == coordinate)
        return;

    c.setCenter(coordinate);
    v = c;
}

/*!
    This property holds the radius of the geocircle in meters.

    The default value for the radius is -1 indicating an invalid geocircle area.
*/
qreal GeoCircleValueType::radius() const
{
    return QGeoCircle(v).radius();
}

void GeoCircleValueType::setRadius(qreal radius)
{
    QGeoCircle c = v;

    if (c.radius() == radius)
        return;

    c.setRadius(radius);
    v = c;
}

QString GeoCircleValueType::toString() const
{
    if (v.type() != QGeoShape::CircleType) {
        qWarning("Not a circle");
        return QStringLiteral("QGeoCircle(not a circle)");
    }

    QGeoCircle c = v;
    return QStringLiteral("QGeoCircle({%1, %2}, %3)")
        .arg(c.center().latitude())
        .arg(c.center().longitude())
        .arg(c.radius());
}

void GeoCircleValueType::setValue(const QVariant &value)
{
    if (value.userType() == qMetaTypeId<QGeoCircle>())
        v = value.value<QGeoCircle>();
    else if (value.userType() == qMetaTypeId<QGeoShape>())
        v = value.value<QGeoShape>();
    else
        v = QGeoCircle();

    onLoad();
}

QVariant GeoCircleValueType::value()
{
    return QVariant::fromValue(QGeoCircle(v));
}

void GeoCircleValueType::write(QObject *obj, int idx, QQmlPropertyPrivate::WriteFlags flags)
{
    QGeoCircle c = v;
    writeProperty(obj, idx, flags, &c);
}

void GeoCircleValueType::writeVariantValue(QObject *obj, int idx, QQmlPropertyPrivate::WriteFlags flags, QVariant *from)
{
    if (from->userType() == qMetaTypeId<QGeoCircle>()) {
        writeProperty(obj, idx, flags, from);
    } else if (from->userType() == qMetaTypeId<QGeoShape>()) {
        QGeoCircle c = from->value<QGeoShape>();
        QVariant v = QVariant::fromValue(c);
        writeProperty(obj, idx, flags, &v);
    } else {
        QVariant v = QVariant::fromValue(QGeoCircle());
        writeProperty(obj, idx, flags, &v);
    }
}

#include "moc_qdeclarativegeocircle.cpp"

QT_END_NAMESPACE
