% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/swish.R
\name{swish}
\alias{swish}
\title{Swish method: differential expression accounting for inferential uncertainty}
\usage{
swish(
  y,
  x,
  cov = NULL,
  pair = NULL,
  interaction = FALSE,
  cor = c("none", "spearman", "pearson"),
  nperms = 100,
  estPi0 = FALSE,
  qvaluePkg = "qvalue",
  pc = 5,
  nRandomPairs = 30,
  fast = 1,
  returnNulls = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{y}{a SummarizedExperiment containing the inferential replicate
matrices of median-ratio-scaled TPM as assays 'infRep1', 'infRep2', etc.}

\item{x}{the name of the condition variable. A factor with two
levels for a two group analysis (possible to adjust for covariate
or matched samples, see next two arguments). The log fold change is
computed as non-reference level over reference level (see vignette:
'Note on factor levels')}

\item{cov}{the name of the covariate for adjustment.
If provided a stratified Wilcoxon in performed.
Cannot be used with \code{pair}, unless using
either \code{interaction} or \code{cor}}

\item{pair}{the name of the pair variable, which should be the
number of the pair. Can be an integer or factor.
If specified, a signed rank test is used
to build the statistic. All samples across \code{x} must be
pairs if this is specified. Cannot be used with \code{cov},
unless using either \code{interaction} or \code{cor}}

\item{interaction}{logical, whether to perform a test of an interaction
between \code{x} and \code{cov}. Can use \code{pair} or not.
See Details.}

\item{cor}{character, whether to compute correlation of \code{x}
with the log counts, and signifance testing on the correlation
as a test statistic.
Either \code{"spearman"} or \code{"pearson"} correlations can be computed.
For Spearman the correlation is computed over ranks of \code{x} and
ranks of inferential replicates. For Pearson, the correlation is
computed for \code{x} and log2 of the inferential replicates plus \code{pc}.
Default is \code{"none"}, e.g. two-group comparison using the rank sum test
or other alternatives listed above.
Additionally, correlation can be computed between a continuous variable
\code{cov} and log fold changes across \code{x} matched by \code{pair}}

\item{nperms}{the number of permutations. if set above the possible
number of permutations, the function will print a message that the
value is set to the maximum number of permutations possible}

\item{estPi0}{logical, whether to estimate pi0}

\item{qvaluePkg}{character, which package to use for q-value estimation,
\code{samr} or \code{qvalue}}

\item{pc}{pseudocount for finite estimation of \code{log2FC}, not used
in calculation of test statistics, \code{locfdr} or \code{qvalue}}

\item{nRandomPairs}{the number of random pseudo-pairs (only used with
\code{interaction=TRUE} and un-matched samples) to use to calculate
the test statistic}

\item{fast}{an integer, toggles different methods based on speed
(\code{fast=1} is default, \code{0} is slower). See Details.}

\item{returnNulls}{logical, only return the \code{stat} vector,
the \code{log2FC} vector, and the \code{nulls} matrix
(default FALSE)}

\item{quiet}{display no messages}
}
\value{
a SummarizedExperiment with metadata columns added:
the statistic (either a centered Wilcoxon Mann-Whitney
or a signed rank statistic, aggregated over inferential replicates),
a log2 fold change (the median over inferential replicates,
and averaged over pairs or groups (if groups, weighted by sample size),
the local FDR and q-value, as estimated by the \code{samr} package.
}
\description{
The Swish method, or "SAMseq With Inferential Samples Helps".
Performs non-parametric inference on rows of \code{y} for
various experimental designs. See References for details.
}
\details{
\strong{interaction:}
The interaction tests are different than the
other tests produced by \code{swish}, in that they focus on a difference
in the log2 fold change across levels of \code{x} when comparing
the two levels in \code{cov}. If \code{pair} is specified, this
will perform a Wilcoxon rank sum test on the two groups
of matched sample LFCs. If \code{pair} is not included, multiple
random pairs of samples within the two groups are chosen,
and again a Wilcoxon rank sum test compared the LFCs across groups.

\strong{fast:}
'0' involves recomputing ranks of the inferential replicates for
each permutation, '1' (default) is roughly 10x faster by avoiding
re-computing ranks for each permutation.
The \code{fast} argument is only relevant for the following three
experimental designs: (1) two group Wilcoxon, (2) stratified Wilcoxon, e.g.
\code{cov} is specified, and (3) the paired interaction test,
e.g. \code{pair} and \code{cov} are specified. For paired design and
general interaction test, there are not fast/slow alternatives.
}
\examples{

library(SummarizedExperiment)
set.seed(1)
y <- makeSimSwishData()
y <- scaleInfReps(y)
y <- labelKeep(y)
y <- swish(y, x="condition")

# histogram of the swish statistics
hist(mcols(y)$stat, breaks=40, col="grey")
cols = rep(c("blue","purple","red"),each=2)
for (i in 1:6) {
  arrows(mcols(y)$stat[i], 20,
         mcols(y)$stat[i], 10,
         col=cols[i], length=.1, lwd=2)
}

# plot inferential replicates
plotInfReps(y, 1, "condition")
plotInfReps(y, 3, "condition")
plotInfReps(y, 5, "condition")

}
\references{
The citation for \code{swish} method is:

Anqi Zhu, Avi Srivastava, Joseph G Ibrahim, Rob Patro, Michael I Love
"Nonparametric expression analysis using inferential replicate counts"
Nucleic Acids Research (2019). \url{https://doi.org/10.1093/nar/gkz622}

The \code{swish} method builds upon the \code{SAMseq} method,
and extends it by incorporating inferential uncertainty, as well
as providing methods for additional experimental designs (see vignette).

For reference, the publication describing the \code{SAMseq} method is:

Jun Li and Robert Tibshirani "Finding consistent patterns:
A nonparametric approach for identifying differential expression
in RNA-Seq data" Stat Methods Med Res (2013).
\url{https://doi.org/10.1177/0962280211428386}
}
