#!/usr/bin/tclsh
#
# raccess4vbox3.tcl
#
# $Id: raccess4vbox3.tcl,v 1.7 2003/10/17 13:14:33 pape Exp $
#
#    Copyright (C) 2000  G. Pape <pape@smarden.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

set tuning(actualmode) ""

###############################################################################
# repeat
proc repeat { } {
    return "REPEAT"
}

###############################################################################
# hangup
proc hangup { } {
    return "HANGUP"
}

###############################################################################
# skip
proc skip { } {
    return "OK"
}

###############################################################################
# log via vboxgetty
proc raccess_log { level msg } {
#    puts "raccess: $msg"
    vbox_log $level "raccess: $msg"
}

###############################################################################
# read rc-file
proc raccess_readrc { { rcfilename "raccess.conf" } } {
    # parse sections [tuning] [run] [dtmf] [events]

    global tuning

    set tuning(rcfilename) $rcfilename

    uplevel #0 {
	if { [ string compare $tuning(rcfilename) "raccess.conf" ] == 0 } {
	    set prefix ""
	    set tuning(rcfilename) "$vbxv_userhome/$tuning(rcfilename)"
	} else {
	    set prefix "$tuning(rcfilename)_"
	    set tuning(rcfilename) "$vbxv_userhome/raccess/$tuning(rcfilename)"
	}
	raccess_log D "readrc: rcfilename $tuning(rcfilename), prefix $prefix"

	if { [ info exists ${prefix}run ] && \
		[ string length $prefix ] > 0 } {
	    raccess_log D "readrc: file $tuning(rcfilename) allready read, skipping"
	    return
	}

	if [ file readable $tuning(rcfilename) ] {
	    set rc [ open $tuning(rcfilename) r ]
	    raccess_log D "readrc: $tuning(rcfilename) opened"
	    
	    for { set i 1 } { ! [ eof $rc ] } { incr i } {
		# empty line
		if { [ gets $rc line ] == 0 } { continue }
		# glue
		while { [ string compare [ string index $line \
			[ expr [ string length $line ] -1 ] ] \\ ] == 0 } {
		    if { [ gets $rc line2 ] == 0 } { continue }
		    set line [ string range $line 0 \
			    [ expr [ string length $line ] -2 ] ]
		    set line "$line$line2"
		}
		# comment char
		if { [ set no [ string first ";" $line ] ] != -1 } {
		    set line [ string range $line 0 [ expr $no - 1 ] ]
		}
		# empty line again
		if { [ llength $line ] == 0 } { continue }
		set word [ lindex $line 0 ]

		# section ?
		if { [ string index $word 0 ] == "\[" } {
		    set word [ string trim $word \[\] ]
		    set arrname "$prefix$word"
		    raccess_log D "readrc: section == $arrname"
		    
		    # extra parse section run
		    if { [ string compare $word "run" ] == 0 } {
			for { set j 0 } { ! [ eof $rc ] } { incr i } {
			    # empty line
			    if { [ gets $rc line ] == 0 } { continue }
			    # comment char
			    if { [ set no [ string first ";" $line ] ] != -1 } {
				set line [ string range $line 0 [ expr $no - 1 ] ]
			    }
			    # empty line again
			    if { [ llength $line ] == 0 } { continue }
			    set word [ lindex $line 0 ]
			    
			    # section ?
			    if { [ string index $word 0 ] == "\[" } {
				set word [ string trim $word \[\] ]
				set arrname "$prefix$word"
				raccess_log D "readrc: section == $arrname"
				break;
			    }
			    set ${arrname}($j) $line
			    raccess_log D \
				    "readrc: setting ${arrname}($j) $line"
			    incr j
			}
		    }
		} else {
		    if { [ llength $line ] >= 2 } {
			if [ info exists arrname ] {
			    set ${arrname}($word) "[ lindex $line 1 ]"
			    raccess_log D \
				    "readrc: setting ${arrname}($word) = [ lindex $line 1 ]"
			    if { [ llength $line ] == 3 } {
				set ${arrname}_eval($word) "[ lindex $line 2 ]"
				raccess_log D \
					"readrc: setting ${arrname}_eval($word) [ lindex $line 2 ]"
			    }
			} else {
			    raccess_log W \
				    "readrc: Ups: arrname not set"
			}
		    } else {
			raccess_log W \
				"readrc: at least 2 words expected in line $i"
		    }
		} 
	    }
	    catch { close $rc }
	} else {
	    raccess_error "readrc: file $tuning(rcfilename) is not readable"
	}
    }
}

###############################################################################
# raccess_error
proc raccess_error { msg } {
    global events

    raccess_log W "ERROR: $msg"
    if { [ info exists events(error) ] } {
	return [ raccess_run "play $events(error)" ]
    }
    return "OK"
}

###############################################################################
# raccess_parseargs
proc raccess_parseargs { carg } {
    
    while { [ set p [ string first "&" $carg ] ] > -1 } {
	lappend rcarg [ raccess_parseargs \
		[ string range $carg 0 [ expr $p -1 ] ] ]
	set carg [ string range $carg [ expr $p +1 ] end ]
    }
    set carg [ string trim $carg " \t\n\r\{\}" ]
    if { $carg != 0 } {
	set carg [ string trimleft $carg "0" ]
    }
    lappend rcarg $carg
    raccess_log D "parseargs: parsed: $rcarg"
    return "$rcarg"
}

###############################################################################
# record
proc record { what } {
    switch $what {
	on {
	    raccess_log I "start record"
	    set rthat [ vbox_voice r start ]
	}
	off {
	    raccess_log I "stop record"
	    set rthat [ vbox_voice r stop ]
	}
	delete {
	    global vbxv_saveulaw vbxv_savevbox
	    
	    raccess_log D "record delete: deleting $vbxv_saveulaw $vbxv_savevbox"
	    file delete $vbxv_saveulaw $vbxv_savevbox
	    set rthat "OK"
	}
	default {
	    raccess_error \
		    "run: \[run\]: usage: record on\|off"
	    set rthat "OK"
	}
    }
    return "$rthat"
}

###############################################################################
# wait
proc wait { what } {
    global vbxv_savetime

    if { [ string compare "$what" "savetime" ] == 0 } {
	set what $vbxv_savetime
    }
    raccess_log D "wait: $what seconds (using play)"
    return [ play "%$what" ]
}

###############################################################################
# play

proc play { args } {
    global tuning
    global vbxv_callerid vbxv_callername vbxv_localphone vbxv_savevbox

    raccess_log D "play: $args"
    set carg [ raccess_parseargs $args ]
    foreach m $carg {
	set m [ string trim $m " \t\{\}" ]
	if { [ string length $m ] == 0 } {
		continue;
	}
	raccess_log D "play: $m"
	
	switch [ string index $m 0 ] {
	    "@" {
		# play from file
		set m [ string range $m 1 end ]
		raccess_log D "play: file \"$m\""
		set fn $m
	    }
	    "|" {
		# run command
		set m [ string range $m 1 end ]
		raccess_log D "play: running \"$m\""
		
		if [ catch { eval exec -- $m } rthat ] {
		    raccess_error "play: error: $m: $rthat"
		    set that "OK"
		} else {
		    raccess_log D "play: command $m: $rthat"
		    if { [ string length $rthat ] > 0 } {
			foreach line [ split "$rthat" \n ] {
			    raccess_log D "play: command $m: part: $line"
			    set that [ raccess_run "play $line" ]
			    if { [ string compare $that "EXIT" ] == 0 } {
				return "EXIT"
			    }
			}
		    } else {
		        set that "OK"
		    }
		}
		continue
	    }
	    "%" {
		# wait (hack)
		set sec [ string range $m 1 end ]
		raccess_log D "play: wait $sec sec"
		set that [ vbox_voice w $sec ]
		continue
	    }
	    default {
		# synth or play cache
		raccess_log D "play: synth or play cache \"$m\""

		# get the filename
		set fntmp ""
		if { [ set len [ string length $m ] ] \
			> $tuning(say_maxfilenamelen) } {
		    set fntmp "[ string range $m 0 $tuning(say_maxfilenamelen) ].$len"
		} else {
		    set fntmp $m
		}

		set fn "$tuning(say_cachedir)/$fntmp"
		if { ! [ file exists $fn ] } {
		    if { $tuning(say) } {
			if { [ info exists events(say_delay) ] } {
			    raccess_run "play $events(say_delay)"
			}
			raccess_log D "running: exec -- $tuning(echo_path) $m | $tuning(say_path) > $fn"
			if [ catch { exec -- $tuning(echo_path) $m | \
				$tuning(say_path) > \
				$fn } that ] {
			    unset fn
			    raccess_log W "play: synth: error: $that"
			}
		    } else {
			raccess_error \
			    "play: say disabled: file not found: \"$fn\""
			unset fn
		    }
		} else {
		    raccess_log D "play: file $fn found"
		}
	    }
	}

	if [ info exists fn ] {
	    raccess_log D "play: filename is: $fn"
	    set that [ vbox_voice p $fn ]
	} else {
	    raccess_error "play: file not found, bad or disabled say."
	    set that "OK"
	}
	if { [ string compare $that "OK" ] != 0 } {
		return "$that"
	}
    }
    return "$that"
}

###############################################################################
# raccess

proc raccess_quit { } {
    return "EXIT"
}

proc raccess_first { } {
    global raccessvar

    set raccessvar(msg_ano) 0
    return "CONT"
}

proc raccess_last { } {
    global raccessvar

    set raccessvar(msg_ano) [ expr $raccessvar(msg_no) -1 ]
    return "CONT"
}

proc raccess_rewind { } {
    global tuning raccessvar
    global $tuning(actualmode)_events

    set raccessvar(msg_ano) [ expr $raccessvar(msg_ano) -1 ]
    if { $raccessvar(msg_ano) < 0 } {
	if { [ info exists $tuning(actualmode)_events(first_msg) ] } {
	    set rthat [ raccess_run "play [ subst $$tuning(actualmode)_events(first_msg) ]" ]
	    switch $rthat {
		EXIT {
		    break
		}
		HANGUP {
		    unset raccessvar
		    return "HANGUP"
		}
	    }
	}
	set raccessvar(msg_ano) 0
    }
    return "CONT"
}

proc raccess_forwind { } {
    global tuning raccessvar
    global $tuning(actualmode)_events

    incr raccessvar(msg_ano)
    if { $raccessvar(msg_ano) >= $raccessvar(msg_no) } {
	if { [ info exists $tuning(actualmode)_events(last_msg) ] } {
	    set rthat [ raccess_run "play [ subst $$tuning(actualmode)_events(last_msg) ]" ]
	    switch $rthat {
		EXIT {
		    break
		}
		HANGUP {
		    unset raccessvar
		    return "HANGUP"
		}
	    }
	}
	set raccessvar(msg_ano) [ expr $raccessvar(msg_no) -1 ]
    }
    return "CONT"
}

proc raccess_repeat { } {
    return "CONT"
}

proc raccess_delete { } {
    global tuning raccessvar
    global $tuning(actualmode)_events
    
    # delete file
    set msg [ lindex $raccessvar(msg_files) $raccessvar(msg_ano) ]
    file delete -force ${msg}.ulaw
    file delete -force ${msg}.vbox

    # delete from filelist
    set raccessvar(msg_files) [ lreplace $raccessvar(msg_files) \
				    $raccessvar(msg_ano) $raccessvar(msg_ano) ]
    incr raccessvar(msg_no) -1
    
    # intro
    if { [ info exists $tuning(actualmode)_events(no_of_msgs) ] } {
	set no $raccessvar(msg_no)
	set what [ subst $$tuning(actualmode)_events(no_of_msgs) ]
	raccess_log D "raccess: no_of_msgs: \"$what\""
	raccess_log D "raccess: no_of_msgs: $raccessvar(msg_no)"
	raccess_log D "raccess: list length: [ llength $raccessvar(msg_files) ]"
	
	set rthat [ raccess_run "play [ subst $what ]" ]
	switch $rthat {
	    EXIT {
		return "EXIT"
	    }
	    HANGUP {
		return "HANGUP"
	    }
	}
    }
    # still msg there?
    if { $raccessvar(msg_no) == 0 } {
	if { [ info exists $tuning(actualmode)_events(no_msgs) ] } {
	    set what [ subst $$tuning(actualmode)_events(no_msgs) ]
	    
	    set rthat [ raccess_run "play [ subst $what ]" ]
	    switch $rthat {
		EXIT {
		    break
		}
		HANGUP {
		    return "HANGUP"
		}
	    }
	}
	raccess_log D "raccess: delete - 0 msgs - EXIT"
	return "EXIT"
    }
    
    # rewind
    return [ raccess_rewind ]
}

proc raccess { dir } {
    global tuning vbxv_userhome
    global $tuning(actualmode)_events
    global raccessvar

    if { [ info exists raccessvar ] } {
	raccess_error "raccess: onother raccess is running... exiting"
	return "REPEAT"
    }
    
    if { [ string compare $dir "start" ] == 0 } {
	set dir "$vbxv_userhome/new"
    }
    raccess_log D "raccess to $dir"
    
    # read directory
    set raccessvar(msg_files) ""
    foreach f [ split [ exec ls -1tr $dir ] \n ] {
	if { [ string compare ".vbox" \
		[ string range $f [ expr [ string length $f ] -5 ] end ] ] \
		== 0 } {
	    continue
	}
	if { [ string compare [ string index $f 0 ] "/" ] != 0 } {
	    set f $vbxv_userhome/new/$f
	}
	lappend raccessvar(msg_files) [ string range $f 0 [ expr [ string length $f ] -6 ] ]
    }
    set raccessvar(msg_no) [ llength $raccessvar(msg_files) ]
    set no $raccessvar(msg_no)
    raccess_log D "raccess: msgs: $raccessvar(msg_files)\nno: $no"
    
    # intro
    if { [ info exists $tuning(actualmode)_events(no_of_msgs) ] } {
	set what [ subst $$tuning(actualmode)_events(no_of_msgs) ]
	raccess_log D "raccess: no_of_msgs: \"$what\""
	
	set rthat [ raccess_run "play [ subst $what ]" ]
	switch $rthat {
	    EXIT {
		unset raccessvar
		return "REPEAT"
	    }
	    HANGUP {
		unset raccessvar
		return "HANGUP"
	    }
	}
    }
    if { $raccessvar(msg_no) == 0 } {
	if { [ info exists $tuning(actualmode)_events(no_msgs) ] } {
	    set what [ subst $$tuning(actualmode)_events(no_msgs) ]
	    
	    set rthat [ raccess_run "play [ subst $what ]" ]
	    switch $rthat {
		EXIT {
		    unset raccessvar
		    return "REPEAT"
		}
		HANGUP {
		    unset raccessvar
		    return "HANGUP"
		}
	    }
	}
	return "EXIT"
    }
    
    # walk on msg_files
    set raccessvar(msg_ano) 0
    set working 1
    while { $working } {
	set msg [ lindex $raccessvar(msg_files) $raccessvar(msg_ano) ]
	# play header
	if { [ info exists $tuning(actualmode)_events(msg_header) ] } {
	    # read header
	    if [ catch { set head [ open ${msg}.vbox r ] } that ] {
		raccess_error "raccess: open ${msg}.vbox: $that"
	    } else {
		gets $head name
		set name [ string range $name 6 end ]
		# _unknown_ hack
		if { [ string compare $name "*** Unknown ***" ] == 0 } {
		    set name ""
		}
		
		gets $head ids
		set ids [ string range $ids 6 end ]
		set id [ string index $ids 0 ]
		for { set i 1 } { $i < [ string length $ids ] } { incr i } {
		    set id "$id & [ string index $ids $i ]"
		}
		gets $head time
		set time [ string range $time 6 end ]
		
		set what [ subst $$tuning(actualmode)_events(msg_header) ]
		set what [ clock format $time -format $what ]
		# play header
		set rthat [ raccess_run "play [ subst $what ]" ]
		switch $rthat {
		    EXIT {
			break
		    }
		    CONT {
			continue
		    }
		    HANGUP {
			unset raccessvar
			return "HANGUP"
		    }
		    default {
		    }
		}
	    }
	}
	
	# play message
	set rthat [ raccess_run "play @${msg}.ulaw" ]
	switch $rthat {
	    EXIT {
		break
	    }
	    CONT {
		continue
	    }
	    HANGUP {
		unset raccessvar
		return "HANGUP"
	    }
	}
	incr raccessvar(msg_ano)
	if { $raccessvar(msg_ano) >= $raccessvar(msg_no) } {
	    if { [ info exists $tuning(actualmode)_events(last_msg) ] } {
		set rthat [ raccess_run "play [ subst $$tuning(actualmode)_events(last_msg) ]" ]
		switch $rthat {
		    EXIT {
			break
		    }
		    HANGUP {
			unset raccessvar
			return "HANGUP"
		    }
		}
	    }
	    set raccessvar(msg_ano) [ expr $raccessvar(msg_no) -1 ]
	}
	if { $raccessvar(msg_ano) < 0 } {
	    if { [ info exists $tuning(actualmode)_events(first_msg) ] } {
		set rthat [ raccess_run "play [ subst $$tuning(actualmode)_events(first_msg) ]" ]
		switch $rthat {
		    EXIT {
			break
		    }
		    HANGUP {
			unset raccessvar
			return "HANGUP"
		    }
		}
	    }
	    set raccessvar(msg_ano) 0
	}
    }
    unset raccessvar
    return "EXIT"
}

###############################################################################
# raccess_run

proc raccess_run { what } {
    
    global tuning dtmf dtmf_eval events
    global $tuning(actualmode)_dtmf $tuning(actualmode)_events \
	    $tuning(actualmode)_dtmf_eval
    
    set cmd [ lindex $what 0 ]
    set carg [ lrange $what 1 end ]
    
    set rthat "REPEAT"
    while { [ string compare $rthat "REPEAT" ] == 0 } {
	set rthat "OK"
	switch $cmd {
	    run {
		proc raccess_run_restore { } {
		    uplevel {
			if { [ info exists bumode ] } {
			    set tuning(actualmode) $bumode
			} else {
			    set tuning(actualmode) ""
			}
			# init dtmf
			vbox_breaklist c
			foreach d [ array names dtmf ] {
			    vbox_breaklist a $d
			}
			foreach d [ array names $tuning(actualmode)_dtmf ] {
			    vbox_breaklist a $d
			}
			vbox_breaklist d
			#
			raccess_log D \
				"run: run_return: restoring tuning(actualmode) = $tuning(actualmode)"
		    }
		}
		
		raccess_log D "run: run: arg: $carg"
		
		set carg [ raccess_parseargs $carg ]
		foreach m "$carg" {
		    
		    raccess_log D "run: run: mode $m"
		    
		    raccess_readrc $m
		    set bumode $tuning(actualmode)
		    
		    set tuning(actualmode) $m
		    raccess_log D \
			    "run: run: setting tuning(actualmode) = $tuning(actualmode)"
		    global ${m}_run ${m}_dtmf ${m}_dtmf_eval ${m}_events
		    
		    # init dtmf
		    vbox_breaklist c
		    foreach d [ array names dtmf ] {
			vbox_breaklist a $d
		    }
		    foreach d [ array names ${m}_dtmf ] {
			vbox_breaklist a $d
		    }
		    vbox_breaklist d
		    #
		    
		    if { ! [ info exists ${m}_run ] } {
			raccess_error "run: run: no section \[run\] in $m defined."
			# skip this mode
			set rthat "OK"
			continue
		    }
		    raccess_log D "run: run: $m"
		    
		    # _do_ hack
		    set vdo 1
		    set vdostart 0
		    while { $vdo != 0 } {
			raccess_log D "run: do $vdo"
			for { set i $vdostart } \
				{ [ info exists ${m}_run($i) ] } { incr i } {
			    set that [ subst $${m}_run($i) ]
			    raccess_log D "run: run: $i: $that"
			    set rthat [ raccess_run $that ]
			    switch $rthat {
				EXIT {
				    raccess_run_restore
				    raccess_log D "run: run: got EXIT: exiting, return BEGIN"
				    return "BEGIN"
				}
				HANGUP {
				    raccess_run_restore
				    raccess_log D "run: run: got HANGUP"
				    return "HANGUP"
				}
				BEGIN {
				    raccess_log D "run: run: got BEGIN: incr vdo"

				    incr vdo 1
				    break
				}
			    }
			}
			incr vdo -1
		    }
		    if { [ info exists ${m}_events(quit) ] } {
			raccess_log D "run: run: ${m}_events(quit) exists. hanging up"
			return "HANGUP"
		    }
		}
		raccess_run_restore
		raccess_log D "run: run: done: exiting, return OK"
		return "OK"
	    }
	    
	    do {
		raccess_log D "run: do"
		upvar i pos vdo rvdo vdostart rvdostart
		set rvdo $carg
		set rvdostart [ expr $pos +1 ]
		raccess_log D \
			"run: do: $rvdo times, start at $rvdostart"
		set rthat "OK"
	    }
	    
	    default {
		raccess_log D "run: \"$cmd $carg\""
		if [ catch { set rwhat [ eval $cmd $carg ] } that ] {
		    raccess_error "run: default: error: eval \"$cmd $carg\": $that"
		    set rthat "OK"
		    break
		} else {
		    if { [ info exists $tuning(actualmode)_dtmf($rwhat) ] } {
			set rthat [ raccess_run \
				"play [ subst $$tuning(actualmode)_dtmf($rwhat)] " ]
		    } else {
			if { [ info exists dtmf($rwhat) ] } {
			    set rthat [ raccess_run \
				    "play $dtmf($rwhat)" ]
			}
		    }
		    
		    if { [ info exists $tuning(actualmode)_dtmf_eval($rwhat) ] } {
			raccess_log D [ subst \
				"run: $cmd $carg: action: \"$$tuning(actualmode)_dtmf_eval($rwhat)\"" ]
			
			if [ catch { set rthat [ raccess_run [ subst \
				$$tuning(actualmode)_dtmf_eval($rwhat) ] ] } that ] {
			    raccess_error \
				    "run: default: error in rc: \"$cmd $carg\": $rwhat: no cmd found: $that"
			    set rthat OK
			}
		    } elseif { [ info exists dtmf_eval($rwhat) ] } {
			raccess_log D "run: $cmd $carg: global action: $dtmf_eval($rwhat)" 		
			if [ catch { set rthat [ raccess_run \
				$dtmf_eval($rwhat) ] } that ] {
			    raccess_error \
				    "run: default: error in rc: \"$cmd $carg\": $rwhat: ???"
			    set rthat "OK"
			}
		    } else {
			switch $rwhat {
			    OK {
				raccess_log D "run: default: OK"
				set rthat "OK"
				break
			    }
			    REPEAT {
				raccess_log D "run: default: REPEAT"
				return "REPEAT"
			    }
			    SKIP {
				raccess_log D "run: default: SKIP"
				return "OK"
			    }
			    HANGUP {
				raccess_log D "run: default: HANGUP"
				return "HANGUP"
			    }
			    CONT {
				raccess_log D "run: default: CONT"
				return "CONT"
			    }
			    EXIT {
				raccess_log D "run: default: EXIT"
				return "EXIT"
			    }
			    BEGIN {
				raccess_log D "run: default: BEGIN"
				return "BEGIN"
			    }
			    default {
				raccess_error "run: default: error in rc: \"$cmd $carg\": $rwhat"
				set rthat "OK"
				break
			    }
			}
		    }
		}
	    }
	}
    }
    raccess_log D "run: leaving with return $rthat"
    return "$rthat"
}

###############################################################################
# main

# set defaults
set tuning(echo_path) /bin/echo
set tuning(say) 0
set tuning(say_path) /usr/local/bin/mbrola-say
set tuning(say_flag) ""
set tuning(say_cachedir) /opt/vbox/share/vbox/msg
set tuning(say_maxfilenamelen) 80
#

raccess_readrc

set ecode OK

if { ! [ info exists run ] } {
    raccess_error "no section \[run\] in raccess.conf defined."
    return "ERROR"
}

for { set batch 0 } { [ info exists run($batch) ] } { incr batch } {
    set that "REPEAT"
    while { [ string compare $that "REPEAT" ] == 0 } {
	set that [ raccess_run $run($batch) ]
	if { [ string compare $that "HANGUP" ] == 0 } {
	    raccess_log D "HANGUP"
	    set ecode "HANGUP"
	    break;
	}
    }
}

if { [ file isfile "$vbxv_saveulaw" ] && [ file isfile "$vbxv_savevbox" ] } {
    if { [ info exists tuning(callhandler) ] } {
        raccess_log I \
                "Running callhandler: $tuning(callhandler) \"$vbxv_saveulaw\" \"$vbxv_callername\" \"$vbxv_callerid\" \"$vbxv_username\" \"$vbxv_localphone\""
        if [ catch { exec -- $tuning(callhandler) \
                "$vbxv_saveulaw" "$vbxv_callername" "$vbxv_callerid" "$vbxv_username" "$vbxv_localphone"} that ] {
            raccess_log E "Error on running callhandler: $that"
        }
    }
}

return "$ecode"
