/*
 This Game is distributed under the GNU GENERAL PUBLIC LICENSE 
 version 2. See COPYING for details.                           
                                                               
 copyright (c) 1999 - 2001 Harry Storbacka                    */

#include <ClanLib/core.h>
#include <ClanLib/application.h>
#include <ClanLib/display.h>
#include <ClanLib/gl.h>
#include <iostream>
#include <cmath>

#include "config.h"
#include "tile_enums.h"
#include "track.h"
#include "map.h"
#include "race.h"
#include "car.h"
#include "game_data.h"
#include "season.h"
#include "sound_controller.h"
#include "game_config.h"
#include "car_config.h"
#include "attractor.h"
#include "camera.h"
#include "info_gfx.h"
#include "object_3d.h"
#include "debug.h"

#include "graphics.h"
#include "fonts.h"

#include "player.h"

Car_Player::Car_Player(float x, float y, char id, short _p_num )
: Car( x, y, id )
{
	RaceDebug::print("Car_Player::Car_Player(...)", 5 );
	
	p_num = _p_num;       // player 1 or 2
	
	start = true;
	out_of_map = false;
	draw_start_highlight = true;
	
	end_position = -1;
	
	RaceDebug::print("Car_Player::Car_Player(...) -- done", 5 );
}

Car_Player::~Car_Player()
{
	RaceDebug::print("Car_Player::~Car_Player(...) -- done", 5 );
}


void Car_Player::show( int center_x, int center_y )
{
	RaceDebug::print("Car_Player::show()",1); 

	if( start )
	{
		// play engine start when player is drawn the first time.
		
		if( Config::sfx_on )
			play_engine_start();
		
		start = false;
		start_highlight_time = CL_System::get_time();
	}
	
	if( draw_start_highlight )
	{
		draw_start_highlight = false;
	}
	
	glPushMatrix();
	
		glTranslatef( x, y, z );
		
		// legacy... Frame is a int (0->31) => 11.25 degrees/step
		// angle is the move direction, frame is the cars rotation :)
		glRotatef( Frame*11.25, 0.0f, 0.0f, 1.0f );
		
		// rotate car so it follows terrain		
		// cout << "rot_x: " << rot_x << "\t\t rot_y: " << rot_y << endl;
		glRotatef( rot_x, 1.0f, 0.0f, 0.0f );
		glRotatef( rot_y, 0.0f, 1.0f, 0.0f );
		
		Gfx::CarGfx->texture[id].bind();
		Gfx::CarGfx->car[id]->draw();
		
	glPopMatrix();

	RaceDebug::print("Car_Player::show() -- done",1); 
}



void Car_Player::reset_special()
{
	RaceDebug::print("Car_Player::reset_special()",1); 

	start = true;
	draw_start_highlight = true;
	end_position = -1;

	RaceDebug::print("Car_Player::reset_special() -- done",1); 
}


void Car_Player::update(float move_time )
{
	RaceDebug::print("Car_Player::update()",1); 
	time_elapsed = move_time;

	dx = move_time * speed*(cos( angle*(M_PI/16)));
	dy = move_time * speed*(sin( angle*(M_PI/16)));

//	cout << "(x, y): " << x << " " << y << endl;
	RaceDebug::print("Car_Player::update() -- done",1); 
}



void Car_Player::update_object_speed()
{
	RaceDebug::print("Car_Player::update_object_speed()", 1); 

	RaceDebug::print("Car_Player::update_speed(): gas", 1); 

	// Gas 
	if( speed < max_speed && !sliding )
	{
		if( GameData::p1_vert_axis->get_pos() < -0.1f )
		{
			accel = time_elapsed * (1.0f/(speed+2.0f)) * 15.0f;
			speed += accel;
		}
	}
	
	RaceDebug::print("Car::update_speed(): brakes", 1);

	// Breaks 
	if( speed > 0 )
	{
		if( GameData::p1_vert_axis->get_pos() > 0.1f )
			speed -= (Config::brakes_speed_decrease*time_elapsed);
		
		// no gas brake
		if( GameData::p1_vert_axis->get_pos() < 0.2f )
			speed -= (Config::no_gas_speed_decrease*time_elapsed);
		
		if( speed < 0 ) speed = 0;
	}

	max_speed = (Config::road_speed * Config::cc->road_speed[id]) * Config::difficulty_level;

	RaceDebug::print("Car::update_speed() -- done", 1); 
}



void Car_Player::update_wanted_dir()
{
	check_out_of_map();

	if( speed > 0.1 )
	{
		if( GameData::p1_hori_axis->get_pos() > 0.1 )
		{
			Frame++;
			wanted_angle++;
			if( Frame == 32 ) Frame = 0;
			if( wanted_angle == 32 ) wanted_angle = 0;
		}
		
		if( GameData::p1_hori_axis->get_pos() < -0.1 )
		{
			Frame--;
			wanted_angle--;
			if( Frame == -1 ) Frame = 31;
			if( wanted_angle == -1 ) wanted_angle = 31;
		}
	}
}


void Car_Player::play_engine_start()
{
	if( !Config::sfx_on ) return;
	
	// sound_controller.play_start();
}




void Car_Player::draw_speed_meter()
{

	InfoGfx::speed->put_screen( 10 + id*320 ,70 );
		
	CL_Display::draw_line(40 + get_p_num()*320,
						  100,
						  ((int)(40 + get_p_num()*320)
						  + (cos(get_speed()/2 + PI) *20 )),
						  ((int)(100 + ( sin(  get_speed()/2 + PI) *20 ))),
						  1.0f, 0.0f, 0.0f, 1.0f );

	// draw position counter... TODO move to its own function.

	if( in_goal == false )
	{
		if( Config::num_players == 1 )	
		{
			Fonts::position->print_right( 635, 390, CL_String(get_current_position()));
		}

		if( Config::num_players == 2 )	
		{
			Fonts::position->print_right( 320 * get_p_num() -10, 390, CL_String(get_current_position()));
		}
	}
	else
	{
		if( Config::num_players == 1 )	
		{
			Fonts::position->print_right( 635, 390, CL_String(get_current_position()));
		}

		if( Config::num_players == 2 )	
		{
			Fonts::position->print_right( 320 * get_p_num() -10, 390, CL_String(get_current_position()));
		}
	}
}


void Car_Player::update_extras()
{

}






void Car_Player::check_out_of_map()
{
	if( x < 0 || y < 0 || x > 40 || y > 40 )
	{
		if( out_of_map == false )
		{
			out_of_map = true;
			out_of_map_start_time = CL_System::get_time();
		}
	}
	else
  		out_of_map = false;

	if( out_of_map )
	{
		if( CL_System::get_time() - out_of_map_start_time > 3000 )
		{
			set_pos( Track::track->attractors[control_points_passed].x,
					Track::track->attractors[control_points_passed].y );
			out_of_map = false;
		}
	}
}











int Car_Player::get_current_position()
{
	if( in_goal == true && end_position != -1 )
	{
		return end_position; // Stop the position counter when player in goal.
	}

	int pos = 1;

	std::vector<Car*>::iterator it;
	
	for( it = GameData::cars.begin();
		 it != GameData::cars.end();
		 it++)
	{
		if( (*it)->id != id )
		{
			if( (*it)->get_lap() > get_lap() )
		  		pos++;
			else
			{
				if( (*it)->control_points_passed_total > control_points_passed_total )
			  		pos++;
				
				if(  (*it)->control_points_passed_total == control_points_passed_total )
				{
					if( (*it)->get_distance_to_next_attractor(
								Track::track->attractors[(*it)->control_points_passed].x,
								Track::track->attractors[(*it)->control_points_passed].y)
						< get_distance_to_next_attractor(
								Track::track->attractors[control_points_passed].x,
								Track::track->attractors[control_points_passed].y ))
					{
						pos++;
					}
				}
			}
		}
	}

	if( in_goal == true )
	{
		end_position = pos;
	}

	return pos;
}

