/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_trace.c,v 1.44.2.2 2005/01/15 23:53:30 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "modules.h"
#include "xmode.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Module MOD_HEADER(m_trace) = {
	"m_trace",
	"/TRACE command",
	6, "$Revision: 1.44.2.2 $"
};

int MOD_LOAD(m_trace)()
{
	if (register_command(&MOD_HEADER(m_trace), &CMD_TRACE, m_trace) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_trace)()
{
	return MOD_SUCCESS;
}

static int report_trace(aClient *sptr, aClient *acptr, int dow, int link_u_p, int link_s_p)
{
	char *name;
	long umode;
	int cnt = 0;

	ASSERT(acptr != NULL);
	ASSERT(acptr->localClient != NULL);

	umode = (IsClient(acptr) || IsUnknown(acptr)) ? UMODE_OPER : UMODE_RSTAFF;
	name = get_client_name(acptr, (HasMode(sptr, umode) ? FALSE : HIDE_IP));

	switch (acptr->status) {
		case STAT_CONNECTING:
			send_me_numeric(sptr, RPL_TRACECONNECTING, acptr->localClient->class->name, name);
			cnt++;
			break;
		case STAT_HANDSHAKE:
			send_me_numeric(sptr, RPL_TRACEHANDSHAKE, acptr->localClient->class->name, name);
			cnt++;
			break;
		case STAT_ME:
			break;
		case STAT_UNKNOWN:
			send_me_numeric(sptr, RPL_TRACEUNKNOWN, acptr->localClient->class->name, name,
				acptr->firsttime ? (timeofday - acptr->firsttime) : -1);
			cnt++;
			break;
		case STAT_CLIENT:
			if (((HasMode(sptr, UMODE_OPER) && MyClient(sptr))
			  || !(dow && HasMode(acptr, UMODE_INVISIBLE))) || !dow
			  || HasMode(acptr, UMODE_OPER)) {
				if (HasMode(acptr, UMODE_OPER)) {
					send_me_numeric(sptr, RPL_TRACEOPERATOR, acptr->localClient->class->name,
						name, timeofday - acptr->firsttime);
					cnt++;
				}
			}
			break;
		case STAT_SERVER:
			send_me_numeric(sptr, RPL_TRACESERVER, acptr->localClient->class->name, link_s_p, link_u_p,
				name, acptr->serv->by_nick,
				*acptr->serv->by_user != '\0' ? acptr->serv->by_user : "*",
				*acptr->serv->by_host != '\0' ? acptr->serv->by_host : "*",
				timeofday - acptr->firsttime);
			cnt++;
			break;
		default:
			send_me_numeric(sptr, RPL_TRACENEWTYPE, name);
			cnt++;
			break;
	}
	return cnt;
}

static int skip_client_trace(aClient *acptr, char *tname, int doall, int dow, int wilds)
{
	if (!doall && wilds && match(tname, acptr->name)) {
		return 1;
	}
	if (!dow && mycmp(tname, acptr->name)) {
		return 1;
	}
	return 0;
}

/*
 * m_trace
 *	parv[0] = sender prefix
 *	parv[1] = server name
 */
int m_trace(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	static time_t last_used = 0L;
	aClient *acptr = NULL;
	int doall, link_s[MAXCONNECTIONS + 1], link_u[MAXCONNECTIONS + 1];
	int cnt = 0, wilds = 0, dow = 0;
	char *tname = parc > 1 ? parv[1] : me.name;
	dlink_node *node;

	if (GeneralConfig.hide_super_servers) {
		if ((acptr = next_client_double(client, tname)) != NULL) {
			if (!HasMode(sptr, UMODE_OPER) && IsULine(acptr)) {
				send_me_numericNA(sptr, ERR_NOPRIVILEGES);
				return 0;
			}
			acptr = NULL;
		}
	}
	if (parc > 2) {
		if (use_or_deliver(cptr, sptr, &CMD_TRACE, "%s :%s", 2, parc, parv)) {
			return 0;
		}
	}
	switch (use_or_deliver(cptr, sptr, &CMD_TRACE, ":%s", 1, parc, parv)) {
		case HUNTED_PASS:
		{
			aClient *acptr2 = next_client_double(client, tname);
			send_me_numeric(sptr, RPL_TRACELINK, ircd_version,
				Internal.debug_mode, tname, (acptr2->from != NULL && !BadPtr(acptr2->from->name)) ?
				  acptr2->from->name : "acptr2_is_NULL!");
			return 0;
		}
		case HUNTED_ISME:
			break;
		default:
			return 0;
	}
	if (!HasMode(sptr, UMODE_OPER)) {
		if (parv[1] != NULL && (strchr(parv[1], '.') == NULL)
		  && (strchr(parv[1], '*') != NULL || strchr(parv[1], '?') != NULL)) {
			send_me_numeric(sptr, RPL_ENDOFTRACE, parv[1]);
			return 0;
		}
		if (FloodConfig.pace_wait_intense && (last_used + FloodConfig.pace_wait_intense > timeofday)) {
			send_me_numericNA(sptr, RPL_LOAD2HI);
			return 0;
		}
		else {
			last_used = timeofday;
		}
		if (GeneralConfig.spy_notices && IsPerson(sptr)) {
			sendto_realops_lev(SPY_LEV, "TRACE requested by %s (%s@%s) [%s]", sptr->name,
				sptr->username, MaskedHost(sptr), sptr->user->server);
		}
	}

	doall = (parv[1] && parc > 1) ? !match(tname, me.name) : TRUE;
	wilds = (parv[1] == NULL || strchr(tname, '*') || strchr(tname, '?'));
	dow = (wilds || doall);

	if (!HasMode(sptr, UMODE_OPER) || !dow) {
		char *name;
		int numeric;

		if ((acptr = hash_find_client(tname, NULL)) == NULL || !IsPerson(acptr)) {
			send_me_numeric(sptr, RPL_ENDOFTRACE, tname);
			return 0;
		}

		ASSERT(acptr->localClient != NULL);

		name = get_client_name(acptr, HasMode(sptr, UMODE_RSTAFF) ? FALSE : HIDE_IP);
		numeric = HasMode(acptr, UMODE_OPER) ? RPL_TRACEOPERATOR : RPL_TRACEUSER;

		send_me_numeric(sptr, numeric, acptr->localClient->class->name, name, timeofday - acptr->lasttime);
		send_me_numeric(sptr, RPL_ENDOFTRACE, tname);
		return 0;
	}

	memset((char *)link_s, '\0', sizeof(link_s));
	memset((char *)link_u, '\0', sizeof(link_u));

	if (doall) {
		for (acptr = client; acptr != NULL; acptr = acptr->next) {
			if (IsPerson(acptr) && (!HasMode(acptr, UMODE_INVISIBLE) || HasMode(sptr, UMODE_OPER))) {
				link_u[acptr->from->localClient->fd]++;
			}
			else if (IsServer(acptr) && (!GeneralConfig.hide_super_servers
			  || (GeneralConfig.hide_super_servers && IsULine(acptr) && !HasMode(sptr, UMODE_OPER)))) {
				link_s[acptr->from->localClient->fd]++;
			}
		}
	}

	DLINK_FOREACH_DATA(lclient_list.head, node, acptr, aClient) {
		if (HasMode(acptr, UMODE_INVISIBLE) && dow && !(MyConnect(sptr) &&
		  HasMode(sptr, UMODE_OPER)) && !HasMode(acptr, UMODE_OPER) && acptr != sptr) {
			continue;
		}
		if (!skip_client_trace(acptr, tname, doall, dow, wilds)) {
			cnt += report_trace(sptr, acptr, dow, 0, 0);
		}
	}
	DLINK_FOREACH_DATA(lserver_list.head, node, acptr, aClient) {
		if (GeneralConfig.hide_super_servers && IsULine(acptr) && !HasMode(sptr, UMODE_OPER)) {
			continue;
		}
		if (!skip_client_trace(acptr, tname, doall, dow, wilds)) {
			cnt += report_trace(sptr, acptr, dow, link_u[acptr->from->localClient->fd],
				link_s[acptr->from->localClient->fd]);
		}
	}
	DLINK_FOREACH_DATA(lunknown_list.head, node, acptr, aClient) {
		if (!skip_client_trace(acptr, tname, doall, dow, wilds)) {
			cnt += report_trace(sptr, acptr, dow, 0, 0);
		}
	}

	if (!OPHasFlag(sptr, OFLAG_WALLOPS) || !cnt) {
		if (cnt) {
			send_me_numeric(sptr, RPL_ENDOFTRACE, tname);
			return 0;
		}

		send_me_numeric(sptr, RPL_TRACESERVER, 0, link_s[me.localClient->fd],
			link_u[me.localClient->fd], me.name, "*", "*", me.name,
			(acptr != NULL) ? timeofday - acptr->firsttime : 0);
		send_me_numeric(sptr, RPL_ENDOFTRACE, tname);
		return 0;
	}
	if (((GeneralConfig.hide_super_servers && HasMode(sptr, UMODE_OPER))
	  || !GeneralConfig.hide_super_servers) && doall) {
		ConfigItem_class *class;

		DLINK_FOREACH_DATA(conf_class_list.head, node, class, ConfigItem_class) {
			if (!class->clients) {
				continue;
			}
			send_me_numeric(sptr, RPL_TRACECLASS, class->name, class->clients);
		}
	}
	send_me_numeric(sptr, RPL_ENDOFTRACE, tname);
	return 0;
}
