/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: module_objects.c,v 1.11.2.1 2004/12/07 03:05:14 pneumatus Exp $
 */

#include "struct.h"
#include "setup.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "h.h"
#include "msg.h"
#include "channel.h"
#include "send.h"
#include "memory.h"
#include "patchlevel.h"
#include "hook.h"
#include "dlink.h"
#include "modules.h"
#include "xmode.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>

Command *register_command(Module *owner, msg_ptr *msg, int (*func)())
{
	Command *cmd;

	ASSERT(owner != NULL);
	ASSERT(msg != NULL);

	if ((cmd = add_command(msg, func)) == NULL) {
		sendto_realops("Couldn't register command %s in module %s", msg->msg_str,
			owner->name);
	}
	else {
		cmd->owner = owner;
		dlink_add(&owner->commands, cmd);
	}
	
	return cmd;
}

Event *register_event(Module *owner, char *name, void (*func)(), void *data, time_t when, int loop)
{
	Event *event;

	ASSERT(owner != NULL);
	ASSERT(!BadPtr(name));

	if ((event = add_event(name, func, data, when, loop)) == NULL) {
		sendto_realops("Couldn't register event %s in module %s", name, owner->name);
	}
	else {
		event->owner = owner;
		dlink_add(&owner->events, event);
	}
	return event;
}

HookEvent *register_hook_event(Module *owner, Hook *hook, int (*event)())
{
	HookEvent *hookevent;

	ASSERT(owner != NULL);
	ASSERT(hook != NULL);

	if ((hookevent = hook_add_event(hook, event)) == NULL) {
		sendto_realops("Couldn't register event for hook %s in module %s", hook->name,
			owner->name);
	}
	else {
		hookevent->owner = owner;
		dlink_add(&owner->hookevents, hookevent);
	}
	return hookevent;
}

Hook *register_hook(Module *owner, char *name)
{
	Hook *hook;

	ASSERT(owner != NULL);
	ASSERT(!BadPtr(name));

	if ((hook = hook_add(name)) == NULL) {
		sendto_realops("Couldn't register hook %s in module %s", name, owner->name);
	}
	else {
		hook->owner = owner;
		dlink_add(&owner->hooks, hook);
	}
	return hook;
};

xMode *register_user_mode(Module *owner, char flag, long *mode, unsigned int options)
{
	xMode *t;

	ASSERT(owner != NULL);

	if ((t = add_user_mode(flag, mode, options)) == NULL) {
		return NULL;
	}
	else {
		t->owner = owner;
		dlink_add(&owner->usermodes, t);
	}
	return t;
}

static void unregister_command_node(Module *owner, Command *cmd, dlink_node *node)
{
	ASSERT(owner != NULL);
	ASSERT(cmd != NULL);

	dlink_del(&owner->commands, cmd, node);
	if (!del_command_cmd(cmd)) {
		sendto_realops("Couldn't unregister command %s from module %s",
			cmd->msg->msg_str, owner->name);
	}
}

static void unregister_event_node(Module *owner, Event *event, dlink_node *node)
{
	ASSERT(owner != NULL);
	ASSERT(event != NULL);
	dlink_del(&owner->events, event, node);
	del_event(event, NULL);
}

static void unregister_hook_event_node(Module *owner, HookEvent *hookevent, dlink_node *node)
{
	ASSERT(owner != NULL);
	ASSERT(hookevent != NULL);
	dlink_del(&owner->hookevents, hookevent, node);
	hook_del_event(hookevent);
}

static void unregister_hook_node(Module *owner, Hook *hook, dlink_node *node)
{
	ASSERT(owner != NULL);
	ASSERT(hook != NULL);
	dlink_del(&owner->hooks, hook, node);
	hook_del(hook);
}

static void unregister_user_mode_node(Module *owner, xMode *t, dlink_node *node)
{
	ASSERT(owner != NULL);
	ASSERT(t != NULL);
	dlink_del(&owner->usermodes, t, node);
	del_user_mode(t->flag, &t->mode);
}

void clear_objects(Module *owner)
{
	dlink_node *node, *next = NULL;

	ASSERT(owner != NULL);

	DLINK_FOREACH_SAFE(owner->commands.head, node, next) {
		unregister_command_node(owner, node->data, node);
	}
	DLINK_FOREACH_SAFE(owner->events.head, node, next) {
		unregister_event_node(owner, node->data, node);
	}
	DLINK_FOREACH_SAFE(owner->hookevents.head, node, next) {
		unregister_hook_event_node(owner, node->data, node);
	}
	DLINK_FOREACH_SAFE(owner->hooks.head, node, next) {
		unregister_hook_node(owner, node->data, node);
	}
	DLINK_FOREACH_SAFE(owner->usermodes.head, node, next) {
		unregister_user_mode_node(owner, node->data, node);
	}
}
