/*
 *  Copyright 2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */


#include "ragel.h"
#include "xmlcodegen.h"
#include "parsetree.h"
#include "fsmgraph.h"

using namespace std;

XMLCodeGen::XMLCodeGen( char *fsmName, ParseData *pd, FsmAp *fsm, 
		std::ostream &out )
:
	fsmName(fsmName),
	pd(pd),
	fsm(fsm),
	out(out),
	nextActionTableId(0)
{
}


void XMLCodeGen::writeActionList()
{
	/* Determine which actions to write. */
	int nextActionId = 0;
	for ( ActionList::Iter act = pd->actionList; act.lte(); act++ ) {
		if ( ! act->isLmAction || act->numRefs() > 0 )
			act->actionId = nextActionId++;
	}

	/* Write the list. */
	out << "    <action_list length=\"" << nextActionId << "\">\n";
	for ( ActionList::Iter act = pd->actionList; act.lte(); act++ ) {
		if ( act->actionId >= 0 )
			writeAction( act );
	}
	out << "    </action_list>\n";
}

void XMLCodeGen::writeActionTableList()
{
	/* Must first order the action tables based on their id. */
	int numTables = nextActionTableId;
	RedActionTable **tables = new RedActionTable*[numTables];
	for ( ActionTableMap::Iter at = actionTableMap; at.lte(); at++ )
		tables[at->id] = at;

	out << "    <action_table_list length=\"" << numTables << "\">\n";
	for ( int t = 0; t < numTables; t++ ) {
		out << "      <action_table length=\"" << tables[t]->key.length() << "\">";
		for ( ActionTable::Iter atel = tables[t]->key; atel.lte(); atel++ ) {
			out << atel->value->actionId;
			if ( ! atel.last() )
				out << " ";
		}
		out << "</action_table>\n";
	}
	out << "    </action_table_list>\n";

	delete[] tables;
}

void XMLCodeGen::reduceActionTables()
{
	/* Reduce the actions tables to a set. */
	for ( StateList::Iter st = fsm->stateList; st.lte(); st++ ) {
		RedActionTable *actionTable = 0;

		/* Reduce To State Actions. */
		if ( st->toStateActionTable.length() > 0 ) {
			if ( actionTableMap.insert( st->toStateActionTable, &actionTable ) )
				actionTable->id = nextActionTableId++;
		}

		/* Reduce From State Actions. */
		if ( st->fromStateActionTable.length() > 0 ) {
			if ( actionTableMap.insert( st->fromStateActionTable, &actionTable ) )
				actionTable->id = nextActionTableId++;
		}

		/* Reduce EOF actions. */
		if ( st->eofActionTable.length() > 0 ) {
			if ( actionTableMap.insert( st->eofActionTable, &actionTable ) )
				actionTable->id = nextActionTableId++;
		}

		/* Loop the transitions and reduce their actions. */
		for ( OutIter out(st); out.lte(); out++ ) {
			if ( out.trans->actionTable.length() > 0 ) {
				if ( actionTableMap.insert( out.trans->actionTable, &actionTable ) )
					actionTable->id = nextActionTableId++;
			}
		}
	}
}

void XMLCodeGen::appendTrans( TransListVect &outList, long lowKey, 
		long highKey, TransAp *trans )
{
	if ( trans->toState != 0 || trans->actionTable.length() > 0 )
		outList.append( TransEl( lowKey, highKey, trans ) );
}

void XMLCodeGen::writeTrans( long lowKey, long highKey, TransAp *trans )
{
	/* First reduce the action. */
	RedActionTable *actionTable = 0;
	if ( trans->actionTable.length() > 0 )
		actionTable = actionTableMap.find( trans->actionTable );

	/* Write the transition. */
	out << "        <t>" << lowKey << " " << highKey;
	if ( trans->toState != 0 )
		out << " " << trans->toState->alg.stateNum;
	else
		out << " x";

	if ( actionTable != 0 )
		out << " " << actionTable->id;
	else
		out << " x";
	out << "</t>\n";
}

void XMLCodeGen::writeTransList( StateAp *state )
{
	KeyOps *keyOps = &pd->keyOps;
	TransAp *defTrans = state->outDefault;
	TransListVect outList;

	/* If there is only are no ranges the task is simple. */
	if ( state->outList.length() == 0 ) {
		/* If there is a default, add a full upper .. lower range. */
		if ( defTrans != 0 ) {
			/* Add the range on the lower and upper bound. */
			appendTrans( outList, keyOps->lowKey, keyOps->highKey, defTrans );
		}
	}
	else {
		/* Check for a gap at the beginning. */
		NextRedTrans tel( state->outList.head );
		if ( defTrans != 0 && keyOps->lt( keyOps->lowKey, tel.lowKey ) ) {
			/* Make the high key and append. */
			long highKey = tel.lowKey;
			keyOps->dec( highKey );

			appendTrans( outList, keyOps->lowKey, highKey, defTrans );
		}

		/* Write the transition. */
		appendTrans( outList, tel.lowKey, tel.highKey, tel.trans );

		/* Keep the last high end. */
		long lastHigh = tel.highKey;

		/* Loop each source range. */
		for ( tel.increment(); tel.trans != 0; tel.increment() ) {
			if ( defTrans != 0 ) {
				/* Make the next key following the last range. */
				long nextKey = lastHigh;
				keyOps->inc( nextKey );

				/* Check for a gap from last up to here. */
				if ( keyOps->lt( nextKey, tel.lowKey ) ) {
					/* Make the high end of the range that fills the gap. */
					long highKey = tel.lowKey;
					keyOps->dec( highKey );

					/* Append the new range. */
					appendTrans( outList, nextKey, highKey, defTrans );
				}
			}

			/* Reduce the transition. If it reduced to anything then add it. */
			appendTrans( outList, tel.lowKey, tel.highKey, tel.trans );

			/* Keep the last high end. */
			lastHigh = tel.highKey;
		}

		/* Now check for a gap on the end to fill. */
		if ( defTrans != 0 && keyOps->lt( lastHigh, keyOps->highKey ) ) {
			/* Get a copy of the default. */
			keyOps->inc( lastHigh );
			appendTrans( outList, lastHigh, keyOps->highKey, defTrans );
		}
	}

	out << "      <trans_list length=\"" << outList.length() << "\">\n";
	for ( TransListVect::Iter tvi = outList; tvi.lte(); tvi++ )
		writeTrans( tvi->lowKey, tvi->highKey, tvi->value );
	out << "      </trans_list>\n";
}

void XMLCodeGen::writeLmSwitch( InlineItem *item )
{
	LongestMatch *longestMatch = item->longestMatch;

	out << "<lm_switch";
	if ( longestMatch->lmSwitchHandlesError )
		out << " handles_error=\"t\"";
	out << ">\n";
	
	for ( LmPartList::Iter lmi = *longestMatch->longestMatchList; lmi.lte(); lmi++ ) {
		if ( lmi->inLmSelect && lmi->action != 0 ) {
			/* Open the action. Write it with the context that sets up _p 
			 * when doing control flow changes from inside the machine. */
			out << "      <sub_action id=\"" << lmi->longestMatchId << "\">";
			writeInlineList( lmi->action->inlineList, item );
			out << "</sub_action>\n";
		}
	}

	out << "    </lm_switch><exec><lm_get_tok_end></lm_get_tok_end></exec>";
}

void XMLCodeGen::writeText( InlineItem *item )
{
	if ( item->prev == 0 || item->prev->type != InlineItem::Text )
		out << "<text>";
	xmlEscapeHost( out, item->data );
	if ( item->next == 0 || item->next->type != InlineItem::Text )
		out << "</text>";
}

void XMLCodeGen::writeWithContext( InlineItem *item, InlineItem *context )
{
	if ( context != 0 ) {
		out << "<sub_action>";

		switch ( context->type ) {
		case InlineItem::LmOnNext:
			out << "<hold></hold>";
			break;
		case InlineItem::LmOnLagBehind:
			out << "<exec><lm_get_tok_end></lm_get_tok_end></exec>";
			break;
		case InlineItem::LmSwitch:
			out << "<exec><lm_get_tok_end></lm_get_tok_end></exec>";
			break;
		default: break;
		}
	}

	switch ( item->type ) {
	case InlineItem::Goto:
		writeGoto( item, context );
		break;
	case InlineItem::GotoExpr:
		writeGotoExpr( item, context );
		break;
	case InlineItem::Call:
		writeCall( item, context );
		break;
	case InlineItem::CallExpr:
		writeCallExpr( item, context );
		break;
	case InlineItem::Next:
		writeNext( item, context );
		break;
	case InlineItem::NextExpr:
		writeNextExpr( item, context );
		break;
	case InlineItem::Break:
		out << "<break></break>";
		break;
	default: break;
	}

	if ( context != 0 )
		out << "</sub_action>";
}

void XMLCodeGen::writeGoto( InlineItem *item, InlineItem *context )
{
	if ( pd->generatingSectionSubset )
		out << "<goto>-1</goto>";
	else {
		EntryMapEl *targ = fsm->entryPoints.find( item->nameTarg->id );
		out << "<goto>" << targ->value->alg.stateNum << "</goto>";
	}
}

void XMLCodeGen::writeCall( InlineItem *item, InlineItem *context )
{
	if ( pd->generatingSectionSubset )
		out << "<call>-1</call>";
	else {
		EntryMapEl *targ = fsm->entryPoints.find( item->nameTarg->id );
		out << "<call>" << targ->value->alg.stateNum << "</call>";
	}
}

void XMLCodeGen::writeNext( InlineItem *item, InlineItem *context )
{
	if ( pd->generatingSectionSubset )
		out << "<next>-1</next>";
	else {
		EntryMapEl *targ = fsm->entryPoints.find( item->nameTarg->id );
		out << "<next>" << targ->value->alg.stateNum << "</next>";
	}
}

void XMLCodeGen::writeGotoExpr( InlineItem *item, InlineItem *context )
{
	out << "<goto_expr>";
	writeInlineList( item->children, 0 );
	out << "</goto_expr>";
}

void XMLCodeGen::writeCallExpr( InlineItem *item, InlineItem *context )
{
	out << "<call_expr>";
	writeInlineList( item->children, 0 );
	out << "</call_expr>";
}

void XMLCodeGen::writeNextExpr( InlineItem *item, InlineItem *context )
{
	out << "<next_expr>";
	writeInlineList( item->children, 0 );
	out << "</next_expr>";
}

void XMLCodeGen::writeEntry( InlineItem * item )
{
	if ( pd->generatingSectionSubset )
		out << "<entry>-1</entry>";
	else {
		EntryMapEl *targ = fsm->entryPoints.find( item->nameTarg->id );
		out << "<entry>" << targ->value->alg.stateNum << "</entry>";
	}
}

void XMLCodeGen::writeActionExec( InlineItem *item )
{
	out << "<exec>";
	writeInlineList( item->children, 0 );
	out << "</exec>";
}

void XMLCodeGen::writeLmOnLast( InlineItem *item )
{
	out << "<lm_set_tok_end>1</lm_set_tok_end>";
	out << "<sub_action>";
	if ( item->longestMatchPart->action != 0 )
		writeInlineList( item->longestMatchPart->action->inlineList, 0 );
	out << "</sub_action>";
}

void XMLCodeGen::writeLmOnNext( InlineItem *item )
{
	out << "<lm_set_tok_end>0</lm_set_tok_end>";
	out << "<sub_action>";
	if ( item->longestMatchPart->action != 0 )
		writeInlineList( item->longestMatchPart->action->inlineList, item );
	out << "</sub_action>";
	out << "<hold></hold>";
}

void XMLCodeGen::writeLmOnLagBehind( InlineItem *item )
{
	out << "<sub_action>";
	if ( item->longestMatchPart->action != 0 )
		writeInlineList( item->longestMatchPart->action->inlineList, item );
	out << "</sub_action>";
	out << "<exec><lm_get_tok_end></lm_get_tok_end></exec>";
}


void XMLCodeGen::writeInlineList( InlineList *inlineList, InlineItem *context )
{
	for ( InlineList::Iter item = *inlineList; item.lte(); item++ ) {
		switch ( item->type ) {
		case InlineItem::Text:
			writeText( item );
			break;
		case InlineItem::Goto: case InlineItem::GotoExpr:
		case InlineItem::Call: case InlineItem::CallExpr:
		case InlineItem::Next: case InlineItem::NextExpr:
		case InlineItem::Break:
			writeWithContext( item, context );
			break;
		case InlineItem::Ret: 
			out << "<ret></ret>";
			break;
		case InlineItem::PChar:
			out << "<pchar></pchar>";
			break;
		case InlineItem::Char: 
			out << "<char></char>";
			break;
		case InlineItem::Hold:
			out << "<hold></hold>";
			break;
		case InlineItem::Curs: 
			out << "<curs></curs>";
			break;
		case InlineItem::Targs: 
			out << "<targs></targs>";
			break;
		case InlineItem::Entry:
			writeEntry( item );
			break;
		case InlineItem::Exec:
			writeActionExec( item );
			break;
		case InlineItem::LmSwitch: 
			writeLmSwitch( item );
			break;
		case InlineItem::LmSetActId:
			out << "<lm_set_act_id>" << 
					item->longestMatchPart->longestMatchId << 
					"</lm_set_act_id>";
			break;
		case InlineItem::LmSetTokEnd:
			out << "<lm_set_tok_end>1</lm_set_tok_end>";
			break;
		case InlineItem::LmOnLast:
			writeLmOnLast( item );
			break;
		case InlineItem::LmOnNext:
			writeLmOnNext( item );
			break;
		case InlineItem::LmOnLagBehind:
			writeLmOnLagBehind( item );
			break;
		case InlineItem::LmInitAct:
			out << "<lm_init_act></lm_init_act>";
			break;
		case InlineItem::LmInitTokStart:
			out << "<lm_init_tok_start></lm_init_tok_start>";
			break;
		case InlineItem::LmSetTokStart:
			out << "<lm_set_tok_start></lm_set_tok_start>";
			break;
		}
	}
}

void XMLCodeGen::writeAction( Action *action )
{
	out << "    <action";
	if ( action->name != 0 ) 
		out << " name=\"" << action->name << "\"";
	out << " line=\"" << action->loc.line << "\" col=\"" << action->loc.col << "\">";
	writeInlineList( action->inlineList, 0 );
	out << "</action>\n";
}

void xmlEscapeHost( std::ostream &out, char *data )
{
	while ( *data != 0 ) {
		switch ( *data ) {
		case '<': out << "&lt;"; break;
		case '>': out << "&gt;"; break;
		case '&': out << "&amp;"; break;
		default: out << *data; break;
		}
		data += 1;
	}
}

void XMLCodeGen::writeStateActions( StateAp *state )
{
	RedActionTable *toStateActions = 0;
	if ( state->toStateActionTable.length() > 0 )
		toStateActions = actionTableMap.find( state->toStateActionTable );

	RedActionTable *fromStateActions = 0;
	if ( state->fromStateActionTable.length() > 0 )
		fromStateActions = actionTableMap.find( state->fromStateActionTable );

	RedActionTable *eofActions = 0;
	if ( state->eofActionTable.length() > 0 )
		eofActions = actionTableMap.find( state->eofActionTable );
	
	if ( toStateActions != 0 || fromStateActions != 0 || eofActions != 0 ) {
		out << "      <state_actions>";
		if ( toStateActions != 0 )
			out << toStateActions->id;
		else
			out << "x";

		if ( fromStateActions != 0 )
			out << " " << fromStateActions->id;
		else
			out << " x";

		if ( eofActions != 0 )
			out << " " << eofActions->id;
		else
			out << " x"; out << "</state_actions>\n";
	}
}

void XMLCodeGen::writeStateList()
{
	/* Write the list of states. */
	out << "    <state_list length=\"" << fsm->stateList.length() << "\">\n";
	for ( StateList::Iter st = fsm->stateList; st.lte(); st++ ) {
		out << "      <state id=\"" << st->alg.stateNum << "\"";
		if ( st->isFinState() )
			out << " final=\"t\"";
		out << ">\n";

		writeStateActions( st );

		writeTransList( st );

		out << "      </state>\n";

		if ( !st.last() )
			out << "\n";
	}
	out << "    </state_list>\n";
}

void XMLCodeGen::writeEntryPoints()
{
	/* List of entry points other than start state. */
	if ( fsm->entryPoints.length() > 0 || pd->lmRequiresErrorState ) {
		out << "    <entry_points";
		if ( pd->lmRequiresErrorState )
			out << " error=\"t\"";
		out << ">\n";
		for ( EntryMap::Iter en = fsm->entryPoints; en.lte(); en++ ) {
			/* Get the name instantiation from nameIndex. */
			NameInst *nameInst = pd->nameIndex[en->key];
			StateAp *state = en->value;
			out << "      <entry name=\"" << nameInst->name << "\">" << 
					state->alg.stateNum << "</entry>\n";
		}
		out << "    </entry_points>\n";
	}
}

void XMLCodeGen::writeMachine()
{
	fsm->setStateNumbers();

	/* Open the machine. */
	out << "  <machine>\n"; 
	
	/* Action tables. */
	reduceActionTables();

	writeActionList();
	writeActionTableList();

	/* Start state. */
	out << "    <start_state>" << fsm->startState->alg.stateNum << 
			"</start_state>\n";

	writeEntryPoints();
	writeStateList();

	out << "  </machine>\n";
}

void XMLCodeGen::writeAlphType()
{
	out << "  <alphtype>";
	switch ( pd->alphType ) {
		case AT_Char:
			out << "signed_char";
			break;
		case AT_UnsignedChar:
			out << "unsigned_char";
			break;
		case AT_Short:
			out << "signed_short";
			break;
		case AT_UnsignedShort:
			out << "unsigned_short";
			break;
		case AT_Int:
			out << "signed_int";
			break;
		case AT_UnsignedInt:
			out << "unsigned_int";
			break;
	}
	out << "</alphtype>\n";
}

void XMLCodeGen::writeGetKeyExpr()
{
	out << "  <getkey>";
	writeInlineList( pd->getKeyExpr, 0 );
	out << "</getkey>\n";
}

void XMLCodeGen::writeAccessExpr()
{
	out << "  <access>";
	writeInlineList( pd->accessExpr, 0 );
	out << "</access>\n";
}

void XMLCodeGen::writeCurStateExpr()
{
	out << "  <curstate>";
	writeInlineList( pd->curStateExpr, 0 );
	out << "</curstate>\n";
}

void XMLCodeGen::writeXML()
{
	/* Open the definition. */
	out << "<ragel_def name=\"" << fsmName << "\">\n";

	if ( pd->alphTypeSet )
		writeAlphType();
	
	if ( pd->getKeyExpr != 0 )
		writeGetKeyExpr();

	if ( pd->accessExpr != 0 )
		writeAccessExpr();

	if ( pd->curStateExpr != 0 )
		writeCurStateExpr();

	writeMachine();

	out <<
		"</ragel_def>\n";
}

