/*
 *  Copyright 2001-2005 Adrian Thurston <thurston@cs.queensu.ca>
 *            2004 Eric Ocean <eric.ocean@ampede.com>
 *            2005 Alan West <alan@alanz.com>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef _FSMCODEGEN_H
#define _FSMCODEGEN_H

#include <iostream>
#include <string>
#include <stdio.h>

using std::string;
using std::ostream;

/* Integer array line length. */
#define IALL 8

/* Forwards. */
struct RedFsmAp;
struct CodeGenData;
struct Action;
struct NameInst;
struct InlineItem;
struct InlineList;
struct RedAction;
struct LongestMatch;
struct LongestMatchPart;

inline string itoa( int i )
{
	char buf[16];
	sprintf( buf, "%i", i );
	return buf;
}

/*
 * class FsmCodeGen
 */
class FsmCodeGen
{
public:
	FsmCodeGen();
	virtual ~FsmCodeGen() {}

	virtual void writeOutData() = 0;
	virtual void writeOutInit();
	virtual void writeOutExec() = 0;
	virtual void writeOutEOF() = 0;
	
	/* Gather various info on the machine. */
	void analyzeAction( Action *act, InlineList *inlineList );
	void analyzeActionList( RedAction *redAct, InlineList *inlineList );
	void analyzeMachine();
	
protected:
	friend struct CodeGenData;

	string FSM_NAME();
	string START_STATE_ID();
	ostream &ACTIONS_ARRAY();
	string GET_KEY();
	ostream &TABS( int level );
	string KEY( long key );
	string LDIR_PATH( char *path );
	void ACTION( ostream &ret, Action *action, int targState, bool inFinish );

	virtual string ARR_OFF( string ptr, string offset ) = 0;
	virtual string CAST( string type ) = 0;
	virtual string UINT( ) = 0;
	virtual string ARRAY_TYPE( unsigned long maxVal ) = 0;
	virtual string ALPH_TYPE() = 0;
	virtual string NULL_ITEM() = 0;
	virtual string POINTER() = 0;
	virtual ostream &SWITCH_DEFAULT() = 0;

	string P() { return "p"; }
	string PE() { return "pe"; }

	string ACCESS();
	string CS();
	string STACK() { return ACCESS() + "stack"; }
	string TOP() { return ACCESS() + "top"; }
	string TOKSTART() { return ACCESS() + "tokstart"; }
	string TOKEND() { return ACCESS() + "tokend"; }
	string ACT() { return ACCESS() + "act"; }

	string DATA_PREFIX();
	string K() { return "_" + DATA_PREFIX() + "k"; }
	string I() { return "_" + DATA_PREFIX() + "i"; }
	string KO() { return "_" + DATA_PREFIX() + "ko"; }
	string IO() { return "_" + DATA_PREFIX() + "io"; }
	string SL() { return "_" + DATA_PREFIX() + "sl"; }
	string RL() { return "_" + DATA_PREFIX() + "rl"; }
	string A() { return "_" + DATA_PREFIX() + "a"; }
	string TA() { return "_" + DATA_PREFIX() + "ta"; }
	string TT() { return "_" + DATA_PREFIX() + "tt"; }
	string TSA() { return "_" + DATA_PREFIX() + "tsa"; }
	string FSA() { return "_" + DATA_PREFIX() + "fsa"; }
	string EA() { return "_" + DATA_PREFIX() + "ea"; }
	string SP() { return "_" + DATA_PREFIX() + "sp"; }
	string START() { return DATA_PREFIX() + "start"; }
	string ERROR() { return DATA_PREFIX() + "error"; }
	string FIRST_FINAL() { return DATA_PREFIX() + "first_final"; }
	string CTXDATA() { return DATA_PREFIX() + "ctxdata"; }

	void INLINE_LIST( ostream &ret, InlineList *inlineList, int targState, bool inFinish );
	virtual void GOTO( ostream &ret, int gotoDest, bool inFinish ) = 0;
	virtual void CALL( ostream &ret, int callDest, int targState, bool inFinish ) = 0;
	virtual void NEXT( ostream &ret, int nextDest, bool inFinish ) = 0;
	virtual void GOTO_EXPR( ostream &ret, InlineItem *ilItem, bool inFinish ) = 0;
	virtual void NEXT_EXPR( ostream &ret, InlineItem *ilItem, bool inFinish ) = 0;
	virtual void CALL_EXPR( ostream &ret, InlineItem *ilItem, 
			int targState, bool inFinish ) = 0;
	virtual void RET( ostream &ret, bool inFinish ) = 0;
	virtual void BREAK( ostream &ret, int targState ) = 0;
	virtual void CURS( ostream &ret, bool inFinish ) = 0;
	virtual void TARGS( ostream &ret, bool inFinish, int targState ) = 0;
	void EXEC( ostream &ret, InlineItem *item, int targState, int inFinish );
	void LM_SWITCH( ostream &ret, InlineItem *item, int targState, int inFinish );
	void LM_SET_ACT_ID( ostream &ret, InlineItem *item );
	void LM_INIT_TOK_START( ostream &ret, InlineItem *item );
	void LM_INIT_ACT( ostream &ret, InlineItem *item );
	void LM_SET_TOK_START( ostream &ret, InlineItem *item );
	void LM_SET_TOK_END( ostream &ret, InlineItem *item );
	void LM_GET_TOK_END( ostream &ret, InlineItem *item );
	void SUB_ACTION( ostream &ret, InlineItem *item, 
			int targState, bool inFinish );

	string ERROR_STATE();
	string FIRST_FINAL_STATE();

	virtual ostream &OPEN_ARRAY( string type, string name ) = 0;
	virtual ostream &CLOSE_ARRAY() = 0;

	int arrayTypeSize( unsigned long maxVal );

	bool anyActions();
	bool anyToStateActions()        { return bAnyToStateActions; }
	bool anyFromStateActions()      { return bAnyFromStateActions; }
	bool anyRegActions()            { return bAnyRegActions; }
	bool anyEofActions()            { return bAnyEofActions; }
	bool anyActionCalls()           { return bAnyActionCalls; }
	bool anyActionRets()            { return bAnyActionRets; }
	bool anyRegActionRets()         { return bAnyRegActionRets; }
	bool anyRegActionByValControl() { return bAnyRegActionByValControl; }
	bool anyRegNextStmt()           { return bAnyRegNextStmt; }
	bool anyRegCurStateRef()        { return bAnyRegCurStateRef; }
	bool anyRegBreak()              { return bAnyRegBreak; }
	bool anyLmSwitchError()         { return bAnyLmSwitchError; }

	/* Set up labelNeeded flag for each state. Differs for each goto style so
	 * is virtual. */
	virtual void setLabelsNeeded() {}

	/* Determine if we should use indicies. */
	virtual void calcIndexSize() {}

	void findFinalActionRefs();
	void assignActionIds();
	void setValueLimits();

	/* Are there any regular transition functions, any out transition functions. */
	char *fsmName;
	CodeGenData *cgd;
	RedFsmAp *redFsm;

protected:
	ostream &out;

	bool bAnyToStateActions;
	bool bAnyFromStateActions;
	bool bAnyRegActions;
	bool bAnyEofActions;
	bool bAnyActionCalls;
	bool bAnyActionRets;
	bool bAnyRegActionRets;
	bool bAnyRegActionByValControl;
	bool bAnyRegNextStmt;
	bool bAnyRegCurStateRef;
	bool bAnyRegBreak;
	bool bAnyLmSwitchError;

	int maxState;
	int maxSingleLen;
	int maxRangeLen;
	int maxKeyOffset;
	int maxIndexOffset;
	int maxIndex;
	int maxActListId;
	int maxActionLoc;
	int maxActArrItem;
	unsigned long long maxSpan;
	int maxFlatIndexOffset;

	bool useIndicies;
};

class CCodeGen : virtual public FsmCodeGen
{
public:
	virtual string ARRAY_TYPE( unsigned long maxVal );
	virtual string ALPH_TYPE();
	virtual string NULL_ITEM();
	virtual string POINTER();
	virtual ostream &SWITCH_DEFAULT();
	virtual ostream &OPEN_ARRAY( string type, string name );
	virtual ostream &CLOSE_ARRAY();
	virtual string ARR_OFF( string ptr, string offset );
	virtual string CAST( string type );
	virtual string UINT( );
};

class DCodeGen : virtual public FsmCodeGen
{
public:
	virtual string ARRAY_TYPE( unsigned long maxVal );
	virtual string ALPH_TYPE();
	virtual string NULL_ITEM();
	virtual string POINTER();
	virtual ostream &SWITCH_DEFAULT();
	virtual ostream &OPEN_ARRAY( string type, string name );
	virtual ostream &CLOSE_ARRAY();
	virtual string ARR_OFF( string ptr, string offset );
	virtual string CAST( string type );
	virtual string UINT( );
};

#endif /* _FSMCODEGEN_H */
