/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "dnsmanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <QMenu>
#include <QSettings>
#include <QShortcut>
#include <RdsUtils>
#include <RdsDnsRecord>
#include <RdsDaemonManager>
#include "dnseditwidget.h"
#include "dnszoneeditwidget.h"
#include "dnsrecordeditwidget.h"
#include "adddomaindialog.h"
#include "addrecorddialog.h"

DnsManagerWidget::DnsManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::DnsManagerWidget()
{
	setupUi(this);
	pushButton_4->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	DeleteButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	ReloadButton->setIcon(QPixmap(findRdsIcon("icons/32x32/reload.png")));

	ReturnValue ret = rdsClient()->getService("DnsManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get DNS manager", ret.errString());
		return;
	}

	_manager = new RdsDnsManager(this);
	*_manager = ret;

	_model = new RdsDnsModel(_manager, this);
	QObject::connect(_model, SIGNAL(loaded()), this, SLOT(onModelLoaded()));
	_sort = new RdsEntitySortModel(_model);
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	View->sortByColumn(0, Qt::AscendingOrder);
	_selection = View->selectionModel();
	_model->setupSelection(_sort, View);
	View->header()->resizeSection(0, 200);


	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(FilterEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));

	DnsEditWidget *w = new DnsEditWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");

	//Actions and Menu stuff
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));

	_adddomainaction = new QAction("Add Domain", this);
	QObject::connect(_adddomainaction, SIGNAL(triggered()), this, SLOT(addDomain()));
	_addoriginaction = new QAction("Add Origin", this);
	QObject::connect(_addoriginaction, SIGNAL(triggered()), this, SLOT(addOrigin()));
	_addrecordaction = new QAction("Add Record", this);
	QObject::connect(_addrecordaction, SIGNAL(triggered()), this, SLOT(addRecord()));
	_removeaction = new QAction("Delete", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(on_DeleteButton_clicked()));
	_showsystemaction = new QAction("Show System Records", this);
	QObject::connect(_showsystemaction, SIGNAL(toggled(bool)), this, SLOT(showSystemObjects(bool)));
	_showsystemaction->setCheckable(true);

	QSettings settings;
	if (settings.value("DNS/ShowSystemsObjects").toBool())
	{
		_showsystemaction->setChecked(true);
		showSystemObjects(true);
	}
	else
	{
		_showsystemaction->setChecked(false);
		showSystemObjects(false);
	}

	QMenu *addmenu = new QMenu(this);
	addmenu->addAction(_adddomainaction);
	addmenu->addAction(_addoriginaction);
	addmenu->addAction(_addrecordaction);
	addmenu->addAction(_removeaction);
	AddButton->setMenu(addmenu);

	_removeaction->setShortcut(QKeySequence("Del"));
}

DnsManagerWidget::~DnsManagerWidget()
{
}

void DnsManagerWidget::onModelLoaded()
{
	RdsEntityModel::Cache *managercache = _model->getCache("manager");
	if (managercache != NULL)
	{
		View->expand(_sort->mapFromSource(_model->indexFromCache(managercache)));
	}
}

void DnsManagerWidget::inputChanged(QString id, QString type)
{
	_input = id;
// 	qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The User you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		delete _edit;
		_edit = NULL;
	}

	if (type == "manager")
	{
		DnsEditWidget *w = new DnsEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "zone")
	{
		DnsZoneEditWidget *w = new DnsZoneEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "record")
	{
		DnsRecordEditWidget *w = new DnsRecordEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else
	{
		DnsEditWidget *w = new DnsEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void DnsManagerWidget::outputsChanged(QStringList outputs)
{
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void DnsManagerWidget::addDomain()
{
	AddDomainDialog dialog(_manager, this);
	dialog.exec();
}

void DnsManagerWidget::addOrigin()
{
	if ((_model->input() == "") || (_model->input() == "manager"))
	{
		QMessageBox::critical(this, "Error", "You must select a domain or origin to add the origin to.");
		return;
	}

	QString zone = _model->getZoneFromId(_model->input());
	QString origin = _model->getOriginFromId(_model->input());

	bool ok = false;
	QString name = QInputDialog::getText(this, "New Origin", "Enter the name for the new origin", QLineEdit::Normal, "", &ok);

	if (!ok)
	{
		return;
	}

	if (name == "")
	{
		QMessageBox::warning(this, "Error", "You must give the new folder a name.");
		return;
	}

	if (!QRegExp("[0-9A-Za-z_-]*").exactMatch(name))
	{
		QMessageBox::critical(NULL, "Error", "The name must contain only numbers, letters, underscores (_), and the hyphen (_).");
		return;
	}

	origin.prepend(name + ".");

	ReturnValue ret = _manager->zone(zone);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get zone " + zone + ": " + ret.errString());
		return;
	}

	RdsDnsZone z = ret;
// 	qDebug() << "Adding:" << origin;
	ret = z.addOrigin(origin);

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to add origin " + name + ": " + ret.errString());
		return;
	}

	int index = _model->input().indexOf("#");
	QString id = _model->input();
	if (index > 0) id = id.left(index);

	id += ":" + name;

	_manager->addEntity(id);

	ret = z.save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save zone " + zone + ": " + ret.errString());
		return;
	}

	save();
}

void DnsManagerWidget::addRecord()
{
	if ((_model->input() == "") || (_model->input() == "manager"))
	{
		QMessageBox::critical(this, "Error", "You must select a domain or origin to add the record to.");
		return;
	}

	QString zone = _model->getZoneFromId(_model->input());
	QString origin = _model->getOriginFromId(_model->input());


	AddRecordDialog dialog(zone, origin, _manager, this);
	dialog.exec();
}

void DnsManagerWidget::on_DeleteButton_clicked()
{
	QStringList outputs = _model->outputs();

	if (outputs.size() == 0)
	{
		QMessageBox::information(this, "Error", "You must select something to delete.");
		return;
	}
	else if (outputs.size() == 1)
	{
		RdsEntityModel::Cache *cache = _model->getCache(outputs[0]);
		if (cache == NULL) return;

		int result;

		if (cache->type() == "zone")
		{
			result = QMessageBox::question(this, "Delete", QString("Are you shure you want to delete the domain %1?").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "origin")
		{
			result = QMessageBox::question(this, "Delete", QString("Do you want to delete the origin %1?").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "record")
		{
			result = QMessageBox::question(this, "Delete", QString("Do you want to delete the record %1?").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else return;

		if (result != QMessageBox::Yes) return;
	}
	else
	{
		int result = QMessageBox::question(this, "Delete Items", QString("Do you want to delete %1 items?").arg(outputs.count()),
		                                   QMessageBox::Yes | QMessageBox::No, QMessageBox::No);

		if (result != QMessageBox::Yes)	return;
	}

	ReturnValue ret, err(true);

	foreach(QString output, outputs)
	{
		RdsEntityModel::Cache *cache = _model->getCache(output);
		if (cache == NULL) continue;

		if (cache->type() == "zone")
		{
			ReturnValue ret = _manager->removeZone(output);
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to delete zone " + output + ": " + ret.errString());
				continue;
			}

			ret = _manager->save();
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to save settings: " + ret.errString());
				continue;
			}

			_manager->removeEntity(output);
		}
		else if (cache->type() == "origin")
		{
			QString zone = _model->getZoneFromId(output);
			QString origin = _model->getOriginFromId(output);

			ret = _manager->zone(zone);
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to get zone " + zone + ": " + ret.errString());
				continue;
			}

			RdsDnsZone z = ret;
// 			qDebug() << "Trying to remove:" << origin;
			ret = z.removeOrigin(origin);

			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to remove origin " + origin + ": " + ret.errString());
				continue;
			}

			ret = z.save();
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to save domain " + zone + ": " + ret.errString());
				continue;
			}

			_manager->removeEntity(output);
		}
		else if (cache->type() == "record")
		{
			QString zone = _model->getZoneFromId(output);

			ret = _manager->zone(zone);
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to get zone " + zone + ": " + ret.errString());
				continue;
			}

			RdsDnsZone z = ret;

			RdsDnsRecord record(output);
			ret = z.removeRecord(record);

			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to remove record " + record.key() + ": " + ret.errString());
				continue;
			}

			ret = z.save();
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to save domain " + zone + ": " + ret.errString());
				continue;
			}

			_manager->removeEntity(output);
		}

		//qDebug() << "Attempting to delete a" << cache->type() << "?";
	}

	View->selectionModel()->clear();
	inputChanged("", "");

	save();

	if (err.isError())
	{
		QMessageBox::warning(this, "Warning", "Some items could not be deleted: " + err.errString());
	}
}

void DnsManagerWidget::showSystemObjects(bool show)
{
	if (show)
		_sort->setHiddenEntitiesRegex(QStringList());
	else
		_sort->setHiddenEntitiesRegex(QStringList() << "^.+[.][:#]_.+$");

	QSettings settings;
	settings.setValue("DNS/ShowSystemsObjects", show);
}


void DnsManagerWidget::contextMenu(const QPoint & pos)
{
	Q_UNUSED(pos);
	QMenu menu;
	menu.addAction(_adddomainaction);
	menu.addAction(_addoriginaction);
	menu.addAction(_addrecordaction);
	menu.addAction(_removeaction);
	menu.addSeparator();
	menu.addAction(_showsystemaction);

	menu.exec(QCursor::pos());
}

void DnsManagerWidget::on_ReloadButton_clicked()
{
	_model->reload();
}

bool DnsManagerWidget::save()
{
	RdsDaemonManager mgr;
	ReturnValue ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.reloadService("Dns");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DNS server: " + ret.errString());
		return(false);
	}

	return(true);
}

