/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "drivemapwidget.h"
#include "selectiondialog.h"
#include <RdsAdObject>
#include <RdsComputer>
#include <RdsUser>
#include <RdsGroup>
#include <RdsUtils>
#include <QMessageBox>
#include <QDebug>
#include <QRegExp>
#include "config.h"

DriveMapWidget::DriveMapWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::DriveMapWidget()
{
	setupUi(this);
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	RemoveButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	
	_usericon = QPixmap(findRdsIcon("./icons/16x16/userconfig.png"));
	_groupicon = QPixmap(findRdsIcon("./icons/hi32-group.png")).scaled(16, 16);
	_computericon = QPixmap(findRdsIcon("./icons/16x16/system.png"));

	_nochange = false;
}

DriveMapWidget::~DriveMapWidget()
{
}

void DriveMapWidget::mapChanged()
{
	if (!_nochange) emit(changed());
	if (NoMap->isChecked())
	{
		DriveLabelEveryone->setEnabled(false);
		DriveComboEveryone->setEnabled(false);
		DriveLabel->setEnabled(false);
		DriveCombo->setEnabled(false);
		UserGroupFrame->setEnabled(false);
	}
	else if (EveryoneMap->isChecked())
	{
		DriveLabelEveryone->setEnabled(true);
		DriveComboEveryone->setEnabled(true);
		DriveLabel->setEnabled(false);
		DriveCombo->setEnabled(false);
		UserGroupFrame->setEnabled(false);
	}
	else if (UserGroupMap->isChecked())
	{
		DriveLabelEveryone->setEnabled(false);
		DriveComboEveryone->setEnabled(false);
		DriveLabel->setEnabled(true);
		DriveCombo->setEnabled(true);
		UserGroupFrame->setEnabled(true);
	}
}

QString DriveMapWidget::mapString() const
{
	if (NoMap->isChecked())
	{
		return("");
	}
	else if (EveryoneMap->isChecked())
	{
		//if a drive letter is selected, just return it
		if (DriveComboEveryone->currentText().endsWith(":"))
		{
			return(DriveComboEveryone->currentText());
		}
		else //if not return *:
		{
			return("*:");
		}
	}
	else if (UserGroupMap->isChecked())
	{
		QString mapstring;
		QStringList users, groups, computers;

		for (int i = 0; i < UserGroupList->count(); i++)
		{
			QListWidgetItem *item = UserGroupList->item(i);
			switch (item->data(Qt::UserRole).toInt())
			{
				case User:
					users << _ids[i];
					break;
				case Group:
					groups << _ids[i];
					break;
				case Computer:
					computers << _ids[i];
					break;
				default:
					break;
			}
		}

		if((users.size() == 0) && (groups.size() == 0) && (computers.size() == 0)) return("");

		if (DriveCombo->currentText().endsWith(":"))
		{
			mapstring = DriveCombo->currentText();
		}
		else //if not return *:
		{
			mapstring = "*:";
		}

		mapstring = QString("%1 %2 %3 %4").arg(mapstring).arg(genList(users)).arg(genList(groups)).arg(genList(computers));
		return(mapstring);
	}

	return("");
}

void DriveMapWidget::setMapString(const QString &mapstring)
{
	_nochange = true;
	//qDebug() << "Map String:" << mapstring;
	UserGroupList->clear();
	_ids.clear();

	if (mapstring == "") NoMap->setChecked(true);
	else if (mapstring.endsWith(":"))
	{
		EveryoneMap->setChecked(true);
		if (mapstring == "*:")
		{
			DriveComboEveryone->setCurrentIndex(0);
		}
		else
		{
			int index = DriveComboEveryone->findText(mapstring, Qt::MatchFixedString);
			if (index == -1) index = 0;
			DriveComboEveryone->setCurrentIndex(index);
		}
	}
	else
	{
		UserGroupMap->setChecked(true);


		QString local = mapstring.trimmed();
		QRegExp regex("^\\s*([a-zA-Z*]*:)\\s*\\((.*)\\)\\s*\\((.*)\\)\\s*\\((.*)\\)\\s*$");
		regex.exactMatch(local);

		if (regex.captureCount() != 4)
		{
			qWarning() << "Malformed map string!" << regex.captureCount() << regex.capturedTexts() ;
			return;
		}

		QString drivestr;
		QStringList users, groups, computers;

		drivestr = regex.cap(1);
		users = parseList(regex.cap(2));
		groups = parseList(regex.cap(3));
		computers = parseList(regex.cap(4));

		int index = DriveCombo->findText(drivestr, Qt::MatchFixedString);
		if (index == -1) index = 0;
		DriveCombo->setCurrentIndex(index);

		foreach(QString tmp, users)
		{
			ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(sAMAccountName="+tmp+")");
			QStringList list = ret.toStringList();
			if(list.size() < 1) continue;
			
			addItem(list[0]);
		}

		foreach(QString tmp, groups)
		{
			ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(sAMAccountName="+tmp+")");
			QStringList list = ret.toStringList();
			if(list.size() < 1) continue;
			
			addItem(list[0]);
		}

		foreach(QString tmp, computers)
		{
			ReturnValue ret = rdsLdapSession()->list(RdsUtils::baseDn(), "(sAMAccountName="+tmp+")");
			QStringList list = ret.toStringList();
			if(list.size() < 1) continue;
			
			addItem(list[0]);
		}
	}
	_nochange = false;
}

QString DriveMapWidget::genList(QStringList list) const
{
	bool first = true;
	QString tmp = "(";
	foreach(QString item, list)
	{
		item = item.replace("\\", "\\\\");
		item = item.replace(",", "\\,");

		if (!first) tmp += ",";
		first = false;

		tmp += item;
	}

	tmp += ")";

	return(tmp);
}

QStringList DriveMapWidget::parseList(QString str) const
{
	QStringList list;
	bool escape = false;
	QString tmp;

	for (int i = 0; i < str.size(); i++)
	{
		if (escape)
		{
			escape = false;
			tmp += str[i];
		}
		else
		{
			if (str[i] == '\\')
			{
				escape = true;
			}
			else if (str[i] == ',')
			{
				if (tmp.trimmed() == "") continue;
				list << tmp.trimmed();
				tmp = "";
			}
			else
			{
				tmp += str[i];
			}
		}
	}

	if (tmp.trimmed() != "")
	{
		list << tmp.trimmed();
		tmp = "";
	}

	return(list);
}

void DriveMapWidget::on_AddButton_clicked()
{
	SelectionDialog dialog(SelectionDialog::User | SelectionDialog::Group | SelectionDialog::Computer, this);
	dialog.setAllowMultipleSelections(true);
	if (!dialog.exec()) return;

	foreach(QString dn, dialog.selectedItems())
	{
		addItem(dn);
	}
	emit(changed());
}

void DriveMapWidget::on_RemoveButton_clicked()
{
	foreach(QListWidgetItem *item, UserGroupList->selectedItems())
	{
		_ids.removeAt(UserGroupList->row(item));
		delete item;
	}
	
	emit(changed());
}

void DriveMapWidget::addItem(QString dn)
{
	QString text;
	QPixmap icon;
	Type type;

	RdsAdObject object(dn);
	ReturnValue ret = object.readAttribute("cn");
	if (!ret.isError())
	{
		LdapValues attr = ret.value<LdapValues>();
		if (attr.size() > 0)
		{
			text = attr[0];

			ret = object.readAttribute("objectClass");
			if (!ret.isError())
			{
				LdapValues attr = ret.value<LdapValues>();
				if (attr.contains(QString("computer").toAscii()))
				{
					type = Computer;
					icon = _computericon;
				}
				else if (attr.contains(QString("user").toAscii()))
				{
					type = User;
					icon = _usericon;
				}
				else if (attr.contains(QString("group").toAscii()))
				{
					
					type = Group;
					icon = _groupicon;
				}
				else
				{
					return;
				}
			}
		}
	}

	bool found = false;

	for (int i = 0; i < UserGroupList->count(); i++)
	{
		QListWidgetItem *item = UserGroupList->item(i);

		if ((item->text() == text) && (item->data(Qt::UserRole).toInt() == type)) found = true;
	}

	if (found) return;

	ret = object.readAttribute("sAMAccountName");
	if (ret.isError())
	{
		QMessageBox::critical(this,"Error","Failed to look up the ID: " + ret.errString());
		return;
	}

	LdapValues value = ret.value<LdapValues>();
	_ids << QString(value[0]);
	
	QListWidgetItem *item = new QListWidgetItem(icon, text, UserGroupList);
	item->setData(Qt::UserRole, type);
}
