/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "newuserdialog.h"
#include <RdsUtils>
#include <QMessageBox>
#include <RdsOrganizationalUnit>
#include <RdsUser>
#include <QDebug>
#include <RdsClient>
#include <QTimer>
#include <RdsSid>
#include <RdsFileManager>
#include <RdsNtAcl>

NewUserDialog::NewUserDialog(QString ou, bool clone, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::NewUserDialog(), _cloneuser(ou)
{
	setupUi(this);
	_clone = clone;
	Aliases->setErrorText("You must specify a valid email address.");
	Aliases->setWeakValidator(QRegExp("[a-zA-Z0-9._%+-@]*"));
	Aliases->setStrongValidator(QRegExp("[a-zA-Z0-9._%+-]*@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,4}"));
	Aliases->setUpDownArrows(false);
	
	if (clone)
	{
		QStringList tmp = ou.split(",");
		if (tmp.size() > 0) tmp.removeAt(0);
		ou = tmp.join(",");
		setWindowTitle("Clone User");

		ReturnValue ret = _cloneuser.groups();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Failed to get groups: ", ret.errString());
		}
		GroupList->setGroups(ret.toStringList());

		ret = _cloneuser.primaryGroup();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Failed to get primary group: ", ret.errString());
		}
		PrimaryGroup->setDn(ret.toString());

		ret = _cloneuser.flags();
		if (ret.isError())
		{
			QMessageBox::critical(this, "Failed to get flags: ", ret.errString());
		}
		RdsUser::UserStates flags = ret.value<RdsUser::UserStates>();
		Disabled->setChecked((RdsUser::Disabled & flags) != 0);
		MustChangePassword->setChecked((RdsUser::MustChangePassword & flags) != 0);
		CannotChangePassword->setChecked((RdsUser::CannotChangePassword & flags) != 0);
		PasswordNeverExpires->setChecked((RdsUser::PasswordNeverExpires & flags) != 0);
	}
	else
	{
		QString groupdn;
		RdsSid sid = RdsSid::getDomainSid();
		sid.setRid(513);
		
		ReturnValue ret = RdsUtils::getObjectBySid(sid);
		if(ret.isError())
		{
			groupdn = "CN=Domain Users,CN=Users," + RdsUtils::baseDn();
		}
		else
		{
			groupdn = ret.toString();
		}
		
		GroupList->setGroups(QStringList() << groupdn);
		PrimaryGroup->setDn(groupdn);
	}

	_ou = ou;

	ReturnValue ret = rdsClient()->getService("UserGroupComputerManager");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get manager: ", ret.errString());
	}
	else
	{
		_manager = ret;
	}

	UserCreatedLabel->setVisible(false);
}

NewUserDialog::~NewUserDialog()
{
}

void NewUserDialog::on_MustChangePassword_toggled(bool state)
{
	if (state) CannotChangePassword->setChecked(false);
}

void NewUserDialog::on_CannotChangePassword_toggled(bool state)
{
	if (state) MustChangePassword->setChecked(false);
}

void NewUserDialog::on_NextButton_clicked()
{
	newUser();
}

void NewUserDialog::on_FinishButton_clicked()
{
	if (newUser()) accept();
}

bool NewUserDialog::newUser()
{
	if (FullName->text() == "")
	{
		QMessageBox::critical(this, "Incomplete Information", "You must enter a full name.");
		return(false);
	}

	if (UserName->text() == "")
	{
		QMessageBox::critical(this, "Incomplete Information", "You must enter a user name.");
		return(false);
	}

	if (!FullName->text().contains(QRegExp("^[0-9a-zA-Z][^+\\;,=]*$")))
	{
		QMessageBox::critical(NULL, "Error", "The full name must not contain any of these characters (+ \\ ; , =).");
		return(false);
	}

	if (!UserName->text().contains(QRegExp("^[0-9a-zA-Z][^\\\\\\[\\]:;|=+?<>*\"]*$")))
	{
		QMessageBox::critical(NULL, "Error", "The user name must not contain any of these characters ([ ] : ; | = + ? < > * \\ \").");
		return(false);
	}

	if (Password->text() != Password2->text())
	{
		QMessageBox::critical(this, "Invalid Information", "The passwords do not match.");
		return(false);
	}

	if ((Email->text() != "") && !Email->text().contains(QRegExp("^[a-zA-Z0-9._%+-]*@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,4}$")))
	{
		QMessageBox::critical(NULL, "Error", "The email address you provided is not valid.");
		return(false);
	}

	RdsOrganizationalUnit ou(_ou);
	ReturnValue ret = ou.createUser(FullName->text(), UserName->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create user: " + ret.errString());
		return(false);
	}

	RdsUser user(ret.toString());

	ret = user.setPassword(Password->text());
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to set password: " + ret.errString());
		ReturnValue ret = user.remove();
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to remove the invalid user: " + ret.errString());
		}
		return(false);
	}


	RdsUser::UserStates flags;
	if (Disabled->isChecked()) flags |= RdsUser::Disabled;
	if (MustChangePassword->isChecked()) flags |= RdsUser::MustChangePassword;
	if (CannotChangePassword->isChecked()) flags |= RdsUser::CannotChangePassword;
	if (PasswordNeverExpires->isChecked()) flags |= RdsUser::PasswordNeverExpires;

	if (_clone)
	{
		ret = _cloneuser.flags();
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to get flags: " + ret.errString());
		}
		RdsUser::UserStates oldflags = ret.value<RdsUser::UserStates>();
		oldflags &= (RdsUser::Disabled | RdsUser::MustChangePassword | RdsUser::CannotChangePassword |
		             RdsUser::PasswordNeverExpires);
		flags |= oldflags;
	}

	ret = user.setFlags(flags);
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to set flags: " + ret.errString());
	}

	ret = user.groups();
	if (!ret.isError())
	{
		QStringList newgroups = GroupList->groups();
		QStringList oldgroups = ret.toStringList();

		//Add new groups
		foreach(QString group, newgroups)
		{
			if (!oldgroups.contains(group))
			{
				ret = user.joinGroup(group);
				//qDebug() << "Joining Group:" << group;
				if (ret.isError())
				{
					QMessageBox::warning(this, "Error", "Failed to join group: " + ret.errString());
				}
			}
		}

		foreach(QString group, oldgroups)
		{
			if (!newgroups.contains(group))
			{
				ret = user.leaveGroup(group);
				//qDebug() << "Leaving Group:" << group;
				if (ret.isError())
				{
					QMessageBox::warning(this, "Error", "Failed to leave: " + ret.errString());
				}
			}
		}
	}
	else
	{
		QMessageBox::warning(this, "Error", "Failed list groups: " + ret.errString());
	}

	ret = user.setPrimaryGroup(PrimaryGroup->dn());
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to set primary group: " + ret.errString());
	}
	
	if(Email->text() != "")
	{
		ret = user.setEmail(processEmail(Email->text()));
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to set email address: " + ret.errString());
		}
	}
	
	if(Aliases->list().size() > 0)
	{
		QStringList aliases;
		
		foreach(QString alias, Aliases->list())
		{
			aliases << processEmail(alias);
		}
		
		ret = user.setEmailAliases(aliases);
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to set email aliases: " + ret.errString());
		}
	}

#define copyString(func, setfunc) ret = _cloneuser.func(); \
	if(ret.isError()) \
	{\
		if(!ret.errString().endsWith("attribute does not exist"))\
		{\
			err = true; \
			qCritical() << "Failed to get data: " #func << ret.errString(); \
		}\
	}\
	else\
	{\
		ret = user.setfunc(ret.toString()); \
		if(ret.isError() && (ret.errString() != "No such attribute")) \
		{\
			err = true; \
			qCritical() << "Failed to set data: "  #setfunc << ret.errString(); \
		}\
	}
	
#define copyStringReplaceUsername(func, setfunc) ret = _cloneuser.func(); \
	if(ret.isError()) \
	{\
		if(!ret.errString().endsWith("attribute does not exist"))\
		{\
			err = true; \
			qCritical() << "Failed to get data: " #func << ret.errString(); \
		}\
	}\
	else\
	{\
		ret = user.setfunc(ret.toString().replace(oldname,UserName->text())); \
		if(ret.isError() && (ret.errString() != "No such attribute")) \
		{\
			err = true; \
			qCritical() << "Failed to set data: "  #setfunc << ret.errString(); \
		}\
	}

	if (_clone)
	{
		bool err = false;

		ret = _cloneuser.userName();
		if(ret.isError())
		{
			err = true;
			qCritical() << "Failed to get original users name: " << ret.errString();
		}
		
		QString oldname = ret.toString();
		
		copyString(description, setDescription);
		copyString(office, setOffice);
		copyString(phoneNumber, setPhoneNumber);
		copyString(webPage, setWebPage);
		copyString(street, setStreet);
		copyString(poBox, setPoBox);
		copyString(city, setCity);
		copyString(state, setState);
		copyString(country, setCountry);
		copyString(postalCode, setPostalCode);
		copyString(homeDrive, setHomeDrive);
		copyStringReplaceUsername(homePath, setHomePath);
		copyStringReplaceUsername(profilePath, setProfilePath);
		copyString(logonScript, setLogonScript);
		copyString(pagerNumber, setPagerNumber);
		copyString(mobileNumber, setMobileNumber);
		copyString(faxNumber, setFaxNumber);
		copyString(homeNumber, setHomeNumber);
		copyString(notes, setNotes);
		copyString(title, setTitle);
		copyString(department, setDepartment);
		copyString(company, setCompany);

		ret = user.homePath();
		if(ret.isError())
		{
			if(!ret.errString().endsWith("attribute does not exist"))
			{
				err = true;
				qCritical() << "Failed to get home path: " << ret.errString();
			}
		}
		else
		{
			QString path = ret.toString().replace("\\","/");
			ret = rdsClient()->hostname();
			path = path.replace("//" + ret.toString(), "@SHARE");
			
			//qDebug() << "PATH:" << path;
			
			RdsFileManager fmgr;
			ret = fmgr.init();
			if(ret.isError())
			{
				
			}
			else
			{
				ret = fmgr.isDirectory(path);
				if(!ret.isError())
				{
					if(!ret.toBool())
					{
						ret = fmgr.createDirectory(path, true);
						if(ret.isError())
						{
							qWarning() << "Failed to create home directory:" << ret;
						}
						
						ret = fmgr.ntPermissions(path);
						if(ret.isError())
						{
							qWarning() << "Failed to get permissions for:" << path << ret;
						}
						else
						{
							RdsNtAcl acl = ret.value<RdsNtAcl>();
							ret = user.sid();
							if(ret.isError())
							{
								qWarning() << "Failed to get sid for user" << ret;
							}
							else
							{
								RdsAce ace;
								ace.setSid(ret.value<RdsSid>());
								ace.setFlags(RdsAce::FileInherit | RdsAce::FolderInherit);
								ace.setAccess(RdsAce::StandardAll | RdsAce::FileAll);
								acl.dacl() << ace;
								
								ret = fmgr.setNtPermissions(path, acl);
								if(ret.isError())
								{
									qWarning() << "Failed to set permissions for path" << path << ret;
								}
							}
							
						}
					}
				}
			}
		}
		
		if (err)
		{
			QMessageBox::warning(this, "Error", "Some attributes failed to copy.");
		}
	}

	_manager.addEntity(user.dn());

	FullName->setText("");
	UserName->setText("");
	Password->setText("");
	Password2->setText("");
	if(Email->text().contains("@"))
	{
		Email->setText("%USERNAME%" + Email->text().mid(Email->text().indexOf("@")));
	}
	else
	{
		Email->clear();
	}
	Aliases->setList(QStringList());
	UserName->setFocus();

	UserCreatedLabel->setVisible(true);
	QTimer::singleShot(3000, this, SLOT(hideUserCreatedLabel()));

	return(true);
}

void NewUserDialog::hideUserCreatedLabel()
{
	UserCreatedLabel->setVisible(false);
}

QString NewUserDialog::processEmail(QString email)
{
	email = email.replace("%USERNAME%",UserName->text());
	if(email.startsWith("@")) email = email;
	return(email);
}

