/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "provisionwizard.h"
#include <QDebug>
#include <QMessageBox>
#include <RdsClient>
#include <QHostAddress>
#include "config.h"
#include "sleeper.h"

#define IP_VALIDATOR new QRegExpValidator(QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$"),this)
#define IP_VALIDATOR_tmp new QRegExpValidator(QRegExp("^(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$"),this)
#define IP_VALIDATOR_STRONG new QRegExpValidator(QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$"),this)

///@todo Test this regex for IP confirmation, it's shorter and simpler...
// (?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)

ProvisionWizard::ProvisionWizard(QString user, QString pass, QString ip, int port, QString service, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::ProvisionWizard()
{
	setupUi(this);
	BannerLabel->setPixmap(QPixmap(findRdsIcon("icons/install-banner.png")));
	BoxLabel->setPixmap(QPixmap(findRdsIcon("icons/install-box.png")));
	setWindowIcon(QPixmap(findRdsIcon("icons/32x32/kcontrol.png")));
	
	_user = user;
	_pass = pass;
	_oldip = ip;
	_port = port;
	_service = service;
	_reverted = false;
	_loadingframe = 0;
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-0.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-1.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-2.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-3.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-4.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-5.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-6.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-7.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-8.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-9.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-10.png"));
	_loadingframes << QPixmap(findRdsIcon("./icons/loading/loading-11.png"));
	QTimer *animtimer = new QTimer(this);
	QObject::connect(animtimer, SIGNAL(timeout()), this, SLOT(onTimerUpdate()));
	animtimer->start(75);

	NextButton->setFocus();
	LogView->setVisible(false);

	_timer = new QTimer(this);
	QObject::connect(_timer, SIGNAL(timeout()), this, SLOT(timeout()));

	QObject::connect(Pages, SIGNAL(currentChanged(int)), this, SLOT(pageChanged(int)));
	Pages->setCurrentIndex(Welcome);

	_provisionator = new RdsProvisionator(NULL);
	QObject::connect(this, SIGNAL(destroyed()), _provisionator, SLOT(deleteLater()));

	ReturnValue ret = rdsClient()->provisionator();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get provisioning service.");
		exit(1);
	}

	*_provisionator = ret;
	QObject::connect(_provisionator, SIGNAL(completed(ReturnValue)), this, SLOT(onCompleted(ReturnValue)));
	QObject::connect(_provisionator, SIGNAL(minorProgress(QString)), this, SLOT(onMinorProgress(QString)));
	QObject::connect(_provisionator, SIGNAL(majorProgress(QString)), this, SLOT(onMajorProgress(QString)));

	IpEdit->setValidator(IP_VALIDATOR);
	NetmaskEdit->setValidator(IP_VALIDATOR);
	GatewayEdit->setValidator(IP_VALIDATOR);
	Dns1Edit->setValidator(IP_VALIDATOR);
	Dns2Edit->setValidator(IP_VALIDATOR);
	DhcpStartEdit->setValidator(IP_VALIDATOR);
	DhcpEndEdit->setValidator(IP_VALIDATOR);
	DhcpNetmaskEdit->setValidator(IP_VALIDATOR);
	HostNameEdit->setValidator(new QRegExpValidator(QRegExp("^[A-Za-z][A-Za-z0-9\\-]{,13}$"), this));
	DomainNameEdit->setValidator(new QRegExpValidator(QRegExp("^[A-Za-z][A-Za-z0-9]*$"), this));
	RealmEdit->setValidator(new QRegExpValidator(QRegExp("^[A-Za-z][A-Za-z0-9.\\-]*$"), this));
}

ProvisionWizard::~ProvisionWizard()
{
}

void ProvisionWizard::setupButtons(QString cancel, QString previous, QString next, bool ce, bool pe, bool ne)
{
	CancelButton->setText(cancel);
	PreviousButton->setText(previous);
	NextButton->setText(next);
	CancelButton->setEnabled(ce);
	PreviousButton->setEnabled(pe);
	NextButton->setEnabled(ne);
}

void ProvisionWizard::setupHeader(QString main, QString desc)
{
	TitleLabel->setText(main);
	//DescriptionLabel->setText(desc);
}

void ProvisionWizard::showBox(bool show)
{
	BoxLabel->setVisible(show);
}

void ProvisionWizard::pageChanged(int page)
{
	ReturnValue ret;
	//qDebug() << "Page" << page;
	switch ((Page) page)
	{
		case Welcome:
			setupButtons("&Cancel", "&Previous", "&Next", true, false, true);
			showBox(true);
			setupHeader("Welcome", "Welcome to the Resara Server setup wizard.");
			break;
		case Network:
			setupButtons("&Cancel", "&Previous", "&Next", true, true, true);
			showBox(false);
			setupHeader("Network Configuration", "Please enter your network settings below.");
			break;
		case NetworkConfig:
			setupButtons("&Cancel", "&Previous", "&Next", true, false, false);
			showBox(false);
			setupHeader("Network Configuration", "Please wait while the network settings are applied.");

			NetworkStatusLabel->setText("Configuring Network");

			_provisionator->setNetworkSettings(NULL , "", _settings);
                        QtRpc::Sleeper::sleep(1);
			rdsClient()->disconnect();

			_ip = _settings["ip"].toString();
			_reverted = false;
			_time.start();
			doConnect();
			break;
		case DateTime:
			setupButtons("&Cancel", "&Previous", "&Next", true, true, true);
			showBox(false);
			setupHeader("Date and Time", "Set the date and time below.");
			break;
		case Domain:
			setupButtons("&Cancel", "&Previous", "&Next", true, true, true);
			showBox(false);
			setupHeader("Domain Details", "Please enter your domain details below.");
			break;
		case Password:
			setupButtons("&Cancel", "&Previous", "&Next", true, true, false);
			showBox(false);
			setupHeader("Administrator Password", "Please enter your administrator password below.");
			on_Password1Edit_textEdited("");
			break;
		case Dhcp:
			setupButtons("&Cancel", "&Previous", "&Next", true, true, true);
			showBox(false);
			setupHeader("DHCP Server", "If you would like to run a DHCP server, configure it now.");
			break;
		case Provisioning:
			setupButtons("&Cancel", "&Previous", "&Next", true, false, false);
			showBox(false);
			setupHeader("Configuring", "Please wait while your settings are applied.");

			_settings["timezone"] = TimeZoneChooser->currentText().replace(" ", "_");
			_settings["datetime"] = _datetime;
			_settings["hostname"] = HostNameEdit->text();
			_settings["domain"] = DomainNameEdit->text().toUpper();
			_settings["realm"] = RealmEdit->text().toUpper();
			_settings["adminpw"] = Password1Edit->text();
			_settings["provisiondhcp"] = DhcpEnabledCheck->isChecked();
			_settings["dhcpstart"] = DhcpStartEdit->text();
			_settings["dhcpend"] = DhcpEndEdit->text();
			_settings["dhcpnetmask"] = DhcpNetmaskEdit->text();

			ret = _provisionator->provision(_settings);
			if (ret.isError()) onCompleted(ret);
			break;
		case Finished:
			setupButtons("&Cancel", "&Previous", "&Finished", false, false, true);
			showBox(true);
			setupHeader("Finished", "Click Finish to exit the setup wizard.");
			break;
		default:
			break;
	}
}

bool ProvisionWizard::isValid(Page page)
{
	switch ((Page) page)
	{
		case Network:
			if (!validTest(IpEdit, "The IP address you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			if (!validTest(NetmaskEdit, "The netmask you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			if (!validTest(GatewayEdit, "The gateway you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			if (Dns2Edit->text() != "")
			{
				if (!validTest(Dns1Edit, "The primary DNS server you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			}
			if (Dns2Edit->text() != "")
			{
				if (!validTest(Dns2Edit, "The secondary DNS server you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			}
			break;
		case Domain:
			if (!validTest(HostNameEdit, "The host name you entered is not valid.")) return(false);
			if (!validTest(DomainNameEdit, "The domain you entered is not valid.")) return(false);
			if (!validTest(RealmEdit, "The realm you entered is not valid.")) return(false);
			break;
		case Password:
			if (Password1Edit->text().size() < 7)
			{
				QMessageBox::warning(this, "Error", "The password must be at least 7 characters long");
				return(false);
			}
			if (Password1Edit->text().toLower().contains("administrator"))
			{
				QMessageBox::warning(this, "Error", "The password must not contain the word \"administrator\".");
				return(false);
			}

			{
				int cat = 0;
				if (Password1Edit->text().contains(QRegExp("[a-z]"))) cat++;
				if (Password1Edit->text().contains(QRegExp("[A-Z]"))) cat++;
				if (Password1Edit->text().contains(QRegExp("[0-9]"))) cat++;
				if (Password1Edit->text().contains(QRegExp("[^A-Za-z0-9]"))) cat++;

				if (cat < 3)
				{
					QMessageBox::warning(this, "Error", QString() + "The password must contain at least 3 of the following categories of characters.\n" +
					                     "Lowercase Letters [a-z]\n" +
					                     "Upercase Letters [A-Z]\n" +
					                     "Numbers [0-9]\n" +
					                     "Non-Aplhanumeric Characters [!#$^@...]");

					return(false);
				}
			}

			return(true);
			break;
		case Dhcp:
		{
			if (!validTest(DhcpStartEdit, "The DHCP start address you entered is not valid.",  IP_VALIDATOR_STRONG)) return(false);
			if (!validTest(DhcpEndEdit, "The DHCP end address you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);
			if (!validTest(DhcpNetmaskEdit, "The DHCP netmask address you entered is not valid.", IP_VALIDATOR_STRONG)) return(false);

			QHostAddress start(DhcpStartEdit->text());
			QHostAddress end(DhcpEndEdit->text());
			if (start.toIPv4Address() > end.toIPv4Address())
			{
				QMessageBox::warning(this, "Error", "The DHCP end address must be lower than the start address.");
				return(false);
			}

			return(true);
		}
		default:
			return(true);
	}

	return(true);
}

bool ProvisionWizard::validTest(QLineEdit *edit, QString error)
{
	QString text = edit->text();
	int size = edit->text().size();
	if ((edit->validator() != NULL) && (edit->validator()->validate(text, size) == QValidator::Acceptable))
		return(true);
	else
	{
		QMessageBox::critical(NULL, "Error", error);
		return(false);
	}
}

bool ProvisionWizard::validTest(QLineEdit *edit, QString error, QRegExpValidator *validator)
{
	QString text = edit->text();
	int size = edit->text().size();
	if ((validator != NULL) && (validator->validate(text, size) == QValidator::Acceptable))
		return(true);
	else
	{
		QMessageBox::critical(NULL, "Error", error);
		return(false);
	}
}

void ProvisionWizard::nextPage(Page page)
{
	switch ((Page) page)
	{
		case Welcome:
			getDefaultSettings();
			Pages->setCurrentIndex(Network);
			break;
		case Network:
			_settings["ip"] = IpEdit->text();
			_settings["netmask"] = NetmaskEdit->text();
			_settings["gateway"] = GatewayEdit->text();
			_settings["interface"] = InterfaceChooser->currentText();

			if (DhcpNetmaskEdit->text() == "")
			{
				DhcpNetmaskEdit->setText(NetmaskEdit->text());
				QHostAddress netmask(NetmaskEdit->text());
				QHostAddress ip(IpEdit->text());
				QHostAddress network(netmask.toIPv4Address() & ip.toIPv4Address());
				QStringList parts = network.toString().split(".");
				if (parts.size() == 4)
				{
					parts[3] = "100";
					DhcpStartEdit->setText(parts.join("."));
					parts[3] = "200";
					DhcpEndEdit->setText(parts.join("."));
				}
			}

			{
				QStringList dns;
				if ((Dns1Edit->text() != "0.0.0.0") && (Dns1Edit->text() != "")) dns << Dns1Edit->text();
				if ((Dns2Edit->text() != "0.0.0.0") && (Dns2Edit->text() != "")) dns << Dns2Edit->text();
				_settings["dns"] = dns;
			}

			if ((_settings["ip"] != _defaultsettings["ip"]) ||
			        (_settings["netmask"] != _defaultsettings["netmask"]) ||
			        (_settings["gateway"] != _defaultsettings["gateway"]) ||
			        (_settings["interface"] != _defaultsettings["interface"]) ||
			        (_settings["dns"] != _defaultsettings["dns"]) ||
			        (!_settings.value("netconfigured").toBool()))
			{
				Pages->setCurrentIndex(NetworkConfig);
			}
			else
			{
				Pages->setCurrentIndex(DateTime);
			}
			break;
		case NetworkConfig:
			Pages->setCurrentIndex(DateTime);
			break;
		case DateTime:
			Pages->setCurrentIndex(Domain);
			break;
		case Domain:
			if(RealmEdit->text().toLower().endsWith(".local"))
			{
				if(QMessageBox::warning(this,"Warning","Your domain ends with .local. This will cause problems with some computers. Are you sure you want to continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::No) break;
			}
			
			if(RealmEdit->text().count('.') == 0)
			{
				QMessageBox::warning(this,"Error","The full domain name must contain at least one period(.). For example: " + RealmEdit->text() + ".LAN");
				break;
			}
			
			if(RealmEdit->text().startsWith(".") || RealmEdit->text().endsWith(".") || RealmEdit->text().contains(".."))
			{
				QMessageBox::warning(this,"Error","The full domain name must not begin or end with a period(.), and must not contain more than one period(.) in a row.");
				break;
			}
			
			Pages->setCurrentIndex(Password);
			break;
		case Password:
			Pages->setCurrentIndex(Dhcp);
			break;
		case Dhcp:
			Pages->setCurrentIndex(Provisioning);
			break;
		case Provisioning:
			Pages->setCurrentIndex(Finished);
			break;
		case Finished:
			_provisionator->installFinished(NULL,"");
			accept();
			break;
		default:
			break;
	}
}

void ProvisionWizard::previousPage(Page page)
{
	switch ((Page) page)
	{
		case Welcome:
			break;
		case Network:
			Pages->setCurrentIndex(Welcome);
			break;
		case NetworkConfig:
			Pages->setCurrentIndex(NetworkConfig);
			break;
		case DateTime:
			Pages->setCurrentIndex(Network);
			break;
		case Domain:
			Pages->setCurrentIndex(DateTime);
			break;
		case Password:
			Pages->setCurrentIndex(Domain);
			break;
		case Dhcp:
			Pages->setCurrentIndex(Password);
			break;
		case Provisioning:
			Pages->setCurrentIndex(Dhcp);
			break;
		case Finished:
			Pages->setCurrentIndex(Dhcp);
			break;
		default:
			break;
	}
}

void ProvisionWizard::cancel(Page page)
{
	Q_UNUSED(page);
	exit(1);
}


void ProvisionWizard::on_CancelButton_clicked()
{
	cancel((Page) Pages->currentIndex());
}

void ProvisionWizard::on_PreviousButton_clicked()
{
	previousPage((Page) Pages->currentIndex());
}

void ProvisionWizard::on_NextButton_clicked()
{
	if (isValid((Page)Pages->currentIndex()))
		nextPage((Page) Pages->currentIndex());
}

void ProvisionWizard::timeout()
{
	qWarning() << "  * Connection timed out.";
	doConnect();
}

void ProvisionWizard::connected(uint, ReturnValue ret)
{
	_timer->stop();

	if (ret.isError())
	{
		qCritical() << "  * Failed to connect:" << ret;
		doConnect();
	}
	else
	{
		qDebug() << " * Connected to:" << _ip;
		NetworkDetailsLabel->setText("Connected To: " + _ip);
		if (_reverted)
		{
			qDebug();
			qCritical() << "Failed to connect to the server!";
			qCritical() << "The server has reverted to its old network settings.";
			QMessageBox::critical(this, "Error", "Failed to connect to the server! The server has reverted to its old network settings.");
			Pages->setCurrentIndex(Network);
			return;
		}

		ret = rdsClient()->provisionator();
		if (ret.isError())
		{
			qDebug() << "Failed to get provision service:" << ret.errNumber() << ret.errString();
			QMessageBox::critical(this, "Error", QString("Failed to get provision service: %1, %2").arg(ret.errNumber()).arg(ret.errString()));
			Pages->setCurrentIndex(Network);
			return;
		}

		*_provisionator = ret;

		ret = _provisionator->networkWorked();
		if (ret.isError())
		{
			qDebug() << "Failed to confirm network settings:" << ret.errNumber() << ret.errString();
			QMessageBox::critical(this, "Error", QString("Failed to confirm network settings: %1, %2").arg(ret.errNumber()).arg(ret.errString()));
			Pages->setCurrentIndex(Network);
			return;
		}

		Pages->setCurrentIndex(DateTime);
	}
}

void ProvisionWizard::doConnect()
{
	if (_reverted)
	{
		if (_time.elapsed() >= 30000)
		{
			qDebug();
			qCritical() << "Failed to connect to the server!";
			qCritical() << "The server tried to revert to its old network settings, but has failed.";
			QMessageBox::critical(this, "Error", "Failed to connect to the server! The server tried to revert to its old network settings, but has failed.");
			Pages->setCurrentIndex(Network);
			return;
		}
	}
	else
	{
		if (_time.elapsed() >= 70000)
		{
			NetworkStatusLabel->setText("Reverting Network Settings");
			qDebug() << " - Failed to connect to the server within 60 seconds.";
			qDebug() << " - Trying the old IP address.";
			_ip = _oldip;
			_reverted = true;
			_time.restart();
		}
	}

	NetworkDetailsLabel->setText("Connecting To: " + _ip);
	rdsClient()->connect(QString("tcp://%1:%2/%3").arg(_ip).arg(_port).arg(_service),
			     this, SLOT(connected(uint, ReturnValue)),
			     QtRpc::AuthToken(_user, _pass));
	_timer->start(10000);
}

void ProvisionWizard::on_DomainNameEdit_textEdited(QString text)
{
	DomainNameEdit->setText(text.toUpper());

	if ((_lasttext == "") && (RealmEdit->text() == ""))
	{
		RealmEdit->setText(text.toUpper() + ".LAN");
	}
	else if (RealmEdit->text().startsWith(_lasttext))
	{
		QString tmp = RealmEdit->text().mid(_lasttext.size());
		RealmEdit->setText(text.toUpper() + tmp);
	}

	_lasttext = text.toUpper();
}

void ProvisionWizard::on_RealmEdit_textEdited(QString text)
{
	RealmEdit->setText(text.toUpper());
}

void ProvisionWizard::on_Password1Edit_textEdited(QString text)
{
	Q_UNUSED(text);
	if (Password2Edit->text() != "")
	{
		if (Password1Edit->text() == Password2Edit->text())
		{
			setupButtons("&Cancel", "&Previous", "&Next", true, true, true);
			PasswordStatusLabel->setText("");
		}
		else
		{
			setupButtons("&Cancel", "&Previous", "&Next", true, true, false);
			PasswordStatusLabel->setText("<b><font color='red'>Passwords do not match!</font></b>");
		}
	}
	else
	{
		setupButtons("&Cancel", "&Previous", "&Next", true, true, false);
		PasswordStatusLabel->setText("");
	}

	int points = 0;
	for (int i = 0; i < Password1Edit->text().length(); i++)
	{
		QChar c = Password1Edit->text()[i];
		if (c.isLetterOrNumber())
		{
			if (c.isUpper())
				points += 2;
			else
				points += 1;
		}
		else
		{
			points += 3;
		}
	}

	PasswordStrengthProgress->setValue(points);
}

void ProvisionWizard::on_Password2Edit_textEdited(QString text)
{
	on_Password1Edit_textEdited(text);
}

void ProvisionWizard::onMinorProgress(const QString &msg)
{
	LogView->append("  * " + msg);
}

void ProvisionWizard::onMajorProgress(const QString &msg)
{
	LogView->append(" - " + msg);
	ProvisionStatusLabel->setText(msg);
}

void ProvisionWizard::onCompleted(const ReturnValue &ret)
{
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to provision:" + ret.errString());
		setupButtons("&Cancel", "&Previous", "&Next", true, true, false);
	}
	else
	{
		Pages->setCurrentIndex(Finished);
	}
}

void ProvisionWizard::onTimerUpdate()
{
	if (_loadingframe == 11) _loadingframe = 0;
	else _loadingframe++;

	AnimLabel1->setPixmap(_loadingframes[_loadingframe]);
	AnimLabel2->setPixmap(_loadingframes[_loadingframe]);
}

void ProvisionWizard::onDateTimeTimer()
{
	_datetime = _datetime.addSecs(1);
	DateEdit->setDate(_datetime.date());
	TimeEdit->setTime(_datetime.time());
}

void ProvisionWizard::on_DateEdit_dateChanged(QDate date)
{
	_datetime.setDate(date);
}

void ProvisionWizard::on_TimeEdit_timeChanged(QTime time)
{
	_datetime.setTime(time);
}

void ProvisionWizard::getDefaultSettings()
{
	ReturnValue ret = _provisionator->getDefaultSettings();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get default settings.");
		exit(1);
	}
	
	_settings = ret.toMap();
	_defaultsettings = ret.toMap();
	
	IpEdit->setText(_settings.value("ip").toString());
	NetmaskEdit->setText(_settings.value("netmask").toString());
	GatewayEdit->setText(_settings.value("gateway").toString());
	
	QStringList dns = _settings.value("dns").toStringList();
	if (dns.size() >= 1) Dns1Edit->setText(dns[0]);
	else Dns1Edit->setText("0.0.0.0");
	if (dns.size() >= 2) Dns2Edit->setText(dns[1]);
	else Dns2Edit->setText("0.0.0.0");
	
	QStringList interfaces = _settings.value("interfaces").toStringList();
	QString interface = _settings.value("interface").toString();
	
	if (interface == "")
		interface = interfaces.first();
	else
	{
		if (!interfaces.contains(interface)) interfaces << interface;
	}
	
	InterfaceChooser->addItems(interfaces);
	InterfaceChooser->setCurrentIndex(InterfaceChooser->findText(interface));
	
	HostNameEdit->setText(_settings.value("hostname").toString());
	DomainNameEdit->setText(_settings.value("domain").toString().toUpper());
	RealmEdit->setText(_settings.value("realm").toString().toUpper());
	_lasttext = _settings.value("domain").toString().toUpper();
	
	DhcpEnabledCheck->setChecked(_settings.value("provisiondhcp").toBool());
	DhcpStartEdit->setText(_settings.value("dhcpstart").toString());
	DhcpEndEdit->setText(_settings.value("dhcpend").toString());
	DhcpNetmaskEdit->setText(_settings.value("dhcpnetmask").toString());
	
	_datetime = _settings.value("datetime").toDateTime();
	QTimer *datetimer = new QTimer(this);
	QObject::connect(datetimer, SIGNAL(timeout()), this, SLOT(onDateTimeTimer()));
	datetimer->start(1000);
	
	QString timezone = _settings.value("timezone").toString();
	if (timezone == "") timezone = "America/New_York";
	QStringList tmp = _settings.value("timezones").toStringList();
	if (!tmp.contains(timezone)) tmp << timezone;
	QStringList zones;
	foreach(QString zone, tmp)
	{
		zones << zone.replace("_", " ");
	}
	zones.sort();
	timezone = timezone.replace("_", " ");
	TimeZoneChooser->addItems(zones);
	TimeZoneChooser->setCurrentIndex(TimeZoneChooser->findText(timezone));
}

