/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "selectiondialog.h"
#include <RdsClient>
#include <QMessageBox>
#include <RdsEntitySortModel>
#include <QDebug>
#include <QSettings>
#include <QMenu>
#include "config.h"

extern QStringList __systemids;

SelectionDialog::SelectionDialog(SelectionTypes type, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::SelectionDialog()
{
	setupUi(this);
	ClearButton->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	_type = type;

	ReturnValue ret = rdsClient()->getService("UserGroupComputerManager");
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get manager", ret.errString());
		return;
	}

	_manager = new RdsEntityManager(this);
	*_manager = ret;
	_model = new RdsUserGroupComputerModel(_manager, this);
	_sort = new RdsUserGroupComputerSortModel(_model);
	
	if((_type & User) == User) 
		_sort->showUsers(true);
	else
		_sort->showUsers(false);
	
	if((_type & Group) == Group)
		_sort->showGroups(true);
	else
		_sort->showGroups(false);
	
	if((_type & Computer) == Computer)
		_sort->showComputers(true);
	else
		_sort->showComputers(false);
	
	if((_type & Special) == Special)
		_sort->showSpecial(true);
	else
		_sort->showSpecial(false);
	
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	_selection = View->selectionModel();
	_model->setupSelection(_sort,View);
	View->sortByColumn(0, Qt::AscendingOrder);
	_model->setConstrainSelection(false);
	
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(SearchEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));
	
	View->header()->resizeSection(0,150);
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));
	_showsystemaction = new QAction("Show System Objects", this);
	QObject::connect(_showsystemaction, SIGNAL(toggled(bool)), this, SLOT(showSystemObjects(bool)));
	_showsystemaction->setCheckable(true);
	
	QSettings settings;
	if (settings.value("Users/ShowSystemsObjects").toBool())
	{
		_showsystemaction->setChecked(true);
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_showsystemaction->setChecked(false);
		_sort->setHiddenEntities(__systemids);
	}
}

SelectionDialog::~SelectionDialog()
{
}

void SelectionDialog::accept()
{
	if(_entities.size() == 0)
	{
		QMessageBox::critical(this, "Error", "Please make a selection");
		return;
	}
	
	QDialog::accept();
}

void SelectionDialog::setAllowMultipleSelections(bool allow)
{
	if(allow)
		View->setSelectionMode(QAbstractItemView::ExtendedSelection);
	else
		View->setSelectionMode(QAbstractItemView::SingleSelection);
}

QString SelectionDialog::selectedItem()
{
	if(_entities.size() == 0) return("");
	return(_entities.first());
}

QStringList SelectionDialog::selectedItems()
{
	return(_entities);
}

void SelectionDialog::outputsChanged(QStringList list)
{
	_entities.clear();
	
	foreach(QString entity, list)
	{
		RdsEntityModel::Cache *cache = _model->getCache(entity);
		if(cache == NULL) continue;
		
		if(((_type & User) == User) && (cache->type() == "user")) _entities << entity;
		if(((_type & Group) == Group) && (cache->type() == "group")) _entities << entity;
		if(((_type & Computer) == Computer) && (cache->type() == "computer")) _entities << entity;
		if(((_type & Special) == Special) && (cache->type() == "special")) _entities << entity;
		
		
	}
}

void SelectionDialog::contextMenu(const QPoint &)
{
	QMenu menu;
	menu.addAction(_showsystemaction);

	menu.exec(QCursor::pos());
}

void SelectionDialog::showSystemObjects(bool show)
{
	if (show)
	{
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_sort->setHiddenEntities(__systemids);
	}
}

void SelectionDialog::AddWidget(QWidget* w)
{
	MainLayout->insertWidget(2,w);
}


