/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/

#include "logmanager.h"
#include "logmanager_p.h"
#include <iostream>
#include <QFile>
#include <QDebug>
#include <QCoreApplication>
#include <QStringList>
#include <syslog.h>

#include "config.h"

using namespace std;

LogManager *_instance;

LogManager::LogManager(const char *name)
{
	Q_UNUSED(name);
	QXT_INIT_PRIVATE(LogManager);

	bool debug = false;
	if (QCoreApplication::arguments().contains("--debug"))
		debug = true;

	_instance = this;
	qxt_d().logs["stdout"] = new LogManagerPrivate::LogStdOut();
	qxt_d().logs["stderr"] = new LogManagerPrivate::LogStdErr();
	qxt_d().logs["syslog"] = new LogManagerPrivate::LogSyslog(name);
	setMap("stdout", "", Debug | Info);
	setMap("stderr", "", Warning | Error | Critical);
	if (debug)
		setMap("syslog", "", Debug | Info | Warning | Error | Critical);
	else
		setMap("syslog", "", Info | Warning | Error | Critical);

	qInstallMsgHandler(messageHandler);
}


LogManager::~LogManager()
{
}

void LogManagerPrivate::LogStdOut::writeLog(LogManager::LogPriority priority, QString section, QString text)
{
	QString msg = "";

	switch (priority)
	{
		case LogManager::Critical:
			msg = "CRITICAL ERROR: ";
			break;
		case LogManager::Error:
			msg = "Error: ";
			break;
		case LogManager::Warning:
			msg = "Warning: ";
			break;
		case LogManager::Info:
		case LogManager::Debug:
			break;
		default:
			break;
	}

	if (section != "") msg += section + ": ";
	msg += text;

	cout << msg.toAscii().data() << endl;
}

void LogManagerPrivate::LogStdErr::writeLog(LogManager::LogPriority priority, QString section, QString text)
{
	QString msg = "";

	switch (priority)
	{
		case LogManager::Critical:
			msg = "CRITICAL ERROR: ";
			break;
		case LogManager::Error:
			msg = "Error: ";
			break;
		case LogManager::Warning:
			msg = "Warning: ";
			break;
		case LogManager::Info:
		case LogManager::Debug:
			break;
		default:
			break;
	}

	if (section != "") msg += section + ": ";
	msg += text;

	cerr << msg.toAscii().data() << endl;
}

LogManagerPrivate::LogFile::LogFile(QString file, bool append)
{
	filename = file;
	if (!append) QFile::remove(file);
}

void LogManagerPrivate::LogFile::writeLog(LogManager::LogPriority priority, QString section, QString text)
{
	QString msg = "";

	switch (priority)
	{
		case LogManager::Critical:
			msg = "CRITICAL ERROR: ";
			break;
		case LogManager::Error:
			msg = "Error: ";
			break;
		case LogManager::Warning:
			msg = "Warning: ";
			break;
		case LogManager::Info:
		case LogManager::Debug:
			break;
		default:
			break;
	}

	if (section != "") msg += section + ": ";
	msg += text + "\n";

	QFile file(filename);
	if (!file.open(QFile::WriteOnly | QFile::Append)) return;

	file.write(msg.toAscii());
	file.close();
}

LogManagerPrivate::LogSyslog::LogSyslog(const char *name)
{
	openlog(name, 0, LOG_USER);
}

LogManagerPrivate::LogSyslog::~LogSyslog()
{
	closelog();
}

void LogManagerPrivate::LogSyslog::writeLog(LogManager::LogPriority priority, QString section, QString text)
{
	QString msg = "";
	int p = 0;

	switch (priority)
	{
		case LogManager::Critical:
			p = LOG_CRIT;
			break;
		case LogManager::Error:
			p = LOG_ERR;
			break;
		case LogManager::Warning:
			p = LOG_WARNING;
			break;
		case LogManager::Info:
			p = LOG_INFO;
		case LogManager::Debug:
			p = LOG_DEBUG;
			break;
		default:
			break;
	}

	if (section != "") msg += section + ": ";
	msg += text;

	///TOTO possable security problem here!
	syslog(p, qPrintable(msg));
}

void LogManager::addFile(QString name, bool append, int p, QString s)
{
	if (qxt_d().logs.keys().contains(name)) delete qxt_d().logs[name];
	qxt_d().logs[name] = new LogManagerPrivate::LogFile(name, append);
	setMap(name, s, p);
}

void LogManager::removeFile(QString name)
{
	if (qxt_d().logs.keys().contains(name)) delete qxt_d().logs[name];
	qxt_d().logs.remove(name);
	qxt_d().maps.remove(name);
}

void LogManager::writeLog(LogPriority p, QString section, QString text)
{
	qxt_d().logmutex.lock();
	foreach(QString key, qxt_d().logs.keys())
	{
		LogDest *dest = qxt_d().logs[key];
		if (qxt_d().maps[key].section != "" && qxt_d().maps[key].section != section) continue;
		if ((qxt_d().maps[key].priority & p) != p) continue;
		dest->writeLog(p, section, text);
	}
	qxt_d().logmutex.unlock();
}

LogManager * LogManager::instance()
{
	return(_instance);
}

void LogManager::setMap(QString dest, QString section, int priority)
{
	LogManagerPrivate::LogMap map(section, (LogPriority)priority);
	qxt_d().maps[dest] = map;
}

void LogManager::removeMap(QString dest, QString section, int priority)
{
	LogManagerPrivate::LogMap map(section, (LogPriority)priority);
	qxt_d().maps.remove(dest);
}

void LogManager::messageHandler(QtMsgType type, const char *msg)
{
	switch (type)
	{
		case QtDebugMsg:
			instance()->writeLog(LogManager::Debug, "", msg);
			break;
		case QtWarningMsg:
			instance()->writeLog(LogManager::Warning, "", msg);
			break;
		case QtCriticalMsg:
			instance()->writeLog(LogManager::Error, "", msg);
			break;
		case QtFatalMsg:
			instance()->writeLog(LogManager::Critical, "", msg);
			abort();
	}

}

LogWriter::LogWriter(LogManager::LogPriority p, QString s) : QDebug(&_data)
{
	QXT_INIT_PRIVATE(LogWriter);
	qxt_d().priority = p;
	qxt_d().section = s;
}

LogWriter::LogWriter(const LogWriter& other) : QDebug(&_data)
{
	QXT_INIT_PRIVATE(LogWriter);
	qxt_d().priority = other.qxt_d().priority;
	qxt_d().section = other.qxt_d().section;
}

// Private
LogWriter::LogWriter() : QDebug(&_data) { }

LogWriter::~LogWriter()
{
	LogManager::instance()->writeLog(qxt_d().priority, qxt_d().section, _data);
}

