/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsnetmonitorthread.h"
#include <QTimer>
#include <RdsNetworkSettings>
#include <QDebug>
#include <QStringList>
#include <QVariant>
#include <QMutexLocker>
#include <QFile>
#include "rdsnetworksettingsthread.h"

RdsNetMonitorThread::RdsNetMonitorThread(QString interface, QString ip, QString netmask, QString gateway, QStringList dns,
        QString oldip, QString oldnetmask, QString oldgateway, QStringList olddns, bool updatehosts, QObject *parent)
		: QThread(parent),
		_interface(interface),
		_ip(ip),
		_netmask(netmask),
		_gateway(gateway),
		_dns(dns),
		_oldip(oldip),
		_oldnetmask(oldnetmask),
		_oldgateway(oldgateway),
		_olddns(olddns),
		_canceled(false),
		_updatehosts(updatehosts)
{
	QObject::connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));
}


RdsNetMonitorThread::~RdsNetMonitorThread()
{
	if (isRunning())
	{
		qCritical() << "RdsNetMonitorThread is still running, but object is being destroyed!";
	}
}

void RdsNetMonitorThread::run()
{
	RdsNetworkSettings net;
	ReturnValue ret = net.loadNetworkConfig();
	if (ret.isError())
	{
		qCritical() << "Failed to load network settings:" << ret.errString();
		return;
	}

	ret = net.listAllInterfaces();
	if (ret.isError())
	{
		qCritical() << "Failed to list interfaces:" << ret.errString();
		return;
	}

	if (!ret.toStringList().contains(_interface))
	{
		qCritical() << "Invalid Interface " + _interface;
		return;
	}

	QMap<QString, QString> options;
	options["address"] = _ip;
	options["netmask"] = _netmask;
	options["gateway"] = _gateway;

	ret = net.setInterface(_interface, "static", true, options, "#Configured by RDS");
	if (ret.isError())
	{
		qCritical() << "Failed to set interface settings:" << ret.errString();
		return;
	}

	ret = net.saveNetworkConfig();
	if (ret.isError())
	{
		qCritical() << "Failed to save network settings:" << ret.errString();
		return;
	}

	if (_updatehosts)
	{
		QVariantMap settings;
		settings["ip"] = _ip;
		ret = RdsNetworkSettingsThread::updateHostsFile(settings);
		if (ret.isError())
		{
			qCritical() << "Failed to update hosts file:" << ret.errString();
			return;
		}
	}

	net.ifDown(_interface);
	net.killDhcpClient();
	ret = net.ifUp(_interface);
	if (ret.isError())
	{
		qCritical() << "Failed to bring up the interface:" << ret.errString();
	}

	QFile resolv("/etc/resolv.conf");
	if (!resolv.open(QFile::WriteOnly | QFile::Truncate))
	{
		qWarning() << "Failed to open /etc/resolv.conf for writing!";
	}
	else
	{
		foreach(QString srv, _dns)
		{
			resolv.write(QString("nameserver %1\n").arg(srv).toAscii());
		}
		resolv.close();
	}

	QTimer::singleShot(60000, this, SLOT(timeout()));
	exec();
	/*
		sleep(60);

		{
			QMutexLocker locker(&_canceledmutex);
			if (_canceled) return;
		}

		qCritical() << "Timed out while waiting for a network connection!";
		qCritical() << "Reverting to the last known good address!";
		net.loadNetworkConfig();
		options.clear();
		options["address"] = _oldip;
		options["netmask"] = _oldnetmask;
		options["gateway"] = _oldgateway;
		net.setInterface(_interface, "static", true, options, "#Reset to DHCP by RDS");
		net.saveNetworkConfig();
		net.ifDown(_interface);
		net.killDhcpClient();
		net.ifUp(_interface);
		*/
}

void RdsNetMonitorThread::timeout()
{
	{
		QMutexLocker locker(&_canceledmutex);
		if (_canceled)
		{
			qCritical() << "The thread was canceled but my timeout still ran!";
			return;
		}
	}
	qCritical() << "Timed out while waiting for a network connection!";
	qCritical() << "Reverting to the last known good address!";
	RdsNetworkSettings net;

	QMap<QString, QString> options;
	options["address"] = _oldip;
	options["netmask"] = _oldnetmask;
	options["gateway"] = _oldgateway;

	ReturnValue ret = net.loadNetworkConfig();
	if (!ret.isError())
		ret = net.setInterface(_interface, "static", true, options, "#Reset to DHCP by RDS");
	if (!ret.isError())
		ret = net.saveNetworkConfig();
	if (!ret.isError())
		ret = net.ifDown(_interface);
	if (!ret.isError())
		ret = net.killDhcpClient();
	if (!ret.isError())
		ret = net.ifUp(_interface);
	if (_updatehosts && !ret.isError())
	{
		QVariantMap settings;
		settings["ip"] = _oldip;
		ret = RdsNetworkSettingsThread::updateHostsFile(settings);
	}
	if (ret.isError())
	{
		qCritical() << "Error while reverting network settings!" << ret;
	}

	QFile resolv("/etc/resolv.conf");
	if (!resolv.open(QFile::WriteOnly | QFile::Truncate))
	{
		qWarning() << "Failed to open /etc/resolv.conf for writing!";
	}
	else
	{
		foreach(QString srv, _olddns)
		{
			resolv.write(QString("nameserver %1\n").arg(srv).toAscii());
		}
		resolv.close();
	}
	quit();
//	wait();
//	deleteLater();
}

void RdsNetMonitorThread::cancel()
{
	QMutexLocker lock(&_canceledmutex);
	_canceled = true;
	quit();
//	deleteLater();
}
