/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSDHCPMANAGER_H
#define RDSDHCPMANAGER_H

#include "RdsEntityManager"
#include "RdsDaemonInstance"
#include <QxtPimpl>
#include <QStringList>
#include <QHostAddress>

class RdsDhcpManagerPrivate;

/**
	@author Chris Vickery <chris@resara.com>
*/
class RdsDhcpManager : public RdsEntityManager, public RdsDaemonInstance
{
	Q_OBJECT;
	QXT_DECLARE_PRIVATE(RdsDhcpManager);
	QTRPC_SERVICEPROXY_PIMPL(RdsDhcpManager);
public:
	RdsDhcpManager();
	RdsDhcpManager(QObject *parent);
	RdsDhcpManager(const RdsDhcpManager& other);
	~RdsDhcpManager();

	/**
	 *        Parses the data and stores configuration information
	 * @param data the config file to parse
	 * @return true
	 */
	bool parse(const QString &data);
	/**
	 *        Gets the general dhcp config values in a print friendly QString
	 * @return QString containing the values in the dhcp config
	 */
	QString toString() const;
	virtual ReturnValue listEntities(const QString &base, bool loadmore) const;
	ReturnValue auth(QtRpc::AuthToken token);

	RdsDhcpManager& operator=(const RdsDhcpManager& other);

	//DHCP sservice functions
	virtual ReturnValue stopService();
	virtual ReturnValue startService();
	virtual ReturnValue restartService();
	virtual ReturnValue reloadService();
	/**
	 *        Reloads the config
	 * @return true on success, error on failure
	 */
	virtual ReturnValue reloadConfig();

public slots:
	/**
	 *        Saves the current configuration
	 * @return true on success, error on failure
	 */
	ReturnValue save() const;
	/**
	 *        Re-parses the configuration file
	 * @return true on success, error on failure
	 */
	ReturnValue reload();
	/**
	 *        Returns the configuration values
	 * @return RdsDhcpValues
	 */
	ReturnValue values() const;
	/**
	 *        Returns the configuration values from a given path
	 * @param path ID from listEntities
	 * @return RdsDhcpValues
	 */
	ReturnValue values(const QString& path) const;
	/**
	 *        Moved given values object to new location
	 * @param path ID from listEntities
	 * @param newpath new ID, formatted the same way as listEntities. Parent must already exist.
	 * @return RdsDhcpValues new moved object
	 */
	ReturnValue move(const QString& path, const QString& newpath);

	/**
	*        List interfaces the DHCP server is listening on. No interfaces indicated that the server is listening on all interfaces
	* @return QStringList of interfaces, or non for all interfaces
	*/
	ReturnValue interfaces();

	/**
	*        Sets the list of interfaces for the DHCP server to listen on. A blank list indicates that it should listen on all interfaces
	* @param interfaces a list of interfaces to listen on
	* @return true on success, and error on failure
	*/
	ReturnValue setInterfaces(QStringList interfaces);

	/**
	*        Returns if the DHCP server is enabled
	* @return true/false if the server is enabled/disabled, or an error
	*/
	ReturnValue enabled();

	/**
	*        Sets weather or no the DHCP server is enabled
	* @param enabled weather or not to enable the DHCP server
	* @return true on success, and error on failure
	*/
	ReturnValue setEnabled(bool enabled);

	/**
	*        Indicates if the DHCP server has been configured
	* @return true if the server has been configured, false otherwise
	*/
	ReturnValue configured();
	
	/**
	*       Checks to see if an existing zone conflicts with the provided range
	* @param id The id of the subnet owns the provided range, blank if it is a new subnet
	* @param start The start address to check
	* @param end The end address to check
	* @return true if there is no conflict, false otherwise
	*/
	ReturnValue checkRange(const QString &id, const QHostAddress &start, const QHostAddress &end);
	
	/**
	*        Returns a list of all the groups
	* @return A QStringList of entity IDs
	*/
	ReturnValue listAllGroups(QString baseid = "");
	
	/**
	*        Returns a list of all the shared networks
	* @return A QStringList of entity IDs
	*/
	ReturnValue listAllSharedNetworks(QString baseid = "");
	
	/**
	*        Returns a list of all the subnets
	* @return A QStringList of entity IDs
	*/
	ReturnValue listAllSubnets(QString baseid = "");
	
	/**
	*        Returns a list of all the hosts
	* @return A QStringList of entity IDs
	*/
	ReturnValue listAllHosts(QString baseid = "");
};

#endif
