/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  backend-db31.c defines the DocBook V.3.1 output backend of refdbd
  markus@mhoenicka.de 6-22-00

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


#include <stdio.h>
#include <string.h>
#include <syslog.h> /* for definitions of log message priorities */
#include <dbi/dbi.h>

#include "refdb.h"
#include "backend.h"
#include "backend-db31.h"
#include "strfncs.h"
#include "linklist.h"
#include "tokenize.h"
#include "dbfncs.h"
#include "risdb.h"
#include "authorinfo.h"
#include "connect.h" /* for get_status_msg() */
#include "refdbd.h"
#include "xmlhelper.h"

/* some globals */
extern int n_log_level; /* numeric version of log_level */

/* forward declarations of local functions */
static char* add_author_db(struct renderinfo* ptr_rendinfo, struct AUTHOR_INFO* ptr_ainfo, char** ptr_chunk_buf, size_t* ptr_chunk_buf_len, struct xmlindent* ptr_indent, int n_ref_format);
static char* add_author_info_db(struct renderinfo* ptr_rendinfo, int author_type, int n_refdb_id, char** ptr_chunk_buf, size_t* ptr_chunk_buf_len, struct xmlindent* ptr_indent, int n_ref_format);
static char* add_analytic_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format);
static char* add_monogr_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format);
static char* add_series_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format);
static char* add_notes_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, struct xmlindent* ptr_indent);
static char* add_abstract_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, struct xmlindent* ptr_indent);
static char* add_biblioids_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent);
static int indent_notbelow_db(const char* name);
static int is_entry_db(const char* name);

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  prepare_render_db31(): writes a header for the docbook output of a
                         query

  int prepare_render_db31 returns 0 if successful,>0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int prepare_render_db31(struct renderinfo* ptr_rendinfo) {
  /* ToDo: control with format_string whether or not to include the document type declaration. For the time being the declaration does go in but is commented out */

  if (ptr_rendinfo->ref_format == REFDOCBK) { /* SGML */
    strcpy(*(ptr_rendinfo->ptr_ref), "<!-- <!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook V3.1//EN\"> -->\n<bibliography>\n<title>Bibliography</title>\n");
  }
  else if (ptr_rendinfo->ref_format == REFDOCBKX) { /* XML */
    char encodingstring[64] = ""; /* todo: is this enough? */
    if (*((ptr_rendinfo->ptr_biblio_info)->encoding)) {
      sprintf(encodingstring, " encoding=\"%s\"", (ptr_rendinfo->ptr_biblio_info)->encoding);
    }

    if (*(ptr_rendinfo->ptr_clrequest->namespace)) {
      sprintf(*(ptr_rendinfo->ptr_ref), "<?xml version=\"1.0\"%s?>\n<%s:bibliography xmlns=\"http://docbook.org/ns/docbook\">\n  <%s:title>Bibliography</%s:title>\n", encodingstring, ptr_rendinfo->ptr_clrequest->namespace, ptr_rendinfo->ptr_clrequest->namespace, ptr_rendinfo->ptr_clrequest->namespace);
    }
    else {
      sprintf(*(ptr_rendinfo->ptr_ref), "<?xml version=\"1.0\"%s?><!DOCTYPE bibliography PUBLIC \"-//OASIS//DTD DocBook XML V4.1.2//EN\" \"http://www.oasis-open.org/docbook/xml/4.1.2/docbookx.dtd\"> -->\n<bibliography>\n<title>Bibliography</title>\n", encodingstring);
    }
  }
  else {
    /* we have a problem */
    LOG_PRINT(LOG_WARNING, get_status_msg(302));
    return 302;
  }

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  finish_render_db31(): writes a footer for the docbook output of a query

  int finish_render_ris returns 0 if successful, >0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int finish_render_db31(struct renderinfo* ptr_rendinfo) {
  char* new_ref;
  char ns[PREFS_BUF_LEN+32];

  if (*(ptr_rendinfo->ptr_clrequest->namespace)) {
    sprintf(ns, "\n</%s:bibliography>\n", ptr_rendinfo->ptr_clrequest->namespace);
  }
  else {
    strcpy(ns, "\n</bibliography>\n");
  }

  if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), ns, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return 801;
  }
  
  *(ptr_rendinfo->ptr_ref) = new_ref;

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  render_db31() renders a RIS dataset for DocBook V.3.1 export

  int render_db31 returns 0 if successful, >0 if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int render_db31(struct renderinfo* ptr_rendinfo) {
  char *new_ref; /* used for reallocating buffer ref */
  char *chunk;
  char *chunk_buf;
  const char *item; /* pointer to string for all db requests */
  const char *type;
  size_t chunk_buf_len;
  unsigned long long n_refdb_id;
  dbi_conn conn;
  struct xmlindent xindent;

  initialize_xmlindent(&xindent, 2, indent_notbelow_db, is_entry_db);

  /* rendering bibliographic information takes up to three levels of
     information:
     analytic: author and title of the article or chapter
     monographic: name of the journal/book, vol/issue/page info, editors
     series: series title, series editors
  */

  chunk_buf_len = 1024;

  if ((chunk_buf = malloc(chunk_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return 801;
  }

  conn = dbi_result_get_conn(ptr_rendinfo->dbires);

  if ((type = get_refdb_type(ptr_rendinfo->dbires)) == NULL) {
    LOG_PRINT(LOG_WARNING, get_status_msg(234));
    free(chunk_buf);
    return 234;
  }

  n_refdb_id = my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id");

  if ((ptr_rendinfo->ptr_biblio_info)->entry_id != NULL) {
    if (print_elstart_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", "id", (ptr_rendinfo->ptr_biblio_info)->entry_id, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
  }
  else {
    if (print_elstart_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
  }

  item = get_refdb_citekey(ptr_rendinfo->dbires);
  if (item && *item) {
    if (print_element_x(item, ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "abbrev", NULL, NULL, NULL, NULL, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
  }

  if (has_part_data(type)) {
    chunk = add_analytic_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  chunk = add_monogr_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX);

  if (chunk && *chunk) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  if (has_set_data(type)) {
    chunk = add_series_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /* add notes and abstract if requested */

  /* notes */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "N1") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
    chunk = add_notes_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, &xindent);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /*----------------------------------------------------------------*/
  /* isbn/issn */
  if (has_periodical_data(type)) {
    chunk = add_biblioids_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, "issn", &xindent);
  }
  else {
    chunk = add_biblioids_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, "isbn", &xindent);
  }

  if (chunk && *chunk) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  /*----------------------------------------------------------------*/
  /* abstract */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "N2") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "AB") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
    chunk = add_abstract_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, &xindent);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /* finish bibliography entry */
  if (print_elend_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    free(chunk_buf);
    return 801;
  }

  free(chunk_buf);

  return 0;
}
 
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  prepare_render_db50(): writes a header for the docbook output of a
                         query

  int prepare_render_db50 returns 0 if successful,>0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int prepare_render_db50(struct renderinfo* ptr_rendinfo) {
  char encodingstring[64] = ""; /* todo: is this enough? */

  if (*((ptr_rendinfo->ptr_biblio_info)->encoding)) {
    sprintf(encodingstring, " encoding=\"%s\"", (ptr_rendinfo->ptr_biblio_info)->encoding);
  }

  if (*(ptr_rendinfo->ptr_clrequest->namespace)) {
    sprintf(*(ptr_rendinfo->ptr_ref), "<?xml version=\"1.0\"%s?>\n<%s:bibliography xmlns:%s=\"http://docbook.org/ns/docbook\">\n<%s:title>Bibliography</%s:title>\n", encodingstring, ptr_rendinfo->ptr_clrequest->namespace, ptr_rendinfo->ptr_clrequest->namespace, ptr_rendinfo->ptr_clrequest->namespace, ptr_rendinfo->ptr_clrequest->namespace);
  }
  else {
    /* always define the default namespace */
    sprintf(*(ptr_rendinfo->ptr_ref), "<?xml version=\"1.0\"%s?>\n<bibliography xmlns=\"http://docbook.org/ns/docbook\">\n<title>Bibliography</title>\n", encodingstring);
  }

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  finish_render_db50(): writes a footer for the docbook output of a query

  int finish_render_ris returns 0 if successful, >0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int finish_render_db50(struct renderinfo* ptr_rendinfo) {
  char* new_ref;

  char ns[PREFS_BUF_LEN+32];

  if (*(ptr_rendinfo->ptr_clrequest->namespace)) {
    sprintf(ns, "\n</%s:bibliography>\n", ptr_rendinfo->ptr_clrequest->namespace);
  }
  else {
    strcpy(ns, "\n</bibliography>\n");
  }

  if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), ns, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return 801;
  }
  
  *(ptr_rendinfo->ptr_ref) = new_ref;

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  render_db50() renders a RIS dataset for DocBook V.5.0 export

  int render_db50 returns 0 if successful, >0 if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int render_db50(struct renderinfo* ptr_rendinfo) {
  char *new_ref; /* used for reallocating buffer ref */
  char *chunk;
  char *chunk_buf;
  const char *item; /* pointer to string for all db requests */
  const char *type;
  size_t chunk_buf_len;
  unsigned long long n_refdb_id;
  dbi_conn conn;
  struct xmlindent xindent;

  /*initialize xindent */
  initialize_xmlindent(&xindent, 2, indent_notbelow_db, is_entry_db);

  /* rendering bibliographic information takes up to three levels of
     information:
     analytic: author and title of the article or chapter
     monographic: name of the journal/book, vol/issue/page info, editors
     series: series title, series editors
  */

  chunk_buf_len = 1024;

  if ((chunk_buf = malloc(chunk_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return 801;
  }

  conn = dbi_result_get_conn(ptr_rendinfo->dbires);

  if ((type = get_refdb_type(ptr_rendinfo->dbires)) == NULL) {
    LOG_PRINT(LOG_WARNING, get_status_msg(234));
    free(chunk_buf);
    return 234;
  }

  n_refdb_id = my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id");


  item = get_refdb_citekey(ptr_rendinfo->dbires);
  if (item && *item) {
    if (print_elstart_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", "xml:id", item, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }

    if (print_element_x(item, ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "abbrev", NULL, NULL, NULL, NULL, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
  }
  else {
    if (print_elstart_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
  }

  if (has_part_data(type)) {
    chunk = add_analytic_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX5);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  chunk = add_monogr_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX5);

  if (chunk && *chunk) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  if (has_set_data(type)) {
    chunk = add_series_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, type, &xindent, REFDOCBKX5);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /* add notes and abstract if requested */

  /* notes */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "N1") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
    chunk = add_notes_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, &xindent);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /*----------------------------------------------------------------*/
  /* isbn/issn */
  if (has_periodical_data(type)) {
    chunk = add_biblioids_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, "issn", &xindent);
  }
  else {
    chunk = add_biblioids_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, "isbn", &xindent);
  }

  if (chunk && *chunk) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  /*----------------------------------------------------------------*/
  /* doi */
  chunk = add_biblioids_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, "doi", &xindent);

  if (chunk && *chunk) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(chunk_buf);
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  /*----------------------------------------------------------------*/
  /* abstract */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "N2") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "AB") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
    chunk = add_abstract_db(ptr_rendinfo, n_refdb_id, &chunk_buf, &chunk_buf_len, &xindent);

    if (chunk && *chunk) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), chunk, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(chunk_buf);
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  /* finish bibliography entry */
  if (print_elend_x(ptr_rendinfo->ptr_ref, ptr_rendinfo->ptr_ref_len, "biblioentry", &xindent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    free(chunk_buf);
    return 801;
  }

  free(chunk_buf);

  return 0;
}
 
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_analytic_db() adds analytic level info to a DocBook bibliography

  static char* add_analytic_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  const char* type ptr to string holding the type of the current dataset

  struct xmlindent* ptr_indent ptr to structure with indentation info

  int n_ref_format reference type (REFDOCBKX|REFDOCBKX5)

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_analytic_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format) {
  int n_have_data = 0; /* set to 1 if we have some data */
  char *new_ref;
  char *entitize_buf;
  char *author_buf;
  char *author_result;
  const char *item;
  size_t author_buf_len = 1024;

  **ptr_chunk_buf = '\0';

  if ((author_buf = malloc(author_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }
    
  /* start analytic information */
  if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", "relation", "analytic", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    free(author_buf);
    return NULL;
  }
    
  /* add authors, if any */
  author_result = add_author_info_db(ptr_rendinfo, 1 /* part */, n_refdb_id, &author_buf, &author_buf_len, ptr_indent, n_ref_format);
  
  if (!author_result) {
    free(author_buf);
    return NULL;
  }
  else if (*author_result) {
    n_have_data++;
    if ((new_ref = mstrcat(*ptr_chunk_buf, author_result, ptr_chunk_buf_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(author_buf);
      return NULL;
    }
    else {
      *ptr_chunk_buf = new_ref;
    }
  }
    
  free(author_buf);
    
  /* get the article or chapter title */
  item = get_refdb_title_copy(ptr_rendinfo->dbires);
    
  if (item != NULL && *item) {
    n_have_data++;
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      free((char*)item);
      return NULL;
    }
      
    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free((char*)item);
      free(entitize_buf);
      return NULL;
    }
      
    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "title", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free((char*)item);
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    free(entitize_buf);
    free((char*)item);
  }

  /* finish analytic information */
  if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }
  
  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_monogr_db() adds monography level info to a DocBook bibliography

  static char* add_monogr_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  const char* type ptr to string holding the type of the current dataset

  struct xmlindent* ptr_indent ptr to structure with indentation info

  int n_ref_format reference type (REFDOCBKX|REFDOCBKX5)

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_monogr_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format) {
  int n_have_data = 0; /* set to 1 if we have some data */
  int i;
  int errcode; /* receives error code for periodical requests */
  char periodical[256];
  char *new_ref;
  char *entitize_buf;
  char *author_buf;
  char *author_result;
  const char *item;
  char journaltype[4][12] = {"name",
			     "custabbrev1",
			     "custabbrev2",
			     "abbrev"};
  size_t author_buf_len;
  dbi_conn conn;

  
  **ptr_chunk_buf = '\0';

  author_buf_len = 1024;

  if ((author_buf = malloc(author_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }
    
  conn = dbi_result_get_conn(ptr_rendinfo->dbires);

  /* start monographic information */
  if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", "relation", "monogr", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    free(author_buf);
    return NULL;
  }

  /* authors and editors */
  author_result = add_author_info_db(ptr_rendinfo, 2, n_refdb_id, &author_buf, &author_buf_len, ptr_indent, n_ref_format);

  if (!author_result) {
    free(author_buf);
    return NULL;
  }
  else if (*author_result) {
    n_have_data++;
    if ((new_ref = mstrcat(*ptr_chunk_buf, author_result, ptr_chunk_buf_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(author_buf);
      return NULL;
    }
    else {
      *ptr_chunk_buf = new_ref;
    }
  }

  free(author_buf);

  if (has_periodical_data(type)) {
    /* journal information, if any */
    for (i = 0; i < 4; i++) {
      /* request journal */
      item = get_periodical(conn, periodical, /*ptr_rendinfo->database*/NULL, i, &errcode, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), NULL /* no frequency required */);
      if (item != NULL) {
	if (!*item) {
	  continue;
	}
	if ((entitize_buf = mstrdup((char*)item)) == NULL) {
	  return NULL;
	}
    
	if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	  free(entitize_buf);
	  return NULL;
	}
	
	if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", "relation", journaltype[i], NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	  LOG_PRINT(LOG_CRIT, get_status_msg(801));
	  return NULL;
	}

	if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "title", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	  free(entitize_buf);
	  LOG_PRINT(LOG_CRIT, get_status_msg(801));
	  return NULL;
	}
	
	free(entitize_buf);

	if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	  LOG_PRINT(LOG_CRIT, get_status_msg(801));
	  return NULL;
	}
	n_have_data++;
      }
    }
  } /* end if has periodical data */

  /* monographic title */
  item = get_refdb_booktitle_copy(ptr_rendinfo->dbires);

  if (item != NULL) {
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      free((char*)item);
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free((char*)item);
      free(entitize_buf);
      return NULL;
    }

    if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", "relation", "booktitle", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "title", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
	
    free(entitize_buf);

    if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    free((char*)item);
    n_have_data++;
  }

  /* publisher and city */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "PB") != NULL
      || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
    if ((item = get_refdb_publisher(ptr_rendinfo->dbires)) != NULL) {
      if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "publisher", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      if ((entitize_buf = mstrdup((char*)item)) == NULL) {
	return NULL;
      }

      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free(entitize_buf);
	return NULL;
      }

      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "publishername", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      free(entitize_buf);

      if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "CY") != NULL
	  || strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ALL") != NULL) {
	/* pubplace - use only if publisher is known */
	if ((item = get_refdb_city(ptr_rendinfo->dbires)) != NULL) {
	  if ((entitize_buf = mstrdup((char*)item)) == NULL) {
	    return NULL;
	  }

	  if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	    free(entitize_buf);
	    return NULL;
	  }

	  if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "address", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	    LOG_PRINT(LOG_CRIT, get_status_msg(801));
	    return NULL;
	  }
	  free(entitize_buf);
	}
      }
      
      if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "publisher", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      n_have_data++;
    }
  }

  /* volume */
  if ((item = get_refdb_volume(ptr_rendinfo->dbires)) != NULL) {
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free(entitize_buf);
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "volumenum", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    free(entitize_buf);
    n_have_data++;
  }

  /* issue */
  if ((item = get_refdb_issue(ptr_rendinfo->dbires)) != NULL) {
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free(entitize_buf);
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "issuenum", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    free(entitize_buf);
    n_have_data++;
  }

  /* start page */
  if ((item = get_refdb_startpage(ptr_rendinfo->dbires)) != NULL) {
    char* entitized_endpage;
    size_t entitize_buf_len;

    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free(entitize_buf);
      return NULL;
    }

    entitize_buf_len = strlen(entitize_buf);

    /* end page */
    if ((item = get_refdb_endpage(ptr_rendinfo->dbires)) != NULL) {
      if ((entitized_endpage = mstrdup((char*)item)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      if (sgml_entitize(&entitized_endpage, NULL) == NULL) {
	free(entitize_buf);
	free(entitized_endpage);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      if ((new_ref = mstrcat(entitize_buf, "-", &entitize_buf_len, 0)) == NULL) {
	free(entitize_buf);
	free(entitized_endpage);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
      else {
	entitize_buf = new_ref;
      }

      if ((new_ref = mstrcat(entitize_buf, entitized_endpage, &entitize_buf_len, 0)) == NULL) {
	free(entitize_buf);
	free(entitized_endpage);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
      else {
	entitize_buf = new_ref;
      }
      free(entitized_endpage);
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "pagenums", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free((char*)item);
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    free(entitize_buf);
    n_have_data++;
  }

  /* publication date, reuse periodical */
  if ((item = get_refdb_pubyear(ptr_rendinfo->dbires, periodical)) != NULL) {
    if (print_element_x(periodical, ptr_chunk_buf, ptr_chunk_buf_len, "pubdate", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free((char*)item);
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    n_have_data++;
  }

  /* finish monographic information */
  if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }

  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_series_db() adds series level info to a DocBook bibliography

  static char* add_series_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  const char* type ptr to string holding the type of the current dataset

  struct xmlindent* ptr_indent ptr to structure with indentation info

  int n_ref_format reference type (REFDOCBKX|REFDOCBKX5)

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_series_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent, int n_ref_format) {
  int n_have_data = 0; /* set to 1 if we have some data */
  char *new_ref;
  char *entitize_buf;
  char *author_buf;
  char *author_result;
  const char *item;
  size_t author_buf_len;

  **ptr_chunk_buf = '\0';

  author_buf_len = 1024;

  if ((author_buf = malloc(author_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }
    
  /* start series info */
  if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", "relation", "series", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    free(author_buf);
    return NULL;
  }

  /* add editors, if any */
  author_result = add_author_info_db(ptr_rendinfo, 3 /* set editors */, n_refdb_id, &author_buf, &author_buf_len, ptr_indent, n_ref_format);

  if (!author_result) {
    free(author_buf);
    return NULL;
  }
  else if (*author_result) {
    n_have_data++;
    if ((new_ref = mstrcat(*ptr_chunk_buf, author_result, ptr_chunk_buf_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(author_buf);
      return NULL;
    }
    else {
      *ptr_chunk_buf = new_ref;
    }
  }

  free(author_buf);

  /* series title, if any */
  item = get_refdb_title_series_copy(ptr_rendinfo->dbires);

  if (item && *item) {
    n_have_data++;
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      free((char*)item);
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free((char*)item);
      free(entitize_buf);
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "title", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free((char*)item);
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }

    free(entitize_buf);
    free((char*)item);
    n_have_data++;
  }

  /* finish series information */
  if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "biblioset", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }

  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_notes_db() adds notes info to a DocBook bibliography

  static char* add_notes_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  struct xmlindent* ptr_indent ptr to structure with indentation info

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_notes_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, struct xmlindent* ptr_indent) {
  int n_have_data = 0; /* set to 1 if we have some data */
  char *entitize_buf;
  const char *item;

  **ptr_chunk_buf = '\0';

  /* note of current user, if any */
  item = get_notes_copy(ptr_rendinfo->dbires, ptr_rendinfo->username);

  if (item && *item) {
    n_have_data++;
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      free((char*)item);
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free((char*)item);
      free(entitize_buf);
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "bibliomisc", "role", "annotation", NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    free(entitize_buf);
  }

  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_abstract_db() adds abstract to a DocBook bibliography

  static char* add_abstract_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  struct xmlindent* ptr_indent ptr to structure with indentation info

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_abstract_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, struct xmlindent* ptr_indent) {
  int n_have_data = 0; /* set to 1 if we have some data */
  char *entitize_buf;
  const char *item;

  **ptr_chunk_buf = '\0';

  /* abstract, if any */
  item = get_refdb_abstract_copy(ptr_rendinfo->dbires);

  if (item != NULL) {
    n_have_data++;
    if ((entitize_buf = mstrdup((char*)item)) == NULL) {
      free((char*)item);
      return NULL;
    }

    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free((char*)item);
      free(entitize_buf);
      return NULL;
    }

    if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "abstract", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      free(entitize_buf);
      return NULL;
    }

    if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "para", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      free((char*)item);
      free(entitize_buf);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    free(entitize_buf);
    free((char*)item);
  }

  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_biblioids_db() adds isbn/issn/doi to a DocBook bibliography

  static char* add_biblioids_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int n_refdb_id ID of the current dataset

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  const char* type type of biblioid (doi/issn/isbn)

  struct xmlindent* ptr_indent ptr to structure with indentation info

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_biblioids_db(struct renderinfo* ptr_rendinfo, int n_refdb_id, char **ptr_chunk_buf, size_t *ptr_chunk_buf_len, const char* type, struct xmlindent* ptr_indent) {
  int n_have_data = 0; /* set to 1 if we have some data */
  char *entitize_buf;
  const char *item;
  dbi_result dbires;

  **ptr_chunk_buf = '\0';

  /* get identifier */
  if (!strcmp(type, "doi") && ptr_rendinfo->ref_format == REFDOCBKX5) {
    dbires = request_ulinks(dbi_result_get_conn(ptr_rendinfo->dbires), n_refdb_id, 0 /* ref entry */, 5 /* link type */, 0 /* is_temp */, NULL);
    if (dbires == NULL) {
      return NULL;
    }

    while ((item = get_ulink(dbires)) != NULL) {
      if ((entitize_buf = mstrdup((char*)item)) == NULL) {
	free((char*)item);
	return NULL;
      }

      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free((char*)item);
	free(entitize_buf);
	return NULL;
      }

      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "biblioid", "class", "doi", NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }

      n_have_data++;

      free(entitize_buf);
    }

    clean_request(dbires);
  }
  else {
    item = get_refdb_issn(ptr_rendinfo->dbires);

    if (item != NULL) {
      if ((entitize_buf = mstrdup((char*)item)) == NULL) {
	free((char*)item);
	return NULL;
      }

      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free((char*)item);
	free(entitize_buf);
	return NULL;
      }

      if (ptr_rendinfo->ref_format == REFDOCBKX5) { /* DocBook >= 5 */
	if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "biblioid", "class", type, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	  free(entitize_buf);
	  LOG_PRINT(LOG_CRIT, get_status_msg(801));
	  return NULL;
	}
      }
      else {
	if (!strcmp(type, "isbn")) {
	  if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "isbn", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	    free(entitize_buf);
	    LOG_PRINT(LOG_CRIT, get_status_msg(801));
	    return NULL;
	  }
	}
	else {
	  if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "issn", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	    free(entitize_buf);
	    LOG_PRINT(LOG_CRIT, get_status_msg(801));
	    return NULL;
	  }
	}
      }
      n_have_data++;

      free(entitize_buf);
    }
  }

  if (!n_have_data) {
    **ptr_chunk_buf = '\0';
  }

  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_author_db(): adds an author in DocBook format

  static char* add_author_db returns a pointer to authorstring

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  struct AUTHOR_INFO* ptr_ainfo ptr to a structure with the name parts

  char **ptr_chunk_buf ptr to an allocated string, may be resized

  size_t *ptr_chunk_buf_len ptr to length of ptr_chunk_buf

  struct xmlindent* ptr_indent ptr to structure with indentation info

  int n_ref_format reference type (REFDOCBKX|REFDOCBKX5)

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_author_db(struct renderinfo* ptr_rendinfo, struct AUTHOR_INFO* ptr_ainfo, char** ptr_chunk_buf, size_t* ptr_chunk_buf_len, struct xmlindent* ptr_indent, int n_ref_format) {
  char* item;
  char* entitize_buf = NULL;

  **ptr_chunk_buf = '\0'; /* start with empty string */

  if (!ptr_ainfo) {
    return *ptr_chunk_buf;
  }

  /* ToDo: check for corporate name */
  if (!*(ptr_ainfo->lastname) && !*(ptr_ainfo->firstname)
      && !*(ptr_ainfo->middlename) && !*(ptr_ainfo->suffix)) {
    if ((entitize_buf = mstrdup(ptr_ainfo->name)) == NULL) {
      return NULL;
    }
    
    if (sgml_entitize(&entitize_buf, NULL) == NULL) {
      free(entitize_buf);
      return NULL;
    }

    if (*(ptr_ainfo->role)) {
      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "corpauthor", "role", ptr_ainfo->role, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
    }
    else {
      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "corpauthor", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
    }
    free(entitize_buf);
  }
  else { /* have name parts */
    if (*(ptr_ainfo->role)) {
      if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "author", "role", ptr_ainfo->role, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(entitize_buf);
	return NULL;
      }
    }
    else {
      if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "author", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(entitize_buf);
	return NULL;
      }
    }

    if (n_ref_format == REFDOCBKX5) {
      if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "personname", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(entitize_buf);
	return NULL;
      }
    }

    if (*(ptr_ainfo->firstname)) {
      if ((entitize_buf = mstrdup(ptr_ainfo->firstname)) == NULL) {
	return NULL;
      }
    
      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free(entitize_buf);
	return NULL;
      }
      
      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "firstname", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
      free(entitize_buf);
    }

    if (*(ptr_ainfo->middlename)) {
      for (item = strtok(ptr_ainfo->middlename, " "); item; item = strtok(NULL, " ")) {
	if ((entitize_buf = mstrdup(item)) == NULL) {
	  return NULL;
	}

	if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	  free(entitize_buf);
	  return NULL;
	}

	if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "othername", "role", "mi", NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	  free(entitize_buf);
	  LOG_PRINT(LOG_CRIT, get_status_msg(801));
	  return NULL;
	}
	free(entitize_buf);
      }
    }

    if (*(ptr_ainfo->lastname)) {
      if ((entitize_buf = mstrdup(ptr_ainfo->lastname)) == NULL) {
	return NULL;
      }

      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free(entitize_buf);
	return NULL;
      }

      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "surname", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
      free(entitize_buf);
    }

    if (*(ptr_ainfo->suffix)) {
      if ((entitize_buf = mstrdup(ptr_ainfo->suffix)) == NULL) {
	return NULL;
      }

      if (sgml_entitize(&entitize_buf, NULL) == NULL) {
	free(entitize_buf);
	return NULL;
      }

      if (print_element_x(entitize_buf, ptr_chunk_buf, ptr_chunk_buf_len, "lineage", NULL, NULL, NULL, NULL, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	free(entitize_buf);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
      free(entitize_buf);
    }

    if (n_ref_format == REFDOCBKX5) {
      if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "personname", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return NULL;
      }
    }

    if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "author", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
  }
    
  return *ptr_chunk_buf;
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  add_author_info_db() adds author info to a DocBook bibliography

  static char* add_author_info_db returns ptr to ref if successful, NULL if not

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  int author_type 1=part 2=publication 3=set

  int n_refdb_id ID of the current dataset

  char** ptr_chunk_buf ptr to a buffer to receive output

  size_t* ptr_chunk_buf_len ptr to length of string

  struct xmlindent* ptr_indent ptr to structure with indentation info

  int n_ref_format reference type (REFDOCBKX|REFDOCBKX5)

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static char* add_author_info_db(struct renderinfo* ptr_rendinfo, int author_type, int n_refdb_id, char** ptr_chunk_buf, size_t* ptr_chunk_buf_len, struct xmlindent* ptr_indent, int n_ref_format) {
  int nis_first = 1;
  size_t author_buf_len = 512;
  char *new_ref;
  char *dbauthor;
  char *author_buf;
  dbi_conn conn;
  dbi_result dbires;
  struct AUTHOR_INFO ainfo;

  if ((author_buf = malloc(author_buf_len)) == NULL) {
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }

  **ptr_chunk_buf = '\0';

  conn = dbi_result_get_conn(ptr_rendinfo->dbires);

  /* get authors */
  dbires = request_authors(conn, author_type, NULL /* all roles */, NULL, 0, n_refdb_id);
  if (dbires == NULL) {
    return NULL;
  }

  /* get buffer to assemble the docbook author output */
  dbauthor = malloc((size_t)2048);  /* fixed length ok afaik */
    
  if (dbauthor == NULL) {
    clean_request(dbires);
    LOG_PRINT(LOG_CRIT, get_status_msg(801));
    return NULL;
  }

  /* fetch all authors of this article */
  while (get_author_parts(dbires, &ainfo) != NULL) {
    char* authorchunk;

    /* begin authorgroup element */
    if (nis_first) {
      nis_first--;
      if (print_elstart_x(ptr_chunk_buf, ptr_chunk_buf_len, "authorgroup", NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0, ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	clean_request(dbires);
	free(dbauthor);
	return NULL;
      }
    }

    authorchunk = add_author_db(ptr_rendinfo, &ainfo, &author_buf, &author_buf_len, ptr_indent, n_ref_format);

    if (authorchunk && *authorchunk) {
      if ((new_ref = mstrcat(*ptr_chunk_buf, authorchunk, ptr_chunk_buf_len, 0)) == NULL) {
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	free(author_buf);
	return NULL;
      }
      else {
	*ptr_chunk_buf = new_ref;
      }
    }
  }
  free(dbauthor);
  free(author_buf);
  clean_request(dbires);

  /* close authorgroup element only if it was started previously */
  if (!nis_first) {
    if (print_elend_x(ptr_chunk_buf, ptr_chunk_buf_len, "authorgroup", ptr_indent, ptr_rendinfo->ptr_clrequest->namespace) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
  }
  else {
    **ptr_chunk_buf = '\0';
  }

  /* if there was no author info, chunk_buf is returned as an empty string */
  return *ptr_chunk_buf;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  indent_notbelow_db(): checks whether or not to indent below the current
                     element

  static int indent_notbelow_db returns 1 if not to indent, 0 if to indent

  const char* name ptr to element name

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static int indent_notbelow_db(const char* name)
{
  if (!strcmp(name, "date")) {
    return 1;
  }
  else {
    return 0;
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  is_entry_db(): checks whether an element starts a new entry

  static int is_entry_db returns 1 if entry starts, 0 if not

  const char* name ptr to element name

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static int is_entry_db(const char* name) {
  if (strstr(name, "biblioentry")) {
    return 1;
  }
  else {
    return 0;
  }
}
