open Str

let compute_predicate () = 
  let s = Sys.ocaml_version in
  if string_match (regexp "^3\\.\\(0[8-9]\\|[1-9]\\).*$") s 0 then
    "3.07_19"
  else if string_match (regexp "^3\\.07\\+\\([0-9]+\\)\\([^0-9].*\\)?$") s 0 then
    let pv = int_of_string (matched_group 1 s) in
    if pv >= 19 then
      "3.07_19"
    else if pv >= 5 then
      "3.07_5"
    else
      "3.07"
  else if string_match (regexp "3\\.06\\+\\([0-9]+\\)$") s 0 then
    let pv = int_of_string (matched_group 1 s) in
    if pv >= 37 then
      "3.07pre"
    else
      "3.06"
  else if string_match (regexp "3\\.06$") s 0 then
    "3.06"
  else
    failwith ("unknown or unsupported OCaml version: " ^ s)

let if_regexp = regexp "^#if +\\([a-z0-9._| ]+\\) *$"
(* let elif_regexp = regexp "^#elif +\\([a-z0-9.| ]+\\) *$" *)
let else_regexp = regexp "^#else *$"
let endif_regexp = regexp "^#endif *$"
let separator_regexp = regexp " *| *"

let process_file ifname ofd pred =
  let nest_level = ref 0 in
  let output_level = ref 0 in
  let ifd = open_in ifname in
  try
    Printf.fprintf ofd "# %d \"%s\"\n" 1 (String.escaped ifname);
    while true do
      let line = input_line ifd in
      let cf = begin
	if string_match if_regexp line 0 then begin
	  if !output_level = !nest_level then begin
	    let conditions = matched_group 1 line in
	    let l = Str.split separator_regexp conditions in
	    if List.mem pred l then
	      incr output_level;
	  end;
	  incr nest_level;
	  true
	end
	else if string_match endif_regexp line 0 then begin
	  decr nest_level;
	  if !output_level > !nest_level then
	    decr output_level;
	  true
	end
	else if string_match else_regexp line 0 then begin
	  if !output_level = !nest_level then
	    decr output_level
	  else if !output_level = !nest_level - 1 then
	    incr output_level;
	  true
	end
	else
	  !output_level < !nest_level
      end
      in
      let line = 
	if cf then "(*" ^ line ^ "*)" else line
      in
      output_string ofd (line ^ "\n")
    done
  with
    End_of_file -> close_in ifd
  | e -> close_in ifd; raise e

open Arg

let ifname = ref None
let pp = ref None
let verbose = ref false
let no_remove = ref false

let arg_set r doc v = 
  match !r with
    None -> r := Some v
  | Some _ -> raise (Bad (doc ^ " specified twice"))

let show_pred () = 
  print_endline (compute_predicate ());
  exit 0

let argspec = 
  [ "-pp", String (arg_set pp "-pp"), "\tpass output to secondary preprocessor";
    "-showpred", Unit show_pred, "\tshow defined predicate";
    "-verbose", Set verbose, "\tverbose message";
    "-no-remove", Set no_remove, "\tdo not remove temporary file";
  ]

let anonfun = arg_set ifname "input files"

let usage_msg = "Usage: " ^ Sys.argv.(0) ^ " [options] input_file"

let () = Random.self_init ()

let msg_EEXIST = Unix.error_message Unix.EEXIST

let rec open_out_temp () = 
  let fname = "camlpp" ^ (string_of_int (Random.bits ())) in
  if !verbose then Printf.eprintf "version_filter: opening temporary file %s\n" fname;
  try
    let fd = open_out_gen [Open_wronly; Open_creat; Open_excl] 0o666 fname in
    fd, fname
  with
    Sys_error s when s = fname ^ ": " ^ msg_EEXIST ->
      (* Printf.eprintf "filename crashed: %s\n" fname; *)
      open_out_temp ()

let main () = 
  parse argspec anonfun usage_msg;
  let ifname = 
    match !ifname with
      None -> usage argspec usage_msg; exit 2
    | Some v -> v
  in
  let pred = compute_predicate () in
  if !verbose then Printf.eprintf "version_filter: reading %s\n" ifname;
  let ifd = open_in ifname in
  begin
    match !pp with
      None ->
	if !verbose then Printf.eprintf "version_filter: output to stdout\n";
	process_file ifname stdout pred
    | Some pp -> begin
	let ofd, ofname = open_out_temp () in
	try
	  begin
	    try
	      if !verbose then Printf.eprintf "version_filter: output to %s\n" ofname;
	      process_file ifname ofd pred;
	      if !verbose then Printf.eprintf "version_filter: closing %s\n" ofname;
	      close_out ofd
	    with 
	      e -> close_out ofd; raise e
	  end;
	  let cmdline = (pp ^ " " ^ Filename.quote ofname) in
	  if !verbose then Printf.eprintf "version_filter: invoking %s\n" cmdline;
	  let rstatus = Sys.command cmdline in
	  if !verbose then Printf.eprintf "version_filter: command %s returns %d\n" cmdline rstatus;
	  if rstatus <> 0 then
	    failwith "secondary preprocessor failed";
	  if not !no_remove then begin
	    if !verbose then Printf.eprintf "version_filter: removing %s\n" ofname;
	    Sys.remove ofname;
	    if !verbose then Printf.eprintf "version_filter: removing %s done\n" ofname
	  end
	with
	  e ->
	    if not !no_remove then begin
	      if !verbose then Printf.eprintf "version_filter: removing %s (err)\n" ofname;
	      Sys.remove ofname; 
	      if !verbose then Printf.eprintf "version_filter: removing %s done\n" ofname
	    end;
	    raise e
    end
  end

let () = main ()

let () = if !verbose then Printf.eprintf "version_filter: exiting\n";
