/***************************************************************************
    file	         : kb_location.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>
#include	<sys/stat.h>


#include	<time.h>

#include	<qstring.h>
#include	<qfile.h>
#include	<qfileinfo.h>
#include	<qtextstream.h>
#include	<qtextcodec.h>

#if		__KB_KDE
#include	<kglobal.h>
#include	<kcharsets.h>
#endif

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"
#include	"kb_location.h"
#include	"kb_basequery.h"
#include	"kb_notifier.h"

#include	"kb_locator.h"


class LIBCOMMON_API	KBLocnTypeMap
{
public	:

	QString		type	;
	QString		extn	;
	QString		legend	;
	KBPartFactory	*factory;

	inline	KBLocnTypeMap
		(	cchar		*type,
			cchar		*extn,
			const QString	&legend,
			KBPartFactory	*factory
		)
		:
		type	(type),
		extn	(extn),
		legend	(legend),
		factory	(factory)
	{
	}
}	;

cchar	*KBLocation::m_pFile	= "!Files"		;
cchar	*KBLocation::m_pInline	= "!Inline"		;
cchar	*KBLocation::m_pStock	= "Stock Components"	;

static 	QList<KBLocnTypeMap>	locnTypeMap	;


/*  KBLocation								*/
/*  KBLocation	: Constructor for location object			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  docType	: cchar *	  : Object type				*/
/*  docLocn	: const QString & : Document location			*/
/*  docName	: const QString & : Document name			*/
/*  docExtn	: const QString & : Document file extension		*/
/*  (returns)	: KBLocation	  :					*/

KBLocation::KBLocation
	(	KBDBInfo	*_dbInfo,
		cchar		*_docType,
		const QString	&_docLocn,
		const QString	&_docName,
		const QString	&_docExtn
	)
	:
	dbInfo	(_dbInfo ),
	docType	(_docType),
	docLocn	(_docLocn),
	docName	(_docName),
	docExtn	(_docExtn)
{
	if (docName.left(2) == "//")
	{
		docName	= docName.mid(2) ;
		docLocn	= m_pFile	 ;
		dbInfo	= 0 ;
	}

//	fprintf
//	(	stderr,
//		"KBLocation::KBLocation [%p] [%s][%s][%s][%s]\n",
//		(void  *)this,
//		(cchar *)docType,
//		(cchar *)docLocn,
//		(cchar *)docName,
//		(cchar *)docExtn
//	)	;
}

/*  KBLocation								*/
/*  KBLocation	: Constructor for location object			*/
/*  (returns)	: KBLocation	  :					*/

KBLocation::KBLocation ()
{
	dbInfo	= 0 		;
	docType	= "unknown"	;
}

/*  KBLocation								*/
/*  findByType	: Find type map entry by type				*/
/*  type	: const QString & : Type				*/
/*  (returns)	: KBLocnTypeMap * : Entry or null if not found		*/

KBLocnTypeMap
	*KBLocation::findByType
	(	const QString	&type
	)
{
	LITER
	(	KBLocnTypeMap,
		locnTypeMap,
		tmp,

		if (tmp->type == type) return tmp ;
	)

	return	0 ;
}

/*  KBLocation								*/
/*  KBLocation								*/
/*  findByExtn	: Find type map entry by extension			*/
/*  extn	: const QString & : Extension				*/
/*  (returns)	: KBLocnTypeMap * : Entry or null if not found		*/

KBLocnTypeMap
	*KBLocation::findByExtn
	(	const QString	&extn
	)
{
	LITER
	(	KBLocnTypeMap,
		locnTypeMap,
		tmp,

		if (tmp->extn == extn) return tmp ;
	)

	return	0 ;
}

/*  KBLocation								*/
/*  operator ==	: Test equality						*/
/*  other	: KBLocation &	: Other location			*/
/*  (returns)	: bool		: Equality				*/

bool	KBLocation::operator ==
	(	KBLocation &other
	)
{
	return	(dbInfo  == other.dbInfo ) && (docType == other.docType) &&
		(docLocn == other.docLocn) && (docName == other.docName) &&
		(docExtn == other.docExtn) ;
}

/*  KBLocation								*/
/*  path	: Get path for file objects				*/
/*  altName	: const QString & : Alternate name			*/
/*  (returns)	: Qstring 	  : File path				*/

QString	KBLocation::path
	(	const QString	&altName
	)
{
	/* Note that we return a path unless the location is inline.	*/
	/* If the location is actually in a database server, we still	*/
	/* return a notional path so that "saveDocumentAs" works as	*/
	/* expected.							*/
	if (isInline()) return QString::null ;

	QString	name = altName.isNull() ? docName : altName ;
	name	+= "." + extnForType (dbInfo, docType, docExtn) ;

	/* If we have a database then the path is resolved relative to	*/
	/* the database directory ....					*/
	if (dbInfo != 0)
		return	dbInfo->getDBPath() + "/"  + name  ;

	/* ... otherwise look in the application global directory.	*/
	return	locateFile ("appdata", name) ;
}

/*  KBLocation								*/
/*  ident	: Get location identifier				*/
/*  (returns)	: QString	: Location identifier			*/

QString	KBLocation::ident ()
{
	QString	dbName	= dbInfo == 0 ? QString("") : dbInfo->getDBName() ;
	return	QString("%1:%2:%3").arg(dbName )
				   .arg(docLocn)
				   .arg(docName) ;
}

/*  KBLocation								*/
/*  title	: Get title for window headers, etc			*/
/*  (returns)	: QString	: Title					*/

QString	KBLocation::title ()
{
	KBLocnTypeMap	*ltm	= findByType (docType) ;
	QString		legend	;

	if ((ltm != 0) && !ltm->legend.isEmpty())
		legend	= QString("%1: ").arg(ltm->legend) ;

	if (docLocn == m_pInline)
		return	QString("%1Inline:%3").arg(legend).arg(docName) ;
	if (docLocn == m_pFile  )
		return	QString("%1!Files:%3").arg(legend).arg(docName) ;

	return	QString	("%1%2:%3").arg(legend).arg(docLocn).arg(docName) ;
}

/*  KBLocation								*/
/*  getData	: Construct and execute select for specified field	*/
/*  field	: cchar *	: Required field			*/
/*  pError	: KBError &	: Error return				*/
/*  data	: QByteArray &	: Result buffer				*/
/*  (returns)	: bool		: Success				*/

bool	KBLocation::getData
	(	cchar		*field,
		KBError		&pError,
		QByteArray	&data
	)
{
	/* Check in case we have been called without a database		*/
	/* information structure; script errors seem to be a major	*/
	/* source of this bug :-)					*/
	if (dbInfo == 0)
	{
		pError	= KBError
			  (	KBError::Fault,
				"KBLocation::getData called without database information",
				QString("%1.%2").arg(docName).arg(docLocn),
				__ERRLOCN
			  )	;
		return	false	;
	}

	KBSQLSelect	*select	;
	KBDBLink	dbLink 	;
	bool		exists	;

	/* Open a link to the database and very that the objects table	*/
	/* exists. If it does then open a select query to retrieve the	*/
	/* specified field for this location.				*/
	if (!dbLink.connect (*this, docLocn, true))
	{	pError	= dbLink.lastError() ;
		return	false	;
	}

	QString	objectTab = dbLink.rekallPrefix (OBJECTTABNAME) ;

	if (!dbLink.tableExists (objectTab, exists))
	{	pError	= dbLink.lastError() ;
		return	false	;
	}

	if (!exists)
	{	pError	= KBError
			  (	KBError::Fault,
				QString(TR("Server %1 does not have a objects table: cannot load")).arg(docLocn),
				QString(TR("Type %1, name %2")).arg(docType).arg(docName),
				__ERRLOCN
			  )	;
		return	false	;
	}



	/* Execute the select query for the named document of the	*/
	/* appropriate type. Note that we select all columns at the	*/
	/* same time in order to get the column types that the actual	*/
	/* server database is using.					*/
	KBValue		args[3]	;
	QString		type	;
	int		nargs	= 2 ;
	QString		qtext	= "select %1 from " +
					QString("%1 where %2 = %3 and %4 = %5")
					 .arg(dbLink.mapExpression(objectTab))
					 .arg(dbLink.mapExpression("Name"   ))
					 .arg(dbLink.placeHolder  (0))
					 .arg(dbLink.mapExpression("Type"   ))
				  	 .arg(dbLink.placeHolder  (1)) ;

	KBLocnTypeMap	*tmp	= findByType (docType) ;

	if (tmp == 0)
	{	pError	= KBError
			  (	KBError::Fault,
			        "KBLocation::getData called with inappropriate type",
				QString("Type code %1").arg(docType),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (docType == "script")
	{
		type	= "script" 	;
		nargs	= 3	   	;
		qtext  += QString(" and %1 = %2 ")
				 .arg(dbLink.mapExpression("Extension"))
				 .arg(dbLink.placeHolder  (2)) ;
	}
	else	type	= docType	;

	if ((select = dbLink.qrySelect
		      (		false,
				QString(qtext).arg(dbLink.mapExpression(field))
		      )) == 0)
	{
		pError	= dbLink.lastError() ;
		return	false	;
	}

	args[0]	= docName	;
	args[1]	= type		;
	args[2] = docExtn	;

	if (!select->execute (nargs, args))
	{	pError	= select->lastError() ;
		delete	select	;
		return	false	;
	}

	// fprintf (stderr, "[%s]\n", (cchar *)select->getRawQuery()) ;
	// fprintf (stderr, "[%s]\n", (cchar *)select->getSubQuery()) ;

	/* ... which should return exactly one row. If it does then	*/
	/* we will return the contents of the field as raw text.	*/
	if (!select->rowExists (0))
	{	pError	= KBError
			  (	KBError::Error,
				"Cannot load document",
				QString	("Document %1 (%2) not found").arg(docName).arg(type),
				__ERRLOCN
			  )	;
		delete	select	;
		return	false	;
	}

	KBValue	value	= select->getField(0, 0) ;
	data.duplicate	(value.dataPtr(), value.dataLength()) ;
	delete	select	;
	return	true	;
}

/*  KBLocation								*/
/*  exists	: See if the document exists				*/
/*  (returns)	: QString	: Location contents or null on error	*/

bool	KBLocation::exists()
{
	/* First case is if document is in the database directory, in	*/
	/* which just check for its existance.				*/
	if (isFile())
		return	QFileInfo(path()).exists () ;

	/* Next case is inline .... easy				*/
	if (isInline())
		return	true	;

	/* Otherwise, we may as well retrieve the contents as this is	*/
	/* going to be almost as quick.					*/
	KBError		error	;
	QByteArray	bytes	;
	return	getData ("Definition", error, bytes) ;
}

/*  KBLocation								*/
/*  contents	: Get the contents of the location			*/
/*  data	: QByteArray &	: Data buffer				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBLocation::contents
	(	QByteArray	&data,
		KBError		&pError
	)
{
//	fprintf
//	(	stderr,
//		"KBLocation::contents [%s][%s][%s][%s]\n",
//		(cchar *)docType,
//		(cchar *)docLocn,
//		(cchar *)docName,
//		(cchar *)docExtn
//	)	;

	/* First case is if document is in the database directory, in	*/
	/* which case read the document from the appropriate file.	*/
	if (isFile())
	{
		QFile	file (path()) ;

		if (!file.open (IO_ReadOnly))
		{
			pError	= KBError
				  (	KBError::Error,
					QString	("Failed to open \"%1\"").arg(path()),
					strerror(errno),
					__ERRLOCN
				  )	;
			return	false	;
		}

		data	= file.readAll();
		return	true	;
	}

	/* Second case is access to a stock component. This is		*/
	/* basically the same as above.					*/
	if (isStock())
	{
		QString	name	= locateFile
				  (	"appdata",
					QString	("stock/%1/%2.%3")
						.arg(docType)
						.arg(docName)
						.arg(docExtn)
			     	  )	;
		QFile	file	(name)	;

		if (!file.open (IO_ReadOnly))
		{
			pError	= KBError
				  (	KBError::Error,
					QString	("Failed to open \"%1\"").arg(name),
					strerror(errno),
					__ERRLOCN
				  )	;
			return	QString::null ;
		}

		data	= file.readAll() ;
		return	true	;
	}

	/* Next case is inline .... easy				*/
	if (isInline())
	{
		data	= QCString(docExtn) ;
		return	true	    ;
	}

	/* Otherwise, we need to read the document source from the	*/
	/* objects table for the specified server...			*/
	return	getData ("Definition", pError, data) ;
}

/*  KBLocation								*/
/*  contents	: Get the contents of the location			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: QString	: Location contents or null on error	*/

QString	KBLocation::contents
	(	KBError	&pError
	)
{
	/* Pick the inline case out specially to save conversions ...	*/
	if (isInline()) return docExtn   ;

	/* ... otherwise go via the QByteArray method ...		*/
	QByteArray data ;
	if (!contents (data, pError)) return QString::null ;

	return	QString::fromUtf8 (data.data(), data.count()) ;
}

/*  KBLocation								*/
/*  timestamp	: Get location timestamp				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: QString	: Timestamp or null on error		*/

QString	KBLocation::timestamp
	(	KBError	&pError
	)
{
	/* If the location is a file then grab the modification time of	*/
	/* the file.							*/
	if (isFile())
	{
		struct	stat statb ;

		if (stat (path(), &statb) != 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	"Cannot determine object modification time",
				  	QString	("%1.%2 (%3): %4")
						.arg(docName)
						.arg(docExtn)
						.arg(path ())
						.arg(strerror(errno)),
				  	__ERRLOCN
				  )	;
			return	QString::null ;
		}

		return	QString("%1").arg(statb.st_mtime) ;
	}

	/* The timestamp of an inline location is arbitrary, so return	*/
	/* the current time.						*/
	if (isInline()) return QString("%1").arg(time(0)) ;

	/* Lastly, for database-stored locations, return the time	*/
	/* stamp from the objects table for this location.		*/
	QByteArray bytes ;
	return	   getData ("SaveDate", pError, bytes) ? bytes : QString::null ;
}

/*  KBLocation								*/
/*  getFactory	: Get part factory for this object type			*/
/*  (returns)	: KBPartFactory * : Factory for the object type		*/

KBPartFactory
	*KBLocation::getFactory ()
{
	KBLocnTypeMap *tmp = findByType (docType) ;

	fprintf
	(	stderr,
		"KBLocation::getFactory: [%s]->[%p]\n",
		(cchar *)docType,
		(void  *)tmp
	)	;
	return	tmp == 0 ? 0 : tmp->factory ;
}

/*  KBLocation								*/
/*  registerType: Register a new type					*/
/*  type	: cchar *	  : Type				*/
/*  extn	: cchar *	  : Extension				*/
/*  legend	: const QString & : Legend for titles			*/
/*  factory	: KBPartFactory * : Factory for the object type		*/
/*  (returns)	: void		  :					*/

void	KBLocation::registerType
	(	cchar		*type,
		cchar		*extn,
		const QString	&legend,
		KBPartFactory	*factory
	)
{
	fprintf (stderr, "KBLocation::registerType(%s,%s)\n", type, extn) ;
	locnTypeMap.append (new KBLocnTypeMap (type, extn, legend, factory)) ;
}

QString	KBLocation::extnForType
	(	KBDBInfo	*dbInfo,
		const QString 	&docType,
		const QString	&docExtn
	)
{
//	fprintf
//	(	stderr,
//		"KBLocation::extnForType(%s,%s)\n",
//		(cchar *)docType,
//		(cchar *)docExtn
//	)	;

	KBLocnTypeMap	*tmp	= findByType (docType) ;

	if (tmp != 0)
		if (tmp->extn.isEmpty())
			return	docExtn	  ;
		else	return	dbInfo->getDBExtn() + "." + tmp->extn ;

//	return	"error"	;
	return	docExtn ;
}


/*  KBLocation								*/
/*  saveToFile	: Save document to a file				*/
/*  file	: const QString & : File location			*/
/*  name	: const QString & : Document name			*/
/*  buffer	: cchar *	  : Data buffer				*/
/*  bufflen	: uint		  : Buffer length			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::saveToFile
	(	const QString	&file,
		const QString	&,
		cchar		*buffer,
		uint		bufflen,
		KBError		&pError
	)
{
	QFile	out	(file) ;

	if (!out.open (IO_WriteOnly))
	{
		pError	= KBError
			  (	KBError::Error,
				QString ("Unable to open %1 file \"%2\"")
					.arg(docType)
					.arg(file),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (out.writeBlock (buffer, bufflen) != (int)bufflen)
	{
		pError	= KBError
			  (	KBError::Error,
				QString ("Error writing file \"%1\"").arg(file),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBLocation								*/
/*  saveToDB	: Save document to database table			*/
/*  locn	: const QString & : Table location			*/
/*  name	: const QString & : Document name			*/
/*  buffer	: cchar *	  : Data buffer				*/
/*  bufflen	: uint		  : Buffer length			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::saveToDB
	(	const QString	&locn,
		const QString	&name,
		cchar		*buffer,
		uint		bufflen,
		KBError		&pError
	)
{
	KBDBLink dbLink	;
	bool	 exists	;
	bool	 rv	= false ;

	if (!dbLink.connect (*this, locn, true))
	{	pError	= dbLink.lastError() ;
		return	false ;
	}

	QString	    objectTab  = dbLink.rekallPrefix (OBJECTTABNAME) ;
	if (!dbLink.tableExists (objectTab, exists))
	{	pError	= dbLink.lastError() ;
		return	false ;
	}

	if (!exists)
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("Server %1 does not have a objects table: cannot save")).arg(locn),
				QString(TR("Type %1, name %2")).arg(docType).arg(name),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* This is an update-or-insert so we need to find out if there	*/
	/* is already an entry. The select query also serves to get	*/
	/* type information for each of the columns in the objects	*/
	/* table (which will vary from server to server).		*/
	KBSQLSelect *qrySelect = 0 ;
	KBSQLInsert *qryInsert = 0 ;
	KBSQLUpdate *qryUpdate = 0 ;
	KBSQLQuery  *save      ;
	int	    nRows      ;

	uint	    argc   = 0 ;
	KBValue	    args[8]    ;

	char	   dstr[32] ;
	time_t	   tnow	    = time   (0)	;
	struct tm  *gmt	    = gmtime (&tnow)	;

	strftime (dstr, sizeof(dstr), "%Y%m%d%H%M%S", gmt) ;

	QString selString =
			QString	("select %1, %2, %3, %4, %5, %6, %7 from %8")
				.arg(dbLink.mapExpression("Id"		))
				.arg(dbLink.mapExpression("Description"	))
				.arg(dbLink.mapExpression("Definition"	))
				.arg(dbLink.mapExpression("SaveDate"	))
				.arg(dbLink.mapExpression("Type"	))
				.arg(dbLink.mapExpression("Name"	))
				.arg(dbLink.mapExpression("Extension"	))
				.arg(dbLink.mapExpression(objectTab	)) ;

	selString += QString(" where %1 = %2")
			.arg(dbLink.mapExpression("Type")) 
			.arg(dbLink.placeHolder  (0)) ;
	selString += QString(" and %1 = %2"  )
			.arg(dbLink.mapExpression("Name")) 
			.arg(dbLink.placeHolder  (1)) ;

	if ((qrySelect = dbLink.qrySelect(false, selString)) == 0)
	{	pError	= dbLink.lastError() ;
		goto	error ;
	}
	args[0] = QString(docType) ;
	args[1]	= name	;
	if (!qrySelect->execute (2, args))
	{	pError	= qrySelect->lastError() ;
		goto	error ;
	}

	/* No row exists then insert data, otherwise update an extant	*/
	/* entry.							*/
	if (!qrySelect->rowExists(0))
	{
		KBValue		newKey    ;
		KBTableSpec	tabSpec	  (objectTab) ;
		bool		idRO	  ;

		if (!dbLink.listFields (tabSpec))
		{
			pError	= dbLink.lastError() ;
			return	false	;
		}
		idRO	= (tabSpec.m_fldList.at(0)->m_flags & KBFieldSpec::ReadOnly) != 0 ;

		KBBaseInsert	insert	(objectTab) ;

		if (!idRO)
			insert.addExpression("Id" ) ;

		insert.addExpression("Description", (void *)0) ;
		insert.addExpression("Definition" ) ;
		insert.addExpression("SaveDate"	  ) ;
		insert.addExpression("Type"	  ) ;
		insert.addExpression("Name"	  ) ;

		if (!docExtn.isEmpty())
			insert.addExpression("Extension") ;

		QString	insString = insert.getQueryText(&dbLink) ;

		/* Construct the insert query and report an error if	*/
		/* this fails. Then if the ID column is not read-only	*/
		/* then see if we get a pre-insert key value.		*/
		if ((qryInsert = dbLink.qryInsert (false, insString, objectTab)) == 0)
		{
			pError	= dbLink.lastError() ;
			goto	error ;
		}

		if (!idRO)
		{
			if (!qryInsert->getNewKey ("Id", newKey, true))
			{
				pError	= dbLink.lastError() ;
				goto	error ;
			}

			args[argc++] = newKey ;
		}

		save	     = qryInsert ;
	}
	else
	{
		KBBaseUpdate	update	(objectTab) ;

		update.addExpression("Definition" ) ; 
		update.addExpression("SaveDate"	  ) ; 
		update.addWhere	    ("Type"	  ) ; 
		update.addWhere	    ("Name"	  ) ;

		if (!docExtn.isEmpty())
			update.addWhere("Extension") ; 

		QString	updString = update.getQueryText(&dbLink) ;

		if ((qryUpdate = dbLink.qryUpdate(false, updString, objectTab)) == 0)
		{
			pError	= dbLink.lastError() ;
			goto	error ;
		}

		save	= qryUpdate ;
	}


	args[argc++] = KBValue
		       (	buffer,
				bufflen,
				qrySelect->getFieldType(2)
		       )	;

	args[argc++] = KBValue (dstr,	 qrySelect->getFieldType(3)) ;
	args[argc++] = KBValue (docType, qrySelect->getFieldType(4)) ;
	args[argc++] = KBValue (name,	 qrySelect->getFieldType(5)) ;

	if (!docExtn.isEmpty())
		args[argc++] = KBValue (docExtn, qrySelect->getFieldType(6)) ;


	/* Execute the query. This should succeed with a sing row being	*/
	/* affected in either case.					*/
	if (!save->execute (argc, args))
	{	pError	= save->lastError() ;
		goto	error ;
	}

	nRows	= qryInsert != 0 ? qryInsert->getNumRows() : qryUpdate->getNumRows() ;

	if (nRows != 1)
	{
		pError	= KBError
			  (	KBError::Error,
				QString	("Error saving %1"	       ).arg(docType),
				QString	("Unexpectedly updated %1 rows").arg(nRows  ),
				__ERRLOCN
		)	;
		goto error ;
	}

	rv		 = true	;

	error	:
		DELOBJ	(qrySelect) ;
		DELOBJ	(qryInsert) ;
		DELOBJ	(qryUpdate) ;
		return	rv	 ;
}

/*  KBLocation								*/
/*  save	: Save document						*/
/*  _locn	: const QString & : Table location			*/
/*  _name	: const QString & : Document name			*/
/*  buffer	: cchar *	  : Data buffer				*/
/*  bufflen	: uint		  : Buffer length			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::save
	(	const QString	&_locn,
		const QString	&_name,
		cchar		*buffer,
		uint		bufflen,
		KBError		&pError
	)
{
	bool	rv	;
	QString	locn	= _locn.isNull() ? docLocn : _locn ;
	QString	name	= _name.isNull() ? docName : _name ;
	
//	fprintf
//	(	stderr,
//		"KBLocation::save       [%d] [%s][%s][%s][%s]\n",
//		this,
//		(cchar *)docType,
//		(cchar *)docLocn,
//		(cchar *)docName,
//		(cchar *)docExtn
//	)	;


	if (locn == KBLocation::m_pFile)
		rv	= saveToFile (path(name), name, buffer, bufflen, pError) ;
	else	rv	= saveToDB   (locn, 	  name, buffer, bufflen, pError) ;

	if (rv)
	{	docLocn	= locn	;
		docName	= name	;
	}

	KBNotifier::self()->nObjectChanged (*this) ;
	return	rv	;
}

/*  KBLocation								*/
/*  save	: Save document						*/
/*  _locn	: const QString & : Table location			*/
/*  _name	: const QString & : Document name			*/
/*  text	: const QString & : Document text			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::save
	(	const QString	&_locn,
		const QString	&_name,
		const QString	&text,
		KBError		&pError
	)
{
	QCString data = text.utf8() ;
	return	 save (_locn, _name, data.data(), data.length(), pError) ;
}

/*  KBLocation								*/
/*  renameFile	: Rename file object					*/
/*  newName	: const QString & : New name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::renameFile
	(	const QString	&newName,
		KBError		&pError
	)
{
	QString	objPath	= path ()	;
	QString	newPath	= path (newName);

	if (::rename ((cchar *)objPath, (cchar *)newPath) != 0)
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Failed to rename %1 %2 as %3"))
					.arg(docType)
					.arg(docName)
					.arg(newName),
			 	QString	(TR("System error: %1 -> %2: %3"))
					.arg(docName)
					.arg(newName)
					.arg(strerror(errno)),
				__ERRLOCN
			   )	;
		pError.setErrno (errno) ;

		return	false	;
	}

	return	true	;
}

/*  KBLocation								*/
/*  renameDB	: Rename database object					*/
/*  newName	: const QString & : New name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::renameDB
	(	const QString	&newName,
		KBError		&pError
	)
{
	KBDBLink dbLink	 ;
	bool	 exists  ;
	KBValue	 args	 [3]  ;

	if (!dbLink.connect (dbInfo, docLocn, true))
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}

	QString	objectTab = dbLink.rekallPrefix (OBJECTTABNAME) ;

	if (!dbLink.tableExists (objectTab, exists))
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}
	if (!exists) return true ;

	KBSQLUpdate *update ;
	QString qryString = QString("update %1 set %2 = %3 where %4 = %5 and %6 = %7")
				   .arg(dbLink.mapExpression(objectTab))
			  	   .arg(dbLink.mapExpression("Name"   ))
				   .arg(dbLink.placeHolder  (0))
			  	   .arg(dbLink.mapExpression("Name"   ))
				   .arg(dbLink.placeHolder  (1))
			  	   .arg(dbLink.mapExpression("Type"   ))
				   .arg(dbLink.placeHolder  (2))
				   ;

	if ((update = dbLink.qryUpdate(false, qryString, objectTab)) == 0)
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}

	args[0] = QString(newName) ;
	args[1] = QString(docName) ;
	args[2] = QString(docType) ;

	if (!update->execute (3, args))	
	{
		pError	= update->lastError() ;
		delete	update	;
		return	false	;
	}

	delete	update	;
	return	true	;
}

/*  KBLocation								*/
/*  rename	: Rename object						*/
/*  newName	: const QString & : New name				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::rename
	(	const QString	&_name,
		KBError		&pError
	)
{
	if (docLocn == KBLocation::m_pFile)
		return	renameFile (_name, pError) ;
	else	return	renameDB   (_name, pError) ;
}

/*  KBLocation								*/
/*  removeFile	: Remove file object					*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::removeFile
	(	KBError		&pError
	)
{
	QString	objPath	= path () ;

	if (!QFile(objPath).remove())
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("Failed to delete %1"))
					.arg(docName),
		 		QString(TR("System error: %1: %1"   ))
		 			.arg(objPath)
					.arg(strerror(errno)),
				__ERRLOCN
			  )	;
		pError.setErrno	(errno) ;

		return	false	;
	}

	return	true	;
}

/*  KBLocation								*/
/*  removeDB	: Remove database object				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::removeDB
	(	KBError		&pError
	)
{
	KBDBLink dbLink	 ;
	bool	 exists  ;
	KBValue	 args	 [2]  ;

	if (!dbLink.connect (dbInfo, docLocn, true))
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}

	QString	objectTab = dbLink.rekallPrefix (OBJECTTABNAME) ;

	if (!dbLink.tableExists (objectTab, exists))
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}
	if (!exists) return true ;

	KBSQLDelete *delrec ;

	QString qryString = QString("delete from %1 where %2 = %3 and %4 = %5")
				   .arg(dbLink.mapExpression(objectTab))
				   .arg(dbLink.mapExpression("Name"   ))
				   .arg(dbLink.placeHolder  (0	      ))
				   .arg(dbLink.mapExpression("Type"   ))
				   .arg(dbLink.placeHolder  (1	      ))
				   ;

	if ((delrec = dbLink.qryDelete
			(	false,
				qryString,
				objectTab
			)) == 0)
	{
		pError	= dbLink.lastError() ;
		return	false ;
	}

	args[0] = QString(docName) ;
	args[1] = QString(docType) ;

	if (!delrec->execute (2, args))	
	{
		pError	= delrec->lastError() ;
		delete	delrec	;
		return	false	;
	}

	delete	delrec	;
	return	true	;
}

/*  KBLocation								*/
/*  remove	: Remove object						*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBLocation::remove
	(	KBError		&pError
	)
{
	if (docLocn == KBLocation::m_pFile)
		return	removeFile (pError) ;
	else	return	removeDB   (pError) ;
}


/*  ------------------------------------------------------------------  */

/*  kbXMLEncoding: Return XML encoding used				*/
/*  (returns)	 : QString	: Encoding				*/

LIBCOMMON_API QString	kbXMLEncoding ()
{
	return	"UTF-8"	;
}

