/***************************************************************************
    file	         : cogen.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<fcntl.h>
#include	<setjmp.h>
#include	<std.h>

#ifdef		_WIN32
#include	<io.h>
#else
#include	<unistd.h>
#endif

#include	"eli.h"
#include	"interp.h"
#include	"syn.h"
#include	"code.h"

#define		PEEPHOLE

LVAR	int	cobuf[10000]	;	/* Code buffer			*/
LVAR	int	coptr		;	/* Current buffer offset	*/
LVAR	int	cobase		;	/* Current function base	*/
LVAR	int	peepb		;	/* Peephole base		*/

LFUNC   void	cg_expr  (ENODE *)	;
LFUNC   void	cg_stmt  (STMT  *)	;
LFUNC   void	cg_assn  (ENODE *)	;
LFUNC   void	cg_call  (ENODE *, ENODE *) ;
LFUNC   void	cg_method(ENODE *, NAME *, ENODE *) ;
LFUNC   void	cg_comma (ENODE *)	;

LVAR    int     lastret 	;
LVAR	int	nlidx		;
LVAR	int	slidx		;

LVAR	CBUFF	*outbuf		;
GVAR	FILE	*_el_errout	;

#define	OPER(i)		((cobuf[i]&OPBITS)>>OPSHIFT)
#define	ARGV(i)		((cobuf[i]&ARBITS))
#define	INST(o,a)	(((o)<<OPSHIFT)|((a)&ARBITS))

LFUNC	int	cg_write
	(	void	*buf,
		int	len
	)
{
	if (outbuf == 0)
	{
		outbuf	= (CBUFF *) malloc  (sizeof(CBUFF) + len) ;
		outbuf->len = 0 ;
	}
	else
		outbuf	= (CBUFF *) realloc (outbuf, sizeof(CBUFF) + outbuf->len + len) ;

	memcpy	(&outbuf->code[outbuf->len], buf, len) ;
	outbuf->len += len ;

	return	len	;
}

/*G _el_outn	: Output opcode with numerical argument			*/
/*  op		: int		: Opcode				*/
/*  num		: int		: Argument				*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_outn
	(	int	op,
		int	num
	)
{
	int	op2	;

#ifdef	PEEPHOLE
	switch (op)
	{			
		case C_RETN :
			/* A pop immediately before a return can be	*/
			/* ignored as the return will pop the stack as	*/
			/* far as needed.				*/
			if (coptr < peepb + 1) break ;
			if (OPER(coptr-1) == C_POP) coptr -= 1 ;
			break	;

		case C_LBLK :
		case C_LOUT :
			/* The sequence SBLK(n)-POP(1)-LBLK(n) (and	*/
			/* similarly for SOUT...LOUT) can have the	*/
			/* latter two instructions removed for the same	*/
			/* effect.					*/
			op2 = op == C_LBLK ? C_SBLK : C_SOUT ;
			if (coptr < peepb + 2) break ;
			if ( ((int)OPER(coptr-1) ==  C_POP) &&
			     ((int)ARGV(coptr-1) ==      1) &&
			     ((int)OPER(coptr-2) ==    op2) &&
			     ((int)ARGV(coptr-2) ==    num) )
			{
				coptr	-= 1 ;
				return	;
			}
			break ;

		case C_POP  :
			/* A POP(1) immediately after CALL instruction	*/
			/* can be replaced with a CALLV instruction	*/
			/* which does not store a result on the stack.	*/
			if (coptr < peepb + 1) break ;
			if ((num == 1) && (OPER(coptr-1) == C_CALL))
			{	cobuf[coptr-1] = INST(C_CALLV,ARGV(coptr-1)) ;
				return	;
			}
			break	;

		default	    :
			break	;
	}

	if (coptr >= peepb + 1)
	{	/* A SBLK or SOUT followed by a pop can be replaced by	*/
		/* a corresponding SBLKP or SOUTP which does the pop.	*/
		/* Note that we check this here since we do not want to	*/
		/* muck up the SBLK(n)-POP-LBLK(n) optimisation above.	*/
		if	( (OPER(coptr-2) == C_SBLK) &&
			  (OPER(coptr-1) == C_POP ) )
		{	cobuf[coptr-2] = INST(C_SBLKP,ARGV(coptr-2)) ;
			coptr -= 1 ;
		}
		else if ( (OPER(coptr-2) == C_SOUT) &&
			  (OPER(coptr-1) == C_POP ) )
		{	cobuf[coptr-2] = INST(C_SOUTP,ARGV(coptr-2)) ;
			coptr -= 1 ;
		}
	}

#endif	/* PEEPHOLE */

	/* If the number has any opcode bit set, or is equal to the	*/
	/* numerical escape value 0x0fff, then output two words, with	*/
	/* the numerical value in the second. Otherwise, the opcode and	*/
	/* the number can be packed into one.				*/
	if (((num & OPBITS) != 0) || (num == ARBITS))
	{	cobuf[coptr++] = INST(op,ARBITS) ;
		cobuf[coptr++] = num   ;
		peepb	       = coptr ;
	}
	else	cobuf[coptr++] = INST(op,num) ;

	/* If the opcode indicates the start of a new function then set	*/
	/* the function offset to minus one. Hence, the function code	*/
	/* starts just after the length word.				*/
	if ((op == C_FDEF) || (op == C_PDEF)) cobase = coptr + 1 ;
}

/*G _el_outd	: Output opcode with double argument			*/
/*  op		: int		: Opcode				*/
/*  dbl		: double	: Argument				*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_outd
	(	int	op,
		double	dbl	
	)
{
	int	*dp	= (int *)&dbl ;
	int	idx	= sizeof(double)/sizeof(int) ;

	cobuf[coptr++] = INST(op,0) ;
	while (idx > 0)
	{	cobuf[coptr++] = *dp	;
		dp	+= 1 ;
		idx	-= 1 ;
	}
}

/*G _el_here	: Get current code generation position			*/
/*  (returns)	: int		: Position				*/

GFUNC	int	_el_here ()
{
	peepb	= coptr	;	/* Must not peephole over label		*/
	lastret	= 0	;	/* Last instruction not a return	*/

	/* The return value is actually the offset into the function	*/
	/* which is currently being defined.				*/
	return	coptr - cobase	;
}

/*G _el_fixn	: Fix up numerical value in instruction			*/
/*  pos		: int		: Offset into current function		*/
/*  val		: int		: Fix-up value				*/
/*  (returns)	: int		: Old value				*/

GFUNC	int	_el_fixn
	(	int	pos,
		int	val
	)
{
	int	old	= cobuf[cobase + pos] & ARBITS ;

	/* Check that the value can be fixed, as it will be impossible	*/
	/* to escape in a value.					*/
	if (((val & OPBITS) != 0) || (val == ARBITS))
	{	fprintf	(_el_errout, "el: fixup not possible\n") ;
		exit	(1) ;
	}

	/* Replace the argument bits with the new value, and return the	*/
	/* old one.							*/
	cobuf[cobase + pos] = (cobuf[cobase + pos] & OPBITS) | val ;
	return	old	;
}

/*L cg_outs	: Output names or strings from list			*/
/*  nptr	: NAME *	: List pointer				*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_outs
	(	NAME	*nptr
	)
{
	int	buff[64];

	for ( ; nptr != NULL ; nptr = nptr->next)
	{
		/* Don't output this name if it is not referenced.	*/
		if (!nptr->ref) continue ;

		/* If this is a local function or variable, then modify	*/
		/* the name with the module name; similarly for	publics	*/
		if 	((nptr->type & N_SCOPE) == N_LOCAL )
			sprintf ((char *)&buff[2], "%s$%s",  _el_lname, nptr->name) ;
		else if ((nptr->type & N_SCOPE) == N_PUBLIC)
			sprintf ((char *)&buff[2], "%s::%s", _el_mname, nptr->name) ;
		else	strcpy	((char *)&buff[2], nptr->name) ;

		buff[0]	= (C_NDEF << OPSHIFT) | (nptr->type & 0xff) ;		
		buff[1]	= (strlen ((char *)&buff[2]) + 2) & ~1 ;

		if (cg_write (buff, 2 * sizeof(int) + buff[1]) !=
				      (int)(2 * sizeof(int) + buff[1]))
			errorE	("el: failed to write name or string: %e\n") ;
	}
}

/*L cg_outl	: Output list lengths					*/
/*  op		: int		: Opcode				*/
/*  llen	: int		: List length				*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_outl
	(	int	op,
		int	llen
	)
{
	int	word	;

	word	= (op << OPSHIFT) | llen ;
	if (cg_write (&word, sizeof(int)) != sizeof(int))
		errorE ("elc: failed to write list length: %e\n") ;
}

/*G _el_cgnode	: Finish code generation				*/
/*  fname	: const char *	: Output file name			*/
/*  (returns)	: CBUFF *	: Code buffer				*/

GFUNC	CBUFF	*_el_cgdone
	(	const char	*fname
	)
{
	long	magic	= MAGIC ;

	if (outbuf != 0)
	{	free ((void *)outbuf) ;	
		outbuf	= 0 ;
	}

	if (cg_write (&magic, sizeof(long)) != sizeof(long))
		errorE	("elc: unable to write to \"%s\": %e\n", fname) ;

	/* Output the numbers of outer names and strings, followed by	*/
	/* the name and string tables themselves.			*/
	cg_outl  (C_NOUT, nlidx) ;
	cg_outl  (C_NSTR, slidx) ;
	cg_outs  (_el_nlist) ;
	cg_outs  (_el_slist) ;

	/* Output the buffered code to the file and the close it.	*/
	if (cg_write (cobuf, coptr * sizeof(int)) != (int)(coptr * sizeof(int)))
		errorE	("el: failed to write code to \"%s\": %e\n", fname) ;

	if (fname != 0)
	{
		int fid = open (fname, O_WRONLY|O_CREAT|O_TRUNC, 0666) ;

		if (fid < 0)
			errorE	("elc: unable to open \"%s\": %e\n", fname) ;

		if (write (fid, outbuf->code, outbuf->len) != outbuf->len)
		{	close	(fid) ;
			errorE	("el: failed to flush code to \"%s\": %e\n", fname) ;
		}

		if (close (fid) < 0)
			errorE	("el: error closing \"%s\": %e\n", fname) ;
	}

	CBUFF	*res	= outbuf ;
	outbuf	= 0	;
	return	res	;
}

/*G _el_cogen   : Code generate                                         */
/*  (returns)   : void          :                                       */

GFUNC   void    _el_cogen ()
{
	FUNC    *fptr   ;
	NAME	*nptr	;

	coptr	= 0	;
	cobase	= 0	;
	peepb	= 0	;
	lastret	= 0	;
	nlidx	= 0	;
	slidx	= 0	;

	for (nptr = _el_nlist ; nptr != NULL ; nptr = nptr->next)
		if (nptr->ref)
			nptr->idx = nlidx++ ;
	for (nptr = _el_slist ; nptr != NULL ; nptr = nptr->next)
		nptr->idx = slidx++ ;

	for (fptr = _el_flist ; fptr != NULL ; fptr = fptr->next)
	{
		int     lpos    ;
		int     npars   ;

		/* Output the function definition marker and then note  */
		/* the position where the function length must be fixed */
		/* up. Output an empty length instruction.              */
		if ((fptr->name->type & N_SCOPE) == N_PUBLIC)
			_el_outn (C_PDEF, fptr->name->idx) ;
		else	_el_outn (C_FDEF, fptr->name->idx) ;
		lpos     = _el_here () ;
		_el_outn (C_LEN,  0) ;

		_el_outn (C_AT, (fptr->body->lno << 12) | (_el_file->idx & 0x0fff)) ;

		/* Output the argument count checker. This is also the  */
		/* initial block size. Only set the block size if it is */
		/* different.                                           */
		npars    = fptr->pars == NULL ? 0 : fptr->pars->idx + 1 ;
		_el_outn (C_ARGC, npars) ;
		if (npars != fptr->size) _el_outn (C_BLKS, fptr->size) ;

		/* Generate the code for the function body, and then    */
		/* a return if the last statement in the function was   */
		/* not one.                                             */
		lastret = 0 ;
		cg_stmt  (fptr->body) ;
		if (!lastret) _el_outn (C_RETN, 0) ;

		/* Fix the length. Note that the difference between the */
		/* current code position and the length instruction is  */
		/* decremented by one, since the length instruction is  */
		/* not counted as part of the function code.            */
		_el_fixn (lpos, _el_here () - lpos - 1) ;
	}
}


/*L cg_patch	: Patch branch chain					*/
/*  chain	: int		: Chain pointer				*/
/*  locn	: int		: Branch target location		*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_patch
	(	int	chain,
		int	locn
	)
{
	while (chain != 0)
		chain	= _el_fixn (chain, locn) ;
}

/*L cg_cond	: Compile conditional branch expression			*/
/*  expr	: ENODE *	: Expression to compile			*/
/*  btrue	: int		: Non-zero -> branch if true		*/
/*  patch	: int		: Patch branch chain			*/
/*  (returns)	: int		: Extended patch branch chain		*/

LFUNC	int	cg_cond
	(	ENODE	*expr,
		int	btrue,
		int	patch
	)
{
	int	addr	;
	int	skip	;
	ENODE	*left	;
	ENODE	*right	;
	int	op	;

	/* A non-expression is always considered to be true, so an	*/
	/* unconditional branch is generated if branch-on-true was	*/
	/* requested.							*/
	if (expr == NULL)
	{	if (btrue)
		{	addr	= _el_here ()   ;
			_el_outn (C_BRA, patch) ;
			patch	= addr		;
		}
		return	patch ;
	}

	/* If the top level of the expression is ! then drop this and	*/
	/* invert the condition.					*/
	while ((expr->tag == E_EXPR) && (expr->val.expr.op == O_NOT))
	{	expr	= expr->val.expr.left	;
		btrue	= !btrue		;
	}

	op	= expr->val.expr.op	;
	left	= expr->val.expr.left	;
	right	= expr->val.expr.right	;

	/* Special cases are the && and || operators. These need not be	*/
	/* compiled to generate results, simply to load each operand in	*/
	/* turn and perform an appropriate branch.			*/
	if	((expr->tag == E_EXPR) && (op == O_ANDIF))
	{
		if (btrue)
		{	skip  = cg_cond (left,  0, 0    ) ;
			patch = cg_cond (right, 1, patch) ;
			cg_patch (skip, _el_here ()) ;
		}
		else
		{	patch = cg_cond (left,  0, patch) ;
			patch = cg_cond (right, 0, patch) ;
		} 
		
	}
	else if ((expr->tag == E_EXPR) && (op == O_ORIF ))
	{
		if (btrue)
		{	patch = cg_cond (left,  1, patch) ;
			patch = cg_cond (right, 1, patch) ;
		}
		else
		{	skip  = cg_cond (left,  1, 0    ) ;
			patch = cg_cond (right, 0, patch) ;
			cg_patch (skip, _el_here ()) ;
		}
	}
	/* We can also deal specially with the == and != operators if	*/
	/* one of the operands is zero. Actually, just test for the	*/
	/* second, since this is almost always the way it is written.	*/
	else if ( (expr->tag == E_EXPR) && (op == O_EQ  ) &&
					   (right->tag	    == E_NUMB) &&
					   (right->val.numb == 0     ) )
	{
		cg_expr	(left)		;
		addr	= _el_here ()	;
		_el_outn  (btrue ? C_BRAF : C_BRAT, patch) ;
		patch	= addr		;
	}
	else if ( (expr->tag == E_EXPR) && (op == O_NEQ ) &&
					   (right->tag	    == E_NUMB) &&
					   (right->val.numb == 0     ) )
	{
		cg_expr	(left)		;
		addr	= _el_here ()	;
		_el_outn  (btrue ? C_BRAT : C_BRAF, patch) ;
		patch	= addr		;
	}
	else
	{
		cg_expr	 (expr)		;
		addr	 = _el_here ()	;
		_el_outn (btrue ? C_BRAT : C_BRAF, patch) ;
		patch	 = addr		;
	}

	return	patch	;
}

/*L cg_popexp	: Compile expression and discard result			*/
/*  expr	: ENODE *	: Expression (or NULL if none)		*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_popexp
	(	ENODE	*expr
	)
{
	if (expr != NULL)
	{	cg_expr	 (expr) ;
		_el_outn (C_POP, 1) ;
	}
}

/*L cg_call     : Compile function call                                 */
/*  func	: ENODE *	: Function				*/
/*  args        : ENODE *       : Arguments (may be NULL if none)       */
/*  (returns)   : void		:					*/

LFUNC   void	cg_call
	(	ENODE	*func,
		ENODE   *args  
	)
{
	int     nargs   = 0 ;

	/* If there are any arguments, compile each expression, keeping */
	/* count of how many there are.                                 */
	if (args != NULL)
	{       for ( ; (args->tag         == E_EXPR ) &&
			(args->val.expr.op == O_COMMA) ; args = args->val.expr.right)
		{
			cg_expr (args->val.expr.left) ;
			nargs   += 1 ;
		}
		cg_expr (args)  ;
		nargs   += 1    ;
	}

	/* Now compile the expression which specifies the function. In	*/
	/* most cases this should turn into a simple L_OUT opcode. The	*/
	/* call opcode can then be output.				*/
	cg_expr  (func)  ;
	_el_outn (C_CALL, nargs) ;
}

/*L cg_method	: Compile method invocation				*/
/*  object	: ENODE *	: Object expression			*/
/*  method	: NAME *	: Method name				*/
/*  args        : ENODE *       : Arguments (may be NULL if none)       */
/*  (returns)   : void		:					*/

LFUNC   void	cg_method
	(	ENODE	*object,
		NAME	*method,
		ENODE   *args  
	)
{
	int     nargs   = 0 ;

	/* First compile code for the expression that generates the	*/
	/* object. This goes first since it will be the first argument	*/
	/* to the method.						*/
	cg_expr  (object)  ;

	/* If there are any arguments, compile each expression, keeping */
	/* count of how many there are.                                 */
	if (args != NULL)
	{       for ( ; (args->tag         == E_EXPR ) &&
			(args->val.expr.op == O_COMMA) ; args = args->val.expr.right)
		{
			cg_expr (args->val.expr.left) ;
			nargs   += 1 ;
		}
		cg_expr (args)  ;
		nargs   += 1    ;
	}

	/* Lastly generate code to load the method name, and to do the	*/
	/* actual invocation.						*/
	_el_outn (C_LSTR, method->idx) ;
	_el_outn (C_METH, nargs)       ;
}

/*L cg_stvar    : Code generate to store to variable                    */
/*  var         : NAME *        : Variable                              */
/*  lno		: int		: Line number				*/
/*  (returns)   : void		:					*/

LFUNC   void	cg_stvar
	(	NAME    *var,
		int	lno
	)
{
	switch (var->type & 0xf0)
	{
		case N_LOCAL  :
		case N_GLOBAL :
		case N_PUBLIC :
			_el_outn (C_SOUT, var->idx) ;
			break ;

		case N_BLOCK  :
			_el_outn (C_SBLK,  var->idx) ;
			break	;

		default       :
			fprintf (_el_errout, "elc: %d: vtype %s %02x\n",
					 lno, var->name, var->type) ;
			exit    (1) ;
	}
}

/*L cg_assn     : Compile assignment expression                         */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void		:					*/

LFUNC   void	cg_assn
	(	ENODE   *expr  
	)
{
	ENODE   *vare   = expr->val.expr.left ;

	cg_expr (expr->val.expr.right) ;

	if (vare->tag == E_VAR) 
	{       /* If the left-hand side if simply a variable name then */
		/* just compile the instruction to store the result.    */
		cg_stvar (vare->val.var, expr->lno) ;
		return  ;
	}

	if ((vare->tag == E_EXPR) && (vare->val.expr.op == O_SUBL))
	{       /* If the left is a subscript expression then compile   */
		/* this to give a pointer at the vector slot. Then      */
		/* store the value and pop the pointer.                 */
		cg_expr  (vare) ;
		_el_outn (C_SIND, 0) ;
		return	;
	}

	/* Anything else should have been trapped at the syntax         */
	/* analysis stage, so is a fatal error here.                    */
	fprintf (_el_errout, "elc: assignment lsh neither name/subscript\n") ;
	exit    (1) ;
}

/*L cg_incdec	: Check binary right-hand-side for increment/decrement	*/
/*  expr	: ENODE *	: Expression right-hand-side		*/
/*  op		: int		: Operator				*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_incdec
	(	ENODE	*expr,
		int	op
	)
{
	if ((expr->tag == E_NUMB) && (op == O_PLUS ))
	{	_el_outn  (C_INCR, expr->val.numb) ;
		return	;
	}

	if ((expr->tag == E_NUMB) && (op == O_MINUS))
	{	_el_outn  (C_DECR, expr->val.numb) ;
		return	;
	}

	cg_expr  (expr) ;
	_el_outn (C_OPER, op) ;
}

/*L cg_assop    : Compile an operator-assign                            */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void		:					*/

LFUNC   void	cg_assop
	(	ENODE   *expr
	)
{
	ENODE   *left   = expr->val.expr.left   ;
	ENODE   *right  = expr->val.expr.right  ;
	int	op	= expr->val.expr.op	;

	if (left->tag == E_VAR)
	{       /* If the left hand side is a variable, then load its   */
		/* value, compile the right hand side and then apply    */
		/* the equivalent non-assign operator. The result is    */
		/* then stored back in the variable.                    */
		cg_expr   (left ) ;
		cg_incdec (right, op - O_APLUS + O_PLUS) ;
		cg_stvar  (left->val.var, expr->lno) ;
		return	  ;
	}

	if ((left->tag == E_EXPR) && (left->val.expr.op == O_SUBL))
	{       /* If the left hand side is a subscript then:           */
		/* (a)  calculate address of value                      */
		/* (b)  duplicate address and load current value        */
		/* (c)  compile right hand side                         */
		/* (d)  perform equivalent non-assign operator          */
		/* (e)  swap address and new value                      */
		/* (f)  store new value                                 */
		cg_expr   (left ) ;
		_el_outn  (C_DUP,  0) ;
		_el_outn  (C_LIND, 0) ;
		cg_incdec (right, op - O_APLUS + O_PLUS) ;
		_el_outn  (C_SWAP, 0) ;
		_el_outn  (C_SIND, 0) ;
		return ;
	}

	/* Anything else should have been trapped at the syntax         */
	/* analysis stage, so is a fatal error here.                    */
	fprintf (_el_errout, "elc: assignment-op lsh neither name/subscript\n") ;
	exit    (1) ;
}

/*L cg_comma    : Compile comma expression                              */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void		:					*/

LFUNC   void	cg_comma
	(	ENODE   *expr
	)
{
	/* This is handled specially as the result of the left-hand     */
	/* operand is thrown away.                                      */
	cg_popexp (expr->val.expr.left ) ;
	cg_expr	  (expr->val.expr.right) ;
}

/*L cg_andif    : Compile the && and-if operator                        */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void          :                                       */

LFUNC   void    cg_andif
	(	ENODE   *expr
	)
{
	int     fixf1   ;
	int     fixf2   ;

	/*              <left>                                          */
	/*              C_BRAF  FALSE                                   */
	/*              <right>                                         */
	/*              C_BRAF  FALSE                                   */
	/*              C_LNUM  1                                       */
	/*              C_BRA   DONE                                    */
	/*      FALSE:  C_LNUM  0                                       */
	/*      DONE:                                                   */
	cg_expr  (expr->val.expr.left ) ;
	fixf1  = _el_here () ;
	_el_outn (C_BRAF, 0) ;
	cg_expr  (expr->val.expr.right) ;
	fixf2  = _el_here () ;
	_el_outn (C_BRAF, 0) ;
	_el_outn (C_LNUM, 1) ;
	_el_outn (C_BRA, _el_here () + 2) ;
	_el_fixn (fixf1, _el_here ()) ;
	_el_fixn (fixf2, _el_here ()) ;
	_el_outn (C_LNUM, 0) ;
}

/*L cg_orif     : Compile the || or-if operator                         */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void          :                                       */

LFUNC   void    cg_orif
	(	ENODE   *expr
	)
{
	int     fixt1   ;
	int     fixt2   ;

	/*              <left>                                          */
	/*              C_BRAT  TRUE                                    */
	/*              <right>                                         */
	/*              C_BRAT  TRUE                                    */
	/*              C_LNUM  0                                       */
	/*              C_BRA   DONE                                    */
	/*      TRUE:   C_LNUM  1                                       */
	/*      DONE:                                                   */
	cg_expr  (expr->val.expr.left ) ;
	fixt1  = _el_here () ;
	_el_outn (C_BRAT, 0) ;
	cg_expr  (expr->val.expr.right) ;
	fixt2  = _el_here () ;
	_el_outn (C_BRAT, 0) ;
	_el_outn (C_LNUM, 0) ;
	_el_outn (C_BRA, _el_here () + 2) ;
	_el_fixn (fixt1, _el_here ()) ;
	_el_fixn (fixt2, _el_here ()) ;
	_el_outn (C_LNUM, 1) ;
}

LFUNC	void	cg_query
	(	ENODE	*expr
	)
{
	/* Basic compiled form is:					*/
	/*		<cond>						*/
	/*		BRAF	false					*/
	/*		<left>						*/
	/*		BRA	true					*/
	/*	false:	<right>						*/
	/*	true:							*/

	int	falsej	;
	int	truej	;

	falsej	= cg_cond  (expr->val.expr.query, 0, 0) ;
	cg_expr  (expr->val.expr.left ) ;
	truej	= _el_here ()  ;
	_el_outn  (C_BRA, 0) ;
	cg_patch  (falsej,  _el_here ()) ;
	cg_expr  (expr->val.expr.right) ;
	_el_fixn  (truej,   _el_here ()) ;
}

/*L cg_expr     : Compile expression                                    */
/*  expr        : ENODE *       : Expression                            */
/*  (returns)   : void		:					*/

LFUNC   void	cg_expr
	(	ENODE   *expr 
	)
{
	ENODE	*left	;
	ENODE	*right	;
	NAME	*vart	;
	int	op	;

	switch (expr->tag)
	{
		case E_VAR      :
			/* Variable to be loaded. Note that this	*/
			/* cannot be the variable on the left of an	*/
			/* assign as this case is handled specially.	*/
			vart	     = expr->val.var ;

			switch (vart->type & N_SCOPE)
			{
				case N_GLOBAL :
				case N_LOCAL  :
				case N_PUBLIC :
					_el_outn (C_LOUT, vart->idx) ;
					break ;

				case N_BLOCK  :
					_el_outn (C_LBLK,  vart->idx) ;
					break	;

				default       :
					fprintf  (_el_errout, "el: vtype %s %04x\n",
							  vart->name,
							  vart->type) ;
					exit     (1) ;
			}

			return	;

		case E_NUMB     :
			_el_outn (C_LNUM, expr->val.numb) ;
			return	;

		case E_DBL     :
			_el_outd (C_LDBL, expr->val.dbl) ;
			return	;

		case E_STRING   :
			_el_outn (C_LSTR, expr->val.str->idx) ;
			return	;

		case E_EXPR     :
		left	= expr->val.expr.left	;
		right	= expr->val.expr.right	;

		switch (op = expr->val.expr.op)
		{
			case O_ASSIGN   : cg_assn  (expr) ; return ;
			case O_APLUS    :
			case O_AMINUS   :
			case O_AMULT    :
			case O_ADIV     :
			case O_AREM     :
			case O_AAND     :
			case O_AOR      :
			case O_AXOR     : cg_assop (expr) ; return ;
			case O_COMMA    : cg_comma (expr) ; return ;
			case O_ANDIF    : cg_andif (expr) ; return ;
			case O_ORIF     : cg_orif  (expr) ; return ;
			case O_QUERY	: cg_query (expr) ; return ;

			case O_VEC	: cg_expr  (left) ;
					  _el_outn (C_OPER, op) ;
					  return ;
			case O_HASH	: cg_expr  (left) ;
					  _el_outn (C_OPER, op) ;
					  return ;

			default         :
				/* There is always an LHS. If there is	*/
				/* an RHS then check this for increment	*/
				/* and decrement, otherwise output the	*/
				/* (unary) operator.			*/
				cg_expr  (left) ;
				if (right == NULL)
					_el_outn  (C_OPER, op) ;
				else	cg_incdec (right,  op) ;
				return	;
		}
		break   ;

		case E_CALL     :
			/* Call is handled separately as the argument   */
			/* expression is handled specially.             */
			cg_call (expr->val.call.func, expr->val.call.alist) ;
			return	;

		case E_METHOD	:
			/* Similarly for method invocation		*/
			cg_method (expr->val.method.object,
				   expr->val.method.method,
				   expr->val.method.alist) ;
			return	;

		default         :
			fprintf (_el_errout, "el: etag %d\n", expr->tag) ;
			exit    (1) ;
	}
}

/*L cg_init     : Generate code for initialised variables               */
/*  vlist       : NAME *        : Variable list                         */
/*  blk         : STMT *        : Block being initialised               */
/*  (returns)   : void          :                                       */

LFUNC   void    cg_init
	(	NAME    *vlist,
		STMT    *blk
	)
{
	ENODE   expr    ;
	ENODE   varn    ;

	if (vlist == NULL) return ;

	cg_init (vlist->next, blk) ;

	if ((vlist->blk != blk) || (vlist->val == NULL)) return ;

	varn.tag            = E_VAR     ;
	varn.val.var        = vlist     ;
	expr.tag            = E_EXPR    ;
	expr.val.expr.op    = O_ASSIGN  ;
	expr.val.expr.left  = &varn     ;
	expr.val.expr.right = vlist->val;

	cg_popexp (&expr) ;
}

/*L cg_iter	: Compile an iterator					*/
/*  stmt	: STMT *	: Iterator statement			*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_iter
	(	STMT	*stmt
	)
{
	int	bcond	;

	stmt->val.iter.breakp	= 0 ;
	stmt->val.iter.contp	= 0 ;

	/* Iterator. The basic form of the code is as below. Continue	*/
	/* and break branches are handled by back-patching.		*/
	/*              	<init>					*/
	/*              	C_POP	1				*/
	/*			BRA	COND				*/
	/*	LOOP:		<body>					*/
	/*	CONT:		<iter>					*/
	/*              	C_POP   1				*/
	/*	COND:		<cond>					*/
	/*			C_BRAT	LOOP				*/
	/*	BREAK:							*/

	/* If there is an initialiser expression then compile that	*/
	/* first.							*/
	cg_popexp (stmt->val.iter.init) ;

	/* Note the position of the initial 'branch to condition' jump	*/
	/* and insert the branch. The body of the loop follows		*/
	/* immediately after.						*/
	bcond	= _el_here () ;
	_el_outn (C_BRA, 0)  		 ;
	cg_stmt  (stmt->val.iter.body)	 ;

	/* Patch a chain of 'continue' jumps to refer to the iterate	*/
	/* (if any) which follows.					*/
	cg_patch  (stmt->val.iter.contp, _el_here ()) ;
	cg_popexp (stmt->val.iter.iter)	;

	_el_fixn  (bcond, _el_here ())	 ;

	/* Conpile the condition, looping back if the condition is true	*/
	/* (of if there is no actual condition).			*/
	cg_patch (cg_cond (stmt->val.iter.cond, 1, 0), bcond + 1)		 ;
	cg_patch (stmt->val.iter.breakp, _el_here ())		 ;

	lastret = 0		 ;
}

/*L cg_ifthen	: Compile an if-then-else statement			*/
/*  stmt	: STMT *	: If-then-else statement		*/
/*  (returns)	: void		:					*/

LFUNC	void	cg_ifthen
	(	STMT	*stmt
	)
{
	/* Conditional. The basic form of the code is:  		*/
	/*              <cond>                          		*/
	/*              C_BRAF  ELSEP                   		*/
	/*              <thenp>                         		*/
	/*              C_BRA   ENDP                    		*/
	/*      ELSEP:                                  		*/
	/*              <elsep>                         		*/
	/*      ENDP:                                   		*/
	int     elsej   ;
	int     thenj   ;

	/* Generate code for the expression, noting the	position for	*/
	/* the conditional branch so that it can be patched when the	*/
	/* location of the else part is known, and output the branch	*/
	/* instruction.							*/
	elsej	= cg_cond (stmt->val.ifthen.cond, 0, 0) ;

	/* Generate code for the then part. We assume that there always	*/
	/* will be one ...						*/
	cg_stmt	(stmt->val.ifthen.thenp) ;

	/* If there is no else part then just patch the else jump.	*/
	/* There is then no more to compile.				*/
	if (stmt->val.ifthen.elsep == NULL)
	{	cg_patch (elsej, _el_here ()) ;
		return ;
	}

	/* Note the position for the branch past the else part and	*/
	/* output the branch. Then branch to the else part can then be	*/
	/* patched up.      						*/
	thenj    = _el_here () ;
	_el_outn (C_BRA, 0) ;
	cg_patch (elsej, _el_here ()) ;

	/* Output the code for the else part and then fix up the jump	*/
	/* from then end of the then part.				*/
	cg_stmt  (stmt->val.ifthen.elsep) ;
	_el_fixn (thenj, _el_here ()) ;

	lastret = 0 ;
}

/*L cg_switch	: Code generate for switch statement			*/
/*  stmt        : STMT *        : Switch statement			*/
/*  (returns)   : void          :                                       */

LFUNC   void    cg_switch
	(	STMT    *stmt
	)
{
	CASE	*cases	;
	int	lastp	= 0 ;
	int	deflp	= 0 ;
	int	skipp	= 0 ;

	stmt->val.swcase.endp = 0 ;

	/* Initially code-generate for the switch expression.		*/
	cg_expr (stmt->val.swcase.value) ;

	for (cases = stmt->val.swcase.cases ; cases != NULL ;
					      cases  = cases->next)
	{
		/* If this is the default case then note where we are	*/
		/* and just code generate the statement. This maintains	*/
		/* the correct order for fall-though to work.		*/
		if (cases->value == NULL)
		{	deflp	= _el_here () ;
			cg_stmt (cases->stmt) ;
			skipp	= _el_here () ;
			_el_outn (C_BRA, 0)   ;
			continue ;
		}

		/* Lastp, if non-zero, refers to the C_BRAF of the	*/
		/* previous case. This will need fixing to the current	*/
		/* case, at this address.				*/
		if (lastp != 0) _el_fixn (lastp, _el_here ()) ;

		/* Duplicate the switch value, so that it is not	*/
		/* destroyed by the case comparisom. Then compare and	*/
		/* branch if not equal, noting the branch instruction	*/
		/* address for the next case.				*/
		_el_outn (C_DUP, 0)	;
		cg_expr  (cases->value) ;
		_el_outn (C_OPER, O_EQ) ;
		lastp	 = _el_here  ()	;
		_el_outn (C_BRAF, 0   ) ;

		/* If there was a previous case (or default) then	*/
		/* skipp will contain the address of the branch at the	*/
		/* end of it past this case.				*/
		if (skipp != 0) _el_fixn (skipp, _el_here ()) ;

		/* Now we can code-generate for the statement.		*/
		cg_stmt	 (cases->stmt) ;
		skipp	 = _el_here () ;
		_el_outn (C_BRA, 0)    ;
	}

	/* Patch the last case skip branches, and pop the stack to get	*/
	/* rid of the case value.					*/
	if (skipp != 0) _el_fixn (skipp, _el_here ()) ;
	if (lastp != 0) _el_fixn (lastp, _el_here ()) ;
	_el_outn (C_POP, 1) ;

	/* If there is a default then jump back to it, since we arrive	*/
	/* here if no case was accepted. Then patch eny endcase jumps.	*/
	if (deflp != 0) _el_outn (C_BRA, deflp) ;

	cg_patch (stmt->val.swcase.endp, _el_here ()) ;
}

/*L cg_stmt     : Code generate for statement list                      */
/*  stmt        : STMT *        : Statement list                        */
/*  (returns)   : void          :                                       */

LFUNC   void    cg_stmt
	(	STMT    *stmt
	)
{
	int	posn	;

	for ( ; stmt != NULL ; stmt = stmt->next)
	{

	_el_outn (C_AT, (stmt->lno << 12) | (_el_file->idx & 0x0fff)) ;

	switch (stmt->tag)
	{
		case S_NULLS    :
			break   ;

		case S_BLOCK    :
			lastret = 0 ;
			cg_init (stmt->val.block.vars, stmt) ;
			cg_stmt (stmt->val.block.stmt) ;
			break   ;

		case S_STAND    :
			lastret  = 0 ;
			cg_popexp (stmt->val.expr) ;
			break   ;

		case S_RETURN   :
			/* If there is no return expression then return */
			/* zero; if the expression is a number then	*/
			/* return that directly. Otherwise, compile	*/
			/* the expression and return with that.		*/
			if	(stmt->val.rete == NULL)
				_el_outn (C_RETN, 0) ;
			else if (stmt->val.rete->tag == E_NUMB)
				_el_outn (C_RETN, stmt->val.rete->val.numb) ;
			else
			{	cg_expr	 (stmt->val.rete) ;
				_el_outn (C_RET,  0) ;
			}

			/* Also, flag that a return has just been	*/
			/* generated.                                   */
			lastret = 1 ;
			break   ;

		case S_ITER     :
			cg_iter   (stmt) ;
			break	;

		case S_IFTHEN   :
			cg_ifthen (stmt) ;
			break	;

		case S_SWITCH	:
			cg_switch (stmt) ;
			break	;

		case S_BREAK    :
			lastret  = 0 ;
			posn	 = _el_here () ;
			_el_outn (C_BRA, stmt->val.iterp->val.iter.breakp)  ;
			stmt->val.iterp->val.iter.breakp  = posn ;
			break   ;

		case S_CONT     :
			lastret  = 0 ;
			posn	 = _el_here () ;
			_el_outn (C_BRA, stmt->val.iterp->val.iter.contp )  ;
			stmt->val.iterp->val.iter.contp   = posn ;
			break   ;

		case S_ENDC	:
			lastret	= 0 ;
			posn	= _el_here () ;
			_el_outn (C_BRA, stmt->val.endc.swit->val.swcase.endp) ;
			stmt->val.endc.swit->val.swcase.endp = posn ;
			break	;

		default :
			errorE ("elc: unknown statement type: %d\n",
							stmt->tag) ;
	}
	}
}

