/***************************************************************************
    file	         : kb_block.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef 	_WIN32
#include	 "kb_block.moc"
#else
#include	 "kb_block.h"
#endif

#include	"kb_location.h"
#include	"kb_formblock.h"
#include	"kb_rowmark.h"
#include	"kb_reportblock.h"
#include	"kb_docroot.h"
#include	"kb_framer.h"

#include	"kb_qrybase.h"
#include	"kb_qrytable.h"
#include	"kb_qryquery.h"
#include	"kb_qrynull.h"
#include	"kb_qrysql.h"

#include	"kb_query.h"
#include	"kb_table.h"
#include	"kb_qryexpr.h"

#include	"kb_hidden.h"
#include	"kb_display.h"
#include	"kb_options.h"
#include	"kb_nodereg.h"
#include	"kb_parse.h"

#include	"kb_writer.h"

#include	"kb_blockevents.h"

/*  KBBlock								*/
/*  KBBlock	: Constructor for extant block element			*/
/*  parent	: KBNode *	: Parent element			*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  element	: cchar *	: Element name				*/
/*  (returns)	: KBBlock	:					*/

KBBlock::KBBlock
	(	KBNode			*parent,
		const QDict<QString>	&aList,
		cchar			*element
	)
	:
	KBItem	   (parent, element,		"master", aList),
	cexpr	   (this,   "child",		aList,  KF_REQD),
	bgcolor	   (this,   "bgcolor",		aList),
	autosync   (this,   "autosync", 	aList,  KF_FORM),
	title      (this,   "title",		aList,  KF_FORM),
	frame      (this,   "frame",		aList,  KF_FORM),
	showbar	   (this,   "showbar",		aList,  KF_FORM),
	rowcount   (this,   "rowcount",		aList,	KF_FORM),
	dx	   (this,   "dx",		aList,	KF_FORM),
	dy	   (this,   "dy",		aList,	KF_FORM)
{
	expr.setFlags (KF_REQD) ;

	init	() ;
	events	= new KBBlockEvents (this, aList) ;

	/* If the element us "KBSubBlock" then the block type is	*/
	/* definitely BTSubBlock. If not then we have to await the	*/
	/* query to figure it out.					*/
	blkType	 = getElement() == "KBFormSubBlock"   ? BTSubBlock :
		   getElement() == "KBReportSubBlock" ? BTSubBlock :
							BTUnknown  ;

	topLevel = (getBlock() == 0) || (getBlock()->getBlkType() == BTNull) ;
}

/*  KBBlock								*/
/*  KBBlock	: Constructor for new block element			*/
/*  parent	: KBObject *	: Parent element			*/
/*  rect	: const QRect &	: Size and position			*/
/*  blkType	: BlkType	: Initial block type			*/
/*  ok		: bool &	: Success				*/
/*  element	: cchar *	: Element name				*/
/*  (returns)	: KBBlock	:					*/

KBBlock::KBBlock
	(	KBObject	*parent,
		const QRect	&rect,
		BlkType		_blkType,
		bool		&ok,
		cchar		*element
	)
	:
	KBItem	   (parent, element, rect, "master", "", 0),
	cexpr	   (this,   "child",	  "",	 KF_REQD),
	bgcolor	   (this,   "bgcolor",	  ""),
	autosync   (this,   "autosync",   true,  KF_FORM),
	title	   (this,   "title",	  "",	 KF_FORM),
	frame	   (this,   "frame",	  "",	 KF_FORM),
	showbar	   (this,   "showbar",	  "No",  KF_FORM),
	rowcount   (this,   "rowcount",	  0,	 KF_FORM),
	dx	   (this,   "dx",	  KBOptions::getDefaultDX(), KF_FORM),
	dy	   (this,   "dy",	  KBOptions::getDefaultDY(), KF_FORM)
{
	expr.setFlags (KF_REQD) ;

	init	() ;
	events	= new KBBlockEvents (this) ;

#if	! __KB_RUNTIME
	/* If this is not a subblock then go through the full setup	*/
	/* procedure. If it is then just grab the query from our parent	*/
	/* which must be a block (or subblock).				*/
	if (_blkType != BTSubBlock)
	{
		if (!setBlkType (_blkType))
		{	ok	= false	;
	//**		delete	this	;
			return	;
		}

		if (!::blockPropDlg (this, "Block", attribs))
		{	ok	= false	;
	//**		delete	this	;
			return	;
		}
	}
	else
#endif
	{
		blkType	= BTSubBlock ;
		m_query	= getBlock()->getQuery() ;
		return	;
	}

	topLevel = (getBlock() == 0) || (getBlock()->getBlkType() == BTNull) ;
	ok	 = true	    ;
}


/*  KBBlock	: Constructor for extant block element			*/
/*  _parent	: KBNode *	: Parent element			*/
/*  _block	: KBBlock *	: Extant block				*/
/*  (returns)	: KBBlock	:					*/

KBBlock::KBBlock
	(	KBNode		*_parent,
		KBBlock		*_block
	)
	:
	KBItem	   (_parent, "master",    _block),
	cexpr	   (this,    "child",	  _block, KF_REQD),
	bgcolor	   (this,    "bgcolor",	  _block),
	autosync   (this,    "autosync",  _block, KF_FORM),
	title	   (this,    "title",	  _block, KF_FORM),
	frame	   (this,    "frame",	  _block, KF_FORM),
	showbar	   (this,    "showbar",	  _block, KF_FORM),
	rowcount   (this,    "rowcount",  _block, KF_FORM),
	dx	   (this,    "dx",	  _block, KF_FORM),
	dy	   (this,    "dy",	  _block, KF_FORM)
{
	expr.setFlags (KF_REQD) ;

	init	   () ;
	events	 = new KBBlockEvents  (this, _block) ;
	blkType	 = _block->getBlkType () ;

	topLevel = (getBlock() == 0) || (getBlock()->getBlkType() == BTNull) ;
}

/*  KBBlock								*/
/*  ~KBBlock	: Destructor for form-block element			*/
/*  (returns)	: void		:					*/

KBBlock::~KBBlock ()
{
	DELOBJ	(events ) ;
	DELOBJ	(blkDisp) ;

	//if (mValues != 0) delete [] mValues ;
}

/*  KBBlock								*/
/*  init	: Common constructor initialisation			*/
/*  (returns)	: void		:					*/

void	KBBlock::init ()
{
	curQRow	= 0	;
	curDRow	= 0	;
	nRows	= 0	;
	pRows	= 0	;
	qryLvl	= 0	;
	header	= 0	;
	footer	= 0	;
	blkDisp	= 0	;
	blkType	= BTUnknown ;
	m_query	= 0	;

	if (getRoot()->isReport())
	{
		QRect	r = geometry () ;

		KBObject::move	 (0, r.y     ()) ;
		KBObject::resize (0, r.height()) ;
		xmode.setValue   (KBObject::FMStretch) ;
		ymode.setValue   (KBObject::FMFixed  ) ;

		attribs.remove (getAttr("x")) ;
		attribs.remove (getAttr("w")) ;
		attribs.remove (&xmode) ;
		attribs.remove (&ymode) ;
	}

	attribs.remove (&defVal) ;
}

/*  KBBlock								*/
/*  newSubBlocks: Build nested sub-blocks				*/
/*  (returns)	: bool		: Success				*/

bool	KBBlock::newSubBlocks ()
{
	/* We need to know the number of levels in the query, so locate	*/
	/* and parse it.						*/
	KBLocation location (getDocRoot()->getDBInfo  (),
			     "query",
			     getDocRoot()->getLocation().docLocn,
			     m_query->getAttrVal ("query")
			    ) ;

	KBError	 	error	;
	KBQuery	 	*qry	;
	QByteArray	doc	;

	if (!location.contents (doc, error))
		return	 false	;

	if ((qry = KBOpenQueryText (location, doc, error)) == 0)
	{
		setError (error) ;
		return	 false	 ;
	}

	QString			svName	 ;
	QList<KBTable>		qryList	 ;
	QList<KBTable>		tabList	 ;
	QList<KBQryExpr>	exprList ;

	qry->getQueryInfo (svName, qryList, exprList) ;
	if (!KBTable::blockUp (qryList, m_query->getAttrVal("toptable"), tabList, error))
	{
		setError (error) ;
		return	 false	 ;
	}


	/* The first subb-lock will use just about half of the area of	*/
	/* this block. Any further subblocks will use successively	*/
	/* smaller areas.						*/
	uint	gridX	= KBOptions::getGridXStep () ;
	uint	gridY	= KBOptions::getGridYStep () ;
	QRect	r	= geometry () ;

	uint	xDelta	= isReportBlock() == 0 ? gridX : 0 ;
	uint	yRedux	= isReportBlock() == 0 ? 0 :
			  r.height() / 2 <= 120 ? 0 : 40 ;

	QRect	 area	(xDelta,
			 r.height() / 2,
			 r.width () - 2 * xDelta,
			 r.height() / 2 - gridY - yRedux) ;


	/* The subblocks are nested inside one-another. We need to note	*/
	/* the top-most subblock, so that we can subsequently pass	*/
	/* down the query level and the query.				*/
	KBObject *insBelow = this ;
	KBBlock  *level1   = 0	  ;
	bool 	 ok	   ;

	fprintf
	(	stderr,
		"KBBlock::newSubBlocks: build for %d\n",
		tabList.count()
	)	;

	for (uint tab = 1 ; tab < tabList.count() ; tab += 1)
	{
		KBBlock	*subBlk	= 0 ;

		if (getForm  () != 0)
			subBlk	= new KBFormBlock
				  (	insBelow,
					area,
					BTSubBlock,
					ok,
					"KBFormSubBlock"
				  )	;

		if (getReport() != 0)
			subBlk	= new KBReportBlock
				  (	insBelow,
					area,
					BTSubBlock,
					ok,
					"KBReportSubBlock"
				  )	;

		if (subBlk == 0)
			KBError::EFatal
			(	TR("Block error"),
				TR("Cannot identify object type"),
				__ERRLOCN
			)	;

		if (tab == 1) level1 = subBlk ;

		uint	 yRedux	= isReportBlock() ==   0 ? 0 :
				  area.height  () <= 120 ? 0 : 40 ;

		insBelow = subBlk ;
		area	 = QRect  (xDelta,
				   2 * gridY,
				   area.width () - 2 * xDelta,
				   area.height() - 3 * gridY - yRedux) ;
	}

	/* Pass down the query and level information. The odd case is	*/
	/* where the KBase-level query has but a single level, in which	*/
	/* case there is nothing to do.					*/
	if (level1 != 0) level1->setQryLevel (1, m_query) ;		

	delete	qry	;
	return	true	;
}

#if	! __KB_RUNTIME
/*  KBBlock								*/
/*  setBlkType	: Set the block type					*/
/*  newType	: BlkType	: Block type				*/
/*  (returns)	: bool		: Success				*/

bool	KBBlock::setBlkType
	(	BlkType	newType
	)
{
	KBQryBase *nQry	= 0 ;

	/* It should not be possible to change a subblock into some	*/
	/* other block type.						*/
	if (blkType == BTSubBlock)
		KBError::EFatal
		(	TR("Attempt to change block from subblock"),
			QString::null,
			__ERRLOCN
		)	;

	switch (newType)
	{
		case BTNull 	:
		case BTTable	:
		case BTQuery	:
		case BTSQL	:
			break	;

		case BTSubBlock	:
			/* This should never happen, the user should	*/
			/* never be able to change the block type to a	*/
			/* sub-block.					*/
			KBError::EFatal
			(	TR("Attempt to change block to subblock"),
				QString::null,
				__ERRLOCN
			)	;

		default	:
			KBError::EFault
			(	TR("Unrecognised block type"),
				QString::null,
				__ERRLOCN
			)	;
			return	false	;
	}

	switch (blkType = newType)
	{
		case BTTable	:
			/* The block accesses a table. Create the	*/
			/* table query object and running its		*/
			/* properties dialog.				*/
			nQry	= new KBQryTable    (this) ;

			if (!nQry->propertyDlg ())
			{	delete	nQry	;
				return	false	;
			}

			break	;

		case BTQuery	:
			/* Like a table ...				*/
			nQry	= new KBQryQuery (this) ;

			if (!nQry->propertyDlg ())
			{	delete	nQry	;
				return	false	;
			}

			break	;

		case BTSQL	:
			/* Also like a table ...			*/
			nQry	= new KBQrySQL (this) ;

			if (!nQry->propertyDlg ())
			{	delete	nQry	;
				return	false	;
			}

			break	;

		case BTNull	:
			/* This is a null block. Create a null query	*/
			/* object; this is a dummy query which always	*/
			/* succeeds, and returns zero rows of zero	*/
			/* fields. Hence, we dont have to check every	*/
			/* place that the block query is used.		*/
			nQry	= new KBQryNull  (this) ;
			break	;

		default	:
			/* Oops! Should have trapped this above! We'd	*/
			/* probably crash soon so lets go cleanly.	*/
			KBError::EFatal
			(	TR("Unrecognised block type"),
				QString::null,
				__ERRLOCN
			)	;
	}

	children.remove (nQry) ;

	while (children.count() > 0) delete children.first() ;

	children.append (nQry) ;
	m_query	= nQry	 ;

	if (blkType == BTQuery)
		if (!newSubBlocks()) return false ;


	return	true	;
}
#endif

/*  KBBlock								*/
/*  remChild	: Remove a child node					*/
/*  child	: KBNode *	: Child in question			*/
/*  (returns)	: void		:					*/

void	KBBlock::remChild
	(	KBNode	*child
	)
{
	/* Intercept this call since, if the is an item, then we must	*/
	/* notify our query, and we also need to check for various	*/
	/* specific pointers.						*/
	if (child->isItem () != 0) m_query->remItem (qryLvl, child->isItem()) ;

	if (child == blkInfo.rowmark) blkInfo.rowmark = 0 ;
	if (child == header) header = 0 ;
	if (child == footer) footer = 0 ;

	KBObject::remChild (child) ;
}

/*  KBBlock								*/
/*  move	: Move the block in its parent				*/
/*  _x		: int		: New X position			*/
/*  _y		: int		: New Y position			*/
/*  (returns)	: void		:					*/

void	KBBlock::move
	(	int	_x,
		int	_y
	)
{
	KBObject::move (_x, _y) ;
	if (blkDisp != 0) blkDisp->move (_x, _y) ;
}

/*  KBBlock								*/
/*  resize	: Resize the block in its parent			*/
/*  _w		: int		: New width				*/
/*  _h		: int		: New height				*/
/*  (returns)	: bool		: Size changed				*/

bool	KBBlock::resize
	(	int	_w,
		int	_h
	)
{
	if (KBObject::resize (_w, _h))
	{
		if (blkDisp != 0)
		{
			blkDisp->resize (_w, _h)  ;

			CITER
			(	Object,
				o,
				o->setGeometry (o->geometry())
			)
		}

		return	true	;
	}

	return	false	;
}

/*  KBBlock								*/
/*  clearValue	: Clear value for specified query and display row	*/
/*  qrow	: uint		: Starting row in query			*/
/*  query	: bool		: Clearing for query			*/
/*  (returns)	: void		:					*/

void	KBBlock::clearValue
	(	uint	,
		bool
	)
{
}

/*  KBBlock								*/
/*  changed	: See if user has changed an item			*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: Item changed				*/

bool	KBBlock::changed
	(	uint
	)
{
	return	false	;
}

/*  KBBlock								*/
/*  isEmpty	: See if item it empty					*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: Item empty				*/

bool	KBBlock::isEmpty
	(	uint
	)
{
	return	true	;
}

/*  KBBlock								*/
/*  isValid	: Test if value is valid to be saved			*/
/*  qrow	: uint		: Query row				*/
/*  allowNull	: bool		: Ignore not-null checks		*/
/*  (returns)	: bool		: Value can be saved			*/

bool	KBBlock::isValid
	(	uint		,
		bool
	)
{
	return	true	;
}

/*  KBBlock								*/
/*  requery	: (Re)query associated query for all rows		*/
/*  (returns)	: bool		  : Success				*/

bool	KBBlock::requery ()
{
	KBValue *mvp	 = getBlockVal () ;
	bool	evRc	;

	// fprintf (stderr, "requery: [%s]\n",
	//		 mvp == 0 ? "None" : value->getRawText()) ;

	/* If this is a sub-block then we don't ever actually perform a	*/
	/* query, this will have been done in the block at the top of	*/
	/* the sub-block chain. A requery is simply interpreted as	*/
	/* returning to the first record.				*/
	if (blkType == BTSubBlock)
	{
		m_query->setCurrentRow (qryLvl, 0) ;
		return	true ;
	}

	curQRow	= 0	;
	curDRow	= 0	;

	m_query->clearItems (qryLvl, 0) ;

	if (!eventHook (events->preQuery, 0, 0, evRc))
		return false ;

	if (!m_query->select
		(	qryLvl,
			mvp,
			cexpr.getValue (),
			m_userFilter,
			m_userSorting,
			true
	   )	)
	{
		setError (m_query->lastError()) ;
		return	 false ;
	}

	if (!eventHook (events->postQuery, 0, 0, evRc)) return false ;

	m_query->setCurrentRow (qryLvl, 0) ;
	return	true	;
}

/*  KBBlock								*/
/*  getQuery	: Get associated query					*/
/*  (returns)	: KBQryBase *	: Query					*/

KBQryBase *KBBlock::getQuery ()
{
	return	m_query ;
}

/*  KBBlock								*/
/*  getNumRows	: Get number of rows in query				*/
/*  (returns)	: uint		: Number of rows			*/

uint	KBBlock::getNumRows ()
{
	return	m_query == 0 ? 0 : m_query->getNumRows (qryLvl) ;
}

/*  KBBlock								*/
/*  getRowValue	: Get value from query					*/
/*  name	: const QString & : Query column name			*/
/*  qrow	: uint		  : Query row				*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBBlock::getRowValue
	(	const QString	&name,
		uint		qrow
	)
{
	if (qrow <= getNumRows())
		CITER
		(	Item,
			i,
			if (i->getName() == name)
				return	m_query->getField (qryLvl, qrow, i->getQueryIdx()) ;
		)

	return	KBValue() ;
}

/*  KBBlock								*/
/*  getRowValue	: Get value from query					*/
/*  item	: KBItem *	: Child item				*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBBlock::getRowValue
	(	KBItem		*item,
		uint		qrow
	)
{
	return	m_query->getField (qryLvl, qrow, item->getQueryIdx()) ;
}

/*  KBBlock								*/
/*  setRowValue	: Set value						*/
/*  name	: const QString & : Query column name			*/
/*  qrow	: uint		  : Query row				*/
/*  value	: const KBValye & : Value to set			*/
/*  (returns)	: void		  :					*/

void	KBBlock::setRowValue
	(	const QString	&name,
		uint		qrow,
		const KBValue	&value
	)
{
	if (qrow <= getNumRows())
		CITER
		(	Item,
			i,
			if (i->getName() == name)
			{	m_query->setField (qryLvl, qrow, i->getQueryIdx(), value) ;
				return	;
			}
		)
}

/*  KBBlock								*/
/*  findQuery	: Find child query node					*/
/*  (returns)	: bool		: Success				*/

bool	KBBlock::findQuery ()
{
	m_query	= 0 ;

	CITER
	(	QryBase,
		q,
		{	m_query	= q	;
			return	true	;
		}
	) 	;

	return	false	;
}

/*  KBBlock								*/
/*  blockSetup	: Prime block for operations				*/
/*  (returns)	: bool		: Success				*/

bool	KBBlock::blockSetup ()
{
	blkInfo.scroll	= blkDisp ;
	blkInfo.rowmark = 0	  ;

	/* If we are not a sub-block (in which case the query will have	*/
	/* been propogated down from the enclosing block) then locate	*/
	/* our query. It should either be a table, a query or a dummy	*/
	/* null query. The total absense of one is bad news, as also	*/
	/* if we cannot tell what sort of query it is.			*/
	if (blkType != BTSubBlock)
	{
		if (!findQuery ())
		{
			setError
			(	KBError::Fault,
				TR("KBBlock lacks a query"),
				TR("Form or Report Error"),
				__ERRLOCN
			) 	;
			return	false	;
		}

		if	(m_query->isQryNull  () != 0) blkType = BTNull   ;
		else if (m_query->isQryTable () != 0) blkType = BTTable  ;
		else if (m_query->isQryQuery () != 0) blkType = BTQuery  ;
		else if (m_query->isQrySQL   () != 0) blkType = BTSQL    ;
		else if (m_query->isQryDesign() != 0) blkType = BTDesign ;
		else
		{
			setError
			(	KBError::Fault,
				TR("KBBlock has unrecognised query"),
				TR("Form or Report Error"),
				__ERRLOCN
			)	;
			return	false	;
		}
	}


	/* Sanity check: If we are a sub-block then make sure that we	*/
	/* have a correct query type.					*/
	if (blkType == BTSubBlock)
	{
		if ((m_query->isQryQuery() == 0) && (m_query->isQryDesign() == 0))
		{
			setError
			(	KBError::Fault,
				TR("KBSubBlock has invalid query"),
				TR("Form or Report Error"),
				__ERRLOCN
			)	;
			return	false	;
		}

		if (m_query->isQryQuery() != 0)
		{
			QString	_mexpr	;
			QString	_cexpr ;

			if (!m_query->isQryQuery()->getLinkage(qryLvl, _mexpr, _cexpr))
			{
				setError
				(	KBError::Error,
					TR("Failed to get subblock child expression"),
					TR("Form Error"),
					__ERRLOCN
				)	;
				return	false	;
			}

			expr .setValue (_mexpr) ;
			cexpr.setValue (_cexpr)	;
		}
	}

	/* If this is the top of a KBase-level query, then scan for	*/
	/* nested sub-blocks, and propogate the query and query level	*/
	/* down. Note that, even if we have a query-query, there might	*/
	/* not be any sub-blocks (ie., if the KBase-level query only	*/
	/* contains on table).						*/
	if ((blkType == BTQuery) || (blkType == BTDesign))
	{
		CITER
		(	Block,
			b,
			if (b->getBlkType() == BTSubBlock)
				b->setQryLevel (1, m_query) ;
		)
		CITER
		(
			Framer,
			f,
			f->setQryLevel(0, m_query) ;
		)
	}

	/* Finally, recurse down to all nested blocks (including sub-	*/
	/* blocks) and framers. This process will traverse the entire	*/
	/* tree of blocks and header, footers, etc.			*/
	CITER
	(	Block,
		b,
		if (!b->blockSetup())
		{	setError (b->lastError ()) ;
			return false ;
		}
	)
	CITER
	(	Framer,
		f,
		if (!f->framerSetup (m_query, qryLvl, &blkInfo))
		{	setError (f->lastError ()) ;
			return false ;
		}
	)

	return	true ;
}

/*  KBBlock								*/
/*  addAllItems	: Add all items to the query				*/
/*  (returns)	: bool		: Any item fetches from database	*/

bool	KBBlock::addAllItems ()
{
	bool	fetch	= blkType == BTNull ;

	/* Pass pointers to all child items (ie., fields and blocks)	*/
	/* to the query so that it knows what to retrieve from the	*/
	/* database; the initial call with a null pointer clears the	*/
	/* query level's list. Also note the row mark if any.		*/
	m_query->addItem (qryLvl, 0) ;

	CITER
	(	Item,
		i,
		{	if (m_query->addItem (qryLvl, i))
				fetch	= true	;
			if (i->isRowMark() != 0)
				blkInfo.rowmark = i->isRowMark() ;
		}
	)	;

	/* We recurse down into nested framers. Note that items in	*/
	/* framers are at the same query level as the block in which	*/
	/* they are embedded.						*/
	CITER
	(
		Framer,
		f,
		if ( f->addAllItems ()) fetch = true  ;
	)

	/* Also recurse down into nested blocks ...			*/
	CITER
	(
		Block,
		b,
		if (!b->addAllItems ()) fetch = false ;
	)

	return	fetch	;
}

/*  KBBlock								*/
/*  setQryLevel	: Pass query level information into sub-blocks		*/
/*  _qryLvl	: uint		: Query level				*/
/*  _query	: KBQryBase *	: Query					*/
/*  (returns)	: void		:					*/

void	KBBlock::setQryLevel
	(	uint		_qryLvl,
		KBQryBase	*_query
	)
{
	qryLvl	= _qryLvl ;
	m_query	= _query  ;

	CITER
	(	Block,
		b,
		if (b->getBlkType() == BTSubBlock)
			b->setQryLevel (qryLvl + 1, m_query) ;
	)

	CITER
	(	Framer,
		f,
		f->setQryLevel (qryLvl, m_query) ;
	)
}

/*  KBBlock								*/
/*  setCurrent	: Note whether this block contains the current item	*/
/*  current	: bool		: True if so				*/
/*  (returns)	: void		:					*/

void	KBBlock::setCurrent
	(	bool	current
	)
{
	if (blkInfo.rowmark != 0) blkInfo.rowmark->setInBlock (current) ;
}

/*  KBBlock								*/
/*  write	: Write block 						*/
/*  writer	: KBWriter *	: Output writer				*/
/*  rect	: QRect		: Field area				*/
/*  extra	: int &		: Return extra space			*/
/*  (returns)	: bool		: Success				*/

bool	KBBlock::write
	(	KBWriter	*writer,
		QPoint		offset,
		bool		first,
		int		&extra
	)
{
	QString	bgcolor	;
	bgcolor.sprintf	("0x%06x", blkDisp->getDisplayWidget()->backgroundColor().rgb() & 0x00ffffff) ;

	new KBWriterBG (writer, geometry(), bgcolor) ;
	if (showingDesign()) new KBWriterBox (writer, geometry()) ;

	QPoint	s = writer->setOffset (false, geometry().topLeft()) ;
	KBNode::write (writer, offset, first, extra) ;
	writer->setOffset (true, s)  ;
	return	true	;
}

/*  KBBlock								*/
/*  getObjects	: Get list of all objects in block			*/
/*  olist	: QList<KBObject> & : List				*/
/*  (returns)	: void		    :					*/

void	KBBlock::getObjects
	(	QList<KBObject>	&olist
	)
{
	CITER
	(	Object,
		o,
		olist.append (o)
	)
}

/*  KBBlock								*/
/*  getItems	: Get list of all items in block			*/
/*  ilist	: QList<KBItem> &   : List				*/
/*  (returns)	: void		    :					*/

void	KBBlock::getItems
	(	QList<KBItem>	&ilist
	)
{
	CITER
	(	Item,
		i,
		ilist.append (i)
	)
}

/*  KBBlock								*/
/*  extendCtrls	: Extend (or decrease) the number of controls		*/
/*  nRows	: uint		: New number of rows			*/
/*  dx		: int		: X offset between rows			*/
/*  dy		: int		: Y offset between rows			*/
/*  (returns)	: void		:					*/

void	KBBlock::extendCtrls
	(	uint	nRows,
		int	,
		int	
	)
{
	pRows	= nRows	;
}

/*  KBBlock								*/
/*  getQueryComment							*/
/*		: Get query comment from associated query		*/
/*  (returns)	: QString	: Comment				*/

QString	KBBlock::getQueryComment ()
{
	return	m_query == 0 ? QString::null : m_query->getComment (qryLvl) ;
}

void	KBBlock::setTabOrder
	(	int
	)
{
}

int	KBBlock::getTabOrder ()
{
	return	-1 ;
}
