/***************************************************************************
    file	         : kb_navigator.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_formblock.h"
#include	"kb_framer.h"

#include	"kb_navigator.h"
#include	"kb_grid.h"



/*  tabCanEnter	: See if control can be entered on tab			*/
/*  obj		: KBObject *	: Object in question			*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: True if entry OK			*/

static	bool	tabCanEnter
	(	KBObject	*obj,
		uint		qrow
	)
{
	/* If the object is an item then check that the item is	visible	*/
	/* and enabled for the sepecified row ...			*/
	KBItem	*item	;

//	fprintf
//	(	stderr,
//		"tabCanEnter[%p][%s] item=[%p]\n",
//		(void  *)obj,
//		(cchar *)obj->getName(),
//		(void  *)obj->isItem ()
//	)	;

	if ((item = obj->isItem()) != 0)
		return	item->isVisible (qrow) && item->isEnabled (qrow) ;

	/* ... and otherwise just check the object as a whole.		*/
	return	obj->isVisible() && obj->isEnabled() ;
}


/*  KBTabOrderList							*/
/*  compareItems  : Ordering function using tab order values		*/
/*  item1	  : KBItem *	: First item				*/
/*  item2	  : KBItem *	: Second item				*/
/*  (returns)	  : int		: Ordering value			*/

int	KBTabOrderList::compareItems
	(	QCollection::Item	item1,
		QCollection::Item	item2
	)
{
	return	(int)((KBObject *)item1)->getTabOrder() -
		(int)((KBObject *)item2)->getTabOrder() ;
}

/*  KBTabOrderList							*/
/*  getFirst	  : Get first object ignoring hidden and disabled items	*/
/*  qrow	  : uint	: Query row number			*/
/*  (returns)	  : KBObject *	: First object or null if none		*/

KBObject
	*KBTabOrderList::getFirst
	(	uint	qrow
	)
{
	for (KBObject *obj = first() ; obj != 0 ; obj = next())
		if (tabCanEnter (obj, qrow))
			return	obj ;

	return	0 ;
}

/*  KBTabOrderList							*/
/*  getPrev	  : Get previous object ignoring hidden and disabled	*/
/*  qrow	  : uint	: Query row number			*/
/*  (returns)	  : KBObject *	: Object or null if none		*/

KBObject
	*KBTabOrderList::getPrev
	(	uint	qrow
	)
{
	for (KBObject *obj = prev() ; obj != 0 ; obj = prev())
		if (tabCanEnter (obj, qrow))
			return	obj ;

	return	0 ;
}

/*  KBTabOrderList							*/
/*  getNext	  : Get next object ignoring hidden and disabled	*/
/*  qrow	  : uint	: Query row number			*/
/*  (returns)	  : KBObject *	: Object or null of none		*/

KBObject
	*KBTabOrderList::getNext
	(	uint	qrow
	)
{
	for (KBObject *obj = next() ; obj != 0 ; obj = next())
		if (tabCanEnter (obj, qrow))
			return	obj ;

	return	0 ;
}

/*  KBTabOrderList							*/
/*  getLast	  : Get last object ignoring hidden and disabled items	*/
/*  qrow	  : uint	: Query row number			*/
/*  (returns)	  : KBObject *	: Object or null if none		*/

KBObject
	*KBTabOrderList::getLast
	(	uint	qrow
	)
{
	for (KBObject *obj = last() ; obj != 0 ; obj = prev())
		if (tabCanEnter (obj, qrow))
			return	obj ;

	return	0 ;
}



/*  ------------------------------------------------------------------	*/

/*  KBNavigator								*/
/*  KBNavigator	: Constructor for navigator object			*/
/*  owner	: KBObject *	  : Owing object			*/
/*  block	: KBFormBlock *	  : Containing form block		*/
/*  childList	: QList<KBNode> & : Block child list			*/
/*  (returns)	: KBNavigator	  :					*/

KBNavigator::KBNavigator
	(	KBObject	*owner,
		KBFormBlock	*block,
		QList<KBNode>	&childList
	)
	:
	m_owner		(owner),
	m_block		(block),
	m_childList	(childList)
{
}

/*  KBNavigator								*/
/*  findGrid	: Find grid control if any				*/
/*  (returns)	: KBGrid *	: Grid or null				*/

KBGrid	*KBNavigator::findGrid ()
{
	TITER
	(	Grid,
		m_childList,
		grid,
		return	grid
	)

	return	0 ;
}

/*  KBNavigator								*/
/*  fixGridLayout: Adjust layout where there is a grid control		*/
/*  (returns)	 : void		:					*/

void	KBNavigator::fixGridLayout ()
{
	KBGrid	*grid	;
	if ((grid = findGrid ()) == 0) return ;

	int	gy = grid->geometry().height() ;
	int	gx = grid->geometry().left  () ;
	int	gw = grid->geometry().width () ;
	int	cx = gx	;
 
	grid->setGeometry (QRect(gx, 0, gw, gy)) ;
	grid->clearItems  (true) ;

	LITER
	(	KBObject,
		m_tabList,
		obj,

		if (obj->isItem() == 0) continue ;

		QRect ig = obj->geometry() ;
		int   iw = ig.width() ;

		if (iw > gw) iw = gw - 32 ;
		if (iw <  0) iw = 32	;

		obj ->setGeometry (QRect(cx, gy, iw, ig.height())) ;
		grid->appendItem  (obj->isItem(), true) ;
		cx += iw ;
	)
}

/*  KBNavigator								*/
/*  orderGridByExpr: Order grid columns by expression (or not ...)	*/
/*  byExpr	   : bool	: Order by expression			*/
/*  (returns)	   : void	:					*/

void	KBNavigator::orderGridByExpr
	(	bool	byExpr
	)
{
	KBGrid	*grid	= findGrid () ;
	if (grid != 0) grid->orderByExpr (byExpr) ;
}

/*  KBNavigator								*/
/*  fixTabOrder	: Fix tab ordering of control items			*/
/*  (returns)	: void		:					*/

void	KBNavigator::fixTabOrder ()
{
	m_tabList.clear () ;

	TITER
	(	Object,
		m_childList,
		o,
		if (o->getTabOrder() > 0) m_tabList.inSort (o) ;
	)
}

/*  KBNavigator								*/
/*  firstItem	: Get the first item					*/
/*  (returns)	: KBItem *	: First item				*/

KBItem	*KBNavigator::firstItem ()
{
	KBItem	*first	= 0 ;

	/* If there are any items on the tab order list we use the	*/
	/* first, otherwise pick the first amongst the children other	*/
	/* than blocks.							*/
	if (m_tabList.count() == 0)
	{
		TITER
		(	Item,
			m_childList,
			i,
			if (i->isBlock() == 0)
			{
				first	= i ;
				break	;
			}
		)
	}
	else	for (uint idx = 0 ; idx < m_tabList.count() ; idx += 1)
			if (m_tabList.at(idx)->isItem() != 0)
			{
				first	= m_tabList.at(idx)->isItem() ;
				break	;
			}

	/* If we still do not have an item then look through any	*/
	/* containers ...						*/
	if (first == 0)
		TITER
		(	Framer,
			m_childList,
			f,
			if ((first = f->firstItem()) != 0) break ;
		)

	return	first	;
}

#if	! __KB_RUNTIME
/*  KBNavigator								*/
/*  newTabOrder	: Invoke tab order dialog				*/
/*  (returns)	: bool		: Success				*/

bool	KBNavigator::newTabOrder ()
{
	QList<KBObject>	objList ;

	TITER
	(	Object,
		m_childList,
		o,
		if (o->getTabOrder() >= 0) objList.append (o) ;
	)

	return	::tabOrderDlg (m_block, objList) ;
}
#endif


/*  KBNavigator								*/
/*  goNext	: Locate next item on tab ordering			*/
/*  curObj	: KBObject *	: Current object			*/
/*  set		: bool		: True to set focus			*/
/*  (returns)	: KBObject *	: Next object or null if none		*/

KBObject
	*KBNavigator::goNext
	(	KBObject	*curObj,
		bool		set
	)
{
	KBItem	 *item	;
	KBObject *res	;
	KBObject *obj	;

//	fprintf
//	(	stderr,
//		"KBNavigator::goNext[%p]: [%s][%d] at %d\n",
//		(void  *)this,
//		(cchar *)curObj->getName(),
//		set,
//		m_block->getCurQRow()
//	)	;

	if (m_tabList.findRef (curObj) >= 0)
	{
		/* First stage is to scan the objects in increasing	*/
		/* tab order (and which are visible and enabled).	*/
		/* Descend into any nested containers, and finish as	*/
		/* soon as we find the next item.			*/
		while ((obj = m_tabList.getNext(m_block->getCurQRow())) != 0)
		{
//			fprintf
//			(	stderr,
//				"KBNavigator::goNext[%p]: ... [%s][%s]\n",
//				(void  *)this,
//				(cchar *)obj->className(),
//				(cchar *)obj->getName  ()
//			)	;

			if (obj->isFramer() != 0)
			{
				if ((res = obj->isFramer()->goFirst(set)) != 0)
					return	res	;

				continue  ;
			}

			if ((item = obj->isItem()) != 0)
			{
				if (set) m_block->moveFocusTo (item) ;
				return	item	;
			}

			obj->setFocus ();
			return	obj	;
		}
	}

	/* No next item at this or lower levels in the hierarchy. If	*/
	/* this is *not* the block level then go back up a level and	*/
	/* search for the next item relative to this object.		*/
	if (m_block != m_owner)
	{
		KBNode		*parent	= m_owner->getParent() ;
		KBFramer	*framer	;
		KBFormBlock	*fblock	;

//		fprintf
//		(	stderr,
//			"KBNavigator::goNext[%p]: up  [%s][%s]\n",
//			(void  *)this,
//			(cchar *)parent->isObject()->className(),
//			(cchar *)parent->isObject()->getName  ()
//		)	;

		if ((framer = parent->isFramer   ()) != 0)
			if ((res = framer->goNext (m_owner, set)) != 0)
				return	res	;

		if ((fblock = parent->isFormBlock()) != 0)
			if ((res = fblock->goNext (m_owner, set)) != 0)
				return	res	;
	}

	/* No next item and this is the block level, which means	*/
	/* wrapping to the first item. If we are not setting focus or	*/
	/* the block is set to wrap on tabs, then just locate said	*/
	/* item; otherwise use the block-level NextForward action to	*/
	/* move row at the same time.					*/
//	fprintf
//	(	stderr,
//		"KBNavigator::goNext[%p]: next (wrap %d) at %d\n",
//		(void  *)this,
//		m_block->tabsWrap  (),
//		m_block->getCurQRow()
//	)	;

	if (!set || m_block->tabsWrap())
		return goFirst (m_block->tabsWrap()) ;

	if (!m_block->doAction (KB::NextForw, &m_tabList))
		m_block->lastError().DISPLAY() ;

	return	m_block->getCurItem () ;
}

/*  KBNavigator								*/
/*  goPrevious	: Locate previois item on tab ordering			*/
/*  curObj	: KBObject *	: Current object			*/
/*  set		: bool		: True to set focus			*/
/*  (returns)	: KBObject *	: Next object or null if none		*/

KBObject
	*KBNavigator::goPrevious
	(	KBObject	*curObj,
		bool		set
	)
{
	KBItem	 *item	;
	KBObject *res	;
	KBObject *obj	;

	if (m_tabList.findRef (curObj) >= 0)
	{
		while ((obj = m_tabList.getPrev(m_block->getCurQRow())) != 0)
		{
			if (obj->isFramer())
			{
				if ((res = obj->isFramer()->goLast(set)) != 0)
					return	res	;

				continue  ;
			}

			if ((item = obj->isItem()) != 0)
			{
				if (set) m_block->moveFocusTo (item) ;
				return	item	;
			}

			obj->setFocus ();
			return	obj	;
		}
	}

	if (m_block != m_owner)
	{
		KBNode		*parent	= m_owner->getParent() ;
		KBFramer	*framer	;
		KBFormBlock	*fblock	;

		if ((framer = parent->isFramer   ()) != 0)
			if ((res = framer->goPrevious (m_owner, set)) != 0)
				return	res	;

		if ((fblock = parent->isFormBlock()) != 0)
			if ((res = fblock->goPrevious (m_owner, set)) != 0)
				return	res	;
	}

	if (!set || m_block->tabsWrap())
		return	goLast (m_block->tabsWrap()) ;

	if (!m_block->doAction (KB::PrevBack, &m_tabList))
		m_block->lastError().DISPLAY() ;

	return	m_block->getCurItem () ;
}

/*  KBNavigator								*/
/*  goFirst	: Locate first item on tab ordering			*/
/*  set		: bool		: True to set focus			*/
/*  (returns)	: KBObject *	: Next object or null if none		*/

KBObject
	*KBNavigator::goFirst
	(	bool		set
	)
{
	KBObject *res	;
	KBItem	 *item	;

	for (KBObject *obj  = m_tabList.getFirst(m_block->getCurQRow()) ;
		       obj != 0 ;
		       obj  = m_tabList.getNext (m_block->getCurQRow()) )
	{
//		fprintf
//		(	stderr,
//			"KBNavigator::goFirst: check [%p][%s]\n",
//			(void  *)obj,
//			(cchar *)obj->getName()
//		)	;

		if (obj->isFramer())
		{
			if ((res = obj->isFramer()->goFirst(set)) != 0)
				return	res	;

			continue  ;
		}

		if ((item = obj->isItem()) != 0)
		{
			if (set) m_block->moveFocusTo (item) ;
			return	item	;
		}

		/* Unlike ::goNext and ::goPrevious, ::goFirst does not	*/
		/* include objects other than framers (eg., buttons).	*/
		/* This simplifies the logic in KBBlock::doOperation,	*/
		/* and in any case is probably the right thing to do.	*/
	}

	fprintf
	(	stderr,
		"KBNavigator::goFirst: nothing in %d!!\n",
		m_tabList.count()
	)	;
	return	0 ;
}

/*  KBNavigator								*/
/*  goLast	: Locate last item on tab ordering			*/
/*  set		: bool		: True to set focus			*/
/*  (returns)	: KBObject *	: Next object or null if none		*/

KBObject
	*KBNavigator::goLast
	(	bool		set
	)
{
	KBObject *res	;
	KBItem	 *item	;

	for (KBObject *obj  = m_tabList.getLast(m_block->getCurQRow()) ;
		       obj != 0 ;
		       obj  = m_tabList.getPrev(m_block->getCurQRow()) )
	{
		if (obj->isFramer())
		{
			if ((res = obj->isFramer()->goLast(set)) != 0)
				return	res	;

			continue  ;
		}

		if ((item = obj->isItem()) != 0)
		{
			if (set) m_block->moveFocusTo (item) ;
			return	item	;
		}

		/* See comment in ::goFirst above re ignoring object	*/
		/* other than framers.					*/
	}

	return	0 ;
}

/*  KBNavigator								*/
/*  keyStroke	: Control has received keystroke			*/
/*  item	: KBItem *	: Item in question			*/
/*  k		: QKeyEvent *	: Key event				*/
/*  (returns)	: bool		: Key consumed				*/

bool	KBNavigator::keyStroke
	(	KBItem		*item,
		QKeyEvent	*k
	)
{
	int	key	= k->key () ;
	bool	shift	= (k->state () & Qt::ShiftButton  ) != 0 ;
	bool	ctrl	= (k->state () & Qt::ControlButton) != 0 ;
	bool	oov	= false	 ;


	/* Item is zero if the control with (notional) focus is out	*/
	/* of view, so get the current control from the block.		*/
	if (item == 0)
	{
		if ((item = m_block->getCurItem ()) == 0)
			return	false	;

		oov	= true	;
	}


	/* Locate the specified item in the tab order list. Abandon	*/
	/* the operation if not found, otherwise this means we can use	*/
	/* next and previous on the list.				*/
	// if (m_tabList.findRef (item) < 0) return false ;


	/* First check for keys that we are interested in, and do some	*/
	/* initial checks.						*/
	switch (key)
	{
		case Qt::Key_Shift	:
		case Qt::Key_Control	:
		case Qt::Key_Meta	:
		case Qt::Key_Alt	:
		case Qt::Key_CapsLock	:
		case Qt::Key_NumLock	:
		case Qt::Key_ScrollLock	:
		case Qt::Key_Pause	:
		case Qt::Key_Print	:
		case Qt::Key_SysReq	:
		case Qt::Key_Super_L	:
		case Qt::Key_Super_R	:
		case Qt::Key_Menu	:
		case Qt::Key_Hyper_L	:
		case Qt::Key_Hyper_R	:
		case Qt::Key_Help	:
#if	QT_VERSION >= 300
		case Qt::Key_Direction_L:
		case Qt::Key_Direction_R:
#endif
		case Qt::Key_unknown	:
			/* Ignore all the modifier keys when on their	*/
			/* own, and a number of others.			*/
			return	false	;

		case Qt::Key_Tab	:
			/* Shift-tab is mapped into back-tab ...	*/
			if (shift) key = Qt::Key_Backtab ;
			break	;

		case Qt::Key_Backtab	:
		case Qt::Key_Up		:
		case Qt::Key_Down	:
		case Qt::Key_Escape	:
			/* These are all navigation keys ...		*/
			break	;

		case Qt::Key_Enter	:
		case Qt::Key_Return	:
			/* Thase are save if the control key is down ..	*/
			if (ctrl)
			{
				if (!m_block->doAction (KB::Save, &m_tabList))
					m_block->lastError().DISPLAY() ;
				return	true  ;
			}

			/* ... or the same as Tab otherwise		*/
			break		;

		case Qt::Key_Left	:
		case Qt::Key_Right	:
			/* This do navigation if the control key is	*/
			/* down ...					*/
			if (!ctrl) return false ;
			break	;

		case Qt::Key_Prior	:
			/* Handle previous page where since we do not	*/
			/* need to scroll if the current item is out of	*/
			/* view ....					*/
			if (!m_block->doAction (KB::PrevPage, &m_tabList))
				m_block->lastError().DISPLAY() ;
			return	true  ;

		case Qt::Key_Next	:
			/* .... and similarly next page.		*/
			if (!m_block->doAction (KB::NextPage, &m_tabList))
				m_block->lastError().DISPLAY() ;
			return	true  ;

		case Qt::Key_F		:
			/* Ctrl-F brings up the find dialog for those	*/
			/* controls that support it.			*/
			if (ctrl)
			{
				item->doSearch	() ;
				return	true	;
			}

			/* Scroll into view if needed, then return for	*/
			/* default handling.				*/
			if (oov)
				m_block->scrollToRow (m_block->getCurQRow()) ;
			return	false	;

		case Qt::Key_A		:
			/* Ctrl-A is used to mark all rows in the	*/
			/* block.					*/
			if (ctrl)
			{
				m_block->setRowMarked (0, KB::MarkOpSetAll) ;
				return	true	;
			}

			/* Scroll into view if needed, then return for	*/
			/* default handling.				*/
			if (oov)
				m_block->scrollToRow (m_block->getCurQRow()) ;
			return	false	;

		default	:
			/* All other keystrokes are passed back for	*/
			/* default handling; scroll into view if needed	*/
			if (oov)
				m_block->scrollToRow (m_block->getCurQRow()) ;
			return	false	;
	}


	if (oov)
		m_block->scrollToRow (m_block->getCurQRow()) ;

	/* If there is a current item and it has changed then mark the	*/
	/* form as changed.						*/
	m_block->markChanged () ;


	/* Now comes the key-specific stuff. Where control is passed to	*/
	/* "doAction" we do not need any more checking here.		*/
	switch (key)
	{
		case Qt::Key_Tab    :
		case Qt::Key_Enter  :
		case Qt::Key_Return :
			/* Get the next item in the tab order ....	*/
			QFocusEvent::setReason (QFocusEvent::Tab    ) ;
			goNext	(item, true) 	;
			return	true	;

		case Qt::Key_Backtab :
			/* .... or the previous one			*/
			QFocusEvent::setReason (QFocusEvent::Backtab) ;
			goPrevious (item, true) ;
			return	true	;

		case Qt::Key_Left :
			/* Move to the first control in the row. We	*/
			/* know the control key is down.		*/
			m_block->goFirst (true) ;
			break	;

		case Qt::Key_Right  :
			/* ... and similarly to the last. Again, the	*/
			/* control key is down.				*/
			m_block->goLast  (true) ;
			break	;

		case Qt::Key_Up :
			/* Either move to the previous row, if any, or	*/
			/* move to the first row, depending on control	*/
			/* key state.					*/
			if (!m_block->doAction
				(	ctrl ? KB::First : KB::Previous,
					&m_tabList
			   )	)
				m_block->lastError().DISPLAY() ;
			return	true ;

		case Qt::Key_Down :
			/* Similarly down, which is either next row or	*/
			/* last row.					*/
			if (!m_block->doAction
				(	ctrl ? KB::Last  : KB::Next,
					&m_tabList
			   )	)
				m_block->lastError().DISPLAY() ;
			return	true  ;

		case Qt::Key_Escape :
			/* This invokes the row-reset function to set	*/
			/* the controls back to the last saved state.	*/
			if (!m_block->doAction (KB::Reset,    &m_tabList))
				m_block->lastError().DISPLAY() ;
			return	true  ;

		default	:
			return	false ;
	}

	/* If control gets here then we are switching controls within	*/
	/* the same row.						*/
	m_block->moveFocusTo (item) ;
	return	true ;
}


/*  KBNavigator								*/
/*  isNavKey	: See if keystroke is a navigation event		*/
/*  k		: QKeyEvent *	: Keystroke event			*/
/*  (returns)	: bool		: True for navigation			*/

bool	KBNavigator::isNavKey
	(	QKeyEvent 	*k
	)
{
	/* This static method can be called to test if a key event is	*/
	/* a navigation event; this is needed for classes like		*/
	/* KBComboBox in order to correctly filter key events when in	*/
	/* read-only mode.						*/
	int	key	= k->key () ;
	bool	ctrl	= (k->state () & Qt::ControlButton) != 0 ;

	switch (key)
	{
		case Qt::Key_Prior	:
		case Qt::Key_Next	:
		case Qt::Key_Tab	:
		case Qt::Key_Backtab	:
		case Qt::Key_Up		:
		case Qt::Key_Down	:
		case Qt::Key_Escape	:
			/* These are all navigation keys ...		*/
			return	true	;

		case Qt::Key_Enter	:
		case Qt::Key_Return	:
		case Qt::Key_Left	:
		case Qt::Key_Right	:
			/* These are navigation keys if the control key	*/
			/* is down ...					*/
			return	ctrl	;

		case Qt::Key_F		:
			/* Ctrl-F brings up the find dialog for those	*/
			/* controls that support it.			*/
			return	ctrl	;

		case Qt::Key_A		:
			/* Ctrl-A is used to mark all rows in the	*/
			/* block.					*/
			return	ctrl	;

		default	:
			/* All other keystrokes are passed back for	*/
			/* default handling; scroll into view if needed	*/
			break	;
	}

	return	false	;
}

