/***************************************************************************
    file	         : kb_attr.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qmetaobject.h>
#include	<qregexp.h>

#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_node.h"
#include	"kb_nodemonitor.h"
#include	"kb_type.h"

#include	"kb_object.h"
#include	"kb_docroot.h"
#include	"kb_nodereg.h"
#include	"kb_attr.h"
#include	"kb_dictionary.h"

#if		! __KB_RUNTIME
#include	"kb_attritem.h"
#endif

static 	struct LIBKBASE_API	AttrInfo
{
	cchar	*name	;
	uint	flags	;
}
	aInfo[]		=
{
{	"x",			KAF_GRPFORMAT				},
{	"y",			KAF_GRPFORMAT				},
{	"w",			KAF_GRPFORMAT				},
{	"h",			KAF_GRPFORMAT				},
{	"xmode",		KAF_GRPFORMAT				},
{	"ymode",		KAF_GRPFORMAT				},
{	"stretch",		KAF_GRPFORMAT				},
{	"fgcolor",		KAF_GRPFORMAT|KAF_COMMON|KAF_CLEAR	},
{	"bgcolor",		KAF_GRPFORMAT|KAF_COMMON|KAF_CLEAR	},
{	"font",			KAF_GRPFORMAT|KAF_COMMON|KAF_CLEAR	},
{	"frame",		KAF_GRPFORMAT|KAF_COMMON|KAF_CLEAR	},
{	"rowcount",		KAF_GRPFORMAT				},
{	"dx",			KAF_GRPFORMAT				},
{	"dy",			KAF_GRPFORMAT				},
{	"showrow",		KAF_GRPFORMAT				},
{	"showbar",		KAF_GRPFORMAT				},
{	"statusbar",		KAF_GRPFORMAT				},
{	"tabtext",		KAF_GRPFORMAT				},
{	"wrap",			KAF_GRPFORMAT				},
{	"wrapchars",		KAF_GRPFORMAT|KAF_CLEAR			},

{	"blktype",		KAF_GRPDATA  |KAF_DUMMY			},
{	"name",			KAF_GRPDATA  |KAF_CLEAR			},
{	"expr",			KAF_GRPDATA  |KAF_CLEAR			},
{	"master",		KAF_GRPDATA  |KAF_CLEAR			},
{	"child",		KAF_GRPDATA				},
{	"default",		KAF_GRPDATA  |KAF_CLEAR			},
{	"values",		KAF_GRPDATA				},
{	"nullval",		KAF_GRPDATA				},
{	"query",		KAF_GRPDATA				},
{	"toptable",		KAF_GRPDATA				},
{	"nullok",		KAF_GRPDATA  				},
{	"evalid",		KAF_GRPDATA  				},
{	"igncase",		KAF_GRPDATA  				},
{	"rdonly",		KAF_GRPDATA  				},
{	"format",		KAF_GRPDATA  |KAF_SINGLE|KAF_CLEAR	},
{	"taborder",		KAF_GRPDATA				},
{	"autosync",		KAF_GRPDATA				},
{	"text",			KAF_GRPDATA  |KAF_CLEAR			},
{	"show",			KAF_GRPDATA				},
{	"server",		KAF_GRPDATA				},
{	"align",		KAF_GRPDATA  |KAF_COMMON		},
{	"mask",			KAF_GRPDATA  |KAF_CLEAR			},
{	"helper",		KAF_GRPDATA  |KAF_CLEAR			},
{	"table",		KAF_GRPDATA				},
{	"primary",		KAF_GRPDATA				},
{	"where",		KAF_GRPDATA  |KAF_CLEAR			},
{	"order",		KAF_GRPDATA  |KAF_CLEAR			},
{	"dynamic",		KAF_GRPDATA				},
{	"parameters",		KAF_GRPDATA				},
{	"noupdate",		KAF_GRPDATA  				},
{	"emptynull",		KAF_GRPDATA  				},

{	"hilite",		KAF_GRPFORMAT|KAF_CLEAR			},
{	"errtext",		KAF_GRPOTHER |KAF_CLEAR			},

{	"lmargin",		KAF_GRPFORMAT				},
{	"rmargin",		KAF_GRPFORMAT				},
{	"tmargin",		KAF_GRPFORMAT				},
{	"bmargin",		KAF_GRPFORMAT				},
}	;




static	QDict<AttrInfo>	aInfoDict ;

LIBKBASE_API	KBDictionary	*getAttrDict()
{
	static	KBDictionary	*dict	;
	if (dict == 0) dict = new KBDictionary ("rekall") ;
	return	dict	;
}


/*  KBAttr								*/
/*  KBAttr	: Constructor for attribute class			*/
/*  owner	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  value	: cchar *	: Initial value				*/
/*  nFlags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttr::KBAttr
	(	KBNode			*owner,
		AType			type,
		cchar			*name,
		cchar			*value,
		uint			nFlags
	)
	:
	owner	(owner),
	type	(type),
	name	(name),
	m_value	(value),
	m_saved	(value),
	nFlags	(nFlags)
{
	attach	() ;
	m_showing = KB::ShowAsUnknown ;
}

/*  KBAttr								*/
/*  KBAttr	: Constructor for attribute class			*/
/*  owner	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  aList	: const QDict<QString> &				*/
/*				: Attribute list			*/
/*  nFlags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttr::KBAttr
	(	KBNode			*owner,
		AType			type,
		cchar			*name,
		const QDict<QString>	&aList,
		uint			nFlags
	)
	:
	owner	(owner),
	type	(type),
	name	(name),
	nFlags	(nFlags)
{
	QString	*v = aList.find (name) ;

	if (v == 0)
		m_value	= "" ;
	else	m_value	= *v ;

	m_saved	= m_value    ;

	attach	() ;
	m_showing = KB::ShowAsUnknown ;
}

/*  KBAttr								*/
/*  KBAttr	: Constructor for attribute class			*/
/*  owner	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  srce	: KBNode *	: Extant source				*/
/*  nFlags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttr::KBAttr
	(	KBNode			*owner,
		cchar			*name,
		KBNode			*srce,
		uint			nFlags
	)
	:
	owner	(owner),
	name	(name),
	nFlags	(nFlags)
{
	const KBAttr *extant	;

	if ((extant = srce->getAttr (name)) != 0)
	{
		type	  = extant->type	;
		m_value	  = extant->m_value	;
		m_saved	  = extant->m_value	;
		m_showing = extant->m_showing	;
	}
	else	m_showing = KB::ShowAsUnknown	;

	attach	() ;
}

/*  KBAttr								*/
/*  attach	: Attach to owner					*/
/*  (returns)	: void		:					*/

void	KBAttr::attach ()
{
	monitor		= 0 ;

	if (owner == 0)
	{	ownerName = "UnknownOwner" ;
		return	  ;
	}

	ownerName	= owner->metaObject()->className() ;

	/* If any node flags are set then see if any corresponding flag	*/
	/* is set for the owner. If not then we do not attach to the	*/
	/* owner; this is used to handle attributes which appear, say,	*/
	/* in a report but not in a form.				*/
	if ((nFlags & KF_FLAGS) != 0)
		if ((owner->getRoot()->getFlags() & nFlags) == 0)
		{	
			owner	= 0 ;
			return	;
		}

	/* Attach ourselves to the owing node. If this is showing a	*/
	/* monitor then the monitor listview item will be returned, and	*/
	/* we add ourselves.						*/
	KBNodeMonitor *pm = owner->addAttr (this, order) ;
	if (pm != 0) showMonitor (pm) ;
}

/*  KBAttr								*/
/*  ~KBAttr	: Destructor for attribute class			*/
/*  (returns)	:		:					*/

KBAttr::~KBAttr ()
{
	/* TRICKY CODE ALERT:						*/
	/* Don't delete a monitor, even though it will have been	*/
	/* allocated here. This is because a monitor is a listviewitem	*/
	/* and will be a child of the parent nodes monitor. When the	*/
	/* latter is deleted, so will the attribute monitor.		*/
	// DELOBJ	(monitor) ;
	if (owner != 0) owner->remAttr (this) ;
}

/*  KBAttr								*/
/*  setValue	: Set attribute value					*/
/*  v		: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBAttr::setValue
	(	const QString	&v
	)
{
	m_value	= v	;
	if (m_showing != KB::ShowAsData) m_saved = v  ;
	if (monitor != 0) monitor->setText (2, v) ;
}

/*  KBAttr								*/
/*  valueOK	: Check if value is OK					*/
/*  v		: const QString & : Value				*/
/*  (returns)	: void		  :					*/

bool	KBAttr::valueOK
	(	const QString	&
	)
{
	return	true	;
}

/*  KBAttr								*/
/*  substitute	: Substiture parameter values				*/
/*  (returns)	: QString	: Substituted text			*/

QString	KBAttr::substitute
	(	const QString	&val
	)
{
	if ((owner != 0) && owner->showingDesign())
		return	m_value	;

	if (val.isNull() || (val.find ("${") < 0))
		return	val	;

	QString	res	= "" 	;
	int	pos1	= 0	;
	int	pos2	;

	while ((pos2 = val.find ("${", pos1)) >= 0)
	{
		res	+= val.mid (pos1, pos2 - pos1) ;
		pos1	 = pos2 + 2 ;

		if ((pos2 = val.find ("}", pos1)) < 0)
		{	res	+= "${" ;
			break	;
		}

		res	+= owner->getDocRoot()->getParamValue (val.mid (pos1, pos2 - pos1)) ;
		pos1	 = pos2 + 1 ;
	}

	res	+= val.mid (pos1) ;
	return	res ;
}

/*  KBAttr								*/
/*  KBAttr								*/
/*  getValue	: Get attribute value					*/
/*  (returns)	: QString	: Value					*/

QString	KBAttr::getValue ()
{
	return	substitute (m_value) ;
}


/*  KBAttr								*/
/*  getLegend	: Get legend for property dialogs			*/
/*  (returns)	: QString	: Legend				*/

QString	KBAttr::getLegend ()
{
	return	getAttrDict()->getAttrLegend (owner->getElement(), name) ;
}

/*  KBAttr								*/
/*  getDescription							*/
/*		: Get description for property dialogs			*/
/*  (returns)	: QString	: Legend				*/

QString	KBAttr::getDescription ()
{
	return	getAttrDict()->getAttrDescription (owner->getElement(), name) ;
}

/*  KBAttr								*/
/*  getNullcheck							*/
/*		: Get null check error for property dialogs		*/
/*  (returns)	: QString	: Error message				*/

QString	KBAttr::getNullcheck ()
{
	return	getAttrDict()->getAttrNullcheck (owner->getElement(), name) ;
}

/*  KBAttr								*/
/*  getOrder	: Get property dialog order value			*/
/*  (returns)	: uint		: Order					*/

uint	KBAttr::getOrder ()
{
	return	(getFlags() & KAF_GRPMASK) | order ;
}

/*  KBAttr								*/
/*  getFlags	: Get attribute flags					*/
/*  (returns)	: uint		: Flags					*/

uint	KBAttr::getFlags ()
{
	if (aInfoDict.count() == 0)
		for (uint idx = 0 ; idx < sizeof(aInfo)/sizeof(aInfo[0]) ; idx += 1)
			aInfoDict.insert (aInfo[idx].name, &aInfo[idx]) ;

	/* If the custom flags flag is not set then pick up the default	*/
	/* flags (if any) and install them. There is a frig so that we	*/
	/* only get one warning message for each missing set.		*/
	if ((nFlags & KAF_CUSTOM) == 0)
	{
		AttrInfo *f = aInfoDict.find (name) ;

		if	(f == 0)
		{
		//	fprintf
		//	(	stderr,
		//		"=====> No attribute flags for [%s]\n",
		//		(cchar *)name
		//	)	;

			f = new AttrInfo	 ;
			f->name  = strdup (name) ;
			f->flags = 0xffffffff	 ;
			aInfoDict.insert  (name, f) ;
		}
		else if	(f->flags == 0xffffffff)
		{
		}
		else	nFlags	|= f->flags ;

		/* Group defaults to "other", and set the custom flag	*/
		/* so that we only fetch once.				*/
		if ((nFlags & KAF_GRPMASK) == 0) nFlags |= KAF_GRPOTHER ;
		nFlags	|= KAF_CUSTOM ;
	}

	return	 nFlags ;
}

/*  KBAttr								*/
/*  escapeText	: Do XML text escaping					*/
/*  val		: const QString & : Text to be escaped			*/
/*  nl		: bool		  : Escape newlines			*/
/*  (returns)	: QString	  : Processed text			*/

QString	KBAttr::escapeText
	(	const QString	&val,
		bool		nl
	)
{
	QString	text	;

	for (uint idx = 0 ; idx < val.length() ; idx += 1)
	{
		QChar	ch = val.at(idx) ;

		if	((ch == '<' )	   ) text += "&lt;"   ;
		else if ((ch == '>' )	   ) text += "&gt;"   ;
		else if ((ch == '&' )	   ) text += "&amp;"  ;
		else if ((ch == '"' )	   ) text += "&quot;" ;
		else if ((ch == '\'')	   ) text += "&#039;" ;
		else if ((ch == '\n') && nl) text += "&#010;" ;
		else		     	     text += ch	      ;
	}

	return	text	;
}

/*  KBAttr								*/
/*  addAttrText	: Add attribute definition text to string		*/
/*  text	: QString &	  : String to append to			*/
/*  name	: const QString & : Attribute name			*/
/*  val		: const QString & : Attribute value			*/
/*  addNull	: bool		  : Add even if null			*/
/*  (returns)	:		  :					*/

void	KBAttr::addAttrText
	(	QString		&text,
		const QString	&name,
		const QString	&val,
		bool		addNull
	)
{
	/* NOTE: This method is static, hence the complete set of	*/
	/*	 arguments.						*/
	QString	escaped	= escapeText(val) ;

	if (!escaped.isEmpty() || addNull) 
		text	+= QString(" %1=\"%2\"").arg(name).arg(escaped) ;
}

/*  KBAttr								*/
/*  showAs	: Set display mode					*/
/*  showAs	: KB::ShowAs	: Display mode				*/
/*  (returns)	: bool		: True if restored value is changed	*/

bool	KBAttr::showAs
	(	KB::ShowAs	showAs
	)
{
//	fprintf
//	(	stderr,
//		"KBAttr::showAs: %d->%d: [%s] [%s][%s]\n",
//		m_showing,
//		showAs,
//		(cchar *)name,
//		(cchar *)value,
//		(cchar *)m_saved
//	)	;

	if (m_showing == showAs)
		return	false	;

	if (showAs == KB::ShowAsData)
	{
		m_showing = KB::ShowAsData ;

		if (m_saved != m_value)
		{
			m_saved	= m_value  ;
			return	true	   ;
		}

		return	false	;
	}

	if (showAs == KB::ShowAsDesign)
	{
		m_showing = KB::ShowAsDesign ;

		if (m_value != m_saved)
		{
			m_value	= m_saved ;
			return	true	  ;
		}

		return	false	;
	}

	return	false	;
}

/*  KBAttr								*/
/*  printAttr	: Print attribute					*/
/*  attrText	: QString &	: Result text for attribute output	*/
/*  node	: QString &	: Result text for node output		*/
/*  indent	: int		: Indent depth for node output		*/
/*  (returns)	: void		:					*/

void	KBAttr::printAttr
	(	QString		&attrText,
		QString		&,
		int
	)
{
	if ((getFlags() & (KAF_HIDDEN|KAF_USER)) != 0)
		return	;

	if ((name == "name") && m_value.isEmpty())
		if ((owner != 0) && (owner->getParent() != 0))
		{
			QString	elem	= owner->getElement() ;
			int	tag	= 1 ;
			QRegExp regexp	(elem + "_([0-9]+)") ;

			LITER
			(	KBNode,
				owner->getParent()->getChildren(),
				s,

				if (s == owner)
					continue ;
				if (regexp.search (s->getAttrVal("name")) < 0)
					continue ;

				if (regexp.cap(1).toInt() >= tag)
					tag	= regexp.cap(1).toInt() + 1 ;
			)

			m_value	= QString("%1_%2").arg(elem).arg(tag) ;
		}

	addAttrText (attrText, name, m_value) ;
}

/*  KBAttr								*/
/*  showMonitor	: Setup monitor for this attribute			*/
/*  parent	: QListViewItem * : Parent monitor entry		*/
/*  (returns)	: void		  :					*/

void	KBAttr::showMonitor
	(	QListViewItem	*parent
	)
{
	if (parent != 0)
	{
		monitor	= new KBNodeMonitor (0,  parent) ;
		monitor->setText	(0, "Attribute") ;
		monitor->setText	(1, name       ) ;
		monitor->setText	(2, m_value    ) ;
		monitor->setSelectable	(false) ;
	}
	else	monitor	= 0 ;
}

/*  KBAttr								*/
/*  replicate	: Replicate attribute with new owner			*/
/*  (returns)	: KBAttr *	: Replicant				*/

KBAttr	*KBAttr::replicate
	(	KBNode	*
	)
{
	return	0 ;
}

/*  KBAttr								*/
/*  isEvent	: Return pointer if attribute is an event		*/
/*  (returns)	: KBEvent *	: Event or null				*/

KBEvent	*KBAttr::isEvent ()
{
	return	0 ;
}

#if	! __KB_RUNTIME

/*  KBAttr								*/
/*  getAttrItem	: Get attribute property item for this attribute	*/
/*  (returns)	: KBAttrItem *	: Item					*/

KBAttrItem *KBAttr::getAttrItem ()
{
	/* This method can be overridden where the attribute item needs	*/
	/* to be more complicated (eg., for an attribute that holds	*/
	/* more than a single value).					*/
	return	new KBAttrItem (this) ;
}

/*  KBAttr								*/
/*  getAttrDlg	: Get component dialog for editing this attribute	*/
/*  parent	: QWidget *		: Parent widget			*/
/*  item	: KBAttrItem *		: Associated item		*/
/*  attrDict	: QDict<KBAttrItem> &	: All attributes		*/
/*  (returns)	: KBAttrDlg *		: Dialog or null		*/

KBAttrDlg
	*KBAttr::getAttrDlg
	(	QWidget			*,
		KBAttrItem		*,
		QDict<KBAttrItem>	&
	)
{
	/* The default is null, in which case the propery editor falls	*/
	/* back on the default cases.					*/
	return	0 ;
}

/*  KBAttr								*/
/*  displayValue: Get display string for value				*/
/*  value	: const QString & : Attribute value			*/
/*  (returns)	: QString	  :					*/

QString	KBAttr::displayValue
	(	const QString	&value
	)
{
	/* Note that the value is passed as an argument so that this	*/
	/* method can be invoked on values in property dialogd that	*/
	/* have not yet been saved.					*/

	/* Default is to remove anything at or beyond a newline and put	*/
	/* ellipsis in in place.					*/
	int	nlidx	= value.find ('\n') ;
	if (nlidx >= 0)
		return	value.left(nlidx) + " ...." ;
	return	value	;
}
#endif

/*  ------------------------------------------------------------------  */

/*  KBAttrInt								*/
/*  KBAttrInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrInt::KBAttrInt	
	(	KBNode			*node,
		cchar			*name,
		const QDict<QString>	&aList,
		uint			nflags
	)
	:
	KBAttr	(node, Int, name, aList, nflags)
{
}

/*  KBAttrInt								*/
/*  KBAttrInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  value	: cchar *	: Initial value				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrInt::KBAttrInt	
	(	KBNode			*node,
		cchar			*name,
		cchar			*value,
		uint			nflags
	)
	:
	KBAttr	(node, Int, name, value, nflags)
{
}

/*  KBAttrInt								*/
/*  KBAttrInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  _v		: int		: Initial value				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrInt::KBAttrInt	
	(	KBNode			*node,
		cchar			*name,
		int			v,
		uint			nflags
	)
	:
	KBAttr	(node, Int, name, QString::number(v), nflags)
{
}

/*  KBAttrInt								*/
/*  KBAttrInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  srce	: KBNode *	: Extant source				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrInt::KBAttrInt	
	(	KBNode			*node,
		cchar			*name,
		KBNode			*srce,
		uint			nflags
	)
	:
	KBAttr	(node, name, srce, nflags)
{
}

/*  KBAttrInt								*/
/*  ~KBAttrInt	: Destructor for integer attribute class		*/
/*  (returns)	:		:					*/

KBAttrInt::~KBAttrInt ()
{
}

/*  KBAttrInt								*/
/*  setValue	: Set attribute						*/
/*  v		: int		: Value					*/
/*  (returns)	: void		:					*/

void	KBAttrInt::setValue
	(	int	v
	)
{
	KBAttr::setValue (QString::number(v)) ;
}

/*  KBAttrInt								*/
/*  valueOK	: Check if value is OK					*/
/*  v		: const QString & : Value				*/
/*  (returns)	: void		  :					*/

bool	KBAttrInt::valueOK
	(	const QString	&v
	)
{
	bool	ok 	;
	if (v.isEmpty())
		ok	= true	;
	else	v.toInt (&ok)	;
	return	ok	;
}

/*  KBAttrInt								*/
/*  replicate	: Replicate attribute with new owner			*/
/*  (returns)	: KBAttr *	: Replicant				*/

KBAttr	*KBAttrInt::replicate
	(	KBNode	*owner
	)
{
	return	new KBAttrInt (owner, name, getValue(), nFlags) ;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrUInt								*/
/*  KBAttrUInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrUInt::KBAttrUInt	
	(	KBNode			*node,
		cchar			*name,
		const QDict<QString>	&aList,
		uint			nflags
	)
	:
	KBAttr	(node, UInt, name, aList, nflags)
{
}

/*  KBAttrUInt								*/
/*  KBAttrUInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  value	: cchar *	: Initial value				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrUInt::KBAttrUInt	
	(	KBNode			*node,
		cchar			*name,
		cchar			*value,
		uint			nflags
	)
	:
	KBAttr	(node, Int, name, value, nflags)
{
}

/*  KBAttrUInt								*/
/*  KBAttrUInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  _v		: uint		: Initial value				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrUInt::KBAttrUInt	
	(	KBNode			*node,
		cchar			*name,
		uint			v,
		uint			nflags
	)
	:
	KBAttr	(node, Int, name, QString::number(v), nflags)
{
}

/*  KBAttrUInt								*/
/*  KBAttrUInt	: Constructor for integer attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  srce	: KBNode *	: Extant source				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrUInt::KBAttrUInt	
	(	KBNode			*node,
		cchar			*name,
		KBNode			*srce,
		uint			nflags
	)
	:
	KBAttr	(node, name, srce, nflags)
{
}

/*  KBAttrUInt								*/
/*  ~KBAttrUInt	: Destructor for integer attribute class		*/
/*  (returns)	:		:					*/

KBAttrUInt::~KBAttrUInt ()
{
}

/*  KBAttrUInt								*/
/*  setValue	: Set attribute						*/
/*  v		: int		: Value					*/
/*  (returns)	: void		:					*/

void	KBAttrUInt::setValue
	(	uint	v
	)
{
	KBAttr::setValue (QString::number(v)) ;
}

/*  KBAttrUInt								*/
/*  valueOK	: Check if value is OK					*/
/*  v		: const QString & : Value				*/
/*  (returns)	: void		  :					*/

bool	KBAttrUInt::valueOK
	(	const QString	&v
	)
{
	bool	ok 	;
	if (v.isEmpty())
		ok	 = true	;
	else	v.toUInt (&ok)	;
	return	ok	;
}

/*  KBAttrUInt								*/
/*  replicate	: Replicate attribute with new owner			*/
/*  (returns)	: KBAttr *	: Replicant				*/

KBAttr	*KBAttrUInt::replicate
	(	KBNode	*owner
	)
{
	return	new KBAttrUInt (owner, name, getValue(), nFlags) ;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrStr	: Constructor for string attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  (returns)	: KBAttr	:					*/

KBAttrStr::KBAttrStr
	(	KBNode			*node,
		cchar			*name,
		const QDict<QString>	&aList,
		uint			nflags
	)
	:
	KBAttr	(node, String, name, aList, nflags)
{
}

/*  KBAttrStr								*/
/*  KBAttrStr	: Constructor for string attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  text	: cchar *	: Initial text value			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrStr::KBAttrStr
	(	KBNode			*node,
		cchar			*name,
		cchar			*text,
		uint			nflags
	)
	:
	KBAttr	(node, String, name, text, nflags)
{
}

/*  KBAttrStr								*/
/*  KBAttrStr	: Constructor for string attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  srce	: KBNode *	: Extant source				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrStr::KBAttrStr
	(	KBNode			*node,
		cchar			*name,
		KBNode			*srce,
		uint			nflags
	)
	:
	KBAttr	(node, name, srce, nflags)
{
}

/*  KBAttrStr								*/
/*  ~KBAttrStr	: Destructor for string attribute class			*/
/*  (returns)	:		:					*/

KBAttrStr::~KBAttrStr ()
{
}

/*  KBAttrStr								*/
/*  replicate	: Replicate attribute with new owner			*/
/*  (returns)	: KBAttr *	: Replicant				*/

KBAttr	*KBAttrStr::replicate
	(	KBNode	*owner
	)
{
	return	new KBAttrStr (owner, name, getValue(), nFlags) ;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrBool								*/
/*  KBAttrBool	: Constructor for boolean attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrBool::KBAttrBool	
	(	KBNode			*node,
		cchar			*name,
		const QDict<QString>	&aList,
		uint			nflags
	)
	:
	KBAttr	(node, Bool, name, aList, nflags)
{
}

/*  KBAttrBool								*/
/*  KBAttrBool	: Constructor for boolean attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  value	: bool		: Value					*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrBool::KBAttrBool	
	(	KBNode			*node,
		cchar			*name,
		bool			value,
		uint			nflags
	)
	:
	KBAttr	(node, Bool, name, value ? "Yes" : "No", nflags)
{
}

/*  KBAttrBool								*/
/*  KBAttrBool	: Constructor for boolean attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  value	: cchar *	: Initial value				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrBool::KBAttrBool	
	(	KBNode			*node,
		cchar			*name,
		cchar			*value,
		uint			nflags
	)
	:
	KBAttr	(node, Bool, name, value, nflags)
{
}

/*  KBAttrBool								*/
/*  KBAttrBool	: Constructor for boolean attribute class		*/
/*  node	: KBNode *	: Owning node				*/
/*  name	: cchar *	: Attribute name			*/
/*  srce	: KBNode *	: Extant source				*/
/*  nflags	: uint		: Required node flags			*/
/*  (returns)	: KBAttr	:					*/

KBAttrBool::KBAttrBool
	(	KBNode			*node,
		cchar			*name,
		KBNode			*srce,
		uint			nflags
	)
	:
	KBAttr	(node, name, srce, nflags)
{
}

/*  KBAttrBool								*/
/*  ~KBAttrBool	: Destructor for integer attribute class		*/
/*  (returns)	:		:					*/

KBAttrBool::~KBAttrBool ()
{
}

/*  KBAttrBool								*/
/*  setValue	: Set attribute						*/
/*  v		: bool		: Value					*/
/*  (returns)	: void		:					*/

void	KBAttrBool::setValue
	(	bool	v
	)
{
	KBAttr::setValue (v ? "Yes" : "No") ;
}

/*  KBAttrBool								*/
/*  getBoolValue: Get value as a boolean				*/
/*  (returns)	: bool		: Value					*/

bool	KBAttrBool::getBoolValue ()
{
	return	getValue() == "Yes" ;
}

/*  KBAttrBool								*/
/*  replicate	: Replicate attribute with new owner			*/
/*  (returns)	: KBAttr *	: Replicant				*/

KBAttr	*KBAttrBool::replicate
	(	KBNode	*owner
	)
{
	return	new KBAttrInt (owner, name, getValue(), nFlags) ;
}
