/***************************************************************************
    file	         : kb_slot.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifdef		_WIN32
#include	"kb_slot.h"
#else
#include	"kb_slot.moc"
#endif

#include	"kb_object.h"
#include	"kb_docroot.h"
#include	"kb_script.h"
#include	"kb_appptr.h"
#include	"kb_callback.h"



/*  KBSlotLink								*/
/*  KBSlotLink	: Constructor for a slot linkage			*/
/*  (returns)	: KBSlotLink	:					*/

KBSlotLink::KBSlotLink ()
{
}

/*  KBSlotLink								*/
/*  KBSlotLink	: Constructor for a slot linkage			*/
/*  name	: const QString & : Link name				*/
/*  target	: const QString & : Target object			*/
/*  event	: const QString & : Event on target object		*/
/*  enabled	: bool		  : Slot enabled			*/
/*  (returns)	: KBSlotLink	  :					*/

KBSlotLink::KBSlotLink
	(	const QString	&name,
		const QString	&target,
		const QString	&event,
		bool		enabled
	)
	:
	m_name	 (name),
	m_target (target),
	m_event	 (event),
	m_enabled(enabled)
{
}


/*  ------------------------------------------------------------------  */

/*  KBSlot								*/
/*  KBSlot	: Constructor for slot					*/
/*  parent	: KBObject *	  : Parent object			*/
/*  name	: const QString & : Slot name				*/
/*  (returns)	: KBSlot	  :					*/

KBSlot::KBSlot
	(	KBObject	*parent,
		const QString	&name
	)
	:
	m_parent(parent),
	m_name	(name)
{
	if (m_parent != 0) m_parent->addSlot (this) ;

	m_disable = false ;
	m_script  = 0	  ;
}

/*  KBSlot								*/
/*  KBSlot	: Constructor for slot					*/
/*  parent	: KBObject *	  : Parent object			*/
/*  extant	: KBSlot *	: Extant slot to be copied		*/
/*  (returns)	: KBSlot	:					*/

KBSlot::KBSlot
	(	KBObject	*parent,
		KBSlot		*extant
	)
	:
	m_parent(parent)
{
	if (m_parent != 0) m_parent->addSlot (this) ;

	m_linkage = extant->m_linkage ;
	m_name	  = extant->m_name    ;
	m_code	  = extant->m_code    ;

	m_disable = false ;
	m_script  = 0	  ;
}

/*  KBSlot								*/
/*  setParent	: Set slot's parent object				*/
/*  parent	: KBobject *	: Parent				*/
/*  (returns)	: void		  :					*/

void	KBSlot::setParent
	(	KBObject		*parent
	)
{
	/* This method is typically used when the user OK's on a	*/
	/* property dialog, and updated slots are returned to the	*/
	/* parent.							*/
	if ((m_parent = parent) != 0) m_parent->addSlot (this) ;
}

/*  KBSlot								*/
/*  addLinkage	: Add a linkage element					*/
/*  name	: const QString & : Link name				*/
/*  target	: const QString & : Target object			*/
/*  event	: const QString & : Event on target object		*/
/*  enabled	: bool		  : Slot enabled			*/
/*  (returns)	: void		  :					*/

void	KBSlot::addLinkage
	(	const QString	&name,
		const QString	&target,
		const QString	&event,
		bool		enabled
	)
{
	m_linkage.append (KBSlotLink (name, target, event, enabled)) ;
}

/*  KBSlot								*/
/*  setCode	: Set code associated with slot				*/
/*  code	: const QString & : Code				*/
/*  append	: bool		  : Append code else replace		*/
/*  (returns)	: void		  :					*/

void	KBSlot::setCode
	(	const QString	&code,
		bool		append
	)
{
	if (append)
	{
		m_code	+= code	;
		return	;
	}

	m_code	= code	;
}

/*  KBSlot								*/
/*  tidy	: Tidy up after loading					*/
/*  (returns)	: void		:					*/

void	KBSlot::tidy ()
{
	/* This is just so we can output the code with a leading 	*/
	/* newline in the XML document.					*/
	m_code	= m_code.stripWhiteSpace() + "\n" ;
}

/*  KBSlot								*/
/*  printNode	: Print as node in XML document				*/
/*  text	: QString &	: Text buffer				*/
/*  indent	: int		: Indent level				*/
/*  (returns)	: void		  :					*/

void	KBSlot::printNode
	(	QString		&text,
		int		indent
	)
{
	text	+= QString("%1<slot").arg("", indent) ;
	KBAttr::addAttrText (text, "name", m_name) ;
	text	+= ">\n" ;

	for (uint idx = 0 ; idx < m_linkage.count() ; idx += 1)
	{
		const KBSlotLink &link = m_linkage[idx] ;

		text	+= QString("%1<slotlink").arg("", indent + 2) ;

		KBAttr::addAttrText (text, "name",   link.name  (), true) ;
		KBAttr::addAttrText (text, "target", link.target(), true) ;
		KBAttr::addAttrText (text, "event",  link.event (), true) ;

		text	+= QString(" enabled=\"%1\"").arg(link.enabled()) ;
		text	+= "/>\n" ;
	}

	text	+= QString("%1<slotcode>\n" ).arg("", indent + 2) ;
	text	+= KBAttr::escapeText (m_code, false) ;
	text	+= QString("%1</slotcode>\n").arg("", indent + 2) ;

	text	+= QString("%1</slot>\n").arg("", indent) ;
}

/*  KBSlot								*/
/*  connectLinks: Connect links to enitters				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		:					*/

bool	KBSlot::connectLinks
	(	KBError		&pError
	)
{
	DELOBJ(m_script)  ;
	m_disable = false ;

	for (uint idx = 0 ; idx < m_linkage.count() ; idx += 1)
	{
		const KBSlotLink &link = m_linkage[idx] ;

		if (!link.enabled())
			continue ;

		KBObject   *target  = m_parent->getNamedObject
				      (		link.target(),
				   		pError,
				   		true
				      )	;
		if (target  == 0) return false ;

		KBEmitter  *emitter = target  ->getEmitter (link.event()) ;
		if (emitter == 0)
		{
			pError	= KBError
				  (	KBError::Error,
					TR("Event not found for slot link"),
				  	QString("%1[%2]")
						.arg(link.target())
						.arg(link.event ()),
					__ERRLOCN
				  )	;
			return	false	;
		}

		if (!connect
			(	emitter,
				SIGNAL(eventSignal(KBObject *, const QString &, uint, KBValue *, KB::ScriptRC &)),
				SLOT  (eventSignal(KBObject *, const QString &, uint, KBValue *, KB::ScriptRC &))
			)
		   )
		{
			pError	= KBError
				  (	KBError::Error,
					TR("Failed to connect to event emitter"),
				  	QString("%1[%2]")
						.arg(link.target())
						.arg(link.event ()),
					__ERRLOCN
				  )	;
			return	false	;
		}

//		fprintf
//		(	stderr,
//			"KBSlot::connectLink: [%s] [%s/%s]\n",
//			(cchar *)m_name,
//			(cchar *)target->getAttrVal("name"),
//			(cchar *)link.event()
//		)	;
	}

	return	true	;
}

/*  KBSlot								*/
/*  eventSignal	: Event handler for the slot				*/
/*  source	: KBObject *	  : Originating object			*/
/*  event	: const QString & : Originators event name		*/
/*  argc	: uint		  : Number of arguments			*/
/*  argv	: KBValue *	  : Vector of arguments			*/
/*  rc		: KB::ScriptRC  & : Script execution return code	*/
/*  (returns)	: void		  :					*/

void	KBSlot::eventSignal
	(	KBObject	*source,
		const QString	&event,
		uint		argc,
		KBValue		*argv,
		KB::ScriptRC	&rc
	)
{
	/* NOTE: The "rc" return is *only* set on error, having been	*/
	/* initialised by the caller to value "KB::ScriptOK". This can	*/
	/* then be detected so that event/slot execution can be aborted	*/
	/* as soon as any slot throws an error.				*/
	if (rc != KB::ScriptOK)
		return	;

//	fprintf
//	(	stderr,
//		"KBSlot::eventSignal [%s][%s]->[%s][%s] [%d/%p]\n",
//		(cchar *)source  ->getAttrVal("name"),
//		(cchar *)event,
//		(cchar *)m_parent->getAttrVal("name"),
//		(cchar *)m_name,
//		argc,
//		(void  *)argv
//	)	;

	bool		ok	  ;
	KBError		error	  ;
	KBDocRoot	*root	  = m_parent->getDocRoot    () ;
	KBScriptIF	*scrIface = root    ->loadScripting (ok, error) ;

	KBAppPtr::getCallback()->logEvent
	(	"Slot",
		m_parent->className (),
		m_parent->getAttrVal("name"),
		m_name,
		argc,
		argv
	)	;

	if (!ok)
	{
		error.DISPLAY () ;
		rc	= KB::ScriptInlineError ;
		return	;
	}
	if (scrIface == 0)
	{
		KBError::EError
		(	TR("No scripting language specified"),
			QString(TR("Trying to execute slot %1.%2"))
				.arg(m_parent->getAttrVal("name"))
				.arg(m_name),
			__ERRLOCN
		)	;
		rc	= KB::ScriptInlineError ;
		return	;
	}

	/* If the event is flagged as disabled then there has been an	*/
	/* earlier error ...						*/
	if (m_disable)
	{
		KBError::EError
		(	TR("Slot has been disabled due to earlier error"),
			QString(TR("Trying to execute slot %1.%2"))
				.arg(m_parent->getAttrVal("name"))
				.arg(m_name),
			__ERRLOCN
		)	;

		m_parent->getDocRoot()->doExecError ()	;
		rc	= KB::ScriptInlineError		;
		return	;
	}

	/* Now on to explicit events, where we compile the code and	*/
	/* keep a copy. First see if the code has already been compiled	*/
	/* and if not then so so now.					*/
	if (m_script == 0)
	{
		QString		eText	;
		QString		ePatt	;
		KBError		error	;

		if ((m_script = scrIface->compileFunc
				(	m_code,
					m_parent->getPath(),
					"slotFunc",
					eText,
					ePatt,
					m_parent->getDocRoot()->getImports(),
					0,
					error
				) ) == 0)
		{
			/* Compile error in the inline code. Display	*/
			/* the error message, and return the form or	*/
			/* report to design mode.			*/
			error.DISPLAY  () ;
			m_parent->getDocRoot()->doExecError () ;

			/* The slot is disabled ...			*/
			m_disable = true ;
			rc	  = KB::ScriptInlineError ;
			return	;
		}
	}

	/* Code was or has been compiled successfully, in which case	*/
	/* execute it. Check for an execution time error ....		*/
	KBValue		resval	;
	KBScript::ExeRC	exerc	= m_script->execute
				  (	m_parent,
					source,
					event,
					argc,
					argv,
					resval
				  )	;

	if ((exerc == KBScript::ExeError) || (exerc == KBScript::ExeFail))
	{
		QString	   errMsg   ;
		uint	   errLno   ;
		QString	   errText  ;
		KBLocation errLocn  = scrIface->exeError (errMsg, errLno, errText) ;

		KBError::EError
		(	errMsg,
			QString (TR("%1, line %2\n%3"))
				.arg(errLocn.docName)
				.arg(errLno)
				.arg(errText),
			__ERRLOCN
		)	;

		m_parent->getDocRoot()->doExecError () ;

		if (errLocn.docName != KBLocation::m_pInline)
		{
			if (exerc != KBScript::ExeFail)
			{
				KBError	   error    ;
				if (!KBAppPtr::getCallback()->editScript (errLocn, errText, "", errLno, error))
					error.DISPLAY () ;
			}

			rc	= KB::ScriptGlobalError ;
			return	;
		}

		rc	  = KB::ScriptInlineError ;
		m_disable = true ;
		return	  ;
	}

	/* All OK. As noted above, the "rc" value is *not* set under	*/
	/* this circumstance.						*/
}

