/***************************************************************************
    file	         : kb_wizard.h
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef		__KB_WIZARD_H
#define		__KB_WIZARD_H

#include	<qwidget.h>
#include	<qwidgetstack.h>
#include	<qlabel.h>
#include	<qcombobox.h>
#include	<qcheckbox.h>
#include	<qtextview.h>
#include	<qpushbutton.h>
#include	<qlayout.h>
#include	<qdict.h>
#include	<qstring.h>
#include	<qlist.h>
#include	<qstack.h>
#include	<qvariant.h>


#include	"libkbase_exports.h"

#include	"kb_dom.h"
#include	"kb_classes.h"
#include	"kb_dialog.h"

#include	"tk_lineedit.h"


class	KBWizardPage	;
class	KBWizard	;
class	KBLocation	;

typedef	struct	_cbuff	CBUFF	;


/*  KBWizardCtrl							*/
/*  -------------							*/
/*  This is a abstract base class used for controls in the wizard pages	*/
/*  It basically exists to hold a widget pointer, a name, and a set/get	*/
/*  value interface.							*/

class	LIBKBASE_API KBWizardCtrl : public QObject
{
	Q_OBJECT

	KBWizardPage	*m_page		;
	QString		m_name		;
	QWidget		*m_widget	;
	bool		m_required	;
	bool		m_wide		;
	bool		m_changed	;

protected slots :

	void		ctrlChanged	() ;

public	:

	KBWizardCtrl			(KBWizardPage *, const QString &) ;
	virtual	~KBWizardCtrl		() ;

	/* The "pageShown" method is called when then page which the	*/
	/* control is embedded in is show, can be overridden to update	*/
	/* the control (for instance, if the control is dependant on	*/
	/* values in other pages). The argument is true for "next" and	*/
	/* false for "previous" movements.				*/
	virtual	void	pageShown	(bool) ;


	virtual	void	setValue	(const QString &)	= 0 ;
	virtual	QString	value		()			= 0 ;


	/* Check if the value is OK when moving off the page that it is	*/
	/* embedded in.							*/
	virtual	bool	ok		() ;


	/* Save is called on all controls when the user exits the	*/
	/* wizard, so that controls can do stuff like save their	*/
	/* current configuration. Argument is dialog return code	*/
	virtual	void	save		(int) ;


	virtual	QVariant
			attribute	(const QString &) ;

	/* Set the underlying widget. This is called from the derived	*/
	/* control classes when thay have created the control's data	*/
	/* widget.							*/
	inline	void	setWidget
		(	QWidget	*widget
		)
	{
		m_widget = widget ;
	}


	inline	void	setRequired
		(	bool	required
		)
	{
		m_required = required	;
	}

	inline	void	setWide
		(	bool	wide
		)
	{
		m_wide	= wide	;
	}

	inline	void	setChanged
		(	bool	changed
		)
	{
		m_changed = changed	;
	}

	/* Methods to return the underlying widget (used during page	*/
	/* layout), the the other stuff stored with the control.	*/
	inline	QWidget		*widget	()	{ return m_widget	; }
	inline	const QString	&name	()	{ return m_name 	; }
	inline	KBWizardPage	*page	()	{ return m_page 	; }
	inline	bool		required()	{ return m_required	; }
	inline	bool		wide	()	{ return m_wide		; }
	inline	bool		changed	()	{ return m_changed	; }
}	;


/*  KBWizardHidden							*/
/*  --------------							*/

class	LIBKBASE_API KBWizardHidden : public KBWizardCtrl
{
	QString		m_value		;

public	:

	KBWizardHidden
	(	KBWizardPage	*, 
		const QString	&,
		const QString	&
	)	;

	virtual	~KBWizardHidden		()	;

	virtual	void	setValue	(const QString &) ;
	virtual	QString	value		()	;
}	;


/*  KBWizardLineEdit							*/
/*  ----------------							*/

class	LIBKBASE_API KBWizardLineEdit : public KBWizardCtrl
{
	Q_OBJECT

	TKLineEdit	*m_lineEdit	;

public	:

	KBWizardLineEdit
	(	KBWizardPage	*,
		const QString	&,
		const QString	&,
		bool		= false
	)	;
	virtual	~KBWizardLineEdit	()	;

	virtual	void	setValue	(const QString &) ;
	virtual	QString	value		()	;

	inline	TKLineEdit *lineEdit	()
	{
		return	m_lineEdit ;
	}
}	;



/*  KBWizardComboBox							*/
/*  ----------------							*/

class	LIBKBASE_API KBWizardComboBox : public KBWizardCtrl
{
	Q_OBJECT

	QComboBox	*m_comboBox	;
	QStringList	m_infoList	;
	QTextView	*m_infoView	;

protected slots :

	void		ctrlChanged	() ;

public	:

	KBWizardComboBox
	(	KBWizardPage		*,
		const QString 		&,
		const QStringList	&,
		const QString 		&,
		bool
	)	;
	virtual	~KBWizardComboBox	()	;

	virtual	void	setValue	(const QString &) ;
	virtual	QString	value		()	;		

	virtual	QVariant
			attribute	(const QString &) ;

	inline	QComboBox *comboBox	()
	{
		return	m_comboBox	;
	}

	void	setInfoList		(const QStringList &) ;
}	;


/*  KBWizardCheckBox							*/
/*  ----------------							*/

class	LIBKBASE_API KBWizardCheckBox : public KBWizardCtrl
{
	QCheckBox	*m_checkBox	;

public	:

	KBWizardCheckBox
	(	KBWizardPage	*,
		const QString	&,
		const QString	&

	)	;
	virtual	~KBWizardCheckBox	()	;

	virtual	void	setValue	(const QString &) ;
	virtual	QString	value		()	;		

	inline	QCheckBox *checkBox	()
	{
		return	m_checkBox	;
	}
}	;


/*  KBWizardPage							*/
/*  ------------							*/
/*  Instances of this class are used to represent pages in the wizard,	*/
/*  each of which can contain one or more controls.			*/

class	LIBKBASE_API KBWizardPage : public QWidget
{
	Q_OBJECT

protected :

	KBWizard		*m_wizard	;
	QDomElement		m_elem		;

	QString			m_name		;
	QString			m_title		;
	QString			m_blurb		;
	uint			m_finish	;

	QGridLayout		*m_layout	;
	QList<KBWizardCtrl>	m_ctrls		;
	QList<QLabel>		m_labels	;
	KBError			m_error		;
	QTextView		*m_info		;

	CBUFF			*m_elNext	;
	CBUFF			*m_elOK		;

	CBUFF			*compile	(const QString &) ;
	QString			execute		(CBUFF *)	  ;

public	:

	KBWizardPage
	(	KBWizard 	*,
		QWidget 	*,
		const QString	&
	)	;

	virtual	~KBWizardPage	() ;

	/* Methods to retrieve the common stuff ...			*/
	inline	const QString	&name		()	{ return m_name   ; }
	inline	const QString	&title		()	{ return m_title  ; }
	inline	const QString	&blurb		()	{ return m_blurb  ; }
	inline	uint		finish		()	{ return m_finish ; }
	inline	KBWizard	*wizard		()	{ return m_wizard ; }

	/* Initialise a page specified by a general XML specification.	*/
	/* Typically called from KBWizard as it builds the set of pages	*/
	/* in the wizard.						*/
	void			init		(const QDomElement &) ;

	/* Methods to build specified types of controls. The versions	*/
	/* that take DOM elements are generally used when building from	*/
	/* an XML specification, the others when building "by hand".	*/
	KBWizardHidden		*addHiddenCtrl	(const QDomElement   &) ;
	KBWizardHidden		*addHiddenCtrl	(const QString &, const QString &, const QString     &) ;
	KBWizardLineEdit	*addTextCtrl	(const QDomElement   &) ;
	KBWizardLineEdit	*addTextCtrl	(const QString &, const QString &, const QString     &, bool = false) ;
	KBWizardComboBox	*addChoiceCtrl	(const QDomElement   &) ;

	KBWizardComboBox	*addChoiceCtrl
				(	const QString		&,
					const QString		&,
					const QStringList	&,
					const QString 		& = QString::null,
					bool		  	= false
				)	;


	KBWizardCheckBox	*addCheckCtrl	(const QDomElement   &) ;
	KBWizardCheckBox	*addCheckCtrl	(const QString &, const QString &, const QString     &) ;

	/* Main method to add a DOM specified control. Handles specific	*/
	/* controls as above, plus controls registered via the		*/
	/* KBWizardCtrlReg class and associated macro.			*/
	KBWizardCtrl		*addCtrl	(const QDomElement &) ;


	/* Add an information "control", used to display additional	*/
	/* information when the user changes control values, for	*/
	/* instance some text about individual database drivers.	*/
	void			setInfoCtrl	(QTextView	*) ;


	/* The "addedAll" method is called after all control have been	*/
	/* added, and is used to finalise the layout.			*/
	void			addedAll	() ;

	/* The "pageShown" method is called when the page is displayed,	*/
	/* and calls "pageShown" on each of the controls in the page.	*/
	/* The argument is true for "next" and false for "previous"	*/
	/* movements.							*/
	void			pageShown	(bool) ;


	void			setCtrl		(const QString &, const QString &) ;
	void			setCtrl		(uint, 		  const QString &) ;
	QString			ctrlValue	(const QString &) ;
	QString			ctrlValue	(uint) ;

	void			ctrlChanged	(KBWizardCtrl   *) ;

	void			settings	(QDict<QString> &, bool = false) ;
	void			clear		() ;

	/* The "ok" method is used to check if the page contents are	*/
	/* acceptable prior to moving to the next page. The "nextPage"	*/
	/* method returns the name of the next page or the empty	*/
	/* string for the default.					*/
	virtual	bool		ok		() ;
	virtual QString		nextPage	() ;


	/* Save is called on all pages when the user exits the wizard,	*/
	/* so that pages can do stuff like save their current		*/
	/* configuration. Argument is the dialog exit code.		*/
	virtual	void		save		(int) ;


	KBWizardCtrl		*findCtrl	(const QString  &, const char * = 0) ;

	inline	uint	numCtrls ()
	{
		return	m_ctrls.count() ;
	}

	inline	void	setBlurb
		(	const QString	&blurb
		)
	{
		m_blurb = blurb ;
	}

	inline	void	setTitle
		(	const QString	&title
		)
	{
		m_title = title ;
	}

	inline	const KBError	&lastError ()
	{
		return	m_error	;
	}
}	;


/*  KBWizard								*/
/*  --------								*/
/*  This is the wizard class. Not based on QWizard so that we can do	*/
/*  our own thing.							*/

class	LIBKBASE_API KBWizard : public _KBDialog
{
	Q_OBJECT

	KBDBInfo		*m_dbInfo	;
	QString			m_server	;

	QLabel			m_title		;
	QTextView		m_blurb		;
	QWidgetStack		m_stack		;
	QPushButton		m_bPrevious	;
	QPushButton		m_bNext		;
	QPushButton		m_bFinish	;
	QPushButton		m_bCancel	;

	QList <KBWizardPage>	m_pages		;
	uint			m_curPage	;
	QStack<KBWizardPage>	m_history	;

	KBError			m_error		;
	QString			m_text		;

	QMap<QString,void *>	m_cookies	;

	void	showPage	(uint, KBWizardPage *, bool) ;

protected slots :

	void	clickPrevious	() ;
	void	clickNext	() ;
	void	clickFinish	() ;
	void	clickCancel	() ;

public	:

	KBWizard	(KBDBInfo *, const QString &);

	KBWizardPage* 	addNewPage(const QDomElement &);

	void		showPage	(uint, bool)		;
	void		showPage	(KBWizardPage *, bool)	;

	void		settings	(QDict<QString> &, bool = false) ;

	bool 		init		(const QDomDocument &)	;
	bool		init		(const QString &)	;
	int		exec		(bool = true)		;
	virtual	int	execute		()			;

	void 		ctrlChanged	(KBWizardPage *, KBWizardCtrl *) ;

	KBWizardPage*	findPage	(const QString  &)	;
	KBWizardCtrl*	findCtrl	(const QString  &, const QString &, const char * = 0) ;
	QString		ctrlValue	(const QString  &, const QString &) ;
	QVariant	ctrlAttribute	(const QString  &, const QString &, const QString  &) ;

	/* Retrieve the current page name. Most useful when the wizard	*/
	/* has terminated in order to check which page the user		*/
	/* finished from.						*/
	QString		currentPageName	() ;

	/* Save is called on all pages when the user exits the wizard,	*/
	/* so that the wizard can do stuff like save its current	*/
	/* configuration. Argument is the dialog exit code.		*/
	virtual	void	save		(int) ;

	/* Cookies allow the caller to pass values into the wizard, and	*/
	/* to retrieve them, without needing explicit knowledge of the	*/
	/* wizard.							*/
	void		setCookie	(const QString &, void *) ;
	void		*cookie		(const QString &)	  ;

	QString			wizForLocale	(const QString &) ;

	inline	const KBError	&lastError ()
	{
		return	m_error	;
	}

	inline	KBDBInfo	*dbInfo	() { return m_dbInfo ; }
	inline	const QString	&server	() { return m_server ; }
}	;

/*  ------------------------------------------------------------------  */

/*  Wizard controls are registered using the WIZARDCTRLREG macro, which */
/*  creates a static instance of the KBWizardCtrlReg class. This 	*/
/*  executes when the library is loaded, and registers the wizard	*/
/*  control.								*/

typedef	KBWizardCtrl	*(*MKWIZARDCTRL)
			(	KBWizardPage		*,
				const QDomElement	&
			)	;

class	LIBKBASE_API	KBWizardCtrlReg
{
public	:

	KBWizardCtrlReg (cchar *, MKWIZARDCTRL) ;

	static	KBWizardCtrl	*makeWizardCtrl
		(	const QString		&name,
			KBWizardPage		*page,
			const QDomElement	&elem
		)	;
}	;

#define	WIZARDCTRLREG(t)						\
									\
	KBWizardCtrl							\
		*newWizardCtrl##t					\
		(	KBWizardPage		*page,			\
			const QDomElement	&elem			\
		)							\
	{								\
		return	new KBWizardCtrl##t (page, elem) ;		\
	}								\
	static	KBWizardCtrlReg  reg##t (#t, newWizardCtrl##t)	;


/*  ------------------------------------------------------------------  */

/*  Similarly, wizards can be registered using the WIZARDREG macro,	*/
/*  which creates a static instance of the KBWizardReg class. This 	*/
/*  executes when the library is loaded, and registers the wizard	*/
/*  control.								*/

typedef	KBWizard	*(*MKWIZARD)
			(	KBLocation	&,
				const QString	&
			)	;

class	LIBKBASE_API	KBWizardReg
{
public	:

	KBWizardReg (cchar *, MKWIZARD) ;

	static	KBWizard*makeWizard
		(	const QString		&name,
			KBLocation		&location,
			const QString		&server
		)	;
}	;

#define	WIZARDREG(t)							\
									\
	KBWizard							\
		*newWizard##t						\
		(	KBLocation		&location,		\
			const QString		&server			\
		)							\
	{								\
		return	new KBWizard##t (location, server) ;		\
	}								\
	static	KBWizardReg  reg##t (#t, newWizard##t)	;

#endif	//	__KB_WIZARD_H

